/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: perform.c,v 1.57 1999/08/29 19:35:16 tiggr Exp $  */

#include "trt.h"
#include <tom/tom-r.h>
#include <stdarg.h>
#include <stdio.h>

#define INVOCATION_BUILD_RESULT()  \
do {						\
  inv->result = xcalloc (1, sizeof (*inv->result)); \
  inv->result->desc = ((selector) INV_SELECTOR (inv))->out; \
						\
  if (sel->out->num > 1)			\
    {						\
      /* XXX Allocate largest size; isn't necessarily a double.  */ \
      char *res = xmalloc ((sel->out->num - 1) * sizeof (double)); \
						\
      inv->result->offsets = xmalloc (sel->out->num \
				      * sizeof (*inv->result->offsets)); \
      inv->result->offsets[0] = 0;		\
      inv->result->mrv = res;			\
						\
      for (i = 1; i < sel->out->num; i++)	\
	switch (sel->out->args[i])		\
	  {					\
	  case TRT_TE_BOOLEAN:			\
	  case TRT_TE_BYTE:			\
	    inv->result->offsets[i] = res - (char *) inv->result->mrv; \
	    APPLY_ARGS_EMIT_POINTER (res);	\
	    res++;				\
	    break;				\
						\
	    /* XXX Is this portable?  Is the alignment necessary?  */ \
	  case TRT_TE_CHAR:			\
	    res = (void *) (((POINTER_INT_TYPE) res + sizeof (tom_char) - 1) \
			    & ~(sizeof (tom_char) - 1)); \
	    inv->result->offsets[i] = res - (char *) inv->result->mrv; \
	    APPLY_ARGS_EMIT_POINTER (res);	\
	    res += sizeof (tom_char);		\
	    break;				\
						\
	  case TRT_TE_INT:			\
	    res = (void *) (((POINTER_INT_TYPE) res + sizeof (tom_int) - 1) \
			    & ~(sizeof (tom_int) - 1)); \
	    inv->result->offsets[i] = res - (char *) inv->result->mrv; \
	    APPLY_ARGS_EMIT_POINTER (res);	\
	    res += sizeof (tom_int);		\
	    break;				\
						\
	  case TRT_TE_LONG:			\
	    res = (void *) (((POINTER_INT_TYPE) res + sizeof (tom_long) - 1) \
			    & ~(sizeof (tom_long) - 1)); \
	    inv->result->offsets[i] = res - (char *) inv->result->mrv; \
	    APPLY_ARGS_EMIT_POINTER (res);	\
	    res += sizeof (tom_long);		\
	    break;				\
						\
	  case TRT_TE_FLOAT:			\
	    res = (void *) (((POINTER_INT_TYPE) res + sizeof (tom_float) - 1) \
			    & ~(sizeof (tom_float) - 1)); \
	    inv->result->offsets[i] = res - (char *) inv->result->mrv; \
	    APPLY_ARGS_EMIT_POINTER (res);	\
	    res += sizeof (tom_float);		\
	    break;				\
						\
	  case TRT_TE_DOUBLE:			\
	    res = (void *) (((POINTER_INT_TYPE) res + sizeof (tom_double) - 1) \
			    & ~(sizeof (tom_double) - 1)); \
	    inv->result->offsets[i] = res - (char *) inv->result->mrv; \
	    APPLY_ARGS_EMIT_POINTER (res);	\
	    res += sizeof (tom_double);		\
	    break;				\
						\
	  case TRT_TE_POINTER:			\
	  case TRT_TE_SELECTOR:			\
	  case TRT_TE_REFERENCE:		\
	    res = (void *) (((POINTER_INT_TYPE) res + sizeof (void *) - 1) \
			    & ~(sizeof (void *) - 1)); \
	    inv->result->offsets[i] = res - (char *) inv->result->mrv; \
	    APPLY_ARGS_EMIT_POINTER (res);	\
	    res += sizeof (void *);		\
	    break;				\
						\
	  case TRT_TE_VOID:			\
	    /* This can not happen?  */		\
	    fatal ("void return value type at %d to selector %s", \
		   i, sel->name.s);		\
	  default:				\
	    ABORT ();				\
	  }					\
						\
      inv->result->size = res - (char *) inv->result->mrv; \
						\
      /* Clear the current values of the multiple-return values, since \
	 garbage collection could visit before the first fire of this \
	 invocation.  */			\
      BZERO (inv->result->mrv, inv->result->size); \
    }						\
} while (0)

/* Fill the invocation struct INV for a method invocation on the OBJECT
   with the selector SEL.  The arguments to the invocation are to be
   retrieved from the va_list pointed to by PAP.  The CMD describes the
   arguments that are supplied.  The first SKIP_IN arguments to the CMD
   are to be ignored; only the remaining arguments are the arguments to be
   passed by the invocation.  */
struct trt_invocation *
invocation_build_args (struct trt_invocation *inv, tom_object object,
		       selector sel, selector cmd, int skip_in, va_list *pap)
{
  int num_args_provided = cmd ? cmd->in->num - skip_in : sel->in->num;
  union apply_args *args;
  int i, args_size;

  if (sel->in->num >= num_args_provided);
  else if (num_args_provided == 1
	   && cmd->in->args[cmd->in->num - 1] == TRT_TE_VOID)
    num_args_provided--;
  else
    fatal ("invocation build argument count mismatch: %s, %s",
	   cmd->name.s, sel->name.s);

  args_size = 2 * sizeof (void *) + APPLY_ARGS_REG_SIZE;
  args = xmalloc (args_size);

  APPLY_ARGS_START (sel);
  args->stack = xmalloc (APPLY_ARGS_STACK_SIZE (sel));
  APPLY_ARGS_CONTINUE ();
  BZERO (inv, sizeof (*inv));
  inv->args = args;
  inv->arg_types = xmalloc (sel->in->num);
  inv->stack = args->stack;
  inv->next_arg = num_args_provided;
  inv->arg_pointers = xmalloc ((2 + sel->in->num) * sizeof (void *));
  inv->arg_pointers[0] = APPLY_ARGS_EMIT_REFERENCE (object);
  inv->arg_pointers[1] = APPLY_ARGS_EMIT_SELECTOR (sel);

  for (i = 0; i < sel->in->num; i++)
    {
      int stored_type = sel->in->args[i];
      int real = i < num_args_provided;
      void *addr;

      if (real && cmd && sel->in->args[i] != cmd->in->args[i + skip_in])
	{
	  fatal ("invocation_build argument type mismatch at %d: %s, %s",
		 i, cmd->name.s, sel->name.s);
	  goto bad;
	}

      switch (sel->in->args[i])
	{
	case TRT_TE_BOOLEAN:
	case TRT_TE_BYTE:
	  addr = APPLY_ARGS_EMIT_BYTE (real ? VA_ARG_BYTE (*pap) : 0);
	  break;
	case TRT_TE_CHAR:
	  addr = APPLY_ARGS_EMIT_CHAR (real ? VA_ARG_CHAR (*pap) : 0);
	  break;
	case TRT_TE_INT:
	  addr = APPLY_ARGS_EMIT_INT (real ? va_arg (*pap, tom_int) : 0);
	  break;
	case TRT_TE_LONG:
	  addr = APPLY_ARGS_EMIT_LONG (real ? va_arg (*pap, tom_long) : 0);
	  break;
	case TRT_TE_FLOAT:
	  addr = APPLY_ARGS_EMIT_FLOAT (real ? VA_ARG_FLOAT (*pap) : 0);
	  break;
	case TRT_TE_DOUBLE:
	  addr = APPLY_ARGS_EMIT_DOUBLE (real ? va_arg (*pap, tom_double) : 0);
	  break;
	case TRT_TE_POINTER:
	case TRT_TE_SELECTOR:
	case TRT_TE_REFERENCE:
	  addr = APPLY_ARGS_EMIT_REFERENCE (real ? va_arg (*pap, void *) : 0);
	  break;
	case TRT_TE_VOID:
	default:
	  /* This can not happen?  */
	  fatal ("argument type %d at %d to selector %s",
		 sel->in->args[i], i, sel->name.s);
	  goto bad;
	}

      inv->arg_pointers[INV_OFFSET_TO_ARGS + i] = addr;
      inv->arg_types[i] = stored_type;
    }

  INVOCATION_BUILD_RESULT ();
  APPLY_ARGS_COMPLETE ();
  inv->args_size = APPLY_ARGS_ACTUAL_SIZE ();
  APPLY_ARGS_END ();

  return inv;

 bad:
  trt_invocation_free (inv);
  return 0;
}

/* Add arguments to the incomplete invocation INV.  The arguments are the
   accessiable through the va_list pointed to by PAP; they are the
   arguments described by CMD, skipping over the first SKIP_IN arguments.
   If MUST_COMPLETE, the the invocation must be a complete (firable)
   invocation afterwards.  */
struct trt_invocation *
invocation_add_args (struct trt_invocation *inv, selector cmd,
		     int skip_in, va_list *pap, int must_complete)
{
  int i, num_args_provided = cmd->in->num - skip_in;
  selector sel = INV_SELECTOR (inv);

  if (num_args_provided == 1 && cmd->in->args[cmd->in->num - 1] == TRT_TE_VOID)
    num_args_provided--;

  if ((!must_complete && sel->in->num - inv->next_arg < cmd->in->num - skip_in)
      || (must_complete
	  && sel->in->num - inv->next_arg != cmd->in->num - skip_in))
    fatal (__FUNCTION__ "argument count mismatch: %s, %s",
	   cmd->name.s, sel->name.s);

  for (i = skip_in; i < cmd->in->num; i++)
    {
      void *addr = inv->arg_pointers[INV_OFFSET_TO_ARGS + inv->next_arg];

      if (sel->in->args[inv->next_arg] != cmd->in->args[i])
	fatal (__FUNCTION__ "argument type mismatch at %d: %s, %s",
	       i, cmd->name.s, sel->name.s);

      switch (cmd->in->args[i])
	{
	case TRT_TE_BOOLEAN:
	case TRT_TE_BYTE:
	  *(tom_int *) addr = VA_ARG_BYTE (*pap);
	  break;
	case TRT_TE_CHAR:
	  *(tom_int *) addr = VA_ARG_CHAR (*pap);
	  break;
	case TRT_TE_INT:
	  *(tom_int *) addr = va_arg (*pap, tom_int);
	  break;
	case TRT_TE_LONG:
	  *(tom_long *) addr = va_arg (*pap, tom_long);
	  break;
	case TRT_TE_FLOAT:
	  *(tom_float *) addr = VA_ARG_FLOAT (*pap);
	  break;
	case TRT_TE_DOUBLE:
	  *(tom_double *) addr = va_arg (*pap, tom_double);
	  break;
	case TRT_TE_POINTER:
	case TRT_TE_SELECTOR:
	case TRT_TE_REFERENCE:
	  *(void **) addr = va_arg (*pap, void *);
	  break;
	case TRT_TE_VOID:
	default:
	  /* This can not happen?  */
	  fatal (__FUNCTION__, "argument type %d at %d to selector %s",
		 cmd->in->args[i], i, sel->name.s);
	}
      inv->next_arg++;
    }

  return inv;
}

struct trt_invocation *
invocation_build_decode (struct trt_invocation *inv, tom_object coder)
{
  tom_object object = TRT_SEND (_PI_, coder, SEL (r_decode));
  selector sel = TRT_SEND (_PI_, coder, SEL (s_decode));
  int num_args_provided = TRT_SEND (_II_, coder, SEL (i_decode));
  union apply_args *args;
  int i, args_size;

  if (sel->in->num < num_args_provided)
    fatal ("invocation build argument count mismatch: %d, %s",
	   num_args_provided, sel->name.s);

  args_size = 2 * sizeof (void *) + APPLY_ARGS_REG_SIZE;
  args = xmalloc (args_size);

  APPLY_ARGS_START (sel);
  args->stack = xmalloc (APPLY_ARGS_STACK_SIZE (sel));
  APPLY_ARGS_CONTINUE ();
  BZERO (inv, sizeof (*inv));
  inv->args = args;
  inv->arg_types = xmalloc (sel->in->num);
  inv->stack = args->stack;
  inv->next_arg = num_args_provided;
  inv->arg_pointers = xmalloc ((2 + sel->in->num) * sizeof (void *));
  inv->arg_pointers[0] = APPLY_ARGS_EMIT_REFERENCE (object);
  inv->arg_pointers[1] = APPLY_ARGS_EMIT_SELECTOR (sel);

  for (i = 0; i < sel->in->num; i++)
    {
      int stored_type = sel->in->args[i];
      int real = i < num_args_provided;
      void *addr;

      switch (sel->in->args[i])
	{
	case TRT_TE_BOOLEAN:
	  addr = APPLY_ARGS_EMIT_BYTE
	    (!real ? 0 : TRT_SEND (_BI_, coder, SEL (o_decode)));
	  break;
	case TRT_TE_BYTE:
	  addr = APPLY_ARGS_EMIT_BYTE
	    (!real ? 0 : TRT_SEND (_BI_, coder, SEL (b_decode)));
	  break;
	case TRT_TE_CHAR:
	  addr = APPLY_ARGS_EMIT_CHAR
	    (!real ? 0 : TRT_SEND (_CI_, coder, SEL (c_decode)));
	  break;
	case TRT_TE_INT:
	  addr = APPLY_ARGS_EMIT_INT
	    (!real ? 0 : TRT_SEND (_II_, coder, SEL (i_decode)));
	  break;
	case TRT_TE_LONG:
	  addr = APPLY_ARGS_EMIT_LONG
	    (!real ? 0 : TRT_SEND (_LI_, coder, SEL (l_decode)));
	  break;
	case TRT_TE_FLOAT:
	  addr = APPLY_ARGS_EMIT_FLOAT
	    (!real ? 0 : TRT_SEND (_FI_, coder, SEL (f_decode)));
	  break;
	case TRT_TE_DOUBLE:
	  addr = APPLY_ARGS_EMIT_DOUBLE
	    (!real ? 0 : TRT_SEND (_DI_, coder, SEL (d_decode)));
	  break;
	case TRT_TE_SELECTOR:
	  addr = APPLY_ARGS_EMIT_REFERENCE
	    (!real ? 0 : TRT_SEND (_PI_, coder, SEL (s_decode)));
	  break;
	case TRT_TE_REFERENCE:
	  addr = APPLY_ARGS_EMIT_REFERENCE
	    (!real ? 0 : TRT_SEND (_PI_, coder, SEL (r_decode)));
	  break;
	case TRT_TE_POINTER:
	case TRT_TE_VOID:
	default:
	  /* This can not happen?  */
	  fatal ("argument type %d at %d to selector %s",
		 sel->in->args[i], i, sel->name.s);
	  goto bad;
	}

      inv->arg_pointers[INV_OFFSET_TO_ARGS + i] = addr;
      inv->arg_types[i] = stored_type;
    }

  INVOCATION_BUILD_RESULT ();
  APPLY_ARGS_COMPLETE ();
  inv->args_size = APPLY_ARGS_ACTUAL_SIZE ();
  APPLY_ARGS_END ();

  return inv;

 bad:
  trt_invocation_free (inv);
  return 0;
}

void
perform_args (builtin_return_type *return_result, int_imp imp,
	      tom_object object, selector sel, selector cmd,
	      int skip_in, va_list *pap)
{
  int args_size = 2 * sizeof (void *) + APPLY_ARGS_REG_SIZE;
  union apply_args *args = alloca (args_size);
  void *result;
  int i;
  builtin_return_type local_result;

  /* The expected return types must match, otherwise the caller expects
     something the callee can't give him.  */
  if (!trt_selector_args_match (cmd->out, sel->out))
    fatal ("perform return value mismatch: %s, %s",
	   cmd->name.s, sel->name.s);

  APPLY_ARGS_START (sel);
  args->stack = alloca (APPLY_ARGS_STACK_SIZE (sel));
  APPLY_ARGS_CONTINUE ();
  APPLY_ARGS_EMIT_REFERENCE (object);
  APPLY_ARGS_EMIT_SELECTOR (sel);

  if (sel->in->num != cmd->in->num - skip_in)
    fatal ("perform argument count mismatch: %s, %s",
	   cmd->name.s, sel->name.s);

  for (i = 0; i < sel->in->num; i++)
    if (sel->in->args[i] != cmd->in->args[i + skip_in])
      fatal ("perform argument type mismatch at %d: %s, %s",
	     i, cmd->name.s, sel->name.s);
    else switch (sel->in->args[i])
      {
	int stored_type;

      case TRT_TE_BOOLEAN:
      case TRT_TE_BYTE:
	APPLY_ARGS_EMIT_BYTE (VA_ARG_BYTE (*pap));
	break;
      case TRT_TE_CHAR:
	APPLY_ARGS_EMIT_CHAR (VA_ARG_CHAR (*pap));
	break;
      case TRT_TE_INT:
	APPLY_ARGS_EMIT_INT (va_arg (*pap, tom_int));
	break;
      case TRT_TE_LONG:
	APPLY_ARGS_EMIT_LONG (va_arg (*pap, tom_long));
	break;
      case TRT_TE_FLOAT:
	APPLY_ARGS_EMIT_FLOAT (VA_ARG_FLOAT (*pap));
	break;
      case TRT_TE_DOUBLE:
	APPLY_ARGS_EMIT_DOUBLE (va_arg (*pap, tom_double));
	break;
      case TRT_TE_POINTER:
      case TRT_TE_SELECTOR:
      case TRT_TE_REFERENCE:
	APPLY_ARGS_EMIT_REFERENCE (va_arg (*pap, void *));
	break;
      case TRT_TE_VOID:
      default:
	/* This can not happen?  */
	fatal ("argument type %d at %d to selector %s",
	       sel->in->args[i], i, sel->name.s);
	break;
      }

  if (sel->out->num > 1)
    {
      for (i = 1; i < cmd->out->num; i++)
	switch (sel->out->args[i])
	  {
	  case TRT_TE_VOID:
	    /* This can not happen?  */
	    fatal ("void return value type at %d to selector %s",
		   i, sel->name.s);
	    break;

	  default:
	    APPLY_ARGS_EMIT_POINTER (va_arg (*pap, void **));
	  }
    }

  APPLY_ARGS_COMPLETE ();

  result = APPLY_ARGS_APPLY ((void (*) ()) imp, args, 
			     APPLY_ARGS_ACTUAL_SIZE (), &local_result);
  memcpy (return_result, result, sizeof (*return_result));

  APPLY_ARGS_END ();
}

static void
perform_objects (builtin_return_type *return_result, int_imp imp,
		 tom_object object, selector sel, selector cmd,
		 tom_object argv, va_list *pap)
{
  int args_size = 2 * sizeof (void *) + APPLY_ARGS_REG_SIZE;
  union apply_args *args = alloca (args_size);
  void *result;
  int i, argc;
  builtin_return_type local_result;

  /* The expected return types must match, otherwise the caller expects
     something the callee can't give him.  */
  if (!trt_selector_args_match (cmd->out, sel->out))
    fatal ("perform return value mismatch: %s, %s", cmd->name.s, sel->name.s);

  APPLY_ARGS_START (sel);
  args->stack = alloca (APPLY_ARGS_STACK_SIZE (sel));
  APPLY_ARGS_CONTINUE ();
  APPLY_ARGS_EMIT_REFERENCE (object);
  APPLY_ARGS_EMIT_SELECTOR (sel);

  argc = argv ? TRT_SEND ((int_imp), argv, SEL (i_length)) : 0;
  if (argc != sel->in->num)
    {
      /* Wrong number of arguments.  */
      fatal ("wrong number of arguments %d to selector %s (should be %d)",
	     (int) argc, sel->name.s, sel->in->num);
    }

  if (sel->in->num)
    for (i = 0; i < sel->in->num; i++)
      switch (sel->in->args[i])
	{
	  int stored_type;

	case TRT_TE_BOOLEAN:
	case TRT_TE_BYTE:
	  APPLY_ARGS_EMIT_BYTE (TRT_SEND (_BI_, argv, SEL (b_at_i), i));
	  break;
	case TRT_TE_CHAR:
	  APPLY_ARGS_EMIT_CHAR (TRT_SEND (_CI_, argv, SEL (c_at_i), i));
	  break;
	case TRT_TE_INT:
	  APPLY_ARGS_EMIT_INT (TRT_SEND (_II_, argv, SEL (i_at_i), i));
	  break;
	case TRT_TE_LONG:
	  APPLY_ARGS_EMIT_LONG (TRT_SEND (_LI_, argv, SEL (l_at_i), i));
	  break;
	case TRT_TE_FLOAT:
	  APPLY_ARGS_EMIT_FLOAT (TRT_SEND (_FI_, argv, SEL (f_at_i), i));
	  break;
	case TRT_TE_DOUBLE:
	  APPLY_ARGS_EMIT_DOUBLE (TRT_SEND (_DI_, argv, SEL (d_at_i), i));
	  break;
	case TRT_TE_REFERENCE:
	  APPLY_ARGS_EMIT_REFERENCE (TRT_SEND (_PI_, argv, SEL (r_at_i), i));
	  break;
	case TRT_TE_VOID:
	case TRT_TE_POINTER:
	case TRT_TE_SELECTOR:
	default:
	  /* This can not happen?  */
	  fatal ("argument type %d at %d to selector %s",
		 sel->in->args[i], i, sel->name.s);
	  break;
	}

  if (sel->out->num > 1)
    {
      for (i = 1; i < cmd->out->num; i++)
	switch (sel->out->args[i])
	  {
	  case TRT_TE_VOID:
	    /* This can not happen?  */
	    fatal ("void return value type at %d to selector %s",
		   i, sel->name.s);
	    break;

	  default:
	    APPLY_ARGS_EMIT_POINTER (va_arg (*pap, void **));
	  }
    }

  APPLY_ARGS_COMPLETE ();

  result = APPLY_ARGS_APPLY ((void (*) ()) imp, args, 
			     APPLY_ARGS_ACTUAL_SIZE (), &local_result);
  memcpy (return_result, result, sizeof (*return_result));

  APPLY_ARGS_END ();
}

GENERIC_RETURN_TYPE
trt_forward (tom_object self, selector cmd, ...)
{
  tom_object delegate = TRT_SEND ((reference_imp), self,
				  SEL (r_forwardDelegate_s), cmd);
  builtin_return_type result;
  int_imp fwd_imp;
  va_list ap;

  va_start (ap, cmd);

  if (delegate != self)
    perform_args (&result, trt_lookup (delegate, cmd),
		  delegate, cmd, cmd, 0, &ap);
  else
    {
      tom_object inv_result;

      fwd_imp = trt_lookup (self, SEL (r_forwardSelector_s_arguments_p));
      if (fwd_imp != (int_imp) trt_forward)
	inv_result = ((reference_imp) fwd_imp)
			(self, SEL (r_forwardSelector_s_arguments_p), cmd, &ap);
      else
	{
	  struct trt_invocation *inv = xmalloc (sizeof (*inv));
	  tom_object r = 0;

	  inv = invocation_build_args (inv, self, cmd, cmd, 0, &ap);
	  r = TRT_SEND (_PI_, CREF (tom_Invocation), SEL (r_alloc));
	  r = TRT_SEND (_PI_, r, SEL (r_init_p), inv);
	  inv_result = TRT_SEND (_PI_, self, SEL (r_forwardInvocation_r), r);
	}

      if (inv_result)
	TRT_SEND (, inv_result, SEL (v_setReturnValues__pp__forSelector_s),
		  &result, &ap, cmd);
    }

  va_end (ap);
  APPLY_ARGS_RETURN (&result);
}

static int_imp
get_imp (struct trtd_extension *x, selector sel)
{
  struct trtd_method *m = trt_extension_get_imp (x, sel);

  if (!m)
    {
      tom_object c = TRT_SEND
	((reference_imp), _mr_c_tom_SelectorCondition,
	 SEL (r_for_r_class_r_message_r_selector_s), x->extension_object,
	 c_tom_Conditions_unrecognized_selector,
	 byte_string_with_c_string ("method not implemented by this extension"),
	 sel);
      TRT_SEND ((reference_imp), c, SEL (v_raise));
    }

  return m->imp;
}

GENERIC_RETURN_TYPE
i_tom_All_x_perform_s_with_x
  (tom_object self, selector cmd, selector sel, ...)
{
  int_imp imp = trt_lookup ((struct trt_instance *) self, sel);
  builtin_return_type result;
  va_list ap;

  va_start (ap, sel);
  perform_args (&result, imp, self, sel, cmd, 1, &ap);
  va_end (ap);

  APPLY_ARGS_RETURN (&result);
}

GENERIC_RETURN_TYPE
i_tom_All_x_perform_s___r
  (tom_object self, selector cmd, selector sel, tom_object argv, ...)
{
  int_imp imp = trt_lookup ((struct trt_instance *) self, sel);
  builtin_return_type result;
  va_list ap;

  va_start (ap, argv);
  perform_objects (&result, imp, self, sel, cmd, argv, &ap);
  va_end (ap);

  APPLY_ARGS_RETURN (&result);
}

GENERIC_RETURN_TYPE
i_tom_Extension_x_perform_s_on_r_with_x
  (tom_object self, selector cmd,
   selector sel, tom_object object, ...)
{
  struct _es_i_tom_Extension *this
    = trt_ext_address (self, _ei_i_tom_Extension);
  int_imp m = get_imp (this->rti, sel);
  builtin_return_type result;
  va_list ap;

  va_start (ap, object);
  perform_args (&result, m, object, sel, cmd, 2, &ap);
  va_end (ap);

  APPLY_ARGS_RETURN (&result);
}

GENERIC_RETURN_TYPE
i_tom_Extension_x_perform_s_on_r___r
  (tom_object self, selector cmd,
   selector sel, tom_object object, tom_object args, ...)
{
  struct _es_i_tom_Extension *this
    = trt_ext_address (self, _ei_i_tom_Extension);
  int_imp m = get_imp (this->rti, sel);
  builtin_return_type result;
  va_list ap;

  va_start (ap, args);
  perform_objects (&result, m, object, sel, cmd, args, &ap);
  va_end (ap);

  APPLY_ARGS_RETURN (&result);
}

/********** invocations **********/

void
trt_invocation_free (struct trt_invocation *inv)
{
  xfree (inv->arg_pointers);
  xfree (inv->stack);
  xfree (inv->args);
  xfree (inv->arg_types);
  if (inv->result)
    trt_invocation_result_free (inv->result);
  xfree (inv);
}

void
trt_invocation_result_free (struct trt_invocation_result *result)
{
  xfree (result->mrv);
  xfree (result->offsets);
  xfree (result);
}

struct trt_invocation_result *
trt_invocation_result_copy (struct trt_invocation_result *in)
{
  struct trt_invocation_result *out = xmalloc (sizeof (*out));

  memcpy (out, in, sizeof (*in));
  if (out->desc->num > 1)
    {
      int n = out->desc->num;

      out->offsets = memcpy (xmalloc (n * sizeof (*out->offsets)),
			     in->offsets, n * sizeof (*out->offsets));
      out->mrv = memcpy (xmalloc (out->size), in->mrv, in->size);
    }

  return out;
}

static void
invocation_result_mark_elements (struct trt_invocation_result *res)
{
  if (res->desc->num > 0)
    {
      int i;

      if (res->desc->args[0] == TRT_TE_REFERENCE)
	mark_if_needed (res->cret.a);

      for (i = 1; i < res->desc->num; i++)
	if (res->desc->args[i] == TRT_TE_REFERENCE)
	  mark_if_needed (*(void **) ((char *) res->mrv + res->offsets[i]));
    }
}

static void
invocation_mark_elements (struct trt_invocation *inv)
{
  selector cmd = INV_SELECTOR (inv);
  int i;

  mark_if_needed (INV_RECEIVER (inv));

  for (i = 0; i < cmd->in->num; i++)
    if (cmd->in->args[i] == TRT_TE_REFERENCE)
      mark_if_needed (*(void **) inv->arg_pointers[INV_OFFSET_TO_ARGS + i]);

  invocation_result_mark_elements (inv->result);
}

tom_object
invocation_fire (tom_object self, selector cmd, struct trt_invocation *inv,
		 int check_complete, int no_result)
{
  builtin_return_type *result;
  int_imp imp;
  builtin_return_type local_result;  

  if (check_complete && INV_SELECTOR (inv)->in->num != inv->next_arg)
    trt_raise (0, self, cmd, c_tom_Conditions_program_condition,
	       "Invocation incomplete: %d args provided, %d needed",
	       inv->next_arg, INV_SELECTOR (inv)->in->num);

  imp = trt_lookup (INV_RECEIVER (inv), INV_SELECTOR (inv));
  result = APPLY_ARGS_APPLY ((void (*) ()) imp, inv->args, inv->args_size,
			     &local_result);

  if (inv->result->desc->num > 0)
    switch (inv->result->desc->args[0])
      {
      case TRT_TE_BOOLEAN:
      case TRT_TE_BYTE:
      case TRT_TE_CHAR:
      case TRT_TE_INT:
	inv->result->cret.i = result->i.i;
	break;
      case TRT_TE_LONG:
	inv->result->cret.l = result->l.l;
	break;
      case TRT_TE_FLOAT:
	inv->result->cret.f = RETURN_RETRIEVE_FLOAT (result);
	break;
      case TRT_TE_DOUBLE:
	inv->result->cret.d = RETURN_RETRIEVE_DOUBLE (result);
	break;
      case TRT_TE_POINTER:
      case TRT_TE_SELECTOR:
      case TRT_TE_REFERENCE:
	inv->result->cret.a = result->p.p;
	break;
      default:
	break;
      }

  return (no_result ? NULL
	  : TRT_SEND (_PI_, CREF (tom_InvocationResult), SEL (r_with_p),
		      trt_invocation_result_copy (inv->result)));
}

static tom_object
invocation_args  (struct trt_invocation **pinv, tom_object self, selector cmd,
		  selector sel, tom_object target, va_list *pap)
{
  struct trt_invocation *inv;
  tom_object r = 0;

  inv = xmalloc (sizeof (*inv));
  inv = invocation_build_args (inv, target, sel, cmd, 2, pap);

  if (pinv)
    *pinv = inv;

  if (inv)
    {
      r = TRT_SEND (_PI_, self, SEL (r_alloc));
      r = TRT_SEND (_PI_, r, SEL (r_init_p), inv);
    }

  return r;
}

tom_object
c_tom_Invocation_r_for_s_to__r_with_x (tom_object self, selector cmd,
				       selector sel, tom_object target, ...)
{
  struct trt_invocation *inv;
  tom_object r;
  va_list ap;

  va_start (ap, target);
  r = invocation_args (&inv, self, cmd, sel, target, &ap);
  va_end (ap);

  return r;
}

tom_object
c_tom_Invocation_r_of_s_to__r_with_x (tom_object self, selector cmd,
				      selector sel, tom_object target, ...)
{
  struct trt_invocation *inv;
  tom_object r;
  va_list ap;

  va_start (ap, target);
  r = invocation_args (&inv, self, cmd, sel, target, &ap);
  va_end (ap);

  if (INV_SELECTOR (inv)->in->num != inv->next_arg)
    trt_raise (0, self, cmd, c_tom_Conditions_program_condition,
	       "Invocation incomplete: %d args provided, %d needed",
	       inv->next_arg, INV_SELECTOR (inv)->in->num);

  return r;
}

tom_object
c_tom_Invocation_r_of_s_to__r_using_p (tom_object self, selector cmd,
				       selector sel, tom_object target,
				       va_list *pap)
{
  return invocation_args (NULL, self, 0, sel, target, pap);
}

static tom_object
invocation_obj (tom_object self, selector cmd, selector sel,
		tom_object target, tom_object args, int must_complete)
{
  struct trt_invocation *inv;
  tom_object r = 0;

  if (args)
      unimplemented (__FUNCTION__ " with non-void args");

  inv = xmalloc (sizeof (*inv));
  inv = invocation_build_args (inv, target, sel, cmd, 3, NULL);

  if (inv)
    {
      r = TRT_SEND (_PI_, self, SEL (r_alloc));
      r = TRT_SEND (_PI_, r, SEL (r_init_p), inv);
    }

  return r;
}

tom_object
c_tom_Invocation_r_for_s_to__r___r (tom_object self, selector cmd, selector sel,
				    tom_object target, tom_object args)
{
  return invocation_obj (self, cmd, sel, target, args, 0);
}

tom_object
c_tom_Invocation_r_of_s_to__r___r (tom_object self, selector cmd,
				   selector sel, tom_object target,
				   tom_object args)
{
  return invocation_obj (self, cmd, sel, target, args, 1);
}

tom_byte
i_tom_Invocation_o_isComplete (tom_object self, selector cmd)
{
  struct _es_i_tom_Invocation *this
    = trt_ext_address (self, _ei_i_tom_Invocation);
  struct trt_invocation *inv = this->invocation;

  return INV_SELECTOR (inv)->in->num == inv->next_arg;
}

void
i_tom_Invocation_v_decodeFromCoder_r (tom_object self, selector cmd,
				      tom_object coder)
{
  struct _es_i_tom_Invocation *this
    = trt_ext_address (self, _ei_i_tom_Invocation);
  struct trt_invocation *inv;

  inv = xmalloc (sizeof (*inv));
  this->invocation = invocation_build_decode (inv, coder);
}

void
i_tom_Invocation_v_encodeToCoder_r (tom_object self, selector cmd,
				    tom_object coder)
{
  struct _es_i_tom_Invocation *this
    = trt_ext_address (self, _ei_i_tom_Invocation);
  struct trt_invocation *inv = this->invocation;
  int i;

  TRT_SEND (, coder, SEL (v_encode_r), INV_RECEIVER (inv));
  TRT_SEND (, coder, SEL (v_encode_s), INV_SELECTOR (inv));
  TRT_SEND (, coder, SEL (v_encode_i), (tom_int) inv->next_arg);

  for (i = 0; i < inv->next_arg; i++)
    {
      void *addr = inv->arg_pointers[INV_OFFSET_TO_ARGS + i];

      switch (INV_SELECTOR (inv)->in->args[i])
	{
	case TRT_TE_BOOLEAN:
	  TRT_SEND (, coder, SEL (v_encode_o), *(tom_byte *) addr);
	  break;
	case TRT_TE_BYTE:
	  TRT_SEND (, coder, SEL (v_encode_b), *(tom_byte *) addr);
	  break;
	case TRT_TE_CHAR:
	  TRT_SEND (, coder, SEL (v_encode_c), *(tom_char *) addr);
	  break;
	case TRT_TE_INT:
	  TRT_SEND (, coder, SEL (v_encode_i), *(tom_int *) addr);
	  break;
	case TRT_TE_LONG:
	  TRT_SEND (, coder, SEL (v_encode_l), *(tom_long *) addr);
	  break;
	case TRT_TE_FLOAT:
	  TRT_SEND (, coder, SEL (v_encode_f), *(tom_float *) addr);
	  break;
	case TRT_TE_DOUBLE:
	  TRT_SEND (, coder, SEL (v_encode_d), *(tom_double *) addr);
	  break;
	case TRT_TE_SELECTOR:
	  TRT_SEND (, coder, SEL (v_encode_s), *(selector *) addr);
	  break;
	case TRT_TE_REFERENCE:
	  TRT_SEND (, coder, SEL (v_encode_r), *(tom_object *) addr);
	  break;
	default:
	  trt_raise (0, self, cmd, c_tom_Conditions_program_condition,
		     "unencodable type at %d in %s",
		     i, INV_SELECTOR (inv)->name.s);
	}
    }
}

void
i_tom_Invocation_v_dealloc (tom_object self, selector cmd)
{
  struct _es_i_tom_Invocation *this
    = trt_ext_address (self, _ei_i_tom_Invocation);
  trt_invocation_free (this->invocation);
}

tom_object
i_tom_Invocation_r_fire (tom_object self, selector cmd)
{
  struct _es_i_tom_Invocation *this
    = trt_ext_address (self, _ei_i_tom_Invocation);
  void *result = invocation_fire (self, cmd, this->invocation, 1, 0);

  this->result = result;
  return result;
}

void
i_tom_Invocation_v_fire (tom_object self, selector cmd)
{
  struct _es_i_tom_Invocation *this
    = trt_ext_address (self, _ei_i_tom_Invocation);

  invocation_fire (self, cmd, this->invocation, 1, 1);
  this->result = NULL;
}

tom_object
i_tom_Invocation_r_fireAt_r (tom_object self, selector cmd, tom_object t)
{
  struct _es_i_tom_Invocation *this
    = trt_ext_address (self, _ei_i_tom_Invocation);
  struct trt_invocation *inv = this->invocation;

  INV_RECEIVER (inv) = (void *) t;
  return TRT_SEND (_PI_, self, SEL (r_fire));
}

void
i_tom_Invocation_v_fireAt_r (tom_object self, selector cmd, tom_object t)
{
  struct _es_i_tom_Invocation *this
    = trt_ext_address (self, _ei_i_tom_Invocation);
  struct trt_invocation *inv = this->invocation;

  INV_RECEIVER (inv) = (void *) t;
  TRT_SEND (_PI_, self, SEL (v_fire));
}

tom_object
i_tom_Invocation_r_forwardSelector_s_arguments_p (tom_object self, selector cmd,
						  selector sel, va_list *pap)
{
  struct _es_i_tom_Invocation *this
    = trt_ext_address (self, _ei_i_tom_Invocation);
  struct trt_invocation *inv = this->invocation;
  selector my_sel = INV_SELECTOR (inv);
  int in_next_arg = inv->next_arg;
  char *s = sel->name.s;
  int l = 0;

  /* Check that the selector completes our selector.  */
  while (l++, *s++ != '_');
  if (memcmp (s, my_sel->name.s + my_sel->name.len - (sel->name.len - l),
	      sel->name.len - l))
    fatal (__FUNCTION__ "%s does not complete %s", sel->name.s, my_sel->name.s);
  
  inv = invocation_add_args (inv, sel, 0, pap, 1);

  /* Restore the incompleteness.  */
  inv->next_arg = in_next_arg;

  return invocation_fire (self, sel, inv, 0, 0);
}

static tom_object
do_fire_with (tom_object self, selector cmd, int no_result, va_list ap)
{
  struct _es_i_tom_Invocation *this
    = trt_ext_address (self, _ei_i_tom_Invocation);
  struct trt_invocation *inv = this->invocation;
  int in_next_arg = inv->next_arg;

  invocation_add_args (inv, cmd, 0, &ap, 1);

  /* Restore the incompleteness.  */
  inv->next_arg = in_next_arg;

  return invocation_fire (self, cmd, inv, 0, no_result);
}

tom_object
i_tom_Invocation_r_fireWith_x (tom_object self, selector cmd, ...)
{
  tom_object r;
  va_list ap;

  va_start (ap, cmd);
  r = do_fire_with (self, cmd, 0, ap);
  va_end (ap);

  return r;
}

void
i_tom_Invocation_v_fireWith_x (tom_object self, selector cmd, ...)
{
  va_list ap;

  va_start (ap, cmd);
  do_fire_with (self, cmd, 1, ap);
  va_end (ap);
}

void
i_tom_Invocation_v_gc_mark_elements (tom_object self, selector cmd)
{
  struct _es_i_tom_Invocation *this
    = trt_ext_address (self, _ei_i_tom_Invocation);

  if (this->invocation)
    invocation_mark_elements (this->invocation);
}

selector
i_tom_Invocation_s_selector (tom_object self, selector cmd)
{
  struct _es_i_tom_Invocation *this
    = trt_ext_address (self, _ei_i_tom_Invocation);
  struct trt_invocation *inv = this->invocation;

  return INV_SELECTOR (inv);
}

tom_object
i_tom_Invocation_r_target (tom_object self, selector cmd)
{
  struct _es_i_tom_Invocation *this
    = trt_ext_address (self, _ei_i_tom_Invocation);
  struct trt_invocation *inv = this->invocation;

  return INV_RECEIVER (inv);
}

tom_object
i_tom_Invocation_r_resultTypeDescription (tom_object self, selector cmd)
{
  struct _es_i_tom_Invocation *this
    = trt_ext_address (self, _ei_i_tom_Invocation);
  struct trt_invocation *inv = this->invocation;

  return TRT_SEND (_PI_, CREF (tom_TypeDescription),
		   SEL (r_for_p), inv->result->desc);
}

void
i_tom_InvocationResult_v_dealloc (tom_object self, selector cmd)
{
  struct _es_i_tom_InvocationResult *this
    = trt_ext_address (self, _ei_i_tom_InvocationResult);

  if (this->values)
    trt_invocation_result_free (this->values);
}

void
i_tom_InvocationResult_v_gc_mark_elements (tom_object self, selector cmd)
{
  struct _es_i_tom_InvocationResult *this
    = trt_ext_address (self, _ei_i_tom_InvocationResult);

  if (this->values)
    invocation_result_mark_elements (this->values);
}

void
i_tom_InvocationResult_v_decodeFromCoder_r (tom_object self, selector cmd,
					    tom_object coder)
{
  abort ();
}

void
i_tom_InvocationResult_v_encodeToCoder_r (tom_object self, selector cmd,
					  tom_object coder)
{
  if (!TRT_SEND (_BI_, coder, SEL (o_hasBeenCodedFor_r),
		 _mr_c_tom_InvocationResult))
    {
      DECL_SR (i_tom_InvocationResult, i_tom_State);

      trt_lookup_super (SR (i_tom_InvocationResult, i_tom_State),
			SEL (v_encodeUsingCoder_r)) (self, cmd, coder);

      {
	struct _es_i_tom_InvocationResult *this
	  = trt_ext_address (self, _ei_i_tom_InvocationResult);
	struct trt_invocation_result *res = this->values;
	struct trtd_selector_args *args = res->desc;

	args = args;
	abort ();
      }
    }
}

tom_object
i_tom_InvocationResult_r_typeDescription (tom_object self, selector cmd)
{
  struct _es_i_tom_InvocationResult *this
    = trt_ext_address (self, _ei_i_tom_InvocationResult);
  struct trt_invocation_result *res = this->values;

  return TRT_SEND (_PI_, CREF (tom_TypeDescription), SEL (r_for_p), res->desc);
}

static void
trt_inv_result_set_values (struct trt_invocation_result *res, tom_object self,
			   selector cmd, builtin_return_type *r, va_list *pap)
{
  int i;

  if (!trt_selector_args_match (res->desc, cmd->out))
    trt_raise (0, self, cmd, c_tom_Conditions_program_condition,
	       "actual return values mismatch expected");

  BZERO (r, sizeof (*r));

  for (i = 0; i < cmd->out->num; i++)
    {
      void *addr = !i ? 0 : (char *) res->mrv + res->offsets[i];

      switch (cmd->out->args[i])
	{
	case TRT_TE_BOOLEAN:
	case TRT_TE_BYTE:
	  if (i)
	    *va_arg (*pap, tom_byte *) = *(tom_byte *) addr;
	  else
	    r->i.i = res->cret.i;
	  break;

	case TRT_TE_CHAR:
	  if (i)
	    *va_arg (*pap, tom_char *) = *(tom_char *) addr;
	  else
	    r->i.i = res->cret.i;
	  break;

	case TRT_TE_INT:
	  if (i)
	    *va_arg (*pap, tom_int *) = *(tom_int *) addr;
	  else
	    r->i.i = res->cret.i;
	  break;

	case TRT_TE_LONG:
	  if (i)
	    *va_arg (*pap, tom_long *) = *(tom_long *) addr;
	  else
	    r->l.l = res->cret.l;
	  break;

	case TRT_TE_FLOAT:
	  if (i)
	    *va_arg (*pap, tom_float *) = *(tom_float *) addr;
	  else
	    RETURN_SET_FLOAT (r, res->cret.f);
	  break;

	case TRT_TE_DOUBLE:
	  if (i)
	    *va_arg (*pap, tom_double *) = *(tom_double *) addr;
	  else
	    RETURN_SET_DOUBLE (r, res->cret.d);
	  break;

	case TRT_TE_POINTER:
	case TRT_TE_SELECTOR:
	case TRT_TE_REFERENCE:
	  if (i)
	    *va_arg (*pap, void **) = *(void **) addr;
	  else
	    r->p.p = res->cret.a;
	  break;

	default:
	  ABORT ();
	}
    }
}

void
i_tom_InvocationResult_v_setReturnValues__pp__forSelector_s
  (tom_object self, selector cmd,
   builtin_return_type *result, va_list *pap, selector sel)
{
  struct _es_i_tom_InvocationResult *this
    = trt_ext_address (self, _ei_i_tom_InvocationResult);
  struct trt_invocation_result *res = this->values;

  trt_inv_result_set_values (res, self, sel, result, pap);
}

GENERIC_RETURN_TYPE
i_tom_InvocationResult_x_components (tom_object self, selector cmd, ...)
{
  struct _es_i_tom_InvocationResult *this
    = trt_ext_address (self, _ei_i_tom_InvocationResult);
  struct trt_invocation_result *res = this->values;
  builtin_return_type r;
  va_list ap;

  va_start (ap, cmd);
  trt_inv_result_set_values (res, self, cmd, &r, &ap);
  va_end (ap);

  APPLY_ARGS_RETURN (&r);
}

GENERIC_RETURN_TYPE
i_tom_InvocationResult_x_component_i (tom_object self, selector cmd,
				      tom_int n, ...)
{
  builtin_return_type r, r1;
  struct _es_i_tom_InvocationResult *this
    = trt_ext_address (self, _ei_i_tom_InvocationResult);
  struct trt_invocation_result *res = this->values;

  if (n > res->desc->num)
    trt_raise (0, self, cmd, c_tom_Conditions_program_condition,
	       "bad index %d", n);

  if (cmd->out->num != 1)
    trt_raise (0, self, cmd, c_tom_Conditions_program_condition,
	       "invalid expected return type");

  BZERO (&r, sizeof (r));
  BZERO (&r1, sizeof (r1));
  if (cmd->out->args[0] == res->desc->args[n]
      || (cmd->out->args[0] == TRT_TE_REFERENCE
	  && res->desc->args[n] != TRT_TE_POINTER))
    {
      switch (res->desc->args[n])
	{
	case TRT_TE_BOOLEAN:
	case TRT_TE_BYTE:
	  r.i.i = (!n ? res->cret.i
		   : *(tom_byte *) ((char *) res->mrv + res->offsets[n]));
	  break;
	case TRT_TE_CHAR:
	  r.i.i = (!n ? res->cret.i
		   : *(tom_char *) ((char *) res->mrv + res->offsets[n]));
	  break;
	case TRT_TE_INT:
	  r.i.i = (!n ? res->cret.i
		   : *(tom_int *) ((char *) res->mrv + res->offsets[n]));
	  break;
	case TRT_TE_LONG:
	  r.l.l = (!n ? res->cret.l
		   : *(tom_long *) ((char *) res->mrv + res->offsets[n]));
	  break;
	case TRT_TE_FLOAT:
	  RETURN_SET_FLOAT (&r, (!n ? res->cret.f
				 : *(tom_float *) ((char *) res->mrv
						   + res->offsets[n])));
	  break;
	case TRT_TE_DOUBLE:
	  RETURN_SET_DOUBLE (&r, (!n ? res->cret.d
				  : *(tom_double *) ((char *) res->mrv
						     + res->offsets[n])));
	  break;
	case TRT_TE_POINTER:
	case TRT_TE_SELECTOR:
	case TRT_TE_REFERENCE:
	  r.p.p = (!n ? res->cret.a
		   : *(void **) ((char *) res->mrv + res->offsets[n]));
	  break;
	default:
	  ABORT ();
	}

      if (cmd->out->args[0] != res->desc->args[n])
	if (cmd->out->args[0] == TRT_TE_REFERENCE)
	  {
	    switch (res->desc->args[n])
	      {
	      case TRT_TE_BOOLEAN:
	      case TRT_TE_BYTE:
	        r1.p.p = TRT_SEND (_PI_, CREF (tom_ByteNumber), SEL (r_with_b),
				   r.i.i);
	        break;
	      case TRT_TE_CHAR:
		r1.p.p = TRT_SEND (_PI_, CREF (tom_CharNumber), SEL (r_with_c),
				   r.i.i);
		break;
	      case TRT_TE_INT:
		r1.p.p = TRT_SEND (_PI_, CREF (tom_IntNumber), SEL (r_with_i),
				   r.i.i);
		break;
	      case TRT_TE_LONG:
		r1.p.p = TRT_SEND (_PI_, CREF (tom_LongNumber), SEL (r_with_l),
				   r.l.l);
		break;
	      case TRT_TE_FLOAT:
		r1.p.p = TRT_SEND ((void *(*)(void *, void *, tom_float)),
				   CREF (tom_FloatNumber), SEL (r_with_f),
				   RETURN_RETRIEVE_FLOAT (&r));
		break;
	      case TRT_TE_DOUBLE:
		r1.p.p = TRT_SEND ((void *(*)(void *, void *, tom_double)),
				   CREF (tom_DoubleNumber), SEL (r_with_d),
				   RETURN_RETRIEVE_DOUBLE (&r));
		break;
	      case TRT_TE_SELECTOR:
		r1.p.p = TRT_SEND (_PI_, CREF (tom_Selector), SEL (r_with_s),
				   r.p.p);
		break;
	      default:
		ABORT ();
	      }
	    memcpy(&r, &r1, sizeof(r));
	  }
	else
	  ABORT ();

      APPLY_ARGS_RETURN (&r);
    }
  else
    trt_raise (0, self, cmd, c_tom_Conditions_program_condition,
	       "type mismatch expected=%d, actual=%d",
	       (int) cmd->out->args[0], (int) res->desc->args[n]);

  /* Not reached.  */
}

tom_byte
i_tom_TypeDescription_o_equal_r (tom_object self, selector cmd,
				 tom_object other)
{
  struct _es_i_tom_TypeDescription *this
    = trt_ext_address (self, _ei_i_tom_TypeDescription);
  void *a = TRT_SEND (_PI_, other, SEL (p_types_description));

  return trt_selector_args_match (a, this->types_description);
}

tom_int
i_tom_TypeDescription_i_length (tom_object self, selector cmd)
{
  struct _es_i_tom_TypeDescription *this
    = trt_ext_address (self, _ei_i_tom_TypeDescription);
  struct trtd_selector_args *d = this->types_description;

  return d->num;
}

tom_int
i_tom_TypeDescription_i_component_i (tom_object self, selector cmd, tom_int n)
{
  struct _es_i_tom_TypeDescription *this
    = trt_ext_address (self, _ei_i_tom_TypeDescription);
  struct trtd_selector_args *d = this->types_description;

  if (n >= d->num)
    trt_raise (0, self, cmd, c_tom_Conditions_program_condition,
	       "bad index %d", (int) n);

  return d->args[n];
}
