/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: thread.c,v 1.29 1999/01/05 11:56:41 tiggr Exp $  */

#include "trt.h"
#include <tom/tom-r.h>
#include <stdarg.h>
#include <stdio.h>
#include <sys/time.h>

/* The number of threads.  */
int trt_thread_num;

/* The capacity of the threads arrays.  */
int trt_thread_cap;

/* The threads, indexed on their thread_id.  */
trt_thread_info *trt_threads;

/* The size of each thread's local data.  */
int trt_thread_local_size;

/* The capacity of each thread's local data block.  This will only unequal
   the size iff we're dynamically loading.  */
int trt_thread_local_cap;

/* The mega lock on modifying the runtime structure, if !0  */
trt_rlock_t trt_runtime_lock;

/* The lock on the object allocation structures.  */
trt_rlock_t trt_alloc_lock;

/* The lock on the gray stack.  */
trt_lock_t trt_gray_lock;

/* Structure to record information for delayed freeing.  */
struct free_delayed
{
  /* The next struct.  */
  struct free_delayed *next;

  /* The data to be freed.  */
  void *data;

  /* Timestamp.  */
  struct timeval stamp;
};

/* Head and tail of the delayed free list.  */
static struct free_delayed *delay_head;
static struct free_delayed **delay_tail = &delay_head;

/* Delayed free.  The runtime must be locked.  */
void
safe_free (void *p)
{
  if (p)
    {
      struct free_delayed *f = xmalloc (sizeof (*f));

      f->data = p;
      gettimeofday (&f->stamp, 0);
      f->next = 0;
      *delay_tail = f;
      delay_tail = &f->next;
    }
}

/* Safe realloc, by malloc and delayed free.  The runtime must be locked.  */
void *
safe_realloc (void *p, int old_size, int new_size)
{
  if (!p)
    return xrealloc (p, new_size);
  if (!new_size)
    {
      safe_free (p);
      return 0;
    }
  {
    void *q = xmalloc (new_size);

    memcpy (q, p, old_size);
    safe_free (p);

    return q;
  }
}

void
trt_thread_pre_init (void *bottom_of_stack)
{
  char *data;

  if (!trt_thread_mach_init ())
    ABORT ();

  /* If we're condemned by the static resolver, set the capacity.  */
  if (!trt_thread_local_cap)
    trt_thread_local_cap = trt_thread_local_size;

  data = trt_init_thread_data;
  trt_thread_set_local_data (data);
  SET_THREAD_LOCAL (data, tom_int, c_tom_Thread_current_id, 0);

  trt_thread_num = 1;
  trt_thread_cap = THREADS_PACKAGE == THREADS_NO_THREADS ? 1 : 8;
  trt_threads = xcalloc (trt_thread_cap, sizeof (*trt_threads));

  trt_threads[0].data = data;
#if !STACK_REF_STRUCT
  trt_threads[0].bottom = bottom_of_stack;
  trt_threads[0].top = &bottom_of_stack;
#endif
}

void
trt_thread_post_init (void)
{
  char *data = trt_thread_get_local_data ();
  tom_object oth;

  c_tom_Thread_threads = TRT_SEND ((reference_imp), _mr_c_tom_MutableEqSet,
				   SEL (r_new));

  oth = TRT_SEND ((reference_imp), _mr_c_tom_Thread, SEL (r_alloc));
  oth = TRT_SEND ((reference_imp), oth, SEL (r_init_i), 0);
  TRT_SEND (, c_tom_Thread_threads, SEL (v_add_r), oth);

  SET_THREAD_LOCAL (data, tom_object, c_tom_Thread_current, oth);
  trt_threads[0].object = oth;
}

/* Initialize this thread.  Not to be used by the main thread.  */
static void
trt_thread_init_thread (trt_thread_t th_id)
{
  char *data;

  trt_rlock_lock (trt_runtime_lock);

  data = xcalloc (1, trt_thread_local_cap);
  trt_thread_set_local_data (data);

  trt_threads[th_id].data = data;
#if !STACK_REF_STRUCT
  trt_threads[th_id].bottom = &data;
  trt_threads[th_id].top = &data;
#endif

  SET_THREAD_LOCAL (data, tom_object, c_tom_Thread_current,
		    trt_threads[th_id].object);
  SET_THREAD_LOCAL (data, tom_int, c_tom_Thread_current_id, th_id);

  trt_rlock_unlock (trt_runtime_lock);
}

void
trt_thread_main (struct trt_thread_invocation *inv)
{
  tom_object th;

  trt_thread_init_thread (inv->th_id);

  invocation_fire (INV_RECEIVER (&inv->inv), INV_SELECTOR (&inv->inv),
		   &inv->inv, 1, 1);

  /* XXX Should post a thread-exit notification with the result.  */
  trt_invocation_free (&inv->inv);

  th = TRT_SEND ((reference_imp), _mr_c_tom_Thread, SEL (r_current));
  TRT_SEND (, th, SEL (v_exit_i), 0);
}

trt_thread_t
trt_thread_self (void)
{
  return GET_THREAD_LOCAL (trt_thread_get_local_data (),
			   tom_int, c_tom_Thread_current_id);
}

tom_object
i_tom_All_r_performInThread_s_with_x (tom_object self, selector cmd,
				      selector sel, ...)
{
  struct trt_thread_invocation *inv;
  tom_object oth = 0;
  trt_thread_t th;
  tom_int th_id;
  va_list ap;

  if (!trt_runtime_lock)
    {
      trt_runtime_lock = trt_rlock_create ();
      trt_alloc_lock = trt_rlock_create ();
      trt_gray_lock = trt_lock_create ();
    }

  trt_rlock_lock (trt_runtime_lock);

  inv = xmalloc (sizeof (*inv));
  va_start (ap, sel);
  invocation_build_args (&inv->inv, self, sel, cmd, 1, &ap);
  va_end (ap);

  /* Find an id for this thread.  */
  if (trt_thread_num == trt_thread_cap)
    {
      int n = trt_thread_cap * sizeof (*trt_threads);

      trt_threads = safe_realloc (trt_threads, n, 2 * n);
      th_id = trt_thread_cap;
      trt_thread_cap *= 2;
    }
  else
    {
      for (th_id = 0; th_id < trt_thread_cap && trt_threads[th_id].object;
	   th_id++);
      if (th_id == trt_thread_cap)
	ABORT ();
    }

  inv->th_id = th_id;
  th = trt_thread_create (th_id, inv);
  if (!th)
    {
      trt_invocation_free (&inv->inv);
      trt_raise (0, self, cmd, c_tom_Conditions_error,
		 "thread creation failed");
    }
  else
    {
      oth = TRT_SEND ((reference_imp), _mr_c_tom_Thread, SEL (r_alloc));
      oth = TRT_SEND ((reference_imp), oth, SEL (r_init_i), th_id);

      TRT_SEND (, c_tom_Thread_threads, SEL (v_add_r), oth);
      trt_threads[th_id].object = oth;
      trt_thread_num++;
    }

  trt_rlock_unlock (trt_runtime_lock);

  return oth;
}

tom_byte
c_tom_Thread_o_functioning (tom_object self, selector cmd)
{
  return THREADS_PACKAGE == THREADS_NO_THREADS ? 0 : 1;
}

void
i_tom_Thread_v_exit_i (tom_object self, selector cmd, tom_int rc)
{
  struct _es_i_tom_Thread *this = trt_ext_address (self, _ei_i_tom_Thread);
  trt_thread_t th = this->thread_id;
  void *data;

  if (th != trt_thread_self ())
    ABORT ();

  trt_rlock_lock (trt_runtime_lock);

  if (trt_thread_num == 1)
    {
      trt_rlock_unlock (trt_runtime_lock);
      TRT_SEND (, _mr_c_tom_Runtime, SEL (v_exit_i), rc);
    }

  BZERO (trt_threads + this->thread_id, sizeof (*trt_threads));
  trt_thread_num--;

  data = trt_thread_get_local_data ();
  if (data != trt_start_thread_data)
    safe_free (data);

  /* Remove us from the existing threads.  Now we (supposedly) are
     unreferenced and gc bait.  */
  TRT_SEND (, c_tom_Thread_threads, SEL (v_remove_r), self);

  trt_rlock_unlock (trt_runtime_lock);

  trt_thread_exit ();

  fatal ("trt_thread_exit returned!");
}

void
trt_thread_update_locals (void)
{
  int i, n;

  if (trt_thread_num > 1)
    ABORT ();

  if (trt_thread_local_size <= trt_thread_local_cap)
    return;

  for (i = 0, n = trt_thread_num; n && i < trt_thread_cap; i++)
    if (trt_threads[i].object)
      {
	void *new;

	if (trt_threads[i].data == trt_start_thread_data)
	  {
	    new = xmalloc (trt_thread_local_size);
	    memcpy (new, trt_start_thread_data, trt_thread_local_size);
	  }
	else
	  new = safe_realloc (trt_threads[i].data,
			      trt_thread_local_cap, trt_thread_local_size);

	BZERO ((char *) new + trt_thread_local_cap,
	       trt_thread_local_size - trt_thread_local_cap);

	if (trt_threads[i].data != trt_start_thread_data)
	  safe_free (trt_threads[i].data);

	trt_threads[i].data = new;
	n--;
      }

  trt_thread_local_cap = trt_thread_local_size;
}

void
i_tom_RecursiveLock_v_dealloc (tom_object self, selector cmd)
{
  struct _es_i_tom_RecursiveLock *this
    = trt_ext_address (self, _ei_i_tom_RecursiveLock);

  trt_rlock_destroy (this->lock);
}

tom_object
i_tom_RecursiveLock_r_init (tom_object self, selector cmd)
{
  struct _es_i_tom_RecursiveLock *this
    = trt_ext_address (self, _ei_i_tom_RecursiveLock);

  this->lock = trt_rlock_create ();

  return self;
}

void
i_tom_RecursiveLock_v_lock (tom_object self, selector cmd)
{
  struct _es_i_tom_RecursiveLock *this
    = trt_ext_address (self, _ei_i_tom_RecursiveLock);

  trt_rlock_lock (this->lock);
}

void
i_tom_RecursiveLock_v_unlock (tom_object self, selector cmd)
{
  struct _es_i_tom_RecursiveLock *this
    = trt_ext_address (self, _ei_i_tom_RecursiveLock);

  trt_rlock_unlock (this->lock);
}

tom_byte
i_tom_RecursiveLock_o_tryLock (tom_object self, selector cmd)
{
  struct _es_i_tom_RecursiveLock *this
    = trt_ext_address (self, _ei_i_tom_RecursiveLock);

  return trt_rlock_try (this->lock);
}

void
i_tom_Semaphore_v_dealloc (tom_object self, selector cmd)
{
  struct _es_i_tom_Semaphore *this
    = trt_ext_address (self, _ei_i_tom_Semaphore);

  trt_sem_destroy (this->sem);
}

tom_object
i_tom_Semaphore_r_init_i (tom_object self, selector cmd, int num)
{
  struct _es_i_tom_Semaphore *this
    = trt_ext_address (self, _ei_i_tom_Semaphore);

  this->sem = trt_sem_create (num);

  return self;
}

void
i_tom_Semaphore_v_lock (tom_object self, selector cmd)
{
  struct _es_i_tom_Semaphore *this
    = trt_ext_address (self, _ei_i_tom_Semaphore);

  trt_sem_lock (this->sem);
}

void
i_tom_Semaphore_v_unlock (tom_object self, selector cmd)
{
  struct _es_i_tom_Semaphore *this
    = trt_ext_address (self, _ei_i_tom_Semaphore);

  trt_sem_unlock (this->sem);
}

tom_byte
i_tom_Semaphore_o_tryLock (tom_object self, selector cmd)
{
  struct _es_i_tom_Semaphore *this
    = trt_ext_address (self, _ei_i_tom_Semaphore);

  return trt_sem_try (this->sem);
}

void
i_tom_SimpleLock_v_dealloc (tom_object self, selector cmd)
{
  struct _es_i_tom_SimpleLock *this
    = trt_ext_address (self, _ei_i_tom_SimpleLock);

  trt_lock_destroy (this->lock);
}

tom_object
i_tom_SimpleLock_r_init (tom_object self, selector cmd)
{
  struct _es_i_tom_SimpleLock *this
    = trt_ext_address (self, _ei_i_tom_SimpleLock);

  this->lock = trt_lock_create ();

  return self;
}

void
i_tom_SimpleLock_v_lock (tom_object self, selector cmd)
{
  struct _es_i_tom_SimpleLock *this
    = trt_ext_address (self, _ei_i_tom_SimpleLock);

  trt_lock_lock (this->lock);
}

void
i_tom_SimpleLock_v_unlock (tom_object self, selector cmd)
{
  struct _es_i_tom_SimpleLock *this
    = trt_ext_address (self, _ei_i_tom_SimpleLock);

  trt_lock_unlock (this->lock);
}

tom_byte
i_tom_SimpleLock_o_tryLock (tom_object self, selector cmd)
{
  struct _es_i_tom_SimpleLock *this
    = trt_ext_address (self, _ei_i_tom_SimpleLock);

  return trt_lock_try (this->lock);
}

/* Provide default implementations for single-threaded machines.  */
#if THREADS_PACKAGE == THREADS_NO_THREADS

int
trt_thread_mach_init (void)
{
  return 1;
}

trt_thread_t
trt_thread_create (trt_thread_t th_id, struct trt_thread_invocation *args)
{
  return 0;
}

__volatile__ void
trt_thread_exit (void)
{
  ABORT ();
}

static void *thread_local_data;

void *
trt_thread_get_local_data (void)
{
  return thread_local_data;
}

void
trt_thread_set_local_data (void *base)
{
  thread_local_data = base;
}

/********** r(ecursive )locks **********/

struct trt_rlock
{
  int count;
};

trt_rlock_t
trt_rlock_create (void)
{
  return xcalloc (1, sizeof (struct trt_rlock));
}

void
trt_rlock_destroy (trt_rlock_t rlock)
{
  xfree (rlock);
}

void
trt_rlock_lock (trt_rlock_t rlock)
{
  struct trt_rlock *rl = rlock;

  rl->count++;
}

void
trt_rlock_unlock (trt_rlock_t rlock)
{
  struct trt_rlock *rl = rlock;

  rl->count--;
}

tom_byte
trt_rlock_try (trt_rlock_t rlock)
{
  struct trt_rlock *rl = rlock;

  rl->count++;
  return 1;
}

/********** semaphores **********/

struct trt_sem
{
  int val;
};

trt_sem_t
trt_sem_create (int allow)
{
  struct trt_sem *sem = xmalloc (sizeof (*sem));

  sem->val = allow;
  return sem;
}

void
trt_sem_destroy (trt_sem_t sem)
{
  xfree (sem);
}

void
trt_sem_lock (trt_sem_t sem)
{
  struct trt_sem *ts = sem;

  if (!ts->val)
    fatal ("deadlock");

  ts->val--;
}

void
trt_sem_unlock (trt_sem_t sem)
{
  struct trt_sem *ts = sem;
  ts->val++;  
}

tom_byte
trt_sem_try (trt_sem_t sem)
{
  struct trt_sem *ts = sem;

  if (!ts->val)
    fatal ("deadlock");

  ts->val--;
  return 1;
}

/********** locks **********/

struct trt_lock
{
  int lock;
};

trt_lock_t
trt_lock_create (void)
{
  return xcalloc (1, sizeof (trt_lock_t));
}

void
trt_lock_destroy (trt_lock_t rl)
{
  trt_lock_lock (rl);
  xfree (rl);
}

void
trt_lock_lock (trt_lock_t rl)
{
  if (rl->lock)
    fatal ("deadlock");
  rl->lock = 1;
}

void
trt_lock_unlock (trt_lock_t rl)
{
  rl->lock = 0;
}

tom_byte
trt_lock_try (trt_lock_t rl)
{
  return rl->lock ? 0 : (rl->lock = 1);
}

#endif

#if THREADS_PACKAGE == THREADS_CTHREADS
#include <mach/cthreads.h>

/********** cthreads **********/

int
trt_thread_mach_init (void)
{
  return 1;
}

int
trt_thread_create (trt_thread_t th_id, struct trt_thread_invocation *args)
{
  cthread_t th = cthread_fork ((any_t (*) ()) trt_thread_main, args);

  if (!th)
    return 0;

  cthread_detach (th);
  return 1;
}

__volatile__ void
trt_thread_exit (void)
{
  cthread_exit (0);

  ABORT ();
}

void *
trt_thread_get_local_data (void)
{
   return cthread_data (cthread_self ());
}

void
trt_thread_set_local_data (void *base)
{
  cthread_set_data (cthread_self (), base);
}

/********** r(ecursive )locks **********/

struct trt_rlock
{
  struct mutex lock;

  cthread_t owner;

  int count;
};

trt_rlock_t
trt_rlock_create (void)
{
  struct trt_rlock *rl = xcalloc (1, sizeof (*rl));

  mutex_init (&rl->lock);

  return rl;
}

void
trt_rlock_destroy (trt_rlock_t rl)
{
  trt_rlock_lock (rl);
  if (rl->count > 1)
    ABORT ();
  mutex_clear (&rl->lock);
  xfree (rl);
}

void
trt_rlock_lock (trt_rlock_t rl)
{
  cthread_t self = cthread_self ();

  if (rl->owner == self)
    rl->count++;
  else
    {
      mutex_lock (&rl->lock);
      if (rl->owner || rl->count)
	ABORT ();
      rl->owner = self;
      rl->count = 1;
    }
}

void
trt_rlock_unlock (trt_rlock_t rl)
{
  cthread_t self = cthread_self ();

  if (rl->owner != self)
    ABORT ();
  else if (!--rl->count)
    {
      rl->owner = 0;
      mutex_unlock (&rl->lock);
    }
}

tom_byte
trt_rlock_try (trt_rlock_t rl)
{
  cthread_t self = cthread_self ();

  if (rl->owner == self)
    rl->count++;
  else if (!mutex_try_lock (&rl->lock))
    return 0;
  else
    {
      rl->owner = self;
      rl->count = 1;
    }
  return 1;
}

/********** semaphores **********/

struct trt_sem
{
  /* The lock to protect this semaphore.  */
  struct mutex lock;

  /* The condition to signal a P is possible.  */
  struct condition cond;

  /* The number of P operations which can successfully be performed.  */
  int possible;

  /* The number of attempted P operations which are pending.  */
  int to_do;
};

trt_sem_t
trt_sem_create (int allow)
{
  struct trt_sem *sem = xmalloc (sizeof (*sem));

  condition_init (&sem->cond);
  mutex_init (&sem->lock);

  sem->possible = allow;
  sem->to_do = 0;

  return sem;
}

void
trt_sem_destroy (trt_sem_t sem)
{
  /* XXX Obtain unique lock... */
  condition_broadcast (&sem->cond);
  condition_clear (&sem->cond);
  mutex_clear (&sem->lock);

  xfree (sem);
}

void
trt_sem_lock (trt_sem_t sem)
{
  mutex_lock (&sem->lock);
  while (!sem->possible)
    {
      sem->to_do++;
      condition_wait (&sem->cond, &sem->lock);
      sem->to_do--;
    }

  sem->possible--;
  mutex_unlock (&sem->lock);
}

void
trt_sem_unlock (trt_sem_t sem)
{
  int n;

  mutex_lock (&sem->lock);
  sem->possible++;
  n = sem->to_do;
  mutex_unlock (&sem->lock);
  if (n)
    condition_signal (&sem->cond);
}

tom_byte
trt_sem_try (trt_sem_t sem)
{
  int r = 0;

  mutex_lock (&sem->lock);
  if (sem->possible)
    {
      sem->possible--;
      r = 1;
    }
  mutex_unlock (&sem->lock);

  return r;
}

/********** locks **********/

struct trt_lock
{
  struct mutex lock;
};

trt_lock_t
trt_lock_create (void)
{
  struct trt_lock *rl = xmalloc (sizeof (*rl));

  mutex_init (&rl->lock);

  return rl;
}

void
trt_lock_destroy (trt_lock_t rl)
{
  trt_lock_lock (rl);
  mutex_clear (&rl->lock);
  xfree (rl);
}

void
trt_lock_lock (trt_lock_t rl)
{
  mutex_lock (&rl->lock);
}

void
trt_lock_unlock (trt_lock_t rl)
{
  mutex_unlock (&rl->lock);
}

tom_byte
trt_lock_try (trt_lock_t rl)
{
  return mutex_try_lock (&rl->lock);
}

#endif

#if THREADS_PACKAGE == THREADS_PTHREADS
#include <pthread.h>

/********** threads **********/

#ifndef PTHREADS_WITHOUT__RECURSIVE_LOCKS

/* The key used to get at the local data.  This is the only key used.  */
static pthread_key_t local_data_key;

/* Mapping from thread id to pthread_t.  */
static pthread_t *thread_pthreads;
static int thread_pthreads_cap;

/* Attributes of fast and recursive mutexes.  */
static pthread_mutexattr_t fast_attr = {PTHREAD_MUTEX_FAST_NP};
static pthread_mutexattr_t recursive_attr = {PTHREAD_MUTEX_RECURSIVE_NP};

/* Conditions of, what, conditions.  */
static pthread_condattr_t cond_attr;

int
trt_thread_mach_init (void)
{
  thread_pthreads_cap = 8;
  thread_pthreads = xcalloc (thread_pthreads_cap, sizeof (*thread_pthreads));
  thread_pthreads[0] = pthread_self ();

  if (pthread_mutexattr_init (&fast_attr)
      || pthread_mutexattr_init (&recursive_attr)
      || pthread_mutexattr_setkind_np (&fast_attr, PTHREAD_MUTEX_FAST_NP)
      || pthread_mutexattr_setkind_np (&recursive_attr,
				       PTHREAD_MUTEX_RECURSIVE_NP)
      || pthread_condattr_init (&cond_attr)
      || pthread_key_create (&local_data_key, NULL))
    ABORT ();

  return 1;
}

int
trt_thread_create (trt_thread_t th_id, struct trt_thread_invocation *args)
{
  if (th_id >= thread_pthreads_cap)
    {
      int n = thread_pthreads_cap * sizeof (*thread_pthreads);

      thread_pthreads = safe_realloc (thread_pthreads, n, 2 * n);
      memset (thread_pthreads + thread_pthreads_cap, 0, n);
	      thread_pthreads_cap *= 2;
    }

  if (pthread_create (&thread_pthreads[th_id], NULL,
		      (void * (*) (void *)) trt_thread_main, (void *) args))
    return 0;

  if (pthread_detach (thread_pthreads[th_id]))
    ABORT ();

  return 1;
}

__volatile__ void
trt_thread_exit (void)
{
  /* XXX This looses for the main thread...  */
  pthread_exit (0);

  ABORT ();
}

void *
trt_thread_get_local_data (void)
{
  return pthread_getspecific (local_data_key);
}

void
trt_thread_set_local_data (void *base)
{
  if (pthread_setspecific (local_data_key, base))
    ABORT ();
}

/********** r(ecursive )locks **********/

struct trt_rlock
{
  pthread_mutex_t lock;
};

trt_rlock_t
trt_rlock_create (void)
{
  struct trt_rlock *rl = xcalloc (1, sizeof (*rl));

  if (pthread_mutex_init (&rl->lock, &recursive_attr))
    ABORT ();

  return rl;
}

void
trt_rlock_destroy (trt_rlock_t rl)
{
  if (pthread_mutex_destroy (&rl->lock))
    ABORT ();

  xfree (rl);
}

void
trt_rlock_lock (trt_rlock_t rl)
{
  if (pthread_mutex_lock (&rl->lock))
    ABORT ();
}

void
trt_rlock_unlock (trt_rlock_t rl)
{
  if (pthread_mutex_unlock (&rl->lock))
    ABORT ();
}

tom_byte
trt_rlock_try (trt_rlock_t rl)
{
  int i = pthread_mutex_trylock (&rl->lock);

  if (i < 0)
    ABORT ();
  return i;
}

/********** locks **********/

struct trt_lock
{
  pthread_mutex_t lock;
};

trt_lock_t
trt_lock_create (void)
{
  struct trt_lock *rl = xcalloc (1, sizeof (*rl));

  if (pthread_mutex_init (&rl->lock, &fast_attr))
    ABORT ();

  return rl;
}

void
trt_lock_destroy (trt_lock_t rl)
{
  if (pthread_mutex_destroy (&rl->lock))
    ABORT ();

  xfree (rl);
}

void
trt_lock_lock (trt_lock_t rl)
{
  if (pthread_mutex_lock (&rl->lock))
    ABORT ();
}

void
trt_lock_unlock (trt_lock_t rl)
{
  if (pthread_mutex_unlock (&rl->lock))
    ABORT ();
}

tom_byte
trt_lock_try (trt_lock_t rl)
{
  int i = pthread_mutex_trylock (&rl->lock);

  if (i < 0)
    ABORT ();
  return i;
}

/********** semaphores **********/

struct trt_sem
{
  /* The lock to protect this semaphore.  */
  pthread_mutex_t lock;

  /* The condition to signal a P is possible.  */
  pthread_cond_t cond;

  /* The number of P operations which can successfully be performed.  */
  int possible;

  /* The number of attempted P operations which are pending.  */
  int to_do;
};

trt_sem_t
trt_sem_create (int allow)
{
  struct trt_sem *sem = xmalloc (sizeof (*sem));

  if (pthread_cond_init (&sem->cond, &cond_attr)
      || pthread_mutex_init (&sem->lock, &fast_attr))
    ABORT ();

  sem->possible = allow;
  sem->to_do = 0;

  return sem;
}

void
trt_sem_destroy (trt_sem_t sem)
{
  if (pthread_cond_broadcast (&sem->cond)
      || pthread_cond_destroy (&sem->cond)
      || pthread_mutex_destroy (&sem->lock))
    ABORT ();

  xfree (sem);
}

void
trt_sem_lock (trt_sem_t sem)
{
  if (pthread_mutex_lock (&sem->lock))
    ABORT ();
  while (!sem->possible)
    {
      sem->to_do++;
      pthread_cond_wait (&sem->cond, &sem->lock);
      sem->to_do--;
    }

  sem->possible--;
  pthread_mutex_unlock (&sem->lock);
}

void
trt_sem_unlock (trt_sem_t sem)
{
  int n;

  pthread_mutex_lock (&sem->lock);
  sem->possible++;
  n = sem->to_do;
  pthread_mutex_unlock (&sem->lock);
  if (n)
    pthread_cond_signal (&sem->cond);
}

tom_byte
trt_sem_try (trt_sem_t sem)
{
  int r = 0;

  pthread_mutex_lock (&sem->lock);
  if (sem->possible)
    {
      sem->possible--;
      r = 1;
    }
  pthread_mutex_unlock (&sem->lock);

  return r;
}

#else /* PTHREADS_WITHOUT__RECURSIVE_LOCKS */

/* The key used to get at the local data.  This is the only key used.  */
static pthread_key_t local_data_key;

/* Mapping from thread id to pthread_t.  */
static pthread_t *thread_pthreads;
static int thread_pthreads_cap;

static pthread_mutexattr_t mattr;

/* Conditions of, what, conditions.  */
static pthread_condattr_t cond_attr;

int
trt_thread_mach_init (void)
{
  thread_pthreads_cap = 8;
  thread_pthreads = xcalloc (thread_pthreads_cap, sizeof (*thread_pthreads));
  thread_pthreads[0] = pthread_self ();

  if (pthread_mutexattr_init (&mattr)
      || pthread_condattr_init (&cond_attr)
      || pthread_key_create (&local_data_key, NULL))
    ABORT ();

  return 1;
}

int
trt_thread_create (trt_thread_t th_id, struct trt_thread_invocation *args)
{
  if (th_id >= thread_pthreads_cap)
    {
      int n = thread_pthreads_cap * sizeof (*thread_pthreads);

      thread_pthreads = safe_realloc (thread_pthreads, n, 2 * n);
      memset (thread_pthreads + thread_pthreads_cap, 0, n);
	      thread_pthreads_cap *= 2;
    }

  if (pthread_create (&thread_pthreads[th_id], NULL,
		      (void * (*) (void *)) trt_thread_main, (void *) args))
    return 0;

  if (pthread_detach (thread_pthreads[th_id]))
    ABORT ();

  return 1;
}

__volatile__ void
trt_thread_exit (void)
{
  /* XXX This looses for the main thread...  */
  pthread_exit (0);

  ABORT ();
}

void *
trt_thread_get_local_data (void)
{
  return pthread_getspecific (local_data_key);
}

void
trt_thread_set_local_data (void *base)
{
  if (pthread_setspecific (local_data_key, base))
    ABORT ();
}

/********** r(ecursive )locks **********/

struct trt_rlock
{
    pthread_mutex_t lock;
    pthread_t held_by;
    int pending_locks;
};

trt_rlock_t
trt_rlock_create (void)
{
  struct trt_rlock *rl = xcalloc (1, sizeof (*rl));

  if (pthread_mutex_init (&rl->lock, &mattr))
    ABORT ();

  rl->pending_locks = 0;
  rl->held_by = NULL;

  return rl;
}

void
trt_rlock_destroy (trt_rlock_t rl)
{
  if (pthread_mutex_destroy (&rl->lock))
    ABORT ();

  xfree (rl);
}

void
trt_rlock_lock (trt_rlock_t rl)
{
  pthread_t s = pthread_self();

  if (rl->held_by == s) {
    rl->pending_locks ++;
    return;
  }
  if (pthread_mutex_lock (&rl->lock))
    ABORT ();
  rl->held_by = s;
  rl->pending_locks = 1;
}

void
trt_rlock_unlock (trt_rlock_t rl)
{
  if (rl->held_by != pthread_self())
    ABORT();

  rl->pending_locks --;
  if (rl->pending_locks > 0) {
    return;
  }
  if (pthread_mutex_unlock (&rl->lock))
    ABORT ();
}

tom_byte
trt_rlock_try (trt_rlock_t rl)
{
  int i;

  if (pthread_self() == rl->held_by)
    return 0;
  i = pthread_mutex_trylock (&rl->lock);

  if (i < 0)
    ABORT ();
  return i;
}

/********** locks **********/

struct trt_lock
{
  pthread_mutex_t lock;
};

trt_lock_t
trt_lock_create (void)
{
  struct trt_lock *rl = xcalloc (1, sizeof (*rl));

  if (pthread_mutex_init (&rl->lock, &mattr))
    ABORT ();

  return rl;
}

void
trt_lock_destroy (trt_lock_t rl)
{
  if (pthread_mutex_destroy (&rl->lock))
    ABORT ();

  xfree (rl);
}

void
trt_lock_lock (trt_lock_t rl)
{
  if (pthread_mutex_lock (&rl->lock))
    ABORT ();
}

void
trt_lock_unlock (trt_lock_t rl)
{
  if (pthread_mutex_unlock (&rl->lock))
    ABORT ();
}

tom_byte
trt_lock_try (trt_lock_t rl)
{
  int i = pthread_mutex_trylock (&rl->lock);

  if (i < 0)
    ABORT ();
  return i;
}

/********** semaphores **********/

struct trt_sem
{
  /* The lock to protect this semaphore.  */
  pthread_mutex_t lock;

  /* The condition to signal a P is possible.  */
  pthread_cond_t cond;

  /* The number of P operations which can successfully be performed.  */
  int possible;

  /* The number of attempted P operations which are pending.  */
  int to_do;
};

trt_sem_t
trt_sem_create (int allow)
{
  struct trt_sem *sem = xmalloc (sizeof (*sem));

  if (pthread_cond_init (&sem->cond, &cond_attr)
      || pthread_mutex_init (&sem->lock, &mattr))
    ABORT ();

  sem->possible = allow;
  sem->to_do = 0;

  return sem;
}

void
trt_sem_destroy (trt_sem_t sem)
{
  if (pthread_cond_broadcast (&sem->cond)
      || pthread_cond_destroy (&sem->cond)
      || pthread_mutex_destroy (&sem->lock))
    ABORT ();

  xfree (sem);
}

void
trt_sem_lock (trt_sem_t sem)
{
  if (pthread_mutex_lock (&sem->lock))
    ABORT ();
  while (!sem->possible)
    {
      sem->to_do++;
      pthread_cond_wait (&sem->cond, &sem->lock);
      sem->to_do--;
    }

  sem->possible--;
  pthread_mutex_unlock (&sem->lock);
}

void
trt_sem_unlock (trt_sem_t sem)
{
  int n;

  pthread_mutex_lock (&sem->lock);
  sem->possible++;
  n = sem->to_do;
  pthread_mutex_unlock (&sem->lock);
  if (n)
    pthread_cond_signal (&sem->cond);
}

tom_byte
trt_sem_try (trt_sem_t sem)
{
  int r = 0;

  pthread_mutex_lock (&sem->lock);
  if (sem->possible)
    {
      sem->possible--;
      r = 1;
    }
  pthread_mutex_unlock (&sem->lock);

  return r;
}

#endif

#endif
