(**
  Implements some classes for drawing object backgrounds in various ways.
**)

MODULE VOBackground;

(*
    Implements some classes for drawing object backgrounds in various ways.
    Copyright (C) 1998  Tim Teulings (rael@edge.ping.de)

    This file is part of VisualOberon.

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D  := VODisplay,
       G  := VOGUIObject,
       I  := VOImage,
       PP := VOPrefsParser;

TYPE
  ImageBackground*     = POINTER TO ImageBackgroundDesc;
  ImageBackgroundDesc* = RECORD (G.BackgroundDesc)
                           image : I.Image;
                         END;

  TileBackground*     = POINTER TO TileBackgroundDesc;
  TileBackgroundDesc* = RECORD (G.BackgroundDesc)
                          image : I.Image;
                        END;

  PROCEDURE (i : ImageBackground) Init*(filename : ARRAY OF CHAR);

  BEGIN
    NEW(i.image);
    i.image.Init;
    i.image.SetFlags({G.horizontalFlex,G.verticalFlex});
    i.image.SetFilename(filename);
  END Init;

  PROCEDURE (i : ImageBackground) CalcSize*(display : D.Display);

  BEGIN
    i.image.CalcSize(display);
  END CalcSize;

  PROCEDURE (i : ImageBackground) Draw*(object : G.Object; x,y,w,h : LONGINT);

  BEGIN
    i.image.Resize(i.source.oWidth,i.source.oHeight);
    i.image.DrawPart(x-i.source.x,y-i.source.y,w,h,object.x,object.y,object.draw);
  END Draw;

  PROCEDURE (b : ImageBackground) Copy*():G.Background;

  VAR
    copy : ImageBackground;

  BEGIN
    NEW(copy);
    copy^:=b^;

    RETURN copy;
  END Copy;

  PROCEDURE (i : ImageBackground) Free*;

  BEGIN
    IF i.image#NIL THEN
      i.image.Free;
      i.image:=NIL;
    END;
  END Free;

  PROCEDURE (t : TileBackground) Init*(filename : ARRAY OF CHAR);

  BEGIN
    NEW(t.image);
    t.image.Init;
    t.image.SetFilename(filename);
  END Init;

  PROCEDURE (t : TileBackground) CalcSize*(display : D.Display);

  BEGIN
    t.image.CalcSize(display);
  END CalcSize;

  PROCEDURE (t : TileBackground) Draw*(object : G.Object; x,y,w,h : LONGINT);

  BEGIN
    t.image.DrawTiled(x,y,w,h,0,0,object.draw);
  END Draw;

  PROCEDURE (t : TileBackground) Copy*():G.Background;

  VAR
    copy : TileBackground;

  BEGIN
    NEW(copy);
    copy^:=t^;

    RETURN copy;
  END Copy;

  PROCEDURE (t : TileBackground) Free*;

  BEGIN
    IF t.image#NIL THEN
      t.image.Free;
      t.image:=NIL;
    END;
  END Free;

  PROCEDURE LoadBackground*(top : PP.Item):G.Background;

  VAR
    type,
    data : ARRAY 256 OF CHAR;

    i    : ImageBackground;
    t    : TileBackground;

  BEGIN
    IF top.GetStringEntry("backgroundType",type) THEN
      IF type="image" THEN
        IF top.GetStringEntry("backgroundName",data) THEN
          NEW(i);
          i.Init(data);
          RETURN i;
        END;
      ELSIF type="imageTile" THEN
        IF top.GetStringEntry("backgroundName",data) THEN
          NEW(t);
          t.Init(data);
          RETURN t;
        END;
      END;
    END;
    RETURN NIL;
  END LoadBackground;

  PROCEDURE SaveBackground*(block : PP.BlockItem; background : G.Background);

  VAR
    value : PP.ValueItem;

  BEGIN
    IF background#NIL THEN
      WITH
        background : ImageBackground DO
          NEW(value);
          value.Init;
          value.SetValue("backgroundType","image");
          block.AddItem(value);
          NEW(value);
          value.Init;
          value.SetValue("backgroundName",background.image.filename^);
          block.AddItem(value);
      | background : TileBackground DO
          NEW(value);
          value.Init;
          value.SetValue("backgroundType","imageTile");
          block.AddItem(value);
          NEW(value);
          value.Init;
          value.SetValue("backgroundName",background.image.filename^);
          block.AddItem(value);
      END;
    END;
  END SaveBackground;

END VOBackground.