(**
   A radio button.
**)

MODULE VORadio;

(*
    Implements a radiobutton.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D  := VODisplay,
       E  := VOEvent,
       G  := VOGUIObject,
       O  := VOObject,
       P  := VOPrefs,
       V  := VOVecImage,
       VM := VOValue;

CONST
  clickedMsg * = 0;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (P.PrefsDesc)
                 image* : LONGINT; (* the frame to use for the button *)
               END;


  Radio*     = POINTER TO RadioDesc;
  RadioDesc* = RECORD (G.GadgetDesc)
                 prefs*   : Prefs;
                 image    : V.VecImage;
                 model-   : VM.ValueModel;
                 number-  : LONGINT;
                 tmpState : LONGINT;
               END;

  (* messages *)

  ClickedMsg*     = POINTER TO ClickedMsgDesc;

  (**
    The PressedMsg generated everytime the button get clicked.
  **)

  ClickedMsgDesc* = RECORD (O.MessageDesc)
                      value* : LONGINT;
                    END;

VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.image:=V.radio;
  END Init;

  PROCEDURE (r : Radio) Init*;

  BEGIN
    r.Init^;

    r.prefs:=prefs;

    INCL(r.flags,G.canFocus);

    r.model:=NIL;
    r.label:=NIL;

    NEW(r.image);
    r.image.Init;
    r.image.SetFlags({G.horizontalFlex,G.verticalFlex});

    r.number:=0;
  END Init;

  PROCEDURE (r : Radio) IsIn(x,y : LONGINT):BOOLEAN;

  BEGIN
    RETURN r.PointIsIn(x,y) OR ((r.label#NIL) & (r.label.PointIsIn(x,y)));
  END IsIn;

  PROCEDURE (r : Radio) SetModel*(model : VM.ValueModel; number : LONGINT);

  BEGIN
    IF r.model#NIL THEN
      r.UnattachModel(r.model);
    END;
    r.number:=number;
    r.model:=model;
    r.AttachModel(model);
  END SetModel;

  PROCEDURE (r : Radio) CalcSize*(display : D.Display);

  BEGIN
    (*
      If the label has a custom way for displaying a focus frame
      we delegate the focus displaying to the label.
    *)
    IF (r.label#NIL) & ~r.label.StdFocus() & r.MayFocus() THEN
      EXCL(r.flags,G.stdFocus);
      INCL(r.label.flags,G.mayFocus);
    END;

    r.image.Set(r.prefs.image);
    r.image.CalcSize(display);

    r.width:=r.image.width;
    r.height:=r.image.height;
    r.minWidth:=r.image.minWidth;
    r.minHeight:=r.image.minHeight;

    r.CalcSize^(display);
  END CalcSize;

  PROCEDURE (r : Radio) GetFocus*(event : E.Event):G.Object;

  BEGIN
    IF ~r.visible OR r.disabled OR (r.model=NIL) THEN
      RETURN NIL;
    END;

    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseDown)
      & (r.IsIn(event.x,event.y))
      & (event.qualifier={}) & (event.button=E.button1) THEN
        r.tmpState:=r.model.GetLongint();
        r.model.SetLongint(r.number);
        RETURN r;
      END;
    ELSE
    END;

    RETURN NIL;
  END GetFocus;

  PROCEDURE (r : Radio) HandleEvent*(event : E.Event):BOOLEAN;

  VAR
    clicked : ClickedMsg;

  BEGIN
    WITH
      event : E.MouseEvent DO
        IF (event.type=E.mouseUp) & (event.button=E.button1) THEN
          IF r.IsIn(event.x,event.y) THEN
            r.model.SetLongint(r.number);
            (* Action: State changed *)
            NEW(clicked);
            clicked.value:=r.model.GetLongint();
            r.Send(clicked,clickedMsg);
          END;
          RETURN TRUE;
        END;
    | event : E.MotionEvent DO
      IF r.IsIn(event.x,event.y) THEN
        r.model.SetLongint(r.number);
      ELSE
        r.model.SetLongint(r.tmpState);
      END;
      RETURN FALSE;
    ELSE
    END;

    RETURN FALSE;
  END HandleEvent;

  PROCEDURE (r : Radio) HandleFocusEvent*(event : E.KeyEvent):BOOLEAN;

  VAR
    keysym : LONGINT;

  BEGIN
    IF event.type=E.keyDown THEN
      keysym:=event.GetKey();
      IF keysym=E.space THEN
        r.model.SetLongint(r.number);
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END HandleFocusEvent;


  PROCEDURE (r : Radio) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    r.Draw^(x,y,draw);

    r.image.Resize(r.width,r.height);

    IF (r.model#NIL) & ~r.model.IsNull() & (r.model.GetLongint()=r.number) THEN
      draw.mode:={D.selected};
    ELSE
      draw.mode:={};
    END;

    IF r.disabled & (G.canDisable IN r.image.flags) THEN
      INCL(r.draw.mode,D.disabled);
    END;

    r.image.Draw(r.x+(r.width-r.image.width) DIV 2,r.y+(r.height-r.image.height) DIV 2,draw);

    draw.mode:={};

    IF r.disabled & ~(G.canDisable IN r.image.flags) THEN
      r.DrawDisabled;
    END;

  END Draw;

  (**
    Draw the keyboard focus.
  **)

  PROCEDURE (r : Radio) DrawFocus*;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF (r.label#NIL) & ~r.label.StdFocus() THEN
      r.label.DrawFocus;
    ELSE
      (* Delegate drawing to the baseclass *)
      r.DrawFocus^;
    END;
  END DrawFocus;

  (**
    Hide the keyboard focus.
  **)

  PROCEDURE (r : Radio) HideFocus*;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF (r.label#NIL) & ~r.label.StdFocus() THEN
      r.label.HideFocus;
    ELSE
      (* Delegate drawing to the baseclass *)
      r.HideFocus^;
    END;
  END HideFocus;

  PROCEDURE (r : Radio) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF r.visible THEN
      r.Redraw;
    END;
  END Resync;

  PROCEDURE (r : Radio) Receive*(message : O.Message);

  BEGIN
    WITH message : G.LabelClickedMsg DO
      IF r.model#NIL THEN
        r.model.SetLongint(r.number);
      END;
    ELSE
    END;
  END Receive;

  PROCEDURE (r : Radio) Hide*;

  BEGIN
    IF r.visible THEN
      r.image.Hide;
      r.Hide^;
    END;
  END Hide;

BEGIN
  NEW(prefs);
  prefs.Init;

END VORadio.