/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : DPTich_edit_cb.cc
|
| Revision    : 1.0
| Date        : 02/05/96
|
| Object      : None
|
| Description : XForms callback code
|
| (c) Richard Kent 1996
|
| $Id$
|
****************************************************************************/

static char DPTich_edit_cb_cc [] = "$Id$";

#include "DPTich_main.h"

/*---------------------------------------------------------------------------
| Defines
---------------------------------------------------------------------------*/

#define MAXRAMPPOINTS 20

/*---------------------------------------------------------------------------
| Global data (sorry !!)
---------------------------------------------------------------------------*/

extern double currentPitch;

long   rampFrames  [MAXRAMPPOINTS];
double rampAmounts [MAXRAMPPOINTS];
long   rampNumber      = 2;
double rateChange      = 100.0;
double framesChange    = 100.0;
double timeChange      = 100.0;
double pitchChange     = 100.0;
long   beats           = 1;
double currentBigSemi  = 60.0;
char   *noteStrings [] =
  { "C","C#","D","D#","E","F","F#","G","G#","A","A#","B" };

/*---------------------------------------------------------------------------
| CALLBACK mixRatio_cb
---------------------------------------------------------------------------*/
void mixRatio_cb (FL_OBJECT *ob,long data)
{
  long temp = (long) fl_get_slider_value (ob);
  char tempString [20];
  sprintf (tempString,"%d",temp);
  fl_set_object_label (mixForm->clipboardText,tempString);
  sprintf (tempString,"%d",100 - temp);
  fl_set_object_label (mixForm->bufferText,tempString);
  updateMixPlot ();
}

/*---------------------------------------------------------------------------
| CALLBACK mixMode_cb
---------------------------------------------------------------------------*/
void mixMode_cb (FL_OBJECT *ob,long data)
{
  fl_call_object_callback (mixForm->inFrames);
}

/*---------------------------------------------------------------------------
| CALLBACK mixInFrames_cb
---------------------------------------------------------------------------*/
void mixInFrames_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  double rate = sample->getRate ();
  long inframes;
  long outframes;
  long length;
  char tempString [50];
  if (!sscanf (fl_get_input (mixForm->inFrames),"%ld",&inframes)) return;
  if (!sscanf (fl_get_input (mixForm->outFrames),"%ld",&outframes)) return;
  if (fl_get_choice (mixForm->mode) == 1)
  {
    if (clip->getValid ())
      length = clip->getFrames ();
    else
      length = 0;
  }
  else
  {
    if (sample->getRangeValid ())
      length = sample->getRangeEnd () - sample->getRangeStart ();
    else
      length = 0;
  }
  if (inframes < 0) inframes = 0;  
  if (inframes > length) inframes = length;
  if ((inframes + outframes) > length)
    outframes = length - inframes;

  sprintf (tempString,"%ld",inframes);
  fl_set_input (mixForm->inFrames,tempString);
  sprintf (tempString,"%.3lf",inframes / rate);
  fl_set_input (mixForm->inTime,tempString);

  sprintf (tempString,"%ld",outframes);
  fl_set_input (mixForm->outFrames,tempString);
  sprintf (tempString,"%.3lf",outframes / rate);
  fl_set_input (mixForm->outTime,tempString);

  updateMixPlot ();
}

/*---------------------------------------------------------------------------
| CALLBACK mixInTime_cb
---------------------------------------------------------------------------*/

void mixInTime_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  double rate = sample->getRate ();
  double time;
  char tempString [50];
  if (!sscanf (fl_get_input (ob),"%lf",&time)) return;
  sprintf (tempString,"%ld",(long) (time * rate));
  fl_set_input (mixForm->inFrames,tempString);
  fl_call_object_callback (mixForm->inFrames);
}

/*---------------------------------------------------------------------------
| CALLBACK mixOutFrames_cb
---------------------------------------------------------------------------*/
void mixOutFrames_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  double rate = sample->getRate ();
  long inframes;
  long outframes;
  long length;
  char tempString [50];
  if (!sscanf (fl_get_input (mixForm->inFrames),"%ld",&inframes)) return;
  if (!sscanf (fl_get_input (mixForm->outFrames),"%ld",&outframes)) return;
  if (fl_get_choice (mixForm->mode) == 1)
  {
    if (clip->getValid ())
      length = clip->getFrames ();
    else
      length = 0;
  }
  else
  {
    if (sample->getRangeValid ())
      length = sample->getRangeEnd () - sample->getRangeStart ();
    else
      length = 0;
  }
  if (outframes < 0) outframes = 0;    
  if (outframes > length) outframes = length;
  if ((inframes + outframes) > length)
    inframes = length - outframes;

  sprintf (tempString,"%ld",inframes);
  fl_set_input (mixForm->inFrames,tempString);
  sprintf (tempString,"%.3lf",inframes / rate);
  fl_set_input (mixForm->inTime,tempString);

  sprintf (tempString,"%ld",outframes);
  fl_set_input (mixForm->outFrames,tempString);
  sprintf (tempString,"%.3lf",outframes / rate);
  fl_set_input (mixForm->outTime,tempString);
  
  updateMixPlot ();
}

/*---------------------------------------------------------------------------
| CALLBACK mixOutTime_cb
---------------------------------------------------------------------------*/

void mixOutTime_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  double rate = sample->getRate ();
  double time;
  char tempString [50];
  if (!sscanf (fl_get_input (ob),"%lf",&time)) return;
  sprintf (tempString,"%ld",(long) (time * rate));
  fl_set_input (mixForm->outFrames,tempString);
  fl_call_object_callback (mixForm->outFrames);
}

/*---------------------------------------------------------------------------
| CALLBACK mixOK_cb
---------------------------------------------------------------------------*/
void mixOK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);

  noSampleWarning
  noRangeWarning
  noClipWarning
  
  int choice = fl_get_choice (mixForm->mode);
  int ratio  = (int) fl_get_slider_value (mixForm->ratio);
  long mixIn;
  long mixOut;

  fl_hide_form (mixForm->mixForm);
  if (!sscanf (fl_get_input (mixForm->inFrames),"%ld",&mixIn)) return;
  if (!sscanf (fl_get_input (mixForm->outFrames),"%ld",&mixOut)) return;
  if (choice == 1)
  {
    updateBuffers ("Undo Mix")
    waitCursorOn ();
    sample->mix (clip,100 - ratio,ratio,mixIn,mixOut,
      fl_get_sample_edit (mainForm->sample));
    waitCursorOff ();
  }
  else
  {
    updateBuffers ("Undo Mix Range")
    waitCursorOn ();
    sample->mixRange (clip,100 - ratio,ratio,mixIn,mixOut,
      fl_get_sample_edit (mainForm->sample));
    waitCursorOff ();
  }
  updateDisplayDetails ();
  updateRangeDetails ();
  updateLoopDetails ();
  updateNameBox ();
  fl_redraw_object (mainForm->sample);
  fl_redraw_object (mainForm->scrollBarSlider);
}

/*---------------------------------------------------------------------------
| CALLBACK mixCancel_cb
---------------------------------------------------------------------------*/
void mixCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (mixForm->mixForm);
}

/*---------------------------------------------------------------------------
| CALLBACK mixPlot_cb
---------------------------------------------------------------------------*/
void mixPlot_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  long  inframes;
  long  outframes;
  long  length;
  float x;
  float y;
  int   n;
  if (fl_get_choice (mixForm->mode) == 1)
  {
    if (clip->getValid ())
      length = clip->getFrames ();
    else
      length = 0;
  }
  else
  {
    if (sample->getRangeValid ())
      length = sample->getRangeEnd () - sample->getRangeStart ();
    else
      length = 0;
  }
  
  char tempString [20];
  fl_get_xyplot (ob,&x,&y,&n);

  if (x < 0.0) x = 0.0;
  else if (x > 100.0) x = 100.0;
  if (y < 0.0) y = 0.0;
  else if (y > 100.0) y = 100.0;
  
  if (n==1)
  {
    fl_set_slider_value (mixForm->ratio,(double) y);
    sprintf (tempString,"%d",(int) y);
    fl_set_object_label (mixForm->clipboardText,tempString);
    sprintf (tempString,"%d",100 - (int) y);
    fl_set_object_label (mixForm->bufferText,tempString);
    inframes = (long) (length * x / 100.0);
    sprintf (tempString,"%ld",inframes);
    fl_set_input (mixForm->inFrames,tempString);
  }
  else if (n==2)
  {
    fl_set_slider_value (mixForm->ratio,(double) y);
    sprintf (tempString,"%d",(int) y);
    fl_set_object_label (mixForm->clipboardText,tempString);
    sprintf (tempString,"%d",100 - (int) y);
    fl_set_object_label (mixForm->bufferText,tempString);
    outframes = (long) (length * (100.0 - x) / 100.0);
    sprintf (tempString,"%ld",outframes);
    fl_set_input (mixForm->outFrames,tempString);
  }
  
  fl_call_object_callback (mixForm->inFrames);
}

/*---------------------------------------------------------------------------
| FUNCTION updateMixPlot
---------------------------------------------------------------------------*/
void updateMixPlot ()
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  double rate = sample->getRate ();
  double mix  = fl_get_slider_value (mixForm->ratio);
  long inframes;
  long outframes;
  long length;
  char tempString [50];
  if (!sscanf (fl_get_input (mixForm->inFrames),"%ld",&inframes)) return;
  if (!sscanf (fl_get_input (mixForm->outFrames),"%ld",&outframes)) return;
  if (fl_get_choice (mixForm->mode) == 1)
  {
    if (clip->getValid ())
      length = clip->getFrames ();
    else
      length = 0;
  }
  else
  {
    if (sample->getRangeValid ())
      length = sample->getRangeEnd () - sample->getRangeStart ();
    else
      length = 0;
  }
  if ((inframes + outframes) > length)
  {
    inframes = (inframes + (length - outframes)) / 2;
    if (inframes < 0) inframes = 0;
    if (inframes > length) inframes = length;
    outframes = length - inframes;
  }
  if (inframes < 0) inframes = 0;
  if (inframes > length) inframes = length;
  if (outframes < 0) outframes = 0;
  if (outframes > length) outframes = length;
  if ((inframes + outframes) > length)
    outframes = length - inframes;
  sprintf (tempString,"%ld",inframes);
  fl_set_input (mixForm->inFrames,tempString);
  fl_redraw_object (mixForm->inFrames);
  sprintf (tempString,"%ld",outframes);
  fl_set_input (mixForm->outFrames,tempString);
  fl_redraw_object (mixForm->outFrames);
  sprintf (tempString,"%ld",length);
  fl_set_object_label (mixForm->framesText,tempString);
  fl_redraw_object (mixForm->framesText);
  sprintf (tempString,"%.3lf",length / rate);
  fl_set_object_label (mixForm->timeText,tempString);
  fl_redraw_object (mixForm->timeText);
  
  float x [4];
  float y [4];
  
  x[0] = 0.0;   
  y[0] = 0.0;

  if (length == 0) x[1] = 0.0;
  else x[1] = (float) 100.0 * inframes / length;
  y[1] = mix;

  if (length == 0) x[2] = 100.0;
  else x[2] = 100.0 - (float) 100.0 * outframes / length;
  y[2] = mix;

  x[3] = 100.0;
  y[3] = 0.0;
  
  fl_set_xyplot_data (mixForm->plot,x,y,4,"","","");
  fl_redraw_object (mixForm->plot);
}

/*---------------------------------------------------------------------------
| CALLBACK channelButton_cb
---------------------------------------------------------------------------*/
void channelButton_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK channelOK_cb
---------------------------------------------------------------------------*/
void channelOK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);

  noSampleWarning
  noRangeWarning
  zeroRangeWarning  
  noStereoWarning

  fl_hide_form (channelForm->channelForm);
  if (fl_get_button (channelForm->leftButton))
  {
    updateBuffers ("Undo Channels (L<-R)")
    waitCursorOn ();
    sample->channelChange (0);
    waitCursorOff ();
  }
  else if (fl_get_button (channelForm->leftrightButton))
  {
    updateBuffers ("Undo Channels (L<->R)")
    waitCursorOn ();
    sample->channelChange (1);
    waitCursorOff ();
  }
  else
  {
    updateBuffers ("Undo Channels (L->R)")
    waitCursorOn ();
    sample->channelChange (2);
    waitCursorOff ();
  }
  fl_redraw_object (mainForm->sample);
}


/*---------------------------------------------------------------------------
| CALLBACK channelCancel_cb
---------------------------------------------------------------------------*/
void channelCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (channelForm->channelForm);
}

/*---------------------------------------------------------------------------
| CALLBACK channel4Button_cb
---------------------------------------------------------------------------*/
void channel4Button_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK channel4OK_cb
---------------------------------------------------------------------------*/
void channel4OK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);

  noSampleWarning
  noRangeWarning
  zeroRangeWarning  
  noStereoWarning

  int mode = 0;
  
  fl_hide_form (channel4Form->channel4Form);
  if (fl_get_button (channel4Form->oneLButton))
    mode = mode | 0;
  else if (fl_get_button (channel4Form->twoLButton))
    mode = mode | 1;
  else if (fl_get_button (channel4Form->threeLButton))
    mode = mode | 2;
  else if (fl_get_button (channel4Form->fourLButton))
    mode = mode | 3;

  if (fl_get_button (channel4Form->oneRButton))
    mode = mode | (0 << 2);
  else if (fl_get_button (channel4Form->twoRButton))
    mode = mode | (1 << 2);
  else if (fl_get_button (channel4Form->threeRButton))
    mode = mode | (2 << 2);
  else if (fl_get_button (channel4Form->fourRButton))
    mode = mode | (3 << 2);
  
  if (fl_get_button (channel4Form->leftButton))
    mode = mode | (0 << 4);
  else if (fl_get_button (channel4Form->leftrightButton))
    mode = mode | (1 << 4);
  else if (fl_get_button (channel4Form->rightButton))
    mode = mode | (2 << 4);
  
  updateBuffers ("Undo Channels (quadro)")
  waitCursorOn ();
  sample->channelChange (mode);
  waitCursorOff ();
  fl_redraw_object (mainForm->sample);
}

/*---------------------------------------------------------------------------
| CALLBACK channel4Cancel_cb
---------------------------------------------------------------------------*/
void channel4Cancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (channel4Form->channel4Form);
}

/*---------------------------------------------------------------------------
| CALLBACK duplicateTimes_cb
---------------------------------------------------------------------------*/
void duplicateTimes_cb (FL_OBJECT *ob,long data)
{
  long times = (long) fl_get_slider_value (ob);
  char tempString [50];
  sprintf (tempString,"%ld",times);
  fl_set_input (duplicateForm->input,tempString);
}

/*---------------------------------------------------------------------------
| CALLBACK duplicateInput_cb
---------------------------------------------------------------------------*/
void duplicateInput_cb (FL_OBJECT *ob,long data)
{
  long times;
  if (!sscanf (fl_get_input (ob),"%ld",&times)) return;
  if (times < 1)
  {
    times = 1;
    fl_set_input (ob,"1");
    fl_set_slider_value (duplicateForm->times,1.0);
  }
  else if (times > 100)
    fl_set_slider_value (duplicateForm->times,100.0);
  else
    fl_set_slider_value (duplicateForm->times,(double) times);
}

/*---------------------------------------------------------------------------
| CALLBACK duplicateOK_cb
---------------------------------------------------------------------------*/
void duplicateOK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);

  noSampleWarning
  noRangeWarning
  zeroRangeWarning  

  long times;

  fl_hide_form (duplicateForm->duplicateForm);
  if (!sscanf (fl_get_input (duplicateForm->input),"%ld",&times)) return;
  updateBuffers ("Undo Duplicate")
  waitCursorOn ();
  sample->duplicate (times,fl_get_sample_edit (mainForm->sample));
  waitCursorOff ();
  updateDisplayDetails ();
  updateRangeDetails ();
  updateLoopDetails ();
  updateNameBox ();
  fl_redraw_object (mainForm->sample);
  fl_redraw_object (mainForm->scrollBarSlider);
}

/*---------------------------------------------------------------------------
| CALLBACK duplicateCancel_cb
---------------------------------------------------------------------------*/
void duplicateCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (duplicateForm->duplicateForm);
}

/*---------------------------------------------------------------------------
| CALLBACK panoramaAmount_cb
---------------------------------------------------------------------------*/
void panoramaAmount_cb (FL_OBJECT *ob,long data)
{
  int amount = (int) fl_get_slider_value (ob);
  char tempString [20];
  sprintf (tempString,"%+d",amount);
  fl_set_object_label (panoramaForm->amountText,tempString);
}

/*---------------------------------------------------------------------------
| CALLBACK panoramaQuadro_cb
---------------------------------------------------------------------------*/
void panoramaQuadro_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK panoramaLtoRSlider_cb
---------------------------------------------------------------------------*/
void panoramaLtoRSlider_cb (FL_OBJECT *ob,long data)
{
  long delay = (long) fl_get_slider_value (ob);
  char tempString [20];
  sprintf (tempString,"%ld",delay);
  fl_set_input (panoramaForm->ltorInput,tempString);
}

/*---------------------------------------------------------------------------
| CALLBACK panoramaLtoRInput_cb
---------------------------------------------------------------------------*/
void panoramaLtoRInput_cb (FL_OBJECT *ob,long data)
{
  long delay;
  if (!sscanf (fl_get_input (ob),"%ld",&delay)) return;
  if (delay < 0)
  {
    delay = 0;
    fl_set_input (ob,"0");
  }
  else if (delay > 1000) delay = 1000;
  fl_set_slider_value (panoramaForm->ltorSlider,(double) delay);
}

/*---------------------------------------------------------------------------
| CALLBACK panoramaRtoLSlider_cb
---------------------------------------------------------------------------*/
void panoramaRtoLSlider_cb (FL_OBJECT *ob,long data)
{
  long delay = (long) fl_get_slider_value (ob);
  char tempString [20];
  sprintf (tempString,"%ld",delay);
  fl_set_input (panoramaForm->rtolInput,tempString);
}

/*---------------------------------------------------------------------------
| CALLBACK panoramaRtoLInput_cb
---------------------------------------------------------------------------*/
void panoramaRtoLInput_cb (FL_OBJECT *ob,long data)
{
  long delay;
  if (!sscanf (fl_get_input (ob),"%ld",&delay)) return;
  if (delay < 0)
  {
    delay = 0;
    fl_set_input (ob,"0");
  }
  else if (delay > 1000) delay = 1000;
  fl_set_slider_value (panoramaForm->rtolSlider,(double) delay);
}

/*---------------------------------------------------------------------------
| CALLBACK panoramaOK_cb
---------------------------------------------------------------------------*/
void panoramaOK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);

  noSampleWarning
  noRangeWarning
  zeroRangeWarning  
  noStereoWarning

  long ltorDelay;
  long rtolDelay;
  double rate = sample->getRate ();
  long ltorFrames;
  long rtolFrames;
  int amount = (int) fl_get_slider_value (panoramaForm->amount);

  fl_hide_form (panoramaForm->panoramaForm);
  if (!sscanf (fl_get_input (panoramaForm->ltorInput),"%ld",&ltorDelay)) return;
  if (!sscanf (fl_get_input (panoramaForm->rtolInput),"%ld",&rtolDelay)) return;
  ltorFrames = (long) ((double) ltorDelay * rate / 1000000.0);
  rtolFrames = (long) ((double) rtolDelay * rate / 1000000.0);
  
  if (sample->getChannels () == 2)
  {
    updateBuffers ("Undo Panorama (stereo)")
    waitCursorOn ();
    sample->panorama (amount,ltorFrames,rtolFrames,0);
    waitCursorOff ();
    fl_redraw_object (mainForm->sample);
  }
  else
  {
    int mode = 0;
    int left  = fl_get_choice (panoramaForm->quadroLeft);
    int right = fl_get_choice (panoramaForm->quadroRight);
    mode = mode | (left - 1);
    mode = mode | ((right - 1) << 2);
    updateBuffers ("Undo Panorama (quadro)")
    waitCursorOn ();
    sample->panorama (amount,ltorFrames,rtolFrames,mode);
    waitCursorOff ();
    fl_redraw_object (mainForm->sample);
  }
}

/*---------------------------------------------------------------------------
| CALLBACK panoramaCancel_cb
---------------------------------------------------------------------------*/
void panoramaCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (panoramaForm->panoramaForm);
}

/*---------------------------------------------------------------------------
| CALLBACK adjustDCAmount_cb
---------------------------------------------------------------------------*/
void adjustDCAmount_cb (FL_OBJECT *ob,long data)
{
  int amount = (int) fl_get_slider_value (ob);
  char tempString [20];
  sprintf (tempString,"%+d",amount);
  fl_set_object_label (adjustDCForm->amountText,tempString);
}

/*---------------------------------------------------------------------------
| CALLBACK adjustDCAutoDC_cb
---------------------------------------------------------------------------*/
void adjustDCAutoDC_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);

  noSampleWarning
  noRangeWarning
  zeroRangeWarning  
  
  long i;
  long rangeStart  = sample->getRangeStart ();
  long rangeEnd    = sample->getRangeEnd ();
  long long frames = rangeEnd - rangeStart;
  long long suml = 0;
  long long sumr = 0;
  long long sum3 = 0;
  long long sum4 = 0;
  long avgl;
  long avgr;
  long avg3;
  long avg4;
  double amountl;
  double amountr;
  double amount3;
  double amount4;

  int mode       = fl_get_sample_edit (mainForm->sample);
  int channels   = sample->getChannels ();
  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));

  for (i=rangeStart; i<rangeEnd; i++)
  {
    if (left)  suml += sample->getFrame24 (i,0);
    if (right) sumr += sample->getFrame24 (i,1);
    if (three) sum3 += sample->getFrame24 (i,2);
    if (four)  sum4 += sample->getFrame24 (i,3);
  }
  
  if (left)  avgl = (long) (suml / frames);
  if (right) avgr = (long) (sumr / frames);
  if (three) avg3 = (long) (sum3 / frames);
  if (four)  avg4 = (long) (sum4 / frames);
  
  if (left)  amountl = - (100.0 * avgl / MAX23);
  if (right) amountr = - (100.0 * avgr / MAX23);
  if (three) amount3 = - (100.0 * avg3 / MAX23);
  if (four)  amount4 = - (100.0 * avg4 / MAX23);
  
  double averageAmount;
  double sumAmount = 0.0;
  int countAmount = 0;
  char tempString [20];

  if (left)  sumAmount += amountl;  
  if (right) sumAmount += amountr;  
  if (three) sumAmount += amount3;  
  if (four)  sumAmount += amount4;
  
  if (left)   countAmount++; 
  if (right)  countAmount++; 
  if (three)  countAmount++; 
  if (four)   countAmount++; 
  
  if (!countAmount) return;
  
  averageAmount = sumAmount / countAmount;

  fl_set_slider_value (adjustDCForm->amount,averageAmount);
  sprintf (tempString,"%+d",(int) averageAmount);
  fl_set_object_label (adjustDCForm->amountText,tempString);
}

/*---------------------------------------------------------------------------
| CALLBACK adjustDCOK_cb
---------------------------------------------------------------------------*/
void adjustDCOK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  
  noSampleWarning
  noRangeWarning
  zeroRangeWarning  

  int amount = (int) fl_get_slider_value (adjustDCForm->amount);

  fl_hide_form (adjustDCForm->adjustDCForm);
  updateBuffers ("Undo Adjust DC")
  waitCursorOn ();
  sample->adjustDC (amount,fl_get_sample_edit (mainForm->sample));
  waitCursorOff ();
  updateDisplayDetails ();
  updateRangeDetails ();
  updateLoopDetails ();
  fl_redraw_object (mainForm->sample);
  fl_redraw_object (mainForm->scrollBarSlider);
}

/*---------------------------------------------------------------------------
| CALLBACK adjustDCCancel_cb
---------------------------------------------------------------------------*/
void adjustDCCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (adjustDCForm->adjustDCForm);
}

/*---------------------------------------------------------------------------
| CALLBACK workspaceFrames_cb
---------------------------------------------------------------------------*/
void workspaceFrames_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  double rate = sample->getRate ();
  long frames;
  char tempString [50];
  if (!sscanf (fl_get_input (workspaceForm->frames),"%ld",&frames)) return;
  sprintf (tempString,"%.3lf",frames / rate);
  fl_set_input (workspaceForm->time,tempString);
}

/*---------------------------------------------------------------------------
| CALLBACK workspaceTime_cb
---------------------------------------------------------------------------*/
void workspaceTime_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  double rate = sample->getRate ();
  double time;
  char tempString [50];
  if (!sscanf (fl_get_input (workspaceForm->time),"%lf",&time)) return;
  sprintf (tempString,"%ld",(long) (time * rate));
  fl_set_input (workspaceForm->frames,tempString);
}

/*---------------------------------------------------------------------------
| CALLBACK workspaceOK_cb
---------------------------------------------------------------------------*/
void workspaceOK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);

  noSampleWarning
  
  double rate = sample->getRate ();
  long frames;
  double time;

  fl_hide_form (workspaceForm->workspaceForm);
  if (!sscanf (fl_get_input (workspaceForm->frames),"%ld",&frames)) return;

  // For common use often enter time without clicking OK so if
  // zero frames grab the time
  if (frames <= 0)
  {
    if (!sscanf (fl_get_input (workspaceForm->time),"%lf",&time)) return;
    frames = (long) (rate * time);
    
    // If still no frames default to one second
    if (frames <= 0) frames = (long) (rate * 1.0);
  }
  updateBuffers ("Undo Add Workspace")
  waitCursorOn ();
  sample->addWorkspace (frames,fl_get_sample_edit (mainForm->sample));
  waitCursorOff ();
  updateDisplayDetails ();
  updateRangeDetails ();
  updateLoopDetails ();
  updateNameBox ();
  fl_redraw_object (mainForm->sample);
  fl_redraw_object (mainForm->scrollBarSlider);
}

/*---------------------------------------------------------------------------
| CALLBACK workspaceCancel_cb
---------------------------------------------------------------------------*/
void workspaceCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (workspaceForm->workspaceForm);
}
/*---------------------------------------------------------------------------
| CALLBACK miscName_cb
---------------------------------------------------------------------------*/
void miscName_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK miscCopyright_cb
---------------------------------------------------------------------------*/
void miscCopyright_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK miscAuthor_cb
---------------------------------------------------------------------------*/
void miscAuthor_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK miscAnnotation_cb
---------------------------------------------------------------------------*/
void miscAnnotation_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK miscOK_cb
---------------------------------------------------------------------------*/
void miscOK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);

  noSampleWarning  

  fl_hide_form (miscForm->miscForm);
  updateBuffers ("Undo Textual Details")
  waitCursorOn ();
  sample->setName       ((char *) fl_get_input (miscForm->name));
  sample->setCopyright  ((char *) fl_get_input (miscForm->copyright));
  sample->setAuthor     ((char *) fl_get_input (miscForm->author));
  sample->setAnnotation ((char *) fl_get_input (miscForm->annotation));
  waitCursorOff ();
}

/*---------------------------------------------------------------------------
| CALLBACK miscCancel_cb
---------------------------------------------------------------------------*/
void miscCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (miscForm->miscForm);
}

/*---------------------------------------------------------------------------
| CALLBACK rateMenu_cb
---------------------------------------------------------------------------*/

void rateMenu_cb (FL_OBJECT *ob,long data)
{
  int  item;
  char tempString [50];
  
  item=fl_get_menu (ob);
  switch (item)
  {
    case 1 :
      sprintf (tempString,"%ld",8000);
      fl_set_input (rateForm->input,tempString);
      break;
    case 2 :
      sprintf (tempString,"%ld",11025);
      fl_set_input (rateForm->input,tempString);
      break;
    case 3 :
      sprintf (tempString,"%ld",16000);
      fl_set_input (rateForm->input,tempString);
      break;
    case 4 :
      sprintf (tempString,"%ld",22050);
      fl_set_input (rateForm->input,tempString);
      break;
    case 5 :
      sprintf (tempString,"%ld",32000);
      fl_set_input (rateForm->input,tempString);
      break;
    case 6 :
      sprintf (tempString,"%ld",44100);
      fl_set_input (rateForm->input,tempString);
      break;
    case 7 :
      sprintf (tempString,"%ld",48000);
      fl_set_input (rateForm->input,tempString);
      break;
    default:
      break;
  }
  fl_call_object_callback (rateForm->input);
}

/*---------------------------------------------------------------------------
| CALLBACK rateInput_cb
---------------------------------------------------------------------------*/

void rateInput_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  long rate;
  
  if (!sscanf (fl_get_input (rateForm->input),"%ld",&rate)) return;
  if (rate < 1)
  {
    rate = 1;
    fl_set_input (rateForm->input,"1");
  }

  long pvBuffer [4];
  pvBuffer [0] = AL_INPUT_RATE;
  pvBuffer [1] = rate;
  pvBuffer [2] = AL_OUTPUT_RATE;
  #ifndef LINUX
  pvBuffer [3] = AL_RATE_INPUTRATE;
  #else
  pvBuffer [3] = rate;
  #endif
  if (fl_get_sample_autoglobals (mainForm->sample))
    ALsetparams (AL_DEFAULT_DEVICE,pvBuffer,4);
  updateAudioGlobals (0,0);
}

/*---------------------------------------------------------------------------
| CALLBACK rateOK_cb
---------------------------------------------------------------------------*/
void rateOK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  long rate;

  noSampleWarning  

  fl_hide_form (rateForm->rateForm);
  if (!sscanf (fl_get_input (rateForm->input),"%ld",&rate)) return;
  if (rate < 1) rate = 1;
  updateBuffers ("Undo Sample Rate")
  waitCursorOn ();
  sample->setRate ((double) rate);
  waitCursorOff ();
  updateSample (sample);
}

/*---------------------------------------------------------------------------
| CALLBACK rateCancel_cb
---------------------------------------------------------------------------*/
void rateCancel_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  long rate = (long) sample->getRate ();
  
  fl_hide_form (rateForm->rateForm);
  updateSample (sample);
}

/*---------------------------------------------------------------------------
| CALLBACK ampSlider_cb
---------------------------------------------------------------------------*/
void ampSlider_cb (FL_OBJECT *ob,long data)
{
  long value = (long) fl_get_slider_value (ob);
  char tempString [20];
  sprintf (tempString,"%ld",value);
  fl_set_input (ampForm->input,tempString);
}

/*---------------------------------------------------------------------------
| CALLBACK ampInput_cb
---------------------------------------------------------------------------*/
void ampInput_cb (FL_OBJECT *ob,long data)
{
  long value;
  if (!sscanf (fl_get_input (ob),"%ld",&value)) return;
  if (value > 200) fl_set_slider_value (ampForm->slider,200.0);
  else if (value < 0) fl_set_slider_value (ampForm->slider,0.0);
  else fl_set_slider_value (ampForm->slider,(double)value);
}

/*---------------------------------------------------------------------------
| CALLBACK ampOK_cb
---------------------------------------------------------------------------*/
void ampOK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);

  noSampleWarning
  noRangeWarning
  zeroRangeWarning  

  long value;

  fl_hide_form (ampForm->ampForm);
  if (!sscanf (fl_get_input (ampForm->input),"%ld",&value)) return;
  updateBuffers ("Undo Amplitude")
  waitCursorOn ();
  if (fl_get_menu_item_mode (mainForm->ampMenu,8) == FL_PUP_BOX)
    sample->percentage
      ((double)value,(double)value,(double)value,(double)value,FALSE,
      fl_get_sample_edit (mainForm->sample));
  else
    sample->percentage
      ((double)value,(double)value,(double)value,(double)value,TRUE,
      fl_get_sample_edit (mainForm->sample));
  waitCursorOff ();
  fl_redraw_object (mainForm->sample);
}

/*---------------------------------------------------------------------------
| CALLBACK ampCancel_cb
---------------------------------------------------------------------------*/
void ampCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (ampForm->ampForm);
}

/*---------------------------------------------------------------------------
| CALLBACK balanceLeftSlider_cb
---------------------------------------------------------------------------*/
void balanceLeftSlider_cb (FL_OBJECT *ob,long data)
{
  long value = (long) fl_get_slider_value (ob);
  char tempString [20];
  sprintf (tempString,"%ld",value);
  fl_set_input (balanceForm->leftInput,tempString);
  if (fl_get_choice (balanceForm->mode) == 1)
  {
    fl_set_slider_value (balanceForm->rightSlider,(double)(100 - value));
    sprintf (tempString,"%ld",100 - value);
    fl_set_input (balanceForm->rightInput,tempString);
    fl_set_slider_value (balanceForm->slider,(double)(100 - value));
  }
}

/*---------------------------------------------------------------------------
| CALLBACK balanceLeftInput_cb
---------------------------------------------------------------------------*/
void balanceLeftInput_cb (FL_OBJECT *ob,long data)
{
  long value;
  char tempString [20];
  if (!sscanf (fl_get_input (ob),"%ld",&value)) return;
  if (fl_get_choice (balanceForm->mode) == 1)
  {
    if (value < 0) value = 0;
    else if (value > 100) value = 100;
    
    sprintf (tempString,"%ld",value);
    fl_set_input (balanceForm->leftInput,tempString);
    fl_set_slider_value (balanceForm->leftSlider,(double)value);
    sprintf (tempString,"%ld",100 - value);
    fl_set_input (balanceForm->rightInput,tempString);
    fl_set_slider_value (balanceForm->rightSlider,(double)(100 - value));
    fl_set_slider_value (balanceForm->slider,(double)(100 - value));
  }
  else
  {
    if (value > 100)
      fl_set_slider_value (balanceForm->leftSlider,100.0);
    else if (value < 0)
      fl_set_slider_value (balanceForm->leftSlider,0.0);
    else
      fl_set_slider_value (balanceForm->leftSlider,(double)value);
  }  
}

/*---------------------------------------------------------------------------
| CALLBACK balanceRightSlider_cb
---------------------------------------------------------------------------*/
void balanceRightSlider_cb (FL_OBJECT *ob,long data)
{
  long value = (long) fl_get_slider_value (ob);
  char tempString [20];
  sprintf (tempString,"%ld",value);
  fl_set_input (balanceForm->rightInput,tempString);
  if (fl_get_choice (balanceForm->mode) == 1)
  {
    fl_set_slider_value (balanceForm->leftSlider,(double)(100 - value));
    sprintf (tempString,"%ld",100 - value);
    fl_set_input (balanceForm->leftInput,tempString);
    fl_set_slider_value (balanceForm->slider,(double)(value));
  }
}

/*---------------------------------------------------------------------------
| CALLBACK balanceRightInput_cb
---------------------------------------------------------------------------*/
void balanceRightInput_cb (FL_OBJECT *ob,long data)
{
  long value;
  char tempString [20];
  if (!sscanf (fl_get_input (ob),"%ld",&value)) return;
  if (fl_get_choice (balanceForm->mode) == 1)
  {
    if (value < 0) value = 0;
    else if (value > 100) value = 100;
    
    sprintf (tempString,"%ld",value);
    fl_set_input (balanceForm->rightInput,tempString);
    fl_set_slider_value (balanceForm->rightSlider,(double)value);
    sprintf (tempString,"%ld",100 - value);
    fl_set_input (balanceForm->leftInput,tempString);
    fl_set_slider_value (balanceForm->leftSlider,(double)(100 - value));
    fl_set_slider_value (balanceForm->slider,(double)value);
  }
  else
  {
    if (value > 100)
      fl_set_slider_value (balanceForm->rightSlider,100.0);
    else if (value < 0)
      fl_set_slider_value (balanceForm->rightSlider,0.0);
    else
      fl_set_slider_value (balanceForm->rightSlider,(double)value);
  }  
}

/*---------------------------------------------------------------------------
| CALLBACK balanceSlider_cb
---------------------------------------------------------------------------*/
void balanceSlider_cb (FL_OBJECT *ob,long data)
{
  char tempString [20];
  long value = (long) fl_get_slider_value (ob);
  fl_set_slider_value (balanceForm->leftSlider,(double)(100 - value));
  sprintf (tempString,"%ld",100 - value);
  fl_set_input (balanceForm->leftInput,tempString);
  fl_set_slider_value (balanceForm->rightSlider,(double)(value));
  sprintf (tempString,"%ld",value);
  fl_set_input (balanceForm->rightInput,tempString);
}

/*---------------------------------------------------------------------------
| CALLBACK balanceMode_cb
---------------------------------------------------------------------------*/
void balanceMode_cb (FL_OBJECT *ob,long data)
{
  if (fl_get_choice (ob) == 1)
  {
    fl_show_object (balanceForm->balanceControls);
    fl_call_object_callback (balanceForm->leftInput);
  }
  else
  {
    fl_hide_object (balanceForm->balanceControls);
  }
}

/*---------------------------------------------------------------------------
| CALLBACK balanceQuadro_cb
---------------------------------------------------------------------------*/
void balanceQuadro_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK balanceOK_cb
---------------------------------------------------------------------------*/
void balanceOK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);

  noSampleWarning
  noRangeWarning
  zeroRangeWarning  
  noStereoWarning

  double valuel;
  double valuer;

  fl_hide_form (balanceForm->balanceForm);
  if (!sscanf (fl_get_input (balanceForm->leftInput),"%lf",&valuel)) return;
  if (!sscanf (fl_get_input (balanceForm->rightInput),"%lf",&valuer)) return;
  if (sample->getChannels () == 2)
  {
    updateBuffers ("Undo Balance (stereo)")
    waitCursorOn ();
    if (fl_get_menu_item_mode (mainForm->ampMenu,8) == FL_PUP_BOX)
      sample->percentage (valuel,valuer,100.0,100.0,FALSE,2);
    else
      sample->percentage (valuel,valuer,100.0,100.0,TRUE,2);
    waitCursorOff ();
    fl_redraw_object (mainForm->sample);
  }
  else if (sample->getChannels () == 4)
  {
    int left  = fl_get_choice (balanceForm->quadroLeft);
    int right = fl_get_choice (balanceForm->quadroRight);
    double values [4] = { 100.0,100.0,100.0,100.0 };
    values [left - 1]  = valuel;
    values [right - 1] = valuer;
    updateBuffers ("Undo Balance (quadro)")
    waitCursorOn ();
    if (fl_get_menu_item_mode (mainForm->ampMenu,8) == FL_PUP_BOX)
      sample->percentage
        (values [0],values [1],values [2],values [3],FALSE,2);
    else
      sample->percentage
        (values [0],values [1],values [2],values [3],TRUE,2);
    waitCursorOff ();
    fl_redraw_object (mainForm->sample);
  }
}

/*---------------------------------------------------------------------------
| CALLBACK balanceCancel_cb
---------------------------------------------------------------------------*/
void balanceCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (balanceForm->balanceForm);
}

/*---------------------------------------------------------------------------
| CALLBACK bounceButton_cb
---------------------------------------------------------------------------*/
void bounceButton_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK bounceOK_cb
---------------------------------------------------------------------------*/
void bounceOK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);

  noSampleWarning
  noRangeWarning
  zeroRangeWarning  
  noStereoWarning

  fl_hide_form (bounceForm->bounceForm);
  if (fl_get_button (bounceForm->LtoR))
  {
    updateBuffers ("Undo Bounce (L->R)")
    waitCursorOn ();
    sample->bounce (0);
    waitCursorOff ();
  }
  else
  {
    updateBuffers ("Undo Bounce (R->L)")
    waitCursorOn ();
    sample->bounce (1);
    waitCursorOff ();
  }
  fl_redraw_object (mainForm->sample);
}

/*---------------------------------------------------------------------------
| CALLBACK bounce4Button_cb
---------------------------------------------------------------------------*/
void bounce4Button_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK bounce4OK_cb
---------------------------------------------------------------------------*/
void bounce4OK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);

  noSampleWarning
  noRangeWarning
  zeroRangeWarning  
  noStereoWarning

  int mode = 0;
  
  fl_hide_form (bounce4Form->bounce4Form);
  if (fl_get_button (bounce4Form->oneLButton))
    mode = mode | 0;
  else if (fl_get_button (bounce4Form->twoLButton))
    mode = mode | 1;
  else if (fl_get_button (bounce4Form->threeLButton))
    mode = mode | 2;
  else if (fl_get_button (bounce4Form->fourLButton))
    mode = mode | 3;

  if (fl_get_button (bounce4Form->oneRButton))
    mode = mode | (0 << 2);
  else if (fl_get_button (bounce4Form->twoRButton))
    mode = mode | (1 << 2);
  else if (fl_get_button (bounce4Form->threeRButton))
    mode = mode | (2 << 2);
  else if (fl_get_button (bounce4Form->fourRButton))
    mode = mode | (3 << 2);
  
  if (fl_get_button (bounce4Form->LtoR))
    mode = mode | (0 << 4);
  else if (fl_get_button (bounce4Form->RtoL))
    mode = mode | (1 << 4);
  
  updateBuffers ("Undo Bounce (quadro)")
  waitCursorOn ();
  sample->bounce (mode);
  waitCursorOff ();
  fl_redraw_object (mainForm->sample);
}

/*---------------------------------------------------------------------------
| CALLBACK bounce4Cancel_cb
---------------------------------------------------------------------------*/
void bounce4Cancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (bounce4Form->bounce4Form);
}

/*---------------------------------------------------------------------------
| CALLBACK bounceCancel_cb
---------------------------------------------------------------------------*/
void bounceCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (bounceForm->bounceForm);
}

/*---------------------------------------------------------------------------
| FUNCTION initialiseRampForm
---------------------------------------------------------------------------*/
void initialiseRampForm ()
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  rampNumber       = 2;
  rampFrames [0]   = 0;
  if (sample->getRangeValid ())
    rampFrames [1]   = sample->getRangeEnd () - sample->getRangeStart ();
  else
    rampFrames [1]   = 0;
  rampAmounts [0]  = 100.0;
  rampAmounts [1]  = 100.0;
  fl_set_input (rampForm->pointNo,"1");
  fl_set_choice (rampForm->mode,1);
  fl_show_object (rampForm->ampLabels);
  fl_hide_object (rampForm->balanceLabels);
  fl_hide_object (rampForm->quadroControls);
  fl_set_choice (rampForm->quadroLeft,1);
  fl_set_choice (rampForm->quadroRight,2);
}

/*---------------------------------------------------------------------------
| FUNCTION updateRampForm
---------------------------------------------------------------------------*/
void updateRampForm ()
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  double rate = sample->getRate ();
  long number;
  char tempString [20];
  
  if (!sscanf (fl_get_input (rampForm->pointNo),"%ld",&number)) return;
  if (number < 1) number = 1;
  else if (number > rampNumber) number = rampNumber;
  
  sprintf (tempString,"%ld",number);
  fl_set_input (rampForm->pointNo,tempString);
  fl_redraw_object (rampForm->pointNo);
  sprintf (tempString,"%ld",(long) rampAmounts [number-1]);
  fl_set_input (rampForm->pointInput,tempString);
  fl_redraw_object (rampForm->pointInput);
  
  if (rampAmounts [number-1] < 0.0)
    fl_set_slider_value (rampForm->pointSlider,0.0);
  else if (rampAmounts [number-1] > 100.0)
    fl_set_slider_value (rampForm->pointSlider,100.0);
  else
    fl_set_slider_value (rampForm->pointSlider,rampAmounts [number-1]);
  
  sprintf (tempString,"%ld",rampFrames [number-1]);
  fl_set_input (rampForm->pointFrames,tempString);
  fl_redraw_object (rampForm->pointFrames);
  sprintf (tempString,"%.3lf",(double) rampFrames [number-1] / rate);
  fl_set_input (rampForm->pointTime,tempString);
  fl_redraw_object (rampForm->pointTime);
}

/*---------------------------------------------------------------------------
| FUNCTION updateRampPlot
---------------------------------------------------------------------------*/
void updateRampPlot ()
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  double rate = sample->getRate ();
  long length;
  long i;
  char tempString [20];
  
  float x [MAXRAMPPOINTS];
  float y [MAXRAMPPOINTS];
  
  if (sample->getRangeValid ())
    length = sample->getRangeEnd () - sample->getRangeStart ();
  else
    length = 0;
  
  // Check validity of current data
  if (rampFrames [rampNumber - 1] != length)
  {
    if (rampFrames [rampNumber - 1] == 0)
    {
      for (i=0; i<rampNumber; i++)
      {
        rampFrames [i] = (long)(i * length / (rampNumber - 1));
        if (rampFrames [i] > length) rampFrames [i] = length;
      }
    }
    else
    {
      // Scale data
      double scale = (double) length / rampFrames [rampNumber - 1];
      for (i=0; i<rampNumber; i++)
      {
        rampFrames [i] = (long) (rampFrames [i] * scale);
        if (rampFrames [i] > length) rampFrames [i] = length;
      }
    }
  }

  rampFrames [0] = 0;
  rampFrames [rampNumber - 1] = length;

  for (i=0; i<rampNumber; i++)
  {
    if (length == 0) x [i] = 100.0 * i / (rampNumber - 1);
    else x [i] = (float) 100.0 * rampFrames [i] / length;

    if (rampAmounts [i] < 0.0) y [i] = 0.0;
    else if (rampAmounts [i] > 100.0) y [i] = 100.0;
    else y [i] = (float) rampAmounts [i];
  }

  sprintf (tempString,"%ld",length);
  fl_set_object_label (rampForm->framesText,tempString);
  fl_redraw_object (rampForm->framesText);
  sprintf (tempString,"%.3lf",length / rate);
  fl_set_object_label (rampForm->timeText,tempString);
  fl_redraw_object (rampForm->timeText);

  fl_set_xyplot_data (rampForm->plot,x,y,rampNumber,"","","");
  fl_redraw_object (rampForm->plot);
}

/*---------------------------------------------------------------------------
| CALLBACK rampPlot_cb
---------------------------------------------------------------------------*/
void rampPlot_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  char  tempString [20];
  long  length;
  long  number;
  float x;
  float y;
  int   n;
  
  if (sample->getRangeValid ())
    length = sample->getRangeEnd () - sample->getRangeStart ();
  else
    length = 0;
  
  if (!sscanf (fl_get_input (rampForm->pointNo),"%ld",&number)) return;

  fl_get_xyplot (ob,&x,&y,&n);

  if (x < 0.0) x = 0.0;
  else if (x > 100.0) x = 100.0;
  if (y < 0.0) y = 0.0;
  else if (y > 100.0) y = 100.0;
  
  if (n == 0) rampFrames [n] = 0;
  else if (n == rampNumber - 1) rampFrames [n] = length;
  else rampFrames [n]  = (long) (length * x / 100.0);
  rampAmounts [n] = (double) y;
  
  sprintf (tempString,"%ld",n+1);
  fl_set_input (rampForm->pointNo,tempString);
  
  updateRampForm ();
  updateRampPlot ();
}

/*---------------------------------------------------------------------------
| CALLBACK rampPointSlider_cb
---------------------------------------------------------------------------*/
void rampPointSlider_cb (FL_OBJECT *ob,long data)
{
  long value = (long) fl_get_slider_value (ob);
  long number;
  char tempString [20];
  
  if (!sscanf (fl_get_input (rampForm->pointNo),"%ld",&number)) return;
  rampAmounts [number-1] = (double) value;
  sprintf (tempString,"%ld",value);
  fl_set_input (rampForm->pointInput,tempString);
  updateRampPlot ();
}

/*---------------------------------------------------------------------------
| CALLBACK rampPointInput_cb
---------------------------------------------------------------------------*/
void rampPointInput_cb (FL_OBJECT *ob,long data)
{
  long value;
  long number;

  if (!sscanf (fl_get_input (rampForm->pointInput),"%ld",&value)) return;
  if (!sscanf (fl_get_input (rampForm->pointNo),"%ld",&number)) return;
  rampAmounts [number-1] = (double) value;
  fl_set_slider_value (rampForm->pointSlider,(double) value);
  updateRampPlot ();
}

/*---------------------------------------------------------------------------
| CALLBACK rampPointNo_cb
---------------------------------------------------------------------------*/
void rampPointNo_cb (FL_OBJECT *ob,long data)
{
  long number;
  char tempString [20];
  if (!sscanf (fl_get_input (rampForm->pointNo),"%ld",&number)) return;
  if (number < 1) number = 1;
  else if (number > rampNumber) number = rampNumber;
  
  sprintf (tempString,"%ld",number);
  fl_set_input (rampForm->pointNo,tempString);
  updateRampForm ();
  updateRampPlot ();
}

/*---------------------------------------------------------------------------
| CALLBACK rampPointFrames_cb
---------------------------------------------------------------------------*/
void rampPointFrames_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  double rate = sample->getRate ();
  long length;
  long frame;
  long number;
  char tempString [20];
  
  if (sample->getRangeValid ())
    length = sample->getRangeEnd () - sample->getRangeStart ();
  else
    length = 0;

  if (!sscanf (fl_get_input (rampForm->pointNo),"%ld",&number)) return;
  if (number == 1)
    frame = 0;
  else if (number == rampNumber)
    frame = length;
  else
  {
    if (!sscanf (fl_get_input (rampForm->pointFrames),"%ld",&frame)) return;
    if (frame < 0) frame = 0;
    else if (frame > length) frame = length;
  }
  
  rampFrames [number-1] = frame;
  updateRampForm ();
  updateRampPlot ();
}

/*---------------------------------------------------------------------------
| CALLBACK rampPointTime_cb
---------------------------------------------------------------------------*/
void rampPointTime_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  double rate = sample->getRate ();
  long length;
  long frame;
  double time;
  long number;
  char tempString [20];
  
  if (sample->getRangeValid ())
    length = sample->getRangeEnd () - sample->getRangeStart ();
  else
    length = 0;

  if (!sscanf (fl_get_input (rampForm->pointNo),"%ld",&number)) return;
  if (number == 1)
    frame = 0;
  else if (number == rampNumber)
    frame = length;
  else
  {
    if (!sscanf (fl_get_input (rampForm->pointTime),"%lf",&time)) return;
    frame = (long) (time * rate);
    if (frame < 0) frame = 0;
    else if (frame > length) frame = length;
  }
  
  rampFrames [number-1] = frame;
  updateRampForm ();
  updateRampPlot ();
}

/*---------------------------------------------------------------------------
| CALLBACK rampPrev_cb
---------------------------------------------------------------------------*/
void rampPrev_cb (FL_OBJECT *ob,long data)
{
  long number;
  char tempString [20];

  if (!sscanf (fl_get_input (rampForm->pointNo),"%ld",&number)) return;
  number--;
  if (number < 1) number = 1;
  else if (number > rampNumber) number = rampNumber;

  sprintf (tempString,"%ld",number);
  fl_set_input (rampForm->pointNo,tempString);
  updateRampForm ();
  updateRampPlot ();
}

/*---------------------------------------------------------------------------
| CALLBACK rampNext_cb
---------------------------------------------------------------------------*/
void rampNext_cb (FL_OBJECT *ob,long data)
{
  long number;
  char tempString [20];

  if (!sscanf (fl_get_input (rampForm->pointNo),"%ld",&number)) return;
  number++;
  if (number < 1) number = 1;
  else if (number > rampNumber) number = rampNumber;

  sprintf (tempString,"%ld",number);
  fl_set_input (rampForm->pointNo,tempString);
  updateRampForm ();
  updateRampPlot ();
}

/*---------------------------------------------------------------------------
| CALLBACK rampAdd_cb
---------------------------------------------------------------------------*/
void rampAdd_cb (FL_OBJECT *ob,long data)
{
  if (rampNumber < MAXRAMPPOINTS)
  {
    DPSample *sample = fl_get_sample (mainForm->sample);
    long length;
    char tempString [20];
    if (sample->getRangeValid ())
      length = sample->getRangeEnd () - sample->getRangeStart ();
    else
      length = 0;
    rampNumber++;
    rampFrames [rampNumber-1]  = length;
    rampAmounts [rampNumber-1] =
    (double)(((long)(rampAmounts [rampNumber-2] + 40.0)) % 101);
    sprintf (tempString,"%ld",rampNumber);
    fl_set_input (rampForm->pointNo,tempString);
    updateRampForm ();
    updateRampPlot ();
  }
}

/*---------------------------------------------------------------------------
| CALLBACK rampDel_cb
---------------------------------------------------------------------------*/
void rampDel_cb (FL_OBJECT *ob,long data)
{
  if (rampNumber > 2)
  {
    DPSample *sample = fl_get_sample (mainForm->sample);
    long length;
    long number;
    char tempString [20];
    if (!sscanf (fl_get_input (rampForm->pointNo),"%ld",&number)) return;
    if (sample->getRangeValid ())
      length = sample->getRangeEnd () - sample->getRangeStart ();
    else
      length = 0;
    for (long i=number-1; i<rampNumber-1; i++)
    {
      rampFrames  [i] = rampFrames  [i+1];
      rampAmounts [i] = rampAmounts [i+1];
    }
    if (number == rampNumber)
    {
      number--;
      if (number < 1) number = 1;
      else if (number > rampNumber-1) number = rampNumber-1;
      sprintf (tempString,"%ld",number);
      fl_set_input (rampForm->pointNo,tempString);
    }
    rampNumber--;
    rampFrames [0] = 0;
    rampFrames [rampNumber-1] = length;
    updateRampForm ();
    updateRampPlot ();
  }
}

/*---------------------------------------------------------------------------
| CALLBACK rampQuadro_cb
---------------------------------------------------------------------------*/
void rampQuadro_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK rampMode_cb
---------------------------------------------------------------------------*/
void rampMode_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  if (fl_get_choice (ob) == 1)
  {
    // Amplitude mode
    fl_show_object (rampForm->ampLabels);
    fl_hide_object (rampForm->balanceLabels);
    fl_hide_object (rampForm->quadroControls);
  }
  else
  {
    // Balance mode
    fl_set_choice (ob,1);
    noStereoWarning
    fl_set_choice (ob,2);
    fl_hide_object (rampForm->ampLabels);
    fl_show_object (rampForm->balanceLabels);
    if (sample->getChannels () == 2)
      fl_hide_object (rampForm->quadroControls);
    else
      fl_show_object (rampForm->quadroControls);
  }
}

/*---------------------------------------------------------------------------
| CALLBACK rampOK_cb
---------------------------------------------------------------------------*/
void rampOK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);

  noSampleWarning
  noRangeWarning
  zeroRangeWarning  

  if (fl_get_choice (rampForm->mode) == 1)
  {
    fl_hide_form (rampForm->rampForm);
    updateBuffers ("Undo Ramp (amplitude)")
    waitCursorOn ();
    if (fl_get_menu_item_mode (mainForm->ampMenu,8) == FL_PUP_BOX)
      sample->draw (rampAmounts,rampFrames,rampNumber,FALSE,
        fl_get_sample_edit (mainForm->sample));
    else
      sample->draw (rampAmounts,rampFrames,rampNumber,TRUE,
        fl_get_sample_edit (mainForm->sample));
    waitCursorOff ();
    fl_redraw_object (mainForm->sample);
  }
  else
  {
    int i;
    int mode;
    noStereoWarning
    fl_hide_form (rampForm->rampForm);
    updateBuffers ("Undo Ramp (balance)")
    waitCursorOn ();
    
    // Correct amounts
    for (i=0; i<rampNumber; i++)
      if (rampAmounts [i] > 100.0)
        rampAmounts [i] = 100.0;
      else if (rampAmounts [i] < 0.0)
        rampAmounts [i] = 0.0;

    // Do right channel
    if (sample->getChannels () == 2) mode = 1;
    else mode = fl_get_choice (rampForm->quadroRight) - 1;
    if (mode > 1) mode++;
    if (fl_get_menu_item_mode (mainForm->ampMenu,8) == FL_PUP_BOX)
      sample->draw (rampAmounts,rampFrames,rampNumber,FALSE,mode);
    else
      sample->draw (rampAmounts,rampFrames,rampNumber,TRUE,mode);
    
    // Invert amounts
    for (i=0; i<rampNumber; i++)
      rampAmounts [i] = 100.0 - rampAmounts [i];
    
    // Do left channel
    if (sample->getChannels () == 2) mode = 0;
    else mode = fl_get_choice (rampForm->quadroLeft) - 1;
    if (mode > 1) mode++;
    if (fl_get_menu_item_mode (mainForm->ampMenu,8) == FL_PUP_BOX)
      sample->draw (rampAmounts,rampFrames,rampNumber,FALSE,mode);
    else
      sample->draw (rampAmounts,rampFrames,rampNumber,TRUE,mode);
    waitCursorOff ();
    fl_redraw_object (mainForm->sample);
  }
}

/*---------------------------------------------------------------------------
| CALLBACK rampCancel_cb
---------------------------------------------------------------------------*/
void rampCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (rampForm->rampForm);
}

/*---------------------------------------------------------------------------
| CALLBACK resampleRateChange_cb
---------------------------------------------------------------------------*/
void resampleRateChange_cb (FL_OBJECT *ob,long data)
{
  char tempString [20];
  if (!sscanf (fl_get_input (ob),"%lf",&rateChange)) return;
  if (rateChange < 1.0) rateChange = 1.0;
  
  if (fl_get_choice (resampleForm->mode) == 1)
  {
    pitchChange = 100.0 * rateChange / framesChange;
    updateResampleForm (1);
  }
  else updateResampleForm (1);
}

/*---------------------------------------------------------------------------
| CALLBACK resampleRateChangeSlider_cb
---------------------------------------------------------------------------*/
void resampleRateChangeSlider_cb (FL_OBJECT *ob,long data)
{
  rateChange = fl_get_slider_value (ob);
  char tempString [20];
  sprintf (tempString,"%.3lf",rateChange);
  fl_set_input (resampleForm->rateChange,tempString);
  fl_call_object_callback (resampleForm->rateChange);
}

/*---------------------------------------------------------------------------
| CALLBACK resampleNewRateMenu_cb
---------------------------------------------------------------------------*/
void resampleNewRateMenu_cb (FL_OBJECT *ob,long data)
{
  int  item;
  char tempString [50];
  
  item=fl_get_menu (ob);
  switch (item)
  {
    case 1 :
      sprintf (tempString,"%ld",8000);
      fl_set_input (resampleForm->newRate,tempString);
      break;
    case 2 :
      sprintf (tempString,"%ld",11025);
      fl_set_input (resampleForm->newRate,tempString);
      break;
    case 3 :
      sprintf (tempString,"%ld",16000);
      fl_set_input (resampleForm->newRate,tempString);
      break;
    case 4 :
      sprintf (tempString,"%ld",22050);
      fl_set_input (resampleForm->newRate,tempString);
      break;
    case 5 :
      sprintf (tempString,"%ld",32000);
      fl_set_input (resampleForm->newRate,tempString);
      break;
    case 6 :
      sprintf (tempString,"%ld",44100);
      fl_set_input (resampleForm->newRate,tempString);
      break;
    case 7 :
      sprintf (tempString,"%ld",48000);
      fl_set_input (resampleForm->newRate,tempString);
      break;
    default:
      break;
  }
  fl_call_object_callback (resampleForm->newRate);
}

/*---------------------------------------------------------------------------
| CALLBACK resampleNewRate_cb
---------------------------------------------------------------------------*/
void resampleNewRate_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  long currentRate = (long) sample->getRate ();
  long newRate;
  char tempString [20];
  
  if (!sscanf (fl_get_input (ob),"%ld",&newRate)) return;
  if (newRate < 1) newRate = 1;
  rateChange = 100.0 * (double) newRate / currentRate;
  sprintf (tempString,"%.3lf",rateChange);
  fl_set_input (resampleForm->rateChange,tempString);
  fl_call_object_callback (resampleForm->rateChange);
}

/*---------------------------------------------------------------------------
| CALLBACK resampleFramesChange_cb
---------------------------------------------------------------------------*/
void resampleFramesChange_cb (FL_OBJECT *ob,long data)
{
  char tempString [20];
  if (!sscanf (fl_get_input (ob),"%lf",&framesChange)) return;
  if (framesChange < 1.0) framesChange = 1.0;
  
  if (fl_get_choice (resampleForm->mode) == 1)
  {
    pitchChange = 100.0 * rateChange / framesChange;
  }
  updateResampleForm (0);
}

/*---------------------------------------------------------------------------
| CALLBACK resampleFramesChangeSlider_cb
---------------------------------------------------------------------------*/
void resampleFramesChangeSlider_cb (FL_OBJECT *ob,long data)
{
  framesChange = fl_get_slider_value (ob);
  char tempString [20];
  sprintf (tempString,"%.3lf",framesChange);
  fl_set_input (resampleForm->framesChange,tempString);
  fl_call_object_callback (resampleForm->framesChange);
}

/*---------------------------------------------------------------------------
| CALLBACK resampleNewFrames_cb
---------------------------------------------------------------------------*/
void resampleNewFrames_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  long currentFrames;
  if (sample->getRangeValid ())
    currentFrames = sample->getRangeEnd () - sample->getRangeStart ();
  else
    currentFrames = 0;
  if (currentFrames == 0) currentFrames = 1;
  long newFrames;
  char tempString [20];
  
  if (!sscanf (fl_get_input (ob),"%ld",&newFrames)) return;
  if (newFrames < 1) newFrames = 1;
  framesChange = 100.0 * (double) newFrames / currentFrames;
  sprintf (tempString,"%.3lf",framesChange);
  fl_set_input (resampleForm->framesChange,tempString);
  fl_call_object_callback (resampleForm->framesChange);
}

/*---------------------------------------------------------------------------
| CALLBACK resampleTimeChange_cb
---------------------------------------------------------------------------*/
void resampleTimeChange_cb (FL_OBJECT *ob,long data)
{
  char tempString [20];
  if (!sscanf (fl_get_input (ob),"%lf",&timeChange)) return;
  if (timeChange < 1.0) timeChange = 1.0;

  updateResampleForm (1);
  if (fl_get_choice (resampleForm->mode) == 1)
  {
    pitchChange = 100.0 * rateChange / framesChange;
  }
  updateResampleForm (1);
}

/*---------------------------------------------------------------------------
| CALLBACK resampleTimeChangeSlider_cb
---------------------------------------------------------------------------*/
void resampleTimeChangeSlider_cb (FL_OBJECT *ob,long data)
{
  timeChange = fl_get_slider_value (ob);
  char tempString [20];
  sprintf (tempString,"%.3lf",timeChange);
  fl_set_input (resampleForm->timeChange,tempString);
  fl_call_object_callback (resampleForm->timeChange);
}

/*---------------------------------------------------------------------------
| CALLBACK resampleNewTime_cb
---------------------------------------------------------------------------*/
void resampleNewTime_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  long currentRate = (long) sample->getRate ();
  long currentFrames;
  double currentTime;
  if (sample->getRangeValid ())
    currentFrames = sample->getRangeEnd () - sample->getRangeStart ();
  else
    currentFrames = 0;
  if (currentFrames == 0) currentFrames = 1;
  
  currentTime = (double) currentFrames / currentRate;
  double newTime;
  char tempString [20];
  
  if (!sscanf (fl_get_input (ob),"%lf",&newTime)) return;
  if (newTime < (1.0 / currentRate)) newTime = 1.0 / currentRate;
  timeChange = 100.0 * newTime / currentTime;
  sprintf (tempString,"%.3lf",timeChange);
  fl_set_input (resampleForm->timeChange,tempString);
  fl_call_object_callback (resampleForm->timeChange);
}

/*---------------------------------------------------------------------------
| CALLBACK resamplePitchChange_cb
---------------------------------------------------------------------------*/
void resamplePitchChange_cb (FL_OBJECT *ob,long data)
{
  char tempString [20];
  if (!sscanf (fl_get_input (ob),"%lf",&pitchChange)) return;
  if (pitchChange < 1.0) pitchChange = 1.0;

  if (fl_get_choice (resampleForm->mode) == 1)
  {
    framesChange = 100.0 * rateChange / pitchChange;
  }
  updateResampleForm (0);
}

/*---------------------------------------------------------------------------
| CALLBACK resamplePitchChangeSlider_cb
---------------------------------------------------------------------------*/
void resamplePitchChangeSlider_cb (FL_OBJECT *ob,long data)
{
  pitchChange = fl_get_slider_value (ob);
  char tempString [20];
  sprintf (tempString,"%.3lf",pitchChange);
  fl_set_input (resampleForm->pitchChange,tempString);
  fl_call_object_callback (resampleForm->pitchChange);
}

/*---------------------------------------------------------------------------
| CALLBACK resamplePitchChangeBigSemi_cb
---------------------------------------------------------------------------*/
void resamplePitchChangeBigSemi_cb (FL_OBJECT *ob,long data)
{
  double bigSemiChange;
  char tempString [20];
  if (!sscanf (fl_get_input (ob),"%lf",&bigSemiChange)) return;
  if (bigSemiChange < -127.0)
    bigSemiChange = -127.0;
  else if (bigSemiChange > 127.0)
    bigSemiChange = 127.0;
  pitchChange = 100.0 * pow (2.0,(bigSemiChange / 12.0));
  sprintf (tempString,"%.3lf",pitchChange);
  fl_set_input (resampleForm->pitchChange,tempString);
  fl_call_object_callback (resampleForm->pitchChange);
}

/*---------------------------------------------------------------------------
| CALLBACK resamplePitchChangeDir_cb
---------------------------------------------------------------------------*/
void resamplePitchChangeDir_cb (FL_OBJECT *ob,long data)
{
  double bigSemiChange;
  char tempString [20];
  int semi;
  int oct;
  int cent;
  int dir;
  if (!sscanf (fl_get_input (resampleForm->pitchChangeSemi),"%d",&semi)) return;
  if (!sscanf (fl_get_input (resampleForm->pitchChangeOct),"%d",&oct)) return;
  if (!sscanf (fl_get_input (resampleForm->pitchChangeCent),"%d",&cent)) return;
  if (fl_get_choice (ob) == 1) dir = 1;
  else dir = -1;
  semiToBigSemi (&bigSemiChange,oct,semi,cent,dir);
  if (bigSemiChange > 127.0)
    bigSemiChange = 127.0;
  else if (bigSemiChange < -127.0)
    bigSemiChange = -127.0;
  sprintf (tempString,"%.3lf",bigSemiChange);
  fl_set_input (resampleForm->pitchChangeBigSemi,tempString);
  fl_call_object_callback (resampleForm->pitchChangeBigSemi);
}

/*---------------------------------------------------------------------------
| CALLBACK resamplePitchChangeSemi_cb
---------------------------------------------------------------------------*/
void resamplePitchChangeSemi_cb (FL_OBJECT *ob,long data)
{
  fl_call_object_callback (resampleForm->pitchChangeDir);
}

/*---------------------------------------------------------------------------
| CALLBACK resamplePitchChangeOct_cb
---------------------------------------------------------------------------*/
void resamplePitchChangeOct_cb (FL_OBJECT *ob,long data)
{
  fl_call_object_callback (resampleForm->pitchChangeDir);
}

/*---------------------------------------------------------------------------
| CALLBACK resamplePitchChangeCent_cb
---------------------------------------------------------------------------*/
void resamplePitchChangeCent_cb (FL_OBJECT *ob,long data)
{
  fl_call_object_callback (resampleForm->pitchChangeDir);
}

/*---------------------------------------------------------------------------
| CALLBACK resampleCurrentSelect_cb
---------------------------------------------------------------------------*/
void resampleCurrentSelect_cb (FL_OBJECT *ob,long data)
{
  fl_set_choice (pitchForm->mode,1);
  initialisePitchForm ();
  updatePitchForm ();
  fl_deactivate_form (resampleForm->resampleForm);
  fl_show_form (pitchForm->pitchForm,
  FL_PLACE_MOUSE,FL_FULLBORDER,"Pitch Select - Current");
}

/*---------------------------------------------------------------------------
| CALLBACK resampleNewSelect_cb
---------------------------------------------------------------------------*/
void resampleNewSelect_cb (FL_OBJECT *ob,long data)
{
  fl_set_choice (pitchForm->mode,2);
  initialisePitchForm ();
  updatePitchForm ();
  fl_deactivate_form (resampleForm->resampleForm);
  fl_show_form (pitchForm->pitchForm,
  FL_PLACE_MOUSE,FL_FULLBORDER,"Pitch Select - New");
}

/*---------------------------------------------------------------------------
| CALLBACK resampleMode_cb
---------------------------------------------------------------------------*/
void resampleMode_cb (FL_OBJECT *ob,long data)
{
  if (fl_get_choice (ob) == 1)
  {
    pitchChange = 100.0 * rateChange / framesChange;
    updateResampleForm (0);
  }
  else
  {
    updateResampleForm (1);
  }
}

/*---------------------------------------------------------------------------
| CALLBACK resampleLow_cb
---------------------------------------------------------------------------*/
void resampleLow_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK resampleMedium_cb
---------------------------------------------------------------------------*/
void resampleMedium_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK resampleHigh_cb
---------------------------------------------------------------------------*/
void resampleHigh_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK resampleBeats_cb
---------------------------------------------------------------------------*/
void resampleBeats_cb (FL_OBJECT *ob,long data)
{
  if (!sscanf (fl_get_input (ob),"%ld",&beats)) return;
  if (beats < 1) beats = 1;
  if (fl_get_choice (resampleForm->mode) == 1)
    updateResampleForm (0);
  else
    updateResampleForm (1);
}

/*---------------------------------------------------------------------------
| CALLBACK resampleNewBPM_cb
---------------------------------------------------------------------------*/
void resampleNewBPM_cb (FL_OBJECT *ob,long data)
{
  long newBPM;
  long newRate;
  double newTime;
  char tempString [20];
  if (!sscanf (fl_get_input (resampleForm->newBPM),"%ld",&newBPM)) return;
  if (!sscanf (fl_get_input (resampleForm->newRate),"%ld",&newRate)) return;
  if (newBPM < 1) newBPM = 1;
  if (newRate < 1) newRate = 1;
  newTime = 60.0 * (double) beats / newBPM;
  sprintf (tempString,"%.3lf",newTime);
  fl_set_input (resampleForm->newTime,tempString);
  fl_call_object_callback (resampleForm->newTime);
}

/*---------------------------------------------------------------------------
| CALLBACK resampleOK_cb
---------------------------------------------------------------------------*/
void resampleOK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);

  noSampleWarning
  noRangeWarning
  zeroRangeWarning

  stretchCancel = 0;
  
  fl_hide_form (resampleForm->resampleForm);
  if (fl_get_choice (resampleForm->mode) == 1)
  {
    long newFrames;
    long newRate;
    if (!sscanf (fl_get_input (resampleForm->newFrames),"%ld",&newFrames))
      return;
    if (!sscanf (fl_get_input (resampleForm->newRate),"%ld",&newRate))
      return;
    if (newRate < 1) newRate = 1;
    updateBuffers ("Undo Resample (non time stretch)")
    waitCursorOn ();
    fl_set_idle_delta (heavyIdle);
    sample->nonTimeStretch (newFrames,(double)newRate,
      fl_get_sample_edit (mainForm->sample));
    fl_set_idle_delta (normalIdle);
    waitCursorOff ();
  }
  else
  {
    long newRate;
    long lowestFreq;
    long quality;
    if (fl_get_button (resampleForm->high))
    {
      lowestFreq = 20;
      quality = 2;
    }
    else if (fl_get_button (resampleForm->medium))
    {
      lowestFreq = 20;
      quality = 1;
    }
    else
    {
      lowestFreq = 45;
      quality = 1;
    }
    if (!sscanf (fl_get_input (resampleForm->newRate),"%ld",&newRate))
      return;
    if (newRate < 1) newRate = 1;
    double actPitchChange = 100.0 * pitchChange / rateChange;
    fl_set_slider_value (stretchForm->percentCompSlider,0.0);
    fl_set_object_label (stretchForm->percentComp,"0");
    fl_show_form (stretchForm->stretchForm,
    FL_PLACE_MOUSE,FL_FULLBORDER,"Time Stretch");
    updateBuffers ("Undo Resample (time stretch)")
    waitCursorOn ();
//    fl_set_idle_delta (heavyIdle);
    sample->timeStretch (actPitchChange,framesChange,(double)newRate,
      lowestFreq,quality,fl_get_sample_edit (mainForm->sample));
//    fl_set_idle_delta (normalIdle);
    waitCursorOff ();
    fl_hide_form (stretchForm->stretchForm);
  }
  
  updateSample (sample);
  updateDisplayDetails ();
  updateRangeDetails ();
  updateLoopDetails ();
  updateNameBox ();
  fl_redraw_object (mainForm->sample);
  fl_redraw_object (mainForm->scrollBarSlider);
}

/*---------------------------------------------------------------------------
| CALLBACK resampleCancel_cb
---------------------------------------------------------------------------*/
void resampleCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (resampleForm->resampleForm);
}

/*---------------------------------------------------------------------------
| CALLBACK stretchPercentCompSlider_cb
---------------------------------------------------------------------------*/
void stretchPercentCompSlider_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK stretchCancel_cb
---------------------------------------------------------------------------*/
void stretchCancel_cb (FL_OBJECT *ob,long data)
{
  if (fl_show_question_old ("Cancel","Are you sure",0))
  {
    stretchCancel = 1;
  }
}

/*---------------------------------------------------------------------------
| FUNCTION initialiseResampleForm
---------------------------------------------------------------------------*/
void initialiseResampleForm ()
{
  rateChange     = 100.0;
  framesChange   = 100.0;
  timeChange     = 100.0;
  pitchChange    = 100.0;
  beats          = 1;
  currentBigSemi = 60.0;
  fl_set_button (resampleForm->low,0);
  fl_set_button (resampleForm->medium,1);
  fl_set_button (resampleForm->high,0);
}

/*---------------------------------------------------------------------------
| FUNCTION updateResampleForm
---------------------------------------------------------------------------*/
void updateResampleForm (int mode)
{
  // mode == 0 update from frames
  // mode == 1 update from time
  
  DPSample *sample = fl_get_sample (mainForm->sample);
  char   tempString [50];
  long   currentRate;
  long   newRate;
  long   currentFrames;
  long   newFrames;
  double currentTime;
  double newTime;
  double bigSemiChange;
  double newBigSemi;
  int    dirChange;
  int    semiChange;
  int    octChange;
  int    centChange;
  int    semiCurrent;
  int    octCurrent;
  int    centCurrent;
  int    dirCurrent;
  int    semiNew;
  int    octNew;
  int    centNew;
  int    dirNew;

  // Get current frames, rate and time and update on screen
  if (sample->getRangeValid ())
    currentFrames = sample->getRangeEnd () - sample->getRangeStart ();
  else
    currentFrames = 0;
  if (currentFrames < 1) currentFrames = 1;
  
  currentRate = (long) sample->getRate ();
  currentTime = (double) currentFrames / currentRate;

  sprintf (tempString,"%ld",currentRate);
  fl_set_object_label (resampleForm->currentRate,tempString);
  sprintf (tempString,"%ld",currentFrames);
  fl_set_object_label (resampleForm->currentFrames,tempString);
  sprintf (tempString,"%.3lf",currentTime);
  fl_set_object_label (resampleForm->currentTime,tempString);
  
  // Update rate change and pitch change on screen
  sprintf (tempString,"%.3lf",rateChange);
  fl_set_input (resampleForm->rateChange,tempString);
  sprintf (tempString,"%.3lf",pitchChange);
  fl_set_input (resampleForm->pitchChange,tempString);
  
  // Calculate new rate and update on screen
  newRate = (long) (currentRate * rateChange / 100.0);
  if (newRate < 1) newRate = 1;

  sprintf (tempString,"%ld",newRate);
  fl_set_input (resampleForm->newRate,tempString);

  // Calculate new frames, new time and new time or frame change
  if (mode == 0)
  {
    newFrames  = (long) (currentFrames * framesChange / 100.0);
    if (newFrames < 1) newFrames = 1;
    newTime    = (double) newFrames / newRate;
    timeChange = 100.0 * newTime / currentTime;
  }
  else
  {
    newTime      = currentTime * timeChange / 100.0;
    if (newTime < (1.0 / newRate)) newTime = 1.0 / newRate;
    newFrames    = (long) (newTime * newRate);
    framesChange = 100.0 * newFrames / currentFrames;
  }

  // Update new frames, new time, frames and time changes on screen
  
  sprintf (tempString,"%ld",newFrames);
  fl_set_input (resampleForm->newFrames,tempString);
  sprintf (tempString,"%.3lf",newTime);
  fl_set_input (resampleForm->newTime,tempString);
  sprintf (tempString,"%.3lf",framesChange);
  fl_set_input (resampleForm->framesChange,tempString);
  sprintf (tempString,"%.3lf",timeChange);
  fl_set_input (resampleForm->timeChange,tempString);
  
  // Update sliders
  
  if (rateChange < 1.0)
    fl_set_slider_value (resampleForm->rateChangeSlider,1.0);
  else if (rateChange > 200.0)
    fl_set_slider_value (resampleForm->rateChangeSlider,200.0);
  else
    fl_set_slider_value
    (resampleForm->rateChangeSlider,rateChange);
  
  if (framesChange < 1.0)
    fl_set_slider_value (resampleForm->framesChangeSlider,1.0);
  else if (framesChange > 200.0)
    fl_set_slider_value (resampleForm->framesChangeSlider,200.0);
  else
    fl_set_slider_value
    (resampleForm->framesChangeSlider,framesChange);
  
  if (timeChange < 1.0)
    fl_set_slider_value (resampleForm->timeChangeSlider,1.0);
  else if (timeChange > 200.0)
    fl_set_slider_value (resampleForm->timeChangeSlider,200.0);
  else
    fl_set_slider_value
    (resampleForm->timeChangeSlider,timeChange);
  
  if (pitchChange < 1.0)
    fl_set_slider_value (resampleForm->pitchChangeSlider,1.0);
  else if (pitchChange > 200.0)
    fl_set_slider_value (resampleForm->pitchChangeSlider,200.0);
  else
    fl_set_slider_value
    (resampleForm->pitchChangeSlider,pitchChange);
    
  // Update pitch change boxes
  bigSemiChange = 12.0 * log (pitchChange / 100.0) / log (2.0);
  sprintf (tempString,"%.3lf",bigSemiChange);
  fl_set_input (resampleForm->pitchChangeBigSemi,tempString);

  bigSemiToSemi
    (bigSemiChange,&(octChange),&(semiChange),
    &(centChange),&(dirChange));

  if (dirChange == 1)
    fl_set_choice (resampleForm->pitchChangeDir,1);
  else
    fl_set_choice (resampleForm->pitchChangeDir,2);
  sprintf (tempString,"%d",semiChange);
  fl_set_input (resampleForm->pitchChangeSemi,tempString);
  sprintf (tempString,"%d",octChange);
  fl_set_input (resampleForm->pitchChangeOct,tempString);
  sprintf (tempString,"%d",centChange);
  fl_set_input (resampleForm->pitchChangeCent,tempString);

  // Calculate and update current and new pitch changes
  newBigSemi = currentBigSemi + bigSemiChange;
  sprintf (tempString,"%.3lf",currentBigSemi);
  fl_set_object_label (resampleForm->currentBigSemi,tempString);
  sprintf (tempString,"%.3lf",newBigSemi);
  fl_set_object_label (resampleForm->newBigSemi,tempString);
  
  bigSemiToSemi
    (currentBigSemi,&octCurrent,&semiCurrent,&centCurrent,&dirCurrent);
  fl_set_object_label (resampleForm->currentSemi,semiToStr (semiCurrent));
  sprintf (tempString,"%d",octCurrent-2);
  fl_set_object_label (resampleForm->currentOct,tempString);
  sprintf (tempString,"%d",centCurrent);
  fl_set_object_label (resampleForm->currentCent,tempString);

  bigSemiToSemi
    (newBigSemi,&octNew,&semiNew,&centNew,&dirNew);
  fl_set_object_label (resampleForm->newSemi,semiToStr (semiNew));
  sprintf (tempString,"%d",octNew-2);
  fl_set_object_label (resampleForm->newOct,tempString);
  sprintf (tempString,"%d",centNew);
  fl_set_object_label (resampleForm->newCent,tempString);
  
  // Beats per minute display
  int currentBPM;
  int newBPM;
  
  if (currentFrames)
    currentBPM = 60 * beats * (long) currentRate / currentFrames;
  else currentBPM = 0;
  if (currentBPM < 1) currentBPM = 1;
  
  if (newFrames)
    newBPM = 60 * beats * (long) newRate / newFrames;
  else newBPM = 0;
  if (newBPM < 1) newBPM = 1;
  
  sprintf (tempString,"%ld",beats);
  fl_set_input (resampleForm->beats,tempString);
  sprintf (tempString,"%d",currentBPM);
  fl_set_object_label (resampleForm->currentBPM,tempString);
  sprintf (tempString,"%d",newBPM);
  fl_set_input (resampleForm->newBPM,tempString);

  
  // Show quality controls if necessary
  long qualityShown = resampleForm->ldata;
  long stretchMode = (fl_get_choice (resampleForm->mode));
  if (stretchMode == 1 && qualityShown)
  {
    fl_hide_object (resampleForm->qualityButtons);
    resampleForm->ldata = 0;
  }
  else if (stretchMode == 2 && !qualityShown)
  {
    fl_show_object (resampleForm->qualityButtons);
    resampleForm->ldata = 1;
  }

}

/*---------------------------------------------------------------------------
| CALLBACK pitchCentSlider_cb
---------------------------------------------------------------------------*/
void pitchCentSlider_cb (FL_OBJECT *ob,long data)
{
  char tempString [20];
  int cent = (int) fl_get_slider_value (ob);
  sprintf (tempString,"%d",cent);
  fl_set_input (pitchForm->cent,tempString);
  fl_call_object_callback (pitchForm->cent);
}

/*---------------------------------------------------------------------------
| CALLBACK pitchCentReset_cb
---------------------------------------------------------------------------*/
void pitchCentReset_cb (FL_OBJECT *ob,long data)
{
  char tempString [20];
  fl_set_input (pitchForm->cent,"0");
  fl_call_object_callback (pitchForm->cent);
}

/*---------------------------------------------------------------------------
| CALLBACK pitchMode_cb
---------------------------------------------------------------------------*/
void pitchMode_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK pitchSemi_cb
---------------------------------------------------------------------------*/
void pitchSemi_cb (FL_OBJECT *ob,long data)
{
  char tempString [20];
  int semi;
  int oct;
  int cent;
  int dir = 1;
  if (sscanf (fl_get_input (pitchForm->semi),"%d",&semi))
  {
    if (semi < 0) semi = 0;
  }
  else
  {
    semi = strToSemi ((char *) fl_get_input (pitchForm->semi));
  }
  if (!sscanf (fl_get_input (pitchForm->oct),"%d",&oct)) return;
  if (!sscanf (fl_get_input (pitchForm->cent),"%d",&cent)) return;
  semiToBigSemi (&currentPitch,oct + 2,semi,cent,dir);
  if (currentPitch < 0.0) currentPitch = 0.0;
  else if (currentPitch > 127.0) currentPitch = 127.0;
  updatePitchForm ();
}

/*---------------------------------------------------------------------------
| CALLBACK pitchOct_cb
---------------------------------------------------------------------------*/
void pitchOct_cb (FL_OBJECT *ob,long data)
{
  fl_call_object_callback (pitchForm->semi);
}

/*---------------------------------------------------------------------------
| CALLBACK pitchCent_cb
---------------------------------------------------------------------------*/
void pitchCent_cb (FL_OBJECT *ob,long data)
{
  fl_call_object_callback (pitchForm->semi);
}

/*---------------------------------------------------------------------------
| CALLBACK pitchBigSemi_cb
---------------------------------------------------------------------------*/
void pitchBigSemi_cb (FL_OBJECT *ob,long data)
{
  if (!sscanf (fl_get_input (ob),"%lf",&currentPitch)) return;
  if (currentPitch < 0.0) currentPitch = 0.0;
  else if (currentPitch > 127.0) currentPitch = 127.0;
  updatePitchForm ();
}

/*---------------------------------------------------------------------------
| CALLBACK pitchOK_cb
---------------------------------------------------------------------------*/
void pitchOK_cb (FL_OBJECT *ob,long data)
{
  int oct;
  int semi;
  int cent;
  int dir;
  char tempString [20];

  fl_hide_form (pitchForm->pitchForm);
  if (fl_get_choice (pitchForm->mode) == 1)
  {
    double newBigSemi;
    double bigSemiChange;
    bigSemiChange = 12.0 * log (pitchChange / 100.0) / log (2.0);
    newBigSemi = currentBigSemi + bigSemiChange;
    currentBigSemi = currentPitch;
    bigSemiChange = newBigSemi - currentBigSemi;
    pitchChange = 100.0 * pow (2.0,(bigSemiChange / 12.0));
  }
  else
  {
    double bigSemiChange = currentPitch - currentBigSemi;
    pitchChange = 100.0 * pow (2.0,(bigSemiChange / 12.0));
  }
  if (pitchChange < 1.0) pitchChange = 1.0;
  if (fl_get_choice (resampleForm->mode) == 1)
  {
    framesChange = 100.0 * rateChange / pitchChange;
  }
  updateResampleForm (0);
  fl_activate_form (resampleForm->resampleForm);
}

/*---------------------------------------------------------------------------
| CALLBACK pitchCancel_cb
---------------------------------------------------------------------------*/
void pitchCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (pitchForm->pitchForm);
  fl_activate_form (resampleForm->resampleForm);
}

/*---------------------------------------------------------------------------
| FUNCTION initialisePitchForm
---------------------------------------------------------------------------*/
void initialisePitchForm ()
{
  if (fl_get_choice (pitchForm->mode) == 1)
  {
    currentPitch = currentBigSemi;
  }
  else
  {
    if (!sscanf (resampleForm->newBigSemi->label,"%lf",&currentPitch))
      currentPitch = 0;
  }
}

/*---------------------------------------------------------------------------
| FUNCTION updatePitchForm
---------------------------------------------------------------------------*/
void updatePitchForm ()
{
  int oct;
  int semi;
  int cent;
  int dir;
  char tempString [20];
  
  bigSemiToSemi (currentPitch,&oct,&semi,&cent,&dir);
  sprintf (tempString,"%.3lf",currentPitch);
  fl_set_input (pitchForm->bigSemi,tempString);
  fl_set_input (pitchForm->semi,semiToStr (semi));
  sprintf (tempString,"%d",oct-2);
  fl_set_input (pitchForm->oct,tempString);
  sprintf (tempString,"%d",cent);
  fl_set_input (pitchForm->cent,tempString);
  fl_set_slider_value (pitchForm->centSlider,(double)cent);
  fl_redraw_object (pitchForm->keyboard);
}

/*---------------------------------------------------------------------------
| FUNCTION bigSemiToSemi
---------------------------------------------------------------------------*/
void bigSemiToSemi (double bigsemi,int *oct,int *semi,int *cent,int *dir)
{
  long centChange;
  if (bigsemi >= 0.0)
  {
    centChange = (long)(rint (100.0 * bigsemi));
    *dir = 1;
  }
  else
  {
    centChange = (long)(rint (100.0 * -bigsemi));
    *dir = -1;
  }
    
  *oct  = centChange / 1200;
  *semi = (centChange % 1200) / 100;
  *cent = (centChange % 1200) % 100;
  
  if (*dir == 1 && *cent >= 50)
  {
    *cent = -100 + *cent;
    *semi += 1;
    if (*semi == 12)
    {
      *semi = 0;
      *oct  += 1;
    }
  }
  else if (*dir == -1)
  {
    if (*cent <= 50)
      *cent = -*cent;
    else
    {
      *cent = 100 - *cent;
      *semi += 1;
      if (*semi == 12)
      {
        *semi = 0;
        *oct  += 1;
      }
    }
  }
}

/*---------------------------------------------------------------------------
| FUNCTION semiToBigSemi
---------------------------------------------------------------------------*/
void semiToBigSemi (double *bigsemi,int oct,int semi,int cent,int dir)
{
  if (dir > 0) dir = 1;
  else dir = -1;
  *bigsemi =
    (double) (dir * ((12 * oct) + semi)) + (cent / 100.0);
}

/*---------------------------------------------------------------------------
| FUNCTION semiToStr
---------------------------------------------------------------------------*/
char *semiToStr (int semi)
{
  return noteStrings [semi % 12];
}

/*---------------------------------------------------------------------------
| FUNCTION strToSemi
---------------------------------------------------------------------------*/
int strToSemi (char *str)
{
  int key;
  int sharp;
  int number;
  static int whitekeylookup [] = { 9,11,0,2,4,5,7 };
  
  if (!str) return 0;
  if (strlen (str) < 1) return 0;
  if (str [0] >= '0' && str [0] <= '9')
  {
    if (!sscanf (str,"%d",&key)) return 0;
    if (key < 0) key = 0;
    else if (key > 12) key = 12;
    return key;
  }
  
  key = toupper (str [0]);
  if (key < 'A') key = 'A';
  else if (key > 'G') key = 'G';
  key = key - 'A';
  key = whitekeylookup [key];
  if (strlen (str) == 1) return key;
  if (str [1] == '#')
  {
    key++;
    if (key == 13) key = 0;
  }
  else if (str [1] == 'b')
  {
    key--;
    if (key == -1) key = 12;
  }
  return key;
}

/*---------------------------------------------------------------------------
| FUNCTION keyboardHandle
---------------------------------------------------------------------------*/
int keyboardHandle (
  FL_OBJECT *ob,
  int event,
  FL_Coord mx,
  FL_Coord my,
  int key,
  void *xev)
{
  static int whitekeylookup [] = { 0,2,4,5,7,9,11                 };
  static int blackkeylookup [] = { 0,1,1,3,3,4,5,6,6,8,8,10,10,11 };
  static int drawlookupx []    = { 0,1,2,3,4,6,7,8,9,10,11,12     };
  static int drawlookupy []    = { 0,1,0,1,0,0,1,0,1,0,1,0        };
  
  char tempString [20];
  
  switch (event)
  {
    case FL_DRAW :
    {
      int key;
      int oct;
      key = strToSemi ((char *) fl_get_input (pitchForm->semi));
      if (!sscanf (fl_get_input (pitchForm->oct),"%d",&oct)) oct = 0;
      int halfkeywidth = 4;
      int octWidth     = 14 * halfkeywidth;
      key = key % 12;
      int x;
      int y;
      int w;
      int h;

      fl_redraw_object (pitchForm->keyboardPixmap);
      
      if (drawlookupy [key])
      {
        // Black
        x = ob->x + ((oct+2)*octWidth) + 
            (halfkeywidth * drawlookupx [key]) + 3;
        y = ob->y + 3;
        w = 3;
        h = 32;
        fl_rectf (x,y,w,h,FL_RED);
      }
      else
      {
        // White
        x = ob->x + ((oct+2)*octWidth) + 
            (halfkeywidth * drawlookupx [key]) + 1;
        y = ob->y + 37;
        w = 7;
        h = 20;
        fl_rectf (x,y,w,h,FL_RED);
        
        x += 1;
        y += 20;
        w -= 2;
        h = 2;
        fl_rectf (x,y,w,h,FL_RED);
        
        if (key == 0 || key == 5)
        {
          x -= 1;
          h = 36;
          y = ob->y + 3;
        }
        else if (key == 4 || key == 11 || (key == 7 && oct == 8))
        {
          x += 1;
          h = 36;
          y = ob->y + 3;
        }
        else
        {
          x += 1;
          w -= 2;
          h = 36;
          y = ob->y + 3;
        }
        fl_rectf (x,y,w,h,FL_RED);
      }
      break;
    }
    
      
    case FL_PUSH :
    
      ALconfig audioConfig;
      long     queueSize;

      // Disable void handler
      ALseterrorhandler (0);

      // Remove idle delta
      fl_set_idle_delta (0);

      // Set up new audio configuration 
      audioConfig = ALnewconfig ();
      if (!audioConfig)
      {
        return 0;
      }
 
      // Set sample format and width
      ALsetsampfmt (audioConfig,AL_SAMPFMT_TWOSCOMP);
      ALsetwidth (audioConfig,AL_SAMPLE_16);

      // Set number of channels
      ALsetchannels (audioConfig,1);

      // Set queue size
      queueSize = SINEOUTQUEUESIZE;
      
      limitQueueSize (1,&queueSize);
      
      if (ALsetqueuesize (audioConfig,queueSize))
      {
        ALfreeconfig (audioConfig);
        return 0;
      }

      // Open new audio port with given configuration
      globalOutputPort =
        ALopenport ("GlobalAudioOutputPort","w",audioConfig);

      ALfreeconfig (audioConfig);
      
      // FOLLOW THROUGH - NO BREAK
    
    case FL_MOUSE :
    {

      int keyheight;
      int keynum;
      int key;
      int cent;
      double newPitch;
      
      int halfkeywidth = 4;
      int midY = ob->y + (ob->h / 2);
      
      if (my > midY) keyheight = 1;
      else keyheight = 0;
      
      keynum = (mx - ob->x) / halfkeywidth;
      if (keynum < 0) keynum = 0;
      else if (keynum > 149) keynum = 149;
      
      if (keyheight == 1)
      {
        // White keys
        keynum = keynum / 2;
        key = (12 * (keynum / 7)) + (whitekeylookup [keynum % 7]);
      }
      else
      {
        // Black keys
        key = (12 * (keynum / 14)) + (blackkeylookup [keynum % 14]);
        if (key > 127) key = 127;
      }
      if (!sscanf (fl_get_input (pitchForm->cent),"%d",&cent)) cent = 0;
      newPitch = (double) key + (cent / 100.0);
      if (newPitch < 0.0) newPitch = 0.0;
      else if (newPitch > 127.0) newPitch = 127.0;
      if (newPitch != currentPitch)
      {
        int oct;
        int semi;
        int cent;
        int dir;
        currentPitch = newPitch;
        bigSemiToSemi (currentPitch,&oct,&semi,&cent,&dir);
        sprintf (tempString,"%.3lf",currentPitch);
        fl_set_input (pitchForm->bigSemi,tempString);
        fl_set_input (pitchForm->semi,semiToStr (semi));
        sprintf (tempString,"%d",oct-2);
        fl_set_input (pitchForm->oct,tempString);
        sprintf (tempString,"%d",cent);
        fl_set_input (pitchForm->cent,tempString);
        fl_set_slider_value (pitchForm->centSlider,(double)cent);
        fl_redraw_object (ob);
        updateAudioGlobals (0,0);
      }
      break;
    }
    
    case FL_RELEASE :
    {
      if (globalOutputPort)
      {
        ALcloseport (globalOutputPort);
        globalOutputPort = 0;
      }

      // Restore idle delta
      fl_set_idle_delta (normalIdle);

      fl_redraw_object (ob);
      break;
    }

    default :
      break;
  }
  return 0;
}

/*---------------------------------------------------------------------------
| CALLBACK bufferBrowser_cb
---------------------------------------------------------------------------*/
void bufferBrowser_cb (FL_OBJECT *ob,long data)
{
  fl_call_object_callback (bufferForm->select);
}

/*---------------------------------------------------------------------------
| CALLBACK bufferSelectDbl_cb
---------------------------------------------------------------------------*/
void bufferSelectDbl_cb (FL_OBJECT *ob,long data)
{
  fl_call_object_callback (bufferForm->select);
}

/*---------------------------------------------------------------------------
| CALLBACK bufferSelect_cb
---------------------------------------------------------------------------*/
void bufferSelect_cb (FL_OBJECT *ob,long data)
{
  int newCurrent = fl_get_browser (bufferForm->browser) - 1;
  if (newCurrent >= 0 && newCurrent != current)
  {
    clearBuffers ();
    current = newCurrent;
    DPSample *sample = buffer [current];
    fl_set_sample (mainForm->sample,sample);
    updateSample (sample);
  }
}

/*---------------------------------------------------------------------------
| CALLBACK bufferOpen_cb
---------------------------------------------------------------------------*/
void bufferOpen_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample;
  char *error;
  const char *filename;

  if (buffer [current]->getChanged ())
    if (!fl_show_question_old
      ("Warning","This buffer contains unsaved data","Really open ?"))
      return;

  fl_use_fselector (0);
  
  if (first0)
  {
    filename = fl_show_fselector (
      "Please enter the filename to load",
      ".",
      "*.aiff",0);
    first0 = 0;
  }
  else
  {
    filename = fl_show_fselector (
      "Please enter the filename to load",0,0,0);
  }

  if (filename)
  {
    sample = new DPSample;
    waitCursorOn ();
    if (error = sample->loadAIFF ((char *) filename))
    {
      waitCursorOff ();
      fl_show_alert ("Warning - Could not load file",
        filename,error,TRUE);
      delete sample;
    }
    else
    {
      waitCursorOff ();
      clearBuffers ();
      fl_set_sample (mainForm->sample,sample);
      delete buffer [current];
      buffer [current] = sample;
      updateSample (sample);
    }
  }
}

/*---------------------------------------------------------------------------
| CALLBACK bufferClear_cb
---------------------------------------------------------------------------*/
void bufferClear_cb (FL_OBJECT *ob,long data)
{
  int selected = fl_get_browser (bufferForm->browser) - 1;
  if (selected < 0) return;
  DPSample *sample = buffer [selected];
  noSampleWarning

  if (sample->getChanged ())
    if (!fl_show_question_old
      ("Warning","This buffer contains unsaved data","Really clear ?"))
      return;

  updateBuffers ("Undo Clear")
  long width    = sample->getWidth ();
  long channels = sample->getChannels ();
  long rate     = (long) (sample->getRate ());
  long frames   = 0;
  waitCursorOn ();
  sample->fresh ((double) rate,width,channels,frames);
  waitCursorOff ();
  if (selected == current)
    updateSample (sample);
}

/*---------------------------------------------------------------------------
| CALLBACK bufferCopy_cb
---------------------------------------------------------------------------*/
void bufferCopy_cb (FL_OBJECT *ob,long data)
{
  fl_set_object_label (bufferForm->text,"Please select buffer to copy over");
  fl_redraw_object (bufferForm->text);
  fl_hide_object (bufferForm->bufferGroup);
  fl_show_object (bufferForm->copyGroup);
  int source = fl_get_browser (bufferForm->browser) - 1;
  updateCopyBrowser (source);
  fl_select_browser_line (bufferForm->browser2,(source+2)%8);
  fl_set_choice (bufferForm->mode,1);
}

/*---------------------------------------------------------------------------
| CALLBACK bufferMove_cb
---------------------------------------------------------------------------*/
void bufferMove_cb (FL_OBJECT *ob,long data)
{
  fl_set_object_label (bufferForm->text,"Please select buffer to move to");
  fl_redraw_object (bufferForm->text);
  fl_hide_object (bufferForm->bufferGroup);
  fl_show_object (bufferForm->copyGroup);
  int source = fl_get_browser (bufferForm->browser)-1;
  updateCopyBrowser (source);
  fl_select_browser_line (bufferForm->browser2,(source+2)%8);
  fl_set_choice (bufferForm->mode,2);
}

/*---------------------------------------------------------------------------
| CALLBACK bufferMixplay_cb
---------------------------------------------------------------------------*/
void bufferMixplay_cb (FL_OBJECT *ob,long data)
{
  fl_deactivate_object (bufferForm->mixplay);
  int item = fl_get_menu (ob);
  
  mixplayForm->ldata = (long) item;
  
  fl_call_object_callback (playForm->stopButton);
  fl_call_object_callback (recordForm->stopButton);
  #ifdef LINUX
  closeGlobalPorts ();
  #endif

  #ifndef LINUX
  long pvBuffer1 [2];
  long pvBuffer2 [2];
  pvBuffer1 [0] = AL_MONITOR_CTL;
  pvBuffer2 [0] = AL_MONITOR_CTL;
  pvBuffer2 [1] = 0;
  ALgetparams (AL_DEFAULT_DEVICE,pvBuffer1,2);
  if (fl_get_sample_autoglobals (mainForm->sample))
    ALsetparams (AL_DEFAULT_DEVICE,pvBuffer2,2);
  #endif
  
  char *error;
  switch (item)
  {
    case 1 :
      fl_set_object_label (mixplayForm->text,"Two Channel Mixing");
      fl_redraw_object (mixplayForm->text);
      initialiseMixplayForm (2);
      if (fl_get_sample_autowindows (mainForm->sample))
      {
    	rememberForms (0);
    	fl_deactivate_all_forms ();
    	fl_activate_form (mixplayForm->mixplayForm);
    	fl_activate_form (selectForm->selectForm);
      }
      fl_hide_object (mixplayForm->fourGroup);
      fl_hide_object (mixplayForm->deskGroup);
      fl_show_form (mixplayForm->mixplayForm,
      FL_PLACE_MOUSE,FL_FULLBORDER,"Mix Play - Two Channel");
      if (error = mixPlay ())
        fl_show_alert ("Warning","Could not play sample",error,TRUE);
      break;
    
    case 2 :
      fl_set_object_label (mixplayForm->text,"Four Channel Mixing");
      fl_redraw_object (mixplayForm->text);
      initialiseMixplayForm (4);
      if (fl_get_sample_autowindows (mainForm->sample))
      {
    	rememberForms (0);
    	fl_deactivate_all_forms ();
    	fl_activate_form (mixplayForm->mixplayForm);
    	fl_activate_form (selectForm->selectForm);
      }
      fl_hide_object (mixplayForm->twoGroup);
      fl_hide_object (mixplayForm->deskGroup);
      fl_show_form (mixplayForm->mixplayForm,
      FL_PLACE_MOUSE,FL_FULLBORDER,"Mix Play - Four Channel");
      if (error = mixPlay ())
        fl_show_alert ("Warning","Could not play sample",error,TRUE);
      break;
      
    case 3 :
      fl_set_object_label (mixplayForm->text,"Mixing Desk");
      fl_redraw_object (mixplayForm->text);
      initialiseMixplayForm (8);
      if (fl_get_sample_autowindows (mainForm->sample))
      {
    	rememberForms (0);
    	fl_deactivate_all_forms ();
    	fl_activate_form (mixplayForm->mixplayForm);
    	fl_activate_form (selectForm->selectForm);
      }
      fl_hide_object (mixplayForm->twoGroup);
      fl_hide_object (mixplayForm->fourGroup);
      fl_show_form (mixplayForm->mixplayForm,
      FL_PLACE_MOUSE,FL_FULLBORDER,"Mix Play - Desk");
      if (error = mixPlay ())
        fl_show_alert ("Warning","Could not play sample",error,TRUE);
      break;
    
    default :
      break;
  }

  #ifndef LINUX
  if (fl_get_sample_autoglobals (mainForm->sample))
    ALsetparams (AL_DEFAULT_DEVICE,pvBuffer1,2);
  #endif
}

/*---------------------------------------------------------------------------
| FUNTION updateBufferBrowser
---------------------------------------------------------------------------*/
void updateBufferBrowser ()
{
  fl_freeze_form (bufferForm->bufferForm);
  fl_clear_browser (bufferForm->browser);
  char *filename;
  char *newFilename;
  int i;
  for (i=0; i<8; i++)
  {
    filename = buffer [i]->getFilename ();
    if (buffer [i]->getValid () && filename && (strlen (filename) > 0))
    {
      filename = &(filename [stripFilename (filename)]);
      newFilename = new char [strlen (filename) + 6];
      strcpy (newFilename+5,filename);
      newFilename [0] = '@';
      newFilename [1] = 'i';
      newFilename [2] = i + '1';
      newFilename [3] = ' ';
      newFilename [4] = ' ';
      if (i == current)
      {
        fl_add_browser_line (bufferForm->browser,newFilename);
      }
      else
      {
        fl_add_browser_line (bufferForm->browser,&(newFilename [2]));
      }
      delete [] newFilename;
    }
    else
    {
      newFilename = new char [50];
      if (buffer [i]->getValid ())
        strcpy (newFilename,"     Untitled");
      else
        strcpy (newFilename,"     ");

      newFilename [0] = '@';
      newFilename [1] = 'i';
      newFilename [2] = i + '1';
      newFilename [3] = ' ';
      newFilename [4] = ' ';
      if (i == current)
      {
        fl_add_browser_line (bufferForm->browser,newFilename);
      }
      else
      {
        fl_add_browser_line (bufferForm->browser,&(newFilename [2]));
      }
      delete [] newFilename;
    }
  }
  fl_select_browser_line (bufferForm->browser,current+1);
  fl_unfreeze_form (bufferForm->bufferForm);
}

/*---------------------------------------------------------------------------
| FUNCTION updateCopyBrowser
---------------------------------------------------------------------------*/
void updateCopyBrowser (int unselect)
{
  fl_freeze_form (bufferForm->bufferForm);
  fl_clear_browser (bufferForm->browser2);
  char *filename;
  char *newFilename;
  int i;
  for (i=0; i<8; i++)
  {
    filename = buffer [i]->getFilename ();
    if (buffer [i]->getValid () && filename && (strlen (filename) > 0))
    {
      filename = &(filename [stripFilename (filename)]);
      newFilename = new char [strlen (filename) + 8];
      strcpy (newFilename+7,filename);
      newFilename [0] = '@';
      newFilename [1] = 'N';
      newFilename [2] = '@';
      newFilename [3] = 'i';
      newFilename [4] = i + '1';
      newFilename [5] = ' ';
      newFilename [6] = ' ';
      if (i == unselect)
      {
        fl_add_browser_line (bufferForm->browser2,newFilename);
      }
      else
      {
        fl_add_browser_line (bufferForm->browser2,&(newFilename [4]));
      }
      delete [] newFilename;
    }
    else
    {
      newFilename = new char [50];
      if (buffer [i]->getValid ())
        strcpy (newFilename,"       Untitled");
      else
        strcpy (newFilename,"       ");

      newFilename [0] = '@';
      newFilename [1] = 'N';
      newFilename [2] = '@';
      newFilename [3] = 'i';
      newFilename [4] = i + '1';
      newFilename [5] = ' ';
      newFilename [6] = ' ';
      if (i == unselect)
      {
        fl_add_browser_line (bufferForm->browser2,newFilename);
      }
      else
      {
        fl_add_browser_line (bufferForm->browser2,&(newFilename [4]));
      }
      delete [] newFilename;
    }
  }
  fl_select_browser_line (bufferForm->browser2,current+1);
  fl_unfreeze_form (bufferForm->bufferForm);
}

/*---------------------------------------------------------------------------
| CALLBACK bufferBrowser2_cb
---------------------------------------------------------------------------*/
void bufferBrowser2_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK bufferMode_cb
---------------------------------------------------------------------------*/
void bufferMode_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK bufferOKDbl_cb
---------------------------------------------------------------------------*/
void bufferOKDbl_cb (FL_OBJECT *ob,long data)
{
  fl_call_object_callback (bufferForm->OK);
}

/*---------------------------------------------------------------------------
| CALLBACK bufferOK_cb
---------------------------------------------------------------------------*/
void bufferOK_cb (FL_OBJECT *ob,long data)
{
  fl_set_object_label (bufferForm->text,"Sample Buffers");
  fl_redraw_object (bufferForm->text);
  fl_hide_object (bufferForm->copyGroup);
  fl_show_object (bufferForm->bufferGroup);
  
  int source = fl_get_browser (bufferForm->browser) - 1;
  int dest   = fl_get_browser (bufferForm->browser2) - 1;
  int success;
  
  if (source == dest) return;
  
  if (buffer [dest]->getChanged ())
  {
    if (fl_get_choice (bufferForm->mode) == 1)
    {
      if (!fl_show_question_old
        ("Warning","The destination buffer contains unsaved data",
        "Really copy over it ?"))
        return;
    }
    else
    {
      if (!fl_show_question_old
        ("Warning","The destination buffer contains unsaved data",
        "Really move over it ?"))
        return;
    }
  }

  waitCursorOn ();
  
  // Copy (if fails then clear buffer)
  success = buffer [dest]->clone (*(buffer [source]));
  if (!success)
  {
    DPSample *sample = buffer [dest];
    long width       = sample->getWidth ();
    long channels    = sample->getChannels ();
    long rate        = (long) (sample->getRate ());
    long frames      = 0;
    sample->fresh ((double) rate,width,channels,frames);
  }
  
  // Clear (but only if move and copy was successful)
  if (fl_get_choice (bufferForm->mode) == 2 && success)
  {
    DPSample *sample = buffer [source];
    long width       = sample->getWidth ();
    long channels    = sample->getChannels ();
    long rate        = (long) (sample->getRate ());
    long frames      = 0;
    sample->fresh ((double) rate,width,channels,frames);
  }

  waitCursorOff ();
  
  // Select new buffer
  fl_select_browser_line (bufferForm->browser,dest + 1);
  current = -1; // Ensures complete redraw
  fl_call_object_callback (bufferForm->select);
}

/*---------------------------------------------------------------------------
| CALLBACK bufferCancel_cb
---------------------------------------------------------------------------*/
void bufferCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_object (bufferForm->copyGroup);
  fl_show_object (bufferForm->bufferGroup);
  fl_set_object_label (bufferForm->text,"Sample Buffers");
}

/*---------------------------------------------------------------------------
| CALLBACK mixplaySlider_cb
---------------------------------------------------------------------------*/
void mixplaySlider_cb (FL_OBJECT *ob,long data)
{
  if (data < 8)
  {
    playList [data].vol = fl_get_slider_value (ob) / 100.0;
  }
  else
  {
    mixplayMasterVol = fl_get_slider_value (ob) / 100.0;
  }
}

/*---------------------------------------------------------------------------
| CALLBACK mixplayTwoSlider_cb
---------------------------------------------------------------------------*/
void mixplayTwoSlider_cb (FL_OBJECT *ob,long data)
{
  double val = fl_get_slider_value (ob) / 200.0;
  if (val < 0.0)
  {
    playList [1].vol = 0.5 + val;
    playList [0].vol = 0.5;
  }
  else
  {
    playList [1].vol = 0.5;
    playList [0].vol = 0.5 - val;
  }
}

/*---------------------------------------------------------------------------
| CALLBACK mixplayFourPositioner_cb
---------------------------------------------------------------------------*/
void mixplayFourPositioner_cb (FL_OBJECT *ob,long data)
{
  double x      = fl_get_positioner_xvalue (ob) / 400.0;
  double y      = fl_get_positioner_yvalue (ob) / 400.0;
  
  if (x < 0)
  {
    if (y > 0.25 + x) y = 0.25 + x;
    else if (y < -0.25 - x) y = -0.25 - x;
  }
  else
  {
    if (y > 0.25 - x) y = 0.25 - x;
    else if (y < -0.25 + x) y = -0.25 + x;
  }
  
  double absx   = x < 0.0 ? -x : x;
  double absx1  = 0.25 - absx;
  double absy   = y < 0.0 ? -y : y;
  double absy1  = 0.25 - absy;
  double absxy  = absx + absy;
  if (absxy > 0.25) absxy = 0.25;
  double absxy1 = 0.25 - absxy;


//   Inefficient version (but easy to read)
//  
//   if (x < 0) vol2 = 0.25 - absy;
//   else vol2 = 0.25 - absxy;
//   
//   if (x > 0) vol4 = 0.25 - absy;
//   else vol4 = 0.25 - absxy;
//   
//   if (y < 0) vol3 = 0.25 - absx;
//   else vol3 = 0.25 - absxy;
//   
//   if (y > 0) vol1 = 0.25 - absx;
//   else vol1 = 0.25 - absxy;
  
  if (x < 0)
  {
    playList [1].vol = absy1;
    playList [3].vol = absxy1;
  }
  else
  {
    playList [1].vol = absxy1;
    playList [3].vol = absy1;
  }
  
  if (y < 0)
  {
    playList [2].vol = absx1;
    playList [0].vol = absxy1;
  }
  else
  {
    playList [2].vol = absxy1;
    playList [0].vol = absx1;
  }
}

/*---------------------------------------------------------------------------
| CALLBACK mixplayPlay_cb
---------------------------------------------------------------------------*/
void mixplayPlay_cb (FL_OBJECT *ob,long data)
{
  if (data < 8)
  {
    playList [data].frame   = 0;
    playList [data].release = 0;
    playList [data].inc     = 1;
  }
  else
  {
    for (int i=0; i<8; i++)
    {
      playList [i].frame   = 0;
      playList [i].release = 0;
      playList [i].inc     = 1;
    }
  }
}

/*---------------------------------------------------------------------------
| CALLBACK mixplayStop_cb
---------------------------------------------------------------------------*/
void mixplayStop_cb (FL_OBJECT *ob,long data)
{
  if (data < 8)
  {
    playList [data].frame = -1;
    fl_set_object_color
    (mixplayForm->playLight [data],FL_COL1,FL_COL1);
    if (data < 4)
    {
      if (data < 2)
        fl_set_object_color
        (mixplayForm->playLight [data+9],FL_COL1,FL_COL1);
      fl_set_object_color
      (mixplayForm->playLight [data+11],FL_COL1,FL_COL1);
    }
  }
  else
  {
    for (int i=0; i<8; i++)
    {
      playList [i].frame = -1;
      fl_set_object_color
      (mixplayForm->playLight [i],FL_COL1,FL_COL1);
      if (i < 4)
      {
        if (i < 2)
          fl_set_object_color
          (mixplayForm->playLight [i+9],FL_COL1,FL_COL1);
        fl_set_object_color
        (mixplayForm->playLight [i+11],FL_COL1,FL_COL1);
      }
    }
  }
}

/*---------------------------------------------------------------------------
| CALLBACK mixplayRelease_cb
---------------------------------------------------------------------------*/
void mixplayRelease_cb (FL_OBJECT *ob,long data)
{
  if (data < 8)
  {
    (playList [data].release)++;
    playList [data].inc = 1;
  }
  else
  {
    for (int i=0; i<8; i++)
    {
      (playList [i].release)++;
      playList [i].inc = 1;
    }
  }
}

/*---------------------------------------------------------------------------
| CALLBACK mixplayBrowser_cb
---------------------------------------------------------------------------*/
void mixplayBrowser_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK mixplayLeftSlider_cb
---------------------------------------------------------------------------*/
void mixplayLeftSlider_cb (FL_OBJECT *ob,long data)
{
  fl_set_slider_value (playForm->leftSlider,fl_get_slider_value (ob));
  fl_call_object_callback (playForm->leftSlider);
}

/*---------------------------------------------------------------------------
| CALLBACK mixplayRightSlider_cb
---------------------------------------------------------------------------*/
void mixplayRightSlider_cb (FL_OBJECT *ob,long data)
{
  fl_set_slider_value (playForm->rightSlider,fl_get_slider_value (ob));
  fl_call_object_callback (playForm->rightSlider);
}

/*---------------------------------------------------------------------------
| CALLBACK mixplayLockButton_cb
---------------------------------------------------------------------------*/
void mixplayLockButton_cb (FL_OBJECT *ob,long data)
{
  fl_set_button (playForm->lockButton,fl_get_button (ob));
  fl_call_object_callback (playForm->lockButton);
}

/*---------------------------------------------------------------------------
| CALLBACK mixplayMuteHiddenButton_cb
---------------------------------------------------------------------------*/
void mixplayMuteHiddenButton_cb (FL_OBJECT *ob,long data)
{
  fl_set_button (playForm->muteHiddenButton,fl_get_button (ob));
  fl_call_object_callback (playForm->muteHiddenButton);
}

/*---------------------------------------------------------------------------
| CALLBACK mixplayMuteButton_cb
---------------------------------------------------------------------------*/
void mixplayMuteButton_cb (FL_OBJECT *ob,long data)
{
  #ifdef LINUX
  fl_set_button (playForm->muteButton,fl_get_button (ob));
  fl_call_object_callback (playForm->muteButton);
  #endif
}

/*---------------------------------------------------------------------------
| CALLBACK mixplaySelect_cb
---------------------------------------------------------------------------*/
void mixplaySelect_cb (FL_OBJECT *ob,long data)
{
  selectForm->ldata = data;
  if (playList [data].bufferNo >= 0)
  {
    fl_select_browser_line
      (selectForm->browser,playList [data].bufferNo + 1);
  }
  else
  {
    fl_deselect_browser (selectForm->browser);
  }
  fl_show_form (selectForm->selectForm,
  FL_PLACE_MOUSE,FL_FULLBORDER,"Select");
  fl_deactivate_form (mixplayForm->mixplayForm);
}

/*---------------------------------------------------------------------------
| CALLBACK mixplaySelectButton_cb
---------------------------------------------------------------------------*/
void mixplaySelectButton_cb (FL_OBJECT *ob,long data)
{
  int i = fl_get_browser (mixplayForm->browser) - 1;
  int max;
  
  if (mixplayForm->ldata == 1) max = 2;
  else if (mixplayForm->ldata == 2) max = 4;
  else if (mixplayForm->ldata == 3) max = 8;
  else return;
 
  if (i >= 0 && i < max)
  {
    selectForm->ldata = i;
    if (playList [i].bufferNo >= 0)
    {
      fl_select_browser_line
        (selectForm->browser,playList [i].bufferNo + 1);
    }
    else
    {
      fl_deselect_browser (selectForm->browser);
    }
    fl_show_form (selectForm->selectForm,
    FL_PLACE_MOUSE,FL_FULLBORDER,"Select");
    fl_deactivate_form (mixplayForm->mixplayForm);
  }
}

/*---------------------------------------------------------------------------
| CALLBACK mixplaySelectDbl_cb
---------------------------------------------------------------------------*/
void mixplaySelectDbl_cb (FL_OBJECT *ob,long data)
{
  fl_call_object_callback (mixplayForm->selectButton);
}

/*---------------------------------------------------------------------------
| CALLBACK mixplaySave_cb
---------------------------------------------------------------------------*/
void mixplaySave_cb (FL_OBJECT *ob,long data)
{
  if (fl_get_button (ob))
  {
    const char *filename;
    if (mixfile)
    {
      AFclosefile (mixfile);
      mixfile = 0;
    }
    fl_use_fselector (1);

    // Cannot select type
    int tempFormat      = fl_get_choice (format);
    int tempCompression = fl_get_choice (compression);
    fl_set_choice (format,1);
    fl_set_choice (compression,1);
    fl_deactivate_object (format);
    fl_deactivate_object (compression);

    if (first1)
    {
      filename = fl_show_fselector (
        "Please enter the filename to save from the mix play",
        ".",
        "*.aiff",0);
        first1 = 0;
    }
    else
    {
      filename = fl_show_fselector (
        "Please enter the filename to save from the mix play",0,0,0);
    }

    if (filename)
    {
      long longOutputRate;
      long outputRate;
      longOutputRate = getOutputRate ();
      if (longOutputRate == AL_RATE_UNDEFINED)
        outputRate = 48000;
      else
        outputRate = longOutputRate;

      AFfilesetup fileSetup;
      int irixFile = open (filename,O_RDONLY);
      if (irixFile != -1)
      {
        close (irixFile);
        if (!fl_show_question_old (filename,"exists already - overwrite ?",0))
        {
          fl_set_button (ob,0);
          return;
        }
      }

      // Open file

      fileSetup = AFnewfilesetup ();
      AFinitfilefmt (fileSetup,AF_FILE_AIFFC);
      AFinitrate (fileSetup,AF_DEFAULT_TRACK,(double) outputRate);
      AFinitsampfmt (fileSetup,AF_DEFAULT_TRACK,AF_SAMPFMT_TWOSCOMP,16);
      AFinitchannels (fileSetup,AF_DEFAULT_TRACK,2);
      mixfile = AFopenfile (filename,"w",fileSetup);
      if (!mixfile)
      {
        fl_set_button (ob,0);
        fl_show_alert ("Warning","Could not open save file","",TRUE);
      }
      AFfreefilesetup (fileSetup);
    }
    else
    {
      fl_set_button (ob,0);
    }
    fl_activate_object (format);
    fl_activate_object (compression);
    fl_set_choice (format,tempFormat);
    fl_set_choice (compression,tempCompression);
  }
  else
  {
    if (mixfile)
    {
      AFclosefile (mixfile);
      mixfile = 0;
    }
  }
}

/*---------------------------------------------------------------------------
| CALLBACK mixplayRateChoice_cb
---------------------------------------------------------------------------*/
void mixplayRateChoice_cb (FL_OBJECT *ob,long data)
{
  fl_set_choice (playForm->rateChoice,fl_get_choice (ob));
  fl_call_object_callback (playForm->rateChoice);
}

/*---------------------------------------------------------------------------
| CALLBACK mixplayDone_cb
---------------------------------------------------------------------------*/
void mixplayDone_cb (FL_OBJECT *ob,long data)
{
  DPMixPlayStop = 1;
  fl_hide_form (mixplayForm->mixplayForm);
  if (fl_get_sample_autowindows (mainForm->sample))
    rememberForms (1);
  fl_activate_all_forms ();
  int item = (int) mixplayForm->ldata;
  
  switch (item)
  {
    case 1 :
      fl_show_object (mixplayForm->fourGroup);
      fl_show_object (mixplayForm->deskGroup);
      break;
      
    case 2 :
      fl_show_object (mixplayForm->twoGroup);
      fl_show_object (mixplayForm->deskGroup);
      break;
      
    case 3 :
      fl_show_object (mixplayForm->twoGroup);
      fl_show_object (mixplayForm->fourGroup);
      break;
    
    default :
      break;
  }
  fl_activate_object (bufferForm->mixplay);
}

/*---------------------------------------------------------------------------
| CALLBACK mixplayRateMenu_cb
---------------------------------------------------------------------------*/

void mixplayRateMenu_cb (FL_OBJECT *ob,long data)
{
  int  item;
  char tempString [50];
  
  item=fl_get_menu (ob);
  switch (item)
  {
    case 1 :
      sprintf (tempString,"%ld",8000);
      fl_set_input (mixplayForm->rateInput,tempString);
      break;
    case 2 :
      sprintf (tempString,"%ld",11025);
      fl_set_input (mixplayForm->rateInput,tempString);
      break;
    case 3 :
      sprintf (tempString,"%ld",16000);
      fl_set_input (mixplayForm->rateInput,tempString);
      break;
    case 4 :
      sprintf (tempString,"%ld",22050);
      fl_set_input (mixplayForm->rateInput,tempString);
      break;
    case 5 :
      sprintf (tempString,"%ld",32000);
      fl_set_input (mixplayForm->rateInput,tempString);
      break;
    case 6 :
      sprintf (tempString,"%ld",44100);
      fl_set_input (mixplayForm->rateInput,tempString);
      break;
    case 7 :
      sprintf (tempString,"%ld",48000);
      fl_set_input (mixplayForm->rateInput,tempString);
      break;
    default:
      break;
  }
  fl_call_object_callback (mixplayForm->rateInput);
}

/*---------------------------------------------------------------------------
| CALLBACK mixplayRateInput_cb
---------------------------------------------------------------------------*/

void mixplayRateInput_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  long rate;
  const char *rateString;
  
  rateString = fl_get_input (mixplayForm->rateInput);
  if (!sscanf (rateString,"%ld",&rate)) return;
  if (rate < 1)
  {
    rate = 1;
    fl_set_input (mixplayForm->rateInput,"1");
  }

  long pvBuffer [2];
  pvBuffer [0] = AL_OUTPUT_RATE;
  pvBuffer [1] = rate;
  ALsetparams (AL_DEFAULT_DEVICE,pvBuffer,2);
  updateAudioGlobals (0,0);
  #ifdef LINUX
  fl_set_input (playForm->rateInput,rateString);
  fl_set_input (postForm->rateInput,rateString);
  #endif
}

/*---------------------------------------------------------------------------
| FUNCTION initialiseMixplayForm
---------------------------------------------------------------------------*/
void initialiseMixplayForm (int mode)
{
  initialiseSelectForm ();
  
  fl_set_button (mixplayForm->save,0);
  char startChar;
  
  switch (mode)
  {
    case 2 :
    {
      fl_set_object_color (mixplayForm->overloaded,FL_COL1,FL_COL1);
      fl_set_slider_bounds (mixplayForm->slider [8],400.0,0.0);
      fl_set_slider_value (mixplayForm->slider [8],200.0);
      mixplayMasterVol = 2.0;
      startChar = 'A';
      fl_set_slider_value (mixplayForm->twoSlider,0.0);
      fl_call_object_callback (mixplayForm->twoSlider);
      break;
    }
    
    case 4 :
    {
      fl_set_object_color (mixplayForm->overloaded,FL_COL1,FL_COL1);
      fl_set_slider_bounds (mixplayForm->slider [8],800.0,0.0);
      fl_set_slider_value (mixplayForm->slider [8],400.0);
      mixplayMasterVol = 4.0;
      startChar = 'A';
      fl_set_positioner_xvalue (mixplayForm->fourPositioner,0.0);
      fl_set_positioner_yvalue (mixplayForm->fourPositioner,0.0);
      fl_call_object_callback (mixplayForm->fourPositioner);
      break;
    }
    
    case 8 :
    {
      int i;
      fl_set_object_color (mixplayForm->overloaded,FL_COL1,FL_COL1);
      fl_set_slider_bounds (mixplayForm->slider [8],200.0,0.0);
      fl_set_slider_value (mixplayForm->slider [8],100.0);
      mixplayMasterVol = 1.0;
      startChar = '1';
      for (i=0; i<8; i++)
        fl_set_slider_value (mixplayForm->slider [i],50.0);
      break;
    }
  
    default :
      return;
  }

  int i;
  int lineNumber;

  // Initialise play list
  for (i=0; i<8; i++)
  {
    playList [i].sample   = 0;
    playList [i].bufferNo = -1;
    playList [i].vol      = 0.5;
    playList [i].frame    = -1;
    playList [i].release  = 0;
    playList [i].inc      = 1;
  }

  // Update browser
  fl_freeze_form (mixplayForm->mixplayForm);
  fl_clear_browser (mixplayForm->browser);
  char *filename;
  char *newFilename;
  i = 0;
  lineNumber = 0;
  while (lineNumber < 8)
  {
    if (lineNumber < mode)
    {
      if (i < 8)
      {
        filename = buffer [i]->getFilename ();
        if (buffer [i]->getValid () && filename && (strlen (filename) > 0))
        {
          filename = &(filename [stripFilename (filename)]);
          newFilename = new char [strlen (filename) + 4];
          strcpy (newFilename+3,filename);
          newFilename [0] = lineNumber + startChar;
          newFilename [1] = ' ';
          newFilename [2] = ' ';
          fl_add_browser_line (mixplayForm->browser,newFilename);
          delete [] newFilename;
          playList [lineNumber].sample   = buffer [i];
          playList [lineNumber].bufferNo = i;
          lineNumber++;
        }
        else if (buffer [i]->getValid ())
        {
          newFilename = new char [50];
          strcpy (newFilename,"   Untitled");
          newFilename [0] = lineNumber + startChar;
          newFilename [1] = ' ';
          newFilename [2] = ' ';
          fl_add_browser_line (mixplayForm->browser,newFilename);
          delete [] newFilename;
          playList [lineNumber].sample   = buffer [i];
          playList [lineNumber].bufferNo = i;
          lineNumber++;
        }
        i++;
      }
      else
      {
        newFilename = new char [50];
        strcpy (newFilename,"   ");
        newFilename [0] = lineNumber + startChar;
        newFilename [1] = ' ';
        newFilename [2] = ' ';
        fl_add_browser_line (mixplayForm->browser,newFilename);
        delete [] newFilename;
        lineNumber++;
      }
    }
    else
    {
      newFilename = new char [50];
      strcpy (newFilename,"   ");          
      fl_add_browser_line (mixplayForm->browser,newFilename);
      delete [] newFilename;
      lineNumber++;
    }
  }
  fl_unfreeze_form (mixplayForm->mixplayForm);
}

/*---------------------------------------------------------------------------
| FUNCTION initialiseSelectForm
---------------------------------------------------------------------------*/
void initialiseSelectForm ()
{
  int i;
  
  // Update browser
  fl_freeze_form (selectForm->selectForm);
  fl_clear_browser (selectForm->browser);
  char *filename;
  char *newFilename;
  for (i=0; i<8; i++)
  {
    filename = buffer [i]->getFilename ();
    if (buffer [i]->getValid () && filename && (strlen (filename) > 0))
    {
      filename = &(filename [stripFilename (filename)]);
      newFilename = new char [strlen (filename) + 4];
      strcpy (newFilename+3,filename);
      newFilename [0] = i + '1';
      newFilename [1] = ' ';
      newFilename [2] = ' ';
      fl_add_browser_line (selectForm->browser,newFilename);
      delete [] newFilename;
    }
    else
    {
      newFilename = new char [50];
      if (buffer [i]->getValid ())
        strcpy (newFilename,"   Untitled");
      else
        strcpy (newFilename,"   ");
        
      newFilename [0] = i + '1';
      newFilename [1] = ' ';
      newFilename [2] = ' ';
      fl_add_browser_line (selectForm->browser,newFilename);
      delete [] newFilename;
    }
  }
  fl_unfreeze_form (selectForm->selectForm);
}

/*---------------------------------------------------------------------------
| CALLBACK selectBrowser_cb
---------------------------------------------------------------------------*/
void selectBrowser_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK selectOKDbl_cb
---------------------------------------------------------------------------*/
void selectOKDbl_cb (FL_OBJECT *ob,long data)
{
  fl_call_object_callback (selectForm->OK);
}

/*---------------------------------------------------------------------------
| CALLBACK selectOK_cb
---------------------------------------------------------------------------*/
void selectOK_cb (FL_OBJECT *ob,long data)
{
  int i = fl_get_browser (selectForm->browser) - 1;
  if (i >= 0 && buffer [i]->getValid ())
  {
    playList [selectForm->ldata].sample   = buffer [i];
    playList [selectForm->ldata].bufferNo = i;
    playList [selectForm->ldata].frame    = -1;
    playList [selectForm->ldata].release  = 0;
    playList [selectForm->ldata].inc      = 1;
      
    char *filename;
    char *newFilename;
    char startChar;
    
    if (mixplayForm->ldata == 1) startChar = 'A';
    else if (mixplayForm->ldata == 2) startChar = 'A';
    else if (mixplayForm->ldata == 3) startChar = '1';
    else return;
    
    filename = buffer [i]->getFilename ();
    if (buffer [i]->getValid () && filename && (strlen (filename) > 0))
    {
      filename = &(filename [stripFilename (filename)]);
      newFilename = new char [strlen (filename) + 4];
      strcpy (newFilename+3,filename);
      newFilename [0] = selectForm->ldata + startChar;
      newFilename [1] = ' ';
      newFilename [2] = ' ';
      fl_replace_browser_line (mixplayForm->browser,
      selectForm->ldata + 1,newFilename);
      delete [] newFilename;
    }
    else
    {
      newFilename = new char [50];
      if (buffer [i]->getValid ())
        strcpy (newFilename,"   Untitled");
      else
        strcpy (newFilename,"   ");
        
      newFilename [0] = selectForm->ldata + startChar;
      newFilename [1] = ' ';
      newFilename [2] = ' ';
      fl_replace_browser_line (mixplayForm->browser,
      selectForm->ldata + 1,newFilename);
      delete [] newFilename;
    }
  }
  fl_hide_form (selectForm->selectForm);
  fl_activate_form (mixplayForm->mixplayForm);
  fl_deselect_browser (mixplayForm->browser);
}

/*---------------------------------------------------------------------------
| CALLBACK selectCancel_cb
---------------------------------------------------------------------------*/
void selectCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (selectForm->selectForm);
  fl_activate_form (mixplayForm->mixplayForm);
  fl_deselect_browser (mixplayForm->browser);
}

/***************************************************************************/
