/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : DPSample.inl
|
| Revision    : 1.0
| Date        : 04/03/96
|
| Object      : DPSample
|
| Description : DPSample inline methods for handling audio samples
|
| (c) Richard Kent 1996
|
| $Id$
|
****************************************************************************/

/*---------------------------------------------------------------------------
| Modifiers
---------------------------------------------------------------------------*/

void DPSample::setChanged (long newChanged)
{
//cerr << "DPSample::setChanged (" << newChanged<< ")" << endl;
  changed = newChanged;
}

void DPSample::setFormat (long newFormat)
{
//cerr << "DPSample::setFormat (" << newFormat << ")" << endl;
  if (format != newFormat) changed = TRUE;
  format = newFormat;
}

void DPSample::setCompression (long newCompression)
{
//cerr << "DPSample::setCompression (" << newCompression << ")" << endl;
  if (compression != newCompression) changed = TRUE;
  compression = newCompression;
}

void DPSample::setRate (double newRate)
{
//cerr << "DPSample::setRate (" << newRate << ")" << endl;
  if (rate != newRate) changed = TRUE;
  rate = newRate;
}

void DPSample::setFilename (char *newFilename)
{
//cerr << "DPSample::setFilename (" << newFilename << ")" << endl;
  if (strcmp (filename,newFilename)) changed = TRUE;
  delete [] filename;
  filename = new char [strlen (newFilename) + 1];
  strcpy (filename,newFilename);
}

void DPSample::setName (char *newName)
{
//cerr << "DPSample::setName (" << newName << ")" << endl;
  if (strcmp (name,newName)) changed = TRUE;
  delete [] name;
  name = new char [strlen (newName) + 1];
  strcpy (name,newName);
}

void DPSample::setCopyright (char *newCopyright)
{
//cerr << "DPSample::setCopyright (" << newCopyright << ")" << endl;
  if (strcmp (copyright,newCopyright)) changed = TRUE;
  delete [] copyright;
  copyright = new char [strlen (newCopyright) + 1];
  strcpy (copyright,newCopyright);
}

void DPSample::setAuthor (char *newAuthor)
{
//cerr << "DPSample::setAuthor (" << newAuthor << ")" << endl;
  if (strcmp (author,newAuthor)) changed = TRUE;
  delete [] author;
  author = new char [strlen (newAuthor) + 1];
  strcpy (author,newAuthor);
}

void DPSample::setAnnotation (char *newAnnotation)
{
//cerr << "DPSample::setAnnotation (" << newAnnotation << ")" << endl;
  if (strcmp (annotation,newAnnotation)) changed = TRUE;
  delete [] annotation;
  annotation = new char [strlen (newAnnotation) + 1];
  strcpy (annotation,newAnnotation);
}

void DPSample::setSusLoopMode (long newSusLoopMode)
{
//cerr << "DPSample::setSusLoopMode (" << newSusLoopMode << ")" << endl;
  if (susLoopMode != newSusLoopMode) changed = TRUE;
  susLoopMode = newSusLoopMode;
}

void DPSample::setSusLoopStart (long newSusLoopStart)
{
//cerr << "DPSample::setSusLoopStart (" << newSusLoopStart << ") ";
  if (susLoopStart != newSusLoopStart) changed = TRUE;
  if (newSusLoopStart < 0)
    susLoopStart = 0;
  else if (newSusLoopStart > frames)
    susLoopStart = frames;
  else
    susLoopStart = newSusLoopStart;
  if (susLoopStart > susLoopEnd)
    susLoopStart = susLoopEnd;
//cerr << "[" << susLoopStart << "," << susLoopEnd << "]" << endl;
}

void DPSample::setSusLoopEnd (long newSusLoopEnd)
{
//cerr << "DPSample::setSusLoopEnd (" << newSusLoopEnd << ") ";
  if (susLoopEnd != newSusLoopEnd) changed = TRUE;
  if (newSusLoopEnd < 0)
    susLoopEnd = 0;
  else if (newSusLoopEnd > frames)
    susLoopEnd = frames;
  else
    susLoopEnd = newSusLoopEnd;
  if (susLoopEnd < susLoopStart)
    susLoopEnd = susLoopStart;
//cerr << "[" << susLoopStart << "," << susLoopEnd << "]" << endl;
}

void DPSample::setSusLoop (long newSusLoopStart,long newSusLoopEnd)
{
//cerr << "DPSample::setSusLoop ";
  if (susLoopStart != newSusLoopStart) changed = TRUE;
  else if (susLoopEnd != newSusLoopEnd) changed = TRUE;
  if (newSusLoopStart < 0)
    susLoopStart = 0;
  else if (newSusLoopStart > frames)
    susLoopStart = frames;
  else
    susLoopStart = newSusLoopStart;
  if (newSusLoopEnd < 0)
    susLoopEnd = 0;
  else if (newSusLoopEnd > frames)
    susLoopEnd = frames;
  else
    susLoopEnd = newSusLoopEnd;
  if (susLoopEnd < susLoopStart)
    susLoopEnd = susLoopStart;
//cerr << "[" << susLoopStart << "," << susLoopEnd << "]" << endl;
}

void DPSample::setRelLoopMode (long newRelLoopMode)
{
//cerr << "DPSample::setRelLoopMode (" << newRelLoopMode << ")" << endl;
  if (relLoopMode != newRelLoopMode) changed = TRUE;
  relLoopMode = newRelLoopMode;
}

void DPSample::setRelLoopStart (long newRelLoopStart)
{
//cerr << "DPSample::setRelLoopStart (" << newRelLoopStart << ") ";
  if (relLoopStart != newRelLoopStart) changed = TRUE;
  if (newRelLoopStart < 0)
    relLoopStart = 0;
  else if (newRelLoopStart > frames)
    relLoopStart = frames;
  else
    relLoopStart = newRelLoopStart;
  if (relLoopStart > relLoopEnd)
    relLoopStart = relLoopEnd;
//cerr << "[" << relLoopStart << "," << relLoopEnd << "]" << endl;
}

void DPSample::setRelLoopEnd (long newRelLoopEnd)
{
//cerr << "DPSample::setRelLoopEnd (" << newRelLoopEnd << ") ";
  if (relLoopEnd != newRelLoopEnd) changed = TRUE;
  if (newRelLoopEnd < 0)
    relLoopEnd = 0;
  else if (newRelLoopEnd > frames)
    relLoopEnd = frames;
  else
    relLoopEnd = newRelLoopEnd;
  if (relLoopEnd < relLoopStart)
    relLoopEnd = relLoopStart;
//cerr << "[" << relLoopStart << "," << relLoopEnd << "]" << endl;
}

void DPSample::setRelLoop (long newRelLoopStart,long newRelLoopEnd)
{
//cerr << "DPSample::setRelLoop ";
  if (relLoopStart != newRelLoopStart) changed = TRUE;
  else if (relLoopEnd != newRelLoopEnd) changed = TRUE;
  if (newRelLoopStart < 0)
    relLoopStart = 0;
  else if (newRelLoopStart > frames)
    relLoopStart = frames;
  else
    relLoopStart = newRelLoopStart;
  if (newRelLoopEnd < 0)
    relLoopEnd = 0;
  else if (newRelLoopEnd > frames)
    relLoopEnd = frames;
  else
    relLoopEnd = newRelLoopEnd;
  if (relLoopEnd < relLoopStart)
    relLoopEnd = relLoopStart;
//cerr << "[" << relLoopStart << "," << relLoopEnd << "]" << endl;
}

void DPSample::setRangeValid (long newRangeValid)
{
//cerr << "DPSample::setRangeValid (" << newRangeValid << ") ";
  rangeValid = newRangeValid;
}

void DPSample::setRangeStart (long newRangeStart)
{
//cerr << "DPSample::setRangeStart (" << newRangeStart << ") ";
  rangeValid = TRUE;
  if (newRangeStart < 0)
    rangeStart = 0;
  else if (newRangeStart > frames)
    rangeStart = frames;
  else
    rangeStart = newRangeStart;
  if (rangeStart > rangeEnd)
  {
    long temp  = rangeEnd;
    rangeEnd   = rangeStart;
    rangeStart = temp;
  }
//cerr << "[" << rangeStart << "," << rangeEnd << "]" << endl;
}

void DPSample::setRangeEnd (long newRangeEnd)
{
//cerr << "DPSample::setRangeEnd (" << newRangeEnd << ") ";
  rangeValid = TRUE;
  if (newRangeEnd < 0)
    rangeEnd = 0;
  else if (newRangeEnd > frames)
    rangeEnd = frames;
  else
    rangeEnd = newRangeEnd;
  if (rangeEnd < rangeStart)
  {
    long temp  = rangeEnd;
    rangeEnd   = rangeStart;
    rangeStart = temp;
  }
//cerr << "[" << rangeStart << "," << rangeEnd << "]" << endl;
}

void DPSample::setRange (long newRangeStart,long newRangeEnd)
{
//cerr << "DPSample::setRange ";
  rangeValid = TRUE;
  if (newRangeStart < 0)
    rangeStart = 0;
  else if (newRangeStart > frames)
    rangeStart = frames;
  else
    rangeStart = newRangeStart;
  if (newRangeEnd < 0)
    rangeEnd = 0;
  else if (newRangeEnd > frames)
    rangeEnd = frames;
  else
    rangeEnd = newRangeEnd;
  if (rangeEnd < rangeStart)
  {
    long temp  = rangeEnd;
    rangeEnd   = rangeStart;
    rangeStart = temp;
  }
//cerr << "[" << rangeStart << "," << rangeEnd << "]" << endl;
}

void DPSample::setDisplayStart (long newDisplayStart)
{
//cerr << "DPSample::setDisplayStart (" << newDisplayStart << ") ";
  if (newDisplayStart < 0)
    displayStart = 0;
  else if (newDisplayStart > frames)
    displayStart = frames;
  else
    displayStart = newDisplayStart;
  if (displayStart > displayEnd)
    displayEnd = displayStart;
//cerr << "[" << displayStart << "," << displayEnd << "]" << endl;
}

void DPSample::setDisplayEnd (long newDisplayEnd)
{
//cerr << "DPSample::setDisplayEnd (" << newDisplayEnd << ")" << endl;
  if (newDisplayEnd < 0)
    displayEnd = 0;
  else if (newDisplayEnd > frames)
    displayEnd = frames;
  else
    displayEnd = newDisplayEnd;
  if (displayEnd < displayStart)
    displayStart = displayEnd;
//cerr << "[" << displayStart << "," << displayEnd << "]" << endl;
}

void DPSample::setDisplay (long newDisplayStart,long newDisplayEnd)
{
//cerr << "DPSample::setDisplay ";
  if (newDisplayStart < 0)
    displayStart = 0;
  else if (newDisplayStart > frames)
    displayStart = frames;
  else
    displayStart = newDisplayStart;
  if (newDisplayEnd < 0)
    displayEnd = 0;
  else if (newDisplayEnd > frames)
    displayEnd = frames;
  else
    displayEnd = newDisplayEnd;
  if (displayStart > displayEnd)
    displayEnd = displayStart;
//cerr << "[" << displayStart << "," << displayEnd << "]" << endl;
}
void DPSample::setRedraw (long newRedraw)
{
//cerr << "DPSample::setRedraw (" << newRedraw << ")" << endl;
  redraw = newRedraw;
}

void DPSample::setCurrentPos (long newCurrentPos)
{
//cerr << "DPSample::setCurrentPos (" << newCurrentPos << ")" << endl;
  if (newCurrentPos < 0 || newCurrentPos > frames)
    currentPos = -1;
  else
    currentPos = newCurrentPos;
}

/*---------------------------------------------------------------------------
| Accessors
---------------------------------------------------------------------------*/

long   DPSample::getValid () const        { return valid;        }
long   DPSample::getChanged () const      { return changed;      }
long   DPSample::getFormat () const       { return format;       }
long   DPSample::getCompression () const  { return compression;  }
double DPSample::getRate () const         { return rate;         }
long   DPSample::getWidth () const        { return width;        }
long   DPSample::getChannels () const     { return channels;     }
long   DPSample::getFrames () const       { return frames;       }
long   DPSample::getSamples () const      { return samples;      }
long   DPSample::getBytes () const        { return bytes;        }
long   DPSample::getAESused () const      { return AESused;      }
unsigned char *DPSample::getAES ()        { return AES;          }
char   *DPSample::getFilename () const    { return filename;     }
char   *DPSample::getName () const        { return name;         }
char   *DPSample::getCopyright () const   { return copyright;    }
char   *DPSample::getAuthor () const      { return author;       }
char   *DPSample::getAnnotation () const  { return annotation;   }
long   DPSample::getSusLoopMode () const  { return susLoopMode;  }
long   DPSample::getSusLoopStart () const { return susLoopStart; }
long   DPSample::getSusLoopEnd () const   { return susLoopEnd;   }
long   DPSample::getRelLoopMode () const  { return relLoopMode;  }
long   DPSample::getRelLoopStart () const { return relLoopStart; }
long   DPSample::getRelLoopEnd () const   { return relLoopEnd;   }
long   DPSample::getRangeValid () const   { return rangeValid;   }
long   DPSample::getRangeStart () const   { return rangeStart;   }
long   DPSample::getRangeEnd () const     { return rangeEnd;     }
long   DPSample::getDisplayStart () const { return displayStart; }
long   DPSample::getDisplayEnd () const   { return displayEnd;   }
long   DPSample::getRedraw () const       { return redraw;       }
long   DPSample::getCurrentPos () const   { return currentPos;   }

long DPSample::getLzero (long frame)
{
  long i=frame;
  long found=FALSE;
  long sign1;
  long sign2;
  long sign3;
  long sign4;
  if (frame == 0) return 0;
  if (channels == 1)
  {
    --i;
    sign1 = sign (getFrame24 (i,0));
    while (i > 0 && !found)
    {
      --i;
      if (sign (getFrame24 (i,0)) != sign1) found = TRUE;
    }
    if (found) return i + 1;
    if (sign (getFrame24 (0,0)) == 0) return 0;
    return frame;
  }
  else if (channels == 2)
  {
    --i;
    sign1 = sign (getFrame24 (i,0));
    sign2 = sign (getFrame24 (i,1));
    while (i > 0 && !found)
    {
      --i;
      if (sign (getFrame24 (i,0)) != sign1) found = TRUE;
      if (sign (getFrame24 (i,1)) != sign2) found = TRUE;
    }
    if (found) return i + 1;
    if (sign (getFrame24 (0,0)) == 0) return 0;
    if (sign (getFrame24 (0,1)) == 0) return 0;
    return frame;
  }
  else
  {
    --i;
    sign1 = sign (getFrame24 (i,0));
    sign2 = sign (getFrame24 (i,1));
    sign3 = sign (getFrame24 (i,2));
    sign4 = sign (getFrame24 (i,3));
    while (i > 0 && !found)
    {
      --i;
      if (sign (getFrame24 (i,0)) != sign1) found = TRUE;
      if (sign (getFrame24 (i,1)) != sign2) found = TRUE;
      if (sign (getFrame24 (i,2)) != sign3) found = TRUE;
      if (sign (getFrame24 (i,3)) != sign4) found = TRUE;
    }
    if (found) return i + 1;
    if (sign (getFrame24 (0,0)) == 0) return 0;
    if (sign (getFrame24 (0,1)) == 0) return 0;
    if (sign (getFrame24 (0,2)) == 0) return 0;
    if (sign (getFrame24 (0,3)) == 0) return 0;
    return frame;
  }
}

long DPSample::getL (long frame,long scrollFrames)
{
  long i=frame;
  i -= scrollFrames;
  if (i > 0) return i;
  else return 0;
}

long DPSample::getR (long frame,long scrollFrames)
{
  long i=frame;
  i += scrollFrames;
  if (i < frames) return i;
  else return frames;
}

long DPSample::getRzero  (long frame)
{
  long i=frame;
  long found=FALSE;
  long sign1;
  long sign2;
  long sign3;
  long sign4;
  if (frame == frames) return frames;
  if (channels == 1)
  {
    sign1 = sign (getFrame24 (i,0));
    while (i < (frames - 1) && !found)
    {
      ++i;
      if (sign (getFrame24 (i,0)) != sign1) found = TRUE;
    }
    if (found) return i;
    if (sign (getFrame24 (frames - 1,0)) == 0) return frames;
    return frame;
  }
  else if (channels == 2)
  {
    sign1 = sign (getFrame24 (i,0));
    sign2 = sign (getFrame24 (i,1));
    while (i < (frames - 1) && !found)
    {
      ++i;
      if (sign (getFrame24 (i,0)) != sign1) found = TRUE;
      if (sign (getFrame24 (i,1)) != sign2) found = TRUE;
    }
    if (found) return i;
    if (sign (getFrame24 (frames - 1,0)) == 0) return frames;
    if (sign (getFrame24 (frames - 1,1)) == 0) return frames;
    return frame;
  }
  else
  {
    sign1 = sign (getFrame24 (i,0));
    sign2 = sign (getFrame24 (i,1));
    sign3 = sign (getFrame24 (i,2));
    sign4 = sign (getFrame24 (i,3));
    while (i < (frames - 1) && !found)
    {
      ++i;
      if (sign (getFrame24 (i,0)) != sign1) found = TRUE;
      if (sign (getFrame24 (i,1)) != sign2) found = TRUE;
      if (sign (getFrame24 (i,2)) != sign3) found = TRUE;
      if (sign (getFrame24 (i,3)) != sign4) found = TRUE;
    }
    if (found) return i;
    if (sign (getFrame24 (frames - 1,0)) == 0) return frames;
    if (sign (getFrame24 (frames - 1,1)) == 0) return frames;
    if (sign (getFrame24 (frames - 1,2)) == 0) return frames;
    if (sign (getFrame24 (frames - 1,3)) == 0) return frames;
    return frame;
  }
}

/*---------------------------------------------------------------------------
| Frame functions
---------------------------------------------------------------------------*/

long DPSample::getFrame24 (long frame,long channel)
{
  // return frame as 24 bit number stored in long
  if (frame < 0 || frame >= frames) return 0;
  if (channels == 1)
    channel = 0;
  else if (channels == 2)
    channel = channel % 2;
  else
    channel = channel % 4;
  if (width == 1)
    return 65536 * ((signed char *) data) [frame*channels + channel];
  else
    return 256 * ((short *) data) [frame*channels + channel];
}

long DPSample::getFrame24Extra (long frame,long channel)
{
  // return frame as 24 bit number stored in long
  if (frame < 0) return 0;
  if (channels == 1)
    channel = 0;
  else if (channels == 2)
    channel = channel % 2;
  else
    channel = channel % 4;
  if (frame < frames)
  {
    if (width == 1)
      return 65536 * ((signed char *) data) [frame*channels + channel];
    else
      return 256 * ((short *) data) [frame*channels + channel];
  }
  
  extraListEntry *e;
  
  e = extra;
  while (e)
  {
    if ((frame >= e->start) && (frame <= e->end))
    {
      if (width == 1)
        return 65536 * ((signed char *) e->data)
          [(frame - e->start)*channels + channel];
      else
        return 256 * ((short *) e->data)
          [(frame - e->start)*channels + channel];
    }
    e = e->next;
  }
  return 0;
}

void DPSample::setFrame24 (long frame,long channel,long newFrame)
{
  // store frame given as 24 bit number
  changed = TRUE;
  if (frame < 0 || frame >= frames) return;
  if (channels == 1)
    channel = 0;
  else if (channels == 2)
    channel = channel % 2;
  else
    channel = channel % 4;
  if (width == 1)
    ((signed char *) data) [frame*channels + channel] = newFrame / 65536;
  else
    ((short *) data) [frame*channels + channel] = newFrame / 256;
}

void DPSample::setFrame24Extra (long frame,long channel,long newFrame)
{
  // store frame given as 24 bit number
  changed = TRUE;
  if (frame < 0) return;
  if (channels == 1)
    channel = 0;
  else if (channels == 2)
    channel = channel % 2;
  else
    channel = channel % 4;
  if (frame < frames)
  {
    if (width == 1)
      ((signed char *) data) [frame*channels + channel] = newFrame / 65536;
    else
      ((short *) data) [frame*channels + channel] = newFrame / 256;
    return;
  }

  extraListEntry *e;

  if (!extra) e = expand (1.0);
  else e = extra;
  
  while (e)
  {
    if ((frame >= e->start) && (frame <= e->end))
    {
      if (width == 1)
        ((signed char *) e->data)
          [(frame - e->start)*channels + channel] = newFrame / 65536;
      else
        ((short *) e->data)
          [(frame - e->start)*channels + channel] = newFrame / 256;
      return;
    }
    e = e->next;
    if (!e) e = expand (1.0);
  }
}

short DPSample::getFrame16 (long frame,long channel)
{
  // return frame as 16 bit number stored in short
  if (frame < 0 || frame >= frames) return 0;
  if (channels == 1)
    channel = 0;
  else if (channels == 2)
    channel = channel % 2;
  else
    channel = channel % 4;
  if (width == 1)
    return 256 * ((signed char *) data) [frame*channels + channel];
  else
    return ((short *) data) [frame*channels + channel];
}

short DPSample::getFrame16Extra (long frame,long channel)
{
  return ((short)((getFrame24Extra (frame,channel)) / 256));
}

void DPSample::setFrame16 (long frame,long channel,short newFrame)
{
  // store frame given as 16 bit number
  changed = TRUE;
  if (frame < 0 || frame >= frames) return;
  if (channels == 1)
    channel = 0;
  else if (channels == 2)
    channel = channel % 2;
  else
    channel = channel % 4;
  if (width == 1)
    ((signed char *) data) [frame*channels + channel] = newFrame / 256;
  else
    ((short *) data) [frame*channels + channel] = newFrame;
}

void DPSample::setFrame16Extra (long frame,long channel,short newFrame)
{
  setFrame24Extra (frame,channel,((long) newFrame) * 256);
}

double DPSample::getFrameDb (long frame,long channel)
{
  // return frame as double value [-1.0,1.0]
  return (((double) getFrame24 (frame,channel)) / MAX23);
}

double DPSample::getFrameDbExtra (long frame,long channel)
{
  // return frame as double value [-1.0,1.0]
  return (((double) getFrame24Extra (frame,channel)) / MAX23);
}

void DPSample::setFrameDb (long frame,long channel,double newFrame)
{
  // store frame given as double value [-1.0,1.0]
  if (newFrame > 1.0) newFrame = 1.0;
  else if (newFrame < -1.0) newFrame = -1.0;
  long value = (long)(MAX23 * newFrame);
  if (value < -MAX23) value = -MAX23;
  else if (value > MAX23_1) value = MAX23_1;
  setFrame24 (frame,channel,value);
}

void DPSample::setFrameDbExtra (long frame,long channel,double newFrame)
{
  // store frame given as double value [-1.0,1.0]
  if (newFrame > 1.0) newFrame = 1.0;
  else if (newFrame < -1.0) newFrame = -1.0;
  long value = (long)(MAX23 * newFrame);
  if (value < -MAX23) value = -MAX23;
  else if (value > MAX23_1) value = MAX23_1;
  setFrame24Extra (frame,channel,value);
}

long DPSample::sign (long x)
{
  if (x > 0) return 1;
  if (x < 0) return -1;
  return 0;
}

/*---------------------------------------------------------------------------
| Time functions
---------------------------------------------------------------------------*/

long DPSample::timeToFrame (double time)
{
//cerr << "DPSample::timeToFrame" << endl;
  return (long) (rate * time);
}

double DPSample::frameToTime (long frame)
{
//cerr << "DPSample::frameToTime" << endl;
  return ((double) frame) / rate;
}

/***************************************************************************/
