/*$Id: d_diode.cc,v 15.10 1999/10/11 06:30:33 al Exp $ -*- C++ -*-
 * diode model.
 * netlist syntax:
 * device:  dxxxx n+ n- mname <area> <off> <ic=vd> <model-card-args>
 * model:   .model mname D <args>
 */
#include "e_aux.h"
#include "d_admit.h"
#include "d_cap.h"
#include "ap.h"
#include "d_diode.h"
/*--------------------------------------------------------------------------*/
//		MODEL_DIODE::MODEL_DIODE();
//	void	MODEL_DIODE::parse(CS& cmd);
// 	void	MODEL_DIODE::print(int,int)const;

//		DIODE_COMMON::DIODE_COMMON(int c)
//		DIODE_COMMON::DIODE_COMMON(const DIODE_COMMON& p, int c)
//	void	DIODE_COMMON::parse(CS& cmd);
// 	void	DIODE_COMMON::print(int)const;
//	void	DIODE_COMMON::expand();

//		DEV_DIODE::DEV_DIODE();
//		DEV_DIODE::DEV_DIODE(const DEV_DIODE& p);
//	void	DEV_DIODE::parse(CS& cmd);
// 	void	DEV_DIODE::print(int,int)const;
//	void	DEV_DIODE::expand();
//	double	DEV_DIODE::probe_tr_num(const std::string& what)const;
//	void	EVAL_DIODE_Yj::tr_eval(COMPONENT* d);
//	void	EVAL_DIODE_Cj::tr_eval(COMPONENT* d);
/*--------------------------------------------------------------------------*/
int DEV_DIODE::Count = 0;
int DIODE_COMMON::Count = -3;
int MODEL_DIODE::Count = 0;
static EVAL_DIODE_Cj Eval_Cj(CC_STATIC);
static EVAL_DIODE_Yj Eval_Yj(CC_STATIC);
static DIODE_COMMON Default_DIODE(CC_STATIC);
/*--------------------------------------------------------------------------*/
MODEL_DIODE::MODEL_DIODE()
  :MODEL_CARD(),
   js	    (1e-14),
   rs	    (0.0),
   n 	    (1.0),
   tt	    (0.0),
   cjo	    (0.0),
   pb	    (1.0),
   mj	    (0.5),
   eg	    (1.11),
   xti	    (3.0),
   kf	    (NOT_INPUT),
   af	    (NOT_INPUT),
   fc	    (0.5),
   bv	    (0.0),	/* infinity */
   ibv	    (1e-3),
   cjsw	    (0.0),
   mjsw	    (0.33),
   gparallel(0.0),
   flags    (USE_OPT),
   fcpb	    (fc * pb)
{
  ++Count;
}
/*--------------------------------------------------------------------------*/
void MODEL_DIODE::parse(CS& cmd)
{
  cmd.skiparg();			/* skip known ".model" */
  parse_label(cmd);
  cmd.skiparg();			/* skip known "d" */
  cmd.skiplparen();
  cmd.stuck();
  do{
    cmd.get("TNOM",	&_tnom,	    mOFFSET,	-ABS_ZERO);
    cmd.get("IS",	&js, 	    mPOSITIVE);
    cmd.get("RS",	&rs, 	    mPOSITIVE);
    cmd.get("N",   	&n,  	    mPOSITIVE);
    cmd.get("TT",  	&tt, 	    mPOSITIVE);
    cmd.get("CJo", 	&cjo,	    mPOSITIVE);
    cmd.get("VJ",  	&pb, 	    mPOSITIVE);
    cmd.get("PB",  	&pb, 	    mPOSITIVE);
    cmd.get("Mj",  	&mj, 	    mPOSITIVE);
    cmd.get("EGap",	&eg, 	    mPOSITIVE);
    cmd.get("XTI", 	&xti,	    mPOSITIVE);
    cmd.get("KF",  	&kf, 	    mPOSITIVE);
    cmd.get("AF",  	&af, 	    mPOSITIVE);
    cmd.get("FC",  	&fc, 	    mPOSITIVE);
    cmd.get("BV",  	&bv,	    mPOSITIVE);
    cmd.get("IBV", 	&ibv,	    mPOSITIVE);
    cmd.get("CJSw",	&cjsw,	    mPOSITIVE);
    cmd.get("MJSw",	&mjsw,	    mPOSITIVE);
    cmd.get("GParallel",&gparallel);
    cmd.get("FLAgs",	&flags,	    mOCTAL);
    cmd.set("NOFLAGS",	&flags,	    USE_OPT);
  }while (cmd.more() && !cmd.stuck());
  cmd.skiprparen();
  cmd.check(bWARNING, "what's this?");
  fcpb = fc * pb;
  {if (_tnom == NOT_INPUT){
    _tnom = OPT::tnom;
  }else{
    untested();
  }}
}
/*--------------------------------------------------------------------------*/
void MODEL_DIODE::print(OMSTREAM where, int)const
{
  where.setfloatwidth(7);
  where << ".model  " << short_label() << "  d  (";
  where	<<   "tnom="<< _tnom+ABS_ZERO;
  where << "  is="  << js;
  where << "  rs="  << rs;
  where << "  n="   << n;
  where << "  tt="  << tt;
  where << "  cjo=" << cjo;
  where << "  vj="  << pb;
  where << "  m="   << mj;
  where << "  eg="  << eg;
  where << "  xti=" << xti;
  if (kf != NOT_INPUT)
    where << "  kf="  << kf;
  if (af != NOT_INPUT)
    where << "  af="  << af;
  where << "  fc="  << fc;
  where << "  bv="  << bv;
  where << "  ibv=" << ibv;
  if (cjsw != 0.){
    where << "  cjsw=" << cjsw;
    where << "  mjsw=" << mjsw;
  }
  if (gparallel != 0){
    where << "  gparallel=" << gparallel;
  }
  if (!(flags & USE_OPT)){
    where << "  flags="<< flags;
  }
  where << ")\n";
}
/*--------------------------------------------------------------------------*/
DIODE_COMMON::DIODE_COMMON(int c)
  :COMPONENT_COMMON(c),
   is	    (NOT_INPUT),
   rs	    (NOT_INPUT),
   cj	    (NOT_INPUT),
   cjsw	    (NOT_INPUT),
   area	    (1.0),
   perim    (0.0),
   gparallel(NOT_INPUT),
   ic	    (NOT_INPUT),
   off	    (false)
{
  calc.is = calc.rs = calc.cj = calc.cjsw = calc.gparallel = false;
  ++Count;
}
/*--------------------------------------------------------------------------*/
DIODE_COMMON::DIODE_COMMON(const DIODE_COMMON& p, int c)
  :COMPONENT_COMMON(p,c),
   is	    (p.is),
   rs	    (p.rs),
   cj	    (p.cj),
   cjsw	    (p.cjsw),
   area	    (p.area),
   perim    (p.perim),
   gparallel(p.gparallel),
   ic	    (p.ic),
   off	    (p.off),
   calc	    (p.calc)
{
  ++Count;
}
/*--------------------------------------------------------------------------*/
void DIODE_COMMON::parse(CS& cmd)
{
  assert(!has_model());

  parse_modelname(cmd);
  if (cmd.is_pfloat()){
    area = cmd.ctopf();
  }
  cmd.stuck();
  do{
    cmd.get("Area",	&area,      mPOSITIVE);
    cmd.get("Perim",	&perim,     mPOSITIVE);
    cmd.get("IC",	&ic);
    cmd.get("OFF",	&off);
    cmd.get("IS",	&is,        mPOSITIVE);
    cmd.get("Rs",	&rs,        mPOSITIVE);
    cmd.get("Cjo",	&cj,        mPOSITIVE);
    cmd.get("CJSW",	&cjsw,	    mPOSITIVE);
    cmd.get("GParallel",&gparallel);
  }while (cmd.more() && !cmd.stuck());
  cmd.check(bWARNING, "what's this?");
}
/*--------------------------------------------------------------------------*/
void DIODE_COMMON::print(OMSTREAM where)const
{
  where << "  " << modelname();

  where.setfloatwidth(7);
  where << ' ' << area;
  if (perim != 0.)
    where << "  perim=" << perim;
  if (off)
    where << "  off";
  if (ic != NOT_INPUT)
    where << "  ic=" <<  ic;
  if (!calc.is  &&  is != NOT_INPUT)
    where << "  is=" <<  is;
  if (!calc.rs  &&  rs != NOT_INPUT)
    where << "  rs=" <<  rs;
  if (!calc.cj  &&  cj != NOT_INPUT)
    where << "  cj=" <<  cj;
  if (!calc.cjsw  &&  cjsw != NOT_INPUT)
    where << "  cjsw=" << cjsw;
  if (!calc.gparallel && gparallel != NOT_INPUT)
    where << "  gparallel=" << gparallel;
  if (calc.is  &&  calc.rs  &&  calc.cj  &&  calc.cjsw)
    where << "\n*+";
  if (calc.is  &&  is != NOT_INPUT)
    where << "  is=" <<  is;
  if (calc.rs  &&  rs != NOT_INPUT)
    where << "  rs=" <<  rs;
  if (calc.cj  &&  cj != NOT_INPUT)
    where << "  cj=" <<  cj;
  if (calc.cjsw  &&  cjsw != NOT_INPUT)
    where << "  cjsw=" << cjsw;
  if (calc.gparallel && gparallel != NOT_INPUT)
    where << "  gparallel=" << gparallel;
  where << '\n';
}
/*--------------------------------------------------------------------------*/
void DIODE_COMMON::expand()
{
  const MODEL_DIODE* m = dynamic_cast<const MODEL_DIODE*>(attach_model());
  if (!m){
    untested();
    error(bERROR, "model " + modelname() + " is not a diode (D)\n");
  }
  
  if (calc.is  ||  is == NOT_INPUT){
    is = m->js * area;
    calc.is = true;
  }
  {if (calc.rs  ||  rs == NOT_INPUT){
    rs = m->rs / area;
    calc.rs = true;
  }else{
    untested();
  }}
  if (calc.cj  ||  cj == NOT_INPUT){
    cj = m->cjo * area;
    calc.cj = true;
  }
  {if (calc.cjsw  ||  cjsw == NOT_INPUT){
    cjsw = m->cjsw * perim;
    calc.cjsw = true;
  }else{
    untested();
  }}
  if (calc.gparallel || gparallel == NOT_INPUT){
    gparallel = m->gparallel * area;
  }
}
/*--------------------------------------------------------------------------*/
DEV_DIODE::DEV_DIODE()
  :BASE_SUBCKT(),
   region(UNKNOWN),
   isat	 (NOT_VALID),
   gd	 (NOT_VALID),
   Yj	 (0),
   Cj	 (0)
{
  attach_common(&Default_DIODE);
  ++Count;
}
/*--------------------------------------------------------------------------*/
DEV_DIODE::DEV_DIODE(const DEV_DIODE& p)
  :BASE_SUBCKT(p),
   region(UNKNOWN),
   isat	 (NOT_VALID),
   gd	 (NOT_VALID),
   Yj	 (0),
   Cj	 (0)
{
  untested();
  ++Count;
}
/*--------------------------------------------------------------------------*/
void DEV_DIODE::parse(CS& cmd)
{
  const DIODE_COMMON* cc = prechecked_cast<const DIODE_COMMON*>(common());
  assert(cc);
  DIODE_COMMON* c = new DIODE_COMMON(*cc);
  assert(c);
  
  parse_Label(cmd);
  parse_nodes(cmd,numnodes(),numnodes());
  c->parse(cmd);
  attach_common(c);
}
/*--------------------------------------------------------------------------*/
void DEV_DIODE::print(OMSTREAM where, int)const
{
  const DIODE_COMMON* c = prechecked_cast<const DIODE_COMMON*>(common());
  assert(c);

  where << short_label();
  printnodes(where,numnodes());
  c->print(where);
}
/*--------------------------------------------------------------------------*/
void DEV_DIODE::expand()
{
  const DIODE_COMMON* cc = prechecked_cast<const DIODE_COMMON*>(common());
  assert(cc);
  DIODE_COMMON* c = const_cast<DIODE_COMMON*>(cc);
  assert(c);
  c->expand();

  {if (c->cj != 0.  ||  c->cjsw != 0.){
    if (!Cj){
      Cj = new DEV_CAPACITANCE;
      subckt().push_front(Cj);
    }
    Cj->set("Cj", this, &Eval_Cj, 0., n[OUT1], n[OUT2]);
  }else{
    if (Cj){
      subckt().erase(Cj);
      Cj = NULL;
      untested();
    }
  }}

  if (!Yj){
    Yj = new DEV_ADMITTANCE;
    subckt().push_front(Yj);
  }
  Yj->set("Yj", this, &Eval_Yj, 0., n[OUT1], n[OUT2]);

  assert(subckt().exists());
  subckt().expand();
  assert(!constant()); /* because it is nonlinear */
}
/*--------------------------------------------------------------------------*/
double DEV_DIODE::probe_tr_num(const std::string& what)const
{
  CS cmd(what);
  
  if (cmd.pmatch("Vd")){
    return n[OUT1].v0() - n[OUT2].v0();
  }else if (cmd.pmatch("Id")){
    return CARD::probe(Yj,"I") + CARD::probe(Cj,"I");
  }else if (cmd.pmatch("IJ")){
    untested();
    return CARD::probe(Yj,"I");
  }else if (cmd.pmatch("IC")){
    untested();
    return CARD::probe(Cj,"I");
  }else if (cmd.pmatch("P")){
    return CARD::probe(Yj,"P") + CARD::probe(Cj,"P");
  }else if (cmd.pmatch("PD")){
    return CARD::probe(Yj,"PD") + CARD::probe(Cj,"PD");
  }else if (cmd.pmatch("PS")){
    return CARD::probe(Yj,"PS") + CARD::probe(Cj,"PS");
  }else if (cmd.pmatch("PJ")){
    untested();
    return CARD::probe(Yj,"P");
  }else if (cmd.pmatch("PC")){
    untested();
    return CARD::probe(Cj,"P");
  }else if (cmd.pmatch("Cap")){
    return CARD::probe(Cj,"EV");
  }else if (cmd.pmatch("Req")){
    return CARD::probe(Yj,"R");
  }else if (cmd.pmatch("Y")){
    return CARD::probe(Yj,"Y") + CARD::probe(Cj,"Y");
  }else if (cmd.pmatch("Z")){
    return port_impedance(n[OUT1], n[OUT2], lu, probe_tr_num("Y"));
  }else if (cmd.pmatch("ZRAW")){
    return port_impedance(n[OUT1], n[OUT2], lu, 0.);
  }else if (cmd.pmatch("REgion")){
    return static_cast<double>(region);
  }else { /* bad parameter */
    untested();
    return NOT_VALID;
  }
}
/*--------------------------------------------------------------------------*/
void EVAL_DIODE_Yj::tr_eval(COMPONENT* d)const
{
  DEV_DIODE* p = prechecked_cast<DEV_DIODE*>(d->owner());
  assert(p);
  const DIODE_COMMON* c = prechecked_cast<const DIODE_COMMON*>(p->common());
  assert(c);
  const MODEL_DIODE* m = prechecked_cast<const MODEL_DIODE*>(c->model());
  assert(m);

  assert(m->_tnom > 0);

  FPOLY1& y = d->y0;

  double volts = y.x;
  double amps  = y.f0;
  trace2(d->long_label(), volts, amps);

  int flags = (m->flags & m->USE_OPT) ? OPT::diodeflags : m->flags;
  double tempratio = SIM::temp / m->_tnom;
  double vt = (K/Q) * SIM::temp * m->n;
  region_t oldregion = p->region;
  p->isat = c->is * pow(tempratio, m->xti) * Exp((m->eg/vt) * (tempratio-1));
  trace4("", tempratio, vt, oldregion, p->isat);

  {if (c->off  &&  STATUS::iter[SIM::mode] <= 1){     /* initially guess off */
    p->region = INITOFF;
    y.f1 = 0.;
    y.f0 = 0.;
    if (flags & 0020){
      untested();
      y.f1 = OPT::gmin;
    }
    trace2("initoff", y.f0, y.f1);
  }else if (volts < 0. /* &&  amps < 0.*/){    		  /* reverse biased */
    p->region = REVERSE;	    		  /* x = volts, f(x) = amps */
    {if (flags & 0010){
      untested();
      y.f1 = y.f0 = 0.;
    }else{
      double expterm = p->isat * Exp(volts/vt);	
      y.f0 = expterm - p->isat;	    /* i = f(x) = isat * (Exp(volts/vt)-1) */
      y.f1 = expterm / vt;	    /* f'(x) = (isat/vt) * Exp(volts/vt)   */
    }}
    if (flags & 0002){
      untested();		// g = gmin, maintain actual current
      y.f1 += OPT::gmin;	// 3 is a resistor, R=1/gmin
    }
    if (flags & 0004){		// 5 is a resistor, R=vt/isat
      y.f1 += p->isat / vt;
    }
    if (flags & 0001){
      untested();		// a resistor, R=1/f1
      y.f0 = y.f1 * volts;
    }
    trace2("reverse", y.f0, y.f1);
  }else if (volts > 0.  &&  amps > 0.){			  /* forward biased */
				    /* x = amps, f(x) = volts */
    /* derivation: */		    /* if f(x) = log(u): f'(x)=(1/u)(du/dx) */
    /* poly1 r; */
    /* r.f0 = vt * log(amps/p->isat +1.); */
    /* r.f1 = vt / (isat + amps); */
    /* y.f1 = 1. / r.f1; */
    /* y.f0 = amps - r.f0*y.f1 + volts*y.f1; */
    
    p->region = FORWARD;
    y.f1 = (p->isat + amps) / vt;
    y.f0 = amps - log(amps/p->isat +1.)*(p->isat + amps) + volts*y.f1;
    trace2("forward", y.f0, y.f1);
  }else{			    /* non-converged, inconsistent	    */
    p->region = UNKNOWN;	    /* volts and amps have different signs  */
    y.f1 = p->isat/vt;		    /* guess that the voltage should be 0   */
    y.f0 = 0.;			    /* (it usually is very close)	    */
    if (flags & 0001){		    /* use the correct value there	    */
      untested();
      y.f0 = volts * y.f1;
    }
    trace2("unknown", y.f0, y.f1);
  }}
  y.f1 += c->gparallel;
  y.f0 += c->gparallel * volts;

  if (oldregion != p->region  &&  OPT::dampstrategy & dsDEVLIMIT){
      SIM::fulldamp = true;
      untested();
      error(bTRACE, p->long_label() + ":device limit damp\n");
  }
  if (flags & 0100){		// twist g to guarantee g >= gmin
    {if (y.f1 < OPT::gmin){	// without changing i
      y.f1 = OPT::gmin;
      untested();
    }else{
      untested();
    }}
  }
  if (flags & 0200){		// add a gmin in parallel
    y.f1 += OPT::gmin;
    y.f0 += OPT::gmin * volts;
    untested();
  }
  if (flags & 0400){		// linearize .. shift I to pass thru 0
    untested();
    y.f0 = y.f1 * volts;
  }
  trace3(d->long_label(), y.x, y.f0, y.f1);
  p->gd = y.f1;
}
/*--------------------------------------------------------------------------*/
void EVAL_DIODE_Cj::tr_eval(COMPONENT* d)const
{
  const DEV_DIODE* p = prechecked_cast<const DEV_DIODE*>(d->owner());
  assert(p);
  const DIODE_COMMON* c = prechecked_cast<const DIODE_COMMON*>(p->common());
  assert(c);
  const MODEL_DIODE* m = prechecked_cast<const MODEL_DIODE*>(c->model());
  assert(m);

  double& volts = d->y0.x;
  trace1(d->long_label(), volts);

  double cb;
  {if (c->cj != 0.){
    {if (volts < m->fcpb){
      cb = c->cj / pow(1. - (volts / m->pb),  m->mj);
    }else{
      cb = (c->cj / pow(1. - m->fc, 1. + m->mj))
	* (1. - m->fc*(1.+m->mj) + (volts/m->pb)*m->mj);
    }}
  }else{
    untested();
    cb = 0.;
  }}
  assert(cb >= 0.);

  double csw;
  {if (c->cjsw != 0.){
    {if (volts < m->fcpb){
      csw = c->cjsw / pow(1. - (volts / m->pb),  m->mjsw);
    }else{
      csw = (c->cjsw / pow(1. - m->fc, 1. + m->mjsw))
	* (1. - m->fc*(1.+m->mjsw) + (volts/m->pb)*m->mjsw);
    }}
  }else{
    csw = 0.;
  }}
  assert(csw >= 0.);
  
  double ctt;
  {if (m->tt != 0.){
    untested();
    ctt = p->gd * m->tt;
  }else{
    ctt = 0.;
  }}
  assert(ctt >= 0.);
  
  trace4("", cb, csw, ctt, cb+csw+ctt);
  d->y0.f1 = cb + csw + ctt;
  d->y0.f0 = d->y0.x * d->y0.f1;
  trace3(d->long_label(), d->y0.x, d->y0.f0, d->y0.f1);
}
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
