/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995,1996 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 */

static	char	*rcsid = "$Id: buffer.c,v 1.2 1998/02/21 10:27:14 cad Exp $";

/* functions used by paste- and move/copy buffer
 */

#include <stdlib.h>
#include <memory.h>

#include "global.h"

#include "buffer.h"
#include "copy.h"
#include "create.h"
#include "crosshair.h"
#include "data.h"
#include "mymem.h"
#include "mirror.h"
#include "misc.h"
#include "parse_l.h"
#include "rats.h"
#include "rotate.h"
#include "remove.h"
#include "search.h"
#include "select.h"
#include "set.h"

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void	*AddViaToBuffer(PinTypePtr);
static	void	*AddLineToBuffer(LayerTypePtr, LineTypePtr);
static	void	*AddArcToBuffer(LayerTypePtr, ArcTypePtr);
static	void	*AddRatToBuffer(RatTypePtr);
static	void	*AddTextToBuffer(LayerTypePtr, TextTypePtr);
static	void	*AddPolygonToBuffer(LayerTypePtr, PolygonTypePtr);
static	void	*AddElementToBuffer(ElementTypePtr);
static	void	*MoveViaToBuffer(PinTypePtr);
static	void	*MoveLineToBuffer(LayerTypePtr, LineTypePtr);
static	void	*MoveArcToBuffer(LayerTypePtr, ArcTypePtr);
static	void	*MoveRatToBuffer(RatTypePtr);
static	void	*MoveTextToBuffer(LayerTypePtr, TextTypePtr);
static	void	*MovePolygonToBuffer(LayerTypePtr, PolygonTypePtr);
static	void	*MoveElementToBuffer(ElementTypePtr);

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	DataTypePtr		Dest,
						Source;

static	ObjectFunctionType	AddBufferFunctions = {
			AddLineToBuffer,
			AddTextToBuffer,
			AddPolygonToBuffer,
			AddViaToBuffer,
			AddElementToBuffer,
			NULL,
			NULL,
			NULL,
			NULL,
			NULL,
			AddArcToBuffer,
			AddRatToBuffer },
							MoveBufferFunctions = {
			MoveLineToBuffer,
			MoveTextToBuffer,
			MovePolygonToBuffer,
			MoveViaToBuffer,
			MoveElementToBuffer,
			NULL,
			NULL,
			NULL,
			NULL,
			NULL,
			MoveArcToBuffer,
			MoveRatToBuffer };

static int ExtraFlag = 0;

/* ---------------------------------------------------------------------------
 * copies a via to paste buffer
 */
static void *AddViaToBuffer(PinTypePtr Via)
{
	return(CreateNewVia(Dest, Via->X, Via->Y, Via->Thickness, Via->Clearance,
		Via->DrillingHole, Via->Name, Via->Flags & ~(FOUNDFLAG | ExtraFlag)));
}

/* ---------------------------------------------------------------------------
 * copies a rat-line to paste buffer
 */
static void *AddRatToBuffer(RatTypePtr Rat)
{
	return(CreateNewRat(Dest, Rat->Point1.X, Rat->Point1.Y,
		Rat->Point2.X, Rat->Point2.Y, Rat->group1, Rat->group2,
		Rat->Thickness, Rat->Flags & ~(FOUNDFLAG | ExtraFlag)));
}

/* ---------------------------------------------------------------------------
 * copies a line to buffer  
 */
static void *AddLineToBuffer(LayerTypePtr Layer, LineTypePtr Line)
{
	LayerTypePtr	layer = &Dest->Layer[GetLayerNumber(Source, Layer)];
	
	return(CreateNewLineOnLayer(layer, Line->Point1.X, Line->Point1.Y,
		Line->Point2.X, Line->Point2.Y, Line->Thickness, Line->Clearance,
		Line->Flags & ~(FOUNDFLAG | ExtraFlag)));
}

/* ---------------------------------------------------------------------------
 * copies an arc to buffer  
 */
static void *AddArcToBuffer(LayerTypePtr Layer, ArcTypePtr Arc)
{
	LayerTypePtr	layer = &Dest->Layer[GetLayerNumber(Source, Layer)];
	
	return(CreateNewArcOnLayer(layer, Arc->X, Arc->Y,
		Arc->Width, Arc->StartAngle, Arc->Delta,
		Arc->Thickness, Arc->Clearance,
		Arc->Flags & ~(FOUNDFLAG | ExtraFlag)));
}

/* ---------------------------------------------------------------------------
 * copies a text to buffer
 */
static void *AddTextToBuffer(LayerTypePtr Layer, TextTypePtr Text)
{
	LayerTypePtr	layer = &Dest->Layer[GetLayerNumber(Source, Layer)];
	
	return(CreateNewText(layer, &PCB->Font, Text->X, Text->Y,
		Text->Direction, Text->Scale, Text->TextString, Text->Flags & ~ExtraFlag));
}

/* ---------------------------------------------------------------------------
 * copies a polygon to buffer
 */
static void *AddPolygonToBuffer(LayerTypePtr Layer, PolygonTypePtr Polygon)
{
	LayerTypePtr	layer = &Dest->Layer[GetLayerNumber(Source, Layer)];
	PolygonTypePtr	polygon;
	
	polygon = GetPolygonMemory(layer);
	CopyPolygonLowLevel(polygon, Polygon);
	polygon->Flags = polygon->Flags & ~(FOUNDFLAG | ExtraFlag);
	return(polygon);
}

/* ---------------------------------------------------------------------------
 * copies a element to buffer
 */
static void *AddElementToBuffer(ElementTypePtr Element)
{
	ElementTypePtr	element;
	
	element = GetElementMemory(Dest);
	CopyElementLowLevel(Dest, element, Element, True);
	element->Flags &= ~ExtraFlag;
	if (ExtraFlag)
	{
		ELEMENTTEXT_LOOP(element,
			text->Flags &= ~ExtraFlag;
		);
		PIN_LOOP(element,
			pin->Flags &= ~ExtraFlag;
		);
		PAD_LOOP(element,
			pad->Flags &= ~ExtraFlag;
		);
	}
	return(element);
}

/* ---------------------------------------------------------------------------
 * moves a via to paste buffer without allocating memory for the name
 */
static void *MoveViaToBuffer(PinTypePtr Via)
{
	PinTypePtr	via;

	via = GetViaMemory(Dest);
	*via = *Via;
	*Via = Source->Via[--Source->ViaN];
	via->Flags &= ~(WARNFLAG | FOUNDFLAG);
	memset(&Source->Via[Source->ViaN], 0, sizeof(PinType));
	return(via);
}

/* ---------------------------------------------------------------------------
 * moves a rat-line to paste buffer
 */
static void *MoveRatToBuffer(RatTypePtr Rat)
{
	RatTypePtr	rat;

	rat = GetRatMemory(Dest);
	*rat = *Rat;
	*Rat = Source->Rat[--Source->RatN];
	Rat->Flags &= ~FOUNDFLAG;
	memset(&Source->Rat[Source->RatN], 0, sizeof(RatType));
	return(rat);
}

/* ---------------------------------------------------------------------------
 * moves a line to buffer  
 */
static void *MoveLineToBuffer(LayerTypePtr Layer, LineTypePtr Line)
{
	LineTypePtr	line;

	line = GetLineMemory(&Dest->Layer[GetLayerNumber(Source, Layer)]);
	*line = *Line;
	line->Flags &= ~FOUNDFLAG;
	*Line = Layer->Line[--Layer->LineN];
	memset(&Layer->Line[Layer->LineN], 0, sizeof(LineType));
	return(line);
}

/* ---------------------------------------------------------------------------
 * moves an arc to buffer  
 */
static void *MoveArcToBuffer(LayerTypePtr Layer, ArcTypePtr Arc)
{
	ArcTypePtr	arc;

	arc = GetArcMemory(&Dest->Layer[GetLayerNumber(Source, Layer)]);
	*arc = *Arc; 
	arc->Flags &= ~FOUNDFLAG;
	*Arc = Layer->Arc[--Layer->ArcN];
	memset(&Layer->Arc[Layer->ArcN], 0, sizeof(ArcType));
	return(arc);
}

/* ---------------------------------------------------------------------------
 * moves a text to buffer without allocating memory for the name
 */
static void *MoveTextToBuffer(LayerTypePtr Layer, TextTypePtr Text)
{
	TextTypePtr	text;

	text = GetTextMemory(&Dest->Layer[GetLayerNumber(Source, Layer)]);
	*text = *Text;
	*Text = Layer->Text[--Layer->TextN];
	memset(&Layer->Text[Layer->TextN], 0, sizeof(TextType));
	return(text);
}

/* ---------------------------------------------------------------------------
 * moves a polygon to buffer. Doesn't allocate memory for the points
 */
static void *MovePolygonToBuffer(LayerTypePtr Layer, PolygonTypePtr Polygon)
{
	PolygonTypePtr	polygon;
	
	polygon = GetPolygonMemory(&Dest->Layer[GetLayerNumber(Source, Layer)]);
	*polygon = *Polygon;
	polygon->Flags &= ~FOUNDFLAG;
	*Polygon = Layer->Polygon[--Layer->PolygonN];
	memset(&Layer->Polygon[Layer->PolygonN], 0, sizeof(PolygonType));
	return(polygon);
}

/* ---------------------------------------------------------------------------
 * moves a element to buffer without allocating memory for pins/names
 */
static void *MoveElementToBuffer(ElementTypePtr Element)
{
	ElementTypePtr	element;

	element = GetElementMemory(Dest);
	*element = *Element;
	PIN_LOOP(element,
		pin->Flags &= ~(WARNFLAG | FOUNDFLAG);
	);
	PAD_LOOP(element,
		pad->Flags &= ~(WARNFLAG | FOUNDFLAG);
	);
	*Element = Source->Element[--Source->ElementN];
	memset(&Source->Element[Source->ElementN], 0, sizeof(ElementType));
	return(element);
}

/* ---------------------------------------------------------------------------
 * calculates the bounding box of the buffer
 */
void SetBufferBoundingBox(BufferTypePtr Buffer)
{
	BoxTypePtr	box = GetDataBoundingBox(Buffer->Data);

	if (box)
		Buffer->BoundingBox = *box;
}

/* ---------------------------------------------------------------------------
 * clears the contents of the paste buffer
 */
void ClearBuffer(BufferTypePtr Buffer)
{
	if (Buffer)
		FreeDataMemory(Buffer->Data);
}

/* ----------------------------------------------------------------------
 * copies all selected and visible objects to the paste buffer
 * returns True if any objects have been removed
 */
void AddSelectedToBuffer(BufferTypePtr Buffer, Position X, Position Y, Boolean LeaveSelected)
{
		/* switch crosshair off because adding objects to the pastebuffer
		 * may change the 'valid' area for the cursor
		 */
	if (!LeaveSelected)
		ExtraFlag = SELECTEDFLAG;
	HideCrosshair(True);
	Source = PCB->Data;
	Dest = Buffer->Data;
	SelectedOperation(&AddBufferFunctions, False);

		/* set origin to passed or current position */
	if (X || Y)
	{
		Buffer->X = X;
		Buffer->Y = Y;
	}
	else
	{
		Buffer->X = Crosshair.X;
		Buffer->Y = Crosshair.Y;
	}
	RestoreCrosshair(True);
	ExtraFlag = 0;
}

/* ---------------------------------------------------------------------------
 * loads element data from file/library into buffer
 * parse the file with disabled 'PCB mode' (see parser)
 * returns False on error
 * if successful, update some other stuff and reposition the pastebuffer
 */
Boolean LoadElementToBuffer(BufferTypePtr Buffer,
	char *Name, Boolean FromFile)
{
	ElementTypePtr	element;
	Cursor			oldCursor;

		/* change cursor shape to 'watch' and save old setting */
	oldCursor = SetOutputXCursor(XC_watch);
	ClearBuffer(Buffer);
	element = GetElementMemory(Buffer->Data);
	if (!(FromFile ?
		ParseElementFile(element, Name) :
		ParseLibraryEntry(element, Name)))
	{
		SetElementBoundingBox(element, &PCB->Font);

			/* always add elements using top-side coordinates */
		if (Settings.ShowSolderSide)
			MirrorElementCoordinates(element, 0);

			/* set buffer offset to 'mark' position */
		PASTEBUFFER->X = element->MarkX;
		PASTEBUFFER->Y = element->MarkY;
		SetOutputXCursor(oldCursor);
		return(True);
	}
		/* release memory which might have been aquired */
	ClearBuffer(Buffer);
	SetOutputXCursor(oldCursor);
	return(False);
}

/*---------------------------------------------------------------------------
 *
 * convert buffer contents into an element
 */
Boolean	ConvertBufferToElement(BufferTypePtr Buffer)
{
	ElementTypePtr	Element;
	Cardinal	group, i;
	Cardinal	pin_n = 1;
	Boolean		hasParts = False;

	Element = CreateNewElement(PCB->Data, NULL, &PCB->Font, NOFLAG,
			NULL, NULL, NULL, PASTEBUFFER->X, PASTEBUFFER->Y, 0, 100, NOFLAG, False);
	if (!Element)
		return(False);
	LINE_LOOP(&Buffer->Data->Layer[MAX_LAYER + COMPONENT_LAYER],
		CreateNewLineInElement(Element, line->Point1.X,
			line->Point1.Y, line->Point2.X, line->Point2.Y, line->Thickness);
		hasParts = True;
	);
	ARC_LOOP(&Buffer->Data->Layer[MAX_LAYER + COMPONENT_LAYER],
		CreateNewArcInElement(Element, arc->X, arc->Y, arc->Width,
			arc->Height, arc->StartAngle, arc->Delta, arc->Thickness);
		hasParts = True;
	);
	group = GetLayerGroupNumberByNumber(MAX_LAYER + COMPONENT_LAYER);
	for (i = 0; i < PCB->LayerGroups.Number[group]; i++)
	{
		LayerTypePtr	padlayer;
		char		num[8];
		Cardinal	number = PCB->LayerGroups.Entries[group][i];
	
		if (number >= MAX_LAYER)
			continue;
		padlayer = &Buffer->Data->Layer[number];
		LINE_LOOP(padlayer,
			if (line->Point1.X == line->Point2.X || line->Point1.Y == line->Point2.Y)
			{
				sprintf(num, "%d", pin_n++);
				CreateNewPad(Element, line->Point1.X,
				line->Point1.Y, line->Point2.X, line->Point2.Y, line->Thickness,
				line->Clearance, line->Thickness + line->Clearance, NULL, num,
				NOFLAG);
				hasParts = True;
			}
		);
	}
	VIA_LOOP(Buffer->Data,
		{
			char		num[8];

			if (via->Name)
				CreateNewPin(Element, via->X, via->Y, via->Thickness,
					via->Clearance, via->Mask, via->DrillingHole,
					NULL, via->Name, (via->Flags &
					~(VIAFLAG | FOUNDFLAG | SELECTEDFLAG
						| WARNFLAG)) | PINFLAG);
			else
			{	
				sprintf(num, "%d", pin_n++);
				CreateNewPin(Element, via->X, via->Y, via->Thickness,
					via->Clearance, via->Mask, via->DrillingHole,
					NULL, num, (via->Flags &
					~(VIAFLAG | FOUNDFLAG | SELECTEDFLAG
						| WARNFLAG)) | PINFLAG);
			}
			hasParts = True;
		}
	);
	if (!hasParts)
	{
		DestroyObject(PCB->Data, ELEMENT_TYPE, Element, Element, Element);
		return(False);
	}
	Element->MarkX = Buffer->X;
	Element->MarkY = Buffer->Y;
	SetElementBoundingBox(Element, &PCB->Font);
	ClearBuffer(Buffer);
	MoveObjectToBuffer(Buffer->Data, PCB->Data, ELEMENT_TYPE, Element, Element, Element);
	SetBufferBoundingBox(Buffer);
	return(True);
}
	
/* ---------------------------------------------------------------------------
 * load PCB into buffer
 * parse the file with enabled 'PCB mode' (see parser)
 * if successful, update some other stuff
 */
Boolean LoadLayoutToBuffer(BufferTypePtr Buffer, char *Filename)
{
	PCBTypePtr	newPCB = CreateNewPCB(False);

		/* new data isn't added to the undo list */
	if (!ParsePCB(newPCB, Filename))
	{
			/* clear data area and replace pointer */
		ClearBuffer(Buffer);
		SaveFree(Buffer->Data);
		Buffer->Data = newPCB->Data;
		newPCB->Data = NULL;
		Buffer->X = newPCB->CursorX;
		Buffer->Y = newPCB->CursorY;
		RemovePCB(newPCB);
		return(True);
	}

		/* release unused memory */
	RemovePCB(newPCB);
	return(False);
}

/* ---------------------------------------------------------------------------
 * rotates the contents of the pastebuffer
 */
void RotateBuffer(BufferTypePtr Buffer, BYTE Number)
{
		/* rotate vias */
	VIA_LOOP(Buffer->Data,
		ROTATE_VIA_LOWLEVEL(via, Buffer->X, Buffer->Y, Number);
	);

		/* elements */
	ELEMENT_LOOP(Buffer->Data,
		RotateElementLowLevel(element, Buffer->X, Buffer->Y, Number);
	);

		/* all layer related objects */
	ALLLINE_LOOP(Buffer->Data,
			RotateLineLowLevel(line, Buffer->X, Buffer->Y, Number);
	);
	ALLARC_LOOP(Buffer->Data,
			RotateArcLowLevel(arc, Buffer->X, Buffer->Y, Number);
	);
	ALLTEXT_LOOP(Buffer->Data,
		RotateTextLowLevel(text, Buffer->X, Buffer->Y, Number);
	);
	ALLPOLYGON_LOOP(Buffer->Data,
		RotatePolygonLowLevel(polygon, Buffer->X, Buffer->Y, Number);
	);

		/* finally the origin and the bounding box */
	ROTATE(Buffer->X, Buffer->Y, Buffer->X, Buffer->Y, Number);
	RotateBoxLowLevel(&Buffer->BoundingBox, Buffer->X, Buffer->Y, Number);
}

/* ---------------------------------------------------------------------------
 * initializes the buffers by allocating memory
 */
void InitBuffers(void)
{
	int	i;

	for (i = 0; i < MAX_BUFFER; i++)
		Buffers[i].Data = CreateNewBuffer();
}

/* ---------------------------------------------------------------------------
 * flip components/tracks from one side to the other
 */
void SwapBuffers(void)
{
	int		i, j, k;
	Cardinal	sgroup, cgroup;
	LayerType	swap;

	for (i = 0; i < MAX_BUFFER; i++)
	{
		ELEMENT_LOOP(Buffers[i].Data,
			MirrorElementCoordinates(element, 0)
		);
			/* set buffer offset to 'mark' position */
		Buffers[i].X = SWAP_X(Buffers[i].X);
		Buffers[i].Y = SWAP_Y(Buffers[i].Y);
		VIA_LOOP(Buffers[i].Data,
			via->X = SWAP_X(via->X);
			via->Y = SWAP_Y(via->Y);
		);
		ALLLINE_LOOP(Buffers[i].Data,
			line->Point1.X = SWAP_X(line->Point1.X);
			line->Point1.Y = SWAP_Y(line->Point1.Y);
			line->Point2.X = SWAP_X(line->Point2.X);
			line->Point2.Y = SWAP_Y(line->Point2.Y);
		);
		ALLARC_LOOP(Buffers[i].Data,
			arc->X = SWAP_X(arc->X);
			arc->Y = SWAP_Y(arc->Y);
			arc->StartAngle = SWAP_ANGLE(arc->StartAngle);
			arc->Delta = SWAP_DELTA(arc->Delta);
		);
		ALLPOLYGON_LOOP(Buffers[i].Data,
			POLYGONPOINT_LOOP(polygon, 
				point->X = SWAP_X(point->X);
				point->Y = SWAP_Y(point->Y);
			);
			SetPolygonBoundingBox(polygon);
		);
		ALLTEXT_LOOP(Buffers[i].Data,
			text->X = SWAP_X(text->X);
			text->Y = SWAP_Y(text->Y);
			TOGGLE_FLAG(ONSOLDERFLAG, text);
			SetTextBoundingBox(&PCB->Font, text);
		);
			/* swap silkscreen layers */
		swap = Buffers[i].Data->Layer[MAX_LAYER + SOLDER_LAYER];
		Buffers[i].Data->Layer[MAX_LAYER + SOLDER_LAYER] =
			Buffers[i].Data->Layer[MAX_LAYER + COMPONENT_LAYER];
		Buffers[i].Data->Layer[MAX_LAYER + COMPONENT_LAYER] = swap;
		
			/* swap layer groups when balanced */
		sgroup = GetLayerGroupNumberByNumber(MAX_LAYER + SOLDER_LAYER);
		cgroup = GetLayerGroupNumberByNumber(MAX_LAYER + COMPONENT_LAYER);
		if (PCB->LayerGroups.Number[cgroup] == PCB->LayerGroups.Number[sgroup])
		{
			for (j = k = 0; j < PCB->LayerGroups.Number[sgroup]; j++)
			{
				int		t1, t2, f1, f2;
				Cardinal	cnumber = PCB->LayerGroups.Entries[cgroup][k];
				Cardinal	snumber = PCB->LayerGroups.Entries[sgroup][j];
	
				if (snumber >= MAX_LAYER)
					continue;
				swap = Buffers[i].Data->Layer[snumber];
				
				while (cnumber >= MAX_LAYER)
				{
					k++;
					cnumber = PCB->LayerGroups.Entries[cgroup][k];
				}
				Buffers[i].Data->Layer[snumber] = Buffers[i].Data->Layer[cnumber];
				Buffers[i].Data->Layer[cnumber] = swap;
				k++;
					/* move the thermal flags with the layers */
				t1 = L0THERMFLAG << snumber;
				t2 = L0THERMFLAG << cnumber;
				ALLPIN_LOOP(Buffers[i].Data,
					f1 = (TEST_FLAG(t1, pin)) ? t2 : 0;
					f2 = (TEST_FLAG(t2, pin)) ? t1 : 0;
					pin->Flags = (pin->Flags & ~t1 & ~t2) | f1 | f2;
				);
				VIA_LOOP(Buffers[i].Data,
					f1 = (TEST_FLAG(t1, via)) ? t2 : 0;
					f2 = (TEST_FLAG(t2, via)) ? t1 : 0;
					via->Flags = (via->Flags & ~t1 & ~t2) | f1 | f2;
				);
			}
		}
		SetBufferBoundingBox(&Buffers[i]);
	}
	SetCrosshairRangeToBuffer();
}

/* ----------------------------------------------------------------------
 * moves the passed object to the passed buffer and removes it
 * from its original place
 */
void *MoveObjectToBuffer(DataTypePtr Destination, DataTypePtr Src,
	int Type, void *Ptr1, void *Ptr2, void *Ptr3)
{
		/* setup local identifiers used by move operations */
	Dest = Destination;
	Source = Src;
	return(ObjectOperation(&MoveBufferFunctions, Type, Ptr1, Ptr2, Ptr3));
}

/* ----------------------------------------------------------------------
 * Adds the passed object to the passed buffer
 */
void *CopyObjectToBuffer(DataTypePtr Destination, DataTypePtr Src,
	int Type, void *Ptr1, void *Ptr2, void *Ptr3)
{
		/* setup local identifiers used by Add operations */
	Dest = Destination;
	Source = Src;
	return(ObjectOperation(&AddBufferFunctions, Type, Ptr1, Ptr2, Ptr3));
}
