// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Radharamanan Radhakrishnan  ramanan@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_VariableAssignmentStatement.cc,v 1.4 1999/08/24 14:48:33 dmartin Exp $
// 
//---------------------------------------------------------------------------
#include "IIR_VariableAssignmentStatement.hh"
#include "IIR_Label.hh"
#include "error_func.hh"
#include "set.hh"
#include "resolution_func.hh"
#include <strstream.h>

IIRScram_VariableAssignmentStatement::~IIRScram_VariableAssignmentStatement() {
}


void
IIRScram_VariableAssignmentStatement::_publish_vhdl(ostream &_vhdl_out) {
  if (get_label() != NULL) {
    get_label()->_publish_vhdl(_vhdl_out);
    _vhdl_out << ": ";
  }
    
  get_target()->_publish_vhdl(_vhdl_out);
  _vhdl_out << " := ";
  if (get_expression()->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION) {
    _vhdl_out << "(";
  }
  get_expression()->_publish_vhdl(_vhdl_out);
  if (get_expression()->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION) {
    _vhdl_out << ")";
  }
}


void 
IIRScram_VariableAssignmentStatement::_publish_cc() {
  if (get_label() != NULL) {
    get_label()->_publish_cc();
    _cc_out << ":\n";
  }

  if (get_target()->get_kind() == IIR_AGGREGATE) {
    // Create a temporary aggregate and use that in the variable assignment
    _cc_out << "{\n";
    ASSERT ( get_target()->_get_subtype() != NULL );
    get_target()->_get_subtype()->_publish_cc_type_name();
    _cc_out << " tempAggregate(";
    get_target()->_publish_cc_constructor_args();
    get_target()->_publish_cc_initialization_value();
    _cc_out << ");\n";
  }
  
  _cc_out << "assignVariable(";
  if (get_target()->get_kind() != IIR_AGGREGATE) {
    get_target()->_publish_cc();
  }
  else {
    _cc_out << "tempAggregate";
  }
  
  _cc_out << ", ";
  get_expression()->_publish_cc();
  _cc_out << ", ";
  
  // If the target or expression is an interface declaration and we are
  // currently publishing a subprogram then pick up the ranges of the
  // target and expression from the interface objects rather than from
  // the actual type definitions because the ranges of the parameters passed
  // to the subprogram can be different from those of the actual type
  // definitions.

  if ((_is_currently_publishing_subprogram() != TRUE) || (get_target()->_is_interface() != TRUE)) {
    get_target()->_publish_cc_bounds();
  }
  else {
    _cc_out << " nullInfo";
  }
  
  _cc_out << ", ";
  
  if ((_is_currently_publishing_subprogram() != TRUE) || (get_expression()->_is_interface() != TRUE)) {  
    get_expression()->_publish_cc_bounds();
  }
  else {
    _cc_out << " nullInfo";
  }
  
  _cc_out << ");\n";

  if (get_target()->get_kind() == IIR_AGGREGATE) {
    _cc_out << "}\n";
  }
}

// This type checking should make sure that any functions are being with
// legal arguments and such...
void 
IIRScram_VariableAssignmentStatement::_type_check(){
  set<IIR_TypeDefinition> *target_lvals = get_target()->_get_rval_set();
  if( target_lvals == NULL ){
    report_undefined_symbol( get_target() );
    return;
  }
  set_expression( get_expression()->_semantic_transform( target_lvals ));

  set<IIR_TypeDefinition> *expression_rvals = get_expression()->_get_rval_set();
  if( expression_rvals == NULL ){
    report_undefined_symbol( get_expression() );
    return;
  }
  set_target( get_target()->_semantic_transform( expression_rvals ));

  get_target()->_type_check( expression_rvals );
  
  // Need to recalculate AFTER the transformation...
  delete target_lvals;
  target_lvals = get_target()->_get_rval_set();
  get_expression()->_type_check( target_lvals );

  delete expression_rvals;
  delete target_lvals;

  target_lvals = get_target()->_get_rval_set();
  expression_rvals = get_expression()->_get_rval_set();

  reconcile_sets( target_lvals, expression_rvals );

  switch( target_lvals->num_elements() ){
  case 0:{
    ostrstream err;

    err << "Incompatible types in assignment: " << *get_target()
	<< " := " << *get_expression() << ends;

    report_error( this, err );
    break;
  }
  case 1:{
    IIR_TypeDefinition *type = target_lvals->get_element(); 
    set_expression( get_expression()->_rval_to_decl( type ));
    
    set_target( get_target()->_rval_to_decl( type ) );

    if( get_expression()->_is_readable() == FALSE ){
      ostrstream err;
      err << "Expression |" << *get_expression() << "| cannot be read from." 
	  << ends;
      report_error( this, err );
    }

    if( get_target()->_is_writable() == FALSE ){
      ostrstream err;
      err << "Target |" << *get_target() << "| is not writable and cannot be assigned to." 
	  << ends;
      report_error( this, err );
    }
    
    if( get_target()->_is_variable() == FALSE ){
      ostrstream err;
      err << "Target |" << *get_target() << "| is not a variable and cannot be assigned"
	  << " through a variable assignment statement." << ends;
      report_error( this, err );
    }

    break;
  }
  default:
    ostrstream err;
    err << "Ambigous assignment - cannot resolve types" << ends;
    report_error( this, err );
    break;    
  }

  delete target_lvals;
  delete expression_rvals;
}


void
IIRScram_VariableAssignmentStatement::_get_list_of_input_signals(set<IIR_Declaration>* list) {
  get_expression()->_get_list_of_input_signals(list);
}

IIR*
IIRScram_VariableAssignmentStatement::_clone() {
  IIR *exprclone;
  IIR_VariableAssignmentStatement *clone = new IIR_VariableAssignmentStatement;
  IIR_SequentialStatement::_clone(clone);

  clone->set_target(get_target()->_clone());
  exprclone = get_expression()->_clone();
  clone->set_expression(exprclone);

  return clone;
}
