/*C*
 *
 * Hatman - The Game of Kings
 * Copyright (C) 1997 James Pharaoh & Timothy Fisken
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *C*/

#include "Console.h"
#include "Keyboard.h"
#include "../util/debug.h"
#include <string.h>

#ifdef SVGALIB
#include <vgakeyboard.h>
#endif

#ifdef GGI
#include <ggi/events.h>
#include <ggi/ggi.h>
#endif

bool Keyboard::initialised = false;

//--------------------------------------------------------------------------------------------------------------------------------

#ifdef SVGALIB

template Collection<KeyboardEvent>;

char keys[256];

void keyboardHandler(int scancode, int newstate)
{
 assert(keyboard);
 if(newstate && keys[scancode]) keyboard->eventQ->add(new KeyRepEvent(scancode));
 if(newstate && !keys[scancode])  keyboard->eventQ->add(new KeyDownEvent(scancode));
 if(!newstate && keys[scancode]) keyboard->eventQ->add(new KeyUpEvent(scancode));
 keys[scancode] = newstate;
}

#endif

//--------------------------------------------------------------------------------------------------------------------------------

Keyboard::Keyboard()
{
#ifdef SVGALIB
 eventQ = new Collection<KeyboardEvent>;
#endif
}

Keyboard::~Keyboard()
{
#ifdef SVGALIB
 delete eventQ;
#endif
 if(initialised) close();
}

//--------------------------------------------------------------------------------------------------------------------------------

bool Keyboard::open()
{
 if(initialised) fatal("Attempted to Keyboard::open() whilst already so\n");
#ifdef SVGALIB
 if(keyboard_init() == -1) return false;
 keyboard_seteventhandler(keyboardHandler);
 for(int i=0; i<256; i++) keys[i] = 0;
#endif
 return initialised = true;
}

void Keyboard::close()
{
 if(!initialised) fatal("Attempted to Keyboard::close() whilst not open\n");
#ifdef SVGALIB
 keyboard_close();
#endif
}

//--------------------------------------------------------------------------------------------------------------------------------

#ifdef GGI

static int ev2scancode(gii_key_event *kev)
{
 switch (kev->label)
  {
  case GIIUC_Escape: return 1;
  case GIIUC_Return: return 28;
  case GIIK_CtrlL: return 97;
  case GIIUC_BackSpace: return 14;
  case GIIUC_Tab: return 15;
  case GIIK_ShiftL: return 42;
  case GIIK_ShiftR: return 54;
  case GIIK_PStar: return 55;
  case GIIK_AltL: return 56;
  case GIIK_CapsLock: return 58;
  case GIIK_F1: return 59;
  case GIIK_F2: return 60;
  case GIIK_F3: return 61;
  case GIIK_F4: return 62;
  case GIIK_F5: return 63;
  case GIIK_F6: return 64;
  case GIIK_F7: return 65;
  case GIIK_F8: return 66;
  case GIIK_F9: return 67;
  case GIIK_F10: return 68;
  case GIIK_NumLock: return 69;
  case GIIK_P7: return 71;
  case GIIK_P8: return 72;
  case GIIK_P9: return 73;
  case GIIK_PMinus: return 74;
  case GIIK_P4: return 75;
  case GIIK_P5: return 76;
  case GIIK_P6: return 77;
  case GIIK_PPlus: return 78;
  case GIIK_P1: return 79;
  case GIIK_P2: return 80;
  case GIIK_P3: return 81;
  case GIIK_P0: return 82;
  case GIIK_PDecimal: return 83;
  case GIIUC_Less: return 86;
  case GIIK_F11: return 87;
  case GIIK_F12: return 88;
  case GIIK_PEnter: return 96;
  case GIIK_CtrlR: return 97;
  case GIIK_PSlash: return 98;
  case GIIK_AltR: return 100;
  case GIIK_Break: return 101;
  case GIIK_Home: return 102;
  case GIIK_Up: return 103;
  case GIIK_PageUp: return 104;
  case GIIK_Left: return 105;
  case GIIK_Right: return 106;
  case GIIK_End: return 107;
  case GIIK_Down: return 108;
  case GIIK_PageDown: return 109;
  case GIIK_Insert: return 110;
  case GIIUC_Delete: return 111;
  case GIIUC_1: return 2;
  case GIIUC_2: return 3;
  case GIIUC_3: return 4;
  case GIIUC_4: return 5;
  case GIIUC_5: return 6;
  case GIIUC_6: return 7;
  case GIIUC_7: return 8;
  case GIIUC_8: return 9;
  case GIIUC_9: return 10;
  case GIIUC_0: return 11;
  case GIIUC_Minus: return 12;
  case GIIUC_Equal: return 13;
  case GIIUC_q: case GIIUC_Q: return 16;
  case GIIUC_w: case GIIUC_W: return 17;
  case GIIUC_e: case GIIUC_E: return 18;
  case GIIUC_r: case GIIUC_R: return 19;
  case GIIUC_t: case GIIUC_T: return 20;
  case GIIUC_y: case GIIUC_Y: return 21;
  case GIIUC_u: case GIIUC_U: return 22;
  case GIIUC_i: case GIIUC_I: return 23;
  case GIIUC_o: case GIIUC_O: return 24;
  case GIIUC_p: case GIIUC_P: return 25;
  case GIIUC_BracketLeft: return 26;
  case GIIUC_BracketRight: return 27;
  case GIIUC_a: case GIIUC_A: return 30;
  case GIIUC_s: case GIIUC_S: return 31;
  case GIIUC_d: case GIIUC_D: return 32;
  case GIIUC_f: case GIIUC_F: return 33;
  case GIIUC_g: case GIIUC_G: return 34;
  case GIIUC_h: case GIIUC_H: return 35;
  case GIIUC_j: case GIIUC_J: return 36;
  case GIIUC_k: case GIIUC_K: return 37;
  case GIIUC_l: case GIIUC_L: return 38;
  case GIIUC_Semicolon: return 39;
  case GIIUC_Apostrophe: return 40;
  case GIIUC_Grave: return 41;
  case GIIUC_BackSlash: return 43;
  case GIIUC_z: case GIIUC_Z: return 44;
  case GIIUC_x: case GIIUC_X: return 45;
  case GIIUC_c: case GIIUC_C: return 46;
  case GIIUC_v: case GIIUC_V: return 47;
  case GIIUC_b: case GIIUC_B: return 48;
  case GIIUC_n: case GIIUC_N: return 49;
  case GIIUC_m: case GIIUC_M: return 50;
  case GIIUC_Comma: return 51;
  case GIIUC_Period: return 52;
  case GIIUC_Slash: return 53;
  case GIIUC_Space: return 57;
  }
 fprintf(stderr, "Unknown LABEL: 0x%x:%c:\n", kev->label, kev->label);
 return 0;
}

#endif

//--------------------------------------------------------------------------------------------------------------------------------

KeyboardEvent* Keyboard::getEvent()
{

#ifdef GGI
 timeval tv = {0, 0};
 if(ggiEventPoll(consoleVisual, emKey, &tv))
  {
   gii_event e;
   ggiEventRead(consoleVisual, &e, emKey);
   int scancode = ev2scancode(&e.key);
   VPRINTF("<keyboard> ggi key event type=%d scancode=%d\n", e.key.type, scancode);
   if(scancode) switch(e.key.type)
    {
    case evKeyPress:
     return new KeyDownEvent(scancode);
    case evKeyRepeat:
     return new KeyRepEvent(scancode);
    case evKeyRelease:
     return new KeyUpEvent(scancode);
    default:
     nonFatal("keyboard got unknown ggi event %d\n", e.any.type);

    }
  }
 return NULL;
#endif

#ifdef SVGALIB
 foreach(*eventQ, i)
  if(eventQ->at(i)) return eventQ->remove(i);
 eventQ->pack();
 keyboard_update();
 return NULL;
#endif

}

//--------------------------------------------------------------------------------------------------------------------------------
