## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import types

# PySol imports
from mfxutil import Struct, kwdefault, SubclassResponsibility       #bundle#
from util import Timer                                              #bundle#
from util import ACE, KING, SUITS                                   #bundle#
from util import ANY_SUIT, ANY_COLOR, ANY_RANK, NO_RANK             #bundle#
from util import NO_REDEAL, UNLIMITED_REDEALS, VARIABLE_REDEALS     #bundle#
from pysoltk import tkname, bind, EVENT_HANDLED, EVENT_PROPAGATE    #bundle#
from pysoltk import CURSOR_DRAG, ANCHOR_NW, ANCHOR_SE               #bundle#
from pysoltk import MfxCanvasGroup, MfxCanvasImage, MfxCanvasText   #bundle#
from pysoltk import Card                                            #bundle#
from pysoltk import after, after_cancel                             #bundle#


# /***********************************************************************
# // Let's start with some test methods for card piles.
# // Empty card-lists return false !
# ************************************************************************/

# check that all cards are face-up
def cardsFaceUp(cards):
    if not cards: return 0
    for c in cards:
        if not c.face_up:
            return 0
    return 1

# check that all cards are face-down
def cardsFaceDown(cards):
    if not cards: return 0
    for c in cards:
        if c.face_up:
            return 0
    return 1

# check that cards are face-up and sorted descending by rank
def isRankSequence(cards, mod=1024, dir=1):
    if not cardsFaceUp(cards):
        return 0
    c1 = cards[0]
    for i in range(1, len(cards)):
        c2 = cards[i]
        if c1.rank != (c2.rank + dir) % mod:
            return 0
        c1 = c2
    return 1

# check that cards are face-up and build by alternate color
def isAlternateColorSequence(cards, mod=1024, dir=1):
    if not cardsFaceUp(cards):
        return 0
    c1 = cards[0]
    for i in range(1, len(cards)):
        c2 = cards[i]
        if c1.rank != (c2.rank + dir) % mod or c1.color == c2.color:
            return 0
        c1 = c2
    return 1

# check that cards are face-up and build by same color
def isSameColorSequence(cards, mod=1024, dir=1):
    if not cardsFaceUp(cards):
        return 0
    c1 = cards[0]
    for i in range(1, len(cards)):
        c2 = cards[i]
        if c1.rank != (c2.rank + dir) % mod or c1.color != c2.color:
            return 0
        c1 = c2
    return 1

# check that cards are face-up and build by same suit
def isSameSuitSequence(cards, mod=1024, dir=1):
    if not cardsFaceUp(cards):
        return 0
    c1 = cards[0]
    for i in range(1, len(cards)):
        c2 = cards[i]
        if c1.rank != (c2.rank + dir) % mod or c1.suit != c2.suit:
            return 0
        c1 = c2
    return 1

# check that cards are face-up and build by any suit but own
def isAnySuitButOwnSequence(cards, mod=1024, dir=1):
    if not cardsFaceUp(cards):
        return 0
    c1 = cards[0]
    for i in range(1, len(cards)):
        c2 = cards[i]
        if c1.rank != (c2.rank + dir) % mod or c1.suit == c2.suit:
            return 0
        c1 = c2
    return 1

def getNumberOfFreeStacks(stacks):
    ##print len(filter(lambda s: not s.cards, stacks)), stacks
    return len(filter(lambda s: not s.cards, stacks))
    n = 0
    for s in stacks:
        if not s.cards:
            n = n + 1
    return n


# /***********************************************************************
# // A StackWrapper is a functor (function object) that creates a
# // new stack when called, i.e. it wraps the constructor.
# //
# // "cap" are the capabilites, see class Stack below.
# ************************************************************************/

# self.cap override any call-time cap
class StackWrapper:
    def __init__(self, stack_class, **cap):
        assert type(stack_class) == types.ClassType
        assert issubclass(stack_class, Stack)
        self.stack_class = stack_class
        self.cap = cap

    # return a new stack (an instance of the stack class)
    def __call__(self, x, y, game, **cap):
        # must preserve self.cap, so create a shallow copy
        c = self.cap.copy()
        apply(kwdefault, (c,), cap)
        return apply(self.stack_class, (x, y, game), c)


# call-time cap override self.cap
class WeakStackWrapper(StackWrapper):
    def __call__(self, x, y, game, **cap):
        apply(kwdefault, (cap,), self.cap)
        return apply(self.stack_class, (x, y, game), cap)


# self.cap only, call-time cap is completely ignored
class FullStackWrapper(StackWrapper):
    def __call__(self, x, y, game, **cap):
        return apply(self.stack_class, (x, y, game), self.cap)


# /***********************************************************************
# //
# ************************************************************************/

class Stack:
    # A generic stack of cards.
    #
    # This is used as a base class for all other stacks (e.g. the talon,
    # the foundations and the row stacks).
    #
    # The default event handlers turn the top card of the stack with
    # its face up on a (single or double) click, and also support
    # moving a subpile around.

    def __init__(self, x, y, game, cap={}):
        # Arguments are the stack's nominal x and y position (the top
        # left corner of the first card placed in the stack), and the
        # game object (which is used to get the canvas; subclasses use
        # the game object to find other stacks).

        #
        # link back to game
        #
        id = len(game.allstacks)
        x = int(round(x))
        y = int(round(y))
        mapkey = (x, y)
        assert not self in game.allstacks
        for s in game.allstacks:
            assert x != s.x or y != s.y
        game.allstacks.append(self)
        assert not game.stackmap.has_key(mapkey)
        game.stackmap[mapkey] = id
        images = game.app.images

        #
        # setup our pseudo MVC scheme
        #
        model, view, controller = self, self, self

        #
        # model
        #
        model.id = id
        model.game = game
        model.cards = []

        # capabilites - the game logic
        model.cap = Struct(
            suit = -1,          # required suit for this stack (-1 is ANY_SUIT)
            color = -1,         # required color for this stack (-1 is ANY_COLOR)
            rank = -1,          # required rank for this stack (-1 is ANY_RANK)
            base_suit = -1,     # base suit for this stack (-1 is ANY_SUIT)
            base_color = -1,    # base color for this stack (-1 is ANY_ANYCOLOR)
            base_rank = -1,     # base rank for this stack (-1 is ANY_RANK)
            dir = 0,            # direction - stack builds up/down
            mod = 1024,         # modulo for wrap around (typically 13 or 1024)
            max_move = 0,       # can move at most # cards at a time
            max_accept = 0,     # can accept at most # cards at a time
            max_cards = 999999, # total number of cards may not exceed this
            # not commonly used:
            min_move = 1,       # must move at least # cards at a time
            min_accept = 1,     # must accept at least # cards at a time
            min_cards = 0,      # total number of cards this stack at least requires
        )
        model.cap.update(cap)
        assert type(model.cap.suit) == types.IntType
        assert type(model.cap.color) == types.IntType
        assert type(model.cap.rank) == types.IntType
        assert type(model.cap.base_suit) == types.IntType
        assert type(model.cap.base_color) == types.IntType
        assert type(model.cap.base_rank) == types.IntType

        #
        # view
        #
        view.x = x
        view.y = y
        view.canvas = game.canvas
        view.CARD_XOFFSET = 0
        view.CARD_YOFFSET = 0
        view.group = MfxCanvasGroup(view.canvas)
        ##view.group.move(view.x, view.y)
        # image items
        view.images = Struct(
            bottom = None,              # canvas item
            redeal = None,              # canvas item
            redeal_img = None,          #   the corresponding PhotoImage
        )
        # text items
        view.texts = Struct(
            ncards = None,              # canvas item
            # by default only used by Talon:
            rounds = None,              # canvas item
            redeal = None,              # canvas item
            redeal_str = None,          #   the corresponding string
            # for use by derived stacks:
            misc = None,                # canvas item
        )
        view.is_visible = view.x >= -100 and view.y >= -100
        view.is_open = -1
        view.can_hide_cards = -1
        # compute hide-off direction (see class Card)
        cx = view.x + images.CARDW / 2
        cy = view.y + images.CARDH / 2
        if cy < 1.5 * images.CARDH:
            view.hide_x, view.hide_y = 0, -10000    # hide at top
        elif cx < 1.5 * images.CARDW:
            view.hide_x, view.hide_y = -10000, 0    # hide at left
        elif cy > view.game.height - 1.5 * images.CARDH:
            view.hide_x, view.hide_y = 0, 10000     # hide at bottom
        else:
            view.hide_x, view.hide_y = 10000, 0     # hide at right

        #
        # controller
        #
        if view.is_visible:
            controller.initBindings()


    # bindings {view widgets bind to controller}
    def initBindings(self):
        ##assertController(self)
        self.magnetic_mouse = Struct(
            timer = None,
            event = None,
        )
        view = self
        bind(view.group, "<1>", self.__clickEventHandler)
        bind(view.group, "<Control-1>", self.__controlclickEventHandler)
        bind(view.group, "<Shift-1>", self.__shiftclickEventHandler)
        bind(view.group, "<Double-1>", self.__doubleclickEventHandler)
        bind(view.group, "<2>", self.__middleclickEventHandler)
        ##bind(view.group, "<Double-2>", "")
        bind(view.group, "<3>", self.__rightclickEventHandler)
        bind(view.group, "<Control-3>", self.__middleclickEventHandler)
        bind(view.group, "<Shift-3>", self.__shiftrightclickEventHandler)
        bind(view.group, "<B1-Motion>", self.__motionEventHandler)
        bind(view.group, "<ButtonRelease-1>", self.__releaseEventHandler)
        ##bind(view.group, "<Enter>", self.__enterEventHandler)
        ##bind(view.group, "<Leave>", self.__leaveEventHandler)


    def prepareView(self):
        ##assertView(self)
        # prepare some variables
        ox, oy = self.CARD_XOFFSET, self.CARD_YOFFSET
        if type(ox) == types.IntType:
            self.CARD_XOFFSET = (ox,)
        else:
            self.CARD_XOFFSET = tuple(ox)
        if type(oy) == types.IntType:
            self.CARD_YOFFSET = (oy,)
        else:
            self.CARD_YOFFSET = tuple(oy)
        if self.can_hide_cards < 0:
            self.can_hide_cards = (self.is_visible and
                                   self.CARD_XOFFSET[0] == 0 and
                                   self.CARD_YOFFSET[0] == 0)
        if self.is_open < 0:
            self.is_open = (self.is_visible and
                            (abs(self.CARD_XOFFSET[0]) >= 10 or
                             abs(self.CARD_YOFFSET[0]) >= 10))
        # bottom image
        if self.is_visible and self.images.bottom is None:
            img = self.getBottomImage()
            if img is not None:
                self.images.bottom = MfxCanvasImage(self.canvas, self.x, self.y,
                                                    image=img, anchor=ANCHOR_NW)
                self.images.bottom.addtag(self.group)


    def prepareStack(self):
        self.prepareView()


    def assertStack(self):
        assert not hasattr(self, "suit")


    #
    # Core access methods {model -> view}
    #

    # Add a card add the top of a stack. Also update display. {model -> view}
    def addCard(self, card, unhide=1, update=1):
        model, view = self, self
        model.cards.append(card)
        card.tkraise(unhide=unhide)
        if view.can_hide_cards and len(model.cards) >= 3:
            # we only need to display the 2 top cards
            model.cards[-3].hide(self)
        card.item.addtag(view.group)
        view._position(card)
        if update:
            view.updateText()
        return card

    # Remove a card from the stack. Also update display. {model -> view}
    def removeCard(self, card=None, unhide=1, update=1):
        model, view = self, self
        assert len(model.cards) > 0
        if card is None:
            card = model.cards[-1]
            # optimized a little bit (compare with the else below)
            card.item.dtag(view.group)
            if unhide and self.can_hide_cards:
                card.unhide()
                if len(self.cards) >= 3:
                    model.cards[-3].unhide()
            del model.cards[-1]
        else:
            card.item.dtag(view.group)
            if unhide and view.can_hide_cards:
                # Note: the 2 top cards ([-1] and [-2]) are already unhidden.
                card.unhide()
                if len(model.cards) >= 3:
                    if card is model.cards[-1] or model is self.cards[-2]:
                        # Make sure that 2 top cards will be un-hidden.
                        model.cards[-3].unhide()
            model.cards.remove(card)
        if update:
            view.updateText()
        return card

    # Get the top card {model}
    def getCard(self):
        if self.cards:
            return self.cards[-1]
        return None

    # get the largest moveable pile {model}
    def getPile(self):
        return None

    # Position the card on the canvas {view}
    def _position(self, card):
        x, y = self.getPositionFor(card)
        card.moveTo(x, y)

    # find card
    def _findCard(self, event):
        model, view = self, self
        if event is not None and model.cards:
            # ask the canvas
            return view.canvas.findCard(self, event)
        return -1

    # find card
    def _findCardXY(self, x, y, cards=None):
        model, view = self, self
        if cards is None: cards = model.cards
        images = self.game.app.images
        index = -1
        for i in range(len(cards)):
            c = cards[i]
            r = (c.x, c.y, c.x + images.CARDW, c.y + images.CARDH)
            if x >= r[0] and x < r[2] and y >= r[1] and y < r[3]:
                index = i
        return index

    #
    # Basic capabilities {model}
    # Used by various subclasses.
    #

    def basicIsBlocked(self):
        # Check if the pile is blocked (e.g. Pyramid)
        return 0

    def basicAcceptsPile(self, from_stack, cards):
        # Check that the limits are ok and the cards are face up
        if from_stack is self or self.basicIsBlocked():
            return 0
        cap = self.cap
        l = len(cards)
        if l < cap.min_accept or l > cap.max_accept:
            return 0
        l = l + len(self.cards)
        if l < cap.min_cards or l > cap.max_cards:
            return 0
        for c in cards:
            if not c.face_up:
                return 0
            if cap.suit >= 0 and c.suit != cap.suit:
                return 0
            if cap.color >= 0 and c.color != cap.color:
                return 0
            if cap.rank >= 0 and c.rank != cap.rank:
                return 0
        if self.cards:
            return self.cards[-1].face_up
        else:
            c = cards[0]
            if cap.base_suit >= 0 and c.suit != cap.base_suit:
                return 0
            if cap.base_color >= 0 and c.color != cap.base_color:
                return 0
            if cap.base_rank >= 0 and c.rank != cap.base_rank:
                return 0
            return 1

    def basicCanMovePile(self, cards):
        # Check that the limits are ok and the cards are face up
        if self.basicIsBlocked():
            return 0
        cap = self.cap
        l = len(cards)
        if l < cap.min_move or l > cap.max_move:
            return 0
        l = len(self.cards) - l
        if l < cap.min_cards or l > cap.max_cards:
            return 0
        return cardsFaceUp(cards)


    #
    # Capabilities - important for game logic {model}
    #

    def acceptsPile(self, from_stack, cards):
        # Do we accept receiving `cards' from `from_stack' ?
        return 0

    def canMovePile(self, cards):
        # Can we move these cards when assuming they are our top-cards ?
        return 0

    def canFlipCard(self):
        # Can we flip our top card ?
        return 0

    def canDropCards(self, stacks):
        # Can we drop the top cards onto one of the stacks ?
        return (None, 0)    # return the stack and the number of cards


    #
    # Appearance {model}
    #

    def __repr__(self):
        # Return a string for debug print statements.
        return "%s(%d)" % (self.__class__.__name__, self.id)

    #
    # Atomic move actions {model -> view}
    #

    def flipMove(self):
        # Flip the top card.
        self.game.flipMove(self)

    def moveMove(self, ncards, to_stack, frames=-1, shadow=-1):
        # Move the top n cards.
        self.game.moveMove(ncards, self, to_stack, frames=frames, shadow=shadow)
        self.fillStack()

    def fillStack(self):
        self.game.fillStack(self)


    #
    # Playing move actions. Better not override.
    #

    def playFlipMove(self):
        self.flipMove()
        if not self.game.checkForWin():
            self.game.autoPlay()
        self.game.finishMove()

    def playMoveMove(self, ncards, to_stack, frames=-1, shadow=-1):
        self.moveMove(ncards, to_stack, frames=frames, shadow=shadow)
        if not self.game.checkForWin():
            # let the player put cards back from the foundations
            if not self in self.game.s.foundations:
                self.game.autoPlay()
        self.game.finishMove()


    #
    # Appearance {view}
    #

    def getBottomImage(self):
        return None

    def getPositionFor(self, card):
        model, view = self, self
        if view.can_hide_cards:
            return view.x, view.y
        x, y = view.x, view.y
        ix, iy, lx, ly = 0, 0, len(view.CARD_XOFFSET), len(view.CARD_YOFFSET)
        for c in model.cards:
            if c is card:
                break
            x = x + view.CARD_XOFFSET[ix]
            y = y + view.CARD_YOFFSET[iy]
            ix = (ix + 1) % lx
            iy = (iy + 1) % ly
        return (x, y)

    # Fully update the view of a stack - updates
    # hiding, card positions and stacking order.
    # Avoid calling this as it is rather slow.
    def refreshView(self):
        model, view = self, self
        cards = model.cards
        if not view.is_visible or len(cards) < 2:
            return
        if view.can_hide_cards:
            # hide all lower cards
            for c in cards[:-2]:
                ##print "refresh hide", c, c.hide_stack
                c.hide(self)
            # unhide the 2 top cards
            for c in cards[-2:]:
                ##print "refresh unhide 1", c, c.hide_stack
                c.unhide()
                ##print "refresh unhide 1", c, c.hide_stack, c.hide_x, c.hide_y
        # update the card postions and stacking order
        item = cards[0].item
        x, y = view.x, view.y
        ix, iy, lx, ly = 0, 0, len(view.CARD_XOFFSET), len(view.CARD_YOFFSET)
        for c in cards[1:]:
            c.item.tkraise(item)
            item = c.item
            if not view.can_hide_cards:
                x = x + view.CARD_XOFFSET[ix]
                y = y + view.CARD_YOFFSET[iy]
                ix = (ix + 1) % lx
                iy = (iy + 1) % ly
                c.moveTo(x, y)

    def updateText(self):
        if self.texts.ncards is None:
            return
        format = "%d"
        if self.texts.ncards.text_format is not None:
            format = self.texts.ncards.text_format
        t = format % len(self.cards)
        if 0 and self.game.app.debug:
            visible = 0
            for c in self.cards:
                if c.isHidden():
                    assert c.hide_stack is not None
                    assert c.hide_x != 0 or c.hide_y != 0
                else:
                    visible = visible + 1
                    assert c.hide_stack is None
                    assert c.hide_x == 0 and c.hide_y == 0
            t  = t + " %2d" % visible
        self.texts.ncards["text"] = t

    def shallHighlightSameRank(self, card):
        assert card in self.cards
        if not self.is_visible or not card.face_up:
            return 0
        if card is self.cards[-1]:
            return 1
        return self.is_open

    def shallHighlightMatch(self, card):
        return self.shallHighlightSameRank(card)

    def highlightSameRank(self, event):
        i = self._findCard(event)
        if i < 0:
            return 0
        card = self.cards[i]
        if not self.shallHighlightSameRank(card):
            return 0
        col = self.game.app.opt.highlight_samerank_colors
        info = [ (self, card, card, col[0], col[1]) ]
        for s in self.game.allstacks:
            for c in s.cards:
                if c is card: continue
                if c.rank != card.rank: continue
                # ask the stack
                if s.shallHighlightSameRank(c):
                    info.append((s, c, c, col[2], col[3]))
        self.game.stats.highlight_samerank = self.game.stats.highlight_samerank + 1
        return self.game._highlightCards(info, self.game.app.opt.highlight_samerank_sleep)

    def highlightMatchingCards(self, event):
        i = self._findCard(event)
        if i < 0:
            return 0
        card = self.cards[i]
        if not self.shallHighlightMatch(card):
            return 0
        col = self.game.app.opt.highlight_cards_colors
        c1 = c2 = card
        info = []
        found = 0
        for s in self.game.allstacks:
            for c in s.cards:
                if c is card: continue
                # ask the stack
                if not s.shallHighlightMatch(c):
                    continue
                # continue if both stacks are foundations
                if self in self.game.s.foundations and s in self.game.s.foundations:
                    continue
                # ask the game
                if self.game.shallHighlightMatch(self, card, s, c):
                    found = 1
                    if s is self:
                        # enlarge rectangle for neighbours
                        j = self.cards.index(c)
                        if i - 1 == j: c1 = c; continue
                        if i + 1 == j: c2 = c; continue
                    info.append((s, c, c, col[2], col[3]))
        if found:
            if info > 0:
                self.game.stats.highlight_cards = self.game.stats.highlight_cards + 1
            info.append((self, c1, c2, col[0], col[1]))
            return self.game._highlightCards(info, self.game.app.opt.highlight_cards_sleep)
        return 0


    #
    # Subclass overridable handlers {contoller -> model -> view}
    #

    def clickHandler(self, event):
        return 0

    def middleclickHandler(self, event):
        # default: show the card if it is overlapped by other cards
        if not self.is_open:
            return 0
        i = self._findCard(event)
        positions = len(self.cards) - i - 1
        if i < 0 or positions <= 0 or not self.cards[i].face_up:
            return 0
        ##print self.cards[i]
        if tkname == "tk":
            self.cards[i].item.tkraise()
        elif tkname == "gnome":
            self.cards[i].item.tkraise(positions)
        self.game.canvas.update_idletasks()
        self.game.top.sleep(self.game.app.opt.raise_card_sleep)
        if tkname == "tk":
            self.cards[i].item.lower(self.cards[i+1].item)
        elif tkname == "gnome":
            self.cards[i].item.lower(positions)
        self.game.canvas.update_idletasks()
        return 1

    def rightclickHandler(self, event):
        return 0

    def doubleclickHandler(self, event):
        return self.clickHandler(event)

    def controlclickHandler(self, event):
        return 0

    def shiftclickHandler(self, event):
        # highlight same rank cards
        if self.game.app.opt.highlight_samerank:
            return self.highlightSameRank(event)
        return 0

    def shiftrightclickHandler(self, event):
        return 0

    def releaseHandler(self, cards):
        # default action: move cards back to their origin position
        for card in cards:
            self._position(card)


    #
    # Event handlers {controller}
    #

    def __defaultClickEventHandler(self, event, handler, start_drag=0):
        self.__leaveEventHandler(None)
        self.game.demo = None
        if self.game.busy: return EVENT_HANDLED
        if self.game.drag.stack:
            self.game.drag.stack.cancelDrag()    # in case we lost an event
        if start_drag:
            # this handler may start a drag operation
            if not handler(event):
                self.startDrag(event)
        else:
            handler(event)
        return EVENT_HANDLED

    def __clickEventHandler(self, event):
        return self.__defaultClickEventHandler(event, self.clickHandler, 1)

    def __doubleclickEventHandler(self, event):
        return self.__defaultClickEventHandler(event, self.doubleclickHandler, 1)

    def __middleclickEventHandler(self, event):
        return self.__defaultClickEventHandler(event, self.middleclickHandler)

    def __rightclickEventHandler(self, event):
        return self.__defaultClickEventHandler(event, self.rightclickHandler)

    def __controlclickEventHandler(self, event):
        return self.__defaultClickEventHandler(event, self.controlclickHandler)

    def __shiftclickEventHandler(self, event):
        return self.__defaultClickEventHandler(event, self.shiftclickHandler)

    def __shiftrightclickEventHandler(self, event):
        return self.__defaultClickEventHandler(event, self.shiftrightclickHandler)

    def __motionEventHandler(self, event):
        self.__leaveEventHandler(None)
        self.game.demo = None
        if self.game.busy: return EVENT_HANDLED
        ##print event.__dict__
        self.keepDrag(event)
        return EVENT_HANDLED

    def __releaseEventHandler(self, event):
        self.__leaveEventHandler(None)
        self.game.demo = None
        if self.game.busy: return EVENT_HANDLED
        self.keepDrag(event)
        self.finishDrag()
        return EVENT_HANDLED

    def __enterEventHandler(self, event):
        self.__leaveEventHandler(None)
        opt = self.game.app.opt
        if opt.magnetic_mouse:
            t = int(opt.magnetic_mouse_time * 1000.0)
            self.magnetic_mouse_timer = after(self.canvas, t, self.__magneticMouseTimerHandler)
        return EVENT_HANDLED

    def __leaveEventHandler(self, event):
        after_cancel(self.magnetic_mouse.timer)
        self.magnetic_mouse.timer = None
        self.magnetic_mouse.event = None
        return EVENT_HANDLED

    def __magneticMouseTimerHandler(self):
        event = self.magnetic_mouse.event
        self.__leaveEventHandler(None)
        ##print "magnetic mouse timer:", self, event
        if event and not self.game.drag.stack:
            ## FIXME: simulate a click ???
            ##self.__clickEventHandler(event)
            pass

    #
    # Drag internals {controller -> model -> view}
    #

    # begin a drag operation
    def startDrag(self, event):
        assert self.game.drag.stack is None
        ##self.game.drag.stack = None
        i = self._findCard(event)
        if i < 0 or not self.canMovePile(self.cards[i:]):
            return
        self.lastx = event.x
        self.lasty = event.y
        game = self.game
        drag = game.drag
        drag.stack = self
        drag.cards = self.cards[i:]
        if tkname == "gnome":
            drag.stack.group.tkraise()
        images = game.app.images
        ##sx, sy = 0, 0
        sx, sy = -images.SHADOW_XOFFSET, -images.SHADOW_YOFFSET
        if game.app.opt.shadow:
            img0, img1 = images.getShadow(0), images.getShadow(len(drag.cards))
#%ifndef BUNDLE
            if 0:
                # Dynamically compute the shadow. Doesn't work because
                # PhotoImage.copy() doesn't preserve transparency.
                img1 = images.getShadow(13)
                if img1:
                    h = images.CARDH - img0.height()
                    h = h + (len(drag.cards) - 1) * self.CARD_YOFFSET[0]
                    if h < img1.height():
                        import Tkinter
                        dest = Tkinter.PhotoImage(width=img1.width(), height=h)
                        dest.blank()
                        img1.tk.call(dest, "copy", img1.name, "-from", 0, 0, img1.width(), h)
                        assert dest.height() == h and dest.width() == img1.width()
                        #print h, img1.height(), dest.height()
                        img1 = dest
                        self._foo = img1 # keep a reference
                    elif h > img1.height():
                        img1 = None
#%endif
            if img0 and img1:
                c = drag.cards[-1]
                if self.CARD_YOFFSET[0] < 0: c = drag.cards[0]
                cx, cy = c.x + images.CARDW, c.y + images.CARDH
                drag.shadows.append(MfxCanvasImage(game.canvas, cx, cy - img0.height(),
                                                   image=img1, anchor=ANCHOR_SE))
                drag.shadows.append(MfxCanvasImage(game.canvas, cx, cy,
                                                   image=img0, anchor=ANCHOR_SE))
        # convert to tuples (speed)
        drag.shadows = tuple(drag.shadows)
        #
        for s in drag.shadows:
            if tkname == "gnome":
                s.addtag(drag.stack.group)
            s.tkraise()
        for card in drag.cards:
            card.tkraise()
            card.moveBy(sx, sy)
        if self.game.app.opt.dragcursor:
            self.game.canvas.config(cursor=CURSOR_DRAG)

    # continue a drag operation
    def keepDrag(self, event):
        drag = self.game.drag
        if not drag.cards:
            return
        assert self is self.game.drag.stack
        dx = event.x - self.lastx
        dy = event.y - self.lasty
        if dx or dy:
            self.lastx = event.x
            self.lasty = event.y
            if self.game.app.opt.shade:
                self._updateShade()
            for s in drag.shadows:
                s.move(dx, dy)
            for card in drag.cards:
                card.moveBy(dx, dy)

    # handle shade within a drag operation
    def _deleteShade(self):
        if self.game.drag.shade_img:
            self.game.drag.shade_img.delete()
        self.game.drag.shade_img = None
        self.game.drag.shade_stack = None

    def _updateShade(self):
        game = self.game
        images = game.app.images
        img = images.getShade()
        if img is None:
            return
        drag = game.drag
        ##stacks = game.allstacks
        c = drag.cards[0]
        stacks = ( game.getClosestStack(c), )
        r1 = (c.x, c.y, c.x + images.CARDW, c.y + images.CARDH)
        sstack, sdiff, sx, sy = None, 999999999, 0, 0
        for s in stacks:
            if s is None or s is drag.stack or s in drag.noshade_stacks:
                continue
            if s.cards:
                c = s.cards[-1]
                r2 = (c.x, c.y, c.x + images.CARDW, c.y + images.CARDH)
            else:
                r2 = (s.x, s.y, s.x + images.CARDW, s.y + images.CARDH)
            if r1[2] <= r2[0] or r1[3] <= r2[1] or r2[2] <= r1[0] or r2[3] <= r1[1]:
                # rectangles do not intersect
                continue
            if s in drag.canshade_stacks:
                pass
            elif s.acceptsPile(drag.stack, drag.cards):
                drag.canshade_stacks.append(s)
            else:
                drag.noshade_stacks.append(s)
                continue
            diff = (r1[0] - r2[0])**2 + (r1[1] - r2[1])**2
            if diff < sdiff:
                sstack, sdiff, sx, sy = s, diff, r2[0], r2[1]
        if sstack is drag.shade_stack:
            return
        if sstack is None:
            self._deleteShade()
            return
        # move or create the shade image
        drag.shade_stack = sstack
        if drag.shade_img:
            img = drag.shade_img
            img.moveTo(sx, sy)
        else:
            img = MfxCanvasImage(game.canvas, sx, sy, image=img, anchor=ANCHOR_NW)
            drag.shade_img = img
        # raise/lower the shade image to the correct position
        if tkname == "tk":
            if drag.shadows:
                img.lower(drag.shadows[0])
            else:
                img.lower(drag.cards[0].item)
        elif tkname == "gnome":
            img.tkraise()
            drag.stack.group.tkraise()

    def _stopDrag(self):
        drag = self.game.drag
        stack, cards = drag.stack, drag.cards
        self._deleteShade()
        drag.canshade_stacks = []
        drag.noshade_stacks = []
        for s in drag.shadows:
            s.delete()
        drag.shadows = []
        drag.stack = None
        drag.cards = []
        return stack, cards

    # finish a drag operation
    def finishDrag(self):
        if self.game.app.opt.dragcursor:
            self.game.canvas.config(cursor=self.game.app.top_cursor)
        stack, cards = self._stopDrag()
        if cards:
            assert stack is self
            self.releaseHandler(cards)

    # cancel a drag operation
    def cancelDrag(self):
        if self.game.app.opt.dragcursor:
            self.game.canvas.config(cursor=self.game.app.top_cursor)
        stack, cards = self._stopDrag()
        if cards:
            assert stack is self
            for card in cards:
                self._position(card)


# /***********************************************************************
# // The Talon is a stack with support for dealing.
# // It is also responsible for creating all cards of a game.
# ************************************************************************/

class TalonStack(Stack):
    def __init__(self, x, y, game, max_rounds=1, num_deal=1, **cap):
        Stack.__init__(self, x, y, game, cap=cap)
        self.max_rounds = max_rounds
        self.num_deal = num_deal
        self.round = 1

    def assertStack(self):
        Stack.assertStack(self)
        n = self.game.game_info.redeals
        if n < 0: assert self.max_rounds == n
        else:     assert self.max_rounds == n + 1

    # Control of dealing is transferred to the game which usually
    # transfers it back to the Talon - see dealCards() below.
    def clickHandler(self, event):
        return self.game.dealCards()

    def rightclickHandler(self, event):
        return self.clickHandler(event)

    # Usually called by Game.canDealCards()
    def canDealCards(self):
        return len(self.cards) > 0

    # Actual dealing, usually called by Game.dealCards().
    # Either deal all cards in Game.startGame(), or subclass responsibility.
    def dealCards(self):
        pass

    # Create all cards for the game. Called from Game.create()
    def createCards(self, decks, progress=None):
        ##timer = Timer("Talon.createCards")
        if progress: pstep = int((100 - progress.percent) / (decks * 4.0))
        cards = []
        id = 0
        for deck in range(decks):
            for suit in range(4):
                for rank in range(13):
                    card = Card(id, deck, suit, rank, self.game, x=self.x, y=self.y)
                    #card.hide(self)
                    cards.append(card)
                    id = id + 1
                if progress and pstep > 0: progress.update(step=pstep)
        if progress: progress.update(percent=100)
        ##print timer
        return cards

    # remove all cards from all stacks
    def removeAllCards(self):
        for stack in self.game.allstacks:
            while stack.cards:
                ##stack.removeCard(update=0)
                stack.removeCard(unhide=0, update=0)
            stack.updateText()

    # Deal a card to each of the RowStacks. Return number of cards dealt.
    def dealRow(self, rows=None, flip=1, reverse=0, frames=-1):
        if rows is None: rows = self.game.s.rows
        return self.dealToStacks(rows, flip, reverse, frames)

    # Same, but no error if too little available cards.
    def dealRowAvail(self, rows=None, flip=1, reverse=0, frames=-1):
        if rows is None: rows = self.game.s.rows
        if len(self.cards) < len(rows):
            rows = rows[:len(self.cards)]
        return self.dealToStacks(rows, flip, reverse, frames)

    def dealToStacks(self, stacks, flip=1, reverse=0, frames=-1):
        if not self.cards or not stacks:
            return 0
        assert len(self.cards) >= len(stacks)
        old_state = self.game.enterState(self.game.S_DEAL)
        if reverse:
            stacks = list(stacks)[:]
            stacks.reverse()
        for r in stacks:
            assert not self.getCard().face_up
            assert r is not self
            if frames == 0 and self.game.moves.state == self.game.S_INIT:
                # optimized a little bit for initial dealing
                c = self.removeCard(update=0)
                r.addCard(c, update=0)
                # doing the flip after the move seems to be a little faster
                if flip:
                    c.showFace()
            else:
                if flip:
                    self.game.flipMove(self)
                self.game.moveMove(1, self, r, frames=frames)
        self.game.leaveState(old_state)
        return len(stacks)

    def updateText(self, update_rounds=1, update_redeal=1):
        ##assertView(self)
        Stack.updateText(self)
        if update_rounds:
            if self.texts.rounds is not None:
                t = "Round %d" % self.round
                self.texts.rounds["text"] = t
        if update_redeal:
            deal = self.canDealCards() != 0
            if self.images.redeal is not None:
                img = (self.getRedealImages())[deal]
                if img is not None and img is not self.images.redeal_img:
                    self.images.redeal.config(image=img)
                    self.images.redeal_img = img
                t = ("", "Redeal")[deal]
            else:
                t = ("Stop", "Redeal")[deal]
            if self.texts.redeal is not None:
                if t != self.texts.redeal_str:
                    self.texts.redeal["text"] = t
                    self.texts.redeal_str = t

    # this must be called *after* prepareView()
    def prepareView(self):
        Stack.prepareView(self)
        if not self.is_visible or self.images.bottom is None:
            return
        if self.images.redeal is not None or self.texts.redeal is not None:
            return
        images = self.game.app.images
        cx, cy, ca = self.x + images.CARDW/2, self.y + images.CARDH/2, "center"
        if images.CARDW >= 54 and images.CARDH >= 54:
            # add a redeal image above the bottom image
            img = (self.getRedealImages())[self.max_rounds != 1]
            if img is not None:
                self.images.redeal = MfxCanvasImage(self.game.canvas, cx, cy,
                                         image=img, anchor="center")
                self.images.redeal_img = img
                self.images.redeal.tkraise(self.images.bottom)
                self.images.redeal.addtag(self.group)
                if images.CARDH >= 90:
                    cy, ca = self.y + images.CARDH - 4, "s"
                else:
                    ca = None
        if ca:
            # add a redeal text above the bottom image
            if self.max_rounds != 1:
                images = self.game.app.images
                self.texts.redeal = MfxCanvasText(self.game.canvas, cx, cy,
                                         anchor=ca)
                self.texts.redeal_str = ""
                self.texts.redeal.tkraise(self.images.bottom)
                self.texts.redeal.addtag(self.group)

    def getBottomImage(self):
        return self.game.app.images.getTalonBottom()

    def getRedealImages(self):
        # return a tuple/list of two PhotoImages
        return self.game.app.redeal_images


# A single click deals one card to each of the RowStacks.
class DealRowTalonStack(TalonStack):
    def dealCards(self):
        return self.dealRowAvail()


# For games where the Talon is only used for the initial dealing.
# It's not really "invisible", just out of play.
class InvisibleTalonStack(TalonStack):
    # no bindings
    def initBindings(self):
        pass
    # no bottom
    def getBottomImage(self):
        return None


# /***********************************************************************
# // An OpenStack is a stack where cards can be placed and dragged
# // (i.e. FoundationStack, RowStack, ReserveStack, ...)
# //
# // Note that it defaults to max_move = 1 and max_accept = 0.
# ************************************************************************/

class OpenStack(Stack):
    def __init__(self, x, y, game, **cap):
        kwdefault(cap, max_move=1, max_accept=0, max_cards=999999)
        Stack.__init__(self, x, y, game, cap=cap)


    #
    # Capabilities {model}
    #

    def acceptsPile(self, from_stack, cards):
        # default for OpenStack: we cannot accept cards (max_accept defaults to 0)
        return self.basicAcceptsPile(from_stack, cards)

    def canMovePile(self, cards):
        # default for OpenStack: we can move the top card (max_move defaults to 1)
        return self.basicCanMovePile(cards)

    def canFlipCard(self):
        # default for OpenStack: we can flip the top card
        if self.basicIsBlocked():
            return 0
        card = self.getCard()
        return card and not card.face_up

    def canDropCards(self, stacks):
        if self.basicIsBlocked():
            return (None, 0)
        card = self.getCard()
        if not card:
            return (None, 0)
        pile = [card]
        if self.canMovePile(pile):
            for s in stacks:
                if s is not self and s.acceptsPile(self, pile):
                    return (s, 1)
        return (None, 0)


    #
    # Access methods {model}
    #

    # get the largest moveable pile - uses canMovePile()
    def getPile(self):
        if self.cap.max_move < 1:
            return None
        if len(self.cards) > self.cap.max_move:
            pile = self.cards[-self.cap.max_move:]
        else:
            pile = self.cards[:]
        while len(pile) >= self.cap.min_move:
            if self.canMovePile(pile):
                return pile
            del pile[0]
        return None


    #
    # Mouse handlers {controller}
    #

    def clickHandler(self, event):
        flipstacks, dropstacks, quickstacks = self.game.getAutoStacks(event)
        if self in flipstacks and self.canFlipCard():
            self.playFlipMove()
            return 0                # continue this event (start a drag)
        return 0

    def rightclickHandler(self, event):
        if self.doubleclickHandler(event):
            return 1
        if self.game.app.opt.quickplay:
            flipstacks, dropstacks, quickstacks = self.game.getAutoStacks(event)
            if self in quickstacks:
                return self.quickPlayHandler(event)
        return 0

    def doubleclickHandler(self, event):
        # Flip or drop a card, if possible
        flipstacks, dropstacks, quickstacks = self.game.getAutoStacks(event)
        if self in flipstacks and self.canFlipCard():
            self.playFlipMove()
            return 0                # continue this event (start a drag)
        if self in dropstacks:
            to_stack, ncards = self.canDropCards(self.game.s.foundations)
            if to_stack:
                self.playMoveMove(ncards, to_stack)
                return 1
        return 0

    def controlclickHandler(self, event):
        # highlight matching cards
        if self.game.app.opt.highlight_cards:
            return self.highlightMatchingCards(event)
        return 0

    def releaseHandler(self, cards):
        stack = self.game.getClosestStack(cards[0])
        if not stack or stack is self or not stack.acceptsPile(self, cards):
            Stack.releaseHandler(self, cards)
        else:
            # this code actually moves the cards
            self.playMoveMove(len(cards), stack, frames=0)

    def quickPlayHandler(self, event=None, tostacks=None):
        if tostacks is None:
            ##tostacks = self.game.s.rows + self.game.s.reserves
            ##tostacks = self.game.sg.dropstacks
            tostacks = self.game.s.foundations + self.game.sg.dropstacks
        pile1, pile2 = None, self.getPile()
        if not pile2:
            return 0
        i = self._findCard(event)
        if i >= 0:
            pile = self.cards[i:]
            if len(pile) != len(pile2) and self.canMovePile(pile):
                pile1 = pile
        moves = []
        for pile in (pile1, pile2):
            if not pile:
                continue
            for s in tostacks:
                if s is not self and s.acceptsPile(self, pile):
                    score = (len(s.cards) == 0)     # prefer non-empty piles
                    moves.append((score, len(moves), s, len(pile)))
            if moves:
                moves.sort()
                ##print moves
                self.playMoveMove(moves[0][3], moves[0][2])
                self.game.stats.quickplay_moves = self.game.stats.quickplay_moves + 1
                return 1
        return 0


# /***********************************************************************
# // Foundations stacks
# ************************************************************************/

class AbstractFoundationStack(OpenStack):
    def __init__(self, x, y, game, suit, **cap):
        kwdefault(cap, suit=suit, base_suit=suit, base_rank=ACE,
                  dir=-1, max_accept=1, max_cards=13)
        apply(OpenStack.__init__, (self, x, y, game), cap)

    def canDropCards(self, stacks):
        return (None, 0)

    def clickHandler(self, event):
        return 0

    def rightclickHandler(self, event):
        return 0

    def quickPlayHandler(self, event):
        return 0

    def getBottomImage(self):
        return self.game.app.images.getSuitBottom(self.cap.base_suit)


# A SameSuit_FoundationStack is the typical Foundation stack.
# It builds up in rank and suit.
class SS_FoundationStack(AbstractFoundationStack):
    def acceptsPile(self, from_stack, cards):
        if not AbstractFoundationStack.acceptsPile(self, from_stack, cards):
            return 0
        if self.cards:
            # check the rank
            if (cards[0].rank + self.cap.dir) % self.cap.mod != self.cards[-1].rank:
                return 0
        return 1


# A Rank_FoundationStack builds up in rank and ignores color and suit.
class RK_FoundationStack(SS_FoundationStack):
    def __init__(self, x, y, game, suit=ANY_SUIT, **cap):
        apply(SS_FoundationStack.__init__, (self, x, y, game, suit), cap)

    def assertStack(self):
        SS_FoundationStack.assertStack(self)
        assert self.cap.suit == ANY_SUIT
        assert self.cap.color == ANY_COLOR


# A AlternateColor_FoundationStack builds up in rank and alternate color.
# It is used in only a few games.
class AC_FoundationStack(SS_FoundationStack):
    def __init__(self, x, y, game, suit, **cap):
        kwdefault(cap, base_suit=suit)
        apply(SS_FoundationStack.__init__, (self, x, y, game, ANY_SUIT), cap)

    def acceptsPile(self, from_stack, cards):
        if not SS_FoundationStack.acceptsPile(self, from_stack, cards):
            return 0
        if self.cards:
            # check the color
            if cards[0].color == self.cards[-1].color:
                return 0
        return 1


# /***********************************************************************
# // Abstract classes for row stacks.
# ************************************************************************/

# Abstract class.
class SequenceStack:
    def _isSequence(self, cards):
        # Are the cards in a basic sequence for our stack ?
        raise SubclassResponsibility

    def _isAcceptableSequence(self, cards):
        return self._isSequence(cards)

    def _isMoveableSequence(self, cards):
        return self._isSequence(cards)

    def acceptsPile(self, from_stack, cards):
        if not self.basicAcceptsPile(from_stack, cards):
            return 0
        # cards must be an acceptable sequence
        if not self._isAcceptableSequence(cards):
            return 0
        # [topcard + cards] must be an acceptable sequence
        if self.cards and not self._isAcceptableSequence([self.cards[-1]] + cards):
            return 0
        return 1

    def canMovePile(self, cards):
        return self.basicCanMovePile(cards) and self._isMoveableSequence(cards)


# Abstract class.
class BasicRowStack(OpenStack):
    def __init__(self, x, y, game, **cap):
        kwdefault(cap, dir=1, base_rank=ANY_RANK)
        apply(OpenStack.__init__, (self, x, y, game), cap)
        self.CARD_YOFFSET = game.app.images.CARD_YOFFSET


# Abstract class.
class SequenceRowStack(SequenceStack, BasicRowStack):
    def __init__(self, x, y, game, **cap):
        kwdefault(cap, max_move=999999, max_accept=999999)
        apply(BasicRowStack.__init__, (self, x, y, game), cap)


# /***********************************************************************
# // Row stacks (the main playing piles on the Tableau).
# ************************************************************************/

#
# Implementation of common row stacks follows here.
#

# An AlternateColor_RowStack builds down by rank and alternate color.
# e.g. Klondike
class AC_RowStack(SequenceRowStack):
    def _isSequence(self, cards):
        return isAlternateColorSequence(cards, self.cap.mod, self.cap.dir)

# A SameSuit_RowStack builds down by rank and suit.
class SS_RowStack(SequenceRowStack):
    def _isSequence(self, cards):
        return isSameSuitSequence(cards, self.cap.mod, self.cap.dir)

# A Rank_RowStack builds down by rank.
class RK_RowStack(SequenceRowStack):
    def _isSequence(self, cards):
        return isRankSequence(cards, self.cap.mod, self.cap.dir)

# A Freecell_AlternateColor_RowStack
class FreeCell_AC_RowStack(AC_RowStack):
    def canMovePile(self, cards):
        max_move = self.game.getNumberOfFreeReserves() + 1
        return AC_RowStack.canMovePile(self, cards) and len(cards) <= max_move

# A Freecell_SameSuit_RowStack (i.e. Baker's Game)
class FreeCell_SS_RowStack(SS_RowStack):
    def canMovePile(self, cards):
        max_move = self.game.getNumberOfFreeReserves() + 1
        return SS_RowStack.canMovePile(self, cards) and len(cards) <= max_move

# A Spider_AlternateColor_RowStack builds down by rank and alternate color,
# but accepts sequences that match by rank only.
class Spider_AC_RowStack(AC_RowStack):
    def _isAcceptableSequence(self, cards):
        return isRankSequence(cards, self.cap.mod, self.cap.dir)

# A Spider_SameSuit_RowStack builds down by rank and suit,
# but accepts sequences that match by rank only.
class Spider_SS_RowStack(SS_RowStack):
    def _isAcceptableSequence(self, cards):
        return isRankSequence(cards, self.cap.mod, self.cap.dir)

# A YukonAlternateColor_RowStack builds down by rank and alternate color,
# but can move any face-up cards regardless of sequence.
class Yukon_AC_RowStack(BasicRowStack):
    def __init__(self, x, y, game, **cap):
        kwdefault(cap, max_move=999999, max_accept=999999)
        apply(BasicRowStack.__init__, (self, x, y, game), cap)

    def _isSequence(self, c1, c2):
        return c1.rank == (c2.rank + self.cap.dir) % self.cap.mod and c1.color != c2.color

    def acceptsPile(self, from_stack, cards):
        if not self.basicAcceptsPile(from_stack, cards):
            return 0
        # [topcard + card[0]] must be acceptable
        if self.cards and not self._isSequence(self.cards[-1], cards[0]):
            return 0
        return 1

# A YukonSameSuit_RowStack builds down by rank and suit,
# but can move any face-up cards regardless of sequence.
class Yukon_SS_RowStack(Yukon_AC_RowStack):
    def _isSequence(self, c1, c2):
        return c1.rank == (c2.rank + self.cap.dir) % self.cap.mod and c1.suit == c2.suit


#
# King-versions of some of the above stacks: they accepts only Kings or
# sequences starting with a King as base_rank cards (i.e. when empty).
#

class KingAC_RowStack(AC_RowStack):
    def __init__(self, x, y, game, **cap):
        kwdefault(cap, base_rank=KING)
        apply(AC_RowStack.__init__, (self, x, y, game), cap)

class KingSS_RowStack(SS_RowStack):
    def __init__(self, x, y, game, **cap):
        kwdefault(cap, base_rank=KING)
        apply(SS_RowStack.__init__, (self, x, y, game), cap)

class KingRK_RowStack(RK_RowStack):
    def __init__(self, x, y, game, **cap):
        kwdefault(cap, base_rank=KING)
        apply(RK_RowStack.__init__, (self, x, y, game), cap)


# /***********************************************************************
# // WasteStack (a helper stack for the Talon, e.g. in Klondike)
# ************************************************************************/

class WasteStack(OpenStack):
    pass


class WasteTalonStack(TalonStack):
    # A single click moves the top cards to the game's waste and
    # moves it face up; if we're out of cards, it moves the waste
    # back to the talon and increases the number of rounds.
    def __init__(self, x, y, game, max_rounds, num_deal=1, waste=None, **cap):
        apply(TalonStack.__init__, (self, x, y, game, max_rounds, num_deal), cap)
        self.waste = waste

    def prepareStack(self):
        TalonStack.prepareStack(self)
        if self.waste is None:
            self.waste = self.game.s.waste

    def canDealCards(self):
        waste = self.waste
        if self.cards:
            num_cards = min(len(self.cards), self.num_deal)
            return len(waste.cards) + num_cards <= waste.cap.max_cards
        elif waste.cards and self.round != self.max_rounds:
            return 1
        return 0

    def dealCards(self):
        old_state = self.game.enterState(self.game.S_DEAL)
        num_cards = 0
        waste = self.waste
        if self.cards:
            num_cards = min(len(self.cards), self.num_deal)
            assert len(waste.cards) + num_cards <= waste.cap.max_cards
            for i in range(num_cards):
                if not self.cards[-1].face_up:
                    self.game.flipMove(self)
                self.game.moveMove(1, self, waste, frames=4, shadow=0)
                self.fillStack()
        elif waste.cards and self.round != self.max_rounds:
            num_cards = len(waste.cards)
            self.game.turnStackMove(waste, self, update_flags=1)
        self.game.leaveState(old_state)
        return num_cards


class FaceUpWasteTalonStack(WasteTalonStack):
    def fillStack(self):
        if self.cards and not self.cards[-1].face_up:
            self.game.flipMove(self)


# /***********************************************************************
# // ReserveStack (free cell)
# ************************************************************************/

class ReserveStack(OpenStack):
    def __init__(self, x, y, game, **cap):
        kwdefault(cap, max_accept=1, max_cards=1)
        apply(OpenStack.__init__, (self, x, y, game), cap)

    def getBottomImage(self):
        return self.game.app.images.getReserveBottom()


# /***********************************************************************
# // InvisibleStack (an internal off-screen stack to hold cards)
# ************************************************************************/

class InvisibleStack(Stack):
    def __init__(self, game, **cap):
        x, y = -500, -500 - len(game.allstacks)
        kwdefault(cap, max_move=0, max_accept=0)
        Stack.__init__(self, x, y, game, cap=cap)

    def assertStack(self):
        Stack.assertStack(self)
        assert not self.is_visible

    # no bindings
    def initBindings(self):
        pass

    # no bottom
    def getBottomImage(self):
        return None


