/***************************************************************************
                          rsnap.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/


/****************************************************************************
** rsnap.cpp 1998/09/01 A. Mustun RibbonSoft
**
** Copyright (C) 1998 RibbonSoft.  All rights reserved.
**
*****************************************************************************/

#include <qbitmap.h>
#include <qcursor.h>

#include "rappwin.h"
#include "rsnap.h"
#include "rstatuspanel.h"
#include "rbehaviordef.h"
#include "rgraphdef.h"
#include "rgraphic.h"
#include "rconfig.h"
#include "rinputpanel.h"
#include "rlog.h"
#include "rmath.h"

// Constructor:
//
RSnap::RSnap()
{
  setFreeMode();
  cx=cy=0.0;
  cEl=0;
  snapRange=10.0;
  onlyElement=0;
  firstElement=0;
  showPreview=true;
  calcDirection=false;
  direction=0.0;

  lastUsedXPos=0.0;
  lastUsedYPos=0.0;
  lastUsedAngle=0.0;
  lastUsedRadius=10.0;
  lastUsedPosMode=0;

  snapDistance=10.0;
}



// Destructor:
//
RSnap::~RSnap()
{
}



// Get snapped point with current snap function:
//
//  _x/_y: returns snapped position
//  _definitive: clicked not only a move
//  _onlyElement: ptr to only element to snap
//  _showPreview: show preview in graphic
//  _calcDirection: calculate direction for arrow preview
//  _direction: returns the direction (angle)
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapPoint(float* _x, float* _y,
                 bool _definitive,
                 RElement* _onlyElement,
                 bool _showPreview,
                 bool _calcDirection,
                 float* _direction) 
{
  bool ret=false;
  
  if(currentDoc()) {

    // Coordinates set by keyboard:
    /*if(currentDoc()->keyCoordsEnabled()) {
      float kx, ky;
      currentDoc()->getKeyCoords(&kx, &ky);
      if(_x) *_x=kx;
      if(_y) *_y=ky;
      return true;
    }*/

    recalculateSnapRange();
    onlyElement=_onlyElement;
    showPreview=_showPreview;
    calcDirection=_calcDirection;
    direction=0.0;

    switch(snapMode) {
      case SNAP_FREE:
        ret = snapFree();
        break;

      case SNAP_GRID:
        ret = snapGrid();
        break;
      
      case SNAP_ENDPOINTS:
        ret = snapEndpoints();
        break;

      case SNAP_CENTERS:
        ret = snapCenters();
        break;

      case SNAP_MIDDLES:
        ret = snapMiddles();
        break;

      case SNAP_AUTOINTER:
        ret = snapAutoInter();
        break;

      case SNAP_MANINTER:
        ret = snapManInter(_definitive);
        break;

      case SNAP_DIST:
        if(true) {
          RInputPanel* inputPanel2=RAppWin::getRAppWin()->getInputPanel2();
          if(inputPanel2) {
            snapDistance = inputPanel2->getValue(0);
          }
          ret = snapDist();
        }
        break;

      case SNAP_ONELEMENT:
        ret = snapOnElement();
        break;

      case SNAP_KEYXY:
        if(true) {
          RInputPanel* inputPanel2=RAppWin::getRAppWin()->getInputPanel2();
          if(inputPanel2) {
            if(inputPanel2->isRadioButtonChecked(0)) {
              cx = inputPanel2->getValue(0);
              cy = inputPanel2->getValue(1);
              if(_definitive) lastUsedPosMode=0;
            }
            else {
              cx = currentDoc()->getRelZeroX() + inputPanel2->getValue(0);
              cy = currentDoc()->getRelZeroY() + inputPanel2->getValue(1);
              if(_definitive) lastUsedPosMode=1;
            }

            if(_definitive) {
              lastUsedXPos=inputPanel2->getValue(0);
              lastUsedYPos=inputPanel2->getValue(1);
            }
          }
          ret=true;
        }
        break;

      case SNAP_KEYAR:
        if(true) {
          RInputPanel* inputPanel2=RAppWin::getRAppWin()->getInputPanel2();
          if(inputPanel2) {
            if(inputPanel2->isRadioButtonChecked(0)) {
              cx = cos(inputPanel2->getValue(0)/ARAD) * inputPanel2->getValue(1);
              cy = sin(inputPanel2->getValue(0)/ARAD) * inputPanel2->getValue(1);
              if(_definitive) lastUsedPosMode=0;
            }
            else {
              cx = currentDoc()->getRelZeroX() + cos(inputPanel2->getValue(0)/ARAD) * inputPanel2->getValue(1);
              cy = currentDoc()->getRelZeroY() + sin(inputPanel2->getValue(0)/ARAD) * inputPanel2->getValue(1);
              if(_definitive) lastUsedPosMode=1;
            }
            if(_definitive) {
              lastUsedAngle=inputPanel2->getValue(0);
              lastUsedRadius=inputPanel2->getValue(1);
            }
          }
          ret=true;
        }
        break;
  
      default:
        break;
    }

    onlyElement=0;
    showPreview=true;
    if(_direction) *_direction=direction;
    direction=0.0;
    calcDirection=false;
  
    if(_x) *_x=cx;
    if(_y) *_y=cy;

    if(cx>=DEF_AREAMAX-DEF_MMTOL || 
       cy>=DEF_AREAMAX-DEF_MMTOL ||
       cx<=DEF_AREAMIN+DEF_MMTOL || 
       cy<=DEF_AREAMIN+DEF_MMTOL   ) ret=false;
    
    if(_showPreview) {
      if(ret) {
        currentDoc()->drawMark(cx, cy);
        //currentDoc()->drawPreviewElement();
        char buf[128];
        sprintf(buf, "Abs: X: %.4f / Y: %.4f\n"
                     "Rel: X: %.4f / Y: %.4f\n",
                     cx, cy,
                     cx-currentDoc()->getRelZeroX(),
                     cy-currentDoc()->getRelZeroY() );
        statusPanel()->setStatus(buf);
      }
      else {
        currentDoc()->delMark();
        //currentDoc()->delPreviewElement();
      }
    }
  }

  return ret;
}



// Stop snapping (delete preview):
//
void
RSnap::stop() 
{
  RLOG("\nStop snapper");
  
  if(currentDoc()) {
    RLOG("\nDel Mark...");
    currentDoc()->delMark();
    cx = cy = DEF_AREAMAX;
    currentDoc()->delPreviewElement();
    currentDoc()->delPreview();
    RLOG("\nReset Preview Element...");
    currentDoc()->resetPreviewElement();
    RLOG("\nReset Highlight Element...");
    currentDoc()->resetHighlightElement();
    RLOG("\nDel Highlight Element...");
    currentDoc()->delHighlightElement();
    RLOG("\nSet First Element to zero...");
    firstElement=0;
  }
  
  RLOG("\nSnapper stopped");
}



// Snap to nothing (free):
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapFree()
{
  currentDoc()->enableMark(true);
  currentDoc()->getRealMousePos(&cx, &cy);
  return true;
}



// Snap to grid:
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapGrid()
{
  double minDist=DEF_AREAMAX;
  double dist;
  double startCX,           // Mousepointers is in this
        stopCX,            // range of grid units
        startCY,
        stopCY;
  double gpX;               // Grid point X
  double gpY;               // Grid point Y
  float  rmx;               // real mousepos X
  float  rmy;               // real mousepos Y

  currentDoc()->getRealMousePos(&rmx, &rmy);

  currentDoc()->enableMark(true);

  // enlarge grid to reasonable size:
  //
  double mg = currentDoc()->getCurrentGridWidth();
  /*
  double mg = RCONFIG->getSettingDouble("CAD:GridWidth");
  if(mg<0.00001) mg = 0.00001;

  while(mg*currentDoc()->getZoom()<6.0 && mg<RIni::getRIni()->getGridWidth()*100000.0) { mg*=10.0; }
  */

  // Get the range, where the mousepointer is (in factor 100 of grid pieces)
  //
  if(rmx>=0.0) {
    startCX = 0.0;
    for(stopCX=0.0; stopCX<=DEF_AREAMAX; stopCX+=mg*100.0) {
      if(stopCX>rmx) break;
      startCX = stopCX;
    }
    //startCX = 0.0;
  }
  else {
    stopCX = 0.0;
    for(startCX=0.0; startCX>=DEF_AREAMIN; startCX-=mg*100.0) {
      if(startCX<rmx) break;
      stopCX = startCX;
    }
    //stopCX = 0.0;
  }

  if(rmy>=0.0) {
    startCY = 0.0;
    for(stopCY=0.0; stopCY<=DEF_AREAMAX; stopCY+=mg*100.0) {
      if(stopCY>rmy) break;
      startCY = stopCY;
    }
    //startCY = 0.0;
  }
  else {
    stopCY = 0.0;
    for(startCY=0.0; startCY>=DEF_AREAMIN; startCY-=mg*100.0) {
      if(startCY<rmy) break;
      stopCY = startCY;
    }
    //stopCY = 0.0;
  }

  // Get the snapped x-coordinate:
  //
  for(gpX=startCX; gpX<=stopCX+mg; gpX+=mg) {
    // Check grid point:
    //
    dist = mtGetDistance(rmx, rmy, gpX, startCY);
    if(dist<minDist) {
      minDist = dist;
      cx = gpX;
    }
    else if(minDist<DEF_AREAMAX) {
      break;
    }
  }

  // Get the snapped y-coordinate:
  //
  minDist = DEF_AREAMAX;
  for(gpY=startCY; gpY<=stopCY+mg; gpY+=mg) {
    // Check grid point:
    //
    dist = mtGetDistance(rmx, rmy, gpX, gpY);
    if(dist<minDist) {
      minDist = dist;
      cy = gpY;
    }
    else if(minDist<DEF_AREAMAX) {
      break;
    }
  }

  return true;
}



// Snap to endpoints:
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapEndpoints()
{
  float rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);

  //float minDist=DEF_AREAMAX;  // minimum measured distance
  float minDist=snapRange;    // minimum measured distance
  float dist;                 // measured distance
  RElement* el;               // element ptr (walks through elements)
  RElement* snappedElement=0;    // snapped element
  bool startPoint=true;       // snapped to startpoint?
  bool found=false;           // snapped to point?

  currentDoc()->enableMark(true);

  if(currentDoc()->count()>0) {
    for(el=currentDoc()->elementFirst(); el!=0; el=currentDoc()->elementNext()) {
      if((onlyElement==0 || el==onlyElement) && 
         el->getFlag(E_VISIBLE)) {
         
        // Check Startpoint of element:
        //
        dist = mtGetDistance(el->getX1(), el->getY1(),
                             rmx, rmy);
        if(dist<minDist && dist<snapRange) {
          minDist = dist;
          startPoint=true;
          snappedElement=el;
          found=true;
        }

        // Check Endpoint of element (if element has an endpoint):
        //
        if(el->hasEndpoint()) {
          dist = mtGetDistance(el->getX2(), el->getY2(),
                               rmx, rmy);
          if(dist<minDist && dist<snapRange) {
            minDist = dist;
            startPoint=false;
            snappedElement=el;
            found=true;
          }
        }
      }
    }
  }
  
  if(found && snappedElement) {
    if(startPoint) {
      cx = snappedElement->getX1();
      cy = snappedElement->getY1();
      if(calcDirection) direction=mtCorrAngle(snappedElement->getDirection1()+180.0);
    }
    else {
      cx = snappedElement->getX2();
      cy = snappedElement->getY2();
      if(calcDirection) direction=mtCorrAngle(snappedElement->getDirection2()+180.0);
    }
  }
  else {
    cx = rmx;
    cy = rmy;
  }

  return found;
}



// Snap to Centers:
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapCenters()
{
  float rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);

  //float minDist=DEF_AREAMAX;  // minimum measured distance
  float minDist=snapRange;    // minimum measured distance
  float dist;                 // measured distance
  RElement* el;               // element ptr (walks through elements)
  RElement* snappedElement=0;    // snapped element
  bool found=false;           // snapped to point?

  currentDoc()->enableMark(true);

  if(currentDoc()->count()>0) {
    for(el=currentDoc()->elementFirst(); el!=0; el=currentDoc()->elementNext()) {
      if((onlyElement==0 || el==onlyElement) && 
         el->getFlag(E_VISIBLE)) {
         
        // Check Startpoint of element:
        //
        dist = el->getDistanceToPoint(rmx, rmy);
        if(dist<minDist && dist<snapRange) {
          minDist = dist;
          snappedElement=el;
          found=true;
        }
      }
    }
  }
  
  if(found && snappedElement) {
    cx = snappedElement->getCenterX();
    cy = snappedElement->getCenterY();
  }
  else {
    cx = rmx;
    cy = rmy;
  }

  return found;
}



// Snap to Middles:
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapMiddles()
{
  float rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);

  //float minDist=DEF_AREAMAX;  // minimum measured distance
  float minDist=snapRange;    // minimum measured distance
  float dist;                 // measured distance
  RElement* el;               // element ptr (walks through elements)
  RElement* snappedElement=0;    // snapped element
  bool found=false;           // snapped to point?

  currentDoc()->enableMark(true);

  if(currentDoc()->count()>0) {
    for(el=currentDoc()->elementFirst(); el!=0; el=currentDoc()->elementNext()) {
      if((onlyElement==0 || el==onlyElement) && 
         el->getFlag(E_VISIBLE)) {
         
        // Check Startpoint of element:
        //
        dist = el->getDistanceToPoint(rmx, rmy);
        if(dist<minDist && dist<snapRange) {
          minDist = dist;
          snappedElement=el;
          found=true;
        }
      }
    }
  }
  
  if(found && snappedElement) {
    cx = snappedElement->getMiddleX();
    cy = snappedElement->getMiddleY();
  }
  else {
    cx = rmx;
    cy = rmy;
  }

  return found;
}



// Snap to Auto Intersections:
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapAutoInter()
{
  float rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);
  
  //float minDist=DEF_AREAMAX;
  float minDist=snapRange;    // minimum measured distance
  float dist;
  float lb, bb, rb, tb;    // Element borders
  RElement* el1;           // element ptr (walks through elements)
  RElement* el2;           // element ptr (walks through elements)
  //RElement* elBak;         // Backup element pos for two loops
  bool  found=false;       // There was a point in catch range
  float ix1, iy1;          // Intersection result 1
  float ix2, iy2;          // Intersection result 2
  bool  ires1, ires2;      // Tells if there are results

  currentDoc()->enableMark(true);

  if(currentDoc()->count()>0) {
    for(el1=currentDoc()->elementFirst(); el1!=0; el1=currentDoc()->elementNext()) {

      /*if(mouseMoveBreak &&
         PeekMessage(&Msg, *np->GetCadWin(), WM_MOUSEMOVE, WM_MOUSEMOVE, PM_NOREMOVE)) {
        cx=ocx;
        cy=ocy;
        break;
      }*/

      el1->getBorders(&lb, &bb, &rb, &tb);

      if( el1->getFlag(E_VISIBLE)        &&
          el1->getElementTyp()!=T_POINT  &&
          //el1->isOnScreen()            &&
          rmx < rb + snapRange           &&
          rmx > lb - snapRange           &&
          rmy < tb + snapRange           &&
          rmy > bb - snapRange              ) {

        currentDoc()->elementNext();
        for(el2=currentDoc()->elementCurrent(); el2!=0; el2=currentDoc()->elementNext()) {
        
          if(el1!=el2) {
            el2->getBorders(&lb, &bb, &rb, &tb);
  
            if( el2->getFlag(E_VISIBLE)        &&
                el2->getElementTyp()!=T_POINT  &&
                //el2->IsOnScreen()            &&
                rmx < rb + snapRange           &&
                rmx > lb - snapRange           &&
                rmy < tb + snapRange           &&
                rmy > bb - snapRange              ) {
  
              ix1=iy1=ix2=iy2=DEF_AREAMAX;
              ires1=ires2=false;
              
              el1->getIntersection(el2, 
                                   &ires1, &ix1, &iy1, 
                                   &ires2, &ix2, &iy2, 
                                   0,
                                   true);
              
              if(ires1) {
                dist = mtGetDistance(rmx, rmy, ix1, iy1);
                if(dist<minDist) {
                  minDist = dist;
                  cx = ix1;
                  cy = iy1;
                  found=true;
                }
              }
  
              if(ires2) {
                dist = mtGetDistance(rmx, rmy, ix2, iy2);
                if(dist<minDist) {
                  minDist = dist;
                  cx = ix2;
                  cy = iy2;
                  found=true;
                }
              }
            }
          }
        }
        currentDoc()->setCurrentElement(el1);
      }
    }
  }
    
  if(!found) {
    cx = rmx;
    cy = rmy;
  }

  return found;

}



// Snap to Manual Intersections:
//
// _definitive: user clicked (not only mouse move)
//
//  return: true: snapped to point
//          false: no point snapped so far
//
bool
RSnap::snapManInter(bool _definitive)
{
  bool ret=true;   // return value
  float rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);

  cx = DEF_AREAMAX;
  cy = DEF_AREAMAX;
  
  currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
  
  if(currentDoc()->count()>0) {
    // Catch first element for intersection:
    //
    if(!firstElement) {
      currentDoc()->setHighlightFilter(T_LONG);
      currentDoc()->enableMark(false);
      // 1st element is definitive:
      //
      if(_definitive) {
        firstElement = cEl;
      }
      ret=false;
    }
    
    // Catch second element for intersection:
    //
    else {
      currentDoc()->setHighlightFilter(T_LONG, firstElement);
      bool ires1, ires2;
      float ix1, iy1, ix2, iy2;
      ires1=false;
      ires2=false;
      ix1=iy1=ix2=iy2=DEF_AREAMAX;
          
      firstElement->getIntersection(cEl, 
                                    &ires1, &ix1, &iy1,
                                    &ires2, &ix2, &iy2,
                                    0, false);
        
      if(ires1) {
        currentDoc()->enableMark(true);
          
        // Test both intersections:
        //
        if(ires2) {
          if(  mtGetDistance(rmx, rmy, ix1, iy1)
             > mtGetDistance(rmx, rmy, ix2, iy2) ){

            cx = ix2;
            cy = iy2;
          }
          else {
            cx = ix1;
            cy = iy1;
          }
        }
        else {
          cx = ix1;
          cy = iy1;
        }
      }
      else {
        cx = rmx;
        cy = rmy;
        currentDoc()->enableMark(false);
        ret=false;
      }
          
      // 2nd element is definitive
      //
      if(_definitive) {
        firstElement=0;
      }
    }
  }

  return ret;

}



// Snap to Dist:
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapDist()
{
  float rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);

  //float minDist=DEF_AREAMAX;  // minimum measured distance
  float minDist=snapRange;    // minimum measured distance
  float dist;                 // measured distance
  RElement* el;               // element ptr (walks through elements)
  RElement* snappedElement=0;    // snapped element
  bool found=false;           // There was a point in range

  currentDoc()->enableMark(true);

  if(currentDoc()->count()>0) {

    // Get nearest element:
    //
    for(el=currentDoc()->elementFirst(); el!=0; el=currentDoc()->elementNext()) {
      if((onlyElement==0 || el==onlyElement) && 
         el->getFlag(E_VISIBLE)) {
         
        // Check Startpoint of element:
        //
        dist = el->getDistanceToPoint(rmx, rmy);
        if(dist<minDist && dist<snapRange) {
          minDist = dist;
          snappedElement=el;
          found=true;
        }
      }
    }
    
    if(found) {
      // Start- or endpoint:
      //
      // Check Startpoint of element:
      //
      minDist = mtGetDistance(rmx, rmy, 
                              snappedElement->getX1(),
                              snappedElement->getY1());
  
      // Check Endpoint of element:
      //
      dist = mtGetDistance(rmx, rmy, 
                           snappedElement->getX2(),
                           snappedElement->getY2());

      // Endpoint is nearer:
      //
      if(dist<minDist) {
        snappedElement->getDistPoint(&cx, &cy, snapDistance, false);
      }

      // Startpoint nearer:
      //
      else {
        snappedElement->getDistPoint(&cx, &cy, snapDistance, true);
      }
    }
  }

  if(!found) {
    cx = rmx;
    cy = rmy;
  }

  return found;

}



// Snap to a point on an element:
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapOnElement()
{
  float rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);

  //float minDist=DEF_AREAMAX;  // minimum measured distance
  float minDist=snapRange;    // minimum measured distance
  float dist;                 // measured distance
  RElement* el;               // element ptr (walks through elements)
  RElement* snappedElement=0; // snapped element
  bool found=false;           // snapped to point?

  currentDoc()->enableMark(true);

  // Find element which is next to cursor:
  //
  if(currentDoc()->count()>0) {
    for(el=currentDoc()->elementFirst(); el!=0; el=currentDoc()->elementNext()) {
      if((onlyElement==0 || el==onlyElement) &&
         el->getFlag(E_VISIBLE)) {

        // Check Distance to this element:
        //
        dist = el->getDistanceToPoint(rmx, rmy);
        if(dist<minDist && dist<snapRange) {
          minDist = dist;
          snappedElement=el;
          found=true;
        }
      }
    }
  }

  if(found && snappedElement) {
    snappedElement->getTouchPoint(&cx, &cy, rmx, rmy);
  }
  else {
    cx = rmx;
    cy = rmy;
  }

  return found;
}



// Snap an element:
//
//   _definitive: true: already chosen (clicked): delete highlighting
//   _show: show highlighting
//   _types: snap only types of group (e.g.: T_ALL, T_STRAIGT, T_CURCULAR, T_LINE, T_ARC, ...)
//   _exception: dont snap this element (or 0)
//
bool 
RSnap::snapElement(bool _show,
                   int _types,
                   RElement* _exception)
{
  float rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);

  recalculateSnapRange();
  
  bool found=false;    // element found
  float minDist=snapRange;    // minimum measured distance
  float dist;
  RElement* el;        // element ptr (walks through elements)
  
  cEl=0;

  //int omx=np->GetCadWin()->GetMX();
  //int omy=np->GetCadWin()->GetMY();

  if(currentDoc()->count()>0) {
    //for(vc=0; vc<np->GetCadGraphic()->GetVNum(); ++vc) {
    for(el=currentDoc()->elementFirst(); el!=0; el=currentDoc()->elementNext()) {
  
      // break on mouse move:
      //if(omx!=np->GetCadWin()->GetMX() ||
      //   omy!=np->GetCadWin()->GetMY()    ) break;
      
      if(el->isInGroup(_types) &&
         el!=_exception        && 
         el->getFlag(E_VISIBLE)   ) {
  
        // Check distance point<->element:
        //
        dist = el->getDistanceToPoint(rmx, rmy);
        if(dist<minDist) {
          minDist = dist;
          cEl = el;
          found=true;
        }
      }
    }

    if(_show) {
      currentDoc()->setHighlightElement(cEl);
      currentDoc()->drawHighlightElement();
    }
  }

  if(found && cEl) return true;
  else             return false;

}



// Recalculate snap range (in real unit)
//
void 
RSnap::recalculateSnapRange()
{
  if(currentDoc()) {
    snapRange = RCONFIG->getSettingInt("CAD:SnapRange")
                / currentDoc()->getZoom();
  }
}


// EOF



















