// adiffile.cpp - LogConv ADIF file support
// Copyright (C) 1999  Robert Barron, KA5WSS
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include "adiffile.h"
#include "ctyfile.h"
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>


extern char *LogConvVersion;

// The official ADIF field name designators.
char *RAdifFile::fieldName[] = {"eor", "call", "qso_date", "time_on", "time_off", "freq", "band", "mode",
                          "tx_pwr", "rst_sent", "rst_rcvd", "qsl_sent", "qsl_rcvd", "name",
                          "qth", "state", "comment", "contest_id", "stx", "srx", "prec", "check",
                          "arrl_sect", "rx_pwr", "cqz", "ituz", "gridsquare", "iota",
                          "tx_count", "category", "age", "ten_ten", "arci_num",
                          "prefecture", "rcv", NULL};

static char *modeName[] = {NULL, "CW", "SSB", "RTTY", "AM", "FM", "TOR", "PKT", "PACTOR",
                            "GTOR", NULL};

static char *bandName[] = {NULL, "160M", "80M", "40M", "30M", "20M", "17M", "15M", "12M",
                          "10M", "6M", "??M", "2M", "1.25M", "70cm", "35cm", "23cm", "13cm",
                          "9cm", "6cm", "3cm", "2cm", "1.25cm", "6mm", "2.5mm", "2mm", "1mm",
                                                                  "SAT", "PACK", "NOV", "LIGHT", NULL};

static char *contestName[] = {NULL, "SS", "ARRLDX", "CQWW", "WPX", "IARU", "ARRL10", "ARRL160", "CQ160",
                                "FD", "CQP", "ARRLDX", "WAE", "VHFQSO", "DXPED", "AA", "NAQP", "GRIDLOC",
                                "YODX", "KCJ", "1010", "SPRINT", "SAC", "TOEC", "JADX", "ARCI",
                                "IOTA", "SprINT", NULL, "STEWP", "HELV", "WAG", "ARI"};

RAdifFile::RAdifFile(char *name)
{
    RAdifFile::open(name);
    ctyfile.open("cqww.cty");
    location = 0;
}

boolean RAdifFile::open(char *name, int mode)
{
    char buffer[BUFFER_SIZE];
    
#ifdef LINUX
    if (!RFile::open(name, mode))
        return FALSE;
#else
    // State ios::binary again just in case user does not.
    if (!RFile::open(name, mode | ios::binary))
        return FALSE;
#endif
        
    if (!(mode & ios::out))
    {
        if (!RFile::getString((char *)buffer, BUFFER_SIZE))
                                return FALSE;
                  while ((strstr(buffer, "<eoh>") == NULL)
                && (strstr(buffer, "<EOH>") == NULL))
            if (!RFile::getString((char *)buffer, BUFFER_SIZE))
                return FALSE;
    }
    
    areaFile = new RAreaFile("stsec.dat");

    return TRUE;
}

boolean RAdifFile::nextQso(qso_data *qso)
{
    char valueBuf[BUFFER_SIZE];
    int fieldNum;
    int fieldSize;
    struct tm dosTime, dosTimeOff;
    unsigned int khz, mhz;
    unsigned char c;
    int bTimeOff = FALSE;
    
    endOfRecord = TRUE;

    memset(valueBuf, '\0', BUFFER_SIZE);
    qso->init();
    memset(&dosTime, '\0', sizeof(struct tm));
    memset(&dosTimeOff, '\0', sizeof(struct tm));
    
    do
    {
        while (RAdifFile::getFieldName(valueBuf, &fieldNum, &fieldSize))
        {
            switch (fieldNum) {
                case ADIF_CALL:
                    qso->setCallsign(valueBuf, fieldSize);
                    break;
                case ADIF_QSO_DATE:
                    dosTime.tm_mday = atoi(&valueBuf[6]);
                    valueBuf[6] = (char)NULL;
                    dosTime.tm_mon = atoi(&valueBuf[4]) - 1;
                    valueBuf[4] = (char)NULL;
                    dosTime.tm_year = atoi(&valueBuf[0]) - 1900;
                    break;
                case ADIF_TIME_ON:
                    if (fieldSize == 6)
                    {
                        dosTime.tm_sec = atoi(&valueBuf[4]);
                        valueBuf[4] = (char)NULL;
                    }
                    dosTime.tm_min = atoi(&valueBuf[2]);
                    valueBuf[2] = (char)NULL;
                    dosTime.tm_hour = atoi(&valueBuf[0]);
                    break;
                case ADIF_TIME_OFF:
                    bTimeOff = TRUE;
                    if (fieldSize == 6)
                    {
                        dosTimeOff.tm_sec = atoi(&valueBuf[4]);
                        valueBuf[4] = (char)NULL;
                    }
                    dosTimeOff.tm_min = atoi(&valueBuf[2]);
                    valueBuf[2] = (char)NULL;
                    dosTimeOff.tm_hour = atoi(&valueBuf[0]);
                    break;
                case ADIF_FREQ:
                    khz = mhz = 0;
                    sscanf(valueBuf, "%d.%d", &mhz, &khz);
                    // Convert to tenths of kilohertz
                    while (khz >= 1000)
                        khz /= 10;
                    qso->setFreq(((mhz * 1000) + khz) * 10);
                    break;
                case ADIF_BAND:
                    for (c = 1; c <= MAX_BAND; c++)
                                                                if (stricmp(valueBuf, bandName[c]) == 0)
                        {
                            qso->setBand(c);
                            break;
                        }
                    break;
                case ADIF_MODE:
                    for (c = 1; c <= MAX_MODE; c++)
                        if (stricmp(valueBuf, modeName[c]) == 0)
                        {
                            qso->setMode(c);
                            break;
                        }
                    if ((qso->getMode() == 0) && (stricmp(valueBuf, "FSK") == 0))
                        qso->setMode(RTTY_MODE);
                    break;
                case ADIF_TX_PWR:
                    qso->setTxPower(valueBuf);
                    break;
                case ADIF_RST_SENT:
                    qso->setSentRST(atoi(valueBuf));
                    break;
                case ADIF_RST_RCVD:
                    qso->setReceivedRST(atoi(valueBuf));
                    break;
                case ADIF_QSL_SENT:
                    qso->setQslSent(valueBuf[0]);
                    break;
                case ADIF_QSL_RCVD:
                    qso->setQslReceived(valueBuf[0]);
                    break;
                case ADIF_NAME:
                    qso->setName(valueBuf);
                    break;
                case ADIF_STATE:
                    qso->setState(valueBuf, QSO_DATA_STATE_SIZE);
                    break;
                case ADIF_QTH:
                    qso->setQth(valueBuf, QSO_DATA_QTH_SIZE);
                    break;
                case ADIF_COMMENT:
                    if (fieldSize < INFO_SIZE)
                        qso->setInfo(valueBuf);
                    else
                        qso->setInfo(valueBuf, INFO_SIZE -1);
                    break;
                case ADIF_CONTEST_ID:
                    for (c = 1; c <= MAX_CONTEST; c++)
                        if ((contestName[c] != NULL)
                                                                          && (strncmp(valueBuf, contestName[c], fieldSize) == 0))
                        {
                            qso->setContest(c);
                            break;
                        }
                    break;
                case ADIF_SERIAL_SENT:
                    qso->setSerialNumberSent(valueBuf);
                    break;
                case ADIF_SERIAL_RCVD:
                    if (header.contest == ARRL_SS_CONTEST)
                        qso->setSerialNumber(valueBuf);
                    else
                        qso->setSerialNumber(valueBuf);
                    break;
                case ADIF_PRECEDENCE:
                    qso->setSSPrecedence(valueBuf[0]);
                    break;
                case ADIF_CHECK:
                    qso->setSSCheck(valueBuf);
                    break;
                case ADIF_SECTION:
                    qso->setSection(valueBuf);
                case ADIF_CQ_ZONE:
                    qso->setCqZone(valueBuf);
                    break;
                case ADIF_ITU_ZONE:
                    qso->setItuZone(valueBuf);
                    break;
                case ADIF_RCVD_PWR:
                case ADIF_GRID:
                case ADIF_IOTA:
                    qso->setInfo(valueBuf);
                    break;
                case ADIF_TX_COUNT:
                    qso->setFdTransmitters(valueBuf);
                    break;
                case ADIF_CATEGORY:
                    qso->setFdCategory(valueBuf[0]);
                    break;
                case ADIF_AGE:
                    qso->setAge(valueBuf);
                    break;
                case ADIF_10_10_NUM:
                    qso->setTenTenNumber(valueBuf);
                    break;
                case ADIF_ARCI_NUMBER:
                    qso->setARCInumber(valueBuf);
                    break;
                case ADIF_PREFECTURE:
                    qso->setInfo(valueBuf);
                    break;
                case ADIF_RCV:
                    // This is a non-standard WriteLog extension.  WriteLog uses this field
                    // to store multiple fields of information instead of breaking them out.
                    // So we have to switch() on contest type here!
                    switch(header.contest)
                    {
                        case FD_CONTEST:
                                                                         c = valueBuf[strlen(valueBuf) - 1];
                            if (isalpha(c))
                                // Don't save the category unless it is a character
                                qso->setFdCategory(c);
                            valueBuf[strlen(valueBuf) - 1] = (char)NULL;
                            qso->setFdTransmitters(valueBuf);
                            break;
                        default:
                            // We currently do not support this contest type.  Do nothing.
                            break;
                    }
                    break;
                default:
                    // Unknown or unimplemented field type.
                    break;
            }
        }
    } while ((!endOfRecord) && (!eof()));

    // Finish off the date and time calculations
    qso->setTime(mktime(&dosTime) - timezone);

    if (bTimeOff)
    {
        dosTimeOff.tm_mday = dosTime.tm_mday;
        dosTimeOff.tm_mon = dosTime.tm_mon;
        dosTimeOff.tm_year = dosTime.tm_year;
        qso->setTimeOff(mktime(&dosTimeOff) - timezone);
        // If the computed off time is less than the on time then the
        //  QSO probably spanned two days.  Add one day to the off time.
        if (qso->getTimeOff() < qso->getTime())
            qso->setTimeOff(qso->getTimeOff() + 60 * 60 * 24);
    }

    location++;
    return TRUE;
}

boolean RAdifFile::getFieldName(char *valuePtr, int *fieldNum, int *fieldSize)
{
    char currentFieldName[20];
    static char *bufPtr = buffer;
    char temp;

    *fieldNum = -1;

    if (endOfRecord)
    {
        RFile::getString((char *)buffer, BUFFER_SIZE);
        bufPtr = buffer;
        endOfRecord = FALSE;
    }
        
    while ((*bufPtr != '<') && (*bufPtr != (char)NULL))
        bufPtr++;
        
    if (*bufPtr == (char)NULL)
    {
        // Could not find a field name through to end of line.
        //  So read a new line, return FALSE so that this routine will
        //  be called again.
        RFile::getString((char *)buffer, BUFFER_SIZE);
        bufPtr = buffer;
        endOfRecord = FALSE;
        return FALSE;
    }
    else
        *bufPtr++;
        
    for (temp = 0; (*bufPtr != ':') && (*bufPtr != '>'); temp++)
        currentFieldName[temp] = *bufPtr++;
    currentFieldName[temp] = (char)NULL;

    for (char c = 0; fieldName[c] != NULL; c++)
                  if (stricmp(fieldName[c], currentFieldName) == 0)
            *fieldNum = c;

    if (*fieldNum == 0)
    {
        endOfRecord = TRUE;
        return FALSE;
    }
        
    if (*bufPtr == ':')
    {
        char size[10];

        memset(size, '\0', 10);
        bufPtr++;
        for (char i = 0; *bufPtr != '>'; i++)
            size[i] = *bufPtr++;
        *fieldSize = atoi(size);
    }
    else
        fieldSize = 0;

    if (fieldSize > 0)
    {
        // The field size has been explicitly stated.
        bufPtr++;
        for (int i = *fieldSize; i > 0; i--)
        {
            if (*bufPtr == (char)NULL)
            {
                RFile::getString((char *)buffer, BUFFER_SIZE);
                bufPtr = buffer;
                endOfRecord = FALSE;
            }
            *valuePtr++ = *bufPtr++;
        }
        *valuePtr = (char)NULL;
    }
    else
    {
        bufPtr++;
        while (*bufPtr != '<')
            // Field values which do not contain size values had
            //  better not span more than one line of the file!
            *valuePtr++ = *bufPtr++;
        *valuePtr = (char)NULL;
    }
    
    return TRUE;
}

boolean RAdifFile::writeQso(qso_data *qso)
{
    char buffer[BUFFER_SIZE];
    char buffer2[BUFFER_SIZE];
    struct tm *dosTime;
    time_t tempTime;
    
    char tempName[40];               // Used for ctyfile querying
    char prefix[8];
    int zone;
    char tempState[10];              // Used to store State/Prov. info

         sprintf(buffer, "<%s:%d>%s", fieldName[ADIF_CALL], strlen(qso->getCallsign()), qso->getCallsign());
    tempTime = qso->getTime();
    dosTime = gmtime(&tempTime);
    sprintf(buffer2, "<%s:8:d>", fieldName[ADIF_QSO_DATE]);
         strcat(buffer, buffer2);
    strftime(buffer2, 100, "%Y%m%d", dosTime);
    strcat(buffer, buffer2);
    sprintf(buffer2, "<%s:6>", fieldName[ADIF_TIME_ON]);
    strcat(buffer, buffer2);
    strftime(buffer2, 100, "%H%M%S", dosTime);
    strcat(buffer, buffer2);
    if ((header.contest == GENERAL_QSO) && (qso->getTimeOff() > 0))
    {
        time_t tempTime;
        
        tempTime = qso->getTimeOff();
        dosTime = gmtime(&tempTime);
        sprintf(buffer2, "<%s:6>", fieldName[ADIF_TIME_OFF]);
        strcat(buffer, buffer2);
        strftime(buffer2, 100, "%H%M%S", dosTime);
        strcat(buffer, buffer2);
    }
    strcat(buffer, "\r\n");
    RFile::write(buffer, strlen(buffer));

    if (qso->getBand() != 0)
        sprintf(buffer, "<%s:%d>%s", fieldName[ADIF_BAND], strlen(bandName[qso->getBand()]), bandName[qso->getBand()]);
    if (qso->getFreq() != 0)
    {
        unsigned int mhz, khz;
        char tmp[15];
        
        mhz = (unsigned int)(qso->getFreq() / 10000);
        khz = (unsigned int)((qso->getFreq() % 10000) / 10);
        sprintf(tmp, "%d.%03d", mhz, khz);
        sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_FREQ], strlen(tmp), tmp);
        strcat(buffer, buffer2);
    }
    sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_MODE], strlen(modeName[qso->getMode()]), modeName[qso->getMode()]);
    strcat (buffer, buffer2);

    switch (header.contest)
    {
        case ARRL_SS_CONTEST:
        case FD_CONTEST:
        case GRIDLOC_CONTEST:
            break;
        default:
            if (qso->getSentRST() > 100)
            {
                sprintf(buffer2, "<%s:3>%d", fieldName[ADIF_RST_SENT], qso->getSentRST());
                strcat(buffer, buffer2);
            }
            else if (qso->getSentRST() != 0)
            {
                sprintf(buffer2, "<%s:2>%d", fieldName[ADIF_RST_SENT], qso->getSentRST());
                strcat(buffer, buffer2);
            }
            
            if (qso->getReceivedRST() > 100)
            {
                sprintf(buffer2, "<%s:3>%d", fieldName[ADIF_RST_RCVD], qso->getReceivedRST());
                strcat(buffer, buffer2);
            }
            else if (qso->getReceivedRST() != 0)
            {
                sprintf(buffer2, "<%s:2>%d", fieldName[ADIF_RST_RCVD], qso->getReceivedRST());
                strcat(buffer, buffer2);
            }
    }
    strcat(buffer, "\r\n");
    RFile::write(buffer, strlen(buffer));

    if ((header.contest > 0) && (header.contest <= MAX_CONTEST)
         && (header.contest != GENERAL_QSO))
        sprintf(buffer, "<%s:%d>%s", fieldName[ADIF_CONTEST_ID],
                    strlen(contestName[header.contest]), contestName[header.contest]);

    switch (header.contest)
    {
        case ARRL_SS_CONTEST:
            sprintf(buffer2, "<%s:%d>%d", fieldName[ADIF_SERIAL_SENT],
                             numOfDigits((long)location + 1), location + 1);
            strcat(buffer, buffer2);
            sprintf(buffer2, "<%s:%d>%d<%s:1>%c<%s:2>%d", fieldName[ADIF_SERIAL_RCVD],
                        numOfDigits((long)qso->getSerialNumber()), qso->getSerialNumber(),
                        fieldName[ADIF_PRECEDENCE], qso->getSSPrecedence(),
                        fieldName[ADIF_CHECK], qso->getSSCheck());
            strcat(buffer, buffer2);
            sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_SECTION], strlen(qso->getSection()),
                        qso->getSection());
            strcat(buffer, buffer2);
            break;
        case ARRL_DX_CONTEST:
            sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_RCVD_PWR], strlen(qso->getInfo()), qso->getInfo());
            strcat(buffer, buffer2);
            break;
        case CQ_WW_CONTEST:
            sprintf(buffer2, "<%s:2>%02d", fieldName[ADIF_CQ_ZONE], qso->getCqZone());
            strcat(buffer, buffer2);
            break;
        case CQ_WPX_CONTEST:
            sprintf(buffer2, "<%s:%d>%d", fieldName[ADIF_SERIAL_RCVD],
                             numOfDigits(qso->getSerialNumber()), qso->getSerialNumber());
            strcat(buffer, buffer2);
            break;
        case IARU_CONTEST:
            if (qso->getItuZone() != 0)
                sprintf(buffer2, "<%s:2>%02d", fieldName[ADIF_ITU_ZONE], qso->getItuZone());
            else
                sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_COMMENT], strlen(qso->getInfo()), qso->getInfo());
            strcat(buffer, buffer2);                
            break;
        case ARRL_10M_CONTEST:
            if (qso->isInfoNumeric())
                sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_SERIAL_RCVD], strlen(qso->getInfo()), qso->getInfo());
            else
                sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_STATE], strlen(qso->getInfo()), qso->getInfo());
            strcat(buffer, buffer2);
            break;
        case ARRL_160M_CONTEST:
            ctyfile.GetInfo(qso->getCallsign(), tempName, prefix, &zone);
            if (strcmp(prefix, "W") == 0 || strcmp(prefix, "KH6") == 0
                 || strcmp(prefix, "KL7") == 0 || strcmp(prefix, "VE") == 0)
                sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_SECTION], strlen(qso->getInfo()), qso->getInfo());
            else
            {
                if (strlen(qso->getQth()))
                    sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_QTH], strlen(qso->getQth()), qso->getQth());
                else
                    sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_QTH], strlen(qso->getInfo()), qso->getInfo());
            }
            strcat(buffer, buffer2);
            break;
        case CQ_160M_CONTEST:
            sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_QTH], strlen(qso->getInfo()), qso->getInfo());
            strcat(buffer, buffer2);
            break;
        case FD_CONTEST:
            sprintf(buffer2, "<%s:%d>%d", "tx_count",
                             numOfDigits((long)qso->getFdTransmitters()), qso->getFdTransmitters());
            strcat(buffer, buffer2);
            sprintf(buffer2, "<%s:1>%c", "category", qso->getFdCategory());
            strcat(buffer, buffer2);
            sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_SECTION], strlen(qso->getSection()),
                        qso->getSection());
            strcat(buffer, buffer2);
            break;
        case CAL_QP_CONTEST:
            sprintf(buffer2, "<%s:%d>%d", fieldName[ADIF_SERIAL_RCVD],
                             numOfDigits(qso->getSerialNumber()), qso->getSerialNumber());
            strcat(buffer, buffer2);
            sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_QTH], strlen(qso->getInfo()),
                    qso->getInfo());
            strcat(buffer, buffer2);
            break;
        case ARRL_DX_CONTEST_DX:
            ctyfile.GetInfo(qso->getCallsign(), tempName, prefix, &zone);
            if (strcmp(prefix, "W") == 0)
                sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_STATE], strlen(qso->getInfo()), qso->getInfo());
            else
                sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_QTH], strlen(qso->getInfo()), qso->getInfo());
            strcat(buffer, buffer2);
            break;
        case WAE_CONTEST:
        case SAC_CONTEST:
            sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_RCVD_PWR], strlen(qso->getInfo()), qso->getInfo());
            strcat(buffer, buffer2);
            break;
        case ARRL_VHF_QSO_PARTY:
        case TOEC_CONTEST:
            sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_GRID], strlen(qso->getInfo()), qso->getInfo());
            strcat(buffer, buffer2);
            break;
        case DXPEDITION_MODE:
            if (strlen(qso->getName()) != 0)
            {
                sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_NAME], strlen(qso->getName()), qso->getName());
                strcat(buffer, buffer2);
            }
            if (strlen(qso->getInfo()) != 0)
            {
                sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_QTH], strlen(qso->getInfo()), qso->getInfo());
                strcat(buffer, buffer2);
            }
            break;
        case ALL_ASIAN_CONTEST:
            sprintf(buffer2, "<%s:%d>%d", fieldName[ADIF_AGE], numOfDigits(qso->getAge()),
                                            qso->getAge());
            strcat(buffer, buffer2);
            break;
        case NA_QSO_PARTY_CONTEST:
            sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_NAME], strlen(qso->getName()), qso->getName());
            strcat(buffer, buffer2);
            ctyfile.GetInfo(qso->getCallsign(), tempName, prefix, &zone);
            if (strcmp(prefix, "W") == 0 || strcmp(prefix, "KL7") == 0
                 || strcmp(prefix, "KH6") == 0)
                sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_STATE], strlen(qso->getInfo()), qso->getInfo());
            else
                sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_QTH], strlen(qso->getInfo()), qso->getInfo());
            strcat(buffer, buffer2);
            break;
        case GRIDLOC_CONTEST:
            sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_NAME], strlen(qso->getName()), qso->getName());
            strcat(buffer, buffer2);
            if (strlen(qso->getInfo()) > 0)
            {
                sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_GRID], strlen(qso->getInfo()), qso->getInfo());
                strcat(buffer, buffer2);
            }
            break;
        case YO_DX_CONTEST:
            sprintf(buffer2, "<%s:2>%02d", fieldName[ADIF_CQ_ZONE], qso->getCqZone());
            strcat(buffer, buffer2);
            if (strlen(qso->getInfo()))     // Is there a YO province name?
            {
                sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_QTH],
                        strlen(qso->getInfo()), qso->getInfo());
                strcat(buffer, buffer2);
            }
            break;
        case KCJ_CONTEST:
            sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_QTH],
                        strlen(qso->getInfo()), qso->getInfo());
            strcat(buffer, buffer2);
            break;
//        case TEN_TEN_CONTEST:
        case NA_SPRINT_CONTEST:
        case INTERNET_SPRINT_CONTEST:
            sprintf(buffer2, "<%s:%d>%d<%s:%d>%s<%s:%d>%s", fieldName[ADIF_SERIAL_RCVD],
                      numOfDigits((long)qso->getSerialNumber()), qso->getSerialNumber(),
                      fieldName[ADIF_NAME], strlen(qso->getName()),
                      qso->getName(), fieldName[ADIF_QTH], strlen(qso->getInfo()), qso->getInfo());
            strcat(buffer, buffer2);
            break;
        case JA_INTL_DX_CONTEST:
            sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_PREFECTURE], strlen(qso->getInfo()),
                             qso->getInfo());
            strcat(buffer, buffer2);
            break;
        case QRP_ARCI_CONTEST:
            sprintf(buffer2, "<%s:%d>%d", fieldName[ADIF_ARCI_NUMBER],
                             numOfDigits((long)qso->getARCInumber()), qso->getARCInumber());
            strcat(buffer, buffer2);
            if (strlen(qso->getInfo()) > 0)
            {
                sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_QTH], strlen(qso->getInfo()), qso->getInfo());
                strcat(buffer, buffer2);
            }
            break;
        case IOTA_CONTEST:
            sprintf(buffer2, "<%s:%d>%d", fieldName[ADIF_SERIAL_RCVD],
                             numOfDigits(qso->getSerialNumber()), qso->getSerialNumber());
            strcat(buffer, buffer2);
            if (strlen(qso->getInfo()) > 0)
            {
                sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_IOTA],
                               strlen(qso->getInfo()), qso->getInfo());
                strcat(buffer, buffer2);
            }
            break;
        case STEW_PERRY_CONTEST:
            sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_GRID], strlen(qso->getInfo()), qso->getInfo());
            strcat(buffer, buffer2);
            break;
        case HELVETIA_CONTEST:
            sprintf(buffer2, "<%s:%d>%d", fieldName[ADIF_SERIAL_RCVD],
                             numOfDigits(qso->getSerialNumber()), qso->getSerialNumber());
            strcat(buffer, buffer2);
            if (strlen(qso->getQth()) > 0)
            {
                sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_QTH], strlen(qso->getQth()), qso->getQth());
                strcat(buffer, buffer2);
            }
            break;
        case WAG_CONTEST:
        case ARI_CONTEST:
            if (qso->getSerialNumber() != 0)
            {
                sprintf(buffer2, "<%s:%d>%d", fieldName[ADIF_SERIAL_RCVD],
                                     numOfDigits(qso->getSerialNumber()), qso->getSerialNumber());
                strcat(buffer, buffer2);
            }
            sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_QTH], strlen(qso->getQth()), qso->getQth());
            strcat(buffer, buffer2);
            break;
        case GENERAL_QSO:
            // More to be added later.
            buffer[0] = (char)NULL;
            writeGeneralInfo(qso, buffer);
            break;
        default:
        /* No other field information is required. */
            break;
    }

    // Write out the U.S. state information, if this can be
    // determined from the contest exchange information.
    ctyfile.GetInfo(qso->getCallsign(), tempName, prefix, &zone);
    if (strcmp(prefix, "W") == 0 || strcmp(prefix, "KL7") == 0
        || strcmp(prefix, "KH6") == 0)
    {
        switch (header.contest)
        {
            case ARRL_SS_CONTEST:
                // Fill in the state field based on the section report.
                if (areaFile->GetInfo(qso->getSection(), tempState))
                {
                    strcat(buffer, "\r\n");
                    sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_STATE],
                                strlen(tempState), tempState);
                    strcat(buffer, buffer2);
                }
                break;
            case ARRL_160M_CONTEST:
                // Fill in the state field based on the section report.
                if (areaFile->GetInfo(qso->getInfo(), tempState))
                {
                    strcat(buffer, "\r\n");
                    sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_STATE],
                                strlen(tempState), tempState);
                    strcat(buffer, buffer2);
                }
                break;
            case FD_CONTEST:
                // Fill in the state field based on the section report.
                if (areaFile->GetInfo(qso->getSection(), tempState))
                {
                    strcat(buffer, "\r\n");
                    sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_STATE],
                                strlen(tempState), tempState);
                    strcat(buffer, buffer2);
                }
                break;
            default:
            // Can't determine the state from the exchange info.
            // Or, the state is already mentioned.
            break;
        }
    }
    
    if (strlen(buffer))
    {
        strcat(buffer, "\r\n");
        RFile::write(buffer, strlen(buffer));
    }
    RFile::write("<eor>\r\n", 7);
    location++;
    return FALSE;
}

void RAdifFile::writeGeneralInfo(qso_data *qso, char *buffer)
{
    char buffer2[BUFFER_SIZE];
    int size;

    if (qso->getContest() && (qso->getContest() <= MAX_CONTEST))
    {
        sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_CONTEST_ID],
                                         strlen(contestName[qso->getContest()]), contestName[qso->getContest()]);
        appendField(buffer, buffer2);
    }
    if (qso->getTxPower())
    {
        sprintf(buffer2, "%d", qso->getTxPower());
        size = strlen(buffer2);
        sprintf(buffer2, "<%s:%d>%d", fieldName[ADIF_TX_PWR], size, qso->getTxPower());
        appendField(buffer, buffer2);
    }
    if (strlen(qso->getName()))
    {
        sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_NAME],
                strlen(qso->getName()), qso->getName());
        appendField(buffer, buffer2);
    }
    if (strlen(qso->getQth()))
    {
        sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_QTH],
                strlen(qso->getQth()), qso->getQth());
        appendField(buffer, buffer2);
    }
    if (strlen(qso->getState()))
    {
        sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_STATE],
                strlen(qso->getState()), qso->getState());
        appendField(buffer, buffer2);
    }
    if (qso->getCqZone())
    {
        sprintf(buffer2, "<%s:2>%02d", fieldName[ADIF_CQ_ZONE], qso->getCqZone());
        appendField(buffer, buffer2);
    }
    if (qso->getItuZone())
    {
        sprintf(buffer2, "<%s:2>%02d", fieldName[ADIF_ITU_ZONE], qso->getItuZone());
        appendField(buffer, buffer2);
    }
    if (qso->getQslSent())
    {
        sprintf(buffer2, "<%s:1>%c", fieldName[ADIF_QSL_SENT], qso->getQslSent());
        appendField(buffer, buffer2);
    }
    if (qso->getQslReceived())
    {
        sprintf(buffer2, "<%s:1>%c", fieldName[ADIF_QSL_RCVD], qso->getQslReceived());
        appendField(buffer, buffer2);
    }
    if (strlen(qso->getInfo()))
    {
        sprintf(buffer2, "<%s:%d>%s", fieldName[ADIF_COMMENT],
                strlen(qso->getInfo()), qso->getInfo());
        appendField(buffer, buffer2);
    }
    return;
}

void RAdifFile::appendField(char *buffer, char *buffer2)
{
    if ((strlen(buffer) + strlen(buffer2)) > 80)
    {
        // Adding this field would expand the line past 80
        //  characters.  So write out the line we have now
        //  and start a new one.
        strcat(buffer, "\r\n");
        RFile::write(buffer, strlen(buffer));
        strcpy(buffer, buffer2);
    }
    else
        // Add this field to the existing line.
        strcat(buffer, buffer2);
        
    return;
}

boolean RAdifFile::writeHeader(header_data newheader)
{
    char buffer[BUFFER_SIZE];
    
    header = newheader;

    if ((header.contest != 0) && (header.contest != GENERAL_QSO))
    {
        sprintf(buffer, "QSO's made during %s contest\r\n", contestName[header.contest]);
        RFile::write(buffer, strlen(buffer));
    }
    RFile::write("\r\n<eoh>\r\n", 9);
    
    return TRUE;
}

int RAdifFile::numOfDigits(long num)
{
    // Warning!  A max return of 8 digits!!!!
    if (num < 10)
        return 1;
    else if (num < 100)
        return 2;
    else if (num < 1000)
        return 3;
    else if (num < 10000)
        return 4;
    else if (num < 100000)
        return 5;
    else if (num < 1000000)
        return 6;
    else if (num < 10000000)
        return 7;
    else
        return 8;
}
    
