// dbfile.cpp
//
// Copyright (C) 1999  Robert Barron, KA5WSS
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include "dbfile.h"
#include "adiffile.h"
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

static char *bands[] = {NULL, "160M", "80M", "40M", "30M", "20M", "17M", "15M", "12M",
                          "10M", "6M", "??M", "2M", "1.5CM", "70CM", "902", "1GHZ", "2GHZ",
                          "3GHZ", "5GHZ", "10GHZ", "24GHZ", "47GHZ", "76GHZ", "119GHZ",
                          "142GHZ", "241GHZ", "SAT", "PACK", "NOV", "LIGHT", NULL};
#if defined(__NT__)
// Store the Mbands and Fbands tables in different formats.  Under DOS, including the
//  the floating point libraries appears to lead to weird linking errors under 16bit
//  compilation.  Instead of trying to find out why, just make it work with strings.              
static double Mbands[] = {0, 160, 80, 40, 30, 20, 17, 15, 12, 10, 6, 4, 2, 1.5, .7, .33,
                            .23, .13, .09, .05, .03, .02, .0125, .004, .002, .002, .001,
                            0, 0, 0, 0};
static double Fbands[] = {0, 1.8, 3.5, 7, 10, 14, 18, 21, 24, 28, 50, 76, 144, 222, 430,
                             902, 1296, 2304, 3300, 5600, 10000, 24000, 47000, 76000, 119000,
                             142000, 241000, 0, 0, 0, 0};
#else
static char *Mbands[] = {NULL, "160.000", " 80.000", " 40.000", " 30.000", " 20.000", " 17.000",
                           " 15.000", " 12.000", " 10.000", "  6.000", "  4.000", "  2.000",
                           "  1.500", "  0.700", "  0.330", "  0.230", "  0.130", "  0.090",
                           "  0.050", "  0.030", "  0.020", "  0.012", "  0.004", "  0.002",
                           "  0.002", "  0.001", "  0.000", "  0.000", "  0.000", "  0.000", NULL};
static char *Fbands[] = {NULL, "     1.800", "     3.500", "     7.000", "    10.000",
                           "    14.000", "    18.000", "    21.000", "    24.000", "    28.000",
                           "    50.000", "    76.000", "   144.000", "   222.000", "   430.000",
                           "   902.000", "  1296.000", "  2304.000", "  3300.000", "  5600.000",
                           " 10000.000", " 24000.000", " 47000.000", " 76000.000", "119000.000",
                           "142000.000", "241000.000", "     0.000", "     0.000", "     0.000",
                           "     0.000", NULL};
#endif                           
static char *modes[] = {NULL, "CW", "SSB", "RTTY", "AM", "FM", "AMTOR", "PACKET", NULL};

RDbFile::RDbFile()
{
    location = 0;
}

RDbFile::RDbFile(char *name)
{
    RDbFile::open(name);
}

boolean RDbFile::open(char *name, int mode)
{
    
#ifdef LINUX
    if (!RFile::open(name, mode))
        return FALSE;
#else
    // State ios::binary again just in case the user does not.
    if (!RFile::open(name, mode | ios::binary))
        return FALSE;
#endif

    if (!(mode & ios::out))          // Not an output file
        RFile::read((void *)&dbheader, sizeof(DBHeader));
    
    location = 0;

    return TRUE;
}

boolean RDbFile::close()
{
    char fieldValue[2];
    
    /* Write end of file character to the file. */
    memset(fieldValue, '\x1a', 1);
    RFile::write(fieldValue, 1);

    /* Position to the start of the file and rewrite the header record.        */
    /*  It should now contain correct information about the number of records. */
    RFile::position(0);
    RFile::write(&dbheader, sizeof(DBHeader));
    RFile::close();

    return TRUE;
}

boolean RDbFile::writeFieldEntry(char *fieldName, char fieldType, unsigned char fieldSize,
                                 unsigned char decimalSize)
{
    FieldEntry field;
    
    memset(&field, '\0', sizeof(FieldEntry));
    strncpy(field.name, fieldName, 11);
    field.type = fieldType;
    field.length = fieldSize;
    field.decimals = decimalSize;
    dbheader.header_length += sizeof(FieldEntry);
    dbheader.record_length += field.length;
    RFile::write(&field, sizeof(FieldEntry));
    
    return TRUE;
}

/* This routine writes out the dBase header information including the data */
/*  dictionary.  It does NOT have room for contest specific information! */
boolean RDbFile::writeHeader(header_data newheader)
{
    time_t tempTime;
    struct tm *structTime;
    
    if (headerWritten)
        return FALSE;

    RQsoFile::storeHeader(newheader);

    memset(&dbheader, '\0', sizeof(DBHeader));

    // Determine today's date (UTC date, not local!) to place in dBase header.
    tempTime = time(NULL);
    structTime = gmtime(&tempTime);
    
    dbheader.db_version = 3;
    dbheader.update_year = (char)structTime->tm_year;
    dbheader.update_month = (char)(structTime->tm_mon + 1);
    dbheader.update_day = (char)structTime->tm_mday;
    dbheader.header_length = sizeof(DBHeader) + 1;
    dbheader.record_length = 1;
    dbheader.num_recs = 0;
    RFile::write(&dbheader, sizeof(DBHeader));

    writeFieldEntry(RAdifFile::fieldName[ADIF_CALL], 'C', 16);
    if (fileOptions.fields & FIELD_QSONUM)
        writeFieldEntry("QSO_NUM", 'N', 6);
    writeFieldEntry(RAdifFile::fieldName[ADIF_QSO_DATE], 'D', 8);
    if (fileOptions.time == TIME_APPROACH)
        writeFieldEntry("TIME_ONTM9", 'C', 8);      // Approach's way of specifying a time field
    else
        writeFieldEntry(RAdifFile::fieldName[ADIF_TIME_ON], 'C', 8);         // ADIF standard
    if (header.contest == GENERAL_QSO)
    {
        if (fileOptions.time == TIME_APPROACH)
            writeFieldEntry("TIME_OFFTM9", 'C', 8);      // Approach's way of specifying a time field
        else
            writeFieldEntry(RAdifFile::fieldName[ADIF_TIME_OFF], 'C', 8);         // ADIF standard
    }
    if ((fileOptions.band == BAND_METERS) || (fileOptions.band == BAND_MHZ))
        writeFieldEntry(RAdifFile::fieldName[ADIF_BAND], 'N', 6, 3);
    else
        writeFieldEntry(RAdifFile::fieldName[ADIF_BAND], 'C', 5);            // ADIF standard
    if (fileOptions.fields & FIELD_FREQ)
        writeFieldEntry(RAdifFile::fieldName[ADIF_FREQ], 'N', 9, 3);
    writeFieldEntry(RAdifFile::fieldName[ADIF_MODE], 'C', 5);
    
    switch (header.contest) {
        case ARRL_SS_CONTEST:
            writeFieldEntry("SERIAL", 'N', 4);
            writeFieldEntry("PREC", 'C', 1);
            writeFieldEntry("CHECK", 'N', 2);
            writeFieldEntry("SECTION", 'C', 4);
            break;
        case ARRL_DX_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'C', 3);
            // Power needs to be a type char because of "KW", etc.
            writeFieldEntry(RAdifFile::fieldName[ADIF_RCVD_PWR], 'C', 8);
            break;
        case CQ_WW_CONTEST:
        case DXPEDITION_MODE:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_CQ_ZONE], 'N', 2);
            break;
        case CQ_WPX_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'C', 3);
            writeFieldEntry("SERIAL", 'N', 5);
            break;
        case IARU_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_ITU_ZONE], 'N', 2);
            writeFieldEntry("HQ_STATION", 'C', 6);
            break;
        case ARRL_160M_CONTEST:
        case CQ_160M_CONTEST:
        case KCJ_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'C', 3);
            writeFieldEntry("QTH", 'C', 8);
            break;
        case CAL_QP_CONTEST:
            writeFieldEntry("SERIAL", 'N', 4);
            writeFieldEntry("QTH", 'C', 8);
            break;
        case FD_CONTEST:
            writeFieldEntry("TX_COUNT", 'N', 2);
            writeFieldEntry("CATEGORY", 'C', 1);
            writeFieldEntry("SECTION", 'C', 4);
            break;
        case ARRL_VHF_QSO_PARTY:
            writeFieldEntry(RAdifFile::fieldName[ADIF_GRID], 'C', 8);
            break;
        case ALL_ASIAN_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'C', 3);
            writeFieldEntry("AGE", 'N', 3);
            break;
        case NA_QSO_PARTY_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_NAME], 'C', 10);
            writeFieldEntry("STATE", 'C', 5);
            break;
        case GRIDLOC_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_NAME], 'C', 10);
            writeFieldEntry(RAdifFile::fieldName[ADIF_GRID], 'C', 5);
            break;
        case YO_DX_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'N', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'N', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_CQ_ZONE], 'N', 2);
            writeFieldEntry("COUNTY", 'C', 2);
            break;
        case TEN_TEN_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_NAME], 'C', 10);
            writeFieldEntry("10-X", 'N', 5);
            writeFieldEntry("QTH", 'C', 8);
            break;
        case NA_SPRINT_CONTEST:
            writeFieldEntry("SERIAL", 'N', 4);
            writeFieldEntry(RAdifFile::fieldName[ADIF_NAME], 'C', 10);
            writeFieldEntry("QTH", 'C', 8);
            break;
        case SAC_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'C', 3);
            writeFieldEntry("SERIAL", 'N', 4);
            break;
        case TOEC_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_GRID], 'C', 5);
            break;
        case JA_INTL_DX_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'C', 3);
            writeFieldEntry("PREF", 'N', 2);
            break;
        case QRP_ARCI_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'C', 3);
            writeFieldEntry("PWR_MEM", 'N', 5);
            writeFieldEntry("QTH", 'C', 8);
            break;
        case IOTA_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'C', 3);
            writeFieldEntry("SERIAL", 'N', 4);
            writeFieldEntry("IOTA", 'C', 6);
            break;
        case INTERNET_SPRINT_CONTEST:
            writeFieldEntry("SERIAL", 'N', 4);
            writeFieldEntry(RAdifFile::fieldName[ADIF_NAME], 'C', 10);
            writeFieldEntry("QTH", 'C', 8);
            break;
        case GENERAL_QSO:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'C', 3);
            break;
        case STEW_PERRY_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_GRID], 'C', 5);
            break;
        case HELVETIA_CONTEST:
        case WAG_CONTEST:
        case ARI_CONTEST:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'C', 3);
            writeFieldEntry("SERIAL", 'N', 4);
            writeFieldEntry("QTH", 'C', 5);
            break;
        case ARRL_10M_CONTEST:
        default:
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_SENT], 'C', 3);
            writeFieldEntry(RAdifFile::fieldName[ADIF_RST_RCVD], 'C', 3);
            writeFieldEntry("INFO", 'C', 8);
    };
        
    RFile::write("\r", 1);     // Write header termination characters

    return TRUE;
}

boolean RDbFile::nextQso(qso_data *qso)
{
    // This feature not yet implemented.
    qso = qso;
    return FALSE;
}

boolean RDbFile::writeQso(qso_data *qso)
{
    char fieldValue[20];
    struct tm *dosTime;
    unsigned int sentRST;
    time_t tempTime;

    if (qso->getSentRST() == 0)
    {
        if ((qso->getMode() == CW_MODE) || (qso->getMode() == RTTY_MODE) || (qso->getMode() == PACKET_MODE))
            sentRST = 599;
        else
            sentRST = 59;
    }
    else
        sentRST = qso->getSentRST();

    // Write the callsign to the file.
    RFile::write(" ", 1);
    memset(fieldValue, ' ', 20);
    memcpy(fieldValue, qso->getCallsign(), strlen(qso->getCallsign()));
    RFile::write(fieldValue, 16);
    if (fileOptions.fields & FIELD_QSONUM)
    {
        sprintf(fieldValue, "%ld", location + 1);
        RFile::write(fieldValue, 6);
    }

    // Write date and time to the file.
    tempTime = qso->getTime();
    dosTime = gmtime(&tempTime);
    strftime(fieldValue, 8, "%Y%m%d", dosTime);
    RFile::write(fieldValue, 8);
    strftime(fieldValue, 8, "%H%M%S00", dosTime);
    RFile::write(fieldValue, 8);
    if (header.contest == GENERAL_QSO && qso->getTimeOff() > 0)
    {
        tempTime = qso->getTimeOff();
        dosTime = gmtime(&tempTime);
        strftime(fieldValue, 8, "%H%M%S00", dosTime);
        RFile::write(fieldValue, 8);
    }

    // Write Amateur band
    memset(fieldValue, ' ', 20);
    if (fileOptions.band == BAND_METERS)
    {
        // Write Amateur band, in Meters numerically
#if defined(__NT__)        
        sprintf(fieldValue, "%6.3f", Mbands[qso->getBand()]);
#else
        strcpy(fieldValue, Mbands[qso->getBand()]);
#endif        
        RFile::write(fieldValue, 6);
    }
    else
    {
        // Write Amateur band, in Meters (ADIF format)
        memcpy(fieldValue, bands[qso->getBand()], strlen(bands[qso->getBand()]));
        RFile::write(fieldValue, 5);
    }
    if (fileOptions.fields & FIELD_FREQ)
    {
        // Write frequency, in MHz
        if (qso->getFreq() == 0)
#if defined(__NT__)        
            sprintf(fieldValue, "%9.3f", Fbands[qso->getBand()]);
#else
            strcpy(fieldValue, Fbands[qso->getBand()]);
#endif            
        else
            // Full frequency information is available.
            sprintf(fieldValue, "%5d.%03d", (qso->getFreq() / 10000), ((qso->getFreq() % 10000) / 10));
        RFile::write(fieldValue, 9);
    }

    // Write mode
    memset(fieldValue, ' ', 20);
    memcpy(fieldValue, modes[qso->getMode()], strlen(modes[qso->getMode()]));
    RFile::write(fieldValue, 5);

    switch(header.contest) {
        case ARRL_SS_CONTEST:
        case FD_CONTEST:
        case ARRL_VHF_QSO_PARTY:
        case CAL_QP_CONTEST:
        case NA_QSO_PARTY_CONTEST:
        case GRIDLOC_CONTEST:
        case TEN_TEN_CONTEST:
        case NA_SPRINT_CONTEST:
        case INTERNET_SPRINT_CONTEST:
            // Don't record the RST info, it's not required.
            break;
        default:
            memset(fieldValue, ' ', 20);
            sprintf(fieldValue, "%3d", sentRST);
            RFile::write(fieldValue, 3);
            
            memset(fieldValue, ' ', 20);
            sprintf(fieldValue, "%3d", qso->getReceivedRST());
            RFile::write(fieldValue, 3);
    }

    switch (header.contest) {
        case ARRL_SS_CONTEST:
            // Write serial number received and precedence received.
            memset(fieldValue, ' ', 20);
            sprintf(fieldValue, "%4d%c", qso->getSerialNumber(), qso->getSSPrecedence());
            RFile::write(fieldValue, 5);
        
            // Write Check received.
            memset(fieldValue, ' ', 20);
            sprintf(fieldValue, "%2d", qso->getSSCheck());
            RFile::write(fieldValue, 2);
        
            // Write the section to the file.
            writeField(qso->getSection(), (unsigned char)strlen(qso->getSection()), 4);
            break;
         case CQ_WW_CONTEST:
            memset(fieldValue, ' ', 20);
            sprintf(fieldValue, "%02d", qso->getCqZone());
            RFile::write(fieldValue, 2);
            break;
         case CQ_WPX_CONTEST:
            memset(fieldValue, ' ', 20);
            sprintf(fieldValue, "%5d", qso->getSerialNumber());
            RFile::write(fieldValue, 5);
            break;
         case DXPEDITION_MODE:
         case JA_INTL_DX_CONTEST:
            writeField(qso->getInfo(), qso->getInfoLength(), 2);
            break;
         case IARU_CONTEST:
            // We need to write out both the ITU Zone and HQ Station fields for every QSO
            // here.  That's the reason for the 6 blank characters in the first case and
            // 2 in the second.
            if (qso->getItuZone() > 0)
            {
                memset(fieldValue, ' ', 20);
                sprintf(fieldValue, "%02d      ", qso->getItuZone());
                RFile::write(fieldValue, 8);
            }
            else
            {
                memset(fieldValue, ' ', 2);
                RFile::write(fieldValue, 2);
                memset(fieldValue, ' ', 6);
                memcpy(fieldValue, qso->getInfo(), qso->getInfoLength());
                RFile::write(fieldValue, 6);
            }

            break;
         case FD_CONTEST:
            //Write Transmitter count and category received.
            memset(fieldValue, ' ', 20);
            sprintf(fieldValue, "%2d%c", qso->getFdTransmitters(), qso->getFdCategory());
            RFile::write(fieldValue, 3);
        
            //Write Section received.
            writeField(qso->getSection(), (unsigned char)strlen(qso->getSection()), 4);
            break;
        case CAL_QP_CONTEST:
            memset(fieldValue, ' ', 20);
            sprintf(fieldValue, "%4d", qso->getSerialNumber());
            RFile::write(fieldValue, 4);
            
            writeField(qso->getInfo(), qso->getInfoLength(), 8);
            break;
        case ALL_ASIAN_CONTEST:
            writeField(qso->getInfo(), qso->getInfoLength(), 3);
            break;
        case NA_QSO_PARTY_CONTEST:
        case GRIDLOC_CONTEST:
            writeField(qso->getName(), qso->getNameLength(), 10);
            writeField(qso->getInfo(), qso->getInfoLength(), 5);
            break;
        case YO_DX_CONTEST:
            memset(fieldValue, ' ', 20);
            sprintf(fieldValue, "%2d", qso->getCqZone());
            RFile::write(fieldValue, 2);
            
            memset(fieldValue, ' ', 20);
            if (strlen(qso->getInfo()) != 0)
                memcpy(fieldValue, qso->getInfo(), qso->getInfoLength());
            RFile::write(fieldValue, 2);
            break;
        case TEN_TEN_CONTEST:
            writeField(qso->getName(), qso->getNameLength(), 10);
            
            memset(fieldValue, ' ', 20);                                // 10-X Number
            if (qso->getTenTenNumber() != 0)
                sprintf(fieldValue, "%5d", qso->getTenTenNumber());
            RFile::write(fieldValue, 5);
            
            writeField(qso->getInfo(), qso->getInfoLength(), 8);        // QTH
            break;
        case NA_SPRINT_CONTEST:
            // Write serial number received.
            memset(fieldValue, ' ', 20);
            sprintf(fieldValue, "%4d", qso->getSerialNumber());
            RFile::write(fieldValue, 4);
            
            writeField(qso->getName(), qso->getNameLength(), 10);       // Name
            
            writeField(qso->getInfo(), qso->getInfoLength(), 8);        // Qth
            break;
        case SAC_CONTEST:
            // Write serial number received.
            memset(fieldValue, ' ', 20);
            memcpy(fieldValue, qso->getInfo(), qso->getInfoLength());
            RFile::write(fieldValue, 4);
            break;
        case TOEC_CONTEST:
            memset(fieldValue, ' ', 20);
            memcpy(fieldValue, qso->getInfo(), qso->getInfoLength());
            RFile::write(fieldValue, 5);
            break;
        case QRP_ARCI_CONTEST:
            memset(fieldValue, ' ', 20);
            sprintf(fieldValue, "%5d", qso->getARCInumber());
            RFile::write(fieldValue, 5);
            
            memset(fieldValue, ' ', 20);
            memcpy(fieldValue, qso->getInfo(), qso->getInfoLength());
            RFile::write(fieldValue, 8);
            break;
        case IOTA_CONTEST:
            memset(fieldValue, ' ', 20);
            sprintf(fieldValue, "%4d", qso->getSerialNumber());
            RFile::write(fieldValue, 4);
            
            memset(fieldValue, ' ', 20);
            memcpy(fieldValue, qso->getInfo(), qso->getInfoLength());
            RFile::write(fieldValue, 6);
            break;
        case INTERNET_SPRINT_CONTEST:
            memset(fieldValue, ' ', 20);
            sprintf(fieldValue, "%4d", qso->getSerialNumber());
            RFile::write(fieldValue, 4);
            
            memset(fieldValue, ' ', 20);                                // Name
            memcpy(fieldValue, qso->getName(), qso->getNameLength());
            RFile::write(fieldValue, 10);
            
            memset(fieldValue, ' ', 20);
            memcpy(fieldValue, qso->getInfo(), qso->getInfoLength());
            RFile::write(fieldValue, 8);
            break;
        case GENERAL_QSO:
            // Should include some common fields here (CQ Zone, DXCC Country, etc).
            break;
        case STEW_PERRY_CONTEST:
            writeField(qso->getInfo(), qso->getInfoLength(), 5);        // Grid square
            break;
        case HELVETIA_CONTEST:
        case WAG_CONTEST:
        case ARI_CONTEST:
            // Write serial number received.
            memset(fieldValue, ' ', 20);
            sprintf(fieldValue, "%4d", qso->getSerialNumber());
            RFile::write(fieldValue, 4);

            // Helvetia - Swiss Canton
            // WAG - DOK or country
            // ARI - Provence ID for Italian stations, country ID for others.
            memset(fieldValue, ' ', 20);
            memcpy(fieldValue, qso->getQth(), qso->getQthLength());
            RFile::write(fieldValue, 5);
            break;
        default:
            //Write generic information.
            writeField(qso->getInfo(), qso->getInfoLength(), 8);
            break;
    };

    dbheader.num_recs++;
    location++;
    
    return TRUE;
}

void RDbFile::writeField(const char *fieldInfo, const unsigned char dataLength,
                          const unsigned char writeLength)
{
    char fieldValue[20];
    
    memset(fieldValue, ' ', 20);
    memcpy(fieldValue, fieldInfo, dataLength);
    RFile::write(fieldValue, writeLength);
    return;
}

