// na9file.cpp
//
// Copyright (C) 1999  Robert Barron, KA5WSS
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include "na9file.h"
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

static char *contests[] = {NULL, NULL, "ARRLDX1", "CQWW", "WPX", "IARU", NULL};
#define MAX_NA9_CONTEST      IARU_CONTEST

static int modes[] = {CW_MODE, SSB_MODE};
#define NA9_MAX_MODE    1

static unsigned char categories[] = {CAT_SINGLE_OP, CAT_SINGLE_OP_UNLIMITED,
                            CAT_SINGLE_OP_SINGLE_BAND, CAT_SINGLE_OP, CAT_SINGLE_OP,
                            CAT_SINGLE_OP, CAT_QRP_SINGLE_OP, CAT_ROVER,
                            CAT_MULTI_SINGLE, CAT_MULTI_SINGLE, CAT_MULTI_TWO,
                            CAT_LIMITED_MULTI, CAT_MULTI_MULTI};
#define NA9_MAX_CATS    12                            

static long bands[] = {0, 2, 4, 8, 11, 15, 19, 22, 25, 30, 54, 77, 148, 225, 450, 928, 1300, 2450,
                        3500, 6000, 10500, 24250, 47200, 77000, 120000, 150000, 250000};
#define NA9_MAX_BAND    BAND_241GHZ                       

RNa9File::RNa9File(char *name)
{
    RNa9File::open(name);
}

boolean RNa9File::open(char *name, int mode)
{
    char tempStr[11];
    
#ifdef LINUX
    if (!RFile::open(name, mode))
        return FALSE;
#else
    // State ios::binary again just in case the user does not.
    if (!RFile::open(name, mode | ios::binary))
        return FALSE;
#endif

    if (!(mode & ios::out))          // Not an output file
    {
        /* Read in the header information and store in neutral format. */        
        RFile::read((void *)&na9header, sizeof(Na9Header));

        NaStringCopy(header.callsign, na9header.callsign, NA9_CALL_SIZE);
        NaStringCopy(header.name, na9header.name, 30);
        
        NaStringCopy(header.address, na9header.address, 30);
        NaStringCopy(header.city, na9header.city, 30);
        NaStringCopy(header.state, na9header.state, 2);
        NaStringCopy(header.zip, na9header.zip, 5);

        header.stationNumber = (unsigned char)na9header.stationNumber;
        
        NaStringCopy(header.club, na9header.clubName, 30);

        for (unsigned char c = 1; c <= MAX_NA9_CONTEST; c++)
            if (strncmp(na9header.contestName, contests[c], strlen(contests[c])) == 0)
                header.contest = c;

        if (na9header.category > NA9_MAX_CATS)
            return FALSE;
        else
            header.category = categories[na9header.category];
            
        // Store the CW messages
        NaStringCopy(header.CQ_CW_Message, na9header.cq_msg, 32);
        NaStringCopy(header.QRZ_CW_Message, na9header.qrz_msg, 32);
        NaStringCopy(header.Exchange_CW_Message, na9header.ex_msg, 32);
        NaStringCopy(header.Dupe_CW_Message, na9header.b4_msg, 32);

        // Don't bother with these fields unless they are specifically needed.
        switch (header.contest) {
            case ARRL_SS_CONTEST:
                header.ss.precedence = na9header.misc.ss.ss_prec;
                NaStringCopy(tempStr, na9header.misc.ss.ss_check, 2);
                header.ss.check = (unsigned char)atoi(tempStr);
                NaStringCopy(header.ss.section, na9header.location, 4);
                break;
            case FD_CONTEST:
                NaStringCopy(tempStr, na9header.misc.ex_info, 8);
                header.fd.category = tempStr[strlen(tempStr) - 1];
                tempStr[strlen(tempStr) - 1] = (char)NULL;
                header.fd.transmitters = (unsigned char)atoi(tempStr);
                break;
            case CQ_WW_CONTEST:
                NaStringCopy(tempStr, na9header.location, 4);
                header.zone = (unsigned char)atoi(tempStr);
            case ARRL_10M_CONTEST:
            case ARRL_160M_CONTEST:
                // Should save QTH somewhere!!!!
            case NA_QSO_PARTY_CONTEST:
                // Should save the nickname somewhere!!!!
            default:
                break;
        };
    }
    
    location = 0;

    return TRUE;
}

/* This routine takes as input the neutral format header from the input file */
/*  and stores it in this object.  It also writes out the header to its file */
/*  in its native format. */
boolean RNa9File::writeHeader(header_data newheader)
{
    // This capability is not yet implemented.
    newheader = newheader;
    
#if 0    
    RQsoFile::storeHeader(newheader);

    memset(&na9header, '\0', sizeof(Na9Header));

    RFile::write(&na9header, sizeof(Na9Header));
    
    headerWritten = TRUE;
    return TRUE;
#endif
    return FALSE;
}

boolean RNa9File::nextQso(qso_data *qso)
{
    Na9Qso na9_qso;
    char tempStr[11];

    qso->init();
    memset(&na9_qso, '\0', sizeof(Na9Qso));
    if (RFile::read((void *)&na9_qso, sizeof(Na9Qso)))
    {
        time_t baseTime = 315532800;                // Unix time at midnight 1/1/1980
            
        long freq = (na9_qso.freq >> 2) / 10000;     // Convert to freq in MHz
    
        NaStringCopy(tempStr, na9_qso.callsign, NA9_CALL_SIZE);
        qso->setCallsign(tempStr);
        qso->setTime((time_t)na9_qso.date + baseTime);
        
        qso->setBand(1);                                  // Determine the proper Amateur band.
        while (freq > bands[qso->getBand()])
            qso->setBand(qso->getBand() + 1);

        if (na9_qso.mode <= NA9_MAX_MODE)
            qso->setMode((char)modes[na9_qso.mode]);       // Convert to generic mode id.
        else
            return FALSE;                               // Not a supported mode!

        switch (header.contest) {
            case ARRL_SS_CONTEST:
                NaStringCopy(tempStr, na9_qso.info, 4);
                qso->setSerialNumber(tempStr);
                qso->setSSPrecedence(na9_qso.info[4]);
                NaStringCopy(tempStr, &na9_qso.info[9], 2);
                qso->setSSCheck(tempStr);
                NaStringCopy(tempStr, na9_qso.qth, 4);
                qso->setSection(tempStr);
                break;
            case CQ_WW_CONTEST:
                NaStringCopy(tempStr, na9_qso.info, 3);
                qso->setReceivedRST(tempStr);
                NaStringCopy(tempStr, na9_qso.qth, 4);
                qso->setCqZone(tempStr);
                break;
            case IARU_CONTEST:
                NaStringCopy(tempStr, na9_qso.info, 3);
                qso->setReceivedRST(tempStr);
                NaStringCopy(tempStr, na9_qso.qth, 4);
                if (isdigit(tempStr[0]))
                    qso->setItuZone(tempStr);
                else
                    qso->setInfo(tempStr);
                break;
            case CQ_WPX_CONTEST:
                NaStringCopy(tempStr, na9_qso.info, 3);
                qso->setReceivedRST(tempStr);
                NaStringCopy(tempStr, &na9_qso.info[3], 4);
                qso->setSerialNumber(tempStr);
                break;
            case FD_CONTEST:
                NaStringCopy(tempStr, na9_qso.info, 8);
                qso->setFdCategory(tempStr[strlen(tempStr) - 1]);       // A-E.
                tempStr[strlen(tempStr) - 1] = (char)NULL;
                qso->setFdTransmitters(tempStr);                       // Transmitter count.
                NaStringCopy(tempStr, na9_qso.qth, 4);         // ARRL section.
                qso->setSection(tempStr);
                break;
            case NA_QSO_PARTY_CONTEST:
                NaStringCopy(tempStr, na9_qso.info, 11);
                qso->setName(tempStr);
                NaStringCopy(tempStr, na9_qso.qth, 4);
                qso->setInfo(tempStr);
                break;
            case WAE_CONTEST:
//            qso->receivedRST = CTqso.rst;
//            strncpy(qso->info, CTqso.info, 5);
//            qso->qtc_num = CTqso.qtc_num;
                break;
            case NA_SPRINT_CONTEST:
                NaStringCopy(tempStr, na9_qso.info, 3);
                qso->setSerialNumber(tempStr);
                NaStringCopy(tempStr, &na9_qso.info[3], 8);
                qso->setName(tempStr);
                NaStringCopy(tempStr, na9_qso.qth, 4);
                qso->setInfo(tempStr);
                break;
            case ARRL_DX_CONTEST:
            case ARRL_10M_CONTEST:
            case ARRL_160M_CONTEST:
            default:
                NaStringCopy(tempStr, na9_qso.info, 3);
                qso->setReceivedRST(tempStr);
                NaStringCopy(tempStr, na9_qso.qth, 4);
                qso->setInfo(tempStr);
                break;
        };
        
        location++;
        return TRUE;
    }
    else
        return FALSE;
}

boolean RNa9File::writeQso(qso_data *qso)
{
    Na9Qso na9_qso;

    memset(&na9_qso, '\0', sizeof(Na9Qso));

    NaStringCopy(na9_qso.callsign, qso->getCallsign(), NA9_CALL_SIZE);

    return TRUE;
}

