// trfile.cpp
//
// Copyright (C) 1999  Robert Barron, KA5WSS
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include "trfile.h"
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include "ctyfile.h"

/* TR-specific formatted information. */
char *RTrFile::months[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul",
                          "Aug", "Sep", "Oct", "Nov", "Dec", NULL};
char *RTrFile::modes[] = {NULL, "CW ", "SSB", "   ", "   ", "FM ", NULL};
#define MAX_TR_MODE   FM_MODE
char *RTrFile::bands[MAX_BAND + 1] = {NULL, "160", " 80", " 40", " 30", " 20", " 17",
                         " 15", " 12", " 10", "  6", "  4", "  2", "222", "432",
                         "902", "1GH", "2GH", "3GH", "5GH", "10G", "24G", NULL,
                         NULL, NULL, NULL, NULL, "SAT", "PKT", "NOV", "LGT"};
                         
char *RTrFile::ContestNames[MAX_CONTEST + 1] = {NULL, "ARRL Sweepstakes", "ARRL DX Test",
                                "CQ WW Contest", "CQ WPX Contest", "IARU Radiosport",
                                "ARRL Ten Meter Contest", "ARRL 160 Contest",
                                "CQ 160 Contest", "Field Day",
                                "California QSO Party", "ARRL DX Test", "WAE Contest",
                                "VHF QSO Party", "General QSOs", "All Asian",
                                "North American QSO Party", "Grid Loc Contest",
                                "YO DX Contest", "KCJ Single Op Contest",
                                "Ten Ten QSO Party", "North American Sprint",
                                "Scandinavian Contest", "TOEC WW GRID CONTEST",
                                "Japan International DX Test",
                                "ARCI QSO PARTY", "RSGB IOTA Contest",
                                "Internet SprINT", " ", "STEW PERRY TOPBAND CHALLENGE",
                                "Helvetia Contest", "Worked All Germany Contest",
                                "ARI International DX Contest"};
                                
char *RTrFile::headings[MAX_CONTEST + 1] = {NULL, "   Rcvd  P Ck  Qth          Mults   Pts",
                             "   Sent Rcvd Power          Mults   Pts",
                             "   Sent Rcvd Zone           Mults   Pts",
                             "   Sent Rcvd Rcvd           Mults   Pts",
                             "   Sent Rcvd Zone  Qth      Mults   Pts",
                             "   Sent Rcvd Rcvd   Qth     Mults   Pts",
                             "   Sent Rcvd  Qth           Mults   Pts",
                             "   Sent Rcvd  Qth           Mults   Pts",
                             "   Class   Qth              Mults   Pts",
                             "   Rcvd   Qth               Mults   Pts",
                             "   Sent Rcvd  Qth           Mults   Pts",
                             "   Sent Rcvd Rcvd           Mults   Pts",
                             "   Sent Rcvd  Qth           Mults   Pts",
                             "   Sent Rcvd Name        Qth       Pts",
                             "   Sent Rcvd Age            Mults   Pts",
                             "   Name         Qth         Mults   Pts",
                             "   Name         Qth         Mults   Pts",
                             "   Sent Rcvd Zone  Qth      Mults   Pts",
                             "   Sent Rcvd  Qth           Mults   Pts",
                             "   Name         1010#  Qth  Mults   Pts",
                             "   Rcvd  Name         Qth   Mults   Pts",
                             "   Sent Rcvd Rcvd           Mults   Pts",
                             "   Sent Rcvd  Qth           Mults   Pts",
                             "   Sent Rcvd Zone           Mults   Pts",
                             "   Sent Rcvd Rcvd   Qth     Mults   Pts",
                             "   Sent Rcvd Rcvd   Qth     Mults   Pts",
                             "   Rcvd  Name         Qth   Mults   Pts",
                             "",
                             "   Sent Rcvd  Qth                  Pts",
                             "   Sent Rcvd Rcvd   Qth     Mults   Pts",
                             "   Sent Rcvd Rcvd   Qth     Mults   Pts",
                             "   Sent Rcvd Rcvd   Qth     Mults   Pts"};
                             
char *RTrFile::underlines[MAX_CONTEST + 1] = {NULL,
                              "   ----  - -- -----         -----   ---",
                              "   ---- ---- -----          -----   ---",
                              "   ---- ---- ----           -----   ---",
                              "   ---- ---- ----           -----   ---",
                              "   ---- ---- ---- -----     -----   ---",
                              "   ---- ---- ----  -----    -----   ---",
                              "   ---- ---- -----          -----   ---",
                              "   ---- ---- -----          -----   ---",
                              "   -----  -----             -----   ---",
                              "   ----  -----              -----   ---",
                              "   ---- ---- -----          -----   ---",
                              "   ---- ---- ----           -----   ---",
                              "   ---- ---- -----          -----   ---",
                              "   ---- ---- ----        ---       ---",
                              "   ---- ---- ---            -----   ---",
                              "   ----        -----        -----   ---",
                              "   ----        -----        -----   ---",
                              "   ---- ---- ---- -----     -----   ---",
                              "   ---- ---- -----          -----   ---",
                              "   ----         ----- ----- -----   ---",
                              "   ----  ----        -----  -----   ---",
                              "   ---- ---- ----           -----   ---",
                              "   ---- ---- -----          -----   ---",
                              "   ---- ---- ----           -----   ---",
                              "   ---- ---- ----  -----    -----   ---",
                              "   ---- ---- ----  -----    -----   ---",
                              "   ----  ----        -----  -----   ---",
                              "",
                              "   ---- ---- -----                ---",
                              "   ---- ---- ----  -----    -----   ---",
                              "   ---- ---- ----  -----    -----   ---",
                              "   ---- ---- ----  -----    -----   ---"};

RTrFile::RTrFile(char *filename)
{
    location = 0;
    RTrFile::open(filename);
}

boolean RTrFile::open(char *filename, int mode)
{
    char buffer[80];
    char temp[40];
    char *tempPtr;
    
    if (!RFile::open(filename, mode))
        return FALSE;

    if (!(mode & ios::out))
    {
        RFile::getString((char *)buffer, 80);
        strcpy(header.year, strtok(buffer, " \r"));

        if ((tempPtr = strrchr(&buffer[5], ' ')) != NULL)
        {
            *tempPtr = (char)NULL;
            strcpy(temp, &buffer[5]);
            tempPtr++;
            strcpy(header.callsign, strtok(tempPtr, " \r"));

            for (char testCount = 1; testCount <= MAX_CONTEST; testCount++)
                if ((ContestNames[testCount] != NULL)
                    && (strncmp(ContestNames[testCount], temp, strlen(ContestNames[testCount])) == 0))
                {
                    if (testCount == ARRL_DX_CONTEST_DX)
                    {
                        char tempName[40];
                        char prefix[8];
                        int zone;
                        RCtyFile ctyfile("cqww.cty");

                        ctyfile.GetInfo(header.callsign, tempName, prefix, &zone);
                        if (strcmp(prefix, "W") != 0)
                            header.contest = testCount;
                    }
                    else
                        header.contest = testCount;
                }

            if (header.contest == 0)
                return FALSE;
        }
        else
            return FALSE;
    }
    
    location = 0;

    return TRUE;
}

boolean RTrFile::nextQso(qso_data *qso)
{
    /* This code is not tested! */
    
    char buffer[MAX_TR_LINE_SIZE];
    unsigned char count;
    char *tempPtr;

    qso->init();

    // Keep getting lines until a valid log entry is found.
    buffer[9] = (char)NULL;
    while (((buffer[9] != '-') || (buffer[19] != ':')) && !RFile::eof())
    {
        memset(buffer, '\0', MAX_TR_LINE_SIZE);
        RFile::getString((char *)buffer, MAX_TR_LINE_SIZE);
        if (RFile::eof())
            return FALSE;
    }

    // Copy callsign of station worked to log_data struct.
    qso->setCallsign((char *)&buffer[29], MAX_TR_CALL_SIZE);

    // Convert the verbose time to time_t format.
    {
        int year, month, day, hour, minutes;
        buffer[9] = buffer[13] = buffer[16] = buffer[19] = buffer[22] = (char)NULL;

        day = atoi(&buffer[7]);
        for (count = 0; months[count] != NULL
                    && stricmp(months[count],&buffer[10]) != 0; count++);
        if (months[count] != NULL)                    
            month = count;
        year = atoi(&buffer[14]);
        hour = atoi(&buffer[17]);
        minutes = atoi(&buffer[20]);
        
        qso->setTime(year, month, day, hour, minutes);
    }
            
    // Store mode into qso_data
    buffer[6] = (char)NULL;
    for (count = 1; count <= MAX_TR_MODE; count++)
         if (modes[count] && (strcmp(modes[count], &buffer[3]) == 0))
             qso->setMode(count);

    // Store band into qso_data
    buffer[3] = (char)NULL;
    for (count = 1; count <= MAX_BAND; count++)
        if ((bands[count] != NULL) && (strcmp(bands[count], &buffer[0]) == 0))
            qso->setBand(count);

    // Store QSO point value
    qso->setQsoPoints(&buffer[76]);
    
    // Store QSO multiplier count (may be more than one)
    //  Don't do this if "*DUPE*" is in the record!
    if ((buffer[68] != '*') && (buffer[68] != ' '))
    {
        char *temp;
        
        buffer[76] = (char)NULL;
        if ((temp = strtok(&buffer[68], " ")) != NULL) // Store 1st mult
            qso->addMult(temp);
        if ((temp = strtok(NULL, " ")) != NULL)        // Is there a 2nd mult?
            qso->addMult(temp);
    }

    switch (header.contest) {
        case ARRL_SS_CONTEST:
            qso->setSerialNumber(strtok(&buffer[43], " "));
            tempPtr = strtok(NULL, " ");
            qso->setSSPrecedence(*tempPtr);
            qso->setSSCheck(strtok(NULL, " "));
            qso->setSection(strtok(NULL, " "));
            break;
        case CQ_WW_CONTEST:
            qso->setSentRST(strtok(&buffer[44], " "));
            qso->setReceivedRST(strtok(NULL, " "));
            qso->setCqZone(strtok(NULL, " "));
            break;
        case CQ_WPX_CONTEST:
            qso->setSentRST(strtok(&buffer[44], " "));
            qso->setReceivedRST(strtok(NULL, " "));
            qso->setSerialNumber(strtok(NULL, " "));
            break;
        case IARU_CONTEST:
            qso->setSentRST(strtok(&buffer[44], " "));
            qso->setReceivedRST(strtok(NULL, " "));
            tempPtr = strtok(NULL, " ");
            if (isdigit(tempPtr[0]))
                qso->setItuZone(tempPtr);
            else
                qso->setInfo(tempPtr);
            break;
        case FD_CONTEST:
            qso->setFdCategory(buffer[46]);
            buffer[46] = (char)NULL;
            qso->setFdTransmitters(&buffer[43]);
            qso->setSection(strtok(&buffer[47], " "));
            break;
        case DXPEDITION_MODE:
            qso->setSentRST(strtok(&buffer[44], " "));
            qso->setReceivedRST(strtok(NULL, " "));
            if (buffer[53] != ' ')
                qso->setName(strtok(&buffer[53], " "));
            if (buffer[65] != ' ')
                qso->setInfo(strtok(&buffer[65], " "));
            break;
        case CAL_QP_CONTEST:
            qso->setSerialNumber(strtok(&buffer[43], " "));
            qso->setInfo(strtok(NULL, " "));
            break;
        case ALL_ASIAN_CONTEST:
            qso->setSentRST(strtok(&buffer[44], " "));
            qso->setReceivedRST(strtok(NULL, " "));
            qso->setAge(strtok(NULL, " "));
            break;
        case NA_QSO_PARTY_CONTEST:
            qso->setName(strtok(&buffer[43], " "));
            qso->setInfo(strtok(NULL, " "));
            break;
        case GRIDLOC_CONTEST:
            qso->setName(strtok(&buffer[43], " "));
            // If user did not give a valid grid then leave it blank!
            if (buffer[55] != ' ')
                qso->setInfo(strtok(NULL, " "));
            break;
        case YO_DX_CONTEST:
            qso->setSentRST(strtok(&buffer[44], " "));
            qso->setReceivedRST(strtok(NULL, " "));
            qso->setCqZone(strtok(NULL, " "));
            if (buffer[58] != ' ')
                // This is a YO QSO.  Also includes county indicator.
                qso->setInfo(strtok(&buffer[58], " "));
            break;
        case TEN_TEN_CONTEST:
            qso->setName(strtok(&buffer[43], " "));
            // If a 10-10 number was exchanged copy it.
            if (isdigit(buffer[59]))
                qso->setTenTenNumber(strtok(&buffer[56], " "));
            qso->setInfo(strtok(&buffer[62], " "));
            break;
        case NA_SPRINT_CONTEST:
            qso->setSerialNumber(strtok(&buffer[43], " "));
            qso->setName(strtok(NULL, " "));
            qso->setInfo(strtok(NULL, " "));
            break;
        case QRP_ARCI_CONTEST:
            qso->setSentRST(strtok(&buffer[44], " "));
            qso->setReceivedRST(strtok(NULL, " "));
            qso->setARCInumber(strtok(NULL, " "));
            tempPtr = strtok(NULL, " ");
            if (!isdigit(tempPtr[0]))
                qso->setInfo(tempPtr);
            break;
        case IOTA_CONTEST:
            qso->setSentRST(strtok(&buffer[44], " "));
            qso->setReceivedRST(strtok(NULL, " "));
            qso->setSerialNumber(strtok(NULL, " "));
            if (buffer[59] != ' ')
                qso->setInfo(strtok(&buffer[59], " "));
            break;
        case INTERNET_SPRINT_CONTEST:
            qso->setSerialNumber(strtok(&buffer[43], " "));
            qso->setName(strtok(NULL, " "));
            qso->setInfo(strtok(NULL, " "));
            break;
        case STEW_PERRY_CONTEST:
            qso->setSentRST(strtok(&buffer[44], " "));
            // Log files may not contain a received RST report.  Handle that
            // special case.
            tempPtr = strtok(NULL, " ");
            if (isdigit(tempPtr[0]))
            {
                qso->setReceivedRST(tempPtr);
                qso->setInfo(strtok(NULL, " "));
            }
            else
                qso->setInfo(tempPtr);
            break;
        case HELVETIA_CONTEST:
            qso->setSentRST(strtok(&buffer[44], " "));
            qso->setReceivedRST(strtok(NULL, " "));
            qso->setSerialNumber(strtok(NULL, " "));
            if (buffer[59] != ' ')
                // Contacts with Swiss stations include canton identifier
                qso->setQth(strtok(NULL, " "));
            break;
        case WAG_CONTEST:
            qso->setSentRST(strtok(&buffer[44], " "));
            qso->setReceivedRST(strtok(NULL, " "));
            if (buffer[56] != ' ')
                // This is a non-German station.
                qso->setSerialNumber(strtok(NULL, " "));
            qso->setQth(strtok(NULL, " "));
            break;
        case ARI_CONTEST:
            qso->setSentRST(strtok(&buffer[44], " "));
            qso->setReceivedRST(strtok(NULL, " "));
            tempPtr = strtok(NULL, " ");
            if (isdigit(tempPtr[0]))
            {
                // This is a non-Italian station.  They send
                // a serial number.  TR also inputs a country ID.
                qso->setSerialNumber(tempPtr);
                qso->setQth(strtok(NULL, " "));
            }
            else
                // Italian stations send letter province ID
                qso->setQth(tempPtr);
            break;
        default:
            qso->setSentRST(strtok(&buffer[44], " "));
            qso->setReceivedRST(strtok(NULL, " "));
            qso->setInfo(strtok(NULL, " "));
            break;
    };

    location++;
    return TRUE;
}

boolean RTrFile::writeHeader(header_data newheader)
{
    /* Test to see if the contest is supported by this TR write code. */
    /*  Return false if it is not supported. */
    if (ContestNames[newheader.contest] == NULL)
        return FALSE;
        
    /* TR log files do not have headers per se. Save */
    /*  header information for the page headers.     */
    RQsoFile::storeHeader(newheader);
    return TRUE;
}

boolean RTrFile::writePageHeader(header_data header, qso_data *qso)
{
    char buffer[MAX_TR_LINE_SIZE];

    /* Unless this is the first page print a page feed character. */
    if (location != 1)
        RFile::write("\f",1);

    /* Used to be a call to localtime().  Now trying gmtime(). */
    if (!isdigit(header.year[0]))
    {
        time_t tempTime;
        tempTime = qso->getTime();
        strftime(header.year, 5, "%Y", gmtime(&tempTime));
    }

    /* Wouldn't a single NULL at buffer[0] do just as well?? */
    memset(buffer, '\0', MAX_TR_LINE_SIZE);
    strncpy(buffer, header.year, 4);
    strcat(buffer, " ");
    strcat(buffer, ContestNames[header.contest]);
    strcat(buffer, " ");
    strncat(buffer, header.callsign, strlen(header.callsign));
    strcat(buffer, "\r\n\r\n\r\n");
    RFile::write(buffer,strlen(buffer));

    strcpy(buffer, " Band    Date    Time  QSO#  Call worked");
    strcat(buffer, headings[header.contest]);
    RFile::write(buffer,strlen(buffer));
    
    strcpy(buffer, "\r\n ----    ----    ----  ----  -----------");
    strcat(buffer, underlines[header.contest]);
    strcat(buffer, "\r\n");
    RFile::write(buffer,strlen(buffer));

    return TRUE;
}

boolean RTrFile::writeQso(qso_data *qso)
{
    char buffer[MAX_TR_LINE_SIZE];
    char buffer2[MAX_TR_LINE_SIZE];
    struct tm *dosTime;
    time_t tempTime;
    
    if (((++location - 1) % 50) == 0)
        writePageHeader(header,qso);            /* Print header at page tops. */
    else if (((location - 1) % 10) == 0)
        RFile::write("\r\n",2);                 /* Skip line every 10 QSOs */

    memset(buffer, '\0', MAX_TR_LINE_SIZE);
    strcpy(buffer, bands[qso->getBand()]);
    if ((qso->getMode() <= MAX_TR_MODE) && modes[qso->getMode()])
        strcat(buffer, modes[qso->getMode()]);
    else
        // Error: Unknown mode, just output blanks.
        strcat(buffer, "    ");
    strcat(buffer, " ");

    tempTime = qso->getTime();
    dosTime = gmtime(&tempTime);
    strftime(buffer2, MAX_TR_LINE_SIZE, "%d-%b-%y %H:%M ", dosTime);
    strcat(buffer, buffer2);
    
    if (location < 10000)
        sprintf(buffer2, "%4d  ", location);
    else
        sprintf(buffer2, "%5d ", location);
    strcat(buffer, buffer2);
    sprintf(buffer2, "%-14s", qso->getCallsign());
    strcat(buffer, buffer2);

    switch (header.contest) {
        case ARRL_SS_CONTEST:
        case FD_CONTEST:
        case NA_QSO_PARTY_CONTEST:
        case GRIDLOC_CONTEST:
        case TEN_TEN_CONTEST:
        case NA_SPRINT_CONTEST:
        case CAL_QP_CONTEST:
        case INTERNET_SPRINT_CONTEST:
            break;
        default:
            sprintf(buffer2, " %-3d ", qso->getSentRST());
            strcat(buffer, buffer2);
            break;
    }
    
    /* Now extract contest specific information. */
    switch (header.contest) {
        case ARRL_SS_CONTEST:
            sprintf(buffer2, "%4d  ", qso->getSerialNumber());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%c ", qso->getSSPrecedence());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%2d ", qso->getSSCheck());
            strcat(buffer, buffer2);
            /* We assume QSO is worth 2 points. */
            sprintf(buffer2, "%-5s         ", qso->getSection());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case CQ_WW_CONTEST:
            sprintf(buffer2, " %-3d  %02d            ", qso->getReceivedRST(),
                    qso->getCqZone());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case CQ_WPX_CONTEST:
            sprintf(buffer2, " %-3d %4d           ", qso->getReceivedRST(), qso->getSerialNumber());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case FD_CONTEST:
            sprintf(buffer2, "%3d%c     %-3s             ", qso->getFdTransmitters(),
                    qso->getFdCategory(), qso->getSection());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case IARU_CONTEST:
            if (qso->getItuZone() > 0)
                /* This is a normal QSO */
                sprintf(buffer2, " %-3d  %02d            ", qso->getReceivedRST(),
                        qso->getItuZone());
            else
                /* This is a QSO with an IARU HQ station */
                sprintf(buffer2, " %-3d     %5s      ", qso->getReceivedRST(),
                        qso->getInfo());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case ARRL_160M_CONTEST:
        case ARRL_DX_CONTEST_DX:
        case CQ_160M_CONTEST:
        case KCJ_CONTEST:
            sprintf(buffer2, " %-3d %-5s          ", qso->getReceivedRST(), qso->getInfo());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case ARRL_10M_CONTEST:
            if (qso->isInfoNumeric())
                sprintf(buffer2, " %-3d %4s           ", qso->getReceivedRST(), qso->getInfo());
            else
                sprintf(buffer2, " %-3d       %-5s    ", qso->getReceivedRST(), qso->getInfo());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case JA_INTL_DX_CONTEST:
            sprintf(buffer2, " %-3d  %-3s           ", qso->getReceivedRST(), qso->getInfo());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case DXPEDITION_MODE:
            sprintf(buffer2, " %-3d %-11s %-11s %d", qso->getReceivedRST(), qso->getName(),
                    qso->getInfo(), qso->getQsoPoints());
            break;
        case CAL_QP_CONTEST:
            sprintf(buffer2, " %3d  %-4s               ", qso->getSerialNumber(), qso->getInfo());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case ALL_ASIAN_CONTEST:
            sprintf(buffer2, " %-3d  %02d            ", qso->getReceivedRST(), qso->getAge());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case NA_QSO_PARTY_CONTEST:
        case GRIDLOC_CONTEST:
            sprintf(buffer2, "%-8s    %-4s         ", qso->getName(), qso->getInfo());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case YO_DX_CONTEST:
            sprintf(buffer2, " %-3d  %02d  %-9s ", qso->getReceivedRST(),
                    qso->getCqZone(), qso->getInfo());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case TEN_TEN_CONTEST:
            if (qso->getTenTenNumber() == 0)
                sprintf(buffer2, "%-8s           %-5s", qso->getName(), qso->getInfo());
            else
                sprintf(buffer2, "%-8s     %5d %-5s", qso->getName(), qso->getTenTenNumber(), qso->getInfo());
            break;
        case NA_SPRINT_CONTEST:
            sprintf(buffer2, "%4d  %-10s  %-5s  ",
                    qso->getSerialNumber(), qso->getName(), qso->getInfo());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case QRP_ARCI_CONTEST:
            if (qso->getARCInumber() >= 10000)
                sprintf(buffer2, " %-3d %5d %-6s     ", qso->getReceivedRST(),
                        qso->getARCInumber(), qso->getInfo());
            else
                sprintf(buffer2, " %-3d %4d  %-6s   ", qso->getReceivedRST(),
                        qso->getARCInumber(), qso->getInfo());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case IOTA_CONTEST:
            sprintf(buffer2, " %-3d %4d  %-8s ", qso->getReceivedRST(), qso->getSerialNumber(),
                    qso->getInfo());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s%2d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case INTERNET_SPRINT_CONTEST:
            sprintf(buffer2, "%4d  %-11s %-6s ", qso->getSerialNumber(), qso->getName(),
                   qso->getInfo());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s%2d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case STEW_PERRY_CONTEST:
            // A received RST report is not guaranteed
            if (qso->getReceivedRST() != 0)
                sprintf(buffer2, " %-3d %4s           ", qso->getReceivedRST(), qso->getInfo());
            else
                sprintf(buffer2, "     %4s           ", qso->getInfo());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
        case HELVETIA_CONTEST:
            sprintf(buffer2, " %-3d %4d %3s       %-8s %d", qso->getReceivedRST(),
                            qso->getSerialNumber(), qso->getQth(),
                            qso->getMultipliers(), qso->getQsoPoints());
            break;
        case WAG_CONTEST:
            if (qso->getSerialNumber() == 0)
                sprintf(buffer2, " %-3d       %-5s    %-8s %d", qso->getReceivedRST(),
                            qso->getQth(), qso->getMultipliers(), qso->getQsoPoints());
            else
                sprintf(buffer2, " %-3d %4d  %-5s    %-8s %d", qso->getReceivedRST(),
                            qso->getSerialNumber(), qso->getQth(),
                            qso->getMultipliers(), qso->getQsoPoints());

            break;
        case ARI_CONTEST:
            if (qso->getSerialNumber() == 0)
                // Italian stations do not send a serial number
                sprintf(buffer2, " %-3d       %-5s    %-8s%2d", qso->getReceivedRST(),
                            qso->getQth(), qso->getMultipliers(), qso->getQsoPoints());
            else
                sprintf(buffer2, " %-3d %4d  %-5s    %-8s%2d", qso->getReceivedRST(),
                            qso->getSerialNumber(), qso->getQth(), qso->getMultipliers(),
                            qso->getQsoPoints());
            break;
        default:
            sprintf(buffer2, " %-3d %4s           ", qso->getReceivedRST(), qso->getInfo());
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-8s %d", qso->getMultipliers(), qso->getQsoPoints());
            break;
    };
    strcat(buffer, buffer2);
    strcat(buffer, "\r\n");
    
    RFile::write(buffer,strlen(buffer));
    
    return TRUE;
}
