/*
 * twclock:  A world clock implemented with Motif widgets
 * Copyright (C) 1997 Ted Williams WA0EIR (ted@bluestone.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge. MA 02139, USA.
 * See the COPYING file in this directory
 *
 * Versions: 1.2 - OCT 1998
 */

/*
 * CALLBACKS FOR TWCLOCK 
 */

#include "twclock.h"

/*
 * CATCH INPUT EVENTS
 *
 * This catches all input events in the GUI.  If it was a
 * button 3 press, popup the menu, else go away.
 */
void input_event (Widget w, XtPointer client_data, XEvent *event, Boolean *cont)
{
   Widget popup_menu = (Widget) client_data;
   if(event->type == ButtonPress && event->xbutton.button == Button3)
   {
      XmMenuPosition(popup_menu, &(event->xbutton));
      XtManageChild(popup_menu); 
   }
}


/*
 * POPUP MENU CALLBACK
 *
 * All menu button presses come here.  The switch checks for
 * which menu button was pressed and either sets the TZ variable 
 * Local, GMT or, in the case of the Others: button, pops
 * up the file selection box dialog.  The tzset() forces the
 * new value in TZ to be recognized.  The clock will adjust
 * to the new time on the next update.
 */
void popup_cb(Widget w, XtPointer client_data, XtPointer cbs)
{
   int set_zone = (int) client_data;
   Widget zone_fsb;

   switch (set_zone)
   {
      case LOCAL_BTN:
         putenv("TZ=localtime");
         tzset();
         doit = 1; /* force the date label to be updated next time */
         break;

      case GMT_BTN:
         putenv("TZ=GMT");
         tzset();
         doit = 1; /* force the date label to be updated next time */
         break;

      case OTHERS_BTN:
         zone_fsb = XmCreateFileSelectionDialog(clock_shell, "zone_fsb",
            (ArgList) NULL, 0);

         XtVaSetValues (XtParent(zone_fsb),
            XmNmwmDecorations, MWM_DECOR_BORDER,
            NULL);

         XtVaSetValues (zone_fsb,
            XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
            NULL);

         XtUnmanageChild (XmFileSelectionBoxGetChild (zone_fsb,
            XmDIALOG_FILTER_TEXT));

         XtUnmanageChild (XmFileSelectionBoxGetChild (zone_fsb,
            XmDIALOG_FILTER_LABEL));

         XtUnmanageChild (XmFileSelectionBoxGetChild (zone_fsb,
            XmDIALOG_HELP_BUTTON));

         XtAddCallback (zone_fsb, XmNokCallback, fsbOkCB, (XtPointer) NULL);
         XtAddCallback (zone_fsb, XmNcancelCallback, fsbCancelCB,
            (XtPointer) NULL);

         XtManageChild (zone_fsb);
         XtPopup (XtParent(zone_fsb), XtGrabNone);
         break;

      default:
         break;
   }
}
 

/*
 * GETTIME FUNCTION
 * A call to this function is hardcoded near the end of widget
 * creation.  At the end of this function, it registers a timeout
 * that forces itself to be called again in one second.  We're off!
 *
 * This function calls localtime(), which bases time on TZ.  The
 * time values are then set into the widgets.  No need to put the
 * same string in the date label 86400 times a day.  So, if the day
 * of the week value has changed, change the label string.  Or, if
 * doit is true, we update the label.  popup_cb() sets doit when
 * the time zone is changed, which requires updating the label.
 */
void  gettime (struct tag1 *ptr)
{

   time_t current;
   struct tm *timestruct;
   char  datestr[40];
   static int oldday;
   XmString labelstr;
   XtIntervalId id;

   unsigned long delta = 1000;

   time (&current);
  	timestruct = localtime (&current);	/* get localtime as per TZ */

   XtVaSetValues (ptr->hr_scale,  XmNvalue, timestruct->tm_hour, NULL);
   XtVaSetValues (ptr->min_scale, XmNvalue, timestruct->tm_min,  NULL);
   XtVaSetValues (ptr->sec_scale, XmNvalue, timestruct->tm_sec,  NULL);

   if ((timestruct->tm_mday != oldday) | doit)
   {
      strftime (datestr, sizeof(datestr), "%e %B %Y %Z", timestruct);
      labelstr = XmStringCreateLocalized (datestr);

      XtVaSetValues (ptr->date_label,
         XmNlabelString, labelstr,
         NULL);

      XmStringFree (labelstr);
      oldday = timestruct->tm_mday;
      doit = 0;
   }

   id=XtAppAddTimeOut(ptr->app, delta,
      (XtTimerCallbackProc)gettime, (XtPointer)ptr);
}


/*
 * FSB OK CALLBACK
 * This function gets data from the fsb and builds a string of
 * the form TZ=pathname.  If the pathname ends with a "/",
 * then you didn't select a city in the fsb, so exit and leave
 * TZ alone.  Otherwise, set TZ to the new timezone and set doit.
 * This forces gettime() to use the new TZ and update the date label.
 */
void fsbOkCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   XmFileSelectionBoxCallbackStruct *fsbcbs =
      (XmFileSelectionBoxCallbackStruct *) cbs;

   int  len;
   char *path;
   char newpath[80];

   XmStringGetLtoR (fsbcbs->value, XmFONTLIST_DEFAULT_TAG, &path);
   strcpy (newpath, "TZ=");
   strcat (newpath, path);
   len = strlen (path);
   if (path[len-1] != '/')   /* If the path ends with /, then do nothing */
   {
      putenv (newpath);
      tzset();
      doit = 1;              /* force the date label to be updated next time */
   }
   XtFree (path);
   XtDestroyWidget(XtParent(w));
}


/*
 * FSB CANCEL CALLBACK
 * Cancel button was pressed, so destroy the fsb and leave
 * things alone.
 */
void fsbCancelCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   XtDestroyWidget(XtParent(w));
}

