/*
 * afpipe.c
 *
 * This module supports pipe channels
 *
 * $Id: afpipe.c,v 1.7 1996/09/04 04:11:10 ron Exp $
 *
 */
#include <unistd.h>
#include <stdio.h>
#include "afos.h"

int pipe_init ( AF_CHANNEL *chan , void (*read_callback)()) ;
int pipe_write ( AF_CHANNEL *chan, unsigned char *dat, int size ) ;
int pipe_getchar ( AF_CHANNEL *chan ) ;

typedef struct _PIPE_STRUCT_ {
	char pipe_name[AFCHAN_NAME_SIZE];
	char pipe_options[128];
	char *argv[20] ;
	int pid ;
	} PIPE_STRUCT ;

/*
	parse_pipe parse out the 'pipe' line in the config file
*/
int
parse_pipe( char *line )
{
	static char args[8][80] ;
	AF_CHANNEL *chan ;
	PIPE_STRUCT *mystuff ;
	int count ;

	count = sscanf ( line,  "%[^ 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]"
							"%[^\n]",
		args[0], args[1], args[2], args[3] )  ;

	if ( count < 3 )
	{
		fprintf ( stderr, "\r\nAFOSD: Unable to parse the following line:\n%s\n"
			"Insufficient parameters for process definition\n"
			"Usage:\n"
			"pipe <CHANNEL NAME> <APPLICATION NAME> [OPTIONS]\n", line);
		exit (-1);
	}

	if (!verify_unique_portname( args[1] ) )
	{
		fprintf(stderr,"\r\nERROR in port configuration file on LINE:\n%s\n"
					   "Port '%s' already defined\n",
						line, args[1] ) ;
		exit (-1);
	}

	chan = (AF_CHANNEL *)malloc( sizeof(*chan) ) ;
	if (!chan)
	{
		fprintf(stderr, "Unable to allocate new channel definition space\n");
		exit (-1);
	}
	
	/* Put new channel on the channel list */
	chan->next = channel_list ;
	channel_list = chan ;
	strncpy ( chan->name, args[1], AFCHAN_NAME_SIZE ) ;

	/* Allocate our private data structure and fill it out with our
	   configuration info */
	mystuff = (PIPE_STRUCT *)malloc( sizeof(*mystuff) ) ;
	if (!mystuff)
	{
		fprintf(stderr, "Unable to allocate new channel definition space\n");
		exit (-1);
	}
	strncpy ( mystuff->pipe_name, args[2], AFCHAN_NAME_SIZE ) ;
	if ( count > 3 )
	{
		strncpy ( mystuff->pipe_options, args[3], 128 ) ;
		mystuff->argv[0] = mystuff->pipe_name ;
		argify (mystuff->pipe_options, mystuff->argv);
	}
	chan->private = mystuff ;

	/* Fill out the rest of the channel info */
	chan->fd = -1 ;
	chan->init = pipe_init ;
	chan->chanwrite = pipe_write ;
	chan->changetchar = pipe_getchar ;
#if 1
	printf( "config line: '%s'\n"
			"channel name: '%s'\n"
			"pipe application name: '%s'\n"
			"pipe application options:'%s'\n",
			line, chan->name, mystuff->pipe_name, mystuff->pipe_options);
#endif
}


/*
	pipe_init initializes a serial port
*/
int
pipe_init ( AF_CHANNEL *chan, void (*read_callback)() )
{
	int pid ;
	int stat ;
	int i ;
	int pipes[2] ;
	PIPE_STRUCT *mystuff = chan->private ;

	if (read_callback) {
		fprintf(stderr,"Pipe channel does not (yet) support read\n");
		exit ( -1 );
	}

	/*
		launch process and establish pipe
	*/
	pipe (pipes) ;
	if ( (mystuff->pid = fork()) > 0 )
	{
		/* parent process */
		chan->fd = pipes[1] ;
	}
	else
	{
		dup2 ( pipes[0], 0 ) ;
		close (pipes[1]);
#if 0
		fprintf(stderr,"afproc: calling '%s' with arguments: ",
			mystuff->pipe_name ) ;
		for ( i = 0 ; mystuff->argv[i] ; i++ )
			fprintf(stderr,"'%s' ",mystuff->argv[i]);
		fprintf(stderr,"<end>\n");
#endif
		execvp( mystuff->pipe_name, mystuff->argv ) ;
		/* Shouldn't get here */
		exit(0);
	}
}

/*
	pipe_write writes a message out the serial port
*/
int
pipe_write ( AF_CHANNEL *chan, unsigned char *dat, int size )
{
	int ret ; 

	/* Check to see if previously initialized */
	if ( chan->fd == -1 )
	{
		/* Not initialized, cope */
		return 0 ;
	}

	chan->write_count++ ;
	chan->bytes_written += size ;
	ret = write ( chan->fd, dat, size ) ;
	if ( ret < 0 ) 
		chan->fd = -1 ;
	return  ret ;
}

/*
	pipe_getchar reads one character from the process ?.  This call is
	non-blocking and may return a negative value if an error occurs
*/
int
pipe_getchar ( AF_CHANNEL *chan )
{
	int i, result;

	i = 0 ;
	result = read ( chan->fd, &i, 1 ) ;

	if ( result <= 0 )
		return -1 ;

	chan->bytes_read++ ;
	return 0xff & i ;
}
