/*  XNECVIEW - a program for visualizing NEC2 structure and gain files
 *
 *  Copyright (C) 1998-1999, P.T. de Boer -- pa3fwm@amsat.org
 *
 *  Distributed on the conditions of the GPL: see the files README and
 *  COPYING, which accompany this source file.
 *
 *  This module parses NEC2's output, extracting the gain as a function of
 *  direction from it.
 */


#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "xnecview.h"


double maxdb;


int read_nec_output(FILE *f)          /* tries to read NEC output data from f; returns 1 if not succesful */
{
   char s[200];
   double phi,theta,db;
   int i;
   int end=0;

   do fgets(s,200,f);  while (!feof(f) && !strstr(s,"RADIATION PATTERNS"));
   if (feof(f)) return 1;

   /* Some versions of NEC2 output extraneous data here before the actual data
    * and column labels (e.g. RANGE = and EXP (-JKR) values ).  Discard until
    * the true bottom of the column labels (look for DB) */
   do fgets(s,200,f);  while (!feof(f) && !strstr(s,"DB"));
   if (feof(f)) return 1;

   fgets(s,200,f);
   removecommas(s);
   if (sscanf(s,"%lg%lg%*g%*g%lg",&theta,&phi,&db)!=3) return 1;
   gphi[0]=phi;
   numphi=1;
   maxdb=db;
   do {
      if (numtheta>=MAXTHETAS) {
         fprintf(stderr,"Output file contains data at more theta values than allowed (%i)\n",MAXTHETAS);
         exit(1);
      }
      gain[numtheta][numphi-1]=db;
      gtheta[numtheta++]=theta*M_PI/180;
      if (db>maxdb) maxdb=db;
      fgets(s,200,f);
      removecommas(s);
      if (sscanf(s,"%lg%lg%*g%*g%lg",&theta,&phi,&db)!=3) { end=1; break; }
   } while (phi==gphi[0]);
   gphi[0]*=M_PI/180;
   if (end) return 0;
   i=0;
   while (1) {
      if (i==0) {
         if (numphi>=MAXPHIS) {
            fprintf(stderr,"Output file contains data at more phi values than allowed (%i)\n",MAXPHIS);
            exit(1);
         }
         gphi[numphi++]=phi*M_PI/180;
      }
      gain[i][numphi-1]=db;
      if (db>maxdb) maxdb=db;
      if (++i==numtheta) i=0;
      fgets(s,200,f);
      removecommas(s);
      if (sscanf(s,"%lg%lg%*g%*g%lg",&theta,&phi,&db)!=3) break;
   }
   printf("Maximum gain = %g dBi\n",maxdb);
   return 0;
}


void process_nec_output()		/* transform the gain data array into an array of points in 3D space */
{
   int i,j;
   double r=0;

   if (extr==0) extr=1;
   else extr*=GAINSIZE;
   for (i=0;i<numtheta;i++)
      for (j=0;j<numphi;j++) {
         switch (gainscale) {
            case GSlin: r=pow(10.0,(gain[i][j]-maxdb)/10); break;        /* linear */
            case GSarrl: r=exp(0.116534*(gain[i][j]-maxdb)/2); break;  /* arrl */
            case GSlog: r=gain[i][j]-maxdb; if (r<-40) r=0; else r=r/40+1; break;  /* log */
         }
         r*=extr;
         gpo[i][j].x = cos(gphi[j])*sin(gtheta[i])*r;
         gpo[i][j].y = sin(gphi[j])*sin(gtheta[i])*r;
         gpo[i][j].z = cos(gtheta[i])*r;
      }
}

