// ---------------------------------------------------------------------------
// - Instance.cpp                                                            -
// - aleph engine - aleph instance class implementation                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Method.hpp"
#include "Closure.hpp"
#include "Instance.hpp"
#include "Exception.hpp"

namespace aleph {
  const String& THIS       = "this";
  const String& META       = "meta";
  const String& SUPER      = "super";
  const String& INITIALIZE = "initialize";

  // create a new instance with a class and some arguments

  Instance::Instance (Interp* interp, Nameset* nset, Cons* args, Class* cls) {
    // save the meta class
    if (cls == nilp)
      throw Exception ("meta-error", "invlid nil meta class with instance");
    p_class = cls;
    Object::iref (cls);
    p_super = nilp;
    d_const = false;
    // create the instance local set
    p_iset = new Localset;
    Object::iref (p_iset);
    // bind the this symbol
    p_iset->symcst (THIS, this);
    // bind the default symbols
    const Strvec& mdata = cls->getmdata ();
    if (mdata.length () != 0) {
      long len  = mdata.length ();
      for (long i = 0; i < len; i++)
	p_iset->symdef (mdata.get (i), (Object*) nilp);
    }
    // find the initial form
    Object* obj  = p_class->find (INITIALIZE);
    Object* form = (obj == nilp) ? nilp : obj->eval (interp, nset);
    if (form != nilp) {
      try {
	p_iset->setparent (nset);
	Object::cref (form->apply (interp, p_iset, args));
	p_iset->setparent ((Nameset*) nilp);
      } catch (...) {
	Object::dref (p_iset);
	Object::dref (p_class);
	Object::dref (p_super);
	throw;
      }
    }
  }

  // destroy this instance

  Instance::~Instance (void) {
    Object::dref (p_iset);
    Object::dref (p_class);
    Object::dref (p_super);
  }

  // return the class name

  String Instance::repr (void) const {
    return "Instance";
  }

  // set the instance super value

  Object* Instance::setsuper (Object* object, const bool flag) {
    if (d_const == true)
      throw Exception ("const-error", "const violation with super member");
    Object::iref (object);
    Object::dref (p_super);
    p_super = object;
    d_const = flag;
    return object;
  }

  // create a const symbol in the instance local set

  Object* Instance::cdef (Interp* interp, Nameset* nset, const String& name,
			  Object* object) {
    if (name == SUPER) return setsuper (object, true);
    return p_iset->cdef (interp, nset, name, object);
  }

  // bind a symbol in the instance local set

  Object* Instance::vdef (Interp* interp, Nameset* nset, const String& name,
			  Object* object) {
    // check for super
    if (name == SUPER) return setsuper (object, false);
    // look in the instance local set
    Object* obj = p_iset->find (name);
    if (obj != nilp) return obj->vdef (interp, nset, object);
    //look in the class
    obj = p_class->find (name);
    if (obj != nilp) return obj->vdef (interp, nset, object);
    // bind locally
    return p_iset->vdef (interp, nset, name, object);
  }
  
  // evaluate an instance member

  Object* Instance::eval (Interp* interp, Nameset* nset, const String& name) {
    // check for super 
    if (name == SUPER) return p_super;
    // check for meta
    if (name == META) return p_class;
    // check in the instance local set
    Object* obj = p_iset->find (name);
    if (obj != nilp) {
      Object* result = obj->eval (interp, nset);
      if (dynamic_cast <Closure*> (result) == nilp) return result;
      return new Method (result, this);
    }
    // check in the class
    obj = p_class->find (name);
    if (obj != nilp) {
      Object* result = obj->eval (interp, nset);
      if (dynamic_cast <Closure*> (result) == nilp) return result;
      return new Method (result, this);
    }
    // check in the super instance
    if (p_super != nilp) {
      Object* result = p_super->eval (interp, nset, name);
      if (dynamic_cast <Closure*> (result) == nilp) return result;
      return new Method (result, this);
    }
    // look in the object
    Object* result = Object::eval (interp, nset, name);
    if (dynamic_cast <Closure*> (result) == nilp) return result;
    return new Method (result, this);
  }

  // evaluate an object by name within this instance

  Object* Instance::apply (Interp* interp, Nameset* nset, const String& name,
			   Cons* args) {
    Object* method = eval (interp, nset, name);
    return apply (interp, nset, method, args);
  }
    
  // evaluate an object within this instance

  Object* Instance::apply (Interp* interp, Nameset* nset, Object* object,
			   Cons* args) {
    // basic reject - as usual
    if (object == nilp) return nilp;
    // rebind the local set before the call
    Localset* lset = new Localset (p_iset);
    Object::iref (lset);
    lset->setparent (nset);
    Object* result = nilp;
    // let's make the call
    try {
      result = object->apply (interp, lset, args);
      Object::dref (lset);
    } catch (...) {
      Object::dref (lset);
      throw;
    }
    return result; 
  }
}
