// ---------------------------------------------------------------------------
// - Buffer.cpp                                                              -
// - standard object library - character buffer class implementation         -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Buffer.hpp"
#include "Exception.hpp"

namespace aleph {

  // default buffer size
  const long BUFFER_SIZE = 1024;

  // Create a new buffer class with a default size of 1024 characters

  Buffer::Buffer (void) {
    p_buffer = new char [BUFFER_SIZE];
    d_size   = BUFFER_SIZE;
    d_start  = 0;
    d_end    = 0;
    d_cursor = 0;
    d_insert = true;
  }

  // Create a new buffer with a predefined size

  Buffer::Buffer (const long size) {
    d_size   = (size <= 0) ? BUFFER_SIZE : size;
    p_buffer = new char[d_size];
    d_start  = 0;
    d_end    = 0;
    d_cursor = 0;
    d_insert = true;
  }
  
  // destroy this buffer
  
  Buffer::~Buffer (void) {
    delete [] p_buffer;
  }
  
  // add a character in this buffer
  
  void Buffer::add (const char value) {
    // if we are in insert mode - we need to shift the buffer if
    // the cursor is not at the end - hence the while loop below
    if (d_insert == true) {
      if (d_cursor == d_end) {
	p_buffer[d_end] = value;
	d_end    = (d_end + 1) % d_size;
	d_cursor = d_end;
      } else {
	long di = d_end;
	while (di != d_cursor) {
	  long si = (di == 0) ? d_size - 1 : di - 1;
	  p_buffer[di] = p_buffer[si];
	  di = si;
	}
	p_buffer[d_cursor] = value;
	d_cursor = (d_cursor + 1) % d_size;
	d_end    = (d_end + 1) % d_size;
      }
    } else {
      if (d_cursor == d_end) {
	p_buffer[d_end] = value;
	d_end    = (d_end + 1) % d_size;
	d_cursor = d_end;
      } else {
	p_buffer[d_cursor] = value;
	d_cursor = (d_cursor + 1) % d_size;
      }
    }
    // check for overflow
    if (d_end == d_start) {
      Exception e ("buffer-overflow", "character buffer overflow");
      e.setnlf (true);
      throw e;
    }
  }
  
  // read a character in this buffer
  
  char Buffer::read (void) {
    // check for no character
    if (d_start == d_end) return '\0';
    // get value and adjust start
    char value = p_buffer[d_start];
    d_start = (d_start + 1) % d_size;
    return value;
  }
  
  // pushback a character in this buffer
  
  void Buffer::pushback (const char value) {
    long index = (d_start == 0) ? d_size - 1 : d_start - 1;
    p_buffer[index] = value;
    if (d_cursor == d_start) d_cursor = index;
    d_start = index;
    // check for overflow
    if (d_end == d_start) {
      Exception e ("buffer-overflow", "character buffer overflow");
      e.setnlf (true);
      throw e;
    }
  }
  
  // pushback a string in this buffer
  
  void Buffer::pushback (const String& value) {
    int len = value.length () - 1;
    for (int i = len; i >= 0; i--)
      pushback (value[i]);
  }
  
  // remove a character before the cursor (backspace mode)
  
  bool Buffer::erase (void) {
    // do nothing if the cursor is at start
    if (d_cursor == d_start) return false;
    // loop and move characters
    long si = d_cursor;
    while (si != d_end) {
      long di = (si == 0) ? d_size - 1 : si - 1;
      p_buffer[di] = p_buffer[si];
      si = (si + 1) % d_size;
    }
    d_cursor = (d_cursor == 0) ? d_size - 1 : d_cursor - 1;
    d_end    = (d_end    == 0) ? d_size - 1 : d_end    - 1;
    return true;
  }
  
  // remove a character at the cursor (delete mode)
  
  bool Buffer::del (void) {
    // do nothing if empty or cursor is at end
    if (d_start  == d_end) return false;
    if (d_cursor == d_end) return false;
    
    // loop and move characters
    long di = d_cursor;
    while (di != d_end) {
      long si = (di + 1) % d_size;
      p_buffer[di] = p_buffer[si];
      di = (di + 1) % d_size;
    }
    d_end = (d_end == 0) ? d_size - 1 : d_end - 1;
    return true;
  }
  
  // kill one or several characters in the buffer
  
  void Buffer::kill (const long num) {
    for (long i = 0; i < num; i++)
      if (erase () == false) return;
  }
  
  // return true if the cursor is moved one character left
  
  bool Buffer::movel (void) {
    if (d_cursor == d_start) return false;
    d_cursor = (d_cursor == 0) ? d_size - 1 : d_cursor - 1;
    return true;
  }
  
  // return true if the cursor is moved one character right
  
  bool Buffer::mover (void) {
    if (d_cursor == d_end) return false;
    d_cursor = (d_cursor + 1) % d_size;
    return true;
  }
  
  // move the cursor to the beginning of the buffer
  
  long Buffer::setcb (void) {
    long result = (d_cursor > d_start) ? d_cursor - d_start : 
                                         d_start - d_cursor;
    d_cursor = d_start;
    return result;
  }

  // move the cursor to the end of the buffer

  long Buffer::setce (void) {
    long result = (d_cursor < d_end) ? d_end - d_cursor: d_cursor - d_end;
    d_cursor = d_end;
    return result;
  }

  // set the insert mode flag
  
  void Buffer::setimode (const bool mode) {
    d_insert = mode;
  }
  
  // return the length of this buffer
  
  long Buffer::length (void) const {
    return (d_end > d_start) ? d_end - d_start : d_start - d_end;
  }

  // return the cursor position
  
  long Buffer::getCursor (void) const {
    return d_cursor;
  }
  
  // return the corresponding string accumulated in this buffer
  
  String Buffer::toString (void) {
    long len = length ();
    if (len == 0) return String ();
    
    // create a temporary buffer to hold the characters
    char* buf = new char[len+1];
    long index = d_start;
    long pos   = 0;
    while (index != d_end) {
      buf[pos++] = p_buffer[index];
      index = (index + 1) % d_size;
    }
    buf[pos] = '\0';
    String result = buf;
    delete [] buf;
    return result;
  }
  
  // reset this buffer but do not change the size
  
  void Buffer::reset (void) {
    d_start  = 0;
    d_end    = 0;
    d_cursor = 0;
  }
}
