// ---------------------------------------------------------------------------
// - NameTable.hpp                                                           -
// - standard object library - name table object class definition            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_NAMETABLE_HPP
#define  ALEPH_NAMETABLE_HPP

#ifndef  ALEPH_STRVEC_HPP
#include "Strvec.hpp"
#endif

namespace aleph {

  /// The NameTable class is similar to the HashTable class except that it
  /// is designed to work with a small set of objects. The NameTable defines
  /// a binding between a name and an object. The class has the same methods
  /// like the HashTable class.
  /// @author amaury darsch

  class NameTable : public virtual Object {
  private:
    /// the name table is a list of buckets
    struct s_bucket* p_table;

  public:
    /// create a name table
    NameTable (void);
  
    /// destroy the name table. 
    ~NameTable (void);

    /// return the class name
    String repr (void) const;

    /// set or create a new object in the table.
    /// @param name the object name to set
    /// @param object the object to set or add
    void add (const String& name, Object* object);

    /// get an object by name. If the object is not found nil is returned.
    /// this means that this function always succeds. You can use lookup 
    /// instead to get an exception if the name is not found. You can also use
    /// the exists method to check for a name but exists and get is very slow.
    /// @param name the object name to search
    /// @return the object or nil
    Object* get (const String& name) const;

    /// find an object by name or throw an exception
    /// @param the object name to find
    /// @return the object
    Object* lookup (const String& name) const;

    /// remove an object in the name table. 
    /// @param name the object name to remove
    void remove (const String& name);
  
    /// @return true if the name exists in this name table
    bool exists (const String& name) const;
  
    /// @return a vector of names defined in this name table
    Strvec names (void) const;

  private:
    // make the copy constructor private
    NameTable (const NameTable&);
    // make the assignment operator private
    NameTable& operator = (const NameTable&);    
  };
}

#endif
