// ---------------------------------------------------------------------------
// - Output.hpp                                                              -
// - standard object library - output stream base class definition           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef ALEPH_OUTPUT_HPP
#define ALEPH_OUTPUT_HPP

#ifndef ALEPH_EXCEPTION_HPP
#include "Exception.hpp"
#endif

namespace aleph {

  /// The Output stream class is a base class which provides a write method for
  /// character in a generic way. There are several ways to create a stream,
  /// including standard Output, file based or socket stream. Various derived 
  /// classes are provided to create an Output stream. The class also provides
  /// support for builtin types serialization.
  /// @author amaury darsch

  class Output : public virtual Object {
  public:
    /// @return true if the output stream is a tty
    virtual bool istty (void) const;

    /// write one character on the output stream.
    /// @param value the character to write
    virtual void write (const char value) =0;

    /// write a character string to the output stream
    /// @param value the character string to write
    virtual void write (const char* value) =0;

    /// write a native integer to the output stream
    /// @param value the integer to write
    virtual void write (const long value) =0;

    /// write a character array to the output stream
    /// @param value the character string to write
    /// @param size  the number of character to write
    virtual void write (const char* value, const long size);

    /// write a string to the output stream
    /// @param value the character string to write
    virtual void write (const String& value);

    /// write a newline to this output stream
    /// function only. Use write for more error detection.
    /// @param line the line to write
    void newline (void);

    /// write a character string followed by a new line. This is a convenient
    /// function only. Use write for more error detection.
    /// @param line the line to write
    void writeln (const String& line);

    /// write an exception string on this output stream.
    /// @param e the exception to write
    void writeln (const Exception& e);

    /// write an exception string on this output stream. Use a file name to
    /// indicate where is the exception comming from.
    /// @param e    the exception to write
    /// @param name the file name to report
    /// @param lnum the line number in the file
    void writeln (const Exception& e, const String& name, const long lnum);

    /// serialize a quad word on this output stream
    /// @param value the quad to serialize
    void serial (const t_quad value);

    /// serialize an octa word on this output stream
    /// @param value the octa word to serialize
    void serial (const t_octa value);

    /// serialize a string on this output stream
    /// @param value the string to serialize
    void serial (const String& value);
    
    /// write a character on the output stream. Just return the stream
    /// @param the character to write
    /// @return the stream  
    Output& operator << (const char value);

    /// write a string on the output stream. Just return the stream.
    /// @param the character to write
    /// @return the stream
    Output& operator << (const String& value);

    /// write an integer on the output stream. Just return the stream.
    /// @param the character to write
    /// @return the stream
    Output& operator << (const long value);

    /// apply this output with a set of arguments and a method name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param argv   the argument vector to apply
    Object* apply (Interp* interp, Nameset* nset, const String& name,
		   Vector* argv);
  };
}

#endif
