// ---------------------------------------------------------------------------
// - OutputTerm.hpp                                                          -
// - standard object library - terminal output stream class definition       -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_OUTPUTTERM_HPP
#define  ALEPH_OUTPUTTERM_HPP

#ifndef  ALEPH_OUTPUT_HPP
#include "Output.hpp"
#endif

namespace aleph {

  /// The OutputTerm class is the terminal output class for writing various
  /// formatted data. The class implements the Output stream interface. The
  /// output stream type is selected at construction and can be either standard
  /// output or error output
  /// @author amaury darsch

  class OutputTerm : public Output {
  public:
    /// mode type for the default output stream
    enum t_mode {OUTPUT,ERROR};

  private:
    /// the stream id
    int d_sid;
    /// the terminfo array
    char** p_tinfo;

  protected:
    /// the insert mode flag
    bool d_insert;

  public:
    /// Create a new output stream.
    OutputTerm (void);

    /// Create a new Output stream. The output stream is created with the
    /// standard output mode (i.e. Output or error)
    /// @param mode the output mode
    OutputTerm (t_mode mode);

    /// destroy this output terminal
    ~OutputTerm (void);

    /// @return the class name
    String repr (void) const;

    /// @return true if the output stream is a tty
    bool istty (void) const;

    /// write one character on the output stream.
    /// @param value the character to write  
    void write (const char value);

    /// write a character string to the output stream
    /// @param value the character string to write
    void write (const char* value);

    /// write an integer to the output stream
    /// @param value the integer to write  
    void write (const long value);

    /// insert a character - depending on the insert mode
    /// @param c the character to insert
    void insert (const char c);

    /// erase left a certain number of characters
    /// @param num the number of backspace to do
    void erasel (const long num);

    /// delete one character at cursor position
    void del (void);

    /// move to the left by a certain amount
    /// @param num the number of backspace to do
    void movel (const long num);

    /// move to the right by a certain amount
    /// @param num the number of move to do
    void mover (const long num);

    /// generate a new output term
    /// @param argv the argument vector
    static Object* mkout (Vector* argv);

    /// generate a new error term
    /// @param argv the argument vector
    static Object* mkerr (Vector* argv);

    /// evaluate an output term member name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to evaluate
    Object* eval (Interp* interp, Nameset* nset, const String& name);

    /// apply this output term with a set of arguments and a method name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param args   the arguments to apply
    Object* apply (Interp* interp, Nameset* nset, const String& name,
		   Cons* args);

  private:
    // make the copy constructor private
    OutputTerm (const OutputTerm&);
    // make the assignment operator private
    OutputTerm& operator = (const OutputTerm&);
  };
}

#endif
