// ---------------------------------------------------------------------------
// - cio.hxx                                                                 -
// - standard system library - c io native function definition               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

/// This package contains c-io related procedures. I/O access is mostly driven
/// by reading and writing operations. Other operations related to file are
/// also defined here. Socket procedures are located in the cnet package.
/// @author amaury darsch

#ifndef ALEPH_CIO_HXX
#define ALEPH_CIO_HXX

namespace aleph {

  /// @return the standard input stream associated with this process
  int c_stdin (void);

  /// @return the standard output stream associated with this process
  int c_stdout (void);

  /// @return the standard error stream associated with this process
  int c_stderr (void);

  /// check if the name is a regular file
  /// @param name the file name
  bool c_isfile (const char* name);

  /// check if the name is a directory
  /// @param name the directory name
  bool c_isdir (const char* name);

  /// open a file for reading. 
  /// @param name the file name
  /// @return a file descriptor or -1 in case of error
  int c_openr (const char* name);

  /// open a file for writing. 
  /// @param name the file name
  /// @return a file descriptor or -1 in case of error
  int c_openw (const char* name);

  /// open a file for reading and writing.
  /// @param name the file name
  /// @return a file descriptor or -1 in case of error
  int c_openrw (const char* name);

  /// wait on a stream id for a certain number of characters.
  /// @param sid the stream id
  /// @param the number of characters we wait for
  /// @param tout the eventual time out in milliseconds
  /// @return true on success
  bool c_select (const int sid, const long size, const long tout);

  /// read n bytes from the specified stream. The stream can be the default 
  /// input stream, a stream file, a socket etc.
  /// @param sid the stream id
  /// @param buffer the character buffer
  /// @param the number of character to read
  /// @return the number of character read or an error code
  long c_read (const int sid, char* buffer, const long count);

  /// write n bytes to the specified stream. The stream can be the default 
  /// output stream, a stream file, a socket etc.
  /// @param sid the stream id
  /// @param buffer the character buffer
  /// @param the number of character to write
  /// @return the number of character writen or an error code
  long c_write (const int sid, const char* buffer, const long count);

  /// write an integer to the specified stream. The stream can be the default 
  /// output stream, a stream file, a socket etc.
  /// @param sid the stream id
  /// @param value the integer value
  /// @return the number of character writen or an error code
  long c_write (const int sid, long value);

  /// close this stream. 
  /// @param sid the stream id
  void c_close (const int sid);

  /// return the size of a file in bytes
  /// @param  sid the file id
  /// @return the file size or -1 in case of error
  long c_fsize (const int sid);

  /// position a file at a certain position
  /// @param  sid the file id
  /// @param  pos the file position to go
  /// @return true on success
  bool c_lseek (const int sid, const long pos);

  /// remove a file from the file system. The behavior of this call is
  /// system dependant but shall work most of the time.
  /// @param name the file name to remove
  void c_rm (const char* name);

  /// get a temporary unique name for a file. never delete the string
  /// @return a unique temporary name
  char* c_tmpname (void);

  /// remove the file extension from a file name
  /// @param name the original file name
  /// @return the file name without the extension
  char* c_rmext (const char* name);
}

#endif
