// ForthVM.cpp
//
// The FORTH Virtual Machine (FVM) to execute Forth byte code.
//
// Copyright (c) 1996--1998 Krishna Myneni, Creative Consulting for
//   Research & Education
//
// This software is provided under the General Public License.
//
// Created: 2-24-96
// Revisions: 
//       10-14-1998
//       4-28-1999  increased stack size to 32768 and ret stack to 4096  KM
//       5-29-1999  moved ABORT to vm.s; added abort/quit error code;
//       6-06-1999  created C++ functions which can be called from vm  KM
//
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <fstream.h>
#include "fbc.h"
#include "ForthCompiler.h"
#include "ForthVM.h"

#define STACK_SIZE 32768
#define RETURN_STACK_SIZE 4096

extern char* WordNames[];
extern byte WordCodes[];
extern int debug;
extern "C" int Base;

// The Dictionary

vector<DictionaryEntry> Dictionary;

// Tables

vector<char*> StringTable;

// stacks; these are global to this module

int ForthStack[STACK_SIZE];                  // the stack
byte ForthTypeStack[STACK_SIZE];             // the value type stack
int ForthReturnStack[RETURN_STACK_SIZE];     // the return stack
byte ForthReturnTypeStack[RETURN_STACK_SIZE];// the return value type stack


// global pointers and the virtual machine are exported from vm.asm

extern "C"
{
    int vm (byte*);     // the machine code virtual machine
    int* GlobalSp;      // the global stack pointer
    byte* GlobalTp;     // the global type stack pointer
    byte* GlobalIp;     // the global instruction pointer
    int* GlobalRp;      // the global return stack pointer
    byte* GlobalRtp;    // the global return type stack pointer
    int* BottomOfStack;
    int* BottomOfReturnStack;
    byte* BottomOfTypeStack;
    byte* BottomOfReturnTypeStack;

}

// global input and output streams

istream* pInStream ;
ostream* pOutStream ;

//---------------------------------------------------------------

int OpenForth ()
{
// Initialize the FORTH dictionary; return the size of
//   the dictionary.

    int i;
    DictionaryEntry d;

    for (i = 0; i < NUMBER_OF_INTRINSIC_WORDS; i++)
    {
        d.WordName = WordNames[i];
        d.WordCode = WordCodes[i];
        d.MemPtr = NULL;
        Dictionary.push_back(d);
    }

    // Initialize the global stack pointers

    BottomOfStack = ForthStack + STACK_SIZE - 1;
    BottomOfReturnStack = ForthReturnStack + RETURN_STACK_SIZE - 1;
    BottomOfTypeStack = ForthTypeStack + STACK_SIZE - 1;
    BottomOfReturnTypeStack = ForthReturnTypeStack + RETURN_STACK_SIZE - 1;

    GlobalSp = BottomOfStack;
    GlobalTp = BottomOfTypeStack;
    GlobalRp = BottomOfReturnStack;
    GlobalRtp = BottomOfReturnTypeStack;

    Base = 10;

    return Dictionary.size();
}
//---------------------------------------------------------------

void CloseForth ()
{
    // Clean up the compiled words

    while (Dictionary.size())
    {
        RemoveLastWord();
    }

    // Clean up the string table

    vector<char*>::iterator j = StringTable.begin();

    while (j < StringTable.end())
    {
        if (*j) delete [] *j;
        ++j;
    }
}

//---------------------------------------------------------------

void RemoveLastWord ()
{
// Remove the last dictionary entry

	vector<DictionaryEntry>::iterator i = Dictionary.end() - 1;
	delete [] i->MemPtr;	// free memory
	
	Dictionary.pop_back(); 
}
//---------------------------------------------------------------

vector<DictionaryEntry>::iterator LocateWord (char* name)
{
// Search the dictionary from end to beginning for an entry
//   with the specified name. Return the iterator to the word
//   or NULL if not found.

	vector<DictionaryEntry>::iterator i;
    
	for (i = Dictionary.end()-1; i >= Dictionary.begin(); i--)
	{
        	if (strcmp(name, i->WordName) == 0) break;
	}

	if (i >= Dictionary.begin())
        	return i;
	else
		return NULL;
}
//---------------------------------------------------------------

int ForthVM (vector<byte>* pFBC, istream& InStream, ostream& OutStream,
    int** pStackPtr, byte** pTypePtr)
{
// The FORTH Virtual Machine
//
// Arguments:
//
//      pFBC        pointer to vector of Forth byte codes
//      OutStream   reference to the output stream
//      pStackPtr   receives pointer to the top item on the stack at exit
//      pTypePtr    receives pointer to the top item on the type stack at exit
//
// Return value:
//
//      0   no error
//      1   illegal operand; require an address
//      2   illegal operand; require an fval or ival
//      3   illegal operand; not expected type
//      4   divide by zero
//      5   return stack has been corrupted
//      6   invalid opcode
//      7   stack underflow
//      8   abort or quit
//      9   dictionary entry already has allocated memory


  if (pFBC->size() == 0) return 0;  // null opcode vector

  // Reset the return stacks on every entry to the vm

  GlobalRp = BottomOfReturnStack;
  GlobalRtp = BottomOfReturnTypeStack;

  // Set the global stream pointers for CPP functions to use

  pInStream = &InStream;
  pOutStream = &OutStream; 

  // Initialize the instruction ptr and error code

  byte *ip = pFBC->begin();
  int ecode = 0;

  // Execute the virtual machine; return when error occurs or
  //   the return stack is exhausted.

  ecode = vm (ip);

  // Set up return information

  *pStackPtr = GlobalSp + 1;
  *pTypePtr = GlobalTp + 1;

  // On stack underflow, update the global stack pointers.

  if (ecode == 7)
    {
      OutStream << "\nStack underflow.\n";
      GlobalSp = BottomOfStack;
      GlobalTp = BottomOfTypeStack;
    }

  return ecode;
}
//---------------------------------------------------------------

int CPP_dot ()
{
  // stack: ( n -- | print n in current base ) 
  
  ++GlobalSp; ++GlobalTp;
  if (GlobalSp > BottomOfStack) 
    return 7;
  else
    {
      int n = *GlobalSp;
      if (n < 0)
	{
	  *pOutStream << '-';
	  *GlobalSp = abs(n);
	}
      --GlobalSp; --GlobalTp;
      CPP_udot();
    }
  return 0;
}
//---------------------------------------------------------------

int CPP_udot ()
{
  // stack: ( u -- | print unsigned single in current base )

  ++GlobalSp; ++GlobalTp;
  if (GlobalSp > BottomOfStack) return 7;
  
  int i, ndig, nchar;
  unsigned int u, utemp, uscale;

  u = *GlobalSp;
  ndig = 1;
  uscale = 1;
  utemp = u;

  while (utemp /= Base) {++ndig; uscale *= Base;}

  for (i = 0; i < ndig; i++) 
    {
      utemp = u/uscale;
      nchar = (utemp < 10) ? (utemp + 48) : (utemp + 55);
      *pOutStream << (char) nchar;
      u -= utemp*uscale;
      uscale /= Base;
    }
  *pOutStream << ' ';
  return 0;
}
//---------------------------------------------------------------

int CPP_fdot ()
{
  // stack: ( f -- | print floating point number )

  ++GlobalSp; ++GlobalTp; ++GlobalSp; ++GlobalTp;
  if (GlobalSp > BottomOfStack)
    return 7;
  else
    {
      --GlobalSp; --GlobalTp;
      *pOutStream << *((double*) GlobalSp) << ' ';
      ++GlobalSp; ++GlobalTp;
    }
  return 0;
}
//---------------------------------------------------------------

int CPP_dots ()
{
  L_depth();  
  ++GlobalSp; ++GlobalTp;
  int depth = *GlobalSp;
  ++GlobalSp; ++GlobalTp;

  if (debug)
    {
      *pOutStream << "\nTop of Stack = " << ((int)ForthStack);
      *pOutStream << "\nBottom of Stack = " << ((int)BottomOfStack);
      *pOutStream << "\nStack ptr = " << ((int)GlobalSp);
      *pOutStream << "\nDepth = " << depth;
    }
 
  if (depth > 0)
    {
      int i;
      byte* bptr;

      for (i = 0; i < depth; i++)
        {
	  if (*(GlobalTp + i) == OP_ADDR)
            {
                bptr = *((byte**) (GlobalSp + i));
                *pOutStream << "\n\taddr\t" << ((int)bptr);
            }
            else
            {
                *pOutStream << "\n\t\t" << *(GlobalSp + i);
            }
        }
    }
  else
    {
        *pOutStream << "<empty>";
    }
  *pOutStream << '\n';
  --GlobalSp; --GlobalTp;
  return 0;
}
//---------------------------------------------------------------


int CPP_emit ()
{
  // stack: ( n -- | display character with ascii code n )

  ++GlobalSp; ++GlobalTp;
  if (GlobalSp > BottomOfStack)
    return 7;
  else
    *pOutStream << (char)(*GlobalSp);
  return 0;
}
//---------------------------------------------------------------

int CPP_cr ()
{
  *pOutStream << '\n';
  return 0;
}
//---------------------------------------------------------------

int CPP_spaces ()
{
  ++GlobalSp; ++GlobalTp;
  if (GlobalSp > BottomOfStack) 
    return 7;
  else
    {
      int n = *GlobalSp;
      if (n > 0)
	for (int i = 0; i < n; i++) *pOutStream << ' ';
    }
  return 0;
}
//---------------------------------------------------------------

int CPP_type ()
{
  ++GlobalSp; ++GlobalTp;
  if (GlobalSp > BottomOfStack) 
    return 7;
  else
    {
      int n = *GlobalSp++; ++GlobalTp;
      if (GlobalSp > BottomOfStack) 
	return 7;
      if (*GlobalTp != OP_ADDR)
	return 3;
      char* cp = *((char**) GlobalSp);
      for (int i = 0; i  < n; i++) *pOutStream << *cp++;
    }
  return 0;
}
//---------------------------------------------------------------

int CPP_words ()
{
  for (int i = 0; i < Dictionary.size(); i++)
    {
      *pOutStream << '\t' << Dictionary[i].WordName;
      if ((i+1) % 8 == 0) *pOutStream << '\n';
    }
  return 0;
}
//---------------------------------------------------------------

int CPP_allot ()
{
  ++GlobalSp; ++GlobalTp;
  if (GlobalSp > BottomOfStack) 
    return 7;
  if (*GlobalTp != OP_IVAL)
    return 3;  // need an int

  vector<DictionaryEntry>::iterator id = Dictionary.end() - 1;
  int n = *GlobalSp;
  if (n > 0)
    {
      if (id->MemPtr == NULL) 
	id->MemPtr = new byte[n];
      else 
	return 9;
    }
  else
    id->MemPtr = NULL;

  return 0;
}
//---------------------------------------------------------------

int CPP_word ()
{
  // stack: ( n -- ^str | parse next word in input stream )
  // n is the delimiting character and ^str is a counted string.


  return 0;
}
//----------------------------------------------------------------

int CPP_find()
{
  // stack: ( ^str -- ^str 0 | xt 1 | xt -1 )

  return 0;
}



