#include "global.h"

VALUE mGdk;

VALUE gdkFont;
VALUE gdkImage;
VALUE gdkGC;
VALUE gdkGCValues;
VALUE gdkCursor;
VALUE gdkAtom;
VALUE gdkEvent;

VALUE mGdkIM;
VALUE gdkIC;
VALUE gdkICAttr;

VALUE gdk_object_list;

VALUE
make_tobj(obj, klass, size)
    gpointer obj;
    VALUE klass;
    int size;
{
    gpointer copy;
    VALUE data;

    copy = xmalloc(size);
    memcpy(copy, obj, size);
    data = Data_Wrap_Struct(klass, 0, free, copy);

    return data;
}

gpointer
get_tobj(obj, klass)
    VALUE obj, klass;
{
    void *ptr;

    if (NIL_P(obj)) return NULL;

    if (!rb_obj_is_instance_of(obj, klass)) {
	rb_raise(rb_eTypeError, "not a %s", rb_class2name(klass));
    }
    Data_Get_Struct(obj, void, ptr);

    return ptr;
}

VALUE
make_gdkfont(font)
    GdkFont *font;
{
    VALUE obj;

    gdk_font_ref(font);
    obj = Data_Wrap_Struct(gdkFont, 0, gdk_font_unref, font);

    return obj;
}

GdkFont*
get_gdkfont(font)
    VALUE font;
{
    GdkFont *gfont;

    if (NIL_P(font)) return NULL;

    if (!rb_obj_is_instance_of(font, gdkFont)) {
	rb_raise(rb_eTypeError, "not a GdkFont");
    }
    Data_Get_Struct(font, GdkFont, gfont);

    return gfont;
}

VALUE
make_gdkcmap(cmap)
    GdkColormap *cmap;
{
    gdk_colormap_ref(cmap);
    return Data_Wrap_Struct(gdkColormap, 0, gdk_colormap_unref, cmap);
}

GdkColormap*
get_gdkcmap(cmap)
    VALUE cmap;
{
    GdkColormap *gcmap;

    if (NIL_P(cmap)) return NULL;

    if (!rb_obj_is_kind_of(cmap, gdkColormap)) {
	rb_raise(rb_eTypeError, "not a GdkColormap");
    }
    Data_Get_Struct(cmap, GdkColormap, gcmap);

    return gcmap;
}

VALUE
make_gdkvisual(visual)
    GdkVisual *visual;
{
    gdk_visual_ref(visual);
    return Data_Wrap_Struct(gdkVisual, 0, gdk_visual_unref, visual);
}

GdkVisual*
get_gdkvisual(visual)
    VALUE visual;
{
    GdkVisual *gvisual;

    if (NIL_P(visual)) return NULL;

    if (!rb_obj_is_kind_of(visual, gdkVisual)) {
	rb_raise(rb_eTypeError, "not a GdkVisual");
    }
    Data_Get_Struct(visual, GdkVisual, gvisual);

    return gvisual;
}

static void
delete_gdkdraw(obj)
    VALUE obj;
{
    GdkDrawable *draw;
    VALUE klass;

    klass = rb_class_of(obj);
    Data_Get_Struct(draw, GdkDrawable, draw);
    if (0);
    else if (klass == gdkWindow) gdk_window_unref(draw);
    else if (klass == gdkBitmap) gdk_bitmap_unref(draw);
    else if (klass == gdkPixmap) gdk_pixmap_unref(draw);
    else {
	rb_p(klass);
	rb_raise(rb_eTypeError, "not a Gdk::Drawable object.");
    }
    rb_hash_aset(gdk_object_list, INT2NUM((VALUE)draw), Qnil);
}

VALUE
make_gdkdraw(klass, draw, ref, unref)
    VALUE klass;
    GdkDrawable *draw;
    void (*ref)();
    void (*unref)();
{
    VALUE obj;

    obj = rb_hash_aref(gdk_object_list, INT2NUM((VALUE)draw));
    if (obj == Qnil) {
	(*ref)(draw);
	obj = Data_Wrap_Struct(klass, 0, delete_gdkdraw, draw);
	rb_hash_aset(gdk_object_list, INT2NUM((VALUE)draw), INT2NUM(obj));
    } else {
	obj = NUM2INT(obj);
    }
    return obj;
}

GdkWindow*
get_gdkdraw(draw, klass, kname)
    VALUE draw, klass;
    char *kname;
{
    GdkDrawable *d;

    if (NIL_P(draw)) return NULL;

    if (!rb_obj_is_kind_of(draw, klass)) {
	rb_raise(rb_eTypeError, "not a %s", kname);
    }
    Data_Get_Struct(draw, GdkDrawable, d);

    return d;
}

static void
delete_gdkwindow(window)
    GdkWindow *window;
{
    gdk_window_unref(window);
    rb_hash_aset(gdk_object_list, INT2NUM((VALUE)window), Qnil);
}

VALUE
new_gdkwindow(window)
    GdkWindow *window;
{
    VALUE obj;

    obj = Data_Wrap_Struct(gdkWindow, 0, delete_gdkwindow, window);
    rb_hash_aset(gdk_object_list, INT2NUM((VALUE)window), INT2NUM(obj));

    return obj;
}

VALUE
make_gdkwindow(window)
    GdkWindow *window;
{
    VALUE obj;

    if (window == NULL) return Qnil;

    obj = rb_hash_aref(gdk_object_list, INT2NUM((VALUE)window));
    if (obj == Qnil) {
	gdk_window_ref(window);
	obj = new_gdkwindow(window);
    } else {
	obj = NUM2INT(obj);
    }
    return obj;
}

static void
delete_gdkpixmap(pixmap)
    GdkPixmap *pixmap;
{
    gdk_pixmap_unref(pixmap);
    rb_hash_aset(gdk_object_list, INT2NUM((VALUE)pixmap), Qnil);
}

VALUE
new_gdkpixmap(pixmap)
    GdkPixmap *pixmap;
{
    VALUE obj;

    obj = Data_Wrap_Struct(gdkPixmap, 0, delete_gdkpixmap, pixmap);
    rb_hash_aset(gdk_object_list, INT2NUM((VALUE)pixmap), INT2NUM(obj));

    return obj;
}

VALUE
make_gdkpixmap(pixmap)
    GdkPixmap *pixmap;
{
    VALUE obj;

    if (pixmap == NULL) return Qnil;

    obj = rb_hash_aref(gdk_object_list, INT2NUM((VALUE)pixmap));
    if (obj == Qnil) {
	gdk_pixmap_ref(pixmap);
	obj = new_gdkpixmap(pixmap);
    } else {
	obj = NUM2INT(obj);
    }
    return obj;
}

static void
delete_gdkbitmap(bitmap)
    GdkBitmap *bitmap;
{
    gdk_bitmap_unref(bitmap);
    rb_hash_aset(gdk_object_list, INT2NUM((VALUE)bitmap), Qnil);
}

VALUE
new_gdkbitmap(bitmap)
    GdkBitmap *bitmap;
{
    VALUE obj;

    obj = Data_Wrap_Struct(gdkBitmap, 0, delete_gdkbitmap, bitmap);
    rb_hash_aset(gdk_object_list, INT2NUM((VALUE)bitmap), INT2NUM(obj));

    return obj;
}

VALUE
make_gdkbitmap(bitmap)
    GdkBitmap *bitmap;
{
    VALUE obj;

    if (bitmap == NULL) return Qnil;

    obj = rb_hash_aref(gdk_object_list, INT2NUM((VALUE)bitmap));
    if (obj == Qnil) {
	gdk_bitmap_ref(bitmap);
	obj = new_gdkbitmap(bitmap);
    } else {
	obj = NUM2INT(obj);
    }
    return obj;
}

VALUE
make_gdkimage(image)
    GdkImage *image;
{
    return Data_Wrap_Struct(gdkImage, 0, gdk_image_destroy, image);
}

GdkImage*
get_gdkimage(image)
    VALUE image;
{
    GdkImage *gimage;

    if (NIL_P(image)) return NULL;

    if (!rb_obj_is_instance_of(image, gdkImage)) {
	rb_raise(rb_eTypeError, "not a GdkImage");
    }
    Data_Get_Struct(image, GdkImage, gimage);
    if (gimage == 0) {
	rb_raise(rb_eArgError, "destroyed GdkImage");
    }

    return gimage;
}

VALUE
make_gdkevent(ev)
    GdkEvent *ev;
{
    GdkEvent *event;
    VALUE ret;

    event = gdk_event_copy(ev);

    switch (event->type) {

    case GDK_EXPOSE:
	ret = Data_Wrap_Struct(gdkEventExpose, 0, gdk_event_free, event);
	break;

    case GDK_NO_EXPOSE:
	ret = Data_Wrap_Struct(gdkEventNoExpose, 0, gdk_event_free, event);
	break;

    case GDK_VISIBILITY_NOTIFY:
	ret = Data_Wrap_Struct(gdkEventVisibility, 0, gdk_event_free, event);
	break;

    case GDK_MOTION_NOTIFY:
	ret = Data_Wrap_Struct(gdkEventMotion, 0, gdk_event_free, event);
	break;

    case GDK_BUTTON_PRESS:
    case GDK_2BUTTON_PRESS:
    case GDK_3BUTTON_PRESS:
    case GDK_BUTTON_RELEASE:
	ret = Data_Wrap_Struct(gdkEventButton, 0, gdk_event_free, event);
	break;

    case GDK_KEY_PRESS:
    case GDK_KEY_RELEASE:
	ret = Data_Wrap_Struct(gdkEventKey, 0, gdk_event_free, event);
	break;

    case GDK_FOCUS_CHANGE:
	ret = Data_Wrap_Struct(gdkEventFocus, 0, gdk_event_free, event);
	break;

    case GDK_CONFIGURE:
	ret = Data_Wrap_Struct(gdkEventConfigure, 0, gdk_event_free, event);
	break;

    case GDK_PROPERTY_NOTIFY:
	ret = Data_Wrap_Struct(gdkEventProperty, 0, gdk_event_free, event);
	break;

    case GDK_SELECTION_CLEAR:
    case GDK_SELECTION_REQUEST:
    case GDK_SELECTION_NOTIFY:
	ret = Data_Wrap_Struct(gdkEventSelection, 0, gdk_event_free, event);
	break;

    case GDK_PROXIMITY_IN:
    case GDK_PROXIMITY_OUT:
	ret = Data_Wrap_Struct(gdkEventProximity, 0, gdk_event_free, event);
	break;
	/* remove 1.2.x
	   case GDK_DRAG_BEGIN:
	   ret = Data_Wrap_Struct(gdkEventDragBegin, 0, gdk_event_free, event);
	   break;

	   case GDK_DRAG_REQUEST:
	   ret = Data_Wrap_Struct(gdkEventDragRequest, 0, gdk_event_free, event);
	   break;

	   case GDK_DROP_ENTER:
	   ret = Data_Wrap_Struct(gdkEventDropEnter, 0, gdk_event_free, event);
	   break;

	   case GDK_DROP_LEAVE:
	   ret = Data_Wrap_Struct(gdkEventDropLeave, 0, gdk_event_free, event);
	   break;

	   case GDK_DROP_DATA_AVAIL:
	   ret = Data_Wrap_Struct(gdkEventDropDataAvailable, 0, gdk_event_free, event);
	   break;
	*/
    case GDK_CLIENT_EVENT:
	ret = Data_Wrap_Struct(gdkEventClient, 0, gdk_event_free, event);
	break;

	/* I don't know which these types below should be related to.
	   Please teach me if you know it.*/
    case GDK_DELETE:
    case GDK_DESTROY:
    case GDK_ENTER_NOTIFY:
    case GDK_LEAVE_NOTIFY:
    case GDK_MAP:
    case GDK_UNMAP:
	/*  case GDK_OTHER_EVENT: remove 1.2.x */

    default:
	ret = Data_Wrap_Struct(gdkEvent, 0, gdk_event_free, event);
    }
    return ret;
}

GdkEvent*
get_gdkevent(event)
    VALUE event;
{
    GdkEvent *gevent;

    if (NIL_P(event)) return NULL;

    if (!rb_obj_is_kind_of(event, gdkEvent)) {
	rb_raise(rb_eTypeError, "not a GdkEvent...");
    }
    Data_Get_Struct(event, GdkEvent, gevent);

    return gevent;
}

VALUE
make_gdkgc(gc)
    GdkGC *gc;
{
    gdk_gc_ref(gc);
    return Data_Wrap_Struct(gdkGC, 0, gdk_gc_destroy, gc);
}

GdkGC*
get_gdkgc(gc)
    VALUE gc;
{
    GdkGC *ggc;

    if (NIL_P(gc)) return NULL;

    if (!rb_obj_is_instance_of(gc, gdkGC)) {
	rb_raise(rb_eTypeError, "not a GdkGC");
    }
    Data_Get_Struct(gc, GdkGC, ggc);
    if (ggc == 0) {
	rb_raise(rb_eArgError, "destroyed GdkGC");
    }

    return ggc;
}


/*
 * Gdk::Font
 */
static VALUE
gdkfnt_load_font(self, name)
    VALUE self, name;
{
    GdkFont *font;

    font = gdk_font_load(STR2CSTR(name));
    return Data_Wrap_Struct(gdkFont, 0, gdk_font_unref, font);
}

static VALUE
gdkfnt_load_fontset(self, name)
    VALUE self, name;
{
    GdkFont *font;

    font = gdk_fontset_load(STR2CSTR(name));
    return Data_Wrap_Struct(gdkFont, 0, gdk_font_unref, font);
}

static VALUE
gdkfnt_string_width(self, str)
    VALUE self, str;
{
    int w;

    w = gdk_string_width(get_gdkfont(self), STR2CSTR(str));
    return INT2NUM(w);
}

static VALUE
gdkfnt_string_height(self, str)
    VALUE self, str;
{
    int h;

    h = gdk_string_height(get_gdkfont(self), STR2CSTR(str));
    return INT2NUM(h);
}

static VALUE
gdkfnt_type(self)
    VALUE self;
{
    return INT2NUM(get_gdkfont(self)->type);
}

static VALUE
gdkfnt_ascent(self)
    VALUE self;
{
    return INT2NUM(get_gdkfont(self)->ascent);
}

static VALUE
gdkfnt_descent(self)
    VALUE self;
{
    return INT2NUM(get_gdkfont(self)->descent);
}

static VALUE
gdkfnt_equal(fn1, fn2)
    VALUE fn1, fn2;
{
    if (gdk_font_equal(get_gdkfont(fn1), get_gdkfont(fn2)))
	return Qtrue;
    return Qfalse;
}

static VALUE
gdkfnt_string_extents(self, str)
    VALUE self, str;
{
    gint lbearing, rbearing, width, ascent, descent;

    gdk_string_extents(get_gdkfont(self), STR2CSTR(str),
		       &lbearing, &rbearing, &width, &ascent, &descent);
    return rb_ary_new3(5,
		       INT2NUM(lbearing), INT2NUM(rbearing),
		       INT2NUM(width), INT2NUM(ascent), INT2NUM(descent));
}

/*
 * Gdk::Image
 */
static VALUE
gdkimage_s_newbmap(klass, visual, data, w, h)
    VALUE klass, visual, data, w, h;
{
    Check_Type(data, T_STRING);
    if (RSTRING(data)->len < w * h) {
	rb_raise(rb_eArgError, "data too short");
    }
    return make_gdkimage(gdk_image_new_bitmap(get_gdkvisual(visual),
					      RSTRING(data)->ptr,
					      NUM2INT(w),NUM2INT(h)));
}

static VALUE
gdkimage_s_new(klass, type, visual, w, h)
    VALUE klass, type, visual, w, h;
{
    return make_gdkimage(gdk_image_new((GdkImageType)NUM2INT(type),
				       get_gdkvisual(visual),
				       NUM2INT(w),NUM2INT(h)));
}

static VALUE
gdkimage_s_get(klass, win, x, y, w, h)
    VALUE klass, win, x, y, w, h;
{
    return make_gdkimage(gdk_image_get(get_gdkdrawable(win),
				       NUM2INT(x),NUM2INT(y),
				       NUM2INT(w),NUM2INT(h)));
}

static VALUE
gdkimage_put_pixel(self, x, y, pix)
    VALUE self, x, y, pix;
{
    gdk_image_put_pixel(get_gdkimage(self),
			NUM2INT(x),NUM2INT(y),NUM2INT(pix));
    return self;
}

static VALUE
gdkimage_get_pixel(self, x, y)
    VALUE self, x, y;
{
    guint32 pix;

    pix = gdk_image_get_pixel(get_gdkimage(self), NUM2INT(x),NUM2INT(y));
    return INT2NUM(pix);
}

static VALUE
gdkimage_destroy(self)
    VALUE self;
{
    gdk_image_destroy(get_gdkimage(self));
    DATA_PTR(self) = 0;
    return Qnil;
}

/*
 * Gdk::GC
 */
static VALUE
gdkgc_s_new(self, win)
    VALUE self, win;
{
    return make_gdkgc(gdk_gc_new(get_gdkdrawable(win)));
}

static VALUE
gdkgc_copy(self, copy)
    VALUE self, copy;
{
    gdk_gc_copy(get_gdkgc(self), get_gdkgc(copy));
    return copy;
}

static VALUE
gdkgc_destroy(self)
    VALUE self;
{
    gdk_gc_destroy(get_gdkgc(self));
    DATA_PTR(self) = 0;
    return Qnil;
}

static VALUE
gdkgc_set_function(self, func)
    VALUE self, func;
{
    GdkFunction f;
    f = (GdkFunction) NUM2INT(func);
    if (f != GDK_COPY && f != GDK_INVERT && f != GDK_XOR)
	rb_raise(rb_eArgError, "function out of range");
  
    gdk_gc_set_function(get_gdkgc(self), f);
    return func;
}

static VALUE
gdkgc_set_foreground(self, color)
    VALUE self, color;
{
    gdk_gc_set_foreground(get_gdkgc(self), get_gdkcolor(color));
    return self;
}

static VALUE
gdkgc_set_background(self, color)
    VALUE self, color;
{
    gdk_gc_set_background(get_gdkgc(self), get_gdkcolor(color));
    return self;
}

static VALUE
gdkgc_set_clip_mask(self, mask)
    VALUE self, mask;
{
    gdk_gc_set_clip_mask(get_gdkgc(self), NIL_P(mask)?NULL:get_gdkbitmap(mask));
    return mask;
}

static VALUE
gdkgc_set_clip_origin(self, x, y)
    VALUE self, x, y;
{
    gdk_gc_set_clip_origin(get_gdkgc(self), NUM2INT(x), NUM2INT(y));
    return self;
}

static VALUE
gdkgc_set_clip_rectangle(self, rectangle)
    VALUE self, rectangle;
{
    gdk_gc_set_clip_rectangle(get_gdkgc(self), get_gdkrectangle(rectangle));
    return rectangle;
}

static VALUE
gdkgc_set_clip_region(self, region)
    VALUE self, region;
{
    gdk_gc_set_clip_region(get_gdkgc(self), get_gdkregion(region));
    return region;
}

static VALUE
gdkgc_set_dashes(self, dash_offset, dash_list)
    VALUE self, dash_offset, dash_list;
{
    gdk_gc_set_dashes(get_gdkgc(self), NUM2INT(dash_offset),
		      RSTRING(dash_list)->ptr, RSTRING(dash_list)->len);
    return self;
}

static VALUE
gdkgc_set_exposures(self, exposures)
    VALUE self, exposures;
{
    gdk_gc_set_exposures(get_gdkgc(self), NUM2INT(exposures));
    return self;
}

static VALUE
gdkgc_set_fill(self, fill)
    VALUE self, fill;
{
    gdk_gc_set_fill(get_gdkgc(self), NUM2INT(fill));
    return self;
}

static VALUE
gdkgc_set_font(self, font)
    VALUE self, font;
{
    gdk_gc_set_font(get_gdkgc(self), get_gdkfont(font));
    return self;
}

static VALUE
gdkgc_set_line_attributes(self, line_width, line_style, cap_style, join_style)
    VALUE self, line_width, line_style, cap_style, join_style;
{
    gdk_gc_set_line_attributes(get_gdkgc(self), NUM2INT(line_width),
			       NUM2INT(line_style), NUM2INT(cap_style),
			       NUM2INT(join_style));
    return self;
}

static VALUE
gdkgc_set_stipple(self, stipple)
    VALUE self, stipple;
{
    gdk_gc_set_stipple(get_gdkgc(self), get_gdkpixmap(stipple)); 
    return self;
}

static VALUE
gdkgc_set_subwindow(self, mode)
    VALUE self, mode;
{
    gdk_gc_set_subwindow(get_gdkgc(self), NUM2INT(mode));
    return self;
}

static VALUE
gdkgc_set_tile(self, tile)
    VALUE self, tile;
{
    gdk_gc_set_tile(get_gdkgc(self), get_gdkpixmap(tile)); 
    return self;
}

static VALUE
gdkgc_set_ts_origin(self, x, y)
    VALUE self, x, y;
{
    gdk_gc_set_ts_origin(get_gdkgc(self), NUM2INT(x), NUM2INT(y)); 
    return self;
}

/* Init */
extern void Init_gtk_gdk_color();
extern void Init_gtk_gdk_const();
extern void Init_gtk_gdk_draw();
extern void Init_gtk_gdk_event();
#ifdef USE_XIM
extern void Init_gtk_gdk_im();
#endif /* USE_XIM */
extern void Init_gtk_gdk_pixmap();
extern void Init_gtk_gdk_region();
extern void Init_gtk_gdk_rgb();
extern void Init_gtk_gdk_window();
extern void Init_gtk_gdkkeysyms();

void
Init_gtk_gdk()
{
    gdk_object_list = rb_hash_new();
    rb_global_variable(&gdk_object_list);

    /* Gdk */
    mGdk = rb_define_module("Gdk");

    /* Gdk::Color, Gdk::Colormap, Gdk::Visual */
    Init_gtk_gdk_color();
    /* Gdk::Drawablem, Gdk::Segment */
    Init_gtk_gdk_draw();
    /* Gdk::Pixmap, Gdk::Bitmap */
    Init_gtk_gdk_pixmap();
    /* Gdk::Window, Gdk::WindowAttr */
    Init_gtk_gdk_window();
    /* Gdk::Point, Gdk::Rectangle, Gdk::Region */
    Init_gtk_gdk_region();
    /* Gdk::Cursor */
    gdkCursor = rb_define_class_under(mGdk, "Cursor", rb_cData);
    /* Gdk::Atom */
    gdkAtom = rb_define_class_under(mGdk, "Atom", rb_cData);
    /* events */
    Init_gtk_gdk_event();

    /*
     * GdkFont
     */
    gdkFont = rb_define_class_under(mGdk, "Font", rb_cData);
    rb_define_singleton_method(gdkFont, "font_load", gdkfnt_load_font, 1);
    rb_define_singleton_method(gdkFont, "fontset_load", gdkfnt_load_fontset, 1);
    rb_define_method(gdkFont, "string_width", gdkfnt_string_width, 1);
    rb_define_method(gdkFont, "string_height", gdkfnt_string_height, 1);
    rb_define_method(gdkFont, "font_type", gdkfnt_type, 0);
    rb_define_method(gdkFont, "ascent", gdkfnt_ascent, 0);
    rb_define_method(gdkFont, "descent", gdkfnt_descent, 0);
    rb_define_method(gdkFont, "==", gdkfnt_equal, 1);
    rb_define_method(gdkFont, "string_extents", gdkfnt_string_extents, 1);

    /*
     * Gdk::GC
     */
    gdkGC = rb_define_class_under(mGdk, "GC", rb_cData);
    rb_define_singleton_method(gdkGC, "new", gdkgc_s_new, 1);
    rb_define_method(gdkGC, "copy", gdkgc_copy, 1);
    rb_define_method(gdkGC, "destroy", gdkgc_destroy, 0);
    rb_define_method(gdkGC, "set_function", gdkgc_set_function, 1);
    rb_define_method(gdkGC, "set_foreground", gdkgc_set_foreground, 1);
    rb_define_method(gdkGC, "set_background", gdkgc_set_background, 1);
    rb_define_method(gdkGC, "set_clip_mask", gdkgc_set_clip_mask, 1);
    rb_define_method(gdkGC, "set_clip_origin", gdkgc_set_clip_origin, 2);
    rb_define_method(gdkGC, "set_clip_rectangle", gdkgc_set_clip_rectangle, 1);
    rb_define_method(gdkGC, "set_clip_region", gdkgc_set_clip_region, 1);
    rb_define_method(gdkGC, "set_dashes", gdkgc_set_dashes, 2);
    rb_define_method(gdkGC, "set_exposures", gdkgc_set_exposures, 1);
    rb_define_method(gdkGC, "set_fill", gdkgc_set_fill, 1);
    rb_define_method(gdkGC, "set_font", gdkgc_set_font, 1);
    rb_define_method(gdkGC, "set_line_attributes", gdkgc_set_line_attributes, 4);
    rb_define_method(gdkGC, "set_stipple", gdkgc_set_stipple, 1);
    rb_define_method(gdkGC, "set_subwindow", gdkgc_set_subwindow, 1);
    rb_define_method(gdkGC, "set_tile", gdkgc_set_tile, 1);
    rb_define_method(gdkGC, "set_ts_origin", gdkgc_set_ts_origin, 2);

    /*
     * Gdk::GCValues
     */
    gdkGCValues = rb_define_class_under(mGdk, "GCValues", rb_cData);

    /*
     * Gdk::Image
     */
    gdkImage = rb_define_class_under(mGdk, "Image", rb_cData);
    rb_define_singleton_method(gdkImage, "new_bitmap", gdkimage_s_newbmap, 4);
    rb_define_singleton_method(gdkImage, "new", gdkimage_s_new, 4);
    rb_define_singleton_method(gdkImage, "get", gdkimage_s_get, 5);
    rb_define_method(gdkImage, "put_pixel", gdkimage_put_pixel, 3);
    rb_define_method(gdkImage, "get_pixel", gdkimage_get_pixel, 2);
    rb_define_method(gdkImage, "destroy", gdkimage_destroy, 0);

    /* Gdk::Rgb */
    Init_gtk_gdk_rgb();
#ifdef USE_XIM
    /* Gdk::IM, Gdk::IC, Gdk::ICAttr */
    Init_gtk_gdk_im();
#endif /* USE_XIM */
    /* Gdk module constants */
    Init_gtk_gdk_const();
    /* keysym constants */
    Init_gtk_gdkkeysyms();

}
