/********************************************************************************
*                                                                               *
*                                D i a l   W i d g e t                          *
*                                                                               *
*********************************************************************************
* Copyright (C) 1998 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* Contributed by: Guoqing Tian                                                  *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXDial.cpp,v 1.5 1999/11/05 18:06:36 jeroen Exp $                        *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXComposite.h"
#include "FXLabel.h"
#include "FXDial.h"


/*
  Notes:
  - Position decoupled from angle.
  - Add some API's.
  - Properly handle cyclic/non cyclic stuff.
  - Callbacks should report position in the void* ptr.
  - Keep notchangle>=0, as % of negative numbers is implementation defined.
  - Dial should work with double, not int.
*/

#define DIALWIDTH     12
#define DIALDIAMETER  40
#define NUMSIDECOLORS 16
#define DIAL_MASK     (DIAL_HORIZONTAL|DIAL_CYCLIC|DIAL_HAS_NOTCH)

/*******************************************************************************/

// Map
FXDEFMAP(FXDial) FXDialMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXDial::onPaint),
  FXMAPFUNC(SEL_MOTION,0,FXDial::onMotion),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXDial::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXDial::onLeftBtnRelease),
  FXMAPFUNC(SEL_UNGRABBED,0,FXDial::onUngrabbed),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_TIP,FXDial::onQueryTip),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_HELP,FXDial::onQueryHelp),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETVALUE,FXDial::onCmdSetValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETINTVALUE,FXDial::onCmdSetIntValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETREALVALUE,FXDial::onCmdSetRealValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETINTVALUE,FXDial::onCmdGetIntValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETREALVALUE,FXDial::onCmdGetRealValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETINTRANGE,FXDial::onCmdSetIntRange),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETINTRANGE,FXDial::onCmdGetIntRange),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETREALRANGE,FXDial::onCmdSetRealRange),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETREALRANGE,FXDial::onCmdGetRealRange),
  };


// Object implementation
FXIMPLEMENT(FXDial,FXFrame,FXDialMap,ARRAYNUMBER(FXDialMap))


// Make a window
FXDial::FXDial(FXComposite* p,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h,FXint pl,FXint pr,FXint pt,FXint pb):
  FXFrame(p,opts,x,y,w,h,pl,pr,pt,pb){
  flags|=FLAG_ENABLED;
  target=tgt;
  message=sel;
  range[0]=0;
  range[1]=360;
  notchangle=0;
  notchspacing=90;
  notchoffset=0;
  notchColor=FXRGB(255,128,0);
  dragpoint=0;
  dragpos=0;
  incr=360;
  pos=0;
  }


// Get minimum width
FXint FXDial::getDefaultWidth(){
  FXint w;
  if(options&DIAL_HORIZONTAL) w=DIALDIAMETER; else w=DIALWIDTH;
  return w+padleft+padright+(border<<1); 
  }


// Get minimum height
FXint FXDial::getDefaultHeight(){ 
  FXint h;
  if(options&DIAL_HORIZONTAL) h=DIALWIDTH; else h=DIALDIAMETER; 
  return h+padtop+padbottom+(border<<1); 
  }


// We were asked about status text
long FXDial::onQueryHelp(FXObject* sender,FXSelector,void*){
  if(!help.empty() && (flags&FLAG_HELP)){ 
    sender->handle(this,MKUINT(ID_SETSTRINGVALUE,SEL_COMMAND),&help);
    return 1;
    }
  return 0;
  }


// We were asked about tip text
long FXDial::onQueryTip(FXObject* sender,FXSelector,void*){
  if(!tip.empty() && (flags&FLAG_TIP)){
    sender->handle(this,MKUINT(ID_SETSTRINGVALUE,SEL_COMMAND),&tip);
    return 1;
    }
  return 0;
  }


// Update value from a message
long FXDial::onCmdSetValue(FXObject*,FXSelector,void* ptr){
  setPosition((FXint)(long)ptr);
  return 1;
  }


// Update value from a message
long FXDial::onCmdSetIntValue(FXObject*,FXSelector,void* ptr){
  setPosition(*((FXint*)ptr));
  return 1;
  }


// Update value from a message
long FXDial::onCmdSetRealValue(FXObject*,FXSelector,void* ptr){
  setPosition((FXint)*((FXdouble*)ptr));
  return 1;
  }


// Obtain value from text field
long FXDial::onCmdGetIntValue(FXObject*,FXSelector,void* ptr){
  *((FXint*)ptr)=getPosition();
  return 1;
  }


// Obtain value from text field
long FXDial::onCmdGetRealValue(FXObject*,FXSelector,void* ptr){
  *((FXdouble*)ptr) = (FXdouble)getPosition();
  return 1;
  }


// Update range from a message
long FXDial::onCmdSetIntRange(FXObject*,FXSelector,void* ptr){
  setRange(((FXint*)ptr)[0],((FXint*)ptr)[1]);
  return 1;
  }


// Get range with a message
long FXDial::onCmdGetIntRange(FXObject*,FXSelector,void* ptr){
  ((FXint*)ptr)[0]=range[0];
  ((FXint*)ptr)[1]=range[1];
  return 1;
  }


// Update range from a message
long FXDial::onCmdSetRealRange(FXObject*,FXSelector,void* ptr){
  setRange((FXint) ((FXdouble*)ptr)[0],(FXint) ((FXdouble*)ptr)[1]);
  return 1;
  }


// Get range with a message
long FXDial::onCmdGetRealRange(FXObject*,FXSelector,void* ptr){
  ((FXdouble*)ptr)[0]=(FXdouble)range[0];
  ((FXdouble*)ptr)[1]=(FXdouble)range[1];
  return 1;
  }


// Pressed LEFT button
long FXDial::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent *event=(FXEvent*)ptr;
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONPRESS),ptr)) return 1;
    if(options&DIAL_HORIZONTAL) 
      dragpoint=event->win_x; 
    else 
      dragpoint=event->win_y;
    dragpos=pos;
    flags|=FLAG_PRESSED;
    flags&=~FLAG_UPDATE;
    return 1;
    }
  return 0;
  }


// Released LEFT button
long FXDial::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    ungrab();
    flags|=FLAG_UPDATE;
    flags&=~FLAG_PRESSED;
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr)) return 1;
    if(flags&FLAG_CHANGED){
      if(target) target->handle(this,MKUINT(message,SEL_COMMAND),(void*)pos);
      }
    flags&=~FLAG_CHANGED;
    return 1;
    }
  return 0;
  }


// The widget lost the grab for some reason
long FXDial::onUngrabbed(FXObject* sender,FXSelector sel,void* ptr){
  FXFrame::onUngrabbed(sender,sel,ptr);
  flags&=~FLAG_PRESSED;
  flags&=~FLAG_CHANGED;
  flags|=FLAG_UPDATE;
  return 1;
  }


// Moving 
long FXDial::onMotion(FXObject*,FXSelector,void* ptr){
  FXEvent *event=(FXEvent*)ptr;
  FXint travel,size,delta,newpos,tmp;
  if(flags&FLAG_PRESSED){
    if(options&DIAL_HORIZONTAL){
      size=width-(border<<1);
      travel=event->win_x-dragpoint;
      }
    else{
      size=height-(border<<1);
      travel=dragpoint-event->win_y;
      }
    if(travel){
      delta=(incr*travel)/(2*size);
      if(options&DIAL_CYCLIC){
        tmp=dragpos+delta-range[0];
        while(tmp<0) tmp+=(range[1]-range[0]+1);
        newpos=range[0]+tmp%(range[1]-range[0]+1);
        }
      else{
        if(dragpos+delta<range[0]) newpos=range[0];
        else if(dragpos+delta>range[1]) newpos=range[1];
        else newpos=dragpos+delta;
        }
      if(pos!=newpos){
        pos=newpos;
        FXASSERT(range[0]<=pos && pos<=range[1]);
        notchangle=(notchoffset+(3600*(pos-range[0]))/incr)%3600;
        update(border+padleft+1,border+padtop+1,width-(border<<1)-padleft-padright-2,height-(border<<1)-padtop-padbottom-2);
        flags|=FLAG_CHANGED;
        if(target) target->handle(this,MKUINT(message,SEL_CHANGED),(void*)pos);
        return 1;
        }
      }
    }
  return 0;
  }


// Handle repaint 
long FXDial::onPaint(FXObject*,FXSelector,void* ptr){
  const FXdouble fac=0.5*PI/((FXdouble)(NUMSIDECOLORS-1));
  FXEvent *ev=(FXEvent*)ptr;
  FXint i,size,u,d,lu,ld,t,r,fm,to,off,ang;
  FXuint rmax,gmax,bmax,red,green,blue;
  FXint lt,rt,tp,bm;
  FXdouble mid,tmp;
  FXDCWindow dc(this,ev);
  
  // Paint background
  dc.setForeground(backColor);
  dc.fillRectangle(ev->rect.x,ev->rect.y,ev->rect.w,ev->rect.h);
  
  FXASSERT(0<=notchangle && notchangle<3600);
  off=(notchangle+3600)%notchspacing;
  fm=off/notchspacing;
  to=(off+1800-notchspacing+1)/notchspacing;
  
  // Rectangle of dial
  lt=border+padleft+1;
  rt=width-border-padright-2;
  tp=border+padtop+1;
  bm=height-border-padbottom-2;
  
  // Colors for sides
  rmax=(126*FXREDVAL(backColor))/100;
  gmax=(126*FXGREENVAL(backColor))/100;
  bmax=(126*FXBLUEVAL(backColor))/100;
  rmax=FXMIN(rmax,255);
  gmax=FXMIN(gmax,255);
  bmax=FXMIN(bmax,255);

  // Horizontal dial
  if(options&DIAL_HORIZONTAL){
    size=rt-lt;
    r=size/2-1;
    mid=0.5*(lt+rt);
    for(i=fm; i<=to; i++){
      ang=i*notchspacing+off;
      t=(FXint)(mid-r*cos(0.1*DTOR*ang));
      if((options&DIAL_HAS_NOTCH) && (ang+3600)%3600==notchangle){
        dc.setForeground(hiliteColor);
        dc.drawLine(t-1,tp,t-1,bm);
        dc.setForeground(notchColor);
        dc.drawLine(t,tp,t,bm);
        dc.drawLine(t+1,tp,t+1,bm);
        dc.setForeground(borderColor);
        dc.drawLine(t+2,tp,t+2,bm);
        }
      else{
        if(ang<200){
          dc.setForeground(shadowColor);
          dc.drawLine(t,tp,t,bm);
          dc.setForeground(borderColor);
          dc.drawLine(t+1,tp,t+1,bm);
          }
        else if(ang<300){
          dc.setForeground(borderColor);
          dc.drawLine(t,tp,t,bm);
          }
        else if(ang<600){
          dc.setForeground(hiliteColor);
          dc.drawLine(t,tp,t,bm);
          dc.setForeground(borderColor);
          dc.drawLine(t+1,tp,t+1,bm);
          }
        else if(ang<1200){
          dc.setForeground(hiliteColor);
          dc.drawLine(t-1,tp,t-1,bm);
          dc.drawLine(t,tp,t,bm);
          dc.setForeground(borderColor);
          dc.drawLine(t+1,tp,t+1,bm);
          }
        else if(ang<1500){
          dc.setForeground(hiliteColor);
          dc.drawLine(t,tp,t,bm);
          dc.setForeground(borderColor);
          dc.drawLine(t+1,tp,t+1,bm);
          }
        else if(ang<1600){
          dc.setForeground(borderColor);
          dc.drawLine(t,tp,t,bm);
          }
        else{
          dc.setForeground(shadowColor);
          dc.drawLine(t,tp,t,bm);
          dc.setForeground(borderColor);
          dc.drawLine(t-1,tp,t-1,bm);
          }
        }
      }
    dc.drawLine(lt,tp,lt,bm);
    dc.drawLine(rt,tp,rt,bm);
    lu=lt;
    ld=rt;
    for(i=0; i<NUMSIDECOLORS; i++){
      tmp=r*cos(fac*i);
      u=(FXint)(mid-tmp);
      d=(FXint)(mid+tmp);
      red=(rmax*i)/(NUMSIDECOLORS-1);
      green=(gmax*i)/(NUMSIDECOLORS-1);
      blue=(bmax*i)/(NUMSIDECOLORS-1);
      dc.setForeground(FXRGB(red,green,blue));
      dc.drawLine(lu,tp,u,tp);
      dc.drawLine(ld,tp,d,tp);
      dc.drawLine(lu,bm,u,bm);
      dc.drawLine(ld,bm,d,bm);
      lu=u;
      ld=d;
      }
    dc.drawLine(lu,tp,ld,tp);
    dc.drawLine(lu,bm,ld,bm);
    }
  
  // Vertical dial
  else{
    size=bm-tp;
    r=size/2-1;
    mid=0.5*(tp+bm);
    for(i=fm; i<=to; i++){
      ang=i*notchspacing+off;
      t=(FXint)(mid+r*cos(0.1*DTOR*ang));
      if((options&DIAL_HAS_NOTCH) && (ang+3600)%3600==notchangle){
        dc.setForeground(hiliteColor);
        dc.drawLine(lt,t-1,rt,t-1);
        dc.setForeground(notchColor);
        dc.drawLine(lt,t,rt,t);
        dc.drawLine(lt,t+1,rt,t+1);
        dc.setForeground(borderColor);
        dc.drawLine(lt,t+2,rt,t+2);
        }
      else{
        if(ang<200){
          dc.setForeground(borderColor);
          dc.drawLine(lt,t,rt,t);
          dc.setForeground(shadowColor);
          dc.drawLine(lt,t-1,rt,t-1);
          }
        else if(ang<300){
          dc.setForeground(borderColor);
          dc.drawLine(lt,t,rt,t);
          }
        else if(ang<600){
          dc.setForeground(hiliteColor);
          dc.drawLine(lt,t,rt,t);
          dc.setForeground(borderColor);
          dc.drawLine(lt,t+1,rt,t+1);
          }
        else if(ang<1200){
          dc.setForeground(hiliteColor);
          dc.drawLine(lt,t-1,rt,t-1);
          dc.drawLine(lt,t,rt,t);
          dc.setForeground(borderColor);
          dc.drawLine(lt,t+1,rt,t+1);
          }
        else if(ang<1500){
          dc.setForeground(hiliteColor);
          dc.drawLine(lt,t,rt,t);
          dc.setForeground(borderColor);
          dc.drawLine(lt,t+1,rt,t+1);
          }
        else if(ang<1600){
          dc.setForeground(borderColor);
          dc.drawLine(lt,t,rt,t);
          }
        else{
          dc.setForeground(borderColor);
          dc.drawLine(lt,t,rt,t);
          dc.setForeground(shadowColor);
          dc.drawLine(lt,t+1,rt,t+1);
          }
        }
      }
    dc.drawLine(lt,tp,rt,tp);
    dc.drawLine(lt,bm,rt,bm);
    lu=tp;
    ld=bm;
    for(i=0; i<NUMSIDECOLORS; i++){
      tmp=r*cos(fac*i);
      u=(FXint)(mid-tmp);
      d=(FXint)(mid+tmp);
      red=(rmax*i)/(NUMSIDECOLORS-1);
      green=(gmax*i)/(NUMSIDECOLORS-1);
      blue=(bmax*i)/(NUMSIDECOLORS-1);
      dc.setForeground(FXRGB(red,green,blue));
      dc.drawLine(lt,lu,lt,u);
      dc.drawLine(lt,ld,lt,d);
      dc.drawLine(rt,lu,rt,u);
      dc.drawLine(rt,ld,rt,d);
      lu=u;
      ld=d;
      }
    dc.drawLine(lt,lu,lt,ld);
    dc.drawLine(rt,lu,rt,ld);
    }
  
  // Border
  drawFrame(dc,0,0,width,height);

  // Inner rectangle
  dc.setForeground(shadowColor);
  dc.drawRectangle(lt-1,tp-1,rt-lt+2,bm-tp+2);
  return 1;
  }


// Set dial range  
void FXDial::setRange(FXint lo,FXint hi){
  if(lo>hi){ fxerror("%s::setRange: trying to set negative range.\n",getClassName()); }
  if(range[0]!=lo || range[1]!=hi){
    range[0]=lo;
    range[1]=hi;
    if(pos<range[0]) pos=range[0];
    if(pos>range[1]) pos=range[1];
    notchangle=(notchoffset+(3600*(pos-range[0]))/incr)%3600;
    update();
    }
  }


// Set dial position
void FXDial::setPosition(FXint p){
  if(p<range[0]) p=range[0];
  if(p>range[1]) p=range[1];
  if(p!=pos){
    pos=p;
    notchangle=(notchoffset+(3600*(pos-range[0]))/incr)%3600;
    update();
    }
  }


// Change increment, i.e. the amount of pos change per revolution
void FXDial::setRevolutionIncrement(FXint i){
  incr=FXMAX(1,i);
  notchangle=(notchoffset+(3600*(pos-range[0]))/incr)%3600;
  update();
  }


// Change notch spacing
void FXDial::setNotchSpacing(FXint spacing){
  if(spacing<1) spacing=1;
  if(spacing>3600) spacing=3600;
  while(3600%spacing) spacing--;    // Should be a divisor of 3600
  if(notchspacing!=spacing){
    notchspacing=spacing;
    update();
    }
  }


// Change notch offset
void FXDial::setNotchOffset(FXint offset){
  if(offset>3600) offset=3600;
  if(offset<-3600) offset=-3600;
  offset=(offset+3600)%3600;
  if(offset!=notchoffset){
    notchangle=(notchoffset+(3600*(pos-range[0]))/incr)%3600;
    update();
    }
  }


// Get dial options
FXuint FXDial::getDialStyle() const {
  return (options&DIAL_MASK); 
  }


// Set dial options
void FXDial::setDialStyle(FXuint style){
  FXuint opts=(options&~DIAL_MASK) | (style&DIAL_MASK);
  if(options!=opts){
    options=opts;
    recalc();
    }
  }


// Save object to stream
void FXDial::save(FXStream& store) const {
  FXFrame::save(store);
  store << range[0] << range[1];
  store << notchColor;
  store << notchangle;
  store << notchspacing;
  store << notchoffset;
  store << incr;
  store << pos;
  store << help;
  store << tip;
  }


// Load object from stream
void FXDial::load(FXStream& store){
  FXFrame::load(store);
  store >> range[0] >> range[1];
  store >> notchColor;
  store >> notchangle;
  store >> notchspacing;
  store >> notchoffset;
  store >> incr;
  store >> pos;
  store >> help;
  store >> tip;
  }  


// Change help text
void FXDial::setHelpText(const FXString& text){
  help=text;
  }


// Change tip text
void FXDial::setTipText(const FXString& text){
  tip=text;
  }


