/********************************************************************************
*                                                                               *
*                          I c o n L i s t   O b j e c t                        *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXIconList.cpp,v 1.7 1999/11/16 18:29:26 jeroen Exp $                    *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "fxkeys.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXFont.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXLabel.h"
#include "FXButton.h"
#include "FXComposite.h"
#include "FXScrollbar.h"
#include "FXScrollWindow.h"
#include "FXHeader.h"
#include "FXIconList.h"


/*
  To do:
  - itemWidth should ALWAYS reflect widest item.
  - item->getWidth() returns actual item width.
  - In detail-mode, some items should be left, some right, and some centered in the field.
  - Typing first few letters of an item should make that item current.
  - Maybe need item-by-name access?
  - Return key simulates double click.
  - Need method to set header columns.
  - Sortfunc's will be hard to serialize, and hard to write w/o secretly #including
    the FXTreeItem header!
  - Rapid key actions are wrongly interpreted as double clicks
*/



#define SIDE_SPACING             4    // Left or right spacing between items

#define DETAIL_LINE_SPACING      1    // Line spacing in detail mode
#define DETAIL_TEXT_SPACING      2    // Spacing between text and icon in detail icon mode

#define MINI_LINE_SPACING        1    // Line spacing in mini icon mode
#define MINI_TEXT_SPACING        2    // Spacing between text and icon in mini icon mode

#define BIG_LINE_SPACING         6    // Line spacing in big icon mode
#define BIG_TEXT_SPACING         2    // Spacing between text and icon in big icon mode

#define ITEM_SPACE             128    // Default space for item

#define ICONLIST_MASK (ICONLIST_MINI_ICONS|ICONLIST_BIG_ICONS|ICONLIST_COLUMNS)
#define SELECT_MASK   (ICONLIST_SINGLESELECT|ICONLIST_BROWSESELECT)


/*******************************************************************************/


// Object implementation
FXIMPLEMENT(FXIconItem,FXObject,NULL,0)


// Helper function
static inline FXint count(const FXchar* label){
  register FXint c=0;
  while(label[c] && label[c]!='\t') c++;
  return c;
  }



// Draw item
void FXIconItem::draw(const FXIconList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const {
  register FXuint options=list->getListStyle();
  if(options&ICONLIST_BIG_ICONS) drawBigIcon(list,dc,x,y,w,h);
  else if(options&ICONLIST_MINI_ICONS) drawMiniIcon(list,dc,x,y,w,h);
  else drawDetails(list,dc,x,y,w,h);
  }


// Draw dotted rectangle for focus
void FXIconItem::drawFocus(const FXIconList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const {
  dc.setFillStyle(FILL_OPAQUESTIPPLED);
  dc.setStipple(STIPPLE_GRAY);
  dc.setForeground(list->getTextColor());
  dc.setBackground(list->getBackColor());
  dc.drawRectangle(x+1,y+1,w-3,h-3);
  dc.setFillStyle(FILL_SOLID);
  dc.setStipple(STIPPLE_NONE);
  }


// Draw big icon
void FXIconItem::drawBigIcon(const FXIconList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const {
  register FXFont *font=list->getFont();
  register FXint iw=0,ih=0,tw=0,th=0;
  register FXint tlen,tdrw,dw,s,sp;
  y+=BIG_LINE_SPACING/2;
  if(bigIcon){
    iw=bigIcon->getWidth();
    ih=bigIcon->getHeight();
    }
  if(!label.empty()){
    tlen=count(label.text());
    tw=4+font->getTextWidth(label.text(),tlen);
    th=4+font->getFontHeight();
    }
  sp=w-SIDE_SPACING;
  if(bigIcon){
    if(isSelected()){
      dc.drawIconShaded(bigIcon,x+(w-iw)/2,y);
      }
    else{
      dc.drawIcon(bigIcon,x+(w-iw)/2,y);
      }
    y+=ih+BIG_TEXT_SPACING;
    }
  if(!label.empty() && tlen>0){
    tdrw=tlen;
    dw=0;
    if(tw>sp){
      dw=font->getTextWidth("...",3);
      s=sp-dw;
      while((tw=4+font->getTextWidth(label.text(),tdrw))>s && tdrw>1) --tdrw;
      if(tw>s) dw=0;
      }
    if(tw<=sp){
      x=x+(w-tw-dw)/2;
      if(isSelected()){
        dc.setForeground(list->getSelBackColor());
        dc.fillRectangle(x,y,tw+dw,th);
        dc.setForeground(list->getSelTextColor());
        }
      else{
        dc.setForeground(list->getTextColor());
        }
      dc.drawText(x+2,y+font->getFontAscent()+2,label.text(),tdrw);
      if(dw) dc.drawText(x+tw-2,y+font->getFontAscent()+2,"...",3);
      if(hasFocus()){
        drawFocus(list,dc,x,y,tw+dw,th);
        }
      }
    }
  }


// Draw mini icon
void FXIconItem::drawMiniIcon(const FXIconList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const {
  register FXFont *font=list->getFont();
  register FXint iw=0,ih=0,tw=0,th=0;
  register FXint tlen,tdrw,dw,s,sp;
  x+=SIDE_SPACING/2;
  y+=MINI_LINE_SPACING/2;
  if(miniIcon){
    iw=miniIcon->getWidth();
    ih=miniIcon->getHeight();
    }
  if(!label.empty()){
    tlen=count(label.text());
    tw=4+font->getTextWidth(label.text(),tlen);
    th=4+font->getFontHeight();
    }
  sp=w-SIDE_SPACING;
  if(miniIcon){
    if(isSelected()){
      dc.drawIconShaded(miniIcon,x,y+(h-ih)/2);
      }
    else{
      dc.drawIcon(miniIcon,x,y+(h-ih)/2);
      }
    x+=iw+MINI_TEXT_SPACING;
    sp-=iw+MINI_TEXT_SPACING;
    }
  if(!label.empty() && tlen>0){
    tdrw=tlen;
    dw=0;
    if(tw>sp){
      dw=font->getTextWidth("...",3);
      s=sp-dw;
      while((tw=4+font->getTextWidth(label.text(),tdrw))>s && tdrw>1) --tdrw;
      if(tw>s) dw=0;
      }
    if(tw<=sp){
      y+=(h-th)/2;
      if(isSelected()){
        dc.setForeground(list->getSelBackColor());
        dc.fillRectangle(x,y,tw+dw,th);
        dc.setForeground(list->getSelTextColor());
        }
      else{
        dc.setForeground(list->getTextColor());
        }
      dc.drawText(x+2,y+font->getFontAscent()+2,label.text(),tdrw);
      if(dw) dc.drawText(x+tw-2,y+font->getFontAscent()+2,"...",3);
      if(hasFocus()){
        drawFocus(list,dc,x,y,tw+dw,th);
        }
      }
    }
  }


// Draw detail
void FXIconItem::drawDetails(const FXIconList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const {
  register const FXchar *text=label.text();
  register FXFont *font=list->getFont();
  FXHeader *header=list->getHeader();
  register FXint iw=0,ih=0,tw=0,th=0;
  register FXint tlen,tdrw,ddw,dw,s,hi,space,tail;
  if(header->getNumItems()==0) return;
  if(miniIcon){
    iw=miniIcon->getWidth();
    ih=miniIcon->getHeight();
    }
  if(!label.empty()){
    th=font->getFontHeight();
    }
  if(isSelected()){
    dc.setForeground(list->getSelBackColor());
    dc.fillRectangle(x,y,header->getWidth(),h);
    }
  if(hasFocus()){
    drawFocus(list,dc,x,y,header->getWidth(),h);
    }
  x+=SIDE_SPACING/2;
  y+=DETAIL_LINE_SPACING/2;
  if(miniIcon){
    dc.drawIcon(miniIcon,x,y+(h-ih)/2);
    x+=iw+DETAIL_TEXT_SPACING;
    }
  if(text){
    ddw=font->getTextWidth("...",3);
    y+=(h-th-4)/2;
    if(isSelected())
      dc.setForeground(list->getSelTextColor());
    else
      dc.setForeground(list->getTextColor());
    tail=iw+DETAIL_TEXT_SPACING+SIDE_SPACING/2;
    for(hi=0; hi<header->getNumItems(); hi++){
      space=header->getItemSize(hi)-tail;
      tlen=count(text);
      if(tlen>0){
        tw=font->getTextWidth(text,tlen);
        tdrw=tlen;
        dw=0;
        if(tw>space-4){
          dw=ddw;
          s=space-4-dw;
          while((tw=font->getTextWidth(text,tdrw))>s && tdrw>1) --tdrw;
          if(tw>space-4) dw=0;
          }
        if(tw<=(space-4)){
          dc.drawText(x+2,y+font->getFontAscent()+2,text,tdrw);
          if(dw) dc.drawText(x+tw+2,y+font->getFontAscent()+2,"...",3);
          }
        }
      if(!text[tlen]) break;
      x+=space;
      text=text+tlen+1;
      tail=0;
      }
    }
  }


// See if item got hit and where: 0 is outside, 1 is icon, 2 is text
FXint FXIconItem::hitItem(const FXIconList* list,FXint rx,FXint ry,FXint rw,FXint rh) const {
  register FXuint options=list->getListStyle();
  register FXint iw=0,tw=0,ih=0,th=0,ix,iy,tx,ty,w,h,sp;
  register FXFont *font=list->getFont();
  //FXTRACE((100,"rx=%d ry=%d rw=%d rh=%d\n",rx,ry,rw,rh));
  if(options&ICONLIST_BIG_ICONS){
    if(bigIcon){
      iw=bigIcon->getWidth();
      ih=bigIcon->getHeight();
      }
    if(!label.empty()){
      tw=4+font->getTextWidth(label.text(),count(label.text()));
      th=4+font->getFontHeight();
      }
    sp=list->getItemSpace()-SIDE_SPACING;
    w=SIDE_SPACING+FXMAX(tw,iw); 
    if(tw>sp) tw=sp;
    h=BIG_LINE_SPACING+ih+th; 
    if(ih && th) h+=BIG_TEXT_SPACING;
    ix=(w-iw)/2;
    iy=BIG_LINE_SPACING/2;
    tx=(w-tw)/2;
    ty=h-th-BIG_LINE_SPACING/2;
    }
  else if(options&ICONLIST_MINI_ICONS){
    if(miniIcon){
      iw=miniIcon->getWidth();
      ih=miniIcon->getHeight();
      }
    if(!label.empty()){
      tw=4+font->getTextWidth(label.text(),count(label.text()));
      th=4+font->getFontHeight();
      }
    sp=list->getItemSpace()-SIDE_SPACING-iw;
    if(iw && tw) sp-=MINI_TEXT_SPACING;
    if(tw>sp) tw=sp;
    h=MINI_LINE_SPACING+FXMAX(th,ih);
    ix=SIDE_SPACING/2;
    tx=SIDE_SPACING/2; 
    if(iw) tx+=iw+MINI_TEXT_SPACING;
    iy=(h-ih)/2;
    ty=(h-th)/2;
    }
  else{
    if(miniIcon){
      iw=miniIcon->getWidth();
      ih=miniIcon->getHeight();
      }
    if(!label.empty()){
      tw=10000000;
      th=4+font->getFontHeight();
      }
    h=DETAIL_LINE_SPACING+FXMAX(th,ih);
    ix=SIDE_SPACING/2;
    tx=SIDE_SPACING/2; 
    if(iw) tx+=iw+DETAIL_TEXT_SPACING;
    iy=(h-ih)/2;
    ty=(h-th)/2;
    }
  
  //FXTRACE((100,"ix=%d iy=%d iw=%d ih=%d tx=%d ty=%d tw=%d th=%d\n",ix,iy,iw,ih,tx,ty,tw,th));
  
  // In icon?
  if(ix<=rx+rw && iy<=ry+rh && rx<ix+iw && ry<iy+ih) return 1;
    
  // In text?
  if(tx<=rx+rw && ty<=ry+rh && rx<tx+tw && ry<ty+th) return 2;
  
  // Outside
  return 0;
  }


// Create icon
void FXIconItem::create(){ 
  if(bigIcon) bigIcon->create(); 
  if(miniIcon) miniIcon->create(); 
  }


// Destroy icon
void FXIconItem::destroy(){ 
  if(bigIcon) bigIcon->destroy(); 
  if(miniIcon) miniIcon->destroy(); 
  }


// Detach from icon resource
void FXIconItem::detach(){ 
  if(bigIcon) bigIcon->detach(); 
  if(miniIcon) miniIcon->detach(); 
  }


// Get item width
FXint FXIconItem::getWidth(const FXIconList* list) const {
  register FXuint options=list->getListStyle();
  register FXint iw=0,tw=0,w=0;
  if(options&ICONLIST_BIG_ICONS){
    if(bigIcon) iw=bigIcon->getWidth();
    if(!label.empty()) tw=4+list->getFont()->getTextWidth(label.text(),count(label.text()));
    w=SIDE_SPACING+FXMAX(tw,iw);
    }
  else if(options&ICONLIST_MINI_ICONS){
    if(miniIcon) iw=miniIcon->getWidth();
    if(!label.empty()) tw=4+list->getFont()->getTextWidth(label.text(),count(label.text()));
    if(iw && tw) iw+=MINI_TEXT_SPACING;
    w=SIDE_SPACING+iw+tw;
    }
  else{
    w=1;
    }
  return w;
  }


// Get item height
FXint FXIconItem::getHeight(const FXIconList* list) const {
  register FXuint options=list->getListStyle();
  register FXint ih=0,th=0,h=0;
  if(options&ICONLIST_BIG_ICONS){
    if(bigIcon) ih=bigIcon->getHeight();
    if(!label.empty()) th=list->getFont()->getFontHeight()+4;
    if(ih && th) ih+=BIG_TEXT_SPACING;
    h=BIG_LINE_SPACING+ih+th;
    }
  else if(options&ICONLIST_MINI_ICONS){
    if(miniIcon) ih=miniIcon->getHeight();
    if(!label.empty()) th=list->getFont()->getFontHeight()+4;
    h=MINI_LINE_SPACING+FXMAX(ih,th);
    }
  else{
    if(miniIcon) ih=miniIcon->getHeight();
    if(!label.empty()) th=list->getFont()->getFontHeight()+4;
    h=DETAIL_LINE_SPACING+FXMAX(ih,th);
    }
  return h;
  }


// Save data
void FXIconItem::save(FXStream& store) const {
  FXObject::save(store);
  store << label;
  store << bigIcon;
  store << miniIcon;
  store << state;
  }


// Load data
void FXIconItem::load(FXStream& store){ 
  FXObject::load(store);
  store >> label;
  store >> bigIcon;
  store >> miniIcon;
  store >> state;
  }


/*******************************************************************************/

// Map
FXDEFMAP(FXIconList) FXIconListMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXIconList::onPaint),
  FXMAPFUNC(SEL_MOTION,0,FXIconList::onMotion),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXIconList::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXIconList::onLeftBtnRelease),
  FXMAPFUNC(SEL_TIMEOUT,FXWindow::ID_AUTOSCROLL,FXIconList::onAutoScroll),
  FXMAPFUNC(SEL_TIMEOUT,FXIconList::ID_TIPTIMER,FXIconList::onTipTimer),
  FXMAPFUNC(SEL_UNGRABBED,0,FXIconList::onUngrabbed),
  FXMAPFUNC(SEL_KEYPRESS,0,FXIconList::onKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,0,FXIconList::onKeyRelease),
  FXMAPFUNC(SEL_ENTER,0,FXIconList::onEnter),
  FXMAPFUNC(SEL_LEAVE,0,FXIconList::onLeave),
  FXMAPFUNC(SEL_FOCUSIN,0,FXIconList::onFocusIn),
  FXMAPFUNC(SEL_FOCUSOUT,0,FXIconList::onFocusOut),
  FXMAPFUNC(SEL_ACTIVATE,0,FXIconList::onActivate),
  FXMAPFUNC(SEL_DEACTIVATE,0,FXIconList::onDeactivate),
  FXMAPFUNC(SEL_CHANGED,0,FXIconList::onChanged),
  FXMAPFUNC(SEL_CLICKED,0,FXIconList::onClicked),
  FXMAPFUNC(SEL_DOUBLECLICKED,0,FXIconList::onDoubleClicked),
  FXMAPFUNC(SEL_TRIPLECLICKED,0,FXIconList::onTripleClicked),
  FXMAPFUNC(SEL_COMMAND,0,FXIconList::onCommand),
  FXMAPFUNC(SEL_SELECTION_LOST,0,FXIconList::onSelectionLost),
  FXMAPFUNC(SEL_SELECTION_GAINED,0,FXIconList::onSelectionGained),
  FXMAPFUNC(SEL_CHANGED,FXIconList::ID_HEADER_CHANGE,FXIconList::onHeaderChanged),
  FXMAPFUNC(SEL_UPDATE,FXIconList::ID_SHOW_DETAILS,FXIconList::onUpdShowDetails),
  FXMAPFUNC(SEL_UPDATE,FXIconList::ID_SHOW_MINI_ICONS,FXIconList::onUpdShowMiniIcons),
  FXMAPFUNC(SEL_UPDATE,FXIconList::ID_SHOW_BIG_ICONS,FXIconList::onUpdShowBigIcons),
  FXMAPFUNC(SEL_UPDATE,FXIconList::ID_ARRANGE_BY_ROWS,FXIconList::onUpdArrangeByRows),
  FXMAPFUNC(SEL_UPDATE,FXIconList::ID_ARRANGE_BY_COLUMNS,FXIconList::onUpdArrangeByColumns),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_TIP,FXIconList::onQueryTip),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_HELP,FXIconList::onQueryHelp),
  FXMAPFUNC(SEL_COMMAND,FXIconList::ID_SHOW_DETAILS,FXIconList::onCmdShowDetails),
  FXMAPFUNC(SEL_COMMAND,FXIconList::ID_SHOW_MINI_ICONS,FXIconList::onCmdShowMiniIcons),
  FXMAPFUNC(SEL_COMMAND,FXIconList::ID_SHOW_BIG_ICONS,FXIconList::onCmdShowBigIcons),
  FXMAPFUNC(SEL_COMMAND,FXIconList::ID_ARRANGE_BY_ROWS,FXIconList::onCmdArrangeByRows),
  FXMAPFUNC(SEL_COMMAND,FXIconList::ID_ARRANGE_BY_COLUMNS,FXIconList::onCmdArrangeByColumns),
  FXMAPFUNC(SEL_COMMAND,FXIconList::ID_SELECT_ALL,FXIconList::onCmdSelectAll),
  FXMAPFUNC(SEL_COMMAND,FXIconList::ID_DESELECT_ALL,FXIconList::onCmdDeselectAll),
  FXMAPFUNC(SEL_COMMAND,FXIconList::ID_SELECT_INVERSE,FXIconList::onCmdSelectInverse),
  };


// Object implementation
FXIMPLEMENT(FXIconList,FXScrollArea,FXIconListMap,ARRAYNUMBER(FXIconListMap))


/*******************************************************************************/


// Serialization
FXIconList::FXIconList(){
  flags|=FLAG_ENABLED;
  header=(FXHeader*)-1;
  items=NULL;
  nitems=0;
  nrows=1;
  ncols=1;
  anchor=-1;
  current=-1;
  extent=-1;
  font=(FXFont*)-1;
  sortfunc=NULL;
  textColor=0;
  selbackColor=0;
  seltextColor=0;
  itemSpace=ITEM_SPACE;
  itemWidth=1;
  itemHeight=1;
  anchorx=0;
  anchory=0;
  currentx=0;
  currenty=0;
  timer=NULL;
  }


// Icon List
FXIconList::FXIconList(FXComposite *p,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
  FXScrollArea(p,opts,x,y,w,h){
  flags|=FLAG_ENABLED;
  header=new FXHeader(this,this,FXIconList::ID_HEADER_CHANGE,HEADER_TRACKING|HEADER_BUTTON|FRAME_RAISED|FRAME_THICK);
  target=tgt;
  message=sel;
  items=NULL;
  nrows=1;
  ncols=1;
  nitems=0;
  anchor=-1;
  current=-1;
  extent=-1;
  font=getApp()->normalFont;
  sortfunc=NULL;
  textColor=getApp()->foreColor;
  selbackColor=getApp()->selbackColor;
  seltextColor=getApp()->selforeColor;
  itemSpace=ITEM_SPACE;
  itemWidth=1;
  itemHeight=1;
  anchorx=0;
  anchory=0;
  currentx=0;
  currenty=0;
  timer=NULL;
  }


// Create window
void FXIconList::create(){
  register FXint i;
  FXScrollArea::create();
  for(i=0; i<nitems; i++){items[i]->create();}
  font->create();
  if(!deleteType){deleteType=getApp()->registerDragType(deleteTypeName);}
  if(!textType){textType=getApp()->registerDragType(textTypeName);}
  dropEnable();
  }


// Detach window
void FXIconList::detach(){
  register FXint i;
  FXScrollArea::detach();
  for(i=0; i<nitems; i++){items[i]->detach();}
  font->detach();
  deleteType=0;
  textType=0;
  }


// If window can have focus
FXbool FXIconList::canFocus() const { return 1; }

 
// Move content
void FXIconList::moveContents(FXint x,FXint y){
  FXint dx=x-pos_x;
  FXint dy=y-pos_y;
  FXint top=0;
  pos_x=x;
  pos_y=y;
  if(!(options&(ICONLIST_MINI_ICONS|ICONLIST_BIG_ICONS))){
    top=header->getDefaultHeight();
    header->move(x,0);
    }
  scroll(0,top,viewport_w,viewport_h,dx,dy);
  }


// Size of a possible column caption
FXint FXIconList::getViewportHeight(){
  return (options&(ICONLIST_MINI_ICONS|ICONLIST_BIG_ICONS)) ? height : height-header->getDefaultHeight();
  }


// Propagate size change
void FXIconList::recalc(){ 
  FXScrollArea::recalc();
  flags|=FLAG_RECALC;
  }


// Recompute interior 
void FXIconList::recompute(){
  register FXint w,h,i;
  
  itemWidth=1;
  itemHeight=1;

  // Measure the items
  for(i=0; i<nitems; i++){
    w=items[i]->getWidth(this);
    h=items[i]->getHeight(this);
    if(w>itemWidth) itemWidth=w;
    if(h>itemHeight) itemHeight=h;
    }

  // Adjust for detail mode
  if(!(options&(ICONLIST_MINI_ICONS|ICONLIST_BIG_ICONS))) itemWidth=header->getDefaultWidth();
  
  // Get number of rows or columns
  getrowscols(nrows,ncols,width,height);
  
  //FXTRACE((100,"%s::recompute: itemWidth=%d itemHeight=%d nrows=%d ncols=%d\n",getClassName(),itemWidth,itemHeight,nrows,ncols));
    
  // Done
  flags&=~FLAG_RECALC;
  }


// Determine number of columns and number of rows
void FXIconList::getrowscols(FXint& nr,FXint& nc,FXint w,FXint h) const {
  if(options&(ICONLIST_BIG_ICONS|ICONLIST_MINI_ICONS)){
    if(options&ICONLIST_COLUMNS){
      nc=w/itemSpace;
      if(nc<1) nc=1;
      nr=(nitems+nc-1)/nc;
      if(nr*itemHeight > h){
        nc=(w-vertical->getDefaultWidth())/itemSpace;
        if(nc<1) nc=1;
        nr=(nitems+nc-1)/nc;
        }
      if(nr<1) nr=1;
      }
    else{
      nr=h/itemHeight;
      if(nr<1) nr=1;
      nc=(nitems+nr-1)/nr;
      if(nc*itemSpace > w){
        nr=(h-horizontal->getDefaultHeight())/itemHeight;
        if(nr<1) nr=1;
        nc=(nitems+nr-1)/nr;
        }
      if(nc<1) nc=1;
      }
    }
  else{
    nr=nitems;
    nc=1;
    }
  }


// Determine content width of icon list
FXint FXIconList::getContentWidth(){
  if(flags&FLAG_RECALC) recompute();
  if(options&(ICONLIST_MINI_ICONS|ICONLIST_BIG_ICONS)) return ncols*itemSpace;
  return header->getDefaultWidth();
  }


// Determine content height of icon list
FXint FXIconList::getContentHeight(){
  if(flags&FLAG_RECALC) recompute();
  return nrows*itemHeight;
  }


// Recalculate layout
void FXIconList::layout(){
  FXint ww;
  
  // Update scroll bars
  FXScrollArea::layout();

  // In detail mode
  if(!(options&(ICONLIST_MINI_ICONS|ICONLIST_BIG_ICONS))){ 
    ww=header->getDefaultWidth();
    if(ww<viewport_w) ww=viewport_w;
    header->position(pos_x,0,ww,header->getDefaultHeight());
    header->show();
    }
  else{
    header->hide();
    }

  // Set line size
  vertical->setLine(itemHeight);
  horizontal->setLine(itemSpace);
  
  // Force repaint
  update();
  
  flags&=~FLAG_DIRTY;
  }


// Changed size:- this is a bit tricky, because
// we don't want to re-measure the items, but the content
// size has changed because the number of rows/columns has...
void FXIconList::resize(FXint w,FXint h){
  if(w!=width || h!=height) getrowscols(nrows,ncols,w,h);
  //FXTRACE((100,"%s::resize: nrows=%d ncols=%d\n",getClassName(),nrows,ncols));
  FXScrollArea::resize(w,h);
  }


// Changed size and/or pos:- this is a bit tricky, because
// we don't want to re-measure the items, but the content
// size has changed because the number of rows/columns has...
void FXIconList::position(FXint x,FXint y,FXint w,FXint h){
  if(w!=width || h!=height) getrowscols(nrows,ncols,w,h);
  //FXTRACE((100,"%s::position: nrows=%d ncols=%d\n",getClassName(),nrows,ncols));
  FXScrollArea::position(x,y,w,h);
  }


// Header subdivision has changed:- this is a bit tricky,
// we want to update the content size w/o re-measuring the items...
long FXIconList::onHeaderChanged(FXObject*,FXSelector,void*){
  flags&=~FLAG_RECALC;
  return 1;
  }


// Append header caption
void FXIconList::appendHeader(const FXString& text,FXIcon *icon,FXint size){
  header->appendItem(text,icon,size);
  }


// Remove header caption
void FXIconList::removeHeader(FXint index){
  if(index<0 || header->getNumItems()<=index){ fxerror("%s::removeHeader: index out of range.\n",getClassName()); } 
  header->removeItem(index);
  }


// Change header caption
void FXIconList::setHeaderText(FXint index,const FXString& text){
  if(index<0 || header->getNumItems()<=index){ fxerror("%s::setHeaderText: index out of range.\n",getClassName()); } 
  header->setItemText(index,text);
  }


// Get header caption
FXString FXIconList::getHeaderText(FXint index) const {
  if(index<0 || header->getNumItems()<=index){ fxerror("%s::getHeaderText: index out of range.\n",getClassName()); } 
  return header->getItemText(index);
  }


// Change header icon
void FXIconList::setHeaderIcon(FXint index,FXIcon *icon){
  if(index<0 || header->getNumItems()<=index){ fxerror("%s::setHeaderIcon: index out of range.\n",getClassName()); } 
  header->setItemIcon(index,icon);
  }


// Get header icon
FXIcon* FXIconList::getHeaderIcon(FXint index) const {
  if(index<0 || header->getNumItems()<=index){ fxerror("%s::getHeaderIcon: index out of range.\n",getClassName()); } 
  return header->getItemIcon(index);
  }


// Change header size
void FXIconList::setHeaderSize(FXint index,FXint size){
  if(index<0 || header->getNumItems()<=index){ fxerror("%s::setHeaderSize: index out of range.\n",getClassName()); } 
  header->setItemSize(index,size);
  }
  

// Get header size
FXint FXIconList::getHeaderSize(FXint index) const {
  if(index<0 || header->getNumItems()<=index){ fxerror("%s::getHeaderSize: index out of range.\n",getClassName()); } 
  return header->getItemSize(index);
  }


// Return number of headers
FXint FXIconList::getNumHeaders() const {
  return header->getNumItems();
  }

// Change item text
void FXIconList::setItemText(FXint index,const FXString& text){
  if(index<0 || nitems<=index){ fxerror("%s::setItemText: index out of range.\n",getClassName()); } 
  items[index]->label=text;
  recalc();
  }


// Get item text
FXString FXIconList::getItemText(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::getItemText: index out of range.\n",getClassName()); } 
  return items[index]->label;
  }


// Set item icon
void FXIconList::setItemBigIcon(FXint index,FXIcon* icon){
  if(index<0 || nitems<=index){ fxerror("%s::setItemBigIcon: index out of range.\n",getClassName()); } 
  items[index]->bigIcon=icon;
  recalc();
  }


// Get item icon
FXIcon* FXIconList::getItemBigIcon(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::getItemBigIcon: index out of range.\n",getClassName()); } 
  return items[index]->bigIcon;
  }


// Set item icon
void FXIconList::setItemMiniIcon(FXint index,FXIcon* icon){
  if(index<0 || nitems<=index){ fxerror("%s::setItemMiniIcon: index out of range.\n",getClassName()); } 
  items[index]->miniIcon=icon;
  recalc();
  }


// Get item icon
FXIcon* FXIconList::getItemMiniIcon(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::getItemMiniIcon: index out of range.\n",getClassName()); } 
  return items[index]->miniIcon;
  }


// Set item data
void FXIconList::setItemData(FXint index,void* ptr){
  if(index<0 || nitems<=index){ fxerror("%s::setItemData: index out of range.\n",getClassName()); } 
  items[index]->data=ptr;
  }


// Get item data
void* FXIconList::getItemData(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::getItemData: index out of range.\n",getClassName()); } 
  return items[index]->data;
  }


// True if item is selected
FXbool FXIconList::isItemSelected(FXint index) const { 
  if(index<0 || nitems<=index){ fxerror("%s::isItemSelected: index out of range.\n",getClassName()); } 
  return (items[index]->state&FXIconItem::SELECTED)!=0; 
  }


// True if item is current
FXbool FXIconList::isItemCurrent(FXint index) const { 
  if(index<0 || nitems<=index){ fxerror("%s::isItemCurrent: index out of range.\n",getClassName()); } 
  return index==current; 
  }


// True if item is enabled
FXbool FXIconList::isItemEnabled(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::isItemEnabled: index out of range.\n",getClassName()); } 
  return (items[index]->state&FXIconItem::DISABLED)==0; 
  }


// True if item (partially) visible
FXbool FXIconList::isItemVisible(FXint index) const {
  register FXbool vis=FALSE;
  register FXint x,y,hh;
  if(index<0 || nitems<=index){ fxerror("%s::isItemVisible: index out of range.\n",getClassName()); } 
  if(options&(ICONLIST_BIG_ICONS|ICONLIST_MINI_ICONS)){
    if(options&ICONLIST_COLUMNS){
      FXASSERT(ncols>0);
      x=pos_x+itemSpace*(index%ncols);
      y=pos_y+itemHeight*(index/ncols);
      }
    else{
      FXASSERT(nrows>0);
      x=pos_x+itemSpace*(index/nrows);
      y=pos_y+itemHeight*(index%nrows);
      }
    if(0<x+itemSpace && x<viewport_w && 0<y+itemHeight && y<viewport_h) vis=TRUE;
    }
  else{
    hh=header->getDefaultHeight();
    y=pos_y+hh+index*itemHeight;
    if(hh<y+itemHeight && y<viewport_h) vis=TRUE;
    }
  return vis; 
  }


// Make item fully visible
void FXIconList::makeItemVisible(FXint index){
  register FXint x,y,hh,px,py;
  if(xid && 0<=index && index<nitems){
    px=pos_x;
    py=pos_y;
    if(options&(ICONLIST_BIG_ICONS|ICONLIST_MINI_ICONS)){
      if(options&ICONLIST_COLUMNS){
        FXASSERT(ncols>0);
        x=itemSpace*(index%ncols);
        y=itemHeight*(index/ncols);
        }
      else{
        FXASSERT(nrows>0);
        x=itemSpace*(index/nrows);
        y=itemHeight*(index%nrows);
        }
      if(px+x+itemSpace >= viewport_w) px=viewport_w-x-itemSpace;
      if(px+x <= 0) px=-x;
      if(py+y+itemHeight >= viewport_h) py=viewport_h-y-itemHeight;
      if(py+y <= 0) py=-y;
      }
    else{
      hh=header->getDefaultHeight();
      y=hh+index*itemHeight;
      if(py+y+itemHeight >= viewport_h+hh) py=hh+viewport_h-y-itemHeight;
      if(py+y <= hh) py=hh-y;
      }
    setPosition(px,py);
    }
  }


// Get item at position x,y (NULL if none)
FXint FXIconList::getItemAt(FXint x,FXint y) const {
  register FXint ix,iy,w,h;
  register FXint r,c,index;
  y-=pos_y;
  x-=pos_x;
  if(options&(ICONLIST_BIG_ICONS|ICONLIST_MINI_ICONS)){
    c=x/itemSpace;
    if(c<0 || c>=ncols) return -1;
    r=y/itemHeight;
    if(r<0 || r>=nrows) return -1;
    index=(options&ICONLIST_COLUMNS) ? ncols*r+c : nrows*c+r;
    }
  else{
    y-=header->getDefaultHeight();
    c=0;
    r=y/itemHeight;
    if(r<0 || r>=nrows) return -1;
    index=r;
    }
  if(index<0 || index>=nitems) return -1;
  w=items[index]->getWidth(this);
  h=items[index]->getHeight(this);
  ix=itemSpace*c;
  iy=itemHeight*r;
  if(options&ICONLIST_BIG_ICONS){
    ix+=(itemSpace-w)/2;
    iy+=itemHeight-h;
    }
  else if(options&ICONLIST_MINI_ICONS){
    iy+=(itemHeight-h)/2;
    }
  else{
    iy+=(itemHeight-h)/2;
    }
  //FXTRACE((100,"maybe at index=%d itemx=%d itemy=%d itemw=%d itemh=%d\n",index,ix,iy,w,h));
  if(items[index]->hitItem(this,x-ix,y-iy)==0) return -1;
  return index;
  }


// Get item at position x,y
FXint FXIconList::findItem(const FXString& text,FXuint len) const {
  register FXint index=nitems-1;
  for(index=nitems-1; 0<=index; index--){
    if(equal(text,items[index]->label,len)) return index;   //// Perhaps we want to compare till the \t separator???
    }
  return index;
  }


// Did we hit the item, and which part of it did we hit
FXint FXIconList::hitItem(FXint index,FXint x,FXint y,FXint ww,FXint hh) const {
  FXint ix,iy,w,h,r,c,hit=0;
  if(0<=index && index<nitems){
    x-=pos_x;
    y-=pos_y;
    if(!(options&(ICONLIST_BIG_ICONS|ICONLIST_MINI_ICONS))) y-=header->getDefaultHeight();
    w=items[index]->getWidth(this);
    h=items[index]->getHeight(this);
    if(options&(ICONLIST_BIG_ICONS|ICONLIST_MINI_ICONS)){
      if(options&ICONLIST_COLUMNS){
        r=index/ncols;
        c=index%ncols;
        }
      else{
        c=index/nrows;
        r=index%nrows;
        }
      }
    else{
      r=index;
      c=0;
      }
    ix=itemSpace*c;
    iy=itemHeight*r;
    if(options&ICONLIST_BIG_ICONS){
      ix+=(itemSpace-w)/2;
      iy+=itemHeight-h;
      }
    else if(options&ICONLIST_MINI_ICONS){
      iy+=(itemHeight-h)/2;
      }
    else{
      iy+=(itemHeight-h)/2;
      }
    hit=items[index]->hitItem(this,x-ix,y-iy,ww,hh);
    }
  return hit;
  }



// Repaint
void FXIconList::updateItem(FXint index){
  if(xid && 0<=index && index<nitems){
    if(options&(ICONLIST_BIG_ICONS|ICONLIST_MINI_ICONS)){
      if(options&ICONLIST_COLUMNS){
        FXASSERT(ncols>0);
        update(pos_x+itemSpace*(index%ncols),pos_y+itemHeight*(index/ncols),itemSpace,itemHeight);
        }
      else{
        FXASSERT(nrows>0);
        update(pos_x+itemSpace*(index/nrows),pos_y+itemHeight*(index%nrows),itemSpace,itemHeight);
        }
      }
    else{
      update(pos_x,pos_y+header->getDefaultHeight()+index*itemHeight,content_w,itemHeight);
      }
    }
  }


// Enable one item
FXbool FXIconList::enableItem(FXint index){
  if(index<0 || nitems<=index){ fxerror("%s::enableItem: index out of range.\n",getClassName()); } 
  if(items[index]->state&FXIconItem::DISABLED){
    items[index]->state&=~FXIconItem::DISABLED;
    updateItem(index);
    return TRUE;
    }
  return FALSE;
  }


// Disable one item
FXbool FXIconList::disableItem(FXint index){
  if(index<0 || nitems<=index){ fxerror("%s::disableItem: index out of range.\n",getClassName()); } 
  if(!(items[index]->state&FXIconItem::DISABLED)){
    items[index]->state|=FXIconItem::DISABLED;
    updateItem(index);
    return TRUE;
    }
  return FALSE;
  }


// Select one item
FXbool FXIconList::selectItem(FXint index){
  if(index<0 || nitems<=index){ fxerror("%s::selectItem: index out of range.\n",getClassName()); } 
  if(!(items[index]->state&FXIconItem::SELECTED)){
    switch(options&SELECT_MASK){
      case ICONLIST_SINGLESELECT:
      case ICONLIST_BROWSESELECT:
        killSelection();
        items[index]->state|=FXIconItem::SELECTED;
        updateItem(index);
        break;
      case ICONLIST_EXTENDEDSELECT:
      case ICONLIST_MULTIPLESELECT:
        items[index]->state|=FXIconItem::SELECTED;
        updateItem(index);
        break;
      }
    return TRUE;
    }
  return FALSE;
  }


// Deselect one item
FXbool FXIconList::deselectItem(FXint index){
  if(index<0 || nitems<=index){ fxerror("%s::deselectItem: index out of range.\n",getClassName()); } 
  if(items[index]->state&FXIconItem::SELECTED){
    switch(options&SELECT_MASK){
      case ICONLIST_EXTENDEDSELECT:
      case ICONLIST_MULTIPLESELECT:
      case ICONLIST_SINGLESELECT:
        items[index]->state&=~FXIconItem::SELECTED;
        updateItem(index);
      case ICONLIST_BROWSESELECT:
        break;
      }
    return TRUE;
    }
  return FALSE;
  }


// Toggle one item
FXbool FXIconList::toggleItem(FXint index){
  if(index<0 || nitems<=index){ fxerror("%s::toggleItem: index out of range.\n",getClassName()); } 
  switch(options&SELECT_MASK){
    case ICONLIST_BROWSESELECT:
      if(!(items[index]->state&FXIconItem::SELECTED)){
        killSelection();
        items[index]->state|=FXIconItem::SELECTED;
        updateItem(index);
        }
      break;
    case ICONLIST_SINGLESELECT:
      if(!(items[index]->state&FXIconItem::SELECTED)){
        killSelection();
        }
      items[index]->state^=FXIconItem::SELECTED;
      updateItem(index);
      break;
    case ICONLIST_EXTENDEDSELECT:
    case ICONLIST_MULTIPLESELECT:
      items[index]->state^=FXIconItem::SELECTED;
      updateItem(index);
      break;
    }
  return TRUE;
  }


// Restore item to marked value
FXbool FXIconList::restoreItem(FXint index){
  if(items[index]->state&FXIconItem::MARKED){
    if(!(items[index]->state&FXIconItem::SELECTED)){
      items[index]->state|=FXIconItem::SELECTED;
      updateItem(index);
      return TRUE;
      }
    }
  else{
    if(items[index]->state&FXIconItem::SELECTED){
      items[index]->state&=~FXIconItem::SELECTED;
      updateItem(index);
      return TRUE;
      }
    }
  return FALSE;
  }


// Mark item so it may be restored
void FXIconList::markItem(FXint index){
  if(items[index]->state&FXIconItem::SELECTED)
    items[index]->state|=FXIconItem::MARKED;
  else
    items[index]->state&=~FXIconItem::MARKED;
  }


// Mark items so they may be restored
void FXIconList::markAllItems(){
  register FXint index;
  for(index=0; index<nitems; index++){
    markItem(index);
    }
  }


// Select items in rectangle
FXbool FXIconList::selectInRectangle(FXint x,FXint y,FXint w,FXint h){
  register FXint rlo,rhi,clo,chi,r,c,index;
  register FXbool changed=FALSE;
  if(options&(ICONLIST_BIG_ICONS|ICONLIST_MINI_ICONS)){
    rlo=(y-pos_y)/itemHeight;
    rhi=(y-pos_y+h-1)/itemHeight;
    if(rlo<0) rlo=0;
    if(rhi>=nrows) rhi=nrows-1;
    clo=(x-pos_x)/itemSpace;
    chi=(x-pos_x+w-1)/itemSpace;
    if(clo<0) clo=0;
    if(chi>=ncols) chi=ncols-1;
    FXTRACE((100,"rlo=%d rhi=%d clo=%d chi=%d\n",rlo,rhi,clo,chi));
    for(r=0; r<nrows; r++){
      for(c=0; c<ncols; c++){
        index=(options&ICONLIST_COLUMNS) ? ncols*r+c : nrows*c+r;
        if(0<=index && index<nitems){
          if(rlo<=r && r<=rhi && clo<=c && c<=chi && hitItem(index,x,y,w,h)){
            changed|=selectItem(index);
            }
          else{
            changed|=restoreItem(index);
            }
          }
        }
      }
    }
  else{
    rlo=(y-pos_y-header->getDefaultHeight())/itemHeight;
    rhi=(y-pos_y+h+itemHeight-header->getDefaultHeight()-2)/itemHeight;
    if(rlo<0) rlo=0;
    if(rhi>=nitems) rhi=nitems-1;
    for(index=0; index<nitems; index++){
      if(rlo<=index && index<=rhi && hitItem(index,x,y,w,h)){
        changed|=selectItem(index);
        }
      else{
        changed|=restoreItem(index);
        }
      }
    }
  return changed;
  }


// Start motion timer while in this window
long FXIconList::onEnter(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onEnter(sender,sel,ptr);
  if(!timer){timer=getApp()->addTimeout(getApp()->menuPause,this,ID_TIPTIMER);}
  return 1;
  }


// Stop motion timer when leaving window
long FXIconList::onLeave(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onLeave(sender,sel,ptr);
  if(timer){getApp()->removeTimeout(timer);timer=NULL;}
  return 1;
  }


// Gained focus
long FXIconList::onFocusIn(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusIn(sender,sel,ptr);
  if(0<=current){
    FXASSERT(current<nitems);
    items[current]->state|=FXIconItem::FOCUS;
    updateItem(current);
    }
  return 1;
  }


// Lost focus
long FXIconList::onFocusOut(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusOut(sender,sel,ptr);
  if(0<=current){
    FXASSERT(current<nitems);
    items[current]->state&=~FXIconItem::FOCUS;
    updateItem(current);
    }
  return 1;
  }


// We have the selection
long FXIconList::onSelectionGained(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onSelectionGained(sender,sel,ptr);
  return 1;
  }


// We lost the selection
long FXIconList::onSelectionLost(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onSelectionLost(sender,sel,ptr);
  killSelection();
  return 1;
  }


// Draw Lasso rectangle
void FXIconList::drawLasso(FXint x0,FXint y0,FXint x1,FXint y1){
  FXDCWindow dc(this);
  dc.setFunction(BLT_NOT_DST);
  x0+=pos_x;
  x1+=pos_x;
  y0+=pos_y;
  y1+=pos_y;
  dc.drawLine(x0,y0,x1,y0);
  dc.drawLine(x1,y0,x1,y1);
  dc.drawLine(x1,y1,x0,y1);
  dc.drawLine(x0,y1,x0,y0);
  }


// Draw item list
long FXIconList::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXDCWindow dc(this,event);
  register FXint x,y,w,h,r,c,index;
  register FXint rlo,rhi,clo,chi,yy,xx;

  // Icon mode
  if(options&(ICONLIST_BIG_ICONS|ICONLIST_MINI_ICONS)){
    
    // Exposed rows
    rlo=(event->rect.y-pos_y)/itemHeight;
    rhi=(event->rect.y+event->rect.h-pos_y)/itemHeight;
    if(rlo<0) rlo=0;
    if(rhi>=nrows) rhi=nrows-1;
 
    // Exposed columns
    clo=(event->rect.x-pos_x)/itemSpace;
    chi=(event->rect.x+event->rect.w-pos_x)/itemSpace;
    if(clo<0) clo=0;
    if(chi>=ncols) chi=ncols-1;

    // Big Icons
    if(options&ICONLIST_BIG_ICONS){
      for(r=rlo; r<=rhi; r++){
        y=pos_y+r*itemHeight;
        for(c=clo; c<=chi; c++){
          x=pos_x+c*itemSpace;
          index=(options&ICONLIST_COLUMNS) ? ncols*r+c : nrows*c+r;
          dc.setForeground(backColor);
          dc.fillRectangle(x,y,itemSpace,itemHeight);
          if(index<nitems){
            w=items[index]->getWidth(this);
            h=items[index]->getHeight(this);
            items[index]->draw(this,dc,x,y,itemSpace,itemHeight);
            }
          }
        }
      }
    
    // Mini icons
    else{
      for(r=rlo; r<=rhi; r++){
        y=pos_y+r*itemHeight;
        for(c=clo; c<=chi; c++){
          x=pos_x+c*itemSpace;
          index=(options&ICONLIST_COLUMNS) ? ncols*r+c : nrows*c+r;
          dc.setForeground(backColor);
          dc.fillRectangle(x,y,itemSpace,itemHeight);
          if(index<nitems){
            w=items[index]->getWidth(this);
            h=items[index]->getHeight(this);
            items[index]->draw(this,dc,x,y,itemSpace,itemHeight);
            }
          }
        }
      }
    
    // Repaint left-over background
    yy=(rhi+1)*itemHeight;
    if(yy<event->rect.y+event->rect.h){
      dc.setForeground(backColor);
      dc.fillRectangle(event->rect.x,yy,event->rect.w,event->rect.y+event->rect.h-yy);
      }
    xx=(chi+1)*itemSpace;
    if(xx<event->rect.x+event->rect.w){
      dc.setForeground(backColor);
      dc.fillRectangle(xx,event->rect.y,event->rect.x+event->rect.w-xx,event->rect.h);
      }
    }
  
  // Detail mode
  else{
    
    // Exposed rows
    rlo=(event->rect.y-pos_y-header->getDefaultHeight())/itemHeight;
    rhi=(event->rect.y+event->rect.h-pos_y-header->getDefaultHeight())/itemHeight;
    if(rlo<0) rlo=0;
    if(rhi>=nitems) rhi=nitems-1;
    
    // Repaint the items
    y=pos_y+rlo*itemHeight+header->getDefaultHeight();
    for(index=rlo; index<=rhi; index++,y+=itemHeight){
      dc.setForeground(backColor);
      dc.fillRectangle(pos_x,y,content_w,itemHeight);
      h=items[index]->getHeight(this);
      items[index]->draw(this,dc,pos_x,y,content_w,itemHeight);
      }
    
    // Repaint left-over background
    if(y<event->rect.y+event->rect.h){
      dc.setForeground(backColor);
      dc.fillRectangle(event->rect.x,y,event->rect.w,event->rect.y+event->rect.h-y);
      }
    }
  return 1;
  }


// We were asked about tip text
long FXIconList::onQueryTip(FXObject* sender,FXSelector,void*){
  FXint x,y; FXuint state;
  if(flags&FLAG_TIP){
    getCursorPosition(x,y,state);
    FXTRACE((250,"%s::onQueryTip %08x (%d,%d)\n",getClassName(),this,x,y));
    FXint index=getItemAt(x,y);
    if(0<=index){
      FXString string=items[index]->label.extract(0,'\t');
      sender->handle(this,MKUINT(ID_SETSTRINGVALUE,SEL_COMMAND),(void*)&string);
      return 1;
      }
    }
  return 0;
  }


// We were asked about status text
long FXIconList::onQueryHelp(FXObject* sender,FXSelector,void*){
  if(!help.empty() && (flags&FLAG_HELP)){
    FXTRACE((250,"%s::onQueryHelp %08x\n",getClassName(),this));
    sender->handle(this,MKUINT(ID_SETSTRINGVALUE,SEL_COMMAND),(void*)&help);
    return 1;
    }
  return 0;
  }


// Arrange by rows
long FXIconList::onCmdArrangeByRows(FXObject*,FXSelector,void*){
  options&=~ICONLIST_COLUMNS;
  recalc();
  return 1;
  }


// Update sender
long FXIconList::onUpdArrangeByRows(FXObject* sender,FXSelector,void* ptr){
  FXuint msg=(options&ICONLIST_COLUMNS)?ID_UNCHECK:ID_CHECK;
  sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
  return 1;
  }


// Arrange by columns
long FXIconList::onCmdArrangeByColumns(FXObject*,FXSelector,void*){
  options|=ICONLIST_COLUMNS;
  recalc();
  return 1;
  }


// Update sender
long FXIconList::onUpdArrangeByColumns(FXObject* sender,FXSelector,void* ptr){
  FXuint msg=(options&ICONLIST_COLUMNS)?ID_CHECK:ID_UNCHECK;
  sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
  return 1;
  }


// Show detailed list
long FXIconList::onCmdShowDetails(FXObject*,FXSelector,void*){
  options&=~ICONLIST_MINI_ICONS;
  options&=~ICONLIST_BIG_ICONS;
  recalc();
  return 1;
  }


// Update sender
long FXIconList::onUpdShowDetails(FXObject* sender,FXSelector,void* ptr){
  FXuint msg=(options&(ICONLIST_MINI_ICONS|ICONLIST_BIG_ICONS))?ID_UNCHECK:ID_CHECK;
  sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
  return 1;
  }


// Show big icons
long FXIconList::onCmdShowBigIcons(FXObject*,FXSelector,void*){
  options&=~ICONLIST_MINI_ICONS;
  options|=ICONLIST_BIG_ICONS;
  recalc();
  return 1;
  }


// Update sender
long FXIconList::onUpdShowBigIcons(FXObject* sender,FXSelector,void* ptr){
  FXuint msg=(options&ICONLIST_BIG_ICONS)?ID_CHECK:ID_UNCHECK;
  sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
  return 1;
  }


// Show small icons
long FXIconList::onCmdShowMiniIcons(FXObject*,FXSelector,void*){
  options|=ICONLIST_MINI_ICONS;
  options&=~ICONLIST_BIG_ICONS;
  recalc();
  return 1;
  }


// Update sender
long FXIconList::onUpdShowMiniIcons(FXObject* sender,FXSelector,void* ptr){
  FXuint msg=(options&ICONLIST_MINI_ICONS)?ID_CHECK:ID_UNCHECK;
  sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
  return 1;
  }


// Select all items
long FXIconList::onCmdSelectAll(FXObject*,FXSelector,void*){
  for(int i=0; i<nitems; i++) selectItem(i);
  return 1;
  }


// Deselect all items
long FXIconList::onCmdDeselectAll(FXObject*,FXSelector,void*){
  for(int i=0; i<nitems; i++) deselectItem(i);
  return 1;
  }


// Select inverse of current selection
long FXIconList::onCmdSelectInverse(FXObject*,FXSelector,void*){
  for(int i=0; i<nitems; i++) toggleItem(i);
  return 1;
  }


// Mark and select
FXbool FXIconList::mark(FXint fm,FXint to,FXuint sel){
  register FXbool changes=FALSE;
  register FXint i;
  if(sel){
    for(i=fm; i<=to; i++){
      if(!(items[i]->state&FXIconItem::SELECTED)){
        items[i]->state&=~FXIconItem::MARKED;
        if(!(items[i]->state&FXIconItem::DISABLED)){
          items[i]->state|=FXIconItem::SELECTED;
          updateItem(i);
          changes=TRUE;
          }
        }
      else{
        items[i]->state|=FXIconItem::MARKED;
        }
      }
    }
  else{
    for(i=fm; i<=to; i++){
      if(items[i]->state&FXIconItem::SELECTED){
        items[i]->state|=FXIconItem::MARKED;
        if(!(items[i]->state&FXIconItem::DISABLED)){
          items[i]->state&=~FXIconItem::SELECTED;
          updateItem(i);
          changes=TRUE;
          }
        }
      else{
        items[i]->state&=~FXIconItem::MARKED;
        }
      }
    }
  return changes;
  }


// Restore to mark
FXbool FXIconList::restore(FXint fm,FXint to){
  register FXbool changes=FALSE;
  register FXint i;
  for(i=fm; i<=to; i++){ changes|=restoreItem(i); }
  return changes;
  }



// Extend selection
FXbool FXIconList::extendSelection(FXint index){
  register FXbool changes=FALSE;
  register FXuint sel;
  if(0<=index && 0<=anchor){
    sel=items[anchor]->state&FXIconItem::SELECTED;
    if(extent<anchor){
      if(index<extent){                     // index < extent < anchor
        changes|=mark(index,extent-1,sel);
        }
      else if(anchor<index){                // extent < anchor < index
        changes|=restore(extent,anchor-1);
        changes|=mark(anchor+1,index,sel);
        }
      else{                                 // extent <= index <= anchor
        changes|=restore(extent,index-1);
        }
      }
    else{
      if(extent<index){                     // anchor <= extent < index
        changes|=mark(extent+1,index,sel);
        }
      else if(index<anchor){                // index < anchor <= extent
        changes|=restore(anchor+1,extent);
        changes|=mark(index,anchor-1,sel);
        }
      else{                                 // anchor <= index <= extent
        changes|=restore(index+1,extent);
        }
      }
    extent=index;
    }
  return changes;
  }


// Kill selection
FXbool FXIconList::killSelection(){
  register FXbool changes=FALSE;
  register FXint i;
  for(i=0; i<nitems; i++){
    if(items[i]->state&FXIconItem::SELECTED){
      items[i]->state&=~FXIconItem::SELECTED;
      updateItem(i);
      changes=TRUE;
      }
    }
  extent=anchor;
  return changes;
  }


// Sort the items based on the sort function
void FXIconList::sortItems(){
  register FXIconItem *v;
  register FXint i,j,h;
  register FXbool exch=FALSE;
  if(sortfunc){
    for(h=1; h<=nitems/9; h=3*h+1);
    if(hasFocus() && 0<=current) items[current]->state&=~FXIconItem::FOCUS;
    for(; h>0; h/=3){
      for(i=h+1;i<=nitems;i++){
        v=items[i-1]; 
        j=i;
        while(j>h && sortfunc(items[j-h-1],v)){
          items[j-1]=items[j-h-1];
          exch=TRUE;
          j-=h;
          }
        items[j-1]=v;
        }
      }
    if(hasFocus() && 0<=current) items[current]->state|=FXIconItem::FOCUS;
    if(exch) update();
    }
  }


// Set current item
void FXIconList::setCurrentItem(FXint index){
  if(index<-1 || nitems<=index){ fxerror("%s::setCurrentItem: index out of range.\n",getClassName()); } 
  if(index!=current){
    
    // Deactivate old item
    if(0<=current){
      
      // No visible change if it doen't have the focus
      if(hasFocus()){
        items[current]->state&=~FXIconItem::FOCUS;
        updateItem(current);
        }
      }

    current=index;

    // Activate new item
    if(0<=current){

      // No visible change if it doen't have the focus
      if(hasFocus()){
        items[current]->state|=FXIconItem::FOCUS;
        updateItem(current);
        }
      }
    }

  // In browse select mode, select this item
  if((options&SELECT_MASK)==ICONLIST_BROWSESELECT){
    if(0<=current && !isItemSelected(current)){
      killSelection();
      selectItem(current);
      }
    }
  }


// Set anchor item
void FXIconList::setAnchorItem(FXint index){
  if(index<-1 || nitems<=index){ fxerror("%s::setAnchorItem: index out of range.\n",getClassName()); } 
  anchor=index;
  extent=index;
  }


// Key Press
long FXIconList::onKeyPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXint index;
  FXDragAction action;
  flags&=~FLAG_TIP;
  if(!isEnabled()) return 0;
  if(target && target->handle(this,MKUINT(message,SEL_KEYPRESS),ptr)) return 1;
  switch(event->code){
    case KEY_Right:
    case KEY_Left:
    case KEY_Up:
    case KEY_Down:
    case KEY_Home:
    case KEY_End:
      index=current;
      switch(event->code){
        case KEY_Right:
          if(!(options&(ICONLIST_BIG_ICONS|ICONLIST_MINI_ICONS))) return 0;
          if(options&ICONLIST_COLUMNS) index+=1; else index+=nrows;
          break;
        case KEY_Left:
          if(!(options&(ICONLIST_BIG_ICONS|ICONLIST_MINI_ICONS))) return 0;
          if(options&ICONLIST_COLUMNS) index-=1; else index-=nrows;
          break;
        case KEY_Up: 
          if(options&ICONLIST_COLUMNS) index-=ncols; else index-=1;
          break;
        case KEY_Down:
          if(options&ICONLIST_COLUMNS) index+=ncols; else index+=1;
          break;
        case KEY_Home: 
          index=0; 
          break;
        case KEY_End:  
          index=nitems-1; 
          break;
        }
      if(index<0) index=0;
      if(index>=nitems) index=nitems-1;
      makeItemVisible(index);
      handle(this,MKUINT(0,SEL_CHANGED),(void*)index);
      if(0<=index && index!=current){
        if(event->state&(SHIFTMASK|CONTROLMASK)){
          if((options&SELECT_MASK)==ICONLIST_EXTENDEDSELECT){
            extendSelection(index);
            }
          }
        }
      if(event->click_count==1){
        if((options&SELECT_MASK)==ICONLIST_BROWSESELECT){
          handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
          }
        }
      flags&=~FLAG_UPDATE;
      return 1;
    case KEY_space:
    case KEY_KP_Space:
      handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
      flags&=~FLAG_UPDATE;
      return 1;
    case KEY_Page_Up:
      setPosition(pos_x,pos_y+verticalScrollbar()->getPage());
      return 1;
    case KEY_Page_Down:
      setPosition(pos_x,pos_y-verticalScrollbar()->getPage());
      return 1;
    case KEY_Control_L:
    case KEY_Control_R:
      if(isDragging()){
        action=DRAG_COPY;
        if(didAccept()!=DRAG_REJECT){
          setDragCursor(getApp()->dndCopyCursor);
          }
        else{
          setDragCursor(getApp()->dontdropCursor);
          }
        }
      return 1;
    case KEY_Shift_L:
    case KEY_Shift_R:
      if(isDragging()){
        action=DRAG_MOVE;
        if(didAccept()!=DRAG_REJECT){
          setDragCursor(getApp()->dndMoveCursor);
          }
        else{
          setDragCursor(getApp()->dontdropCursor);
          }
        }
      return 1;
    case KEY_Alt_L:
    case KEY_Alt_R:
      if(isDragging()){
        action=DRAG_LINK;
        if(didAccept()!=DRAG_REJECT){
          setDragCursor(getApp()->dndLinkCursor);
          }
        else{
          setDragCursor(getApp()->dontdropCursor);
          }
        }
      return 1;
    }
  return 0;
  }


// Key Release 
long FXIconList::onKeyRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXDragAction action;
  if(!isEnabled()) return 0;
  flags|=FLAG_UPDATE;
  if(target && target->handle(this,MKUINT(message,SEL_KEYRELEASE),ptr)) return 1;
  switch(event->code){
    case KEY_Right:
    case KEY_Left:
      if(!(options&(ICONLIST_BIG_ICONS|ICONLIST_MINI_ICONS))) return 0;
    case KEY_Up:
    case KEY_Down:
    case KEY_Home:
    case KEY_End:
      flags|=FLAG_UPDATE;
      if(event->click_count==1){
        if((options&SELECT_MASK)==ICONLIST_BROWSESELECT){
          handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
          }
        }
      return 1;
    case KEY_space:
    case KEY_KP_Space:
      flags|=FLAG_UPDATE;
      handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
      return 1;
    case KEY_Page_Up:
    case KEY_Page_Down:
      return 1;
    case KEY_Shift_L:
    case KEY_Shift_R:
    case KEY_Control_L:
    case KEY_Control_R:
    case KEY_Alt_L:
    case KEY_Alt_R:
      if(isDragging()){
        action=DRAG_COPY;
        if(isDropTarget()) action=DRAG_MOVE;
        if(didAccept()!=DRAG_REJECT){
          if(action==DRAG_MOVE)
            setDragCursor(getApp()->dndMoveCursor);
          else
            setDragCursor(getApp()->dndCopyCursor);
          }
        else{
          setDragCursor(getApp()->dontdropCursor);
          }
        }
      return 1;
    }
  return 0;
  }


// We timed out, i.e. the user didn't move for a while
long FXIconList::onTipTimer(FXObject*,FXSelector,void*){
  FXTRACE((250,"%s::onTipTimer %08x\n",getClassName(),this));
  timer=NULL;
  flags|=FLAG_TIP;
  return 1;
  }


// Timeout
long FXIconList::onAutoScroll(FXObject* sender,FXSelector sel,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXint lx,rx,ty,by;
  
  // Lasso mode
  if(flags&FLAG_LASSO){
    
    // Hide the lasso before scrolling
    drawLasso(anchorx,anchory,currentx,currenty);
    
    // Scroll the content
    FXScrollArea::onAutoScroll(sender,sel,ptr);

    currentx=event->win_x-pos_x;
    currenty=event->win_y-pos_y;
    FXMINMAX(lx,rx,anchorx,currentx);
    FXMINMAX(ty,by,anchory,currenty);

    // Perform selection
    selectInRectangle(pos_x+lx,pos_y+ty,rx-lx+1,by-ty+1);

    getApp()->flush();

    // Show lasso again
    drawLasso(anchorx,anchory,currentx,currenty);
    
    return 1;
    }
  
  // Drag and drop mode
  if(flags&FLAG_DODRAG){
    
    // Scroll the content
    FXScrollArea::onAutoScroll(sender,sel,ptr);
    
    // Content scrolled, so perhaps something else under cursor
    handle(this,MKUINT(0,SEL_DRAGGED),ptr);
  
    return 1;
    }

  return 0;
  }


// Mouse moved
long FXIconList::onMotion(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXint lx,rx,ty,by;
  
  // Lasso mode
  if(flags&FLAG_LASSO){
    flags&=~FLAG_TIP;

    // Start auto scrolling?
    if(startAutoScroll(event->win_x,event->win_y,FALSE)) return 1;

    // Hide lasso
    drawLasso(anchorx,anchory,currentx,currenty);
    currentx=event->win_x-pos_x;
    currenty=event->win_y-pos_y;
    FXMINMAX(lx,rx,anchorx,currentx);
    FXMINMAX(ty,by,anchory,currenty);

    // Perform selection
    selectInRectangle(pos_x+lx,pos_y+ty,rx-lx+1,by-ty+1);

    getApp()->flush();

    // Show lasso again
    drawLasso(anchorx,anchory,currentx,currenty);
    return 1;
    }

  // Drag and drop mode
  if(flags&FLAG_DODRAG){
    flags&=~FLAG_TIP;

    // Start auto scrolling?
    if(startAutoScroll(event->win_x,event->win_y,TRUE)) return 1;

    handle(this,MKUINT(0,SEL_DRAGGED),ptr);
    return 1;
    }
  
  // Tentative drag and drop
  if(flags&FLAG_TRYDRAG){
    flags&=~FLAG_TIP;
    if(event->moved){
      flags&=~FLAG_TRYDRAG;
      if(handle(this,MKUINT(0,SEL_BEGINDRAG),ptr)){
        flags|=FLAG_DODRAG;
        }
      }
    return 1;
    }

  // Reset tip timer
  if(timer) getApp()->removeTimeout(timer);
  timer=getApp()->addTimeout(getApp()->menuPause,this,ID_TIPTIMER);

  // Movement kills tip
  if(flags&FLAG_TIP){
    flags&=~FLAG_TIP;
    return 1;
    }
  
  // No change
  return 0; 
  }


// Pressed a button
long FXIconList::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXint index;
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONPRESS),ptr)) return 1;
    index=getItemAt(event->win_x,event->win_y);
    //FXTRACE((100,"%s::getItemAt(%d,%d) -> %d\n",getClassName(),event->win_x,event->win_y,index));
    handle(this,MKUINT(0,SEL_CHANGED),(void*)index);
    handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
    if(0<=index){
      if(items[index]->state&FXIconItem::DRAGGABLE){
        flags|=(FLAG_PRESSED|FLAG_TRYDRAG);
        }
      }
    else if(!(options&(ICONLIST_SINGLESELECT|ICONLIST_BROWSESELECT))){
      markAllItems();
      anchorx=currentx=event->win_x-pos_x;
      anchory=currenty=event->win_y-pos_y;
      drawLasso(anchorx,anchory,currentx,currenty);
      flags|=(FLAG_PRESSED|FLAG_LASSO);
      }
    flags&=~FLAG_UPDATE;
    return 1;
    }
  return 0;
  }


// Released button
long FXIconList::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    ungrab();
    stopAutoScroll();
    flags&=~(FLAG_PRESSED|FLAG_TRYDRAG);
    flags|=FLAG_UPDATE;
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr)) return 1;
    if(flags&FLAG_LASSO){
      drawLasso(anchorx,anchory,currentx,currenty);
      flags&=~FLAG_LASSO;
      }
    if(flags&FLAG_DODRAG){
      handle(this,MKUINT(0,SEL_ENDDRAG),ptr);
      flags&=~FLAG_DODRAG;
      }
    handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
    makeItemVisible(current);
    return 1; 
    }
  return 0;
  }


// The widget lost the grab for some reason
long FXIconList::onUngrabbed(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onUngrabbed(sender,sel,ptr);
  flags&=~(FLAG_DODRAG|FLAG_LASSO|FLAG_TRYDRAG|FLAG_PRESSED|FLAG_CHANGED);
  flags|=FLAG_UPDATE;
  stopAutoScroll();
  return 1;
  }



// Button or Key activate
long FXIconList::onActivate(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  if((0<=current) && !(items[current]->state&FXIconItem::DISABLED)){
    switch(options&SELECT_MASK){
      case ICONLIST_EXTENDEDSELECT:
        if(event->state&SHIFTMASK){
          if(0<=anchor){
            selectItem(anchor);
            extendSelection(current);
            }
          else{
            selectItem(current);
            setAnchorItem(current);
            }
          }
        else if(event->state&CONTROLMASK){
          //if(!isItemSelected(current)) selectItem(current);
          //toggleItem(current);
          setAnchorItem(current);
          }
        else{
          killSelection();
          selectItem(current);
          setAnchorItem(current);
          }
        break;
      case ICONLIST_SINGLESELECT:
      case ICONLIST_MULTIPLESELECT:
        toggleItem(current);
        setAnchorItem(current);
        break;
      case ICONLIST_BROWSESELECT:
        break;
      }
    }
  else{
    switch(options&SELECT_MASK){
      case ICONLIST_EXTENDEDSELECT:
      case ICONLIST_SINGLESELECT:
        if(!(event->state&(SHIFTMASK|CONTROLMASK))) killSelection();
        break;
      case ICONLIST_MULTIPLESELECT:
      case ICONLIST_BROWSESELECT:
        break;
      }
    }
  return 1;
  }
  

// Button or Key deactivate
long FXIconList::onDeactivate(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  setAnchorItem(current);
  if((0<=current) && !(items[current]->state&FXIconItem::DISABLED)){
    switch(options&SELECT_MASK){
      case ICONLIST_EXTENDEDSELECT:
        if(!event->moved && (event->state&CONTROLMASK)) toggleItem(current);
        break;
      case ICONLIST_MULTIPLESELECT:
      case ICONLIST_SINGLESELECT:
      case ICONLIST_BROWSESELECT:
        break;
      }
    }
  if(event->click_count==1){
    handle(this,MKUINT(0,SEL_CLICKED),(void*)current);
    }
  else if(event->click_count==2){
    handle(this,MKUINT(0,SEL_DOUBLECLICKED),(void*)current);
    }
  else if(event->click_count==3){
    handle(this,MKUINT(0,SEL_TRIPLECLICKED),(void*)current);
    }
  if((0<=current) && !(items[current]->state&FXIconItem::DISABLED)){
    handle(this,MKUINT(0,SEL_COMMAND),(void*)current);
    }
  return 1;
  }


// Current item changed
long FXIconList::onChanged(FXObject*,FXSelector,void* ptr){
  setCurrentItem((FXint)(long)ptr);
  if(target) target->handle(this,MKUINT(message,SEL_CHANGED),ptr);
  return 1;
  }


// Command message
long FXIconList::onCommand(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_COMMAND),ptr);
  }


// Clicked in list
long FXIconList::onClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_CLICKED),ptr);
  }


// Double Clicked in list; ptr may or may not point to an item
long FXIconList::onDoubleClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_DOUBLECLICKED),ptr);
  }


// Triple Clicked in list; ptr may or may not point to an item
long FXIconList::onTripleClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_TRIPLECLICKED),ptr);
  }


// Create custom item
FXIconItem *FXIconList::createItem(const FXString& text,FXIcon *big,FXIcon* mini,void* ptr){
  return new FXIconItem(text,big,mini,ptr);
  }


// Retrieve item
FXIconItem *FXIconList::retrieveItem(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::retrieveItem: index out of range.\n",getClassName()); } 
  return items[index];
  }


// Insert item
FXint FXIconList::insertItem(FXint index,FXIconItem* item){
  if(!item){ fxerror("%s::insertItem: item is NULL.\n",getClassName()); } 
  if(index<0 || nitems<index){ fxerror("%s::insertItem: index out of range.\n",getClassName()); } 
  FXRESIZE(&items,FXIconItem*,nitems+1);
  memmove(&items[index+1],&items[index],sizeof(FXIconItem*)*(nitems-index));
  items[index]=item;
  if(anchor>=index)  anchor++;
  if(extent>=index)  extent++;
  if(current>=index) current++;
  nitems++;
  recalc();
  return index;
  }


// Insert item
FXint FXIconList::insertItem(FXint index,const FXString& text,FXIcon *big,FXIcon* mini,void* ptr){
  return insertItem(index,createItem(text,big,mini,ptr));
  }


// Replace item with another
FXint FXIconList::replaceItem(FXint index,FXIconItem* item){
  if(!item){ fxerror("%s::replaceItem: item is NULL.\n",getClassName()); } 
  if(index<0 || nitems<=index){ fxerror("%s::replaceItem: index out of range.\n",getClassName()); } 
  item->state=items[index]->state;    // Copy the state over
  delete items[index];
  items[index]=item;
  recalc();
  return index;
  }


// Replace item with another
FXint FXIconList::replaceItem(FXint index,const FXString& text,FXIcon *big,FXIcon* mini,void* ptr){
  return replaceItem(index,createItem(text,big,mini,ptr));
  }


// Append item at end
FXint FXIconList::appendItem(FXIconItem* item){
  if(!item){ fxerror("%s::appendItem: item is NULL.\n",getClassName()); } 
  FXRESIZE(&items,FXIconItem*,nitems+1);
  items[nitems]=item;
  nitems++;
  recalc();
  return nitems-1;
  }


// Append item at end
FXint FXIconList::appendItem(const FXString& text,FXIcon *big,FXIcon* mini,void* ptr){
  return appendItem(createItem(text,big,mini,ptr));
  }


// Remove node from list
void FXIconList::removeItem(FXint index){
  if(index<0 || nitems<=index){ fxerror("%s::removeItem: index out of range.\n",getClassName()); } 
  nitems--;
  delete items[index];
  memmove(&items[index],&items[index+1],sizeof(FXIconItem*)*(nitems-index));
  if(anchor>index)  anchor--;  else if(anchor==index)  anchor=-1;
  if(extent>index)  extent--;  else if(extent==index)  extent=-1;
  if(current>index) current--; else if(current==index) current=-1;
  recalc();
  }


// Remove all items
void FXIconList::clearItems(){
  FXint i;
  for(i=0; i<nitems; i++){delete items[i];}
  FXFREE(&items);
  nitems=0;
  current=-1;
  anchor=-1;
  extent=-1;
  recalc();
  }


// Change the font
void FXIconList::setFont(FXFont* fnt){
  if(!fnt){ fxerror("%s::setFont: NULL font specified.\n",getClassName()); }
  if(font!=fnt){
    font=fnt;
    recalc();
    update();
    }
  }


// Set text color
void FXIconList::setTextColor(FXColor clr){
  if(clr!=textColor){
    textColor=clr;
    update();
    }
  }


// Set select background color
void FXIconList::setSelBackColor(FXColor clr){
  if(clr!=selbackColor){
    selbackColor=clr;
    update();
    }
  }


// Set selected text color
void FXIconList::setSelTextColor(FXColor clr){
  if(clr!=seltextColor){
    seltextColor=clr;
    update();
    }
  }


// Set text width
void FXIconList::setItemSpace(FXint s){
  if(s<1) s=1;
  if(itemSpace!=s){
    itemSpace=s;
    recalc();
    }
  }


// Change list style
void FXIconList::setListStyle(FXuint style){
  FXuint opts=(options&~ICONLIST_MASK) | (style&ICONLIST_MASK);
  if(options!=opts){
    options=opts;
    recalc();
    }
  }


// Get list style
FXuint FXIconList::getListStyle() const { 
  return (options&ICONLIST_MASK); 
  }


// Change help text
void FXIconList::setHelpText(const FXString& text){
  help=text;
  }


// Save data
void FXIconList::save(FXStream& store) const {
  register FXint i;
  FXScrollArea::save(store);
  store << header;
  store << nitems;
  for(i=0; i<nitems; i++){store<<items[i];}
  store << nrows;
  store << ncols;
  store << anchor;
  store << current;
  store << extent;
  store << font;
  store << textColor;
  store << selbackColor;
  store << seltextColor;
  store << itemSpace;
  store << itemWidth;
  store << itemHeight;
  store << help;
  }


// Load data
void FXIconList::load(FXStream& store){ 
  register FXint i;
  FXScrollArea::load(store);
  store >> header;
  store >> nitems;
  FXRESIZE(&items,FXIconItem*,nitems);
  for(i=0; i<nitems; i++){store>>items[i];}
  store >> nrows;
  store >> ncols;
  store >> anchor;
  store >> current;
  store >> extent;
  store >> font;
  store >> textColor;
  store >> selbackColor;
  store >> seltextColor;
  store >> itemSpace;
  store >> itemWidth;
  store >> itemHeight;
  store >> help;
  }


// Cleanup
FXIconList::~FXIconList(){
  clearItems();
  if(timer){getApp()->removeTimeout(timer);}
  header=(FXHeader*)-1;
  items=(FXIconItem**)-1;
  font=(FXFont*)-1;
  timer=(FXTimer*)-1;
  }


