/* $Id: visual.c,v 1.47 1999/02/13 01:52:23 marcus Exp $
******************************************************************************

   Display-X: initialization

   Copyright (C) 1995 Andreas Beck      [becka@ggi-project.org]
   Copyright (C) 1997 Jason McMullan    [jmcc@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>

#include <ggi/internal/ggi-dl.h>
#include <ggi/display/x.h>

#define NUM_OPTS  2

static gg_option x_options[NUM_OPTS] =
{
	{ "inroot", "no" },
	{ "inwin",  "no" }
};


void _GGI_X_freedbs(ggi_visual *vis, struct Xhooks *priv) {
	int i;
	int first = LIBGGI_APPLIST(vis)->first_targetbuf;
	int last = LIBGGI_APPLIST(vis)->last_targetbuf;

	if (first < 0) {
		return;
	}
	for (i = (last - first); i >= 0; i--) {

		if (priv->ximage_list[i]) {
			XDestroyImage(priv->ximage_list[i]);
			priv->ximage_list[i] = NULL;
		}

#ifdef HAVE_SHM
		if (priv->shminfo[i].shmid != -1) {
			XShmDetach(priv->display, &(priv->shminfo[i]));
			GGIDPRINT_MISC("_GGI_X_freedbs: XShmDetach(%p, %d)\n",
			       priv->display, priv->shminfo[i]);
			if (LIBGGI_APPLIST(vis)->bufs[i]->write != NULL)
				shmdt(LIBGGI_APPLIST(vis)->bufs[i]->write);
			priv->shminfo[i].shmid = -1;
		}
#endif
		_ggi_db_free(LIBGGI_APPLIST(vis)->bufs[i+first]);
		_ggi_db_del_buffer(LIBGGI_APPLIST(vis), i+first);
	}
	LIBGGI_APPLIST(vis)->first_targetbuf = -1;
	priv->ximage = NULL;
}


static inline Cursor make_cursor(Display *disp)
{
	char data[] = { 0xf8, 0xfa, 0xf8 };
	char mask[] = { 0xfa, 0xff, 0xfa };
	Pixmap crsrpix, crsrmask;
	XColor black = { 0, 0x0, 0x0, 0x0 },
	       white = { 0, 0xffff, 0xffff, 0xffff };
	Cursor mycrsr;

	crsrpix = XCreateBitmapFromData(disp, DefaultRootWindow(disp),
					data, 3, 3);
	crsrmask = XCreateBitmapFromData(disp, DefaultRootWindow(disp),
					mask, 3, 3);
	mycrsr = XCreatePixmapCursor(disp, crsrpix, crsrmask,
				     &black, &white, 1, 1);
	XFreePixmap(disp, crsrpix);
	XFreePixmap(disp, crsrmask);

	return mycrsr;
}


int GGIdlinit(ggi_visual *vis, const char *args, void *argptr)
{
	struct Xhooks *priv;
	Display *disp;
	void *lock;
	int i;

	if (args) {
		args = ggParseOptions((char *)args, x_options,
				      NUM_OPTS);
		if (args == NULL) {
			fprintf(stderr, "display-x: error in "
				"arguments.\n");
			return GGI_DL_ERROR;
		}
	}

	GGIDPRINT_MISC("X-target wants display %s\n", args);
	disp = XOpenDisplay(args);
	if (disp == NULL) return GGI_DL_ERROR;
	GGIDPRINT_MISC("X: has display %s\n", args);

	if ((lock = ggLockCreate()) == NULL) {
		XCloseDisplay(disp);
		return GGI_DL_ERROR;
	}

	priv = _ggi_malloc(sizeof(struct Xhooks));

	priv->display = disp;
	priv->window = DefaultRootWindow(disp);
	priv->ximage = NULL;
	priv->cmap = 0;
	priv->gc = 0;
	priv->XLibLock = lock;

	priv->screen = DefaultScreen(priv->display);
	GGIDPRINT_MISC("X: has screen %d\n", priv->screen);

	/* Get size of root window */
	{
		Window dummywin;
		int dummy;
		unsigned int w, h, udummy;

		XGetGeometry(disp, DefaultRootWindow(disp), &dummywin,
			     &dummy, &dummy, &w, &h,
			     &udummy, &udummy);
		if (w > 640) w = 640;
		if (h > 480) h = 480;
		priv->defsize.x = w;
		priv->defsize.y = h;
	}

	if (x_options[0].result[0] != 'n') {
		priv->wintype = GGIX_ROOT;
		GGIDPRINT_MISC("X: using root window\n");
	} else if (x_options[1].result[0] != 'n') {
		priv->wintype = GGIX_WIN;
		priv->window = strtol(x_options[1].result, NULL, 0);
		GGIDPRINT_MISC("X: using window id 0x%x\n", priv->window);
	} else {
		priv->wintype = GGIX_NORMAL;
	}

	priv->opmansync = _ggi_malloc(sizeof(_ggi_opmansync));

	if(!_ggiAddDL(vis, "helper-mansync", NULL, priv->opmansync, 0)) {
		fprintf(stderr,
			"display-X: Cannot load required helper-mansync!\n");
		XCloseDisplay(disp);
		free(priv);
		return GGI_DL_ERROR;
	}

	priv->cursor = make_cursor(disp);

	for(i = 0; i<8; i++) {
		priv->ximage_list[i] = NULL;
#ifdef HAVE_SHM
		priv->shminfo[i].shmid = -1;
#endif
	}

#ifdef HAVE_SHM
	if (XShmQueryExtension(priv->display)) {
		
		GGIDPRINT_MISC("X: Enabling use of XSHM extension\n");
		priv->have_shm = 1;
	} else {
		priv->have_shm = 0;
	}
#endif

	LIBGGI_PRIVATE(vis) = priv;

	MANSYNC_init(vis);
	
	/* Has mode management */
	vis->opdisplay->flush=GGI_X_flush;
	vis->opdisplay->getmode=GGI_X_getmode;
	vis->opdisplay->setmode=GGI_X_setmode;
	vis->opdisplay->checkmode=GGI_X_checkmode;
	vis->opdisplay->getapi =GGI_X_getapi;
	vis->opdisplay->setflags=GGI_X_setflags;

	{
		gii_inputxwin_arg args;
		gii_input *inp;
		
		args.disp = priv->display;
		args.ptralwaysrel = 0;
		args.wait = 1;
		args.exposefunc = (gii_inputxwin_exposefunc*)_ggi_x_do_blit;
		args.exposearg = priv;
		args.gglock = lock;
		
		if ((inp = giiOpen("xwin", &args, NULL)) == NULL) {
			GGIDPRINT_MISC("Unable to open xwin inputlib\n");
			GGIdlcleanup(vis);
			return GGI_DL_ERROR;
		}

		priv->inp = inp;
		/* Now join the new event source in. */
		vis->input = giiJoinInputs(vis->input, inp);
        }

	LIBGGI_GC(vis) = _ggi_malloc(sizeof(ggi_gc));

	GGIDPRINT_MISC("X-target fully up\n");

	return GGI_DL_OPDISPLAY;
}

int GGIdlcleanup(ggi_visual *vis)
{
	struct Xhooks *priv = LIBGGI_PRIVATE(vis);

	MANSYNC_deinit(vis);

	if (priv->cmap)	XFreeColormap(priv->display, priv->cmap);
	_GGI_X_freedbs(vis, priv);
	if (priv->gc) XFreeGC(priv->display, priv->gc);
	if (priv->window) XDestroyWindow(priv->display, priv->window);
	if (priv->cursor != None) XFreeCursor(priv->display, priv->cursor);
	XSync(priv->display,0);
	if (priv->display) {
		XCloseDisplay(priv->display);
	}
	free(priv->opmansync);
	
	ggLockDestroy(priv->XLibLock);
	
	free(priv);

	if(vis->palette) free(vis->palette);
	free(LIBGGI_GC(vis));

	GGIDPRINT_MISC("X-target closed\n");
	
	return 0;
}

#include <ggi/internal/ggidlinit.h>
