//=-- Dots.c - Fun with flying dots... ---------------------------------------=
//
//  A bunch of neeto flying dots.  A cool LibGGI demo.
//
//=---------------------------------------------------------------------------=
//  This file is copyright (c) 1997 Chris Lattner 
//    Do with it as you please, but at least let me know of the changes you 
//    make to it...  This may be freely distributed and made a part of the GGI
//    distribution.
//=---------------------------------------------------------------------------=

#include <math.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <ggi/ggi.h>

// DOTDENSITY - Number of particles to generate per frame
int DOTDENSITY=10;

// NUMSTREAMS - The number of seperate streams
int NUMSTREAMS=8;

// NUMPARTICLES - Maximum number of particles on screen
int NUMPARTICLES=4000;

ggi_visual_t *vis;
ggi_color Palette[256];
ggi_pixel Lookup[256];

struct PtType {
  int X, Y;
};

struct ParticleSt {
  struct PtType Pos, Vel, Acc;
  int sx, sy;
  int Life;
  uint8 Alive;     /* Is this record filled?  */
  uint8 Color;
};

struct ParticleSt *Particles;
struct PtType     *Stream, *Grav, *SVel;

int UsedParticles = 0;
int FrameNum;

void SetupPalette(void) {
  int i;
  for (i = 0; i < 64; i++) {
    Palette[i].r = i*256*4;
    Palette[i].g = 0;
    Palette[i].b = i*256*4;

    Palette[i+64].r = 0; 
    Palette[i+64].g = i*256*4; 
    Palette[i+64].b = i*256*4;

    Palette[i+128].r = 0; 
    Palette[i+128].g = 0; 
    Palette[i+128].b = i*256*4;

    Palette[i+192].r = i*256*4; 
    Palette[i+192].g = i*256*4; 
    Palette[i+192].b = 0;
  }
}

void LookupPalette(void) {
  int i;
  for (i = 0; i < 256; i++) {
    Lookup[i] = ggiMapColor(vis, Palette+i);
  }
}

int FindParticle(void) {
  int i;
  for (i = 0; i < NUMPARTICLES; i++)
    if (!Particles[i].Alive) 
      return i;
  return -1;
}

void GenerateParticles(int Number, int SourceX, int SourceY, int Color) {
  int i;

  while (Number-- > 0) {
    i = FindParticle();
    if (i == -1) return;

    UsedParticles++;

    Particles[i].Pos.X = SourceX;
    Particles[i].Pos.Y = SourceY;
    Particles[i].Vel.X = 0;
    Particles[i].Vel.Y = 0;
    Particles[i].Acc.X = (random() % 6000) - 3000;
    Particles[i].Acc.Y = (random() % 6000) - 3000;

    Particles[i].Life = (random() % 40) + 87;
    Particles[i].Alive = 1;
    Particles[i].Color = Color;
  }
}

#define Clamp(X,Max)  \
	(((X) > (Max)) ? (Max) : (((X) < -(Max)) ? -(Max) : (X)))

void DoSource(void) {
  int i, dx, dy;
  int Dist;

  if (!(FrameNum & 63)) {
    for (i = 0; i < NUMSTREAMS; i++) {
      Grav[i].X = (random() % (280 << 16)) - (140 << 16);
      Grav[i].Y = (random() % (160 << 16)) - (80 << 16);
    }
  }

  for (i = 0; i < NUMSTREAMS; i++) {
    dx = Grav[i].X-Stream[i].X;
    dy = Grav[i].Y-Stream[i].Y;

    Dist = (int)sqrt((double) (((dx*dx) >> 16) + ((dy*dy) >> 16))) /(7+i);

    if (Dist == 0) Dist = 1;
    SVel[i].X += dx / Dist;
    SVel[i].Y += dy / Dist;

    SVel[i].X = Clamp(SVel[i].X, 3 << 16);  // Don't go too fast now!!!
    SVel[i].Y = Clamp(SVel[i].Y, 3 << 16);

    Stream[i].X += SVel[i].X;
    Stream[i].Y += SVel[i].Y;

    GenerateParticles(DOTDENSITY, Stream[i].X, Stream[i].Y, (i & 3)*64);
  }
}

static int TimeDiff(struct timeval start, struct timeval end)
{
	return (end.tv_sec  - start.tv_sec)  * 1000 +
	       (end.tv_usec - start.tv_usec) / 1000;
}

int main(int argc,char *argv[]) { 
  int i, j, x, y;
  int xmax, ymax;
  ggi_mode mode;
  struct timeval start_time, end_time;
  int cur_frame=0, total_time;
  char *progname=argv[0];

  argv++; argc--;

  if (ggiInit() != 0) {
	fprintf(stderr, "%s: unable to initialize libggi, exiting.\n",
		progname);
	exit(1);
  }

  ggiParseMode("", &mode);   /* sets all fields to GGI_AUTO */

  if ((argc>=2) && 
      ((strcmp(argv[0], "-m") == 0) ||
       (strcmp(argv[0], "--mode") == 0))) {

	ggiParseMode(argv[1], &mode);
	argv += 2; argc -= 2;
  }
  
  if (argc>=1) { NUMPARTICLES=atoi(*argv); argv++; argc--; }
  if (argc>=1) { NUMSTREAMS  =atoi(*argv); argv++; argc--; }
  if (argc>=1) { DOTDENSITY  =atoi(*argv); argv++; argc--; }

  if (! (Stream   =malloc(NUMSTREAMS  *sizeof(Stream   [0]))) ) exit(1);
  if (! (Grav     =malloc(NUMSTREAMS  *sizeof(Grav     [0]))) ) exit(1);
  if (! (SVel     =malloc(NUMSTREAMS  *sizeof(SVel     [0]))) ) exit(1);

  if (! (Particles=malloc(NUMPARTICLES*sizeof(Particles[0]))) ) exit(1);

  if ((vis=ggiOpen(NULL)) == NULL) {
	  fprintf(stderr,
		  "%s: unable to open default visual, exiting.\n",
		  progname);
	  exit(1);
  }

  ggiSetFlags(vis, GGIFLAG_ASYNC);
  
  if (ggiSetMode(vis, &mode) != 0) {
    ggiPanic("dots: Cannot set mode!\n");
  }

  ggiGetMode(vis, &mode);
  xmax = mode.virt.x;
  ymax = mode.virt.y;

  SetupPalette();
  if(mode.graphtype==GT_8BIT)
	  ggiSetPalette(vis, 0, 256, Palette);
  LookupPalette();

  memset(Particles, 0, sizeof(Particles[0])*NUMPARTICLES);

  gettimeofday(&start_time, NULL);

  while (! ggiKbhit(vis)) {

    cur_frame = FrameNum % mode.frames;
    ggiSetReadFrame(vis, cur_frame);
    ggiSetWriteFrame(vis, cur_frame);

    DoSource();
    j = 0;
    for (i = 0; (i < NUMPARTICLES) && (j < UsedParticles); i++) 
      if (Particles[i].Alive) {
        Particles[i].Vel.X += Particles[i].Acc.X;
        Particles[i].Vel.Y += Particles[i].Acc.Y;
        Particles[i].Pos.X += Particles[i].Vel.X;
        Particles[i].Pos.Y += Particles[i].Vel.Y;
        x = (Particles[i].Pos.X >> 16) + (xmax>>1);
        y = (Particles[i].Pos.Y >> 16) + (ymax>>1);
        if ((x < 0) || (x >= xmax) || (y < 0) || (y >= ymax) ||
           (Particles[i].Life == 1) ) {
          Particles[i].Alive = 0;
          UsedParticles--;
        } else {
          Particles[i].Life--;
          Particles[i].sx = x;
          Particles[i].sy = y;
          j++;
        }
      }

    j = 0;
    ggiFillscreen(vis);

      for (i = 0; i < 256; i++)
        ggiPutPixel(vis, i, 0, Lookup[i]); // Draw palette

      for (i = 0; (i < NUMPARTICLES) && (j < UsedParticles); i++) 
        if (Particles[i].Alive) {
          ggiPutPixel(vis, Particles[i].sx, Particles[i].sy, 
             Lookup[(Particles[i].Life >> 1) + Particles[i].Color]);
          j++;
        }

    ggiFlush(vis);
    ggiSetDisplayFrame(vis, cur_frame);

    FrameNum++;
    ggUSleep(100);
  }

  ggiGetc(vis);
  ggiClose(vis);
  ggiExit();

  gettimeofday(&end_time, NULL);
  total_time = TimeDiff(start_time, end_time);

  printf("Time: %.2f s  Frames: %d  FPS: %.2f  TPF: %.2f ms\n", 
	 (float) total_time / 1000.0, FrameNum, (total_time > 0) ?
	 (float) FrameNum * 1000.0 / (float) total_time : 0.0, 
	 (FrameNum > 0) ? (float) total_time / (float) FrameNum : 0.0);

  return 0;
}
