/*
 * shmdemo.c - written in 1998 by Andreas Beck   becka@ggi-project.org
 *
 * This is a demonstration of LibGGI's functions and can be used as a 
 * reference programming example.
 *
 *   This software is placed in the public domain and can be used freely
 *   for any purpose. It comes without any kind of warranty, either
 *   expressed or implied, including, but not limited to the implied
 *   warranties of merchantability or fitness for a particular purpose.
 *   Use it at your own risk. the author is not responsible for any damage
 *   or consequences raised by use or inability to use this program.
 */

/* Include the necessary headers used for e.g. error-reporting.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

/* Include the LibGGI declarations.
 */
#include <ggi/gii.h>
#include <ggi/ggi.h>

/* We do shm here !
 */
#include <sys/ipc.h>
#include <sys/shm.h>
#include <signal.h>

#define NUMSHMEM	6
/* A cube has 6 sides ... :-) */
                     
/* In case we were called with wrong parameters, give an explanation.
 */
void usage(const char *prog)
{
	fprintf(stderr, "Usage:\n\n"
		        "%s [<xsize>x<ysize>[#<virtx>x<virty>]"
			"['['<bpp>']']\n\n"
		       "Example: %s 320x200[8]\n"
		"For textmode, try %s 80x25T\n",prog,prog,prog);
	exit(1);
}


/* The main routine.
 * It will set up a graphics mode as requested on the commandline and 
 * do an extensive test using about all graphics primitives LibGGI
 * knows.
 */
int main(int argc, char **argv)
{
	/* First we define a bunch of variables we will access throughout the
	 * main() function. Most of them are pretty meaningless loop-counters
	 * and helper-variables.
	 */

	const char *prog;	/* Make an alias for the program name */
	
	/* This is an enum holding the requested type of graphics mode.
	 * See the mode setting code below for details.
	 */
	 
	ggi_graphtype type;

	/* The depth is used for chosing a graphics type.  it is mapped
	 * to the right GT_* by a switch-statement.
	 */
	 
	int depth=0;

	/* These holde the visible screen size (sx,sy) and
	 * the virtual size (vx,vy). On most targets you can 
	 * request a certain larger area on which the visible
	 * area can be placed.
	 */

	int sx,sy,vx,vy;
	
	/* Pretty meaningless helpers used everywhere. */

	int err, x;
	char text[128];

 	ggi_visual_t vis;
 	
 	ggi_visual_t memvis[NUMSHMEM];
 	int shmid[NUMSHMEM];

	enum effect { STEADY, SLIDE_LEFT, SLIDE_RIGHT, SLIDE_UP, SLIDE_DOWN,
			JAL, BLEND };
	struct {
		enum effect effect;
		int speed;
		int position;
		int currcons;
		int wantcons;
		int keeprunning;
		int is_escape;
	} scrstate;

	/* This is a struct containing visual and virtual screen size as
	 * well as the bpp/textmode information 
	 */

	ggi_mode mode = { /* This will cause the default mode to be set */
		1,                      /* 1 frame */
		{GGI_AUTO,GGI_AUTO},    /* Default size */
		{GGI_AUTO,GGI_AUTO},    /* Virtual */
		{0,0},                  /* size in mm don't care */
		GT_AUTO,               /* Mode */
		{GGI_AUTO,GGI_AUTO}     /* Font size */
	};

	/* Get the arguments from the command line. 
	 * Set defaults for optional arguments.
	 */

	prog=*argv; argv++; argc--;
	if (strrchr(prog, '/') != NULL) prog=strrchr(prog, '/')+1;

	if ((argc > 0) &&
	    ((strcmp(*argv, "-h")==0) || 
	     (strcmp(*argv, "--help")==0))) {

		usage(prog);	/* This is not an allowed call format. */
		return 1;	/* Tell the user how to call and fail. */
	}

	if (argc > 1) {
		usage(prog);	/* This is not an allowed call format. */
		return 1;	/* Tell the user how to call and fail. */
	}
	
	/* Set up the random number generator. */
	srandom(time(NULL));

	for(x=0;x<NUMSHMEM;x++)
	{
		shmid[x]=shmget(ftok("/dev/null",'0'+x), 1024*1024, IPC_CREAT|0666);
		printf("shmid[%d]=%d\n",x,shmid[x]);
	}

	/* Initialize the GGI library. This must be called before any other 
	 * GGI function. Otherwise behaviour is undefined.
	 */
	if (ggiInit() != 0) {
		fprintf(stderr, "%s: unable to initialize libggi, exiting.\n",
			prog);
		exit(1);
	}

	/* Open the default visual. This is automatically selected depending
	 * on the calling environment (Linux console, X, ...). You can 
	 * explicitly set this in the LIBGGI_DISPLAY environment variable.
	 */
	if ((vis=ggiOpen(NULL)) == NULL) {
		fprintf(stderr,
			"%s: unable to open default visual, exiting.\n",
			prog);
		ggiExit();
		exit(1);
	}
	ggiSetFlags(vis,GGIFLAG_ASYNC);

	if (argc == 1) { /* A mode-string was given - so we parse it */
		if (ggiParseMode(*argv, &mode) != 0) {
			/* Error parsing mode */
			fprintf(stderr,
				"Error parsing mode: %s\n"
				"Valid modes are:\n"
"  X x Y # XVIRT x YVIRT D dpp.x x dpp.y F frames (T) [bpp] + XOFF + YOFF\n"
"  X x Y # XVIRT x YVIRT D dpp.x x dpp.y F frames [graphtype] + XOFF + YOFF\n\n"
"  Everything (!) can be omitted, all ommitted values default to GGI_AUTO,\n"
"  with the exception of a textmode with unspecified size which defaults\n"
"  to TEXT16.\n  Whitespace is ignored.\n"
				, *argv);
			exit(1);
		}
		/* mode now holds the requested visible 
		 * and virtual size of the screen.
		 */
	}

	/* that's what we try. See what we get ... */
	printf("Trying mode ");
	ggiPrintMode(&mode);
	printf("\n");

	/* Is the mode possible ? If not, a better one will be
	 * suggested. 
	 */

	ggiCheckMode(vis,&mode);

	printf("Suggested mode ");
	ggiPrintMode(&mode);
	printf("\n");

       	err=ggiSetMode(vis,&mode);   /* now try it. it *should* work! */
	ggiSetFlags(vis,GGIFLAG_ASYNC);
	
	/* Check if there were errors. Almost all LibGGI functions
	 * return 0 on success and an error code otherwise. No messing
	 * with errno, as this is not thread- friendly. The functions
	 * that are supposed to return a pointer normally return NULL on
	 * error.
	 */

	if (err) {
		fprintf(stderr,"Can't set mode\n");
		ggiClose(vis);
		ggiExit();
		return 2;
	}


	/* Now we read back the set mode, as it might have been
	 * autoselected or changed.
	 */

	type=mode.graphtype;
	vx=mode.virt.x;    vy=mode.virt.y;
	sx=mode.visible.x; sy=mode.visible.y;
	depth=GT_DEPTH(mode.graphtype);

	/* Set a colorful palette for the tests.
	 */
	 
	if (GT_SCHEME(mode.graphtype) == GT_PALETTE) {
		ggiSetColorfulPalette(vis);
	}
	

	for(x=0;x<NUMSHMEM;x++)
	{
		/* Open a shared "memory-visual" which is simply a simulated 
		 * display in shared memory.
		 */
		sprintf(text,"display-memory:-input:shmid:%d",shmid[x]);

		if ((memvis[x]=ggiOpen(text, NULL)) == NULL) {
			ggiPanic("Ouch - can't open the shmem target %d !",x);
		}

		/* Set it to the same mode as the main visual */
		err=ggiSetMode(memvis[x],&mode);
		
		if (GT_SCHEME(mode.graphtype) == GT_PALETTE) {
			ggiSetColorfulPalette(memvis[x]);
		}

		/* Check for errors */
		if (err) ggiPanic("Ouch - can't setmode on shmem target %d !",x);

	}

	printf("mode: %s\n",text);
	printf("Server is running - start another GGI demo (e.g. flying_ggis)\n");
	printf("with the following variable settings:\n");
	printf("GGI_DISPLAY=\"display-memory:-input:keyfile:%d:[0-%d]:%s\"\n",1024*1024,NUMSHMEM-1,"/dev/null");
	ggiSPrintMode(text,&mode);
	printf("GGI_DEFMODE=\"%s\"\n",text);
	printf("export GGI_DEFMODE GGI_DISPLAY\n");


	{
		ggi_color black  = { 0x0000, 0x0000, 0x0000 };
		ggi_color white  = { 0xffff, 0xffff, 0xffff };
		ggiSetGCForeground(memvis[0],ggiMapColor(memvis[0],&white));
		ggiSetGCBackground(memvis[0],ggiMapColor(memvis[0],&black));
	}
	ggiPuts(memvis[0],0,  0,"Keyboard:");
	ggiPuts(memvis[0],0, 10,"#   : Escape char. Twice to send it itself.");
	ggiPuts(memvis[0],0, 20,"#0-5: Select screen to view.");
	ggiPuts(memvis[0],0, 30,"#+/-: Next/previous screen.");
	ggiPuts(memvis[0],0, 40,"#'r': Pick random screen.");

#if 0
	y=0;dy=5;
	while(1)
	{
		ggiCrossBlit(memvis[0],0,vy-1-y,vx,   y,vis,0,0);
		ggiCrossBlit(memvis[1],0,     0,vx,vy-y,vis,0,y);
		ggiFlush(vis);
		y+=dy;
		if (y<=   0) { dy=-dy; y+=dy; }
		if (y>=vy-1) { dy=-dy; y+=dy; }
		ggUSleep(50000);
	}
#else
	scrstate.effect=JAL;
	scrstate.speed=10;
	scrstate.currcons=0;
	scrstate.wantcons=-1;
	scrstate.keeprunning=1;
	scrstate.is_escape=0;
	while(scrstate.keeprunning)
	{
		struct timeval t={0,50000};
		
		if (ggiEventPoll(vis,emKey,&t))
		{
			ggi_event event;
			ggiEventRead(vis,&event,emKey);
			if (event.any.type==evKeyPress)
			{
				switch(event.key.sym)
				{
					case '#': scrstate.is_escape=!scrstate.is_escape;
						  if (!scrstate.is_escape) goto sendit;
						  break;
					case '0': if (scrstate.is_escape) {
							scrstate.wantcons=0;
							scrstate.position=0;
						  }
						  else goto sendit;
						scrstate.is_escape=0;break;
					case '1': if (scrstate.is_escape) {
							scrstate.wantcons=1;
							scrstate.position=0;
						  }
						  else goto sendit;
						scrstate.is_escape=0;break;
					case '2': if (scrstate.is_escape) {
							scrstate.wantcons=2;
							scrstate.position=0;
						  }
						  else goto sendit;
						scrstate.is_escape=0;break;
					case '3': if (scrstate.is_escape) {
							scrstate.wantcons=3;
							scrstate.position=0;
						  }
						  else goto sendit;
						scrstate.is_escape=0;break;
					case '4': if (scrstate.is_escape) {
							scrstate.wantcons=4;
							scrstate.position=0;
						  }
						  else goto sendit;
						scrstate.is_escape=0;break;
					case '5': if (scrstate.is_escape) {
							scrstate.wantcons=5;
							scrstate.position=0;
						  }
						  else goto sendit;
						scrstate.is_escape=0;break;
					case '+': if (!scrstate.is_escape) goto sendit;
						  scrstate.is_escape=0;
						  scrstate.wantcons++;
						  if (scrstate.wantcons>=NUMSHMEM)  
						  	scrstate.wantcons=0;
						  scrstate.position=0;
						  break;
					case '-': if (!scrstate.is_escape) goto sendit;
						  scrstate.is_escape=0;
						  scrstate.wantcons--;
						  if (scrstate.wantcons<0) scrstate.wantcons=NUMSHMEM-1;
						  scrstate.position=0;
						  break;
					case 'r': if (!scrstate.is_escape) goto sendit;
						  scrstate.is_escape=0;
						  scrstate.wantcons=rand()%NUMSHMEM;
						  scrstate.position=0;
						  break;
					case 'q': if (!scrstate.is_escape) goto sendit;
						  scrstate.is_escape=0;
						  scrstate.keeprunning=0;
						  break;
					default:
					sendit:
						giiEventSend(ggiJoinInputs(memvis[scrstate.currcons],NULL),&event);
						scrstate.is_escape=0;break;
				}
			}
		}
		if (scrstate.wantcons>=0)
		{
		  switch(scrstate.effect)
		  { case STEADY:
		  		scrstate.currcons=scrstate.wantcons;
		  		scrstate.wantcons=-1;
				ggiCrossBlit(memvis[scrstate.currcons],0,0,vx,vy,vis,0,0);
		  		break;
		    case SLIDE_LEFT: 
		    		if (scrstate.position<vx) {
					ggiCrossBlit(memvis[scrstate.currcons],scrstate.position,0,vx-scrstate.position-1,vy,vis,0,0);
					ggiCrossBlit(memvis[scrstate.wantcons],0,0,scrstate.position,vy,vis,vx-scrstate.position-1,0);
					scrstate.position+=scrstate.speed;
		    		} else {
			  		scrstate.currcons=scrstate.wantcons;
			  		scrstate.wantcons=-1;
		    		}
		    		break;
		    case BLEND:
		    case SLIDE_RIGHT:
		    case SLIDE_UP:
		    case SLIDE_DOWN:	/*left as excercise to the reader */
		  		scrstate.currcons=scrstate.wantcons;
		  		scrstate.wantcons=-1;
				ggiCrossBlit(memvis[scrstate.currcons],0,0,vx,vy,vis,0,0);
		  		break;
		    	break;
		    case JAL:
		    		if (scrstate.position<scrstate.speed) {
		    			for(x=0;x<vy;x++)
						ggiCrossBlit(memvis[((x%scrstate.speed)<scrstate.position) ? scrstate.wantcons : scrstate.currcons ],0,x,vx,1,vis,0,x);
					scrstate.position++;
		    		} else {
			  		scrstate.currcons=scrstate.wantcons;
			  		scrstate.wantcons=-1;
		    		}
		    		break;
		  }
		}
		else
			ggiCrossBlit(memvis[scrstate.currcons],0,0,vx,vy,vis,0,0);
		ggiFlush(vis);
	}
#endif

	for(x=0;x<NUMSHMEM;x++)
	{
		ggiClose(memvis[x]);

		/* this makes the memory detach when all programs using this memory
		 * complete (according to SVR4 standard).
		 * note - this may not work on all platforms....
		 * note2: This may _not be called at an earlier place.
		 * Calling before our own shmat() will destroy _instantly_,
		 * calling before the other programs are running will deny
		 * the shmem to them.
		 */
		shmctl(shmid[x],IPC_RMID, NULL);
	}

	ggiClose(vis);
	
	/* Now close down LibGGI. */
	ggiExit();	

	/* Terminate the program.*/
	return 0;
}
