///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998-2000, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

#ifndef _MEMSTAT_H_
#define _MEMSTAT_H_

#include <assert.h>
#include <sys/time.h>
#include <unistd.h>


class MemoryRequest {
public:

  enum { FREE_OP = 0,
	 MALLOC_OP,
	 REALLOC_OP,
	 REFREE_OP,
	 ALLOCATE_OP,
	 DEALLOCATE_OP
  };

  void malloc (void * addr,
	       size_t sz)
  {
    _size = sz;
    _address = (unsigned long) addr | MALLOC_OP;
    markTime (_sec, _usec);
    // printf ("malloc %d (%f)\n", sz, getTime());
  }


  void free (void * addr)
  {
    _address = (unsigned long) addr | FREE_OP;
    markTime (_sec, _usec);
    // printf ("free %d (%f)\n", _address, getTime());
  }


  void allocate (int sz)
  {
    _address = ALLOCATE_OP;
    _size = sz;
    markTime (_sec, _usec);
    // printf ("allocate %d (%f)\n", sz, getTime());
  }


  // Set sec & usec to the current time.
  void markTime (long& sec, long& usec)
  {
    struct timeval tv;
    struct timezone tz;
    gettimeofday (&tv, &tz);
    sec = tv.tv_sec;
    usec = tv.tv_usec;
  }

  int getType (void) {
    return _address & 7;
  }

  int getAllocated (void) {
    return _size;
  }

  unsigned long getAddress (void) {
    return _address & ~7;
  }

  int getSize (void) {
    return _size;
  }

  double getTime (void) {
    return (double) _sec + (double) _usec / 1000000.0;
  }

  long getSeconds (void) {
    return _sec;
  }

  long getUseconds (void) {
    return _usec;
  }

  friend int operator< (MemoryRequest& m, MemoryRequest& n) {
    return ((m._sec < n._sec)
	    || ((m._sec == n._sec)
		&& (m._usec < n._usec)));
  }

private:
  int	_size;     	// in bytes 
  unsigned long _address;  // The address returned by malloc/realloc   
  long	_sec;      	// seconds as returned by gettimeofday      
  long	_usec;     	// microseconds as returned by gettimeofday 
};




#endif // _MEMSTAT_H_ 
