///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998-2000, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

/*
  superblock.cpp
  ------------------------------------------------------------------------
  The superblock class controls a number of blocks (which are
  allocatable units of memory).
  ------------------------------------------------------------------------
  @(#) $Id: superblock.cpp,v 1.25 2000/02/24 03:11:03 emery Exp $
  ------------------------------------------------------------------------
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

#include <new.h>
#include <string.h>

#include "arch-specific.h"
#include "config.h"
#include "heap.h"
#include "processheap.h"
#include "superblock.h"


superblock::superblock (int numBlocks,	// The number of blocks in the sb.
			int szclass,	// The size class of the blocks.
			hoardHeap * o)	// The heap that "owns" this sb.
  :
#if HEAP_DEBUG
    _magic (SUPERBLOCK_MAGIC),
#endif
    _sizeClass (szclass),
    _numBlocks (numBlocks),
    _numAvailable (0),
    _fullness (0),
    _freeList (NULL),
    _owner (o),
    _next (NULL),
    _prev (NULL)
{
  assert (_numBlocks >= 1);

  // Determine the size of each block.
  const int blksize =
    hoardHeap::align (sizeof(block) + hoardHeap::sizeFromClass(_sizeClass));

  // Make sure this size is in fact aligned.
  assert ((blksize & hoardHeap::ALIGNMENT_MASK) == 0);

  // Set the first block to just past this superblock header.
  block * b
    = (block *) hoardHeap::align ((unsigned long) (this + 1));

  // Initialize all the blocks,
  // and insert the block pointers into the linked list.
  for (int i = 0; i < _numBlocks; i++) {
    // Make sure the block is on a double-word boundary.
    assert (((unsigned int) b & hoardHeap::ALIGNMENT_MASK) == 0);
    new (b) block (this);
    assert (b->getSuperblock() == this);
    b->setNext (_freeList);
    _freeList = b;
    b = (block *) ((char *) b + blksize);
  }
  _numAvailable = _numBlocks;
  _fullness = computeFullness();
  assert ((unsigned long) b <= hoardHeap::align (sizeof(superblock) + blksize * _numBlocks) + (unsigned long) this);
}


superblock * superblock::makeSuperblock (int sizeclass)
{
  // Determine how many blocks will go in this superblock.
  const size_t s = hoardHeap::sizeFromClass (sizeclass);
  assert (s > 0);

  // Compute the aligned size of a block plus its header.
  const int blksize =
    hoardHeap::align (sizeof(block) + s);

  // Compute the number of blocks that will go into this superblock.
  int numBlocks = MAX (1, ((hoardHeap::SUPERBLOCK_SIZE - sizeof(superblock)) / blksize));

  // We need to extend the break value to get more memory.

  // Compute how much memory we need.
  unsigned long moreMemory;
  if (numBlocks > 1) {
    moreMemory = hoardHeap::SUPERBLOCK_SIZE;
    assert (moreMemory >= hoardHeap::align(sizeof(superblock) + blksize * numBlocks));
  } else {
    moreMemory =
      hoardHeap::align (sizeof(superblock)  //   one superblock (header)
			+ blksize * numBlocks);	// + n blocks (headers + data).
  }

  // Set aside some memory for the new superblock.
  char * buf;

  // Is the block we obtain a recycled superblock?
  int isRecycled = 0;

  // Allocate more memory (plus enough to guarantee alignment).
  if (numBlocks > 1) {
    // This is an ordinary superblock.
    // Grab one from the process heap.
    isRecycled = processHeap::getSuperblockBuffer (buf);
  } else {
    buf = (char *) hoardSbrk (moreMemory + hoardHeap::ALIGNMENT - 1);
  }
  
  // Make sure that we actually got the memory.
  if (buf == NULL) {
    return 0;
  }
  buf = (char *) hoardHeap::align ((unsigned long) buf);

  // Make sure this buffer is double-word aligned.
  assert (buf == (char *) hoardHeap::align ((unsigned long) buf));
  assert ((((unsigned long) buf) & hoardHeap::ALIGNMENT_MASK) == 0);

  // If the superblock is recycled, check to see if it's for the same
  // size class.  If so, we don't have to reformat it.
  if (isRecycled) {
    if (((superblock *) buf)->getBlockSizeClass() == sizeclass) {
      return (superblock *) buf;
    }
  }
  // Instantiate the new superblock in the buffer.
  superblock * sb = new (buf) superblock (numBlocks, sizeclass, NULL);
  return sb;
}
