/* File "display_process.c":
 * Invoking the display process from Malaga. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* includes =================================================================*/

#define _XOPEN_SOURCE
#define _XOPEN_SOURCE_EXTENDED 1
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/wait.h>
#include <signal.h>
#include <pwd.h>
#include <fcntl.h>
#include "basic.h"
#include "files.h"
#include "input.h"

#undef GLOBAL
#define GLOBAL

#include "display_process.h"

/* constants ================================================================*/

#define MAX_ARGS 10 /* maximum number of arguments */

/* variables ================================================================*/

LOCAL pid_t process_id;     /* ID of display process */

/* functions ================================================================*/

GLOBAL void stop_display_process (void)
/* Stop the Malaga display process. */
{
  if (display_stream != NULL)
    fclose (display_stream);

  if (process_id != 0)
    kill (process_id, SIGTERM);

  display_stream = NULL;
  process_id = 0;
}

/*---------------------------------------------------------------------------*/

GLOBAL void start_display_process (void)
/* Start the Malaga display process whose name is in the environment variable
 * MALAGA_DISPLAY if it is not already running.
 * Return the input stream for the pipe to this process. */
{
  int pipe_fd[2];
  string_t malaga_display;

  if (process_id != 0)
  {
    if (waitpid (process_id, 0, WNOHANG) == 0)
      return;

    process_id = 0;
    fclose (display_stream);
    display_stream = NULL;
  }

  if (pipe (pipe_fd) == -1)
    error ("can't create pipe to display process: %s", strerror (errno));
    
  malaga_display = getenv ("MALAGA_DISPLAY");
  if (malaga_display == NULL) 
    error ("missing environment variable \"MALAGA_DISPLAY\"");

  switch (process_id = fork ())
  {
  case -1:
    error ("can't create display process: %s", strerror (errno));
    break;
    
  case 0:
  {
    string_t args[MAX_ARGS + 1];
    long_t num_args;

    dup2 (pipe_fd[0], STDIN_FILENO);
    close (pipe_fd[0]);
    close (pipe_fd[1]);

    parse_whitespace (&malaga_display);
    num_args = 0;
    while (*malaga_display != EOS)
    {
      if (num_args == MAX_ARGS)
      {
	fprintf (stderr, "too many arguments in \"MALAGA_DISPLAY\"\n");
	exit (1);
      }
      args[num_args] = parse_word (&malaga_display);
      num_args++;
    }
    args[num_args] = NULL;

    execvp (args[0], args);
    fprintf (stderr, "can't start display process \"%s\": %s\n", 
	     args[0], strerror (errno));
    exit (1);
  }
  default:
    close (pipe_fd[0]);
    display_stream = fdopen (pipe_fd[1], "w");
    if (display_stream == NULL)
      error ("can't open data stream: %s", strerror (errno));

#ifdef HANGUL
    fprintf (display_stream, "hangul\n");
#endif
#ifdef LATIN1
    fprintf (display_stream, "latin1\n");
#endif
  }
}

/*---------------------------------------------------------------------------*/
