/* File "lexicon.c":
 * All definitions for the run-time lexicon. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "tries.h"
#include "files.h"
#include "malaga_files.h"
#include "lex_type.h"

#undef GLOBAL
#define GLOBAL

#include "lexicon.h"

/*---------------------------------------------------------------------------*/

LOCAL void *lexicon_data; /* address of the lexicon file mapped into memory */
LOCAL long_t lexicon_length; /* the length of *<lexicon_data> */

/*---------------------------------------------------------------------------*/

GLOBAL void read_lexicon_file (string_t file_name)
/* Read lexicon from file <file_name>. */
{
  lexicon_header_t *header; /* lexicon file header */

  /* Map the lexicon file into memory. This saves a lot of memory. */
  map_file (file_name, &lexicon_data, &lexicon_length);

  /* Check lexicon header. */
  header = (lexicon_header_t *) lexicon_data;
  check_header (&header->common_header, file_name, LEXICON_FILE,
 		LEXICON_CODE_VERSION);
  
  /* Init trie. */
  lexicon.trie_size = header->trie_size;
  lexicon.trie = (long_t *) (header + 1);
  lexicon.trie_root = header->trie_root;

  /* Init cat lists. */
  lexicon.cat_lists_size = header->cat_lists_size;
  lexicon.cat_lists = (long_t *) (lexicon.trie + lexicon.trie_size);

  /* Init values. */
  lexicon.values_size = header->values_size;
  lexicon.values = (cell_t *) (lexicon.cat_lists + lexicon.cat_lists_size);

  /* Init strings. */
  lexicon.strings_size = header->strings_size;
  lexicon.strings = (char *) (lexicon.values + lexicon.values_size);
}

/*---------------------------------------------------------------------------*/

GLOBAL void free_lexicon (void)
/* Free all memory used by the lexicon. */
{
  unmap_file (lexicon_data, lexicon_length);
}

/*---------------------------------------------------------------------------*/
