/* File "rule_type.h":
 * The data types for rules */
 
/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

typedef enum /* possible rule types */
{
  ALLO_RULE, COMBI_RULE, END_RULE, FILTER_RULE, PRUNING_RULE, ROBUST_RULE,
  SUBRULE
} rule_type_t;

typedef struct /* defines a rule */
{
  long_t name;        /* string table index to rule name */
  long_t first_instr; /* code index of rule start */
  rule_type_t type;   /* type of the rule */
  long_t num_params;  /* number of parameters the rule takes */
} rule_t;

typedef struct
/* establishes a correspondence between a source line and a rule code index. */
{
  long_t line;  /* number of source line or -1 if instruction can't be
		 * associated with a specific source line */
  long_t file;  /* string table index to source file name or -1 (see "line") */
  long_t instr; /* index of first instruction that belongs to src_line */
} src_line_t;

typedef struct /* defines a variable name and all its scopes */
{
  long_t name;             /* string table index to variable name */
  long_t first_scope;      /* index to first scope of this variable
                            * in <var_scopes> */
  long_t number_of_scopes; /* number of scopes of this variable */ 
} var_t;

typedef struct /* defines a single scope of a variable */
{
  long_t first_instr; /* index of first instruction where var is defined */
  long_t last_instr;  /* index of last instruction where var is defined */
  long_t stack_index; /* index of variable in rule stack */
} var_scope_t;

typedef struct
/* contains a rule system of a compiled rule file.
 * A "..._size" variable contains the number of elements
 * in the following table. */
{
  long_t initial_rule_set; /* rules index of the initial combi_rule set */
  long_t initial_cat;      /* values index of initial combi_rule category */

  long_t robust_rule;     /* number of robust_rule or -1 */

  long_t pruning_rule;     /* number of pruning_rule or -1 */

  long_t allo_rule;        /* number of allo_rule or -1 */

  long_t filter_rule;      /* number of filter_rule or -1 */

  long_t input_rule;       /* number of input_rule or -1 */

  rule_t *rules;           /* name and code of every rule */
  long_t rules_size;

  long_t *rule_sets;       /* a collection of double lists; each double list is
			    * a series of rules, followed by -1, followed by
			    * a series of default rules, followed by -1 */
  long_t rule_sets_size;

  instr_t *instrs;         /* the actual rule instructions */
  long_t instrs_size;

  cell_t *values;          /* all constant Malaga values */
  long_t values_size;

  char *strings;           /* names of files, variables, rules, patterns */
  long_t strings_size;
  
  src_line_t *src_lines;   /* correspondence between source lines
			    * and rule instructions */
  long_t src_lines_size;

  var_t *vars;             /* variable names */
  long_t vars_size;

  var_scope_t *var_scopes; /* variable scopes */
  long_t var_scopes_size;
} rule_sys_t;
