/* File "values.h":
 * Operations to build, modify, and print Malaga values. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* types ====================================================================*/

typedef u_short_t cell_t;
/* A value is stored in one or more cells.
 * Use this type if you want to allocate memory (pools etc.) for values. */ 

typedef cell_t *value_t; /* Reference to a Malaga values by this type. */
typedef cell_t symbol_t;

/* some standard symbols */
enum {NIL_SYMBOL, YES_SYMBOL, NO_SYMBOL, 
      SYMBOL_SYMBOL, STRING_SYMBOL, NUMBER_SYMBOL, LIST_SYMBOL, RECORD_SYMBOL};

#define SYMBOL_MAX 8192 /* symbols must be in the range of 0..SYMBOL_MAX-1 */

/* variables ================================================================*/

GLOBAL string_t (*values_symbol_name) (symbol_t symbol);
/* Return the name of "symbol".
 * This is a callback function that must be set externally. */

GLOBAL value_t (*values_atoms_of_symbol) (symbol_t symbol);
/* Return the list of atoms of <symbol>,
 * if that symbol is a multi-symbol or NULL else.
 * This is a callback function that must be set externally. */

GLOBAL long_t num_hidden_attributes;
GLOBAL symbol_t *hidden_attributes;
/* The values of attributes in <hidden_attributes> are not printed
 * by the value output functions "value_as_text", etc. */

typedef enum {INTERNAL_ORDER, 
	      ALPHABETIC_ORDER, DEFINITION_ORDER} attribute_order_t;
extern attribute_order_t attribute_order;
/* The order in which attributes in a record are printed.
 * INTERNAL_ORDER is the order in which attributes are stored internally.
 * ALPHABETIC_ORDER means the alphabetic order of the attribute names.
 * DEFINITION_ORDER is the order in which the attributes are defined in the 
 *  symbol file.
 * Used by value output functions "value_as_text", etc. */

/* value storage support ====================================================*/

extern long_t length_of_value (value_t value);
/* Return the length of <value> in cells. */

extern value_t new_value (value_t value);
/* Allocate space for <value> and copy it.
 * Values that are allocated are not deleted
 * when "clear_value_heap" or "clear_value_heap_except" is called.
 * Use "free" to free the value space. */

extern value_t copy_value_to_pool (pool_t value_pool, 
				   value_t value, 
				   long_t *index);
/* Copy <value> to the pool <value_pool> and store its index in *<index>. */

/* value heap ===============================================================*/

/* The value heap will be filled with temporary values
 * like the results of value operations. It is of fixed size. */

extern void set_value_heap_size (int size);
/* Set the value heap size to <size> cells. */

extern long_t get_value_heap_size (void);
/* Get the current heap size in cells. */

extern void free_value_heap (void);
/* Free the memory that is allocated by the value heap. */

extern void clear_value_heap (void);
/* Clear all values on the heap. */

extern void clear_value_heap_except (long_t n, value_t values[]);
/* If the value heap is almost full, clear all values from the heap,
 * except the <n> values <values>[0..<n>-1]. */

/* value operations =========================================================*/

extern symbol_t get_value_type (value_t value);
/* Return the type of <value>. Depending of the type, the result value may be
 * SYMBOL_SYMBOL, STRING_SYMBOL, NUMBER_SYMBOL, 
 * LIST_SYMBOL or RECORD_SYMBOL. */

/*---------------------------------------------------------------------------*/

extern symbol_t value_to_symbol (value_t value);
/* Return <value> as a symbol. It is an error if <value> is no symbol.
 * The corresponding function "symbol_to_value" is located in symbols.h. */

extern value_t canonise_atom_list (value_t atom_list);
/* Convert the list of <atom_list> to canonical format. */

extern value_t symbols_to_list (long_t n, symbol_t symbol_list[]);
/* Convert <symbol_list>, consisting of <n> symbols, to a value. */

extern value_t symbol_to_atoms (symbol_t symbol);
/* Return the list of atoms for <symbol>. */

/*---------------------------------------------------------------------------*/

extern string_t value_to_string (value_t string);
/* Return the value of <string> as a C style string. */

extern value_t string_to_value (string_t string_start, string_t string_end);
/* Return the string starting at <string_start> as a Malaga value.
 * If <string_end> != NULL, it marks the end of the string. */

extern value_t concat_string_values (value_t string1, value_t string2);
/* Concat <string1> and <string2> and return the result.
 * <string1> and <string2> must be strings. */

/*---------------------------------------------------------------------------*/

extern value_t build_record (long_t n, value_t values[]);
/* Return a record of <n> attributes. The <i>-th attribute-value pair is
 * <values>[2<i>-2] : <values>[2<i>-1]" for <i>=1,..,<n>. */

extern value_t join_records (value_t record1, value_t record2);
/* Returns a record of all attributes of <record1> and <record2>, and their
 * associated values. If an attribute has different values in <record1> and 
 * <record2>, the value in <record2> will be taken). 
 * <record1> and <record2> must be records. */

extern value_t get_attribute (value_t record, symbol_t attribute);
/* Return the value of <attribute> in the record <record> 
 * or NULL if it doesn't exist. */

extern value_t select_attribute (value_t record, symbol_t attribute);
/* Return the record that contains <attribute> and its value in <record>.
 * <record> must be a record. */

extern value_t select_attributes (value_t record, value_t list);
/* Return the record with all attributes of <record> that are in <list>.
 * <record> must be a record and <list> must be a list of symbols. */

extern value_t remove_attribute (value_t record, symbol_t attribute);
/* Remove <attribute> from <record> and return the result.
 * <record> must be a record. */

extern value_t remove_attributes (value_t record, value_t list);
/* Remove all attributes in <list> from <record> and return the result.
 * <list> must be a list of symbols; <record> must be a record. */

extern value_t replace_attribute (value_t record, 
				  symbol_t attribute, value_t value);
/* Replace the attribute <attribute> of <record> by <value>.
 * <record> must be a record which contains <attribute>. */

/*---------------------------------------------------------------------------*/

extern value_t build_list (long_t n, value_t elements[]);
/* Return a list of the <n> given elements in <elements>. */

extern value_t concat_lists (value_t list1, value_t list2);
/* Concat <list1> and <list2> and return the result.
 * <list1> and <list2> must be lists. */

extern value_t get_list_difference (value_t list1, value_t list2);
/* Compute the list difference of <list1> and <list2> and return it.
 * An element that appears <m> times in <list1> and <n> times in <list2> 
 * will appear <m-n> times in the result.
 * <list1> and <list2> must be lists. */

extern value_t get_set_difference (value_t list1, value_t list2);
/* Compute the set difference of <list1> and <list2> and return it.
 * Each element of <list1> will be in the result list if it is not 
 * an element of <list2>.
 * <list1> and <list2> must be lists. */

extern value_t intersect_lists (value_t list1, value_t list2);
/* Return the list intersection of <list1> and <list2>.
 * An element that occurs <m> times in <list1> and <n> times in <list2>
 * will occur min(<m>, <n>) times in the result. */

extern long_t get_list_length (value_t list);
/* Return the number of elements in <list>. 
 * <list> must be a list. */

extern value_t get_element (value_t list, long_t n);
/* Return the <n>-th element of the list <list>,
 * or NULL, if that element doesn't exist.
 * If <n> is positive, elements will be counted from the left border.
 * If it's negative, elements will be counted from the right border. */

extern value_t remove_element (value_t list, long_t n);
/* Return the list <list> without element at index <n>.
 * If <n> is positive, the elements will be counted from the left border;
 * if <n> is negative, they will be counted from the right border.
 * If the list contains less than abs(<n>) elements, do nothing. */

extern value_t replace_element (value_t list, long_t n, value_t value);
/* Replace the <n>-th element of <list> by <value>.
 * If <n> is negative, count from the right end.
 * <list> must be a list which contains at least <n> Elements. */

extern value_t list_to_set (value_t list);
/* Return the list <list>, but with multiple occurrences of elements deleted.
 * That means, convert the multi-set <list> to a set. */

/*---------------------------------------------------------------------------*/

extern double value_to_double (value_t value);
/* Return <value> as a C double. */

extern long_t value_to_long (value_t value);
/* Return the value of <value> which must be an integral number value. */

extern value_t double_to_value (double number);
/* Make a Malaga value out of <number>. */

/* type dependent Malaga operations =========================================*/

extern value_t dot_operation (value_t value1, value_t value2);
/* Perform "."-operation, which depends on the type of the values. */

extern value_t plus_operation (value_t value1, value_t value2);
/* Perform "+"-operation, which depends on the type of the values. */

extern value_t minus_operation (value_t value1, value_t value2);
/* Perform "-"-operation, which depends on the type of the values. */

extern value_t asterisk_operation (value_t value1, value_t value2);
/* Perform "*"-operation, which depends on the type of the values. */

extern value_t slash_operation (value_t value1, value_t value2);
/* Perform "/"-operation, which depends on the type of the values. */

extern value_t unary_minus_operation (value_t value);
/* Perform unary "-"-operation, which depends on the type of the values. */

/* functions for value paths ================================================*/

extern value_t build_path (long_t n, value_t elements[]);
/* Return a path which contains of  <n> given elements in <elements>. 
 * Any element must be a numbe, a symbol or a list of numbers and symbols. 
 * If an element is a list, the elements of this list are inserted into the
 * path. */

extern value_t modify_value_part (value_t value, value_t path, 
                                  value_t mod_value,
				  value_t (*modifier) (value_t old_part, 
						       value_t new_value));
/* Return <value>, but modify the part that is described by <path>.
 * <path> must be a list of symbols and numbers <e1, e2, .. , en>.
 * They will be used as nested attributes and indexes, so the part of <value>
 * that is actually modified is <value>.<e1>.<e2>..<en>. If this part does not
 * exist, an error will be reported. Else the function <modifier> will be 
 * called with this part and <new_value> as arguments.
 * The value returned by <modifier> will be entered in <value> in place of
 * <old_part>. */

extern value_t modifier_replace (value_t old_part, value_t new_value);
/* A modifier for "modify_value_part" that replaces <old_part> 
 * by <new_value>. */

extern value_t get_value_part (value_t value, value_t path);
/* Return the value part of <value> that is specified by the attribute path
 * <path>. If that value part doesn't exist, return NULL. */

/* functions for list/record iteration ======================================*/

extern value_t first_element (value_t value);
/* If <value> is a list, return the first element.
 * If <value> is a record, return the first attribute.
 * Return NULL if such an element doesn't exist. */

extern value_t next_element (value_t value, value_t element);
/* Return successor of <element> in <value> or NULL, if no successor exists.
 * <value> must be a list or a record and <element> must be an element or
 * attribute that has been extracted of <value> either by 
 * "first_element" or by "next_element". */

extern value_t value_from_element_onward (value_t value, value_t element);
/* Return the list/record of all elements/attributes of <value> from <element>
 * onward. If <element> is NULL, return an empty list/record.
 * If <value> is a list, <element> must be NULL or an element of it.
 * If <value> is a record, <element> must be NULL or an attribute of it. */

extern value_t value_without_element (value_t value, value_t element);
/* Return the value <value> without its element <element>.
 * If <value> is a list, <element> must be one of its elements.
 * If <value> is a record, <element> must be one of its attributes.
 * If <element> is NULL, the original list/record is returned. */

/* functions to compare values ==============================================*/

extern bool_t values_equal (value_t value1, value_t value2);
/* Return a truth value indicating whether <value1> and <value2> are equal.
 * <value1> an <value2> must be of same type.
 * Refer to documentation to see what "equal" in Malaga really means. */

extern bool_t values_congruent (value_t value1, value_t value2);
/* Return a truth value indicating whether <value1> and <value2> have
 * at least one element in common.
 * <value1> and <value2> must both be symbols or lists. */

extern bool_t value_in_value (value_t value1, value_t value2);
/* Return bool value saying if <value1> is element or attribute of <value2>.
 * <value2> must be a list or a record.
 * If <value2> is a record, then <value1> must be a symbol. */

/* functions to convert values to text ======================================*/

extern string_t value_as_text (string_t output, 
			       value_t value, 
			       string_t output_end,
			       bool_t full_value);
/* Convert <value> to a format readable for humans and store it in <output>
 * which extends to <output_end> (this is a pointer to the first byte after
 * <output>. The pointer returned points to the EOS of the built string.
 * If <full_value> == TRUE, show all attributes, even those that are hidden. */

extern short_t fprint_value (FILE *stream, value_t value);
/* Print <value> in a format readable for humans on <stream>. */

extern short_t print_value (value_t value);
/* Print <value> in a format readable for humans on stdout. */

/*---------------------------------------------------------------------------*/
