/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_sysvideo.c,v 1.16 1999/07/15 18:42:03 slouken Exp $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <windows.h>

#include "SDL.h"
#include "SDL_mutex.h"
#include "SDL_syswm.h"
#include "SDL_sysvideo.h"
#include "SDL_video_c.h"
#include "SDL_sysevents.h"
#include "SDL_events_c.h"
#include "SDL_pixels_c.h"
#include "SDL_syswm_c.h"
#include "SDL_lowvideo.h"

/* Various local structures */
static HBITMAP screen_bmp = NULL;
static HPALETTE screen_pal = NULL;

static void SDL_UpdateVideoInfo(SDL_VideoInfo *info)
{
	return;
}

int SDL_SYS_VideoInit(SDL_PixelFormat *vformat)
{
	HDC    hdc;

#if 0
	/* The window creation runs some messages through WinProc() */
	if ( SDL_StartEventLoop() < 0 ) {
		return(-1);
	}
#endif
	/* Create the window */
	if ( SDL_CreateWindow() < 0 ) {
		return(-1);
	}
	SDL_SYS_SoundFocus(SDL_Window);

	/* Determine the screen depth */
	hdc = GetDC(SDL_Window);
	vformat->BitsPerPixel = GetDeviceCaps(hdc, PLANES) * 
						GetDeviceCaps(hdc, BITSPIXEL);
	switch (vformat->BitsPerPixel) {
		case 16:
			/* GDI defined as 5-5-5 */
			vformat->Rmask = 0x00007c00;
			vformat->Gmask = 0x000003e0;
			vformat->Bmask = 0x0000001f;
			break;
		case 24:
		case 32:
			/* GDI defined as 8-8-8 */
			vformat->Rmask = 0x00ff0000;
			vformat->Gmask = 0x0000ff00;
			vformat->Bmask = 0x000000ff;
			break;
		default:
			break;
	}
	ReleaseDC(SDL_Window, hdc);

	/* Grab an identity palette if we are in a palettized mode */
	if ( vformat->BitsPerPixel <= 8 ) {
		LOGPALETTE *palette;
		int ncolors;
		int i;

		ncolors = 1;
		for ( i=0; i<vformat->BitsPerPixel; ++i ) {
			ncolors *= 2;
		}
		palette = (LOGPALETTE *)alloca(sizeof(*palette)+
					ncolors*sizeof(PALETTEENTRY));
		palette->palVersion = 0x300;
		palette->palNumEntries = ncolors;
		hdc = GetDC(SDL_Window);
		GetSystemPaletteEntries(hdc, 0, ncolors, palette->palPalEntry);
		ReleaseDC(SDL_Window, hdc);
		screen_pal = CreatePalette(palette);
	}

	/* Fill in some window manager capabilities */
	SDL_HWCaps.info.wm_available = 1;
	SDL_WMCaps.SetIcon = SDL_SYS_SetWMIcon;
	SDL_WMCaps.SetCaption = SDL_SYS_SetWMCaption;
	SDL_WMCaps.GetWMInfo = SDL_SYS_GetWMInfo;

	/* Fill in the video hardware capabilities */
	SDL_UpdateVideoInfo(&SDL_HWCaps.info);

	/* We're done! */
	return(0);
}

/* We support any format at any dimension */
SDL_Rect **SDL_SYS_ListModes(SDL_Surface *screen,
					SDL_PixelFormat *format, Uint32 flags)
{
	return((SDL_Rect **)-1);
}

/* Various screen update functions available */
static void SDL_NormalUpdate(SDL_Surface *screen,int numrects,SDL_Rect *rects);

SDL_Surface *SDL_SYS_SetVideoMode(SDL_Surface *current,
				int width, int height, int bpp, Uint32 flags)
{
	SDL_Surface *video;
	BITMAPINFO *binfo;
	HDC hdc;
	RECT bounds;
	int x, y;

	/* Recalculate the bitmasks if necessary */
	if ( bpp == current->format->BitsPerPixel ) {
		video = current;
	} else {
		Uint32 Rmask, Gmask, Bmask;

		switch (bpp) {
			case 16:
				/* GDI defined as 5-5-5 */
				Rmask = 0x00007c00;
				Gmask = 0x000003e0;
				Bmask = 0x0000001f;
				break;
			case 24:
			case 32:
				/* GDI defined as 8-8-8 */
				Rmask = 0x00ff0000;
				Gmask = 0x0000ff00;
				Bmask = 0x000000ff;
				break;
			default:
				Rmask = 0x00000000;
				Gmask = 0x00000000;
				Bmask = 0x00000000;
				break;
		}
		video = SDL_AllocSurface(SDL_SWSURFACE,
					0, 0, bpp, Rmask, Gmask, Bmask, 0);
		if ( video == NULL ) {
			SDL_OutOfMemory();
			return(NULL);
		}
	}

	/* Fill in part of the video surface */
	video->flags = 0;	/* Clear flags */
	video->w = width;
	video->h = height;
	video->pitch = SDL_CalculatePitch(video);

	/* Fill in the bitmap info header */
	if ( video->format->palette != NULL ) {
		binfo = (BITMAPINFO *)alloca(sizeof(*binfo) + 
			video->format->palette->ncolors*sizeof(RGBQUAD));
	} else {
		binfo = (BITMAPINFO *)alloca(sizeof(*binfo));
	}
	if ( binfo == NULL ) {
		if ( video != current ) {
			SDL_FreeSurface(video);
		}
		SDL_OutOfMemory();
		return(NULL);
	}
	binfo->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	binfo->bmiHeader.biWidth = video->w;
	binfo->bmiHeader.biHeight = -video->h;
	binfo->bmiHeader.biPlanes = 1;
	binfo->bmiHeader.biBitCount = video->format->BitsPerPixel;
	binfo->bmiHeader.biCompression = BI_RGB;
	binfo->bmiHeader.biSizeImage = video->h * video->pitch;
	binfo->bmiHeader.biXPelsPerMeter = 0;
	binfo->bmiHeader.biYPelsPerMeter = 0;
	binfo->bmiHeader.biClrUsed = 0;
	binfo->bmiHeader.biClrImportant = 0;
	if ( video->format->palette ) {
		memset(binfo->bmiColors, 0,
			video->format->palette->ncolors*sizeof(RGBQUAD));
	}

	/* Create the offscreen bitmap buffer */
	if ( screen_bmp != NULL ) {
		DeleteObject(screen_bmp);
	}
	hdc = GetDC(SDL_Window);
	screen_bmp = CreateDIBSection(hdc, binfo, DIB_RGB_COLORS,
				(void **)(&video->pixels), NULL, 0);
	ReleaseDC(SDL_Window, hdc);
	if ( screen_bmp == NULL ) {
		if ( video != current ) {
			SDL_FreeSurface(video);
		}
		SDL_SetError("Couldn't create DIB section");
		return(NULL);
	}
	SDL_SYS_UpdateRects = SDL_NormalUpdate;


	/* Set video surface flags */
	if ( bpp <= 8 ) {
		/* BitBlt() maps colors for us */
		video->flags |= SDL_HWPALETTE;
	}

	/* Resize the window */
	SDL_resizing = 1;
	bounds.left = 0;
	bounds.top = 0;
	bounds.right = video->w;
	bounds.bottom = video->h;
	AdjustWindowRect(&bounds, GetWindowLong(SDL_Window, GWL_STYLE), FALSE);
	width = bounds.right-bounds.left;
	height = bounds.bottom-bounds.top;
	x = (GetSystemMetrics(SM_CXSCREEN)-width)/2;
	y = (GetSystemMetrics(SM_CYSCREEN)-height)/2;
	if ( y < 0 ) { /* Cover up title bar for more client area */
		y -= GetSystemMetrics(SM_CYCAPTION)/2;
	}
	SetWindowPos(SDL_Window, NULL, x, y, width, height,
			(SWP_NOCOPYBITS | SWP_NOZORDER | SWP_SHOWWINDOW));
	SDL_resizing = 0;
	SetForegroundWindow(SDL_Window);

	/* We're live! */
	return(video);
}

static void SDL_NormalUpdate(SDL_Surface *screen, int numrects, SDL_Rect *rects)
{
	HDC hdc, mdc;
	int i;

	hdc = GetDC(SDL_Window);
	if ( screen_pal ) {
		SelectPalette(hdc, screen_pal, FALSE);
	}
	mdc = CreateCompatibleDC(hdc);
	SelectObject(mdc, screen_bmp);
	for ( i=0; i<numrects; ++i ) {
		BitBlt(hdc, rects[i].x, rects[i].y, rects[i].w, rects[i].h,
					mdc, rects[i].x, rects[i].y, SRCCOPY);
	}
	DeleteDC(mdc);
	ReleaseDC(SDL_Window, hdc);
}

int SDL_SYS_SetColors(SDL_Surface *screen, int firstcolor, int ncolors)
{
	SDL_Palette *palette;
	RGBQUAD *pal;
	int i, j;
	HDC hdc, mdc;

	/* Update the display palette */
	hdc = GetDC(SDL_Window);
	palette = screen->format->palette;
	if ( screen_pal ) {
		PALETTEENTRY *entries;

		entries = (PALETTEENTRY *)alloca(ncolors*sizeof(PALETTEENTRY));
		for ( i=0, j=firstcolor; i<ncolors; ++i, ++j ) {
			entries[i].peRed   = palette->colors[j].r;
			entries[i].peGreen = palette->colors[j].g;
			entries[i].peBlue  = palette->colors[j].b;
			entries[i].peFlags = PC_NOCOLLAPSE;
		}
		SetPaletteEntries(screen_pal, firstcolor, ncolors, entries);
		SelectPalette(hdc, screen_pal, FALSE);
		RealizePalette(hdc);
	}

	/* Copy palette colors into DIB palette */
	pal = (RGBQUAD *)alloca(palette->ncolors*sizeof(RGBQUAD));
	for ( i=0; i<palette->ncolors; ++i ) {
		pal[i].rgbRed = palette->colors[i].r;
		pal[i].rgbGreen = palette->colors[i].g;
		pal[i].rgbBlue = palette->colors[i].b;
		pal[i].rgbReserved = 0;
	}

	/* Set the DIB palette and update the display */
	mdc = CreateCompatibleDC(hdc);
	SelectObject(mdc, screen_bmp);
	SetDIBColorTable(mdc, 0, palette->ncolors, pal);
	BitBlt(hdc, 0, 0, screen->w, screen->h, mdc, 0, 0, SRCCOPY);
	DeleteDC(mdc);
	ReleaseDC(SDL_Window, hdc);
	return(1);
}

void SDL_SYS_VideoQuit(SDL_Surface *screen)
{
	/* Destroy the window and everything associated with it */
	if ( SDL_Window ) {
		SDL_DestroyWindow();

		/* Delete the screen bitmap (also frees screen->pixels) */
		if ( screen_bmp ) {
			DeleteObject(screen_bmp);
			screen_bmp = NULL;
		}
		if ( screen ) {
			screen->pixels = NULL;
		}
		if ( screen_icn ) {
			DestroyIcon(screen_icn);
			screen_icn = NULL;
		}
		SDL_Window = NULL;
	}
}
void SDL_SYS_FinalQuit(void) { }

/* Exported for the windows message loop only */
static void SDL_FocusPalette(int foreground)
{
	if ( screen_pal != NULL ) {
		HDC hdc;

		hdc = GetDC(SDL_Window);
		SelectPalette(hdc, screen_pal, FALSE);
		if ( RealizePalette(hdc) )
			InvalidateRect(SDL_Window, NULL, FALSE);
		ReleaseDC(SDL_Window, hdc);
	}
}
void SDL_RealizePalette(void)
{
	SDL_FocusPalette(1);
}
void SDL_PaletteChanged(HWND window)
{
	if ( window != SDL_Window ) {
		SDL_FocusPalette(0);
	}
}

/* Exported for the windows message loop only */
void SDL_WinPAINT(HDC hdc)
{
	HDC mdc;

	if ( screen_pal ) {
		SelectPalette(hdc, screen_pal, FALSE);
	}
	mdc = CreateCompatibleDC(hdc);
	SelectObject(mdc, screen_bmp);
	BitBlt(hdc, 0, 0, SDL_VideoSurface->w, SDL_VideoSurface->h,
							mdc, 0, 0, SRCCOPY);
	DeleteDC(mdc);
}

