/* 
 * tkXIM.c --
 *
 *      This file contains modules to implement the XIM protocol session.
 *
 *	Author:	m-hirano@sra.co.jp
 *
 * Copyright 1998 Software Research Associates, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Software Research Associates not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  Software Research
 * Associates makes no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 */

#ifndef lint
static char rcsid[] = "$Header: /home/m-hirano/cvsroot/tcltk/tk8/unix/tkXIM.c,v 1.10 1998/11/27 13:25:15 m-hirano Exp $";
#endif /* !lint */

#include "tkPort.h"
#include "tkInt.h"

#include "tkFont.h"

#ifdef TK_USE_INPUT_METHODS
#ifdef XIM_IMPROVE

static XIMStyle		GetStyleValueByStyleString _ANSI_ARGS_ ((char *str));
static char *		GetStyleStringByStyleValue _ANSI_ARGS_ ((XIMStyle val));
static void		TkpIMGenericHandler _ANSI_ARGS_ ((ClientData clientData, XEvent *eventPtr));
#ifdef CHECK_XTT
static Tk_Uid		GetSaneFontName _ANSI_ARGS_ ((Tk_Window tkwin, unsigned long atom));
#endif /* CHECK_XTT */
static Tk_Uid		GetFontNames _ANSI_ARGS_ ((Tcl_Interp *interp, Tk_Window tkwin, char *str,
						   char **aName, char **kName));
static Window		TkpIMGetICFocusWindow _ANSI_ARGS_ ((XIC ic));
static XPoint *		TkpIMSendSpotLocation _ANSI_ARGS_ ((XIC ic, XPoint *spot));
static XPoint *		TkpIMGetSpotLocation _ANSI_ARGS_ ((XIC ic, XPoint *spot));
static XRectangle *	TkpIMSendArea _ANSI_ARGS_ ((XIC ic, char *attr, XRectangle *rect));
static XFontSet		TkpIMSendFontSet _ANSI_ARGS_ ((XIC ic, XFontSet fontset));
static int		TkpIMSendColor _ANSI_ARGS_ ((XIC ic,
						     unsigned long fg, unsigned long bg, Colormap cmap));
static TkWindow *	TkpIMGetPreferredArea _ANSI_ARGS_ ((XIC ic, TkWindow *winPtr));

static void		DumpICAttribute _ANSI_ARGS_ ((Tcl_Interp *interp, TkWindow *winPtr));
static int		ConfigGetICAttribute _ANSI_ARGS_ ((Tcl_Interp *interp, TkWindow *winPtr,
							   int objc, Tcl_Obj *CONST objv[]));
static int		ConfigSetICAttribute _ANSI_ARGS_ ((Tcl_Interp *interp, TkWindow *winPtr,
							   int objc, Tcl_Obj *CONST objv[], int force));

static XIC		TkpCreateIC _ANSI_ARGS_ ((TkWindow *winPtr));
static void		TkpDestroyIC _ANSI_ARGS_ ((TkWindow *winPtr));
static int		TkpConfigIC _ANSI_ARGS_ ((Tcl_Interp *interp, TkWindow *winPtr));

#define TkIMPreeditMask	(XIMPreeditArea|XIMPreeditCallbacks|XIMPreeditPosition|XIMPreeditNothing|XIMPreeditNone)
#define TkIMStatusMask	(XIMStatusArea|XIMStatusCallbacks|XIMStatusNothing|XIMStatusNone)

typedef struct {
    char *styleString;
    XIMStyle styleVal;
} TkpIMStyle;

static TkpIMStyle imStyle[] = {
    { "PreeditArea", XIMPreeditArea },
    { "PreeditCallbacks", XIMPreeditCallbacks },
    { "PreeditPosition", XIMPreeditPosition },
    { "PreeditNothing", XIMPreeditNothing },
    { "PreeditNone", XIMPreeditNone },
    { "StatusArea", XIMStatusArea },
    { "StatusCallbacks", XIMStatusCallbacks },
    { "StatusNothing", XIMStatusNothing },
    { "StatusNone", XIMStatusNone },
};
#define StyleTableSize(a) (sizeof(a) / sizeof(a[0]))

static XIMStyle 
GetStyleValueByStyleString(str)
     char *str;
{
    int i;

    if (str == NULL || *str == 0) return (XIMStyle)0L;

    for (i = 0; i < StyleTableSize(imStyle); i++) {
	if (strcmp(imStyle[i].styleString, str) == 0) {
	    return imStyle[i].styleVal;
	}
    }
    return (XIMStyle)0L;
}


static char imStrBuf[1024];
static char *
GetStyleStringByStyleValue(val)
     XIMStyle val;
{
    int len = 0;
    int i;

    if (val == (XIMStyle)0L) return NULL;
    for (i = 0; i < StyleTableSize(imStyle); i++) {
	if (val & imStyle[i].styleVal) {
	    sprintf(imStrBuf + len, "%s ", imStyle[i].styleString);
	    len += (strlen(imStyle[i].styleString) + 1);
	}
    }
    if (len == 0) return NULL;
    if (imStrBuf[len - 1] == ' ') {
	imStrBuf[len - 1] = 0;
    } else {
	imStrBuf[len] = 0;
    }
    return imStrBuf;
}


static void
TkpIMGenericHandler(clientData, eventPtr)
     ClientData clientData;
     XEvent *eventPtr;
{
    Tk_Window tkwin = (Tk_Window)clientData;
    TkWindow *winPtr = (TkWindow *)clientData;
    
    if (eventPtr->xany.window != Tk_WindowId(tkwin) ||
	winPtr->inputContext == NULL ||
	!(winPtr->flags & TK_CHECKED_IC)) {
	/*
	 * Why ME ???
	 */
	return;
    }

    /*
     * Well, should I care about Enter/Leave ?
     */
    switch (eventPtr->type) {
        case KeyPress:
        case FocusIn: {
	    Window root, child;
	    int rootX, rootY;
	    int wX, wY;
	    unsigned int mask;

	    /*
	     * Care about case mouse pointer is not on winPtr->window.
	     * In such a case, IM server can't fetch any events from Tk.
	     */

	    if (XQueryPointer(winPtr->display,
			      RootWindow(winPtr->display, winPtr->screenNum),
			      &root, &child, &rootX, &rootY, &wX, &wY, &mask) == True) {
		TkWindow *pWin = (TkWindow *)Tk_CoordsToWindow(rootX, rootY,
							       (Tk_Window)winPtr);
		if (pWin == NULL) goto doFocus;
		if (pWin->window != winPtr->window) {
#if 0
		    /*
		     * Code belows are what I REALLY want to do. But,
		     * in XIMPreeditPosition mode, IM server use the
		     * focus window as key event source window and as
		     * PreeditArea window by X11 specification. I want
		     * IM server to use this focus window ONLY as key
		     * event source. mean:
		     *
		     * 	o PreeditArea and PreeditPosition take place
		     * 	  within client window (winPtr->window).
		     *	o KeyPress event source is the window in which
		     *	  mouse pointer is (pWin->window).
		     */

		    if (XSetICValues(winPtr->inputContext, XNFocusWindow,
				     pWin->window, NULL) != NULL) {
			fprintf(stderr, "debugIC: can't set IC focus to pointer window 0x%08x\n",
				pWin->window);
		    } else {
			fprintf(stderr, "debugIC: set IC focus to pointer window 0x%08x\n",
				pWin->window);
		    }
#endif
		    doFocus:
		    TkpChangeFocus(winPtr, 1);
#ifdef XIM_DEBUG
		    fprintf(stderr, "debugIC: X focus to '%s' 0x%08x\n", winPtr->pathName,
			    winPtr->window);
#endif /* XIM_DEBUG */
		    
		}
	    }
#ifdef XIM_DEBUG
	    fprintf(stderr, "debugIC: IC focus to '%s' 0x%08x\n", winPtr->pathName,
		    winPtr->window);
#endif /* XIM_DEBUG */
	    XSetICFocus(winPtr->inputContext);
	    break;
	}

        case FocusOut: {
	    XUnsetICFocus(winPtr->inputContext);
	    break;
	}

	case DestroyNotify: {
	    XUnsetICFocus(winPtr->inputContext);
	    TkpDeleteIMGenericHandler(tkwin);
	    break;
	}
    }
}


#ifdef CHECK_XTT
static Tk_Uid
GetSaneFontName(tkwin, atom)
     Tk_Window tkwin;
     unsigned long atom;
{
    Tk_Uid aName = Tk_GetUid(Tk_GetAtomName(tkwin, atom));
    Tk_Uid sName = NormalizeXLFD(aName);

    if (aName == sName) {
	return aName;
    } else {
	return sName;
    }
    return NULL;
}
#else
#define GetSaneFontName(tkwin, atom) Tk_GetUid(Tk_GetAtomName(tkwin, atom))
#endif /* CHECK_XTT */


/*
 * returned value is Tk_GetUid()'d.
 */
Tk_Uid
GetFontNames(interp, tkwin, str, aNameRet, kNameRet)
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *str;
     char **aNameRet;
     char **kNameRet;
{
    char *aName = NULL;
    char *kName = NULL;
    char *ret = NULL;
    /*
     * Byte length of a font name is smaller than 256.
     * (If you doubt it, check Xlib code :^>)
     * Maybe 1024 is enuff for room of fontset name.
     */
    char fontNames[1024];

    Tk_Font tkfont = Tk_GetFont(interp, tkwin, Tk_GetUid(str));

    if (tkfont != NULL) {
	if (Tk_FontType(tkfont) == TK_FONT_COMPOUND) {
	    aName = TkpGetFontPropertyName(tkwin, TkpGetAsciiFontStruct(tkfont));
	    kName = TkpGetFontPropertyName(tkwin, TkpGetKanjiFontStruct(tkfont));
	} else {
	    Tk_Font defFont;
	    aName = TkpGetFontPropertyName(tkwin, TkpGetFontStruct(tkfont));
	    defFont = TkpGetDefaultFontByDisplay(Tk_Display(tkwin));
	    if (defFont != NULL) {
		kName = TkpGetFontPropertyName(tkwin, TkpGetFontStruct(defFont));
	    }
	}

	if (aName != NULL && kName != NULL) {
	    sprintf(fontNames, "%s,%s", aName, kName);
	    if (aNameRet != NULL) {
		*aNameRet = aName;
	    }
	    if (kNameRet != NULL) {
		*kNameRet = kName;
	    }
	    ret = Tk_GetUid(fontNames);
	}
	Tk_FreeFont(tkfont);
    }

    return ret;
}


static Window
TkpIMGetICFocusWindow(ic)
     XIC ic;
{
    Window ret = None;
    Window fw;
    if (XGetICValues(ic, XNFocusWindow, &fw, NULL) != NULL) {
	ret = None;
    } else {
	ret = fw;
    }
    return ret;
}


static XPoint *
TkpIMSendSpotLocation(ic, spot)
     XIC ic;
     XPoint *spot;
{
    XPoint *ret = spot;
    XVaNestedList list;
    list = XVaCreateNestedList(0, XNSpotLocation, spot, NULL);
    if (XSetICValues(ic, XNPreeditAttributes, list, NULL) != NULL) {
	ret = NULL;
    }
    XFree(list);
    return ret;
}


static XPoint *
TkpIMGetSpotLocation(ic, spot)
     XIC ic;
     XPoint *spot;
{
    XPoint *ret;
    XVaNestedList list = XVaCreateNestedList(0, XNSpotLocation, &ret, NULL);
    if (XGetICValues(ic, XNPreeditAttributes, list, NULL) != NULL) {
	return NULL;
    } else {
	*spot = *ret;
	XFree(ret);
	return spot;
    }
}


static XRectangle *
TkpIMSendArea(ic, attr, rect)
     XIC ic;
     char *attr;
     XRectangle *rect;
{
    XRectangle *ret = rect;
    XVaNestedList list;
    list = XVaCreateNestedList(0, XNArea, rect, NULL);
    if (XSetICValues(ic, attr, list, NULL) != NULL) {
	ret = NULL;
    }
    XFree(list);
    return ret;
}


static XFontSet
TkpIMSendFontSet(ic, fontset)
     XIC ic;
     XFontSet fontset;
{
    XFontSet ret = fontset;
    XVaNestedList peList = XVaCreateNestedList(0, XNFontSet, fontset, NULL);
    if (XSetICValues(ic,
		     XNPreeditAttributes, peList,
		     NULL) != NULL) {
	ret = NULL;
    }
    XFree(peList);
    return ret;
}


static int
TkpIMSendColor(ic, fg, bg, cmap)
     XIC ic;
     unsigned long fg;
     unsigned long bg;
     Colormap cmap;
{
    int ret = 1;
    unsigned long fgVal = fg;
    unsigned long bgVal = bg;

    XVaNestedList peList = XVaCreateNestedList(0, XNColormap, cmap, XNForeground, fgVal,
					       XNBackground, bgVal, NULL);
    if (XSetICValues(ic,
		     XNPreeditAttributes, peList,
		     XNStatusAttributes, peList,
		     NULL) != NULL) {
	ret = 0;
    }
    XFree(peList);
    return ret;
}


static TkWindow *
TkpIMGetPreferredArea(ic, winPtr)
     XIC ic;
     TkWindow *winPtr;
{
    TkWindow *ret = NULL;
    if ((winPtr->icAttr->styleVal & XIMStatusArea) &&
	winPtr->inputContext != NULL) {
	XRectangle area, *pArea;
	XVaNestedList list;

	area.x = area.y = area.height = area.width = 0;

	list = XVaCreateNestedList(0, XNAreaNeeded, &area, NULL);
	if (XSetICValues(ic, XNStatusAttributes, list, NULL) != NULL) {
	    winPtr->icAttr->stPrefAreaVal.x = 
	    winPtr->icAttr->stPrefAreaVal.y =
	    winPtr->icAttr->stPrefAreaVal.width =
	    winPtr->icAttr->stPrefAreaVal.height = 0;
	    return ret;
	}
	XFree(list);
	list = XVaCreateNestedList(0, XNAreaNeeded, &pArea, NULL);
	if (XGetICValues(ic, XNStatusAttributes, list, NULL) != NULL) {
	    winPtr->icAttr->stPrefAreaVal.x = 
	    winPtr->icAttr->stPrefAreaVal.y =
	    winPtr->icAttr->stPrefAreaVal.width =
	    winPtr->icAttr->stPrefAreaVal.height = 0;
	} else {
	    winPtr->icAttr->stPrefAreaVal.x = pArea->x;
	    winPtr->icAttr->stPrefAreaVal.y = pArea->y;
	    winPtr->icAttr->stPrefAreaVal.width = pArea->width;
	    winPtr->icAttr->stPrefAreaVal.height = pArea->height;
	    XFree(pArea);
	    ret = winPtr;
	}
	XFree(list);
    }
    return ret;
}


static char *imOptionStrings[] = {
    "-status",
    "-supportedStyle",
    "-style",
    "-spot",
    "-preeditArea",
    "-statusArea",
    "-preferredStatusArea",
    "-font",
    "-foreground",
    "-background",
    NULL
};

typedef enum {
    IM_STATUS,
    IM_SUPPORTEDSTYLE,
    IM_STYLE,
    IM_SPOT,
    IM_PREEDITAREA,
    IM_STATUSAREA,
    IM_PREFERREDSTATUSAREA,
    IM_FONT,
    IM_FOREGROUND,
    IM_BACKGROUND,
} imOptions;


#define isNotNULL(A) (((icAttr->A) != NULL) ? (icAttr->A) : "{}")
#define isNotLISTNULL(A) (((icAttr->A) != NULL) ? (icAttr->A) : "")

static void
DumpICAttribute(interp, winPtr)
     Tcl_Interp *interp;
     TkWindow *winPtr;
{
    Tcl_Obj *resultPtr = Tcl_GetObjResult(interp);
    TkpICAttribute *icAttr = winPtr->icAttr;
    int i;
    char buf[4096];
    int len = 0;
#define WRITEIT(X, Y) \
    sprintf(buf + len, ((Y == 1) ? "{%s} " : "%s "), (X)); \
    len = strlen(buf);
    int n = sizeof(imOptionStrings) / sizeof(char *) -1;

    for (i = 0; i < n; i++) {

	sprintf(buf + len, "%s ", imOptionStrings[i]);
	len = strlen(buf);

	switch ((imOptions)i) {

	    case IM_STATUS: {
		if (winPtr->inputContext != NULL &&
		    (winPtr->flags & TK_CHECKED_IC)) {
		    sprintf(buf + len, "started ");
		    len += 8;
		} else {
		    sprintf(buf + len, "never ");
		    len += 6;
		}
		break;
	    }

	    case IM_SUPPORTEDSTYLE: {
		char buf2[1024];
		char *style = NULL;
		int len2 = 0;
		int j;
		if (winPtr->dispPtr->imSupportedStyle == NULL) {
		    sprintf(buf + len, "{} ");
		    len += 3;
		} else {
		    for (j = 0; j < winPtr->dispPtr->imSupportedStyle->count_styles; j++) {
			style = GetStyleStringByStyleValue(winPtr->dispPtr->imSupportedStyle->supported_styles[j]);
			sprintf(buf2 + len2, "{%s} ", (style != NULL) ? style : "");
			len2 = strlen(buf2);
		    }
		    if (buf2[len2 - 1] == ' ') {
			buf2[len2 - 1] = 0;
		    } else {
			buf2[len2] = 0;
		    }
		    sprintf(buf + len, "{%s} ", buf2);
		    len = strlen(buf);
		}
		break;
	    }

	    case IM_STYLE: {
		WRITEIT(isNotLISTNULL(style), 1);
		break;
	    }
	    
	    case IM_SPOT: {
		if ((icAttr->styleVal & XIMPreeditPosition) &&
		    winPtr->inputContext != NULL) {
		    char spBuf[1024];
		    XPoint spot;
		    if (TkpIMGetSpotLocation(winPtr->inputContext, &spot) == &spot) {
			sprintf(spBuf, "%d %d", spot.x, spot.y);
			sprintf(buf + len, "{%s} ", spBuf);
			len = strlen(buf);
		    } else {
			WRITEIT(isNotLISTNULL(spot), 1);			
		    }
		} else {
		    WRITEIT(isNotLISTNULL(spot), 1);
		}
		break;
	    }
	    
	    case IM_PREEDITAREA: {
		WRITEIT(isNotLISTNULL(peArea), 1);
		break;
	    }

	    case IM_STATUSAREA: {
		WRITEIT(isNotLISTNULL(stArea), 1);
		break;
	    }

	    case IM_PREFERREDSTATUSAREA: {
		if (TkpIMGetPreferredArea(winPtr->inputContext, winPtr) != winPtr) {
		    sprintf(buf + len, "{} ");
		} else {
		    sprintf(buf + len, "{%d %d %d %d} ",
			    icAttr->stPrefAreaVal.x,
			    icAttr->stPrefAreaVal.y,
			    icAttr->stPrefAreaVal.width,
			    icAttr->stPrefAreaVal.height);
		}
		len = strlen(buf);
		break;
	    }

	    case IM_FONT: {
		WRITEIT(isNotNULL(font), 0);
		break;
	    }

	    case IM_FOREGROUND: {
		WRITEIT(isNotNULL(fg), 0);
		break;
	    }

	    case IM_BACKGROUND: {
		WRITEIT(isNotNULL(bg), 0);
		break;
	    }
	}
    }
    buf[len] = 0;
    
    Tcl_AppendStringsToObj(resultPtr, buf, NULL);
#undef WRITEIT
#undef isNotNULL
#undef isNotLISTNULL
}


#define isNotNULL(A) (((icAttr->A) != NULL) ? (icAttr->A) : NULL)
static int
ConfigGetICAttribute(interp, winPtr, objc, objv)
     Tcl_Interp *interp;
     TkWindow *winPtr;
     int objc;
     Tcl_Obj *CONST objv[];
{
    char *string = NULL;
    char buf[1024];
    int index;
    TkpICAttribute *icAttr = winPtr->icAttr;
    Tcl_Obj *resultPtr = Tcl_GetObjResult(interp);
    
    if (Tcl_GetIndexFromObj(interp, objv[0], imOptionStrings, "option", 0, &index)
	!= TCL_OK) {
	return TCL_ERROR;
    }

    switch ((imOptions)index) {
	
	case IM_STATUS: {
	    if (winPtr->inputContext != NULL &&
		(winPtr->flags & TK_CHECKED_IC)) {
		string = "started";
	    } else {
		string = "never";
	    }
	    break;
	}

	case IM_SUPPORTEDSTYLE: {
	    int len = 0;
	    int j;
	    if (winPtr->dispPtr->imSupportedStyle == NULL) {
		string = "";
	    } else {
		for (j = 0; j < winPtr->dispPtr->imSupportedStyle->count_styles; j++) {
		    sprintf(buf + len, "{%s} ",
			    GetStyleStringByStyleValue(winPtr->dispPtr->imSupportedStyle->supported_styles[j]));
		    len = strlen(buf);
		}
		buf[len] = 0;
		string = buf;
	    }
	    break;
	}

	case IM_STYLE: {
	    string = isNotNULL(style);
	    break;
	}
	
	case IM_SPOT: {
	    if ((icAttr->styleVal & XIMPreeditPosition) &&
		winPtr->inputContext != NULL) {
		XPoint spot;
		if (TkpIMGetSpotLocation(winPtr->inputContext, &spot) == &spot) {
		    sprintf(buf, "%d %d", spot.x, spot.y);
		    string = buf;
		} else {
		    string = isNotNULL(spot);
		}
	    } else {
		string = isNotNULL(spot);
	    }
	    break;
	}

	case IM_PREEDITAREA: {
	    string = isNotNULL(peArea);
	    break;
	}

	case IM_STATUSAREA: {
	    string = isNotNULL(stArea);
	    break;
	}

	case IM_PREFERREDSTATUSAREA: {
	    if (TkpIMGetPreferredArea(winPtr->inputContext, winPtr) != winPtr) {
		string = NULL;
	    } else {
		sprintf(buf, "%d %d %d %d",
			icAttr->stPrefAreaVal.x,
			icAttr->stPrefAreaVal.y,
			icAttr->stPrefAreaVal.width,
			icAttr->stPrefAreaVal.height);
		string = buf;
	    }
	    break;
	}
	
	case IM_FONT: {
	    string = isNotNULL(font);
	    break;
	}

	case IM_FOREGROUND: {
	    string = isNotNULL(fg);
	    break;
	}

	case IM_BACKGROUND: {
	    string = isNotNULL(bg);
	    break;
	}
    }

    if (string != NULL) {
	Tcl_AppendStringsToObj(resultPtr, string, NULL);
    }
    return TCL_OK;
}
#undef isNotNULL


static int
ConfigSetICAttribute(interp, winPtr, objc, objv, force)
     Tcl_Interp *interp;
     TkWindow *winPtr;
     int objc;
     Tcl_Obj *CONST objv[];
     int force;
{
    int i;
    int index;
    char *string;
    TkpICAttribute *icAttr = winPtr->icAttr;
    Tcl_Obj *resultPtr = Tcl_GetObjResult(interp);
    int lObjc;
    Tcl_Obj **lObjv;
    int oIdx;
    Tk_Uid newVal;

    if (objc & 1) {
	string = Tcl_GetStringFromObj(objv[objc - 1], NULL);
        Tcl_AppendStringsToObj(Tcl_GetObjResult(interp), "missing value for \"",
			       string, "\" option", (char *) NULL);
        return TCL_ERROR;
    }

    icAttr->isChanged = IM_NOTHING_CHANGE;

    for (i = 0; i < objc; i += 2) {
	if (Tcl_GetIndexFromObj(interp, objv[i], imOptionStrings, "option", 0, &index)
	    != TCL_OK) {
	    return TCL_ERROR;
	}
	oIdx = i + 1;
	string = Tcl_GetStringFromObj(objv[oIdx], NULL);
	newVal = Tk_GetUid(string);
	
	switch ((imOptions)index) {

	    case IM_STATUS:
	    case IM_SUPPORTEDSTYLE: {
		break;
	    }

	    case IM_STYLE: {
		if (Tcl_ListObjGetElements(interp, objv[oIdx], &lObjc, &lObjv) != TCL_OK) {
		    return TCL_ERROR;
		}
		if (lObjc < 2) {
		    Tcl_AppendStringsToObj(resultPtr, "style must be {preeditSpec statusSpec}", NULL);
		    return TCL_ERROR;
		}
		if (newVal != icAttr->style || icAttr->style == NULL) {
		    XIMStyle preedit = (XIMStyle)0L;
		    XIMStyle status = (XIMStyle)0L;
		    char *pStyle = NULL;
		    char *sStyle = NULL;
		    int j;
		    int match = 0;

		    pStyle = Tcl_GetStringFromObj(lObjv[0], NULL);
		    sStyle = Tcl_GetStringFromObj(lObjv[1], NULL);
		    preedit = GetStyleValueByStyleString(pStyle);
		    if (preedit == (XIMStyle)0L) {
			Tcl_AppendStringsToObj(resultPtr, "invalid style \"", pStyle, "\".", NULL);
			return TCL_ERROR;
		    }
		    status = GetStyleValueByStyleString(sStyle);
		    if (status == (XIMStyle)0L) {
			Tcl_AppendStringsToObj(resultPtr, "invalid style \"", sStyle, "\".", NULL);
			return TCL_ERROR;
		    }
		    
		    /* Check the value is in the style list. */
		    for (j = 0; j < winPtr->dispPtr->imSupportedStyle->count_styles; j++) {
			if ((preedit|status) == winPtr->dispPtr->imSupportedStyle->supported_styles[j]) {
			    match = 1;
			    break;
			}
		    }
		    if (match == 0) {
			Tcl_AppendStringsToObj(resultPtr, "style \"", newVal, "\" is not supported by the input method server.", NULL);
			return TCL_ERROR;
		    }
		    icAttr->styleVal = preedit|status;
		    icAttr->style = newVal;
		    icAttr->isChanged |= IM_STYLE_CHANGE;
		}
		break;
	    }
	    
	    case IM_SPOT: {
		if (Tcl_ListObjGetElements(interp, objv[oIdx], &lObjc, &lObjv) != TCL_OK) {
		    return TCL_ERROR;
		}
		if (lObjc != 2) {
		    Tcl_AppendStringsToObj(resultPtr, "spot must be {x y}.", NULL);
		    return TCL_ERROR;
		}
		if (newVal != icAttr->spot || icAttr->spot == NULL || force) {
		    int x, y;
		    icAttr->spot = newVal;
		    if (Tcl_GetIntFromObj(interp, lObjv[0], &x) != TCL_OK) {
			return TCL_ERROR;
		    }
		    if (Tcl_GetIntFromObj(interp, lObjv[1], &y) != TCL_OK) {
			return TCL_ERROR;
		    }
		    icAttr->spotVal.x = x;
		    icAttr->spotVal.y = y;
		    icAttr->isChanged |= IM_SPOT_CHANGE;
		}
		break;
	    }

	    case IM_PREEDITAREA: {
		if (Tcl_ListObjGetElements(interp, objv[oIdx], &lObjc, &lObjv) != TCL_OK) {
		    return TCL_ERROR;
		}
		if (lObjc != 4) {
		    Tcl_AppendStringsToObj(resultPtr, "area must be {x y w h}.", NULL);
		    return TCL_ERROR;
		}
		if (newVal != icAttr->peArea || icAttr->peArea == NULL || force) {
		    int x, y, w, h;
		    icAttr->peArea = newVal;
		    if (Tcl_GetIntFromObj(interp, lObjv[0], &x) != TCL_OK) {
			return TCL_ERROR;
		    }
		    if (Tcl_GetIntFromObj(interp, lObjv[1], &y) != TCL_OK) {
			return TCL_ERROR;
		    }
		    if (Tcl_GetIntFromObj(interp, lObjv[2], &w) != TCL_OK) {
			return TCL_ERROR;
		    }
		    if (Tcl_GetIntFromObj(interp, lObjv[3], &h) != TCL_OK) {
			return TCL_ERROR;
		    }
		    icAttr->peAreaVal.x = x;
		    icAttr->peAreaVal.y = y;
		    icAttr->peAreaVal.width = w;
		    icAttr->peAreaVal.height = h;
		    icAttr->isChanged |= IM_PREEDITAREA_CHANGE;
		}
		break;
	    }

	    case IM_STATUSAREA: {
		if (Tcl_ListObjGetElements(interp, objv[oIdx], &lObjc, &lObjv) != TCL_OK) {
		    return TCL_ERROR;
		}
		if (lObjc != 4) {
		    Tcl_AppendStringsToObj(resultPtr, "area must be {x y w h}.", NULL);
		    return TCL_ERROR;
		}
		if (newVal != icAttr->stArea || icAttr->stArea == NULL || force) {
		    int x, y, w, h;
		    icAttr->stArea = newVal;
		    if (Tcl_GetIntFromObj(interp, lObjv[0], &x) != TCL_OK) {
			return TCL_ERROR;
		    }
		    if (Tcl_GetIntFromObj(interp, lObjv[1], &y) != TCL_OK) {
			return TCL_ERROR;
		    }
		    if (Tcl_GetIntFromObj(interp, lObjv[2], &w) != TCL_OK) {
			return TCL_ERROR;
		    }
		    if (Tcl_GetIntFromObj(interp, lObjv[3], &h) != TCL_OK) {
			return TCL_ERROR;
		    }
		    icAttr->stAreaVal.x = x;
		    icAttr->stAreaVal.y = y;
		    icAttr->stAreaVal.width = w;
		    icAttr->stAreaVal.height = h;
		    icAttr->isChanged |= IM_STATUSAREA_CHANGE;
		}
		break;
	    }

	    case IM_PREFERREDSTATUSAREA: {
		break;
	    }

	    case IM_FONT: {
#ifdef USE_XFONTSET
		char **ml;
		int mc;
		char *ds;
#endif /* USE_XFONTSET */
		char *aName;
		char *kName;

		Tk_Uid new = GetFontNames(interp, (Tk_Window)winPtr, newVal, &aName, &kName);
		XFontSet nf = NULL;

		if (new == NULL) {
		    Tcl_AppendStringsToObj(resultPtr, "Can't find font \"", newVal,
					   "\".", NULL);
		    return TCL_ERROR;
		}
		if (new != icAttr->fontlist || icAttr->fontlist == NULL || force) {
		    if (icAttr->fontset != NULL) {
#ifdef USE_XFONTSET
			XFreeFontSet(Tk_Display(winPtr), icAttr->fontset);
#else
			TkpFreeFontSet((Tk_Window)winPtr, icAttr->fontset);
#endif /* USE_XFONTSET */
		    }
#ifdef USE_XFONTSET
		    nf = XCreateFontSet(Tk_Display(winPtr), new, &ml, &mc, &ds);
#else
		    nf = TkpCreateFontSet((Tk_Window)winPtr, new, aName, kName);
#endif /* USE_XFONTSET */
		    if (nf == NULL) {
			icAttr->font = NULL;
			icAttr->fontlist = NULL;
			icAttr->fontset = NULL;
			Tcl_AppendStringsToObj(resultPtr, "Can't create fontset \"", newVal,
					       "\" -> \"", new, "\".", NULL);
			return TCL_ERROR;
		    }
		    icAttr->font = newVal;
		    icAttr->fontlist = new;
		    icAttr->fontset = nf;
		    icAttr->isChanged |= IM_FONT_CHANGE;
		}
		break;
	    }

	    case IM_FOREGROUND: {
		if (newVal != icAttr->fg || icAttr->fg == NULL || force) {
		    XColor *c = Tk_GetColor(interp, (Tk_Window)winPtr, newVal);
		    if (c == NULL) {
			Tcl_AppendStringsToObj(resultPtr, "Can't find color \"", string, 
					       "\".", NULL);
			return TCL_ERROR;
		    }
		    icAttr->fg = newVal;
		    icAttr->fgVal = c->pixel;
		    icAttr->isChanged |= IM_FOREGROUND_CHANGE;
		}
		break;
	    }

	    case IM_BACKGROUND: {
		if (newVal != icAttr->bg || icAttr->bg == NULL || force) {
		    XColor *c = Tk_GetColor(interp, (Tk_Window)winPtr, newVal);
		    if (c == NULL) {
			Tcl_AppendStringsToObj(resultPtr, "Can't find color \"", string, 
					       "\".", NULL);
			return TCL_ERROR;
		    }
		    icAttr->bg = newVal;
		    icAttr->bgVal = c->pixel;
		    icAttr->isChanged |= IM_BACKGROUND_CHANGE;
		}
		break;
	    }
	}
    }
    return TCL_OK;
}


static XIC
TkpCreateIC(winPtr)
     TkWindow *winPtr;
{
    XIC ret = NULL;

    XIMStyle style = winPtr->icAttr->styleVal;
    XIMStyle preedit = winPtr->icAttr->styleVal & TkIMPreeditMask;
    XIMStyle status = winPtr->icAttr->styleVal & TkIMStatusMask;

    XVaNestedList pList = NULL;
    XVaNestedList sList = NULL;
    XPoint spot;
    XRectangle pArea;
    XRectangle sArea;
    XFontSet fontset = winPtr->icAttr->fontset;
    int doFont = (fontset == NULL) ? 0 : 1;
    int doSpot = 0;
    int doPArea = 0;
    int doSArea = 0;

    switch (preedit) {
	
	case XIMPreeditArea: {
	    pArea.x = winPtr->icAttr->peAreaVal.x;
	    pArea.y = winPtr->icAttr->peAreaVal.y;
	    pArea.width = (winPtr->icAttr->peAreaVal.width != 0) ?
	    	winPtr->icAttr->peAreaVal.width : winPtr->reqWidth;
	    pArea.height = (winPtr->icAttr->peAreaVal.height != 0) ?
	    	winPtr->icAttr->peAreaVal.height : winPtr->reqHeight;
	    if (doFont) {
		pList = XVaCreateNestedList(0, XNArea, &pArea, XNFontSet, fontset, NULL);
	    } else {
		pList = XVaCreateNestedList(0, XNArea, &pArea, NULL);
	    }
	    doPArea = 1;
	    break;
	}

	case XIMPreeditCallbacks: {
	    /*
	     * Sorry, not supported yet. changing the style to PreeditNothing.
	     */
	    if (doFont) {
		pList = XVaCreateNestedList(0, XNFontSet, fontset, NULL);
	    }
	    style &= ~(XIMPreeditCallbacks);
	    style |= XIMPreeditNothing;
	    break;
	}

	case XIMPreeditPosition: {
	    spot.x = winPtr->icAttr->spotVal.x;
	    spot.y = winPtr->icAttr->spotVal.y;
	    if (doFont) {
		pList = XVaCreateNestedList(0, XNSpotLocation, &spot, XNFontSet, fontset, NULL);
	    } else {
		pList = XVaCreateNestedList(0, XNSpotLocation, &spot, NULL);
	    }
	    doSpot = 1;
	    break;
	}

	case XIMPreeditNothing:
	case XIMPreeditNone: {
	    /*
	     * Maybe this is useless..
	     */
	    if (doFont) {
		pList = XVaCreateNestedList(0, XNFontSet, fontset, NULL);
	    }
	    break;
	}
    }

    switch (status) {
	
	case XIMStatusArea: {
	    sArea.x = winPtr->icAttr->stAreaVal.x;
	    sArea.y = winPtr->icAttr->stAreaVal.y;
	    sArea.width = (winPtr->icAttr->stAreaVal.width != 0) ?
	    	winPtr->icAttr->stAreaVal.width : winPtr->reqWidth;
	    sArea.height = (winPtr->icAttr->stAreaVal.height != 0) ?
	    	winPtr->icAttr->stAreaVal.height : winPtr->reqHeight;
	    if (doFont) {
		sList = XVaCreateNestedList(0, XNArea, &sArea, XNFontSet, fontset, NULL);
	    } else {
		sList = XVaCreateNestedList(0, XNArea, &sArea, NULL);
	    }
	    doSArea = 1;
	    break;
	}

	case XIMStatusCallbacks: {
	    /*
	     * Sorry, not supported yet. changing the style to PreeditNothing.
	     */
	    if (doFont) {
		sList = XVaCreateNestedList(0, XNFontSet, fontset, NULL);
	    }
	    style &= ~(XIMStatusCallbacks);
	    style |= XIMStatusNothing;
	    break;
	}
	
	case XIMStatusNothing:
	case XIMStatusNone: {
	    if (doFont) {
		sList = XVaCreateNestedList(0, XNFontSet, fontset, NULL);
	    }
	    break;
	}
    }

    /*
     * OK, roll'em.
     */
    if (pList != NULL && sList != NULL) {
	ret = XCreateIC(winPtr->dispPtr->inputMethod,
			XNInputStyle, style,
			XNClientWindow, winPtr->window,
			XNFocusWindow, winPtr->window,
			XNPreeditAttributes, pList,
			XNStatusAttributes, sList,
			NULL);
	XFree(pList);
	XFree(sList);
    } else if (pList != NULL && sList == NULL) {
	ret = XCreateIC(winPtr->dispPtr->inputMethod,
			XNInputStyle, style,
			XNClientWindow, winPtr->window,
			XNFocusWindow, winPtr->window,
			XNPreeditAttributes, pList,
			NULL);
	XFree(pList);
    } else if (pList == NULL && sList != NULL) {
	ret = XCreateIC(winPtr->dispPtr->inputMethod,
			XNInputStyle, style,
			XNClientWindow, winPtr->window,
			XNFocusWindow, winPtr->window,
			XNStatusAttributes, sList,
			NULL);
	XFree(sList);
    } else {
	ret = XCreateIC(winPtr->dispPtr->inputMethod,
			XNInputStyle, style,
			XNClientWindow, winPtr->window,
			XNFocusWindow, winPtr->window,
			NULL);
    }

    if (ret != NULL) {
#if 0
	{
	    unsigned long eMask;
	    XVaNestedList eList = XVaCreateNestedList(0, XNFilterEvents, &eMask, NULL);
	    if (XGetICValues(ret, XNFilterEvents, &eMask, NULL) != NULL) {
		fprintf(stderr, "debugIC: can't get event mask\n");
	    }
	    fprintf(stderr, "debugIC: event mask 0x%08x\n", eMask);
	}
#endif	    
	if (doSArea) {
	    /*
	     * Get the IM's preferred status area.
	     */
	    XRectangle area;
	    if (TkpIMGetPreferredArea(ret, winPtr) == winPtr) {
		area.width = winPtr->icAttr->stPrefAreaVal.width;
		area.height = winPtr->icAttr->stPrefAreaVal.height;
		if (doSpot) {
		    area.x = spot.x;
		    area.y = spot.y;
		} else if (doPArea) {
		    area.x = pArea.x;
		    area.y = pArea.y;
		} else {
		    area.x = 0;
		    area.y = 0;
		}
		TkpIMSendArea(ret, XNStatusAttributes, &area);
	    }
	}

	/*
	 * Register a generic event handler for Input focus.
	 */
	TkpCreateIMGenericHandler((Tk_Window)winPtr);

	winPtr->icAttr->isChanged &= ~(IM_STYLE_CHANGE);
	if (doFont) {
	    winPtr->icAttr->isChanged &= ~(IM_FONT_CHANGE);
	}
	if (doSpot) {
	    winPtr->icAttr->isChanged &= ~(IM_SPOT_CHANGE);
	}
	if (doPArea) {
	    winPtr->icAttr->isChanged &= ~(IM_PREEDITAREA_CHANGE);
	}
	if (doSArea) {
	    winPtr->icAttr->isChanged &= ~(IM_STATUSAREA_CHANGE);
	}
	winPtr->flags |= TK_CHECKED_IC;
	winPtr->inputContext = ret;
	if (winPtr->icAttr->styleVal != style) {
	    winPtr->icAttr->style = Tk_GetUid(GetStyleStringByStyleValue(style));
	    winPtr->icAttr->styleVal = style;
	}
    } else {
	winPtr->flags &= ~(TK_CHECKED_IC);
	winPtr->inputContext = NULL;
    }
    return ret;
}


static void
TkpDestroyIC(winPtr)
     TkWindow *winPtr;
{
    XDestroyIC(winPtr->inputContext);
    winPtr->flags &= ~(TK_CHECKED_IC);
    winPtr->inputContext = NULL;
}


static int
TkpConfigIC(interp, winPtr)
     Tcl_Interp *interp;
     TkWindow *winPtr;
{
    TkpICAttribute *icAttr = winPtr->icAttr;
    Tcl_Obj *resultPtr = Tcl_GetObjResult(interp);
    XIC ic;

#define isAttrChanged(X) ( icAttr->isChanged & (X) )

    if (!(winPtr->flags & TK_CHECKED_IC) ||
	(winPtr->inputContext == NULL) ||
	isAttrChanged(IM_STYLE_CHANGE)) {
	if (isAttrChanged(IM_STYLE_CHANGE)) {
	    if (winPtr->inputContext != NULL) {
		TkpDestroyIC(winPtr);
	    }
	    icAttr->isChanged &= ~(IM_STYLE_CHANGE);
	}
	if (icAttr->styleVal != (XIMStyle)0L) {
	    if (TkpCreateIC(winPtr) == NULL) {
		Tcl_AppendStringsToObj(resultPtr, "can't create IC.", NULL);
		return TCL_ERROR;
	    }
	} else {
	    panic("unknown IM style.");
	}
    }

    if (winPtr->inputContext == NULL) {
	Tcl_AppendStringsToObj(resultPtr, "IC is not created.", NULL);
	return TCL_ERROR;
    }

    if (icAttr->isChanged == IM_NOTHING_CHANGE) {
	return TCL_OK;
    }

    ic = winPtr->inputContext;
    if (isAttrChanged(IM_PREEDITAREA_CHANGE)) {
	TkpIMSendArea(ic, XNPreeditAttributes, &(winPtr->icAttr->peAreaVal));
    }
    if (isAttrChanged(IM_STATUSAREA_CHANGE)) {
	TkpIMSendArea(ic, XNStatusAttributes, &(winPtr->icAttr->stAreaVal));
    }
    if (isAttrChanged(IM_SPOT_CHANGE)) {
	TkpIMSendSpotLocation(ic, &(winPtr->icAttr->spotVal));
    }
    if (isAttrChanged(IM_FONT_CHANGE)) {
	TkpIMSendFontSet(ic, winPtr->icAttr->fontset);
    }
    if (isAttrChanged(IM_FOREGROUND_CHANGE) ||
	isAttrChanged(IM_BACKGROUND_CHANGE)) {
#ifdef XIM_BUGFIX
	fprintf(stderr, "debugIC: send color '%s'(0x%08x) '%s'(0x%08x) cmap 0x%08x\n",
		winPtr->icAttr->fg, winPtr->icAttr->fgVal,
		winPtr->icAttr->bg, winPtr->icAttr->bgVal,
		winPtr->atts.colormap);
	if (TkpIMSendColor(ic, winPtr->icAttr->fgVal, winPtr->icAttr->bgVal,
			   winPtr->atts.colormap) != 1) {
	    fprintf(stderr, "debugIC: color send fail.\n");
	}
#else
	TkpIMSendColor(ic, winPtr->icAttr->fgVal, winPtr->icAttr->bgVal,
		       winPtr->atts.colormap);
#endif /* XIM_DEBUG */
    }

    icAttr->isChanged = IM_NOTHING_CHANGE;
    return TCL_OK;
}



/*
 * Exported Stuff.
 */

TkpICAttribute *
TkpAllocICAttribute(tkwin)
     Tk_Window tkwin;
{
    TkpICAttribute *ret = (TkpICAttribute *)ckalloc(sizeof(TkpICAttribute));
    memset((VOID *)ret, 0, sizeof(TkpICAttribute));
    return ret;
}


void
TkpDeleteICAttribute(tkwin, attr)
     Tk_Window tkwin;
     TkpICAttribute *attr;
{
    if (attr->fontset != NULL) {
#ifdef USE_XFONTSET
	XFreeFontSet(Tk_Display(tkwin), attr->fontset);
#else
	TkpFreeFontSet(tkwin, attr->fontset);
#endif /* USE_XFONTSET */
    }
    ckfree(attr);
}


void
TkpCreateIMGenericHandler(tkwin)
     Tk_Window tkwin;
{
    Tk_CreateEventHandler(tkwin, FocusChangeMask|StructureNotifyMask|KeyPressMask
#if 0
			  EnterWindowMask|LeaveWindowMask,
#else
			  ,
#endif
			  (Tk_EventProc *)TkpIMGenericHandler,
			  (ClientData)tkwin);
}


void
TkpDeleteIMGenericHandler(tkwin)
     Tk_Window tkwin;
{
    Tk_DeleteEventHandler(tkwin, FocusChangeMask|StructureNotifyMask|KeyPressMask
#if 0
			  EnterWindowMask|LeaveWindowMask,
#else
			  ,
#endif
			  (Tk_EventProc *)TkpIMGenericHandler,
			  (ClientData)tkwin);
}


int
Tk_ImconfigureObjCmd(clientData, interp, objc, objv)
     ClientData clientData;	/* Main window associated with
				 * interpreter. */
     Tcl_Interp *interp;	/* Current interpreter. */
     int objc;			/* Number of arguments. */
     Tcl_Obj *CONST objv[];	/* Argument objects. */
{
    TkWindow *winPtr;
    Tk_Window tkwin = (Tk_Window)clientData;
    char *string;
    Tcl_Obj *resultPtr = Tcl_GetObjResult(interp);
    int doForce = 0;
    int idx = 2;
    int nObjc = objc;

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "path ?option? ?arg? ...");
	return TCL_ERROR;
    }
    string = Tcl_GetStringFromObj(objv[1], NULL);
    tkwin = Tk_NameToWindow(interp, string, tkwin);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }
    winPtr = (TkWindow *)tkwin;

    if (winPtr->dispPtr->inputMethod == NULL) {
	Tcl_AppendStringsToObj(resultPtr, "No IM server is available.", NULL);
	return TCL_ERROR;
    } else if (winPtr->dispPtr->imSupportedStyle == NULL) {
	Tcl_AppendStringsToObj(resultPtr, "IM server doesn't support any input style.", NULL);
	return TCL_ERROR;
    }

    if (objc == 2) {
	DumpICAttribute(interp, winPtr);
	return TCL_OK;
    }

    if (objc >= 3) {
	string = Tcl_GetStringFromObj(objv[2], NULL);
	if (strcmp(string, "-force") == 0) {
	    nObjc--;
	    doForce = 1;
	    idx++;
	}
    }

    if (nObjc == 3) {
	return ConfigGetICAttribute(interp, winPtr, objc - idx, objv + idx);
    }

    if (ConfigSetICAttribute(interp, winPtr, objc - idx, objv + idx, doForce) != TCL_OK) {
	return TCL_ERROR;
    }

    if (winPtr->icAttr->isChanged != IM_NOTHING_CHANGE ||
	winPtr->inputContext == NULL) {
	return TkpConfigIC(interp, winPtr);
    }
    return TCL_OK;
}


#ifdef XIM_DEBUG
Display *dpy = NULL;
char *Yes = "YES";
char *No = "NO";
char *Unknown = "unknown";

prologue (eventp, event_name)
    XEvent *eventp;
    char *event_name;
{
    XAnyEvent *e = (XAnyEvent *) eventp;

    printf ("\n%s event, serial %ld, synthetic %s, window 0x%lx,\n",
	    event_name, e->serial, e->send_event ? Yes : No, e->window);
    return;
}


do_KeyPress (eventp)
    XEvent *eventp;
{
    XKeyEvent *e = (XKeyEvent *) eventp;
    KeySym ks;
    char *ksname;
    int nbytes;
    char str[256+1];

    nbytes = XLookupString (e, str, 256, &ks, NULL);
    if (ks == NoSymbol)
	ksname = "NoSymbol";
    else if (!(ksname = XKeysymToString (ks)))
	ksname = "(no name)";
    printf ("    root 0x%lx, subw 0x%lx, time %lu, (%d,%d), root:(%d,%d),\n",
	    e->root, e->subwindow, e->time, e->x, e->y, e->x_root, e->y_root);
    printf ("    state 0x%x, keycode %u (keysym 0x%x, %s), same_screen %s,\n",
	    e->state, e->keycode, ks, ksname, e->same_screen ? Yes : No);
    if (nbytes < 0) nbytes = 0;
    if (nbytes > 256) nbytes = 256;
    str[nbytes] = '\0';
    printf ("    XLookupString gives %d characters:  \"%s\"\n", nbytes, str);

    return;
}

do_KeyRelease (eventp)
    XEvent *eventp;
{
    do_KeyPress (eventp);		/* since it has the same info */
    return;
}

do_ButtonPress (eventp)
    XEvent *eventp;
{
    XButtonEvent *e = (XButtonEvent *) eventp;

    printf ("    root 0x%lx, subw 0x%lx, time %lu, (%d,%d), root:(%d,%d),\n",
	    e->root, e->subwindow, e->time, e->x, e->y, e->x_root, e->y_root);
    printf ("    state 0x%x, button %u, same_screen %s\n",
	    e->state, e->button, e->same_screen ? Yes : No);

    return;
}

do_ButtonRelease (eventp)
    XEvent *eventp;
{
    do_ButtonPress (eventp);		/* since it has the same info */
    return;
}

do_MotionNotify (eventp)
    XEvent *eventp;
{
    XMotionEvent *e = (XMotionEvent *) eventp;

    printf ("    root 0x%lx, subw 0x%lx, time %lu, (%d,%d), root:(%d,%d),\n",
	    e->root, e->subwindow, e->time, e->x, e->y, e->x_root, e->y_root);
    printf ("    state 0x%x, is_hint %u, same_screen %s\n",
	    e->state, e->is_hint, e->same_screen ? Yes : No);

    return;
}

do_EnterNotify (eventp)
    XEvent *eventp;
{
    XCrossingEvent *e = (XCrossingEvent *) eventp;
    char *mode, *detail;
    char dmode[10], ddetail[10];

    switch (e->mode) {
      case NotifyNormal:  mode = "NotifyNormal"; break;
      case NotifyGrab:  mode = "NotifyGrab"; break;
      case NotifyUngrab:  mode = "NotifyUngrab"; break;
      case NotifyWhileGrabbed:  mode = "NotifyWhileGrabbed"; break;
      default:  mode = dmode, sprintf (dmode, "%u", e->mode); break;
    }

    switch (e->detail) {
      case NotifyAncestor:  detail = "NotifyAncestor"; break;
      case NotifyVirtual:  detail = "NotifyVirtual"; break;
      case NotifyInferior:  detail = "NotifyInferior"; break;
      case NotifyNonlinear:  detail = "NotifyNonlinear"; break;
      case NotifyNonlinearVirtual:  detail = "NotifyNonlinearVirtual"; break;
      case NotifyPointer:  detail = "NotifyPointer"; break;
      case NotifyPointerRoot:  detail = "NotifyPointerRoot"; break;
      case NotifyDetailNone:  detail = "NotifyDetailNone"; break;
      default:  detail = ddetail; sprintf (ddetail, "%u", e->detail); break;
    }

    printf ("    root 0x%lx, subw 0x%lx, time %lu, (%d,%d), root:(%d,%d),\n",
	    e->root, e->subwindow, e->time, e->x, e->y, e->x_root, e->y_root);
    printf ("    mode %s, detail %s, same_screen %s,\n",
	    mode, detail, e->same_screen ? Yes : No);
    printf ("    focus %s, state %u\n", e->focus ? Yes : No, e->state);

    return;
}

do_LeaveNotify (eventp)
    XEvent *eventp;
{
    do_EnterNotify (eventp);		/* since it has same information */
    return;
}

do_FocusIn (eventp)
    XEvent *eventp;
{
    XFocusChangeEvent *e = (XFocusChangeEvent *) eventp;
    char *mode, *detail;
    char dmode[10], ddetail[10];

    switch (e->mode) {
      case NotifyNormal:  mode = "NotifyNormal"; break;
      case NotifyGrab:  mode = "NotifyGrab"; break;
      case NotifyUngrab:  mode = "NotifyUngrab"; break;
      case NotifyWhileGrabbed:  mode = "NotifyWhileGrabbed"; break;
      default:  mode = dmode, sprintf (dmode, "%u", e->mode); break;
    }

    switch (e->detail) {
      case NotifyAncestor:  detail = "NotifyAncestor"; break;
      case NotifyVirtual:  detail = "NotifyVirtual"; break;
      case NotifyInferior:  detail = "NotifyInferior"; break;
      case NotifyNonlinear:  detail = "NotifyNonlinear"; break;
      case NotifyNonlinearVirtual:  detail = "NotifyNonlinearVirtual"; break;
      case NotifyPointer:  detail = "NotifyPointer"; break;
      case NotifyPointerRoot:  detail = "NotifyPointerRoot"; break;
      case NotifyDetailNone:  detail = "NotifyDetailNone"; break;
      default:  detail = ddetail; sprintf (ddetail, "%u", e->detail); break;
    }

    printf ("    mode %s, detail %s\n", mode, detail);
    return;
}

do_FocusOut (eventp)
    XEvent *eventp;
{
    do_FocusIn (eventp);		/* since it has same information */
    return;
}

do_KeymapNotify (eventp)
    XEvent *eventp;
{
    XKeymapEvent *e = (XKeymapEvent *) eventp;
    int i;

    printf ("    keys:  ");
    for (i = 0; i < 32; i++) {
	if (i == 16) printf ("\n           ");
	printf ("%-3u ", (unsigned int) e->key_vector[i]);
    }
    printf ("\n");
    return;
}

do_Expose (eventp)
    XEvent *eventp;
{
    XExposeEvent *e = (XExposeEvent *) eventp;

    printf ("    (%d,%d), width %d, height %d, count %d\n",
	    e->x, e->y, e->width, e->height, e->count);
    return;
}

do_GraphicsExpose (eventp)
    XEvent *eventp;
{
    XGraphicsExposeEvent *e = (XGraphicsExposeEvent *) eventp;
    char *m;
    char mdummy[10];

    switch (e->major_code) {
      case X_CopyArea:  m = "CopyArea";  break;
      case X_CopyPlane:  m = "CopyPlane";  break;
      default:  m = mdummy; sprintf (mdummy, "%d", e->major_code); break;
    }

    printf ("    (%d,%d), width %d, height %d, count %d,\n",
	    e->x, e->y, e->width, e->height, e->count);
    printf ("    major %s, minor %d\n", m, e->minor_code);
    return;
}

do_NoExpose (eventp)
    XEvent *eventp;
{
    XNoExposeEvent *e = (XNoExposeEvent *) eventp;
    char *m;
    char mdummy[10];

    switch (e->major_code) {
      case X_CopyArea:  m = "CopyArea";  break;
      case X_CopyPlane:  m = "CopyPlane";  break;
      default:  m = mdummy; sprintf (mdummy, "%d", e->major_code); break;
    }

    printf ("    major %s, minor %d\n", m, e->minor_code);
    return;
}

do_VisibilityNotify (eventp)
    XEvent *eventp;
{
    XVisibilityEvent *e = (XVisibilityEvent *) eventp;
    char *v;
    char vdummy[10];

    switch (e->state) {
      case VisibilityUnobscured:  v = "VisibilityUnobscured"; break;
      case VisibilityPartiallyObscured:  v = "VisibilityPartiallyObscured"; break;
      case VisibilityFullyObscured:  v = "VisibilityFullyObscured"; break;
      default:  v = vdummy; sprintf (vdummy, "%d", e->state); break;
    }

    printf ("    state %s\n", v);
    return;
}

do_CreateNotify (eventp)
    XEvent *eventp;
{
    XCreateWindowEvent *e = (XCreateWindowEvent *) eventp;

    printf ("    parent 0x%lx, window 0x%lx, (%d,%d), width %d, height %d\n",
	    e->parent, e->window, e->x, e->y, e->width, e->height);
    printf ("border_width %d, override %s\n",
	    e->border_width, e->override_redirect ? Yes : No);
    return;
}

do_DestroyNotify (eventp)
    XEvent *eventp;
{
    XDestroyWindowEvent *e = (XDestroyWindowEvent *) eventp;

    printf ("    event 0x%lx, window 0x%lx\n", e->event, e->window);
    return;
}

do_UnmapNotify (eventp)
    XEvent *eventp;
{
    XUnmapEvent *e = (XUnmapEvent *) eventp;

    printf ("    event 0x%lx, window 0x%lx, from_configure %s\n",
	    e->event, e->window, e->from_configure ? Yes : No);
    return;
}

do_MapNotify (eventp)
    XEvent *eventp;
{
    XMapEvent *e = (XMapEvent *) eventp;

    printf ("    event 0x%lx, window 0x%lx, override %s\n",
	    e->event, e->window, e->override_redirect ? Yes : No);
    return;
}

do_MapRequest (eventp)
    XEvent *eventp;
{
    XMapRequestEvent *e = (XMapRequestEvent *) eventp;

    printf ("    parent 0x%lx, window 0x%lx\n", e->parent, e->window);
    return;
}

do_ReparentNotify (eventp)
    XEvent *eventp;
{
    XReparentEvent *e = (XReparentEvent *) eventp;

    printf ("    event 0x%lx, window 0x%lx, parent 0x%lx,\n",
	    e->event, e->window, e->parent);
    printf ("    (%d,%d), override %s\n", e->x, e->y, 
	    e->override_redirect ? Yes : No);
    return;
}

do_ConfigureNotify (eventp)
    XEvent *eventp;
{
    XConfigureEvent *e = (XConfigureEvent *) eventp;

    printf ("    event 0x%lx, window 0x%lx, (%d,%d), width %d, height %d,\n",
	    e->event, e->window, e->x, e->y, e->width, e->height);
    printf ("    border_width %d, above 0x%lx, override %s\n",
	    e->border_width, e->above, e->override_redirect ? Yes : No);
    return;
}

do_ConfigureRequest (eventp)
    XEvent *eventp;
{
    XConfigureRequestEvent *e = (XConfigureRequestEvent *) eventp;
    char *detail;
    char ddummy[10];

    switch (e->detail) {
      case Above:  detail = "Above";  break;
      case Below:  detail = "Below";  break;
      case TopIf:  detail = "TopIf";  break;
      case BottomIf:  detail = "BottomIf"; break;
      case Opposite:  detail = "Opposite"; break;
      default:  detail = ddummy; sprintf (ddummy, "%d", e->detail); break;
    }

    printf ("    parent 0x%lx, window 0x%lx, (%d,%d), width %d, height %d,\n",
	    e->parent, e->window, e->x, e->y, e->width, e->height);
    printf ("    border_width %d, above 0x%lx, detail %s, value 0x%lx\n",
	    e->border_width, e->above, detail, e->value_mask);
    return;
}

do_GravityNotify (eventp)
    XEvent *eventp;
{
    XGravityEvent *e = (XGravityEvent *) eventp;

    printf ("    event 0x%lx, window 0x%lx, (%d,%d)\n",
	    e->event, e->window, e->x, e->y);
    return;
}

do_ResizeRequest (eventp)
    XEvent *eventp;
{
    XResizeRequestEvent *e = (XResizeRequestEvent *) eventp;

    printf ("    width %d, height %d\n", e->width, e->height);
    return;
}

do_CirculateNotify (eventp)
    XEvent *eventp;
{
    XCirculateEvent *e = (XCirculateEvent *) eventp;
    char *p;
    char pdummy[10];

    switch (e->place) {
      case PlaceOnTop:  p = "PlaceOnTop"; break;
      case PlaceOnBottom:  p = "PlaceOnBottom"; break;
      default:  p = pdummy; sprintf (pdummy, "%d", e->place); break;
    }

    printf ("    event 0x%lx, window 0x%lx, place %s\n",
	    e->event, e->window, p);
    return;
}

do_CirculateRequest (eventp)
    XEvent *eventp;
{
    XCirculateRequestEvent *e = (XCirculateRequestEvent *) eventp;
    char *p;
    char pdummy[10];

    switch (e->place) {
      case PlaceOnTop:  p = "PlaceOnTop"; break;
      case PlaceOnBottom:  p = "PlaceOnBottom"; break;
      default:  p = pdummy; sprintf (pdummy, "%d", e->place); break;
    }

    printf ("    parent 0x%lx, window 0x%lx, place %s\n",
	    e->parent, e->window, p);
    return;
}

do_PropertyNotify (eventp)
    XEvent *eventp;
{
    XPropertyEvent *e = (XPropertyEvent *) eventp;
    char *aname = XGetAtomName (dpy, e->atom);
    char *s;
    char sdummy[10];

    switch (e->state) {
      case PropertyNewValue:  s = "PropertyNewValue"; break;
      case PropertyDelete:  s = "PropertyDelete"; break;
      default:  s = sdummy; sprintf (sdummy, "%d", e->state); break;
    }

    printf ("    atom 0x%lx (%s), time %lu, state %s\n",
	   e->atom, aname ? aname : Unknown, e->time,  s);

    if (aname) XFree (aname);
    return;
}

do_SelectionClear (eventp)
    XEvent *eventp;
{
    XSelectionClearEvent *e = (XSelectionClearEvent *) eventp;
    char *sname = XGetAtomName (dpy, e->selection);

    printf ("    selection 0x%lx (%s), time %lu\n",
	    e->selection, sname ? sname : Unknown, e->time);

    if (sname) XFree (sname);
    return;
}

do_SelectionRequest (eventp)
    XEvent *eventp;
{
    XSelectionRequestEvent *e = (XSelectionRequestEvent *) eventp;
    char *sname = XGetAtomName (dpy, e->selection);
    char *tname = XGetAtomName (dpy, e->target);
    char *pname = XGetAtomName (dpy, e->property);

    printf ("    owner 0x%lx, requestor 0x%lx, selection 0x%lx (%s),\n",
	    e->owner, e->requestor, e->selection, sname ? sname : Unknown);
    printf ("    target 0x%lx (%s), property 0x%lx (%s), time %lu\n",
	    e->target, tname ? tname : Unknown, e->property,
	    pname ? pname : Unknown, e->time);

    if (sname) XFree (sname);
    if (tname) XFree (tname);
    if (pname) XFree (pname);

    return;
}

do_SelectionNotify (eventp)
    XEvent *eventp;
{
    XSelectionEvent *e = (XSelectionEvent *) eventp;
    char *sname = XGetAtomName (dpy, e->selection);
    char *tname = XGetAtomName (dpy, e->target);
    char *pname = XGetAtomName (dpy, e->property);

    printf ("    selection 0x%lx (%s), target 0x%lx (%s),\n",
	    e->selection, sname ? sname : Unknown, e->target,
	    tname ? tname : Unknown);
    printf ("    property 0x%lx (%s), time %lu\n",
	    e->property, pname ? pname : Unknown, e->time);

    if (sname) XFree (sname);
    if (tname) XFree (tname);
    if (pname) XFree (pname);

    return;
}

do_ColormapNotify (eventp)
    XEvent *eventp;
{
    XColormapEvent *e = (XColormapEvent *) eventp;
    char *s;
    char sdummy[10];

    switch (e->state) {
      case ColormapInstalled:  s = "ColormapInstalled"; break;
      case ColormapUninstalled:  s = "ColormapUninstalled"; break;
      default:  s = sdummy; sprintf (sdummy, "%d", e->state); break;
    }

    printf ("    colormap 0x%lx, new %s, state %s\n",
	    e->colormap, e->new ? Yes : No, s);
    return;
}

do_ClientMessage (eventp)
    XEvent *eventp;
{
    XClientMessageEvent *e = (XClientMessageEvent *) eventp;
    char *mname = XGetAtomName (dpy, e->message_type);

    printf ("    message_type 0x%lx (%s), format %d\n",
	    e->message_type, mname ? mname : Unknown, e->format);

    if (mname) XFree (mname);
    return;
}

do_MappingNotify (eventp)
    XEvent *eventp;
{
    XMappingEvent *e = (XMappingEvent *) eventp;
    char *r;
    char rdummy[10];

    switch (e->request) {
      case MappingModifier:  r = "MappingModifier"; break;
      case MappingKeyboard:  r = "MappingKeyboard"; break;
      case MappingPointer:  r = "MappingPointer"; break;
      default:  r = rdummy; sprintf (rdummy, "%d", e->request); break;
    }

    printf ("    request %s, first_keycode %d, count %d\n",
	    r, e->first_keycode, e->count);
    return;
}


void
DumpXEvent(display, event)
     Display *display;
     XEvent *event;
{
  dpy = display;
  switch (event->type) {
    case KeyPress: {
      prologue (event, "KeyPress");
      do_KeyPress (event);
      break;
    }
    case KeyRelease: {
      prologue (event, "KeyRelease");
      do_KeyRelease (event);
      break;
    }
    case ButtonPress: {
      prologue (event, "ButtonPress");
      do_ButtonPress (event);
      break;
    }
    case ButtonRelease: {
      prologue (event, "ButtonRelease");
      do_ButtonRelease (event);
      break;
    }
    case MotionNotify: {
      prologue (event, "MotionNotify");
      do_MotionNotify (event);
      break;
    }
    case EnterNotify: {
      prologue (event, "EnterNotify");
      do_EnterNotify (event);
      break;
    }
    case LeaveNotify: {
      prologue (event, "LeaveNotify");
      do_LeaveNotify (event);
      break;
    }
    case FocusIn: {
      prologue (event, "FocusIn");
      do_FocusIn (event);
      break;
    }
    case FocusOut: {
      prologue (event, "FocusOut");
      do_FocusOut (event);
      break;
    }
    case KeymapNotify: {
      prologue (event, "KeymapNotify");
      do_KeymapNotify (event);
      break;
    }
    case Expose: {
      prologue (event, "Expose");
      do_Expose (event);
      break;
    }
    case GraphicsExpose: {
      prologue (event, "GraphicsExpose");
      do_GraphicsExpose (event);
      break;
    }
    case NoExpose: {
      prologue (event, "NoExpose");
      do_NoExpose (event);
      break;
    }
    case VisibilityNotify: {
      prologue (event, "VisibilityNotify");
      do_VisibilityNotify (event);
      break;
    }
    case CreateNotify: {
      prologue (event, "CreateNotify");
      do_CreateNotify (event);
      break;
    }
    case DestroyNotify: {
      prologue (event, "DestroyNotify");
      do_DestroyNotify (event);
      break;
    }
    case UnmapNotify: {
      prologue (event, "UnmapNotify");
      do_UnmapNotify (event);
      break;
    }
    case MapNotify: {
      prologue (event, "MapNotify");
      do_MapNotify (event);
      break;
    }
    case MapRequest: {
      prologue (event, "MapRequest");
      do_MapRequest (event);
      break;
    }
    case ReparentNotify: {
      prologue (event, "ReparentNotify");
      do_ReparentNotify (event);
      break;
    }
    case ConfigureNotify: {
      prologue (event, "ConfigureNotify");
      do_ConfigureNotify (event);
      break;
    }
    case ConfigureRequest: {
      prologue (event, "ConfigureRequest");
      do_ConfigureRequest (event);
      break;
    }
    case GravityNotify: {
      prologue (event, "GravityNotify");
      do_GravityNotify (event);
      break;
    }
    case ResizeRequest: {
      prologue (event, "ResizeRequest");
      do_ResizeRequest (event);
      break;
    }
    case CirculateNotify: {
      prologue (event, "CirculateNotify");
      do_CirculateNotify (event);
      break;
    }
    case CirculateRequest: {
      prologue (event, "CirculateRequest");
      do_CirculateRequest (event);
      break;
    }
    case PropertyNotify: {
      prologue (event, "PropertyNotify");
      do_PropertyNotify (event);
      break;
    }
    case SelectionClear: {
      prologue (event, "SelectionClear");
      do_SelectionClear (event);
      break;
    }
    case SelectionRequest: {
      prologue (event, "SelectionRequest");
      do_SelectionRequest (event);
      break;
    }
    case SelectionNotify: {
      prologue (event, "SelectionNotify");
      do_SelectionNotify (event);
      break;
    }
    case ColormapNotify: {
      prologue (event, "ColormapNotify");
      do_ColormapNotify (event);
      break;
    }
    case ClientMessage: {
      prologue (event, "ClientMessage");
      do_ClientMessage (event);
      break;
    }
    case MappingNotify: {
      prologue (event, "MappingNotify");
      do_MappingNotify (event);
      break;
    }
    default: {
      printf ("Unknown event type %d\n", event->type);
      break;
    }
  }
}
#endif /* XIM_DEBUG */

#endif /* XIM_IMPROVE */
#endif /* TK_USE_INPUT_METHODS */
