/*
                       MIDI File Format Library

	Functions:

	void swap32(int32_t *i)
	void swapu32(u_int32_t *i)

	int MidiFGetVarlenNumber(FILE *fp)
	int MidiGetVarlenNumber(char *buf, int max)

	int MidiIsFileMidi(char *filename)
	int MidiRead(char *filename, MidiDataStruct *md)
	void MidiDestroy(MidiDataStruct *md)

	---

 */

#include <stdio.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>

#include "midi.h"



void swap32(int32_t *i);
void swapu32(u_int32_t *i);

int MidiFGetVarlenNumber(FILE *fp);
int MidiGetVarlenNumber(char *buf, int max);



void swap32(int32_t *i)
{
	int32_t tmp;
	int8_t *src, *tar;

	if(i == NULL)
	    return;

	tmp = *i;
	src = (int8_t *)&tmp;
	tar = (int8_t *)i;

	tar[3] = src[0];
	tar[2] = src[1];
	tar[1] = src[2];
	tar[0] = src[3];

	return;
}

void swapu32(u_int32_t *i)
{
        u_int32_t tmp;
        u_int8_t *src, *tar;

        if(i == NULL)
            return;

        tmp = *i;
        src = (u_int8_t *)&tmp;
        tar = (u_int8_t *)i;

        tar[3] = src[0];
        tar[2] = src[1];
        tar[1] = src[2];
        tar[0] = src[3];

        return;
}


/*
 *	Reads variable-length number from FILE pointer fp,
 *	fp will be incremented to the point just after the bytes
 *	making up the variable-length number.
 */
int MidiFGetVarlenNumber(FILE *fp)
{
	int c, p = 0, val = 0;


	if(fp == NULL)
	    return(0);

	do
	{
	    c = fgetc(fp);
	    if(c == EOF)
		break;

	    val += (((u_int8_t)c & 0x7f) << (p * 8));

	    p++;
	}
	while(((u_int8_t)c & 0x80) && (p <= 4));


	return(val);
}


/*
 *      Reads variable-length number from buffer pointer buf.
 *	Will not read more than max bytes.
 */
int MidiGetVarlenNumber(char *buf, int max)
{
        int p = 0, val = 0;


        if(buf == NULL)
            return(0);
 
        do
        {
            val += (((u_int8_t)*buf & 0x7f) << (p * 8));

            buf++;
            p++;
        }
        while(((u_int8_t)*buf & 0x80) && (p <= max));


        return(val);
}


/*
 *	Returns MidiSuccess if filename is a MIDI file or
 *	MidiBadValue if filename is not a MIDI file.
 */
int MidiIsFileMidi(char *filename)
{
	int i, c;
	struct stat stat_buf;
	FILE *fp;
	char chunk_type_name[MidiChunkTypeNameLen + 1];


	if(filename == NULL)
	    return(MidiBadValue);


        /* Check if file exists. */
        if(stat(filename, &stat_buf))
            return(MidiBadValue);

        /* Open file. */
        fp = fopen(filename, "rb");
        if(fp == NULL)
            return(MidiBadValue);


	/* Read first few bytes. */
	*chunk_type_name = '\0';
	for(i = 0; i < MidiChunkTypeNameLen; i++)
	{
	    c = fgetc(fp);
	    if(c == EOF)
		break;

	    chunk_type_name[i] = (char)c;
	}
	chunk_type_name[MidiChunkTypeNameLen] = '\0';

	/* Close file. */
        fclose(fp);
        

	if(!strcmp(chunk_type_name, MidiChunkTypeNameHeader) ||
           !strcmp(chunk_type_name, MidiChunkTypeNameTrack)
	)
	    return(MidiSuccess);
	else
	    return(MidiBadValue);
}


/*
 *	Read an midi file and store data into md.
 */
int MidiRead(char *filename, MidiDataStruct *md)
{
	int need_break;
	int i, c;
	FILE *fp;
	struct stat stat_buf;
	off_t file_pos, file_size;

	char chunk_type_name[MidiChunkTypeNameLen + 1];
	u_int32_t chunk_len;
	MidiChunkStruct *chunk_ptr;

	u_int8_t *buf_ptr;
	int buf_len;


	/* Error checks. */
	if((filename == NULL) ||
           (md == NULL)
	)
	    return(MidiNoBuffers);

	/* Reset md. */
	memset(md, 0x00, sizeof(MidiDataStruct));
	md->chunk = NULL;	/* Just in case. */
	md->total_chunks = 0;


	/* Check if file exists. */
	if(stat(filename, &stat_buf))
	    return(MidiBadValue);

	/* Open file. */
	fp = fopen(filename, "rb");
	if(fp == NULL)
	    return(MidiBadValue);

	/* Record file name. */
	i = strlen(filename);
	md->filename = (char *)malloc((i + 1) * sizeof(char));
	if(md->filename == NULL)
	{
	    fclose(fp);
	    return(MidiNoBuffers);
	}
	strcpy(md->filename, filename);


	/* Read each chunk in the file. */
	file_pos = 0;
	file_size = stat_buf.st_size;
	while(file_pos < file_size)
	{
	    need_break = 0;

	    /* Chunk type (4 bytes). */
	    for(i = 0; i < MidiChunkTypeNameLen; i++)
	    {
		c = fgetc(fp);
                if(c == EOF)
		{
		    need_break = 1;
                    break;
		}

		chunk_type_name[i] = (char)c;
	    }
	    chunk_type_name[MidiChunkTypeNameLen] = '\0'; /* Safe. */
	    if(need_break)
		break;

	    /* Read chunk length (4 bytes). */
	    fread(&chunk_len, sizeof(u_int32_t), 1, fp);
	    swapu32(&chunk_len);

	    /* Chunk length on file does not include chunk type and
             * length bytes so add + 8 to it.
	     */
	    chunk_len += (MidiChunkTypeNameLen + sizeof(u_int32_t));


	    /* Allocate raw data buffer and read raw chunk data into it. */
	    buf_len = (int)chunk_len - 8;
	    if(buf_len > 0)
	    {
	        buf_ptr = (u_int8_t *)malloc(buf_len * sizeof(u_int8_t));
		if(buf_ptr == NULL)
		    break;

		fread(buf_ptr, sizeof(u_int8_t), buf_len, fp);
	    }
	    else
	    {
		buf_ptr = NULL;
		buf_len = 0;
	    }


	    /* Allocate a new chunk structure. */
	    i = md->total_chunks;
	    md->total_chunks += 1;
	    md->chunk = (MidiChunkStruct **)realloc(
		md->chunk,
		md->total_chunks * sizeof(MidiChunkStruct *)
	    );
	    if(md->chunk == NULL)
	    {
		md->total_chunks = 0;
		continue;
	    }

	    md->chunk[i] = (MidiChunkStruct *)calloc(
		1,
		sizeof(MidiChunkStruct)
	    );
	    if(md->chunk[i] == NULL)
	    {
		md->total_chunks -= 1;
		continue;
	    }
	    chunk_ptr = md->chunk[i];



	    /* Put fetched data into chunk structure. */

	    /* Chunk type. */
	    if(!strcmp(chunk_type_name, MidiChunkTypeNameHeader))
		chunk_ptr->type = MidiChunkTypeHeader;
	    else if(!strcmp(chunk_type_name, MidiChunkTypeNameTrack))
		chunk_ptr->type = MidiChunkTypeTrack;
	    else
		chunk_ptr->type = MidiChunkTypeUnknown;

	    /* Chunk length. */
	    chunk_ptr->len = chunk_len;

	    /* Raw data buffer and length of it. */
	    chunk_ptr->buf_len = buf_len;
	    chunk_ptr->buf = buf_ptr;



	    /* If header chunk, parse raw data. */
	    if(chunk_ptr->type == MidiChunkTypeHeader)
	    {
		if(buf_len >= 6)
		{
		    md->format = (int16_t)(
                        ((u_int16_t)buf_ptr[0] << 8) +
			(u_int16_t)buf_ptr[1]
		    );

                    md->tracks = (int16_t)(
                        ((u_int16_t)buf_ptr[2] << 8) +
                        (u_int16_t)buf_ptr[3]  
                    );

                    md->divisions = (int16_t)(
                        ((u_int16_t)buf_ptr[4] << 8) +
                        (u_int16_t)buf_ptr[5]
                    );
		}
	    }







	    /* Increment file position. */
	    file_pos += chunk_len;
	    i = fseek(fp, file_pos, SEEK_SET);
	    if(i == -1)
		break;
	}




	/* Close file. */
	fclose(fp);
	fp = NULL;


	return(MidiSuccess);
}

/*
 *	Deallocates all allocated substructures in md.
 */
void MidiDestroy(MidiDataStruct *md)
{
	int i, n;
	MidiChunkStruct **chunk_ptr;
	MidiEventStruct **event_ptr;


	if(md == NULL)
	    return;

	free(md->filename);
	md->filename = NULL;


	/* Free each chunk. */
	for(i = 0, chunk_ptr = md->chunk;
            i < md->total_chunks;
            i++, chunk_ptr++
	)
	{
	    if(*chunk_ptr == NULL)
		continue;

	    /* Free each event in the chunk. */
	    for(n = 0, event_ptr = (*chunk_ptr)->event;
                n < (*chunk_ptr)->total_events;
                n++, event_ptr++
	    )
		free(*event_ptr);
	    free((*chunk_ptr)->event);

	    /* Free raw data buffer. */
	    free((*chunk_ptr)->buf);

	    /* Free structure itself. */
	    free(*chunk_ptr);
	}

	free(md->chunk);
	md->chunk = NULL;

	md->total_chunks = 0;


	return;
}
