/*
                   RAW File Format Library

	Functions:

	void RawReportError(char *filename, char *reason, int how_bad)
	FILE *RawOpenFile(char *filename, char *mode, off_t *size_rtn)

	int RawIsFileRaw(char *filename)

	int RawReadHeader(char *filename, raw_data_struct *rd)
	int RawReadPartialData(
	        raw_data_struct *rd,
	        long offset,
	        long max_chunk_size,
	        int read_opt
	)

	---

 */

#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "raw.h"


void RawReportError(char *filename, char *reason, int how_bad);
FILE *RawOpenFile(char *filename, char *mode, off_t *size_rtn);


void RawReportError(char *filename, char *reason, int how_bad)
{
        if(filename != NULL)
        {
            fprintf(stderr, "%s: ", filename);
        }
        if(reason != NULL)
        {
            fprintf(stderr, "%s (Error level %i)", reason, how_bad);
        }
        fprintf(stderr, "\n");


        return;
}


void RawDestroyData(raw_data_struct *rd)
{
	if(rd == NULL)
	    return;

	free(rd->filename);
	rd->filename = NULL;

	rd->data_offset = 0;
	rd->data_length = 0;

        rd->block_align = 0;
        rd->sample_size = 0;
        rd->channels = 0;
        rd->sample_rate = 0;

	free(rd->data);
	rd->data = NULL;

	rd->data_len = 0;

	return;
}


FILE *RawOpenFile(char *filename, char *mode, off_t *size_rtn)
{
        FILE *fp;
        struct stat stat_buf;


        if(filename == NULL)
        {
            fprintf(stderr, "Cannot open file with no name.\n");
            *size_rtn = 0;
            return(NULL);
        }
        if(mode == NULL)
        {
            fprintf(stderr, "%s: Open mode not givin.\n", filename);
            *size_rtn = 0;
            return(NULL);
        }

        if(stat(filename, &stat_buf))
        {
            fprintf(stderr, "%s: No such file.\n", filename);
            *size_rtn = 0;
            return(NULL);
        }
        *size_rtn = stat_buf.st_size;

        fp = fopen(filename, mode);
        if(fp == NULL)
        {
            fprintf(stderr, "%s: Cannot open.\n", filename);
            *size_rtn = 0;
            return(NULL);
        }


        return(fp);
}


int RawIsFileRaw(char *filename)
{
        FILE *fp;
        off_t filesize;


        /* Open file. */
        fp = RawOpenFile(filename, "rb", &filesize);
        if(fp == NULL)
            return(RawErrorNoAccess);
        if(filesize == 0)
            return(RawErrorNoAccess);

/* What do we check? */


        /* Close file. */
        fclose(fp);


        return(RawSuccess); 
}


int RawReadHeader(char *filename, raw_data_struct *rd)
{
	int len;
        FILE *fp;
        off_t filesize;


        /* Error checks. */
        if(rd == NULL)
            return(RawErrorNoBuffers);
        if(filename == NULL)
            return(RawErrorBadValue);


        /* Reset values. */
	memset(rd, 0x00, sizeof(raw_data_struct));


        /* Open file. */
        fp = RawOpenFile(filename, "rb", &filesize);
        if(fp == NULL)
            return(RawErrorNoAccess);
        if(filesize == 0)
            return(RawErrorNoAccess);


        /* Record file name. */
	len = strlen(filename);
        rd->filename = (char *)malloc((len + 1) * sizeof(char));
	if(rd->filename != NULL)
	{
            strncpy(rd->filename, filename, len);
	    rd->filename[len] = '\0';
	}

/* Check for offset? */
	/* Get offset. */
	rd->data_offset = 0;

	/* Get length of data. */
	rd->data_length = (long)filesize - (long)rd->data_offset;
	if(rd->data_length < 0)
	    rd->data_length = 0;


	/* Standard values. */
	rd->block_align = 0;
	rd->sample_size = 8;
        rd->channels = 1;
	rd->sample_rate = 8000;


        /* Close file. */
        fclose(fp);


        return(RawSuccess);
}


int RawReadPartialData(
        raw_data_struct *rd,
        long offset,            /* In bytes. */
        long max_chunk_size,    /* In bytes. */
        int read_opt
)
{
	int i;
	FILE *fp;
	off_t filesize;
	char *buf_ptr;


        /* Error checks. */
        if(rd == NULL)
            return(RawErrorNoBuffers);
        if(max_chunk_size <= 0)
            return(RawErrorBadValue);
        if(offset < 0)
            return(RawErrorBadValue);


	/* Sanitize values. */
	if(offset >= rd->data_length)
        {
            return(RawErrorEndOfData);
	}

	if((offset + max_chunk_size) >= rd->data_length)
	{
            max_chunk_size = rd->data_length - (offset + 1);
	}

        if(max_chunk_size <= 0)
        {
            return(RawErrorEndOfData);
        }


        /* ********************************************************* */

        /* Open file. */
        fp = RawOpenFile(rd->filename, "rb", &filesize);
        if(fp == NULL)
            return(RawErrorNoAccess);
        if(filesize == 0)
            return(RawErrorNoAccess);

        /* Reallocate memory as needed. */
	if(rd->data_len != max_chunk_size)
	{
	    rd->data_len = max_chunk_size;

	    rd->data = (char *)realloc(
		rd->data,
		rd->data_len * sizeof(char)
	    );
	}
	/* Allocation error? */
        if(rd->data == NULL)
        {
	    rd->data_len = 0;
            fclose(fp);
            return(RawErrorNoBuffers);
        }

        /* Position file pointer. */
        fseek(fp, rd->data_offset + offset, SEEK_SET);

        /* Read the data, raw file format stores DSP as signed 8. */
	buf_ptr = rd->data;
        switch(read_opt)
        {
          /* Shift byte value by + 128. */
          case RawReadUnsigned8:
            for(i = 0; i < max_chunk_size; i++)
                *buf_ptr++ = (char)(fgetc(fp) + 128);
            break;

	  /* Get data as is (signed 8). */
          default:
            for(i = 0; i < max_chunk_size; i++)
                *buf_ptr++ = (char)fgetc(fp);
            break;
        }


        /* Close the file. */
        fclose(fp);


        return(RawSuccess);
}
