;;
;; ELMO -- Elisp Library for Message Orchestration
;;
;; Copyright (C) 1998 Yuuichi Teranishi <teranisi@gohome.org>
;;
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.
;;

(require 'elmo-vars)
(require 'elmo-msgdb)
(require 'elmo-cache)
(require 'elmo-util)
(require 'elmo-dop)
(provide 'elmo2)

(eval-when-compile
  (require 'elmo-localdir)
  (require 'elmo-imap4)
  (require 'elmo-nntp)
  (require 'elmo-pop3)
  (require 'elmo-multi)
  (require 'elmo-filter)
  (require 'elmo-archive))

(if (or (featurep 'dbm)
	(featurep 'gnudbm)
	(featurep 'berkdb))
    (require 'elmo-database))

;; version info...
(defconst elmo-version "0.9.7")
(defconst elmo-appname "ELMO")

(defun elmo-version ()
  (format "%s %s" elmo-appname elmo-version))

(defun elmo-quit ()
  (interactive)
  (if (featurep 'elmo-imap4)
      (elmo-imap4-flush-connection))
  (if (featurep 'elmo-nntp)
      (elmo-nntp-flush-connection))
  (if (featurep 'elmo-pop3)
      (elmo-pop3-flush-connection))
  (if (get-buffer elmo-work-buf-name)
      (kill-buffer elmo-work-buf-name))
  )

(defun elmo-cleanup-variables ()
  (setq elmo-folder-info-hashtb nil
	elmo-nntp-groups-hashtb nil
	elmo-nntp-list-folders-cache nil
	))

;;  (cons of max . estimated message number) elmo-max-of-folder (folder)
(defun elmo-max-of-folder (folder)
  (if elmo-plugged
      (elmo-call-func folder "max-of-folder")
    (elmo-dop-max-of-folder folder)))

;;  list elmo-list-folder (folder)
(defun elmo-list-folder (folder)
  (if elmo-plugged
      (elmo-call-func folder "list-folder")
    (elmo-dop-list-folder folder)))

;;  list elmo-list-folders (folder)

(defun elmo-list-folders (folder &optional hierarchy)
  (elmo-call-func folder "list-folders" hierarchy))

;; bool elmo-folder-exists-p (folder)
(defun elmo-folder-exists-p (folder)
  (if elmo-plugged
      (elmo-call-func folder "folder-exists-p")
    (elmo-dop-folder-exists-p folder)))

;; bool elmo-create-folder (folder)
;; create folder
(defun elmo-create-folder (folder)
  (if elmo-plugged
      (elmo-call-func folder "create-folder")
    (elmo-dop-create-folder folder)))

(defun elmo-read-msg-no-cache (folder msg outbuf &optional msgdb)
  "Read messsage into outbuf without cacheing.
If msgdb is specified, use cache."
  (let (ret-val)
    (when msgdb
      (set-buffer outbuf)
      (erase-buffer)
      (setq ret-val
	    (elmo-cache-read 
	     ;; message-id
	     (cdr (assq msg (elmo-msgdb-get-number-alist msgdb)))
	     folder msg)))
    (if (not ret-val)
	(elmo-call-func folder "read-msg" msg outbuf))))

(defun elmo-force-cache-msg (folder number msgid)
  "Force cache message."
  (let* ((cache-file (elmo-cache-get-path msgid))
	 dir)
    (when cache-file
      (setq dir (directory-file-name (file-name-directory cache-file)))
      (if (not (file-exists-p dir))
	  (elmo-make-directory dir))
      (if (elmo-local-file-p folder number)
	  (elmo-copy-file (elmo-get-msg-filename folder number)
			  cache-file)
	(with-temp-buffer
	  (elmo-call-func folder "read-msg" number (current-buffer))
	  (as-binary-output-file 
	   (write-region (point-min) (point-max) cache-file nil 'no-msg)))))))

(defun elmo-prefetch-msg (folder msg outbuf msgdb)
  "Read message into outbuf with cacheing."
  (let* ((number-alist (elmo-msgdb-get-number-alist 
			(or msgdb (elmo-msgdb-load folder))))
	 (dir (elmo-msgdb-expand-path folder))
	 (message-id (cdr (assq msg number-alist)))
	 type
	 cache-status 
	 ret-val part-num real-fld-num)
    (set-buffer outbuf)
    (if (elmo-cache-exists-p message-id)
	t
      ;; cache doesn't exist.
      (setq real-fld-num (elmo-get-real-folder-number
			  folder msg))
      (setq type (elmo-folder-get-type (car real-fld-num)))
      (cond ((eq type 'imap4)
	     (setq ret-val (elmo-imap4-prefetch-msg 
			    (elmo-folder-get-spec (car real-fld-num))
			    (cdr real-fld-num) 
			    outbuf)))
	    ((or (eq type 'localdir)
		 (eq type 'archive))
	     ())
	    (t (setq ret-val (elmo-call-func (car real-fld-num) 
					     "read-msg" 
					     (cdr real-fld-num) outbuf))))
      (if ret-val
	  (elmo-cache-save message-id
			   (elmo-string-partial-p ret-val)
			   folder msg))
      (and ret-val t))))

;;  elmo-read-msg (folder msg outbuf msgdb)
;;; read message 
(defun elmo-read-msg (folder msg outbuf msgdb &optional force-reload)
  "Read message into outbuf."
  (let ((inhibit-read-only t))
    (if (not (elmo-use-cache-p folder msg))
	(elmo-read-msg-no-cache folder msg outbuf msgdb)
      (elmo-read-msg-with-cache folder msg outbuf msgdb force-reload))))

(defun elmo-read-msg-with-cache (folder msg outbuf msgdb 
					&optional force-reload)
  "Read message into outbuf with cacheing."
  (let* ((number-alist (elmo-msgdb-get-number-alist 
			(or msgdb (elmo-msgdb-load folder))))
	 (dir (elmo-msgdb-expand-path folder))
	 (message-id (cdr (assq msg number-alist)))
	 (type (elmo-folder-number-get-type folder msg))
	 cache-status 
	 ret-val part-num real-fld-num)
    (set-buffer outbuf)
    (if (and (not force-reload)
	     (not (eq type 'localdir)))
	(setq ret-val (elmo-cache-read message-id folder msg)))
    (if ret-val 
	t
      ;; cache doesn't exist.
      (setq real-fld-num (elmo-get-real-folder-number
			  folder msg))
      (if (setq ret-val (elmo-call-func (car real-fld-num) 
					"read-msg" 
					(cdr real-fld-num) outbuf))
	  (if (not (eq type 'localdir))
	      (elmo-cache-save message-id
			       (elmo-string-partial-p ret-val)
			       folder msg)))
      (and ret-val t))))

(defun elmo-copy-msgs (src-folder msgs dst-folder expunge-it &optional same-number)
  (elmo-call-func src-folder "copy-msgs" msgs 
		  (elmo-folder-get-spec dst-folder)
		  expunge-it
		  same-number))

(defun elmo-move-msgs (src-folder msgs dst-folder 
				  &optional msgdb all done
				  no-delete-info no-delete same-number)
  (save-excursion
    (let* ((db (or msgdb (elmo-msgdb-load src-folder)))
	   (number-alist (elmo-msgdb-get-number-alist db))
	   (messages msgs)
	   (len (length msgs))
	   (all-msg-num (or all len))
	   (done-msg-num (or done 0))
	   (tmp-buf (get-buffer-create " *elmo-move-msg*"))
	   ;elmo-no-cache-flag
	   ret-val real-fld-num done-copy
	   mes-string message-id src-cache i percent)
      (setq i done-msg-num)
      (set-buffer tmp-buf)
      ;; for elmo-archive (localdir -> archive)
      (when (and (memq (elmo-folder-get-type dst-folder) '(localdir localnews archive))
		 (memq (elmo-folder-get-type src-folder) '(localdir localnews)))
	(message (concat (if no-delete "copying" "moving")
			 " %d message(s)...") (length messages))
	(unless (elmo-copy-msgs src-folder
				messages
				dst-folder 
				nil
				same-number)
	  (error "copy message to %s failed." dst-folder))
	(setq done-copy t))
      (while messages
	(setq real-fld-num (elmo-get-real-folder-number src-folder
							(car messages)))
	(setq message-id (cdr (assq (car messages) number-alist)))
	(unless (or (eq dst-folder 'null) done-copy)
	  (if (and elmo-plugged
		   (elmo-folder-identical-system-p (car real-fld-num)
						   dst-folder))
	      ;; online and identical system...so copy 'em!
		(unless
		    (elmo-copy-msgs (car real-fld-num) 
				    (list (cdr real-fld-num))
				    dst-folder 
				    (and no-delete (not (cdr messages)))
				    same-number)
		  (error "copy message to %s failed." dst-folder))
	    ;; use cache if exists.
	    (elmo-read-msg src-folder (car messages) tmp-buf msgdb)
	    (unless (elmo-append-msg dst-folder (buffer-string) message-id
				     (if same-number (car messages)))
	      (error "move: append message to %s failed." dst-folder))))
	;; delete src cache if it is partial.
	(elmo-cache-delete-partial message-id src-folder (car messages))
	(setq ret-val (append ret-val (list (car messages))))
	(setq i (+ i 1))
	(setq percent (/ (* i 100) all-msg-num))
	(message 
	 (concat (if no-delete "copying" "moving")
		 " messages...%d%%") percent)
	(setq messages (cdr messages)))
      (kill-buffer tmp-buf)
      (if (and (not no-delete) ret-val)
	  (progn
	    (if (not no-delete-info) 
		(message "cleaning up src folder..."))
	    (if (and (elmo-delete-msgs src-folder ret-val db)
		     (elmo-msgdb-delete-msgs src-folder ret-val db t))
		(setq ret-val t)
	      (message "move: delete messages from %s failed." src-folder)
	      (setq ret-val nil)
	      )
	    (if (and ret-val
		     (not no-delete-info))
		(message "cleaning up src folder...done.")
	      )
	    ret-val)
	(if no-delete
	    (progn
	      (message "copying messages...done.")
	      t)
	  (message "moving messages failed.")
	  nil ; failure
	  )
	)
      )))

;;  boolean elmo-delete-msgs (folder msgs)
(defun elmo-delete-msgs (folder msgs &optional msgdb)
  ;; remove from real folder.
  (if elmo-plugged
      (elmo-call-func folder "delete-msgs" msgs)
    (elmo-dop-delete-msgs folder msgs msgdb)))

;;  boolean elmo-msgdb-sync (folder &optional msgdb) 
(defun elmo-msgdb-sync (folder &optional msgdb)
  (if elmo-plugged
      ();; flush event queue
    )
  )

;;
;; Server side search.
;;
(defun elmo-search (folder key &optional from-msgs)
  (let ((type (elmo-folder-get-type folder)))
    (if (or (eq type 'localdir)
	    (and elmo-plugged
		 (eq type 'nntp))
	    elmo-plugged)
	(elmo-call-func folder "search" key from-msgs)
      (elmo-cache-search folder key from-msgs))))

(defun elmo-msgdb-create (folder numlist new-mark already-mark 
				 seen-mark important-mark seen-list)
  (if elmo-plugged
      (elmo-call-func folder "msgdb-create" numlist new-mark already-mark 
		      seen-mark important-mark seen-list)
    (elmo-dop-msgdb-create folder numlist new-mark already-mark
			   seen-mark important-mark seen-list)))

(defun elmo-make-folder-numbers-list (folder msgs)
  (let ((msg-list msgs)
	pair fld-list
	ret-val)
    (while msg-list
      (when (> (car msg-list) 0)
	(setq pair (elmo-get-real-folder-number folder (car msg-list)))
	(if (setq fld-list (assq (car pair) ret-val))
	    (setcdr fld-list (cons (cdr pair) (cdr fld-list)))
	  (setq ret-val (cons (cons (car pair) (list (cdr pair))) ret-val))))
      (setq msg-list (cdr msg-list)))
    ret-val))

(defun elmo-call-func-on-markable-msgs (folder func-name msgs msgdb)
  (save-match-data
    (let ((folder-numbers (elmo-make-folder-numbers-list folder msgs))
	  type)
      (while folder-numbers
	(if (or (eq 
		 (setq type (car
			     (elmo-folder-get-spec 
			      (car (car folder-numbers)))))
		 'imap4)
		(memq type '(maildir internal)))
	    (if elmo-plugged
		(elmo-call-func (car (car folder-numbers)) func-name
				(cdr (car folder-numbers)))
	      (if elmo-enable-disconnected-operation
		  (elmo-dop-call-func-on-msgs 
		   (car (car folder-numbers)) ; real folder
		   func-name 
		   (cdr (car folder-numbers)) ; real number
		   msgdb)
		(error "Unplugged."))))
	(setq folder-numbers (cdr folder-numbers))))))

(defun elmo-unmark-important (folder msgs msgdb)
  (elmo-call-func-on-markable-msgs folder "unmark-important" msgs msgdb))
  
(defun elmo-mark-as-important (folder msgs msgdb)
  (elmo-call-func-on-markable-msgs folder "mark-as-important" msgs msgdb))

(defun elmo-mark-as-read (folder msgs msgdb)
  (elmo-call-func-on-markable-msgs folder "mark-as-read" msgs msgdb))

(defun elmo-mark-as-unread (folder msgs msgdb)
  (elmo-call-func-on-markable-msgs folder "mark-as-unread" msgs msgdb))

(defun elmo-msgdb-create-as-numlist (folder numlist new-mark already-mark 
					    seen-mark important-mark seen-list)
  (if elmo-plugged
      (elmo-call-func folder "msgdb-create-as-numlist" numlist 
		      new-mark already-mark seen-mark important-mark seen-list)
    (elmo-dop-msgdb-create-as-numlist 
     folder numlist new-mark already-mark
     seen-mark important-mark seen-list)))

;;   msgdb elmo-msgdb-load        (folder)
(defun elmo-msgdb-load (folder &optional spec)
  (message "Loading msgdb for %s..." folder)
  (let* ((path (elmo-msgdb-expand-path folder spec))
	 (ret-val 
	  (list (elmo-msgdb-overview-load path)
		(elmo-msgdb-number-load path)
		(elmo-msgdb-mark-load path)
		(elmo-msgdb-location-load path))))
    (message "Loading msgdb for %s...done." folder)
    (elmo-folder-set-info-max-by-numdb folder (nth 1 ret-val))
    ret-val))

;;   boolean elmo-msgdb-save (folder msgdb)
(defun elmo-msgdb-save (folder msgdb)
  (message "Saving msgdb for %s..." folder)
  (save-excursion
    (let ((path (elmo-msgdb-expand-path folder)))
      (elmo-msgdb-overview-save path (car msgdb))
      (elmo-msgdb-number-save path (cadr msgdb))
      (elmo-msgdb-mark-save path (caddr msgdb))
      (elmo-msgdb-location-save path (cadddr msgdb))
    ;(elmo-sync-validity folder);; for validity check!!
      ))
  (message "Saving msgdb for %s...done." folder)
  (elmo-folder-set-info-max-by-numdb folder (cadr msgdb)))

(defun elmo-msgdb-add-msgs-to-seen-list (folder msgs msgdb seen-marks)
  "Add to seen list."
  (unless (eq folder 'null) ;; black hole
    (let* ((seen-mark-list (string-to-char-list seen-marks))
	   (number-alist (elmo-msgdb-get-number-alist msgdb))
	   (mark-alist   (elmo-msgdb-get-mark-alist msgdb))
	   (dir (elmo-msgdb-expand-path folder))
	   (seen-list (elmo-msgdb-seen-load dir))
	   ent)
      (while msgs
	(if (setq ent (assq (car msgs) mark-alist))
	    (if (memq (string-to-char (cadr ent)) seen-mark-list)
		(setq seen-list
		      (cons (cdr (assq (car msgs) number-alist)) seen-list)))
	  ;; no mark ... seen...
	  (setq seen-list
		(cons (cdr (assq (car msgs) number-alist)) seen-list)))
	(setq msgs (cdr msgs)))
      (elmo-msgdb-seen-save dir seen-list)
      )
    ))

;;  msgdb elmo-append-msg (folder string)
(defun elmo-append-msg (folder string &optional message-id msg)
  (let ((type (elmo-folder-get-type folder))
	filename)
    (if (or (not (eq type 'imap4))
	    elmo-plugged)
	(progn
	  (elmo-call-func folder "append-msg" string msg)
	  )
      (elmo-dop-append-msg folder string message-id))))

(defun elmo-check-validity (folder)
  (elmo-call-func folder "check-validity" 
		  (expand-file-name
		   elmo-msgdb-validity-filename
		   (elmo-msgdb-expand-path folder))))

(defun elmo-pack-number (folder msgdb arg)
  (if (string-match "^[\\+=].*" folder)
      (elmo-call-func folder "pack-number" msgdb arg)
    (error "pack-number not supported.")))

(defun elmo-sync-validity (folder)
  (elmo-call-func folder "sync-validity" 
		  (expand-file-name
		   elmo-msgdb-validity-filename
		   (elmo-msgdb-expand-path folder))))

(defun elmo-use-cache-p (folder number)
  (elmo-call-func folder "use-cache-p" number)
  )

(defun elmo-local-file-p (folder number)
  (elmo-call-func folder "local-file-p" number))

(defun elmo-get-msg-filename (folder number)
  "Available if elmo-local-file-p is t."
  (elmo-call-func folder "get-msg-filename" number))

(defun elmo-strict-folder-diff (fld &optional number-alist) 
  (interactive)
  (let* ((dir (elmo-msgdb-expand-path fld))
	 (nalist (or number-alist 
		     (elmo-msgdb-number-load dir)))
	 (in-db (sort (mapcar 'car nalist) '<))
	 (in-folder  (elmo-list-folder fld))
	 append-list delete-list diff)
    (cons (if (equal in-folder in-db)
	      0
	    (setq diff (elmo-list-diff
			in-folder in-db
			nil
			))
	    (setq append-list (car diff))
	    (setq delete-list (cadr diff))
	    (if append-list 
		(length append-list)
	      (if delete-list
		  (- 0 (length delete-list))
		0)))
	  (length in-folder))))
		     
;; returns cons cell of (unsync . number-of-messages-in-folder)
(defun elmo-folder-diff (fld &optional number-alist)
  (interactive)
  (let ((type (elmo-folder-get-type fld)))
    (cond ((eq type 'multi)
	   (elmo-multi-folder-diff fld))
	  ((and (eq type 'filter)
		(or (elmo-multi-p fld)
		    (not (eq (nth 1 (elmo-folder-get-spec fld)) 'partial)))
		;; not partial...unsync number is unknown.
		(cons nil 
		      (cdr (elmo-folder-diff 
			    (nth 4 (elmo-folder-get-spec fld)))))))
	  (t 
	   (let ((cached-in-db-max (elmo-folder-get-info-max fld))
		 (in-folder (elmo-max-of-folder fld))
		 (in-db t)
		 unsync nomif
		 in-db-max)
	     (if (or number-alist
		     (not cached-in-db-max))
		 (let* ((dir (elmo-msgdb-expand-path fld))
			(nalist (or number-alist 
				    (elmo-msgdb-number-load dir))))
		   ;; No info-cache.
		   (setq in-db (sort (mapcar 'car nalist) '<))
		   (setq in-db-max (or (nth (max 0 (1- (length in-db))) in-db)
				       0))
		   (if (not number-alist)
		       ;; Number-alist is not used.
		       (elmo-folder-set-info-hashtb fld in-db-max
						    nil))
;; 						   (or 
;; 						    (and in-db (length in-db)) 
;; 						    0)))
		   )
	       ;; info-cache exists.
	       (setq in-db-max cached-in-db-max))
	     (setq unsync (if (and in-db 
				   (car in-folder))
			      (- (car in-folder) in-db-max)
			    (if (and in-folder
				     (null in-db))
				(cdr in-folder)
			      (if (null (car in-folder))
				  nil))))
	     (setq nomif (cdr in-folder))
	     (if (and unsync nomif (> unsync nomif))
		 (setq unsync nomif))
	     (cons (or unsync 0) (or nomif 0)))))))
    
(defsubst elmo-folder-get-info (folder &optional hashtb)
  (elmo-get-hash-val folder
		     (or hashtb elmo-folder-info-hashtb)))

(defun elmo-folder-set-info-hashtb (folder max numbers &optional new unread)
  (let ((info (elmo-folder-get-info folder)))
    (when info
      (or new     (setq new     (nth 0 info)))
      (or unread  (setq unread  (nth 1 info)))
      (or numbers (setq numbers (nth 2 info)))
      (or max     (setq max     (nth 3 info))))
    (elmo-set-hash-val folder
		       (list new unread numbers max)
		       elmo-folder-info-hashtb)))

(defun elmo-folder-set-info-max-by-numdb (folder msgdb-number)
  (let ((num-db (sort (mapcar 'car msgdb-number) '<)))
    (elmo-folder-set-info-hashtb
     folder
     (or (nth (max 0 (1- (length num-db))) num-db) 0)
     nil ;;(length num-db)
     )))

(defun elmo-folder-get-info-max (folder)
  "Get folder info from cache."
  (nth 3 (elmo-folder-get-info folder)))

(defun elmo-folder-get-info-length (folder)
  (nth 2 (elmo-folder-get-info folder)))

(defun elmo-folder-get-info-unread (folder)
  (nth 1 (elmo-folder-get-info folder)))

(defun elmo-folder-info-make-hashtb (info-alist hashtb)
  (let* ((hashtb (or hashtb
		     (elmo-make-hash (length info-alist)))))
    (mapcar
     '(lambda (x)
	(let ((info (cadr x)))
	  (and (intern-soft (car x) hashtb)
	       (elmo-set-hash-val (car x)
				  (list (nth 2 info)   ;; new
					(nth 3 info)   ;; unread
					(nth 1 info)   ;; length
					(nth 0 info))  ;; max
				  hashtb))))
     info-alist)
    (setq elmo-folder-info-hashtb hashtb)))

(defun elmo-crosspost-message-set (message-id folder-list &optional type)
  (if (assoc message-id elmo-crosspost-message-alist)
      (setcdr (assoc message-id elmo-crosspost-message-alist)
	      (list folder-list type))
    (setq elmo-crosspost-message-alist
	  (append elmo-crosspost-message-alist
		  (list (list message-id folder-list type))))))

(defun elmo-crosspost-message-delete (message-id folder)
  (let* ((id-fld (assoc message-id elmo-crosspost-message-alist))
	 (folder-list (nth 1 id-fld)))
    (when id-fld
      (if (setq folder-list (delete folder folder-list))
	  (setcar (cdr id-fld) folder-list)
	(setq elmo-crosspost-message-alist
	      (delete id-fld elmo-crosspost-message-alist))))))


(defun elmo-get-msgs-with-mark (mark-alist mark)
  (let (ret-val)
    (while mark-alist
      (if (string= (cadr (car mark-alist)) mark)
	  (cons (car (car mark-alist)) ret-val))
      (setq mark-alist (cdr mark-alist)))
    (nreverse ret-val)))
  
;; autoloads
(autoload 'elmo-imap4-commit "elmo-imap4")
(autoload 'elmo-imap4-get-connection "elmo-imap4")
(autoload 'elmo-nntp-make-groups-hashtb "elmo-nntp")
(autoload 'elmo-nntp-post "elmo-nntp")
(autoload 'elmo-localdir-max-of-folder "elmo-localdir")
(autoload 'elmo-localdir-msgdb-create-overview-entity-from-file "elmo-localdir")
(autoload 'elmo-multi-folder-diff "elmo-multi")
