;;;
;;; Wanderlust -- Yet Another Message Interface on Emacsen.
;;;
;;; Copyright (C) 1998 Yuuichi Teranishi <teranisi@gohome.org>
;;;
;;; Time-stamp: <99/06/15 16:35:21 teranisi>

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to the
;;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;;; Boston, MA 02111-1307, USA.
;;;

;;; SEMI interface

(require 'mime-view)
(require 'mime-edit)
(eval-when-compile
  (require 'mime-partial))

(require 'mmelmo)

(defvar wl-draft-buffer nil)
(defun wl-mime-yank-method (buf)
  (set-buffer wl-draft-buffer)
  (let ((mail-reply-buffer buf))
    (wl-yank-content nil))
  (kill-buffer buf))

(defun wl-mime-yank-original ()
  (let ((wl-draft-buffer (current-buffer))
	(mime-view-following-method-alist 
	 (list (cons 'mmelmo-original-mode 
		     (function wl-mime-yank-method))))
	(mime-preview-following-method-alist 
	 (list (cons 'mmelmo-original-mode
		     (function wl-mime-yank-method))))
	mes-buf)
    (if (setq mes-buf (get-buffer wl-message-buf-name))
	(save-excursion
	  (set-buffer mes-buf)
	  (widen)
	  (mime-preview-follow-current-entity)))))

(defun wl-mime-following-method (buf)
  (wl-draft-reply buf 'with-arg))

(defun wl-mime-decode-message (outbuf inbuf)
  (let ((mime-view-content-header-filter-hook 'wl-highlight-headers)
	(mime-display-header-hook 'wl-highlight-headers)
	)
    (mime-view-mode nil nil nil inbuf outbuf)
    ))

(defun wl-mime-decode-message-with-all-header (outbuf inbuf)
  (let ((mime-view-ignored-field-regexp "^:$")
	(mime-view-content-header-filter-hook 'wl-highlight-headers)
	;; for SEMI-1.7.0
	(mime-display-header-hook 'wl-highlight-headers)
	mime-view-ignored-field-list
	)
    (mime-view-mode nil nil nil inbuf outbuf))
  )

(defun wl-mime-delete-mime-out-buf ()
  (let (mime-out-buf mime-out-win)
    (if (setq mime-out-buf (get-buffer mime-echo-buffer-name))
	(if (setq mime-out-win (get-buffer-window mime-out-buf))
	    (delete-window mime-out-win)))))

(defun wl-mime-request-partial-message (folder number msgdb)
  (elmo-set-work-buf
   (elmo-read-msg-no-cache folder number (current-buffer) msgdb)
   (mime-parse-buffer)))

(defun wl-mime-combine-message/partial-pieces (entity situation)
  "Internal method for wl to combine message/partial messages
automatically."
  (interactive)
  (let* ((msgdb (save-excursion 
		  (set-buffer wl-message-buffer-cur-summary-buffer)
		  wl-summary-buffer-msgdb))
	 (mime-display-header-hook 'wl-highlight-headers)
	 (folder wl-message-buffer-cur-folder)
	 (id (cdr (assoc "id" situation)))
	 (mother (current-buffer))
	 subject-id overviews
	 (root-dir (expand-file-name
		    (concat "m-prts-" (user-login-name))
		    temporary-file-directory))
	 full-file)
    (setq root-dir (concat root-dir "/" (replace-as-filename id)))
    (setq full-file (concat root-dir "/FULL"))
    (if (or (file-exists-p full-file)
	    (not (y-or-n-p "Merge partials?"))
	    )
	(mime-store-message/partial-piece entity situation)
      (setq subject-id 
	    (elmo-eword-decode-string
	     (decode-mime-charset-string 
	      (mime-read-field 'Subject entity)
	      wl-summary-buffer-mime-charset)))
      (if (string-match "[0-9\n]+" subject-id)
	  (setq subject-id (substring subject-id 0 (match-beginning 0)))
	)
      (setq overviews (elmo-msgdb-get-overview msgdb))
      (catch 'tag
	(while overviews
	  (when (string-match
		 (regexp-quote subject-id)
		 (elmo-msgdb-overview-entity-get-subject (car overviews)))
	    (let* ((message
		    ;; request message at the cursor in Subject buffer.
		    (wl-mime-request-partial-message
		     folder
		     (elmo-msgdb-overview-entity-get-number (car overviews))
		     msgdb))
		   (situation (mime-entity-situation message))
		   (the-id (cdr (assoc "id" situation))))
	      (when (string= the-id id)
		(with-current-buffer 
		 mother
		 (mime-store-message/partial-piece message situation)
		 )
		(if (file-exists-p full-file)
		    (throw 'tag nil)
		  ))))
	  (setq overviews (cdr overviews)))
	(message "Not all partials found.")))))

(defun wl-mime-summary-burst ()
  (interactive)
  (let ((raw-buf (wl-message-get-original-buffer))
	(i 0)
	target 
	children message-entity content-type)
    (save-excursion
      (setq target wl-summary-buffer-folder-name)
      (while (not (elmo-folder-writable-p target))
	(setq target 
	      (wl-summary-read-folder wl-default-folder "to extract to"))
	)
      (wl-summary-set-message-buffer-or-redisplay)
      (set-buffer raw-buf)
      (setq children (mime-entity-children mime-message-structure))
      (message "Bursting...")
      (while children
	(setq content-type (mime-entity-content-type (car children)))
	(when (and (eq (cdr (assq 'type content-type)) 'message)
		   (eq (cdr (assq 'subtype content-type)) 'rfc822))
	  (message (format "Bursting...%s" (setq i (+ 1 i))))
	  (setq message-entity
		(car (mime-entity-children (car children))))
	  (save-restriction
	    (narrow-to-region (mime-entity-point-min message-entity)
			      (mime-entity-point-max message-entity))
	    (elmo-append-msg target
			     ;;(mime-entity-content (car children))))
			     (buffer-substring (point-min) (point-max))
			     (std11-field-body "Message-ID"))))
	(setq children (cdr children)))
      (message "Bursting...done."))
    (if (and (eq (elmo-folder-get-type target) 'imap4)
	     wl-plugged)
	(elmo-imap4-commit target))
    (wl-summary-sync-update3)))

(defun wl-mime-setup ()
  (set-alist 'mime-preview-quitting-method-alist
	     'mmelmo-original-mode 'wl-message-exit)
  (set-alist 'mime-view-over-to-previous-method-alist
	     'mmelmo-original-mode 'wl-message-exit)
  (set-alist 'mime-view-over-to-next-method-alist 
	     'mmelmo-original-mode 'wl-message-exit)
  (set-alist 'mime-preview-over-to-previous-method-alist
	     'mmelmo-original-mode 'wl-message-exit)
  (set-alist 'mime-preview-over-to-next-method-alist
	     'mmelmo-original-mode 'wl-message-exit)
  (add-hook 'wl-summary-redisplay-hook  'wl-mime-delete-mime-out-buf)
  (add-hook 'wl-message-exit-hook 'wl-mime-delete-mime-out-buf)
  (setq wl-mime-decode-func 'wl-mime-decode-message)
  (setq wl-mime-decode-with-all-header-func 
	'wl-mime-decode-message-with-all-header)
  (setq wl-editor-mode-func 'mime-edit-mode)
  (setq wl-edit-again-func  'mime-edit-again)
  (setq wl-insert-mail-func 'mime-edit-insert-mail)
  (defalias 'wl-summary-burst 'wl-mime-summary-burst)
;  (autoload 'mime-method-to-combine-message/partial-pieces "mime-partial" nil)
;  (autoload 'mime-combine-message/partial-pieces-automatically 
;    "mime-partial" nil)
;  (require 'mime-partial)
  (ctree-set-calist-strictly
   'mime-acting-condition
   '((type . message) (subtype . partial)
     (method .  wl-mime-combine-message/partial-pieces)
     (request-partial-message-method . wl-mime-request-partial-message)
     (major-mode . mmelmo-original-mode)
     ))
  (set-alist 'mime-preview-following-method-alist 
	     'mmelmo-original-mode
	     (function wl-mime-following-method))
  (set-alist 'mime-view-following-method-alist 
	     'mmelmo-original-mode
	     (function wl-mime-following-method))
  (set-alist 'mime-edit-message-inserter-alist
	     'wl-draft-mode (function wl-draft-insert-message))
  (set-alist 'mime-edit-mail-inserter-alist
	     'wl-draft-mode (function wl-draft-insert-message))
  (set-alist 'mime-edit-split-message-sender-alist
	     'wl-draft-mode 
	     (cdr (assq 'mail-mode mime-edit-split-message-sender-alist)))
  (setq wl-yank-original-func 'wl-mime-yank-original)  
  (setq wl-caesar-region-func 'mule-caesar-region)
;  (setq mime-view-content-header-filter-hook 'wl-highlight-headers)
;  (setq mime-display-header-hook 'wl-highlight-headers)
  (setq wl-message-read-func 'mime-preview-scroll-up-entity)
  (setq wl-message-next-content-func 'mime-preview-move-to-next)
  (setq wl-message-prev-content-func 'mime-preview-move-to-previous)
  (setq wl-message-play-content-func 'mime-preview-play-current-entity)
  (setq wl-message-extract-content-func 'mime-preview-extract-current-entity)
  (setq wl-message-quit-func 'mime-preview-quit)
  (setq wl-message-button-dispatcher-func 'mime-button-dispatcher)
  
  (set-alist 'mime-raw-representation-type-alist
	     'mmelmo-original-mode 'binary)
  (setq mmelmo-sort-field-list wl-message-sort-field-list)
  )

(provide 'wl-mime)
