;;;
;;; Wanderlust -- Yet Another Message Interface on Emacsen.
;;;
;;; Copyright (C) 1998 Yuuichi Teranishi <teranisi@gohome.org>
;;;
;;; Time-stamp: <99/06/28 17:52:59 teranisi>

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to the
;;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;;; Boston, MA 02111-1307, USA.
;;;

(require 'elmo2)

(require 'wl-message)
(require 'wl-vars)
(require 'wl-highlight)
(require 'wl-refile)
(require 'wl-util)
(condition-case ()
    (progn
      (require 'timezone)
      (require 'easymenu))
  (error))
(require 'elmo-date)

(condition-case nil
    (require 'ps-print)
  (error))

(eval-when-compile
  (mapcar
   (function
    (lambda (symbol)
      (unless (boundp symbol)
	(make-local-variable symbol)
	(eval (list 'setq symbol nil)))))
   '(dragdrop-drop-functions scrollbar-height mail-reply-buffer))
  (or (fboundp 'ps-print-buffer-with-faces)
      (defun ps-print-buffer-with-faces (a)))
  (unless (featurep 'xemacs)
    ;; dummy.
    (defun elmo-database-msgid-put (a b c))
    (defun elmo-database-close ()) 
    (defun elmo-database-msgid-get (a))
    ))

(defvar wl-summary-buffer-name "Summary")
(defvar wl-summary-mode-map nil)

(defvar wl-summary-buffer-msgdb       nil)
(defvar wl-summary-buffer-folder-name nil)
(defvar wl-summary-buffer-disp-msg    nil)
(defvar wl-summary-buffer-disp-folder nil)
(defvar wl-summary-buffer-refile-list nil) 
(defvar wl-summary-buffer-delete-list nil) 
(defvar wl-summary-buffer-last-displayed-msg nil)
(defvar wl-summary-buffer-current-msg nil)
(defvar wl-summary-buffer-unread-status " (0 new/0 unread)")
(defvar wl-summary-buffer-unread-count 0)
(defvar wl-summary-buffer-new-count    0)
(defvar wl-summary-buffer-mime-charset  nil)
(defvar wl-summary-buffer-weekday-name-lang  nil)
(defvar wl-summary-buffer-thread-indent-set-alist  nil)
(defvar wl-summary-buffer-message-redisplay-func nil)
(defvar wl-summary-buffer-view 'thread)
(defvar wl-summary-buffer-message-modified nil)
(defvar wl-summary-buffer-mark-modified nil)
(defvar wl-summary-buffer-number-column nil)
(defvar wl-summary-buffer-number-regexp nil)
(defvar wl-summary-buffer-persistent nil)
(defvar wl-summary-buffer-thread-nodes nil)
(defvar wl-summary-buffer-temp-mark-list nil)
(defvar wl-summary-buffer-copy-list nil) 
(defvar wl-summary-buffer-prev-refile-destination nil)
(defvar wl-summary-buffer-prev-copy-destination nil)
(defvar wl-thread-indent-level-internal nil)
(defvar wl-thread-have-younger-brother-str-internal nil)
(defvar wl-thread-youngest-child-str-internal nil)
(defvar wl-thread-vertical-str-internal nil)
(defvar wl-thread-horizontal-str-internal nil)
(defvar wl-thread-space-str-internal nil)
(defvar wl-summary-last-visited-folder nil)
(defvar wl-read-folder-hist nil)

(defvar wl-summary-message-regexp "^ *\\([0-9]+\\)")

(defvar wl-summary-shell-command-last "")

(defvar wl-ps-preprint-hook nil)
(defvar wl-ps-print-hook nil)

(mapcar 
 (function make-variable-buffer-local)
 (list 'wl-summary-buffer-msgdb
       'wl-summary-buffer-disp-msg
       'wl-summary-buffer-disp-folder
       'wl-summary-buffer-refile-list
       'wl-summary-buffer-copy-list
       'wl-summary-buffer-temp-mark-list
       'wl-summary-buffer-delete-list
       'wl-summary-buffer-folder-name
       'wl-summary-buffer-last-displayed-msg
       'wl-summary-buffer-unread-status
       'wl-summary-buffer-unread-count
       'wl-summary-buffer-new-count
       'wl-summary-buffer-mime-charset
       'wl-summary-buffer-weekday-name-lang
       'wl-summary-buffer-thread-indent-set
       'wl-summary-buffer-message-redisplay-func
       'wl-summary-buffer-view
       'wl-summary-buffer-message-modified
       'wl-summary-buffer-mark-modified
       'wl-summary-buffer-number-column
       'wl-summary-buffer-number-regexp
       'wl-summary-buffer-persistent
       'wl-summary-buffer-thread-nodes
       'wl-summary-buffer-prev-refile-destination
       'wl-thread-indent-level-internal
       'wl-thread-have-younger-brother-str-internal
       'wl-thread-youngest-child-str-internal
       'wl-thread-vertical-str-internal
       'wl-thread-horizontal-str-internal
       'wl-thread-space-str-internal
       ))

;; internal functions (dummy)
(defun wl-summary-append-message-func-internal (entity overview 
						       mark-alist update))
(defun wl-summary-from-func-internal (from)
  from)
(defun wl-summary-subject-func-internal (subject)
  subject)
(defun wl-summary-subject-filter-func-internal (subject)
  subject)

(defmacro wl-summary-sticky-buffer-name (folder)
  (` (concat wl-summary-buffer-name ":" (, folder))))

(defun wl-summary-default-subject (subject-string)
  (if (string-match "^[ \t]*\\[[^:]+[,: ][0-9]+\\][ \t]*" subject-string)
      (substring subject-string (match-end 0))
    subject-string
    ))

(defun wl-summary-default-from (string)
  (if wl-use-petname
      (wl-address-get-petname string)
    string))

(defvar wl-summary-mode-menu-spec
  '("Summary"
    ["Read" wl-summary-read t]
    ["Prev page" wl-summary-prev-page t]
    ["Next page" wl-summary-next-page t]
    ["Top"       wl-summary-display-top t]
    ["Bottom"    wl-summary-display-bottom t]
    ["Prev"      wl-summary-prev t]
    ["Next"      wl-summary-next t]
    ["Up"        wl-summary-up t]
    ["Down"      wl-summary-down t]
    ["Parent message" wl-summary-jump-to-parent-message t]
    "----"
    ["Sync"            wl-summary-sync t]
    ["Execute"         wl-summary-exec t]
    ["Go to other folder" wl-summary-goto-folder t]
    ["Pick" wl-summary-pick t]
    ["Mark as read all" wl-summary-mark-as-read-all t]
    ["Unmark all"      wl-summary-unmark-all t]
    ["Toggle display message" wl-summary-toggle-disp-msg t]
    ["Display folder" wl-summary-toggle-disp-folder t]
    ["Toggle threading" wl-summary-toggle-thread t]
    ["Stick" wl-summary-stick t]
    ("Sort"
     ["By Number" wl-summary-sort-by-number t]
     ["By Date" wl-summary-sort-by-date t]
     ["By From" wl-summary-sort-by-from t]
     ["By Subject" wl-summary-sort-by-subject t]
     )    
    "----"
    ("Message Operation"
     ["Mark as read"    wl-summary-mark-as-read t]
     ["Set delete mark" wl-summary-delete t]
     ["Set refile mark" wl-summary-refile t]
     ["Set copy mark"   wl-summary-copy t]
     ["Prefetch"        wl-summary-prefetch t]
     ["Mark"            wl-summary-temp-mark-line t]
     ["Unmark"          wl-summary-unmark-line t]
     ["Mark as important" wl-summary-mark-as-important t]
     ["Mark as unread"   wl-summary-mark-as-unread t]
     ["Cancel posted news" wl-summary-cancel-message t]
     ["Enter the message" wl-summary-jump-to-current-message t]
     ["Pipe message" wl-summary-pipe-message t]
     ["Print message" wl-summary-print-message t])
    ("Thread Operation"
     ["Open or Close" wl-thread-open-close (eq wl-summary-buffer-view 'thread)]
     ["Open all"     wl-thread-open-all (eq wl-summary-buffer-view 'thread)]
     ["Close all"    wl-thread-close-all (eq wl-summary-buffer-view 'thread)]
     ["Mark as read" wl-thread-mark-as-read (eq wl-summary-buffer-view 'thread)]
     ["Set delete mark"  wl-thread-delete (eq wl-summary-buffer-view 'thread)]
     ["Set refile mark"  wl-thread-refile (eq wl-summary-buffer-view 'thread)]
     ["Set copy mark"    wl-thread-copy (eq wl-summary-buffer-view 'thread)]
     ["Prefetch"     wl-thread-prefetch (eq wl-summary-buffer-view 'thread)]
     ["Mark"        wl-thread-temp-mark (eq wl-summary-buffer-view 'thread)]
     ["Unmark"      wl-thread-unmark (eq wl-summary-buffer-view 'thread)]
     ["Execute"      wl-thread-exec (eq wl-summary-buffer-view 'thread)]
     )
    ("Region Operation"
     ["Mark as read" wl-summary-mark-as-read-region t]
     ["Set delete mark" wl-summary-delete-region t]
     ["Set refile mark" wl-summary-refile-region t]
     ["Set copy mark" wl-summary-copy-region t]
     ["Prefetch" wl-summary-prefetch-region t]
     ["Mark" wl-summary-temp-mark-region t]
     ["Unmark" wl-summary-unmark-region t]
     ["Execute" wl-summary-exec-region t]
     )
    ("Mark Operation"
     ["Mark as read" wl-summary-temp-mark-mark-as-read t]
     ["Set delete mark" wl-summary-temp-mark-delete t]
     ["Set refile mark" wl-summary-temp-mark-refile t]
     ["Set copy mark" wl-summary-temp-mark-copy t]
     ["Prefetch" wl-summary-temp-mark-prefetch t]
     ["Reply with citation" wl-summary-temp-mark-reply-with-citation t]
     ["Forward" wl-summary-temp-mark-forward t]
     ["uudecode" wl-summary-temp-mark-uudecode t]
     )
    "----"
    ("Writing Messages"
     ["Write a message" wl-draft t]
     ["Reply" wl-summary-reply t]
     ["Reply with citation" wl-summary-reply-with-citation t]
     ["Forward" wl-summary-forward t]
     )
    "----"
    ["Toggle Plug Status" wl-toggle-plugged t]
    "----"
    ["Exit Current Folder" wl-summary-exit t]
    ))

(if wl-on-xemacs
    (defun wl-summary-setup-mouse ()
      (define-key wl-summary-mode-map 'button4 'wl-summary-prev)
      (define-key wl-summary-mode-map 'button5 'wl-summary-next)
      (define-key wl-summary-mode-map [(shift button4)] 
	'wl-summary-up)
      (define-key wl-summary-mode-map [(shift button5)] 
	'wl-summary-down)
      (define-key wl-summary-mode-map 'button2 'wl-summary-click))
  (if wl-on-nemacs
      (defun wl-summary-setup-mouse ())
    (defun wl-summary-setup-mouse ()
      (define-key wl-summary-mode-map [mouse-4] 'wl-summary-prev)
      (define-key wl-summary-mode-map [mouse-5] 'wl-summary-next)
      (define-key wl-summary-mode-map [S-mouse-4] 'wl-summary-up)
      (define-key wl-summary-mode-map [S-mouse-5] 'wl-summary-down)
      (define-key wl-summary-mode-map [mouse-2] 'wl-summary-click))))

(if wl-summary-mode-map
    ()
  (setq wl-summary-mode-map (make-sparse-keymap))
  (define-key wl-summary-mode-map " "    'wl-summary-read)
  (define-key wl-summary-mode-map "."    'wl-summary-redisplay)
  (define-key wl-summary-mode-map "<"    'wl-summary-display-top)
  (define-key wl-summary-mode-map ">"    'wl-summary-display-bottom)
  (define-key wl-summary-mode-map "\177" 'wl-summary-prev-page)
  (define-key wl-summary-mode-map "\r"   'wl-summary-next-line-content)
  (define-key wl-summary-mode-map "\C-m" 'wl-summary-next-line-content)
  (define-key wl-summary-mode-map "/"    'wl-thread-open-close)
  (define-key wl-summary-mode-map "["    'wl-thread-open-all)
  (define-key wl-summary-mode-map "]"    'wl-thread-close-all)
  (define-key wl-summary-mode-map "-"    'wl-summary-prev-line-content)
  (define-key wl-summary-mode-map "\e\r" 'wl-summary-prev-line-content)
  (define-key wl-summary-mode-map "g"    'wl-summary-goto-folder)
  (define-key wl-summary-mode-map "c"    'wl-summary-mark-as-read-all)
  (define-key wl-summary-mode-map "D"    'wl-summary-drop-unsync)
  
  (define-key wl-summary-mode-map "a"    'wl-summary-reply)
  (define-key wl-summary-mode-map "A"    'wl-summary-reply-with-citation)
  (define-key wl-summary-mode-map "C"    'wl-summary-cancel-message)
  (define-key wl-summary-mode-map "E"    'wl-summary-reedit)
  (define-key wl-summary-mode-map "f"    'wl-summary-forward)
  (define-key wl-summary-mode-map "$"    'wl-summary-mark-as-important)
  (define-key wl-summary-mode-map "@"    'wl-summary-edit-petname)

  (define-key wl-summary-mode-map "y"    'wl-summary-save)
  (define-key wl-summary-mode-map "n"    'wl-summary-next)
  (define-key wl-summary-mode-map "p"    'wl-summary-prev)
  (define-key wl-summary-mode-map "N"    'wl-summary-down)
  (define-key wl-summary-mode-map "P"    'wl-summary-up)
  (define-key wl-summary-mode-map "w"    'wl-draft)
  (define-key wl-summary-mode-map "W"    'wl-summary-write-current-newsgroup)
;  (define-key wl-summary-mode-map "e"     'wl-draft-open-file)
  (define-key wl-summary-mode-map "e"     'wl-summary-save)
  (define-key wl-summary-mode-map "\C-c\C-o" 'wl-jump-to-draft-buffer)
  (define-key wl-summary-mode-map "H"    'wl-summary-redisplay-all-header)
  (define-key wl-summary-mode-map "M"    'wl-summary-redisplay-no-mime)
  (define-key wl-summary-mode-map "B"    'wl-summary-burst)
  (define-key wl-summary-mode-map "Z"    'wl-status-update)
  (define-key wl-summary-mode-map "#"    'wl-summary-print-message)
  (define-key wl-summary-mode-map "|"    'wl-summary-pipe-message)
  (define-key wl-summary-mode-map "q"    'wl-summary-exit)
  (define-key wl-summary-mode-map "Q"    'wl-summary-force-exit)
  
  (define-key wl-summary-mode-map "j"    'wl-summary-jump-to-current-message)
  (define-key wl-summary-mode-map "J"    'wl-summary-jump-to-msg)
  (define-key wl-summary-mode-map "I"    'wl-summary-incorporate)
  (define-key wl-summary-mode-map "\M-j" 'wl-summary-jump-to-msg-by-message-id)
  (define-key wl-summary-mode-map "^"    'wl-summary-jump-to-parent-message)
  (define-key wl-summary-mode-map "!"    'wl-summary-mark-as-unread)
  
  (define-key wl-summary-mode-map "s"    'wl-summary-sync)
  (define-key wl-summary-mode-map "S"    'wl-summary-sort)
  (define-key wl-summary-mode-map "\M-s"    'wl-summary-stick)
  (define-key wl-summary-mode-map "T"    'wl-summary-toggle-thread)

  (define-key wl-summary-mode-map "l"    'wl-summary-toggle-disp-folder)
  (define-key wl-summary-mode-map "v"    'wl-summary-toggle-disp-msg)
  (define-key wl-summary-mode-map "V"    'wl-summary-virtual)

  (define-key wl-summary-mode-map "\C-i"  'wl-summary-goto-last-displayed-msg)
  (define-key wl-summary-mode-map "?"    'wl-summary-pick)
  (define-key wl-summary-mode-map "\ee"  'wl-summary-expire)

  ;; line commands
  (define-key wl-summary-mode-map "R"    'wl-summary-mark-as-read)
  (define-key wl-summary-mode-map "i"    'wl-summary-prefetch)
  (define-key wl-summary-mode-map "x"    'wl-summary-exec)
  (define-key wl-summary-mode-map "*"    'wl-summary-temp-mark-line)
  (define-key wl-summary-mode-map "o"    'wl-summary-refile)
  (define-key wl-summary-mode-map "O"    'wl-summary-copy)
  (define-key wl-summary-mode-map "\M-o" 'wl-summary-refile-prev-destination)
;  (define-key wl-summary-mode-map "\M-O" 'wl-summary-copy-prev-destination)
  (define-key wl-summary-mode-map "\C-o" 'wl-summary-auto-refile)  
  (define-key wl-summary-mode-map "d"    'wl-summary-delete)
  (define-key wl-summary-mode-map "u"    'wl-summary-unmark-line)
  (define-key wl-summary-mode-map "U"    'wl-summary-unmark-all)

  ;; region commands
  (define-key wl-summary-mode-map "r"    (make-sparse-keymap))
  (define-key wl-summary-mode-map "rR"   'wl-summary-mark-as-read-region)
  (define-key wl-summary-mode-map "ri"   'wl-summary-prefetch-region)
  (define-key wl-summary-mode-map "rx"   'wl-summary-exec-region)
  (define-key wl-summary-mode-map "r*"   'wl-summary-temp-mark-region)
  (define-key wl-summary-mode-map "ro"   'wl-summary-refile-region)
  (define-key wl-summary-mode-map "rO"   'wl-summary-copy-region)
  (define-key wl-summary-mode-map "rd"   'wl-summary-delete-region)
  (define-key wl-summary-mode-map "ru"   'wl-summary-unmark-region)

  ;; thread commands
  (define-key wl-summary-mode-map "t"	(make-sparse-keymap))
  (define-key wl-summary-mode-map "tR" 'wl-thread-mark-as-read)
  (define-key wl-summary-mode-map "ti" 'wl-thread-prefetch)
  (define-key wl-summary-mode-map "tx" 'wl-thread-exec)
  (define-key wl-summary-mode-map "t*" 'wl-thread-temp-mark)
  (define-key wl-summary-mode-map "to" 'wl-thread-refile)
  (define-key wl-summary-mode-map "tO" 'wl-thread-copy)
  (define-key wl-summary-mode-map "td" 'wl-thread-delete)
  (define-key wl-summary-mode-map "tu" 'wl-thread-unmark)

  ;; temp-mark commands
  (define-key wl-summary-mode-map "m"	  (make-sparse-keymap))
  (define-key wl-summary-mode-map "mi"   'wl-summary-temp-mark-prefetch)
  (define-key wl-summary-mode-map "mR"   'wl-summary-temp-mark-mark-as-read)
  (define-key wl-summary-mode-map "mo"   'wl-summary-temp-mark-refile)
  (define-key wl-summary-mode-map "mO"   'wl-summary-temp-mark-copy)
  (define-key wl-summary-mode-map "md"   'wl-summary-temp-mark-delete)
  
  (define-key wl-summary-mode-map "mu"   'wl-summary-delete-all-temp-marks)
  (define-key wl-summary-mode-map "mU"   'wl-summary-temp-mark-uudecode)
  (define-key wl-summary-mode-map "ma"   'wl-summary-temp-mark-all)
  (define-key wl-summary-mode-map "mt"   'wl-summary-temp-mark-thread)
  (define-key wl-summary-mode-map "mr"   'wl-summary-temp-mark-region)
  (define-key wl-summary-mode-map "mA"   'wl-summary-temp-mark-reply-with-citation)
  (define-key wl-summary-mode-map "mf"   'wl-summary-temp-mark-forward)

  (define-key wl-summary-mode-map "\M-t" 'wl-toggle-plugged)
  ;;
  (wl-summary-setup-mouse)
  (easy-menu-define
   wl-summary-mode-menu
   wl-summary-mode-map
   "Menu used in Summary mode."
   wl-summary-mode-menu-spec)
  )

(defun wl-status-update ()
  (interactive)
  (wl-address-setup))

(defun wl-summary-display-top ()
  (interactive)
  (goto-char (point-min))
  (if wl-summary-buffer-disp-msg
      (wl-summary-redisplay)))

(defun wl-summary-display-bottom ()
  (interactive)
  (goto-char (point-max))
  (forward-line -1)
  (if wl-summary-buffer-disp-msg
      (wl-summary-redisplay)))

(defun wl-summary-collect-unread (mark-alist &optional folder)
  (let (mark ret-val)
    (while mark-alist
      (setq mark (cadr (car mark-alist)))
      (and mark
	   (or (string= mark wl-summary-new-mark)
	       (string= mark wl-summary-unread-uncached-mark)
	       (string= mark wl-summary-unread-cached-mark))
	   (setq ret-val (cons (car (car mark-alist)) ret-val)))
      (setq mark-alist (cdr mark-alist)))
    ret-val))

(defun wl-summary-count-unread (mark-alist &optional folder)
  (let ((new 0)
	(unread 0)
	mark)
    (while mark-alist
      (setq mark (cadr (car mark-alist)))
      (and mark
	   (cond 
	    ((string= mark wl-summary-new-mark)
	     (setq new (+ 1 new))
	     )
	    ((or (string= mark wl-summary-unread-uncached-mark)
		 (string= mark wl-summary-unread-cached-mark))
	     (setq unread (+ 1 unread)))))
      (setq mark-alist (cdr mark-alist)))
    (if (eq major-mode 'wl-summary-mode)
	(setq wl-summary-buffer-new-count new 
	      wl-summary-buffer-unread-count unread))
    (+ new unread)))

(defun wl-summary-make-modeline ()
  "Create new modeline format for Wanderlust"
  (let* ((duplicated (copy-sequence mode-line-format))
	 (cur-entry duplicated)
	 return-modeline)
    (if (memq 'wl-plug-state-indicator mode-line-format)
	duplicated
      (catch 'done
	(while cur-entry
	  (if (or (and (symbolp (car cur-entry))
		       (eq 'mode-line-buffer-identification 
			      (car cur-entry)))
		  (and (consp (car cur-entry))
		       (or 
			(eq 'modeline-buffer-identification 
			       (car (car cur-entry)))
			(eq 'modeline-buffer-identification 
			       (cdr (car cur-entry))))))
	      (progn
		(setq return-modeline (append return-modeline
					      (list 
					       'wl-plug-state-indicator
					       (car cur-entry)
					       'wl-summary-buffer-unread-status)
					      (cdr cur-entry)))
		(throw 'done return-modeline))
	    (setq return-modeline (append return-modeline
					  (list (car cur-entry)))))
	  (setq cur-entry (cdr cur-entry)))))))

(defun wl-summary-reedit ()
  "Re-edit current message."
  (interactive)
  (if (string= wl-summary-buffer-folder-name wl-draft-folder)
      (if (wl-summary-message-number)
	  (unwind-protect
	      (wl-draft-reedit (wl-summary-message-number))
	    (mail-position-on-field "To")
	    (delete-other-windows)))
  (save-excursion
    (let ( ;(orig-message-buf (wl-message-get-original-buffer))
	  (mmelmo-force-fetch-entire-message t)
	  )
;     (wl-summary-redisplay)
;     (set-buffer orig-message-buf)
      (wl-summary-set-message-buffer-or-redisplay)
      (set-buffer (wl-message-get-original-buffer))
      (wl-draft-edit-string (buffer-substring (point-min) (point-max)))))))

(defun wl-summary-msgdb-load-async (folder)
  "Loading msgdb and selecting folder is executed asynchronously in IMAP4.
Returns nil if selecting folder was in failure."
  (if (and wl-plugged
	   (eq (elmo-folder-get-type folder) 'imap4))
      (let* ((spec (elmo-folder-get-spec folder))
	     (imap-fld (nth 1 spec))
	     (user (nth 3 spec))
	     (server (nth 2 spec))
	     (auth (nth 4 spec))
	     (port (nth 5 spec))
	     (ssl (nth 6 spec))
	     (connection (elmo-imap4-get-connection user server auth port
						    ssl))
	     (buffer (nth 0 connection))
	     (process (nth 1 connection))
	     msgdb response)
	(save-excursion
	  (unwind-protect
	      (progn
		(elmo-imap4-send-command buffer process
					 (format "select %s" imap-fld))
		(setq msgdb (elmo-msgdb-load (wl-string folder)))
		(setq response (elmo-imap4-read-response buffer process)))
	    (setcar (cddr connection) imap-fld))
	  (if response msgdb)))
    (elmo-msgdb-load (wl-string folder))))

(defun wl-summary-mode (folder)
  "Major mode for reading threaded messages.
The keys that are defined for this mode are:

SPC	Read messages. 
DEL	Back-scroll this message. 
.	Force to display this message. 
RET	Make this message scroll up with one line.
M-RET -	Make this message scroll down with one line.

C-n	Go to the next line.
C-p	Go to the previous line.
n	Move to below then display. 
N       Move to next unread.
p	Move to above then display. 
P       Move to previous unread.
s	Sync current folder.
t       Same as 's' but force update. 
g	Go to the folder which you input.
w	Write a message. A new draft is prepared.
a	Answer to this message. A new draft is prepared in Draft mode. 
f	Forward this message to a third person. A new draft is prepared in 
	Draft mode and this message is automatically attached.
v	Toggle \"Summary and Folder view\".
        You can quickly put the delete marks since the next message is not 
        displayed.
i       Prefetch message if uncached.
o	Put the refile mark('o') on this message. 
!	Mark current message as unread.
$	Toggle mark current message as important.
d	Put the delete mark('D') on this message.
c       Check all messages as read.
*	Put the temporal mark('*') on this message. 
u	Cancel the mark on this message.
x	Process marked messages. 

mo	Put the refile mark onto all messages marked with '*'.
	This is very convenient to refile all messages picked by '?'.
md	Put the delete mark onto all messages marked with '*'.
mi      Prefetch all messages marked with '*'.
mu	Unmark all temp-marked messages.
mt      Put the '*' mark onto all messages which belong to th current thread.
ma      Put the '*' mark onto all messages.
?	Pick messages according to a pick pattern which you input, 
	then put the '*' mark onto them.
q	Goto folder mode.
"
  (interactive)
  (setq major-mode 'wl-summary-mode)
  (setq mode-name "Summary")
  (use-local-map wl-summary-mode-map)
  (setq wl-summary-buffer-folder-name folder)
;  (setq wl-summary-buffer-msgdb (save-excursion (elmo-msgdb-load folder)))
;  (setq wl-summary-buffer-msgdb (wl-summary-msgdb-load-async folder))
  (setq wl-summary-buffer-refile-list nil)
  (setq wl-summary-buffer-temp-mark-list nil)
  (setq wl-summary-buffer-delete-list nil)
  (setq default-directory (or wl-tmp-dir (expand-file-name "~/")))  
  (setq buffer-read-only t)
  (setq truncate-lines t)
  (make-local-variable 'tab-width)
  (setq tab-width 1)
  (buffer-disable-undo)
  (setq wl-summary-buffer-mime-charset (or (wl-get-assoc-list-value 
					    wl-folder-mime-charset-alist
					    folder) 
					   wl-mime-charset))
  (setq wl-summary-buffer-weekday-name-lang 
	(or (wl-get-assoc-list-value 
	     wl-folder-weekday-name-lang-alist
	     folder) 
	    wl-summary-weekday-name-lang))
  (setq wl-summary-buffer-thread-indent-set
	(wl-get-assoc-list-value 
	 wl-folder-thread-indent-set-alist
	 folder))
  (setq wl-summary-buffer-persistent (wl-folder-persistent-p folder))
  (setq wl-summary-buffer-number-column 
	(or (wl-get-assoc-list-value wl-summary-number-column-alist folder)
	    wl-summary-default-number-column))
  (setq wl-summary-buffer-number-regexp 
	(wl-repeat-string "." wl-summary-buffer-number-column))
  (if wl-use-semi
      (setq wl-summary-buffer-message-redisplay-func 
	    'wl-mmelmo-message-redisplay)
    (setq wl-summary-buffer-message-redisplay-func
	  'wl-normal-message-redisplay))
  (wl-xmas-setup-summary) ; setup toolbar, dnd, etc.
  (when wl-show-plug-status-on-modeline 
    (setq mode-line-format (wl-summary-make-modeline)))
  (setq 
   wl-thread-indent-level-internal
   (or (nth 0 wl-summary-buffer-thread-indent-set)
       wl-thread-indent-level)
   wl-thread-have-younger-brother-str-internal
   (or (nth 1 wl-summary-buffer-thread-indent-set)
       wl-thread-have-younger-brother-str)
   wl-thread-youngest-child-str-internal
   (or (nth 2 wl-summary-buffer-thread-indent-set)
       wl-thread-youngest-child-str)
   wl-thread-vertical-str-internal
   (or (nth 3 wl-summary-buffer-thread-indent-set)
       wl-thread-vertical-str)
   wl-thread-horizontal-str-internal
   (or (nth 4 wl-summary-buffer-thread-indent-set)
       wl-thread-horizontal-str)
   wl-thread-space-str-internal
   (or (nth 5 wl-summary-buffer-thread-indent-set)
       wl-thread-space-str))
  (setq wl-thread-indent-regexp 
	(concat (regexp-quote wl-thread-have-younger-brother-str-internal) "\\|"
		(regexp-quote wl-thread-youngest-child-str-internal) "\\|"
		(regexp-quote wl-thread-vertical-str-internal) "\\|"
		(regexp-quote wl-thread-horizontal-str-internal) "\\|"
		(regexp-quote wl-thread-space-str-internal)))
  (easy-menu-add wl-summary-mode-menu)
  (run-hooks 'wl-summary-mode-hook)
  )

(defun wl-summary-overview-entity-compare-by-date (x y)
  "Compare entity by date"
  (string<
   (timezone-make-date-sortable 
    (elmo-msgdb-overview-entity-get-date x))
   (timezone-make-date-sortable 
    (elmo-msgdb-overview-entity-get-date y))))

(defun wl-summary-overview-entity-compare-by-number (x y)
  "Compare entity by number"
  (<
   (elmo-msgdb-overview-entity-get-number x)
   (elmo-msgdb-overview-entity-get-number y)))

(defun wl-summary-overview-entity-compare-by-from (x y)
  "Compare entity by from"
  (string<
   (wl-address-header-extract-address
    (or (elmo-msgdb-overview-entity-get-from x)
	wl-summary-no-from-message))
   (wl-address-header-extract-address
    (or (elmo-msgdb-overview-entity-get-from y)
	wl-summary-no-from-message))))

(defun wl-summary-overview-entity-compare-by-subject (x y)
  "Compare entity by subject"
  (string< (elmo-msgdb-overview-entity-get-subject x)
	   (elmo-msgdb-overview-entity-get-subject y)))

(defun wl-summary-sort-by-date ()
  (interactive)
  (wl-summary-rescan "date"))
(defun wl-summary-sort-by-number ()
  (interactive)
  (wl-summary-rescan "number"))
(defun wl-summary-sort-by-subject ()
  (interactive)
  (wl-summary-rescan "subject"))
(defun wl-summary-sort-by-from ()
  (interactive)
  (wl-summary-rescan "from"))

(defun wl-summary-rescan (&optional sort-by)
  "Rescan current folder without updating."
  (interactive)
  (let* ((cur-buf (current-buffer))
	 (msgdb wl-summary-buffer-msgdb) 
	 (overview (elmo-msgdb-get-overview msgdb))
	 ;;(number-alist (elmo-msgdb-get-number-alist msgdb))
	 (mark-alist (elmo-msgdb-get-mark-alist msgdb))
	 (elmo-mime-charset wl-summary-buffer-mime-charset)
	 i percent num
	 gc-message entity
	 curp
	 (inhibit-read-only t)
	 (buffer-read-only nil)
	 )
    (fset 'wl-summary-append-message-func-internal 
	  (wl-summary-get-append-message-func))
    (erase-buffer)
    (message "Re-scanning...")
    (setq i 0)
    (setq num (length overview))
    (when sort-by
      (message "Sorting by %s..." sort-by)
      (setq overview
	    (sort overview
		  (intern (format "wl-summary-overview-entity-compare-by-%s"
				  sort-by))))
      (message "Sorting by %s...done" sort-by)
      (elmo-msgdb-set-overview wl-summary-buffer-msgdb
			       overview)
      )
    (setq curp overview)
    (set-buffer cur-buf)
    (setq wl-thread-entity-hashtb (elmo-make-hash))
    (setq wl-thread-entity-list nil)
    (setq wl-thread-entities nil)
    (setq wl-summary-buffer-temp-mark-list nil)
    (setq wl-summary-buffer-refile-list nil)
    (setq wl-summary-buffer-delete-list nil)
    (message "Constructing summary structure..." percent)
    (while curp
      (setq entity (car curp))
      (wl-summary-append-message-func-internal entity overview mark-alist
					       nil)
      (setq curp (cdr curp))
      (setq i (+ i 1))
      (message "Constructing summary structure...%d%%" (/ (* i 100) num))
      )
    (message "Constructing summary structure...done." percent)
    (set-buffer cur-buf)
    (when (eq wl-summary-buffer-view 'thread)
      (message "Inserting thread...")
      (wl-thread-insert-top)
      (message "Inserting thread...done."))
    (wl-summary-set-message-modified)
    (wl-summary-count-unread mark-alist)
    (wl-summary-update-modeline)    
    (goto-char (point-max))
    (forward-line -1)
    (set-buffer-modified-p nil)))
    
(defun wl-summary-next-folder-or-exit (&optional next-entity)
  (if (and next-entity
	   wl-auto-select-next)
      (let (retval)
	(wl-summary-toggle-disp-msg 'off)
	(unwind-protect
	    (setq retval
		  (wl-summary-goto-folder-subr next-entity 
					       'force-update
					       nil
					       nil ; not sticky
					       t   ; interactive!
					       ))
	  (wl-folder-set-current-entity-id (wl-folder-get-entity-id next-entity))
	  (if (and (eq retval 'more-next)
		   wl-summary-next-no-unread
		   (memq this-command wl-summary-next-no-unread-command))
	      (wl-summary-down t))))
    (wl-summary-exit)))

(defun wl-summary-set-message-modified ()
  (setq wl-summary-buffer-message-modified t))
(defun wl-summary-message-modified-p ()
  wl-summary-buffer-message-modified)
(defun wl-summary-set-mark-modified ()
  (setq wl-summary-buffer-mark-modified t))
(defun wl-summary-mark-modified-p ()
  wl-summary-buffer-mark-modified)

(defun wl-summary-msgdb-save ()
  "Save msgdb if modified."
  (when wl-summary-buffer-msgdb
    (save-excursion
      (let (path)
	(when (wl-summary-message-modified-p)
	  (setq path (elmo-msgdb-expand-path
		      wl-summary-buffer-folder-name))
	  (elmo-msgdb-overview-save 
	   path 
	   (elmo-msgdb-get-overview wl-summary-buffer-msgdb))
	  (elmo-msgdb-number-save 
	   path 
	   (elmo-msgdb-get-number-alist wl-summary-buffer-msgdb))
	  (elmo-folder-set-info-max-by-numdb
	   (wl-string wl-summary-buffer-folder-name)
	   (elmo-msgdb-get-number-alist
	    wl-summary-buffer-msgdb))
	  (setq wl-summary-buffer-message-modified nil))
	(when (wl-summary-mark-modified-p)
	  (or path 
	      (setq path (elmo-msgdb-expand-path
			  wl-summary-buffer-folder-name)))
	  (elmo-msgdb-mark-save 
	   path
	   (elmo-msgdb-get-mark-alist wl-summary-buffer-msgdb))
;; 	  (elmo-folder-set-info-hashtb
;; 	   (wl-string wl-summary-buffer-folder-name)
;; 	   nil nil
;; 	   0
;; 	   (+ wl-summary-buffer-new-count wl-summary-buffer-unread-count))
;;	  (setq wl-folder-info-alist-modified t)
	  (setq wl-summary-buffer-mark-modified nil))))))

(defsubst wl-summary-cleanup-temp-marks (&optional sticky)
  (if (or wl-summary-buffer-refile-list
	  wl-summary-buffer-copy-list
	  wl-summary-buffer-delete-list)
      (if (y-or-n-p "Marks remain to be executed. Execute them?")
	  (wl-summary-exec)
	;; delete temp-marks
	(message "")
	(wl-summary-delete-all-refile-marks)
	(wl-summary-delete-all-copy-marks)
	(wl-summary-delete-all-delete-marks)))
  (wl-summary-delete-all-temp-marks-on-buffer sticky))

;; a subroutine for wl-summary-exit/wl-save-status
(defun wl-summary-save-status (&optional sticky)
  ;; already in summary buffer.
  (wl-summary-cleanup-temp-marks sticky)
  (when wl-summary-buffer-persistent
    ;; save the current summary buffer view.
    (if (and wl-summary-cache-use 
	     (or (wl-summary-message-modified-p)
		 (wl-summary-mark-modified-p)))
	(wl-summary-save-view-cache sticky))
    ;; save msgdb ... 
    (wl-summary-msgdb-save)))

(defun wl-summary-force-exit ()
  "Exit current summary. Buffer is deleted even the buffer is sticky"
  (interactive)
  (wl-summary-exit 'force-exit))

(defun wl-summary-exit (&optional force-exit)
  "Exit current summary. if FORCE-EXIT, exits even the summary is sticky."
  (interactive "P")
  (let ((summary-buf (current-buffer))
	(sticky (wl-summary-sticky-p))
	summary-win
	message-buf message-win
	folder-buf folder-win)
    (unwind-protect
	;; save summary status
	(progn
	  (wl-summary-save-status sticky)
	  ;(wl-summary-msgdb-save)
	  )
      ;; for sticky summary
      (wl-delete-all-overlays)
      (setq wl-summary-buffer-disp-msg nil)
      ;; delete message window if displayed.
      (if (setq message-buf (get-buffer wl-message-buf-name))
	  (if (setq message-win (get-buffer-window message-buf))
	      (delete-window message-win)))
      (if (setq folder-buf (get-buffer wl-folder-buffer-name))
	  (if (setq folder-win (get-buffer-window folder-buf))
	      ;; folder win is already displayed.
	      (select-window folder-win)
	    ;; folder win is not displayed.
	    (switch-to-buffer folder-buf))
	;; currently no folder buffer
	(wl-folder)
	)
      (and wl-folder-move-cur-folder
	   wl-folder-buffer-cur-point
	   (goto-char wl-folder-buffer-cur-point))
      (setq wl-folder-buffer-cur-path nil)
      (setq wl-folder-buffer-cur-entity-id nil)
      (wl-delete-all-overlays)
      (if wl-summary-exit-next-move
	  (if wl-plugged
	      (wl-folder-next-unsync)
	    (wl-folder-next-entity))
	(beginning-of-line))
      (if (setq summary-win (get-buffer-window summary-buf))
	  (delete-window summary-win))
      (if (or force-exit 
	      (not sticky))
	  (kill-buffer summary-buf))
      (when (get-buffer wl-message-buf-name)
	(kill-buffer wl-message-buf-name))
      (run-hooks 'wl-summary-exit-hook))
    ))

(defun wl-summary-sync-force-update (&optional unset-cursor)
  (interactive)
  (let ((msgdb-dir (elmo-msgdb-expand-path wl-summary-buffer-folder-name))
	ret-val seen-list)
    (unwind-protect
	(progn
	  (setq seen-list (elmo-msgdb-seen-load msgdb-dir))
	  (setq ret-val (wl-summary-sync-update3 seen-list unset-cursor))
	  (elmo-msgdb-seen-save msgdb-dir nil))
      (set-buffer (current-buffer)))
    (if (interactive-p)
	(message "%s" ret-val))
    ret-val))

(defun wl-summary-sync (&optional unset-cursor force-range)
  (interactive)
  (let* ((folder wl-summary-buffer-folder-name)
	 (inhibit-read-only t)
	 (buffer-read-only nil)
	 (msgdb-dir (elmo-msgdb-expand-path
		     folder))
	 (range (or force-range (wl-summary-input-range folder)))
	 mes seen-list)
    (cond ((string= range "all")
	   ;; initialize buffer local databases.
	   (if (and (not wl-plugged) ; forbidden
		    (elmo-folder-contains-type folder 'imap4)) 
	       (error "Unplugged."))
	   (setq seen-list
		 (nconc
		  (elmo-msgdb-mark-alist-to-seen-list
		   (elmo-msgdb-get-number-alist
		    wl-summary-buffer-msgdb)
		   (elmo-msgdb-get-mark-alist 
		    wl-summary-buffer-msgdb)
		   (concat wl-summary-important-mark
			   wl-summary-read-uncached-mark))
		  (elmo-msgdb-seen-load msgdb-dir)))
	   (setq wl-summary-buffer-msgdb '(nil nil nil nil))
	   (setq wl-thread-entity-hashtb (elmo-make-hash))
	   (setq wl-thread-entity-list nil)
	   (setq wl-thread-entities nil)
	   (setq wl-summary-buffer-temp-mark-list nil)
	   (setq wl-summary-buffer-refile-list nil)
	   (setq wl-summary-buffer-copy-list nil)
	   (setq wl-summary-buffer-delete-list nil)
	   (setq mes (wl-summary-sync-update3 seen-list unset-cursor))
	   (elmo-msgdb-seen-save msgdb-dir nil) ; delete all seen.
	   (if mes (message "%s" mes)))
;	   (wl-summary-sync-all folder t))
	  ((string= range "rescan")
	   (let ((msg (wl-summary-message-number)))
	     (wl-summary-rescan)
	     (and msg (wl-summary-jump-to-msg msg))))
	  ((or (string-match "last:" range)
	       (string-match "first:" range))
	   (wl-summary-goto-folder-subr (concat "/" range "/" folder)
					'force-update nil nil t))
	  ((string= range "no-sync")
	   ;; do nothing.
	   )
	  (t 
	   (setq seen-list (elmo-msgdb-seen-load msgdb-dir))
	   (setq mes (wl-summary-sync-update3 seen-list unset-cursor))
	   (elmo-msgdb-seen-save msgdb-dir nil) ; delete all seen.
	   (if mes (message "%s" mes))))))

(defun wl-summary-edit-petname (&optional from-string)
  "Add/change/delete petname to/from wl-address-filename interactively."
  (interactive (if current-prefix-arg
		   (list (read-from-minibuffer "Email-address: "))))
  (let* ((entity (assoc (cdr  
			 (assq 
			  (wl-summary-message-number)
			  (elmo-msgdb-get-number-alist 
			   wl-summary-buffer-msgdb)))
			(elmo-msgdb-get-overview wl-summary-buffer-msgdb)))
	 (from-string (or from-string
			  (elmo-msgdb-overview-entity-get-from entity)))
	 name-in-from the-email)
    (when from-string
      (setq the-email (wl-address-header-extract-address from-string))
      (setq name-in-from (wl-address-header-extract-realname from-string))
      
      (if (elmo-get-hash-val the-email wl-address-petname-hash)
	  (if (y-or-n-p (format "Change %s in your petname entries?: " 
				the-email))
	      ;;;; Change Petname
	      (wl-address-petname-add-or-change 
	       the-email
	       (elmo-get-hash-val the-email wl-address-petname-hash)
	       (wl-address-header-extract-realname
		(cdr (assoc (downcase the-email) wl-address-completion-list))) t)
	      ;;;; Delete Petname
	    (if (y-or-n-p (format "Delete %s from your petname entries?: " the-email))
		(wl-address-petname-delete the-email)
	      (message nil)))
	;;;; Add Petname
	(if (y-or-n-p (format "Add %s to your petname entries?: " the-email))
	    (wl-address-petname-add-or-change 
	     the-email name-in-from name-in-from)
	  (message nil)))
      ;; update alias
      (wl-status-update)
      ;; i'd like to update summary-buffer, but...
      ;;(wl-summary-rescan)
      (run-hooks 'wl-summary-edit-petname-hook)
      )))

(defun wl-summary-incorporate ()
  (interactive)
  (wl-summary-sync-force-update)
  (wl-summary-prefetch-all-new)
  )

(defun wl-summary-prefetch-msg (number)
  "Returns status-mark."
  ;; prefetching procedure.
  (interactive)
  (save-excursion
    (let* ((msgdb wl-summary-buffer-msgdb)
	   (mark-alist (elmo-msgdb-get-mark-alist msgdb))	  
	   (number-alist (elmo-msgdb-get-number-alist msgdb))
	   (message-id (cdr (assq number number-alist)))
	   (ov (assoc message-id 
		      (elmo-msgdb-get-overview msgdb)))
	   (entity ov)
	   (size (elmo-msgdb-overview-entity-get-size ov))
	   (inhibit-read-only t)
	   (buffer-read-only nil)	   
	   (force-read (and size
			    (< size wl-fetch-confirm-threshold)))
	   mark new-mark)
      (unwind-protect
	  (progn
	    (if (and size (not force-read))
		(setq force-read
		      (y-or-n-p
		       (format
			"Message from %s has %d bytes. Prefetch it?" 
			(concat 
			 "[ "
			 (save-match-data 
			   (wl-set-string-width 
			    wl-from-width
			    (wl-summary-from-func-internal
			     (elmo-eword-decode-string
			      (elmo-delete-char 
			       ?\"
			       (or 
				(elmo-msgdb-overview-entity-get-from ov)
				"??")
			       ))))) " ]")
			size))))
	    (when force-read
	      (save-excursion
		(save-match-data
		  (elmo-prefetch-msg wl-summary-buffer-folder-name
				     number 
				     (wl-message-get-original-buffer)
				     msgdb 
				     )))
	      (setq mark (cadr (assq number mark-alist)))
	      (setq new-mark
		    (cond 
		     ((string= mark 
			       wl-summary-unread-uncached-mark)
		      wl-summary-unread-cached-mark)
		     ((string= mark wl-summary-new-mark)
		      (setq wl-summary-buffer-new-count 
			    (- wl-summary-buffer-new-count 1))
		      (setq wl-summary-buffer-unread-count
			    (+ wl-summary-buffer-unread-count 1))
		      wl-summary-unread-cached-mark)
		     ((string= mark wl-summary-read-uncached-mark)
		      nil)))
	      (setq mark-alist (elmo-msgdb-mark-set 
				mark-alist number new-mark)))
	    (setq new-mark (cadr (assq number mark-alist))))
	(elmo-msgdb-set-mark-alist msgdb mark-alist)
	(wl-summary-set-mark-modified)
	(wl-summary-update-modeline))
      (or new-mark " "))))

(defun wl-summary-prefetch-all-new (&optional arg)
  (interactive "P")
  (save-excursion
    (let* ((cur-buf (current-buffer))
	   (inhibit-read-only t)
	   (buffer-read-only nil)
	   (msgdb wl-summary-buffer-msgdb)
	   (number-alist (elmo-msgdb-get-number-alist msgdb))
	   (mark-alist (elmo-msgdb-get-mark-alist msgdb))	  
	   (case-fold-search nil)
	   uncached-msgs
	   msg start-point
	   (count 0))
      (setq start-point (point))
      (unwind-protect
	  (progn
	    (goto-char (point-min))
	    (while (re-search-forward 
		    (format "^ *\\([0-9]+\\)[^0-9]\\(%s\\)"
			    (regexp-quote wl-summary-new-mark))
		    nil t)
	      (setq uncached-msgs (append
				   uncached-msgs
				   (list 
				    (wl-match-buffer 1)))))
	    (goto-char (point-min))
	    (while (re-search-forward 
		    (format "^ *\\([0-9]+\\)[^0-9]\\(%s\\)"
			    (regexp-quote wl-summary-new-mark))
		    nil t)
	      (setq msg (string-to-int (wl-match-buffer 1)))
	      ;;(setq mark (wl-match-buffer 2))
	      (let* (;;(number-alist (elmo-msgdb-get-number-alist msgdb))
		     (message-id (cdr (assq msg number-alist)))
		     (ov (assoc message-id 
				(elmo-msgdb-get-overview msgdb)))
		     (size (elmo-msgdb-overview-entity-get-size ov))
		     (force-read (and size 
				      (< size wl-fetch-confirm-threshold))))
		(if (and size
			 (not force-read))
		    (let ((from 
			   (concat 
			    "[ "
			    (save-match-data 
			      (wl-set-string-width 
			       wl-from-width
			       (wl-summary-from-func-internal
				(elmo-eword-decode-string
				 (elmo-delete-char 
				  ?\"
				  (decode-mime-charset-string
				   (or 
				    (elmo-msgdb-overview-entity-get-from ov)
				    "??")
				   wl-mime-charset)))))) " ]")))
		      (setq 
		       force-read
		       (y-or-n-p
			(format
			 "Message from %s has %d bytes. Prefetch it?" 
			 from
			 size)))))
		(when force-read
		  (save-excursion
		    (save-match-data
		      (elmo-prefetch-msg wl-summary-buffer-folder-name
					 msg 
					 (wl-message-get-original-buffer)
					 msgdb 
					 )))
		  (delete-region (match-beginning 2) (match-end 2))
		  (insert wl-summary-unread-cached-mark)
		  (setq mark-alist
			(elmo-msgdb-mark-set 
			 mark-alist
			 msg 
			 ;(cdr (assq msg
			 ;number-alist)) 
			 wl-summary-unread-cached-mark))
		  (if wl-summary-highlight	
		      (wl-highlight-summary-current-line nil nil t))
		  (message "Prefetching... %d/%d message(s)" 
			   (setq count (+ 1 count))
			   (length uncached-msgs)
			   )))
	      (save-excursion
		;; redisplay
		(goto-char start-point)
		(sit-for 0))
	      ))
	;; hidden messages.
	(set-buffer cur-buf); why is this needed???
	(elmo-msgdb-set-mark-alist msgdb mark-alist)
	(wl-summary-set-mark-modified)
	(setq wl-summary-buffer-new-count 
	      (- wl-summary-buffer-new-count count))
	(setq wl-summary-buffer-unread-count
	      (+ wl-summary-buffer-unread-count count))
	(wl-summary-update-modeline)
	(message "Prefetched %d message(s)." count)))))

(defun wl-summary-prefetch-region (beg end)
  (interactive "r")
  (let ((count 0)
	(len 0)
	mark
	entity msg
	children start-pos)
    (save-excursion
      (setq start-pos (point))
      (save-restriction
	(narrow-to-region beg end)
	;; first, count all unread msgs in the region.
	(goto-char (point-min))
	(while (not (eobp))
	  (beginning-of-line)
	  (when (looking-at "^ *\\([0-9]+\\)[^0-9]\\([^0-9]\\)")
	    (setq mark (wl-match-buffer 2))
	    (if (or (string= mark wl-summary-new-mark)
		    (string= mark wl-summary-read-uncached-mark)
		    (string= mark wl-summary-unread-uncached-mark))
		(setq len (+ 1 len)))
	    (setq msg (string-to-int (wl-match-buffer 1)))
	    (setq entity (wl-thread-get-entity msg))
	    (if (or (not (eq wl-summary-buffer-view 'thread))
		    (wl-thread-entity-get-opened entity))
		(); opened. no hidden children.
	      ;; hidden children!!
	      (setq len (- (+ len
			      (length 
			       (wl-thread-get-children-msgs-uncached msg)))
			   1))));; delete myself
	  (forward-line 1))
	(goto-char (point-min))
	(while (not (eobp))
	  (beginning-of-line)
	  (when (looking-at "^ *\\([0-9]+\\)[^0-9]\\([^0-9]\\)")
	    (setq mark (wl-match-buffer 2))
	    (when (or (string= mark wl-summary-new-mark)
		      (string= mark wl-summary-read-uncached-mark)
		      (string= mark wl-summary-unread-uncached-mark))
	      (wl-summary-prefetch)
	      (message "Prefetching... %d/%d message(s)"
		       (setq count (+ 1 count)) len))
	    (setq msg (string-to-int (wl-match-buffer 1)))
	    (setq entity (wl-thread-get-entity msg))
	    (if (or (not (eq wl-summary-buffer-view 'thread))
		    (wl-thread-entity-get-opened entity))
		;; opened...no hidden children.
		()
	      ;; hidden children is here.
	      (setq children (wl-thread-get-children-msgs-uncached msg))
	      (while children
		(save-excursion (wl-summary-prefetch-msg (car children)))
		(message "Prefetching... %d/%d message(s)"
			 (setq count (+ 1 count)) len)
		(setq children (cdr children))))
	    (forward-line 1)
	    ;; redisplay!
	    (save-excursion
	      (save-restriction
		(widen)
		(goto-char start-pos)
		(sit-for 0)))
	    ))
	(message "Prefetched %d message(s)" count)))))

(defun wl-summary-prefetch ()
  "Prefetch current message."
  (interactive)
  (save-excursion
    (save-match-data
      (beginning-of-line)
      (when (looking-at "^ *\\([0-9]+\\)[^0-9]\\([^0-9]\\)")
	(goto-char (match-beginning 2))
	(let ((inhibit-read-only t)
	      (buffer-read-only nil)
	      mark)
	  (setq mark (save-excursion (wl-summary-prefetch-msg 
				      (string-to-int (wl-match-buffer 1)))))
	  (delete-region (match-beginning 2)
			 (match-end 2))
	  (insert mark)
	  (set-buffer-modified-p nil)
	  (if wl-summary-highlight
	      (wl-highlight-summary-current-line)))))))

(defun wl-summary-delete-all-status-marks-on-buffer ()
  (interactive)
  (save-excursion
    (goto-char (point-min))
    (let ((inhibit-read-only t)
	  (buffer-read-only nil)
	  (case-fold-search nil)
	  )
      (while (re-search-forward 
	      (concat "^" wl-summary-buffer-number-regexp ".\\(.\\)") nil t)
	(delete-region (match-beginning 1) (match-end 1))
	(insert " ")))))

(defun wl-summary-delete-copy-marks-on-buffer(cpys)
  (mapcar (function
	   (lambda (x)
	     (wl-summary-jump-to-msg x)
	     (wl-summary-unmark-line)))
	  cpys))

(defun wl-summary-delete-all-refile-marks ()
  (mapcar (function
	   (lambda (x)
	     (wl-summary-jump-to-msg (car x))
	     (wl-summary-unmark-line))) wl-summary-buffer-refile-list))

(defun wl-summary-delete-all-delete-marks ()
  (mapcar (function
	   (lambda (x)
	     (wl-summary-jump-to-msg x)
	     (wl-summary-unmark-line))) wl-summary-buffer-delete-list))

(defun wl-summary-delete-all-copy-marks ()
  (mapcar (function
	   (lambda (x)
	     (wl-summary-jump-to-msg (car x))
	     (wl-summary-unmark-line))) wl-summary-buffer-copy-list))

(defun wl-summary-delete-all-temp-marks-on-buffer (&optional sticky)
  ;; for summary view cache saving.
  (interactive)
  (save-excursion
    (goto-char (point-min))
    (let ((inhibit-read-only t)
	  (buffer-read-only nil)
	  (case-fold-search nil)
	  (regexp (concat "^" wl-summary-buffer-number-regexp "\\([^ ]\\)" ))
	  )
      (while (re-search-forward regexp nil t)
	(delete-region (match-beginning 1) (match-end 1))
	(insert " ")
	(if (and sticky
		 wl-summary-highlight)
	    (wl-highlight-summary-current-line))))))

(defun wl-summary-delete-all-marks (mark-alist mark)
  "Delete all MARKs in MARK-ALIST"
  (let ((malist mark-alist)
	(ret-val mark-alist)
	entity)
    (while malist
      (setq entity (car malist))
      (if (string= (cadr entity) mark)
	  ;; delete this entity
	  (setq ret-val (delete entity ret-val)))
      (setq malist (cdr malist)))
    ret-val))

;; Does not work correctly...
(defun wl-summary-mark-as-read-region (beg end)
  (interactive "r")
  (save-excursion
    (save-restriction
      (narrow-to-region beg end);(save-excursion (goto-char end)
					;    (end-of-line) (point)))
      (goto-char (point-min))
      (if (eq wl-summary-buffer-view 'thread)
	  (progn
	    (while (not (eobp))
	      (let* ((number (wl-summary-message-number))
		     (entity (wl-thread-get-entity number))
		     children)
		(if (wl-thread-entity-get-opened entity)
		    ;; opened...mark line.
		    ;; Crossposts are not processed
		    (wl-summary-mark-as-read t)
		  ;; closed
		  (wl-summary-mark-as-read t) ; mark itself.
		  (setq children (wl-thread-get-children-msgs number))
		  (while children 
		    (wl-thread-msg-mark-as-read (car children))
		    (setq children (cdr children))))
		(forward-line 1))))
	(while (not (eobp))
	  (wl-summary-mark-as-read t)
	  (forward-line 1)))))
  (wl-summary-count-unread (elmo-msgdb-get-mark-alist wl-summary-buffer-msgdb))
  (wl-summary-update-modeline))

(defun wl-summary-mark-as-read-all ()
  (interactive)
  (if (or (not (interactive-p))
	  (y-or-n-p "Mark all messages as read?"))
      (let* ((folder wl-summary-buffer-folder-name)
	     (cur-buf (current-buffer))
	     (msgdb wl-summary-buffer-msgdb)
	     ;;(number-alist (elmo-msgdb-get-number-alist msgdb))
	     (mark-alist (elmo-msgdb-get-mark-alist msgdb))
	     (malist mark-alist)
	     (inhibit-read-only t)
	     (buffer-read-only nil)
	     (case-fold-search nil)
	     msg mark
	     )
	(message "setting all msgs as read...")
	(elmo-mark-as-read folder (wl-summary-collect-unread mark-alist)
			   msgdb)
	(save-excursion
	  (goto-char (point-min))
	  (while (re-search-forward "^ *\\([0-9]+\\)[^0-9]\\([^0-9 ]\\)" nil t)
	    (setq msg (string-to-int (wl-match-buffer 1)))
	    (setq mark (wl-match-buffer 2))
	    (when (and (not (string= mark wl-summary-important-mark))
		       (not (string= mark wl-summary-read-uncached-mark)))
	      (delete-region (match-beginning 2) (match-end 2))
	      (if (or (not (elmo-use-cache-p folder msg))
		      (string= mark wl-summary-unread-cached-mark))
		  (progn
		    (insert " ")
		    (setq mark-alist
			  (elmo-msgdb-mark-set 
			   mark-alist
			   msg ;(cdr (assq msg number-alist)) 
			   nil)))
		;; New mark and unread-uncached mark
		(insert wl-summary-read-uncached-mark)
		(setq mark-alist
		      (elmo-msgdb-mark-set mark-alist
					   msg
					; (cdr (assq msg number-alist)) 
					   wl-summary-read-uncached-mark)))
	      (if wl-summary-highlight
		  (wl-highlight-summary-current-line)))
	    ))
	(setq mark-alist (wl-summary-set-as-read-mark-alist mark-alist))
	(wl-summary-set-mark-modified)
	(set-buffer cur-buf); why is this needed???
	(elmo-msgdb-set-mark-alist msgdb mark-alist)
	(wl-folder-update-unread wl-summary-buffer-folder-name 0)
	(setq wl-summary-buffer-unread-count 0)
	(setq wl-summary-buffer-new-count    0)	
	(wl-summary-update-modeline)
	(message "setting all msgs as read...done.")
	(set-buffer-modified-p nil)
	)))

(defun wl-summary-delete-cache ()
  "Delete cache of current message."
  (interactive)
  (save-excursion
    (let* ((inhibit-read-only t)
	   (buffer-read-only nil)
	   (folder wl-summary-buffer-folder-name)
	   (msgdb wl-summary-buffer-msgdb)
	   (mark-alist (elmo-msgdb-get-mark-alist msgdb))
	   (number-alist (elmo-msgdb-get-number-alist msgdb))
	   (case-fold-search nil)
	   mark number unread new-mark
	   )
;      (re-search-backward "^ *[0-9]+..[0-9]+/[0-9]+" nil t) ; set cursor line
      (beginning-of-line)
      (when (looking-at "^ *\\([0-9]+\\)[^0-9]\\([^0-9]\\)")
	(progn
	  (setq mark (wl-match-buffer 2))
	  (cond 
	   ((string= mark wl-summary-new-mark)
	    (string= mark wl-summary-unread-uncached-mark)
	    (string= mark wl-summary-important-mark)
	    ;; noop
	    )
	   ((string= mark wl-summary-unread-cached-mark)
	    (setq new-mark wl-summary-unread-uncached-mark))
	   (t 
	    (setq new-mark wl-summary-read-uncached-mark)))
	  (when new-mark
	    (setq number (string-to-int (wl-match-buffer 1)))
	    (delete-region (match-beginning 2) (match-end 2))
	    (goto-char (match-beginning 2))
	    (insert new-mark)
	    (elmo-cache-delete (cdr (assq number number-alist))
			       wl-summary-buffer-folder-name
			       number)
	    (setq mark-alist
		  (elmo-msgdb-mark-set mark-alist number new-mark))
	    (elmo-msgdb-set-mark-alist msgdb mark-alist)
	    (wl-summary-set-mark-modified)
	    (if wl-summary-highlight
		(wl-highlight-summary-current-line nil nil t))
	    (set-buffer-modified-p nil)))))))
  
(defun wl-summary-resume-cache-status ()
  "Resume the cache status of all messages in the current folder."
  (interactive)
  (let* ((folder wl-summary-buffer-folder-name)
	 (cur-buf (current-buffer))
	 (msgdb wl-summary-buffer-msgdb)
	 (number-alist (elmo-msgdb-get-number-alist msgdb))
	 (mark-alist (elmo-msgdb-get-mark-alist msgdb))
	 (inhibit-read-only t)
	 (buffer-read-only nil)
	 (case-fold-search nil)
	 msg mark msgid set-mark
	 )
    (message "Resuming cache status...")
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward "^ *\\([0-9]+\\)[^0-9]\\([^0-9]\\)" nil t)
	(setq msg (string-to-int
		   (wl-match-buffer 1)))
	(setq mark (wl-match-buffer 2))
	(setq msgid (cdr (assq msg number-alist)))
	(setq set-mark nil)
	(if (elmo-cache-exists-p msgid folder msg)
	    (if (or
		 (string= mark wl-summary-unread-uncached-mark) ; U -> !
		 (string= mark wl-summary-new-mark)             ; N -> !
		 )
		(setq set-mark wl-summary-unread-cached-mark)
	      (if (string= mark wl-summary-read-uncached-mark)  ; u -> ' '
		  (setq set-mark " ")))
	  (if (string= mark " ")                            
	      (setq set-mark wl-summary-read-uncached-mark)     ;' ' -> u
	    (if (string= mark wl-summary-unread-cached-mark) 
		(setq set-mark wl-summary-unread-uncached-mark) ; !  -> U
	      )))
	(when set-mark
	  (delete-region (match-beginning 2) (match-end 2))
	  (insert set-mark)
	  (setq mark-alist
		(elmo-msgdb-mark-set 
		 mark-alist msg ; msgid 
		 (if (string= set-mark " ") nil set-mark)))
	  (if wl-summary-highlight
	      (wl-highlight-summary-current-line))))
      (wl-summary-set-mark-modified)
      (set-buffer cur-buf); why is this needed???
      (elmo-msgdb-set-mark-alist msgdb mark-alist)
      (wl-summary-count-unread mark-alist)
      (wl-summary-update-modeline)
      (message "Resuming cache status...done.")
      (set-buffer-modified-p nil))))

(defun wl-summary-resume-marks-and-highlight ()
  (let* ((msgdb wl-summary-buffer-msgdb)
	 (mark-alist (elmo-msgdb-get-mark-alist msgdb))
	 ;;(number-alist (elmo-msgdb-get-number-alist msgdb))
	 (count (count-lines (point-min)(point-max)))
	 (i 0)
	 msg-num percent smark)
    (save-excursion
      (goto-char (point-min))
      (message "resuming all marks...")
      (while (not (eobp))
	(setq msg-num (wl-summary-message-number))
	(setq smark (car (cdr (assq msg-num mark-alist))))
	(if (looking-at (format "^ *%s \\( \\)" msg-num))
	    (progn
	      (goto-char (match-end 1))
	      (delete-region (match-beginning 1) (match-end 1))
	      (insert (or smark " "))))
	(wl-highlight-summary-current-line smark)
	(setq i (+ i 1))
	(setq percent (/ (* i 100) count))
	(message "resuming all marks...%d%%" percent)
	(forward-line 1)))
    (message "resuming all marks...done.")))

(defun wl-summary-resume-marks ()
  (let* ((msgdb wl-summary-buffer-msgdb)
	 (mark-alist (elmo-msgdb-get-mark-alist msgdb))
	 (number-alist (elmo-msgdb-get-number-alist msgdb))
	 (count (length mark-alist))
	 (i 0)
	 entity msg-num percent)
    (save-excursion
      (message "resuming all marks...")
      (while mark-alist
	(setq entity (car mark-alist))
	(if (setq msg-num (car (rassoc (car entity) number-alist)))
	    (progn ;(goto-char (point-min))
	      (if (re-search-forward (format "^ *%s \\( \\)" msg-num) nil t)
		  (progn
		    (delete-region (match-beginning 1) (match-end 1))
		    (insert (or (cadr entity)
				" ")))
		(if (re-search-backward (format "^ *%s \\( \\)" msg-num) nil t)
		    (progn
		      (goto-char (match-end 1))
		      (delete-region (match-beginning 1) (match-end 1))
		      (insert (or (cadr entity)
				  " ")))))))
	(setq i (+ i 1))
	(setq percent (/ (* i 100) count))
	(message "resuming all marks...%d%%" percent)
	(setq mark-alist (cdr mark-alist)))
      (message "resuming all marks...done."))))

(defun wl-summary-delete-messages-on-buffer (msgs &optional deleting-info)
  (interactive)
  (save-excursion
    (let ((inhibit-read-only t)
	  (buffer-read-only nil)
	  (msgs2 msgs)
	  (len (length msgs))
	  (i 0)
	  update-list)
      (while msgs
	(if (eq wl-summary-buffer-view 'thread)
	    (progn
	      (wl-append update-list
			 (wl-thread-delete-message (car msgs)))
	      (setq update-list (delete (car msgs) update-list))
	      )
	  (goto-char (point-min))
	  (if (re-search-forward (format "^ *%d[^0-9]\\([^0-9]\\).*$" 
					 (car msgs)) nil t)
	      (progn
		(delete-region (match-beginning 0) (match-end 0))
		(delete-char 1) ; delete '\n'
		)))
	(when deleting-info
	  (setq i (1+ i))
	  (and (zerop (% i 10))
	       (message "Deleting...%d%%" (/ (* i 100) len))))
	(setq msgs (cdr msgs)))
      (if (eq wl-summary-buffer-view 'thread)
	  (wl-thread-update-line-msgs (wl-uniq-list update-list)))
      (wl-thread-cleanup-symbols msgs2)
      (wl-summary-count-unread 
       (elmo-msgdb-get-mark-alist wl-summary-buffer-msgdb))
      (wl-summary-update-modeline)	    
      (wl-folder-update-unread
       wl-summary-buffer-folder-name
       (+ wl-summary-buffer-unread-count wl-summary-buffer-new-count)))))

(defun wl-summary-set-as-read-mark-alist (mark-alist)
  (let ((marks (list (cons wl-summary-unread-cached-mark 
			   nil)
		     (cons wl-summary-unread-uncached-mark 
			   wl-summary-read-uncached-mark)
		     (cons wl-summary-new-mark
			   wl-summary-read-uncached-mark)))
	(ret-val mark-alist)
	entity pair)
    (while mark-alist
      (setq entity (car mark-alist))
      (when (setq pair (assoc (cadr entity) marks))
	(if (elmo-use-cache-p wl-summary-buffer-folder-name
			      (caar mark-alist))
	    (if (cdr pair)
		(setcar (cdr entity) (cdr pair))
	      (setq ret-val (delete entity ret-val)))
	  (setq ret-val (delete entity ret-val))))
      (setq mark-alist (cdr mark-alist)))
    ret-val))

(defun wl-summary-set-status-marks (mark-alist before after)
  "Set the BEFORE marks to AFTER"
  (let ((ret-val mark-alist)
	entity)
    (while mark-alist
      (setq entity (car mark-alist))
      (when (string= (cadr entity) before)
	(if after
	    (setcar (cdr entity) after)
	  (setq ret-val (delete entity ret-val))))
      (setq mark-alist (cdr mark-alist)))
    ret-val
    ))

(defun wl-summary-set-status-marks-on-buffer (before after)
  "Set the MARKS marks on buffer"
  (interactive)
  (save-excursion
    (goto-char (point-min))
    (let ((inhibit-read-only t)
	  (buffer-read-only nil)
	  (regexp (concat "^" wl-summary-buffer-number-regexp ".\\(\\%s\\)"))
	  )
      (while (re-search-forward 
	      (format regexp (regexp-quote before)) nil t)
	(delete-region (match-beginning 1) (match-end 1))
	(insert after)
	(if wl-summary-highlight
	    (wl-highlight-summary-current-line))
	))))

(defun wl-summary-get-delete-folder (folder)
  (if (string= folder wl-trash-folder)
      'null
    (let* ((type (or (wl-get-assoc-list-value wl-delete-folder-alist folder)
		     'trash)))
      (cond ((stringp type)
	     type)
	    ((or (equal type 'remove) (equal type 'null))
	     'null)
	    (t;; (equal type 'trash)
	     wl-trash-folder)))))

(defun wl-summary-delete-important-msgs-from-list (delete-list 
						   mark-alist)
  (let ((dlist delete-list))
    (while dlist
      (if (string= wl-summary-important-mark 
		   (car (cdr (assq (car dlist) mark-alist))))
	  (setq delete-list (delete (car dlist) delete-list)))
      (setq dlist (cdr dlist)))
    delete-list))
  
(defun wl-summary-get-append-message-func ()
  (if (eq wl-summary-buffer-view 'thread)
      'wl-summary-insert-thread-entity
;      'wl-summary-insert-thread
    'wl-summary-insert-summary
    ))

(defun wl-summary-sort ()
  (interactive)
  (let ((sort-by (let ((input-range-list '("number" "date" "subject" "from"))
		       (default "date")
		       in)
		   (setq in
			 (completing-read 
			  (format "Sort by (%s): " default)
			  (mapcar 
			   (function (lambda (x) (cons x x)))
			   input-range-list)))
		   (if (string= in "")
		       default
		     in))))
    (if (not (member sort-by '("number" "date" "subject" "from")))
	(error "Sort by %s is not implemented."  sort-by))
    (wl-summary-rescan sort-by)
    ))



(defun wl-summary-sync-update3 (&optional seen-list unset-cursor)
  "Update the summary view."
  (interactive)
  (let* ((folder wl-summary-buffer-folder-name)
	 (cur-buf (current-buffer))
	 (msgdb wl-summary-buffer-msgdb)
	 (number-alist (elmo-msgdb-get-number-alist msgdb))
	 (mark-alist (elmo-msgdb-get-mark-alist msgdb))
	 (overview (elmo-msgdb-get-overview msgdb))
	 ;;(location (elmo-msgdb-get-location msgdb))
	 (case-fold-search nil)
	 (elmo-mime-charset wl-summary-buffer-mime-charset)
	 (inhibit-read-only t)
	 (buffer-read-only nil)
	 diff append-list delete-list
	 i percent num result
	 gc-message
	 in-folder
	 in-db curp
	 overview-append
	 entity ret-val crossed crossed2 sync-all
	 top-num update-top-list
	 )
    (fset 'wl-summary-append-message-func-internal 
	  (wl-summary-get-append-message-func))
    ;; Flush pending append operations (disconnected operation).
    (wl-summary-flush-pending-append-operations)
    (goto-char (point-max))
    (message "Checking folder diff...")
    (setq in-folder (elmo-list-folder folder))
;    (setq in-db (mapcar 'int-to-string
;			(sort 
;			 (mapcar 'string-to-int 
;				 (mapcar 'car number-alist)) '<)))
    (setq in-db (sort (mapcar 'car number-alist) '<))
    (when (or (eq msgdb nil) ; trick for unplugged...
	      (equal msgdb '(nil nil nil nil)))
      (setq sync-all t)
      (erase-buffer))
    (setq diff (if (eq (elmo-folder-get-type folder) 'multi)
		   (elmo-multi-list-bigger-diff in-folder in-db)
		 (elmo-list-bigger-diff in-folder in-db)))
    (setq append-list (car diff))
    (setq delete-list (cadr diff))
    (message "Checking folder diff...done.")
    ;; don't delete important-marked msgs.
    (setq delete-list
	  (wl-summary-delete-important-msgs-from-list delete-list 
						      mark-alist))
    (if (or (equal diff '(nil nil))
	    (equal diff '(nil)))
	(progn
	  (setq ret-val (format "No update is needed for %s" folder))
	  (wl-folder-set-folder-updated folder 
					(list 0 
					      (wl-summary-count-unread 
					       mark-alist folder)
					      (length in-folder))))
      (when delete-list
	(message "Deleting...")
	(elmo-msgdb-delete-msgs folder delete-list msgdb t) ; reserve cache.
	(set-buffer cur-buf)
	(wl-summary-delete-messages-on-buffer delete-list t)
	(message "Deleting...done."))
      (set-buffer cur-buf)
      (message "Updating message status marks...")
      ;; Change "New" marks to "Uncached Unread" marks.
      (wl-summary-set-status-marks mark-alist 
				   wl-summary-new-mark 
				   wl-summary-unread-uncached-mark)
      (wl-summary-set-status-marks-on-buffer 
       wl-summary-new-mark 
       wl-summary-unread-uncached-mark)
      (message "Updating message status marks...done.")
      (message "Updating thread...")
      (setq num (length append-list))
      (if append-list
	  (progn
	    (setq i 0)
	    (setq result (elmo-msgdb-create ;-range
			  folder 
			  append-list
			  wl-summary-new-mark
			  wl-summary-unread-cached-mark ; !
			  wl-summary-read-uncached-mark ; u ;; XXXX
			  wl-summary-important-mark
			  seen-list
			  ))
	    ;; delete duplicated messages.
	    (when (elmo-folder-contains-multi folder)
	      (setq crossed (elmo-multi-delete-crossposts
			     msgdb result))
	      (setq result (cdr crossed))
	      (setq crossed (car crossed))
	      )
	    (setq overview-append (car result))
	    (setq msgdb (elmo-msgdb-append msgdb result))
	    ;; set these value for append-message-func
	    (setq overview (elmo-msgdb-get-overview msgdb))
	    (setq number-alist (elmo-msgdb-get-number-alist msgdb))
	    (setq mark-alist (elmo-msgdb-get-mark-alist msgdb))
	    ;; (setq location (elmo-msgdb-get-location msgdb))
	    (setq curp overview-append)
	    (setq num (length curp))
	    (while curp
	      (setq entity (car curp))
	      (setq top-num
		    (wl-summary-append-message-func-internal 
		     entity overview mark-alist 
		     (not sync-all)
		     ))
	      (when top-num
		(wl-append update-top-list (list top-num)))
	      (if elmo-use-database
		  (elmo-database-msgid-put 
		   (car entity) folder
		   (elmo-msgdb-overview-entity-get-number entity)))
	      (setq curp (cdr curp))
	      (setq i (+ i 1))
	      (setq percent (/ (* i 100) num))
	      (message "Updating thread...%d%%" percent)
	      )
	    (setq update-top-list
		  (wl-uniq-list update-top-list))
	    (when (and (eq wl-summary-buffer-view 'thread)
		       update-top-list )
	      (message "Updating indent...")
	      (wl-thread-update-indent-string-thread update-top-list)
	      (message "Updating indent...done."))
	    (message "Updating thread...done.")
	    (set-buffer cur-buf)
	    ))
      ;; 
      (wl-summary-set-message-modified)
      (wl-summary-set-mark-modified)
      (setq wl-summary-buffer-msgdb msgdb)
      (when (and sync-all (eq wl-summary-buffer-view 'thread))
	(message "Inserting thread...")
	(setq wl-thread-entity-cur 0)
	(wl-thread-insert-top)
	(message "Inserting thread...done."))
      (if elmo-use-database
	  (elmo-database-close))
      (set-buffer cur-buf)
      ;; Update Folder mode
      (wl-folder-set-folder-updated folder (list 0 
						 (wl-summary-count-unread 
						  mark-alist folder)
						 (length in-folder)))
      (run-hooks 'wl-summary-sync-updated-hook)
      (setq ret-val (format "Updated (-%d/+%d) message(s)" 
			    (length delete-list) num))
      (set-buffer cur-buf))
    (wl-summary-count-unread mark-alist)
    (wl-summary-update-modeline)
;;     (wl-folder-update-unread 
;;      wl-summary-buffer-folder-name
;;      (+ wl-summary-buffer-unread-count 
;; 	wl-summary-buffer-new-count))
    (setq crossed2 (wl-summary-update-crosspost))
    (if (or crossed crossed2)
	(let ((crosses (+ (or crossed 0)
			  (or crossed2 0))))
	  (setq ret-val
		(if ret-val
		    (concat ret-val
			    (format " (%d crosspost)" crosses))
		  (format "%d crosspost message(s)" crosses))))
      (and ret-val
	   (setq ret-val (concat ret-val "."))))
    (unless unset-cursor
      (goto-char (point-min))
      (if (not (wl-summary-cursor-down t))
	  (progn
	    (goto-char (point-max))
	    (forward-line -1))
	(if (not (get-text-property (point) 'face))
	    (save-excursion
	      (unless wl-summary-partial-highlight-above-lines
		(goto-char (point-max)))
	      (forward-line (- 0 
			       (or
				wl-summary-partial-highlight-above-lines
				wl-summary-highlight-partial-threshold)))
	      (wl-highlight-summary (point) (point-max))))))
    (wl-delete-all-overlays)
    (set-buffer-modified-p nil)
    ret-val))



(defun wl-summary-update-modeline ()
  (setq wl-summary-buffer-unread-status 
	(format " {%s}(%d new/%d unread)"
		(if (eq wl-summary-buffer-view 'thread)
		    "T" "S")
		wl-summary-buffer-new-count
		(+ wl-summary-buffer-new-count
		   wl-summary-buffer-unread-count))))

(defsubst wl-summary-jump-to-msg (&optional number)
  (interactive)
  (let ((num (or number 
		 (string-to-int 
		  (read-from-minibuffer "Jump to Message(No.): ")))))
    (setq num (int-to-string num))
    (if (re-search-forward (concat "^[ \t]*" num "[^0-9]") nil t)
	(progn
	  (beginning-of-line)
	  t)
      (if (re-search-backward (concat "^[ \t]*" num "[^0-9]") nil t)
	  (progn
	    (beginning-of-line)
	    t)
	nil))))

(defun wl-summary-highlight-msgs (msgs)
  (save-excursion
    (let ((len (length msgs))
	  i)
      (message "Hilighting...")
      (setq i 0)
      (while msgs
	(setq i (+ i 1))
	(message "Highlighting...%d%%" (/ (* i 100) len))
	(if (wl-summary-jump-to-msg (car msgs))
	    (wl-highlight-summary-current-line))
	(setq msgs (cdr msgs)))
      (message "Highlighting...done."))))

(defun wl-summary-message-number ()
  (save-excursion
    (beginning-of-line)
    (if (looking-at "^ *\\([0-9]+\\)")
	(string-to-int (wl-match-buffer 1))
      nil)))

(defun wl-summary-move (src dsts-msgs)
  (let* ((dsts (car dsts-msgs))		; (+foo +bar)
;;	 (msgs (cdr dsts-msgs))		; (1 2 3)
;;	 (msgdb wl-summary-buffer-msgdb)
;;	 result)
	 )
    (while dsts
      (setq dsts (cdr dsts)))
    ))

(defun wl-summary-flush-pending-append-operations ()
  "Execute append operations that are done while offline status."
  (when (and wl-plugged
	     elmo-enable-disconnected-operation)
    (let* ((append-list (elmo-dop-append-list-load 
			 wl-summary-buffer-folder-name))
	   (appends append-list)
	   (number-alist (elmo-msgdb-get-number-alist wl-summary-buffer-msgdb))
	   dels pair)
      (when append-list
	(while appends
	  (if (setq pair (rassoc (car appends) number-alist))
	      (setq dels (append dels (list (car pair)))))
	  (setq appends (cdr appends)))
	(message "Resuming summary status...")
	(elmo-msgdb-delete-msgs wl-summary-buffer-folder-name
				dels wl-summary-buffer-msgdb t)
	(wl-summary-delete-messages-on-buffer dels)
	(message "Resuming summary status...done.")
	(elmo-dop-flush-pending-append-operations 
	 wl-summary-buffer-folder-name append-list)))))

(defun wl-summary-delete-all-msgs ()
  (interactive)
  (let ((cur-buf (current-buffer))
	(dels (elmo-list-folder wl-summary-buffer-folder-name)))
    (set-buffer cur-buf)
    (if (null dels)
	(message "No message to delete.")
      (if (y-or-n-p (format "%s has %d message(s). Delete all?"
			    wl-summary-buffer-folder-name
			    (length dels)
			    ))
	  (progn
	    (message "Deleting...")
	    (elmo-delete-msgs wl-summary-buffer-folder-name dels
			      wl-summary-buffer-msgdb)
	    (elmo-msgdb-delete-msgs wl-summary-buffer-folder-name
				    dels wl-summary-buffer-msgdb)
	    ;;(elmo-msgdb-save wl-summary-buffer-folder-name nil)
	    (wl-summary-set-message-modified)
	    (wl-summary-set-mark-modified)
	    (wl-folder-set-folder-updated wl-summary-buffer-folder-name
					  (list 0 0 0))
	    ;; for thread.
	    ;; (setq wl-thread-top-entity '(nil t nil nil))
	    (setq wl-summary-buffer-unread-count 0)
	    (setq wl-summary-buffer-new-count    0)
	    (wl-summary-update-modeline)
	    (set-buffer cur-buf)
	    (let ((inhibit-read-only t)
		  (buffer-read-only nil))
	      (erase-buffer))
	    ;;	  (if wl-summary-cache-use (wl-summary-save-view-cache))      
	    (message "Deleting...done.")
	    t
	    )
	nil))))

(defun wl-summary-toggle-thread (&optional arg)
  "Toggle thread status (T)hread and (S)equencial."
  (interactive "P")
  (when (or arg
	    (y-or-n-p (format "Toggle threading? (y=%s): " 
			      (if (eq wl-summary-buffer-view 'thread)
				  "\"off\"" "\"on\""))))
    (if (eq wl-summary-buffer-view 'thread)
	(setq wl-summary-buffer-view 'sequence)
      (setq wl-summary-buffer-view 'thread))
    (wl-summary-update-modeline)
    (force-mode-line-update)
    (wl-summary-rescan)))

(defun wl-summary-load-file-object (filename)
  "Load lisp object from dir."
  (save-excursion
    (let ((tmp-buffer (get-buffer-create " *wl-summary-load-file-object*"))
	  insert-file-contents-pre-hook   ; To avoid autoconv-xmas...
	  insert-file-contents-post-hook 
	  ret-val)
      (if (not (file-readable-p filename))
	  ()
	(set-buffer tmp-buffer)
	(as-binary-input-file (insert-file-contents filename))
	(setq ret-val
	      (condition-case nil
		  (read (current-buffer)) 
		(error (error "reading failed")))))
      (kill-buffer tmp-buffer)
      ret-val)))

(defun wl-summary-goto-folder (&optional arg)
  (interactive "P")
  (wl-summary-goto-folder-subr nil nil nil arg t))

(defun wl-summary-goto-last-visited-folder ()
  (interactive)
  (wl-summary-goto-folder-subr wl-summary-last-visited-folder nil nil nil t))

(defun wl-summary-sticky-p (&optional fld)
  (if fld
      (get-buffer (wl-summary-sticky-buffer-name fld))
    (not (string= wl-summary-buffer-name (buffer-name)))))

(defun wl-summary-stick ()
  "Make current summary buffer sticky."
  (interactive)
  (if (wl-summary-sticky-p)
      (message "Current summary buffer is already sticky.")
    (when (y-or-n-p "Stick current summary buffer?")
      (wl-summary-switch-to-clone-buffer 
       (wl-summary-sticky-buffer-name 
	wl-summary-buffer-folder-name)))
;;; ???hang up
;      (rename-buffer (wl-summary-sticky-buffer-name 
;		      wl-summary-buffer-folder-name)))
    (message "")))

(defun wl-summary-switch-to-clone-buffer (buffer-name)
  (let ((cur-buf (current-buffer))
	(msg (wl-summary-message-number))
	(buf (get-buffer-create buffer-name))
	(folder wl-summary-buffer-folder-name)
	(view wl-summary-buffer-view)
	(refile-list wl-summary-buffer-refile-list)
	(delete-list wl-summary-buffer-delete-list)
	(copy-list wl-summary-buffer-copy-list)
	(hashtb wl-thread-entity-hashtb)
	(entities wl-thread-entities)
	(entity-list wl-thread-entity-list)
	(msgdb wl-summary-buffer-msgdb)
	(message-modified wl-summary-buffer-message-modified)
	(mark-modified wl-summary-buffer-mark-modified))
    (set-buffer buf)
    (wl-summary-mode folder)
    (setq wl-summary-buffer-msgdb msgdb)
    (let ((buffer-read-only nil))
      (insert-buffer cur-buf))
    (set-buffer-modified-p nil)
    (when (eq wl-summary-buffer-view 'thread)
      (setq wl-thread-entity-hashtb hashtb)
      (setq wl-thread-entities entities)
      (setq wl-thread-entity-list entity-list))
    (setq wl-summary-buffer-refile-list refile-list)
    (setq wl-summary-buffer-delete-list delete-list)
    (setq wl-summary-buffer-copy-list copy-list)
    (setq wl-summary-buffer-view view)
    (setq wl-summary-buffer-message-modified message-modified)
    (setq wl-summary-buffer-mark-modified mark-modified)
    (switch-to-buffer buf)
    (kill-buffer cur-buf)
    (setq mode-line-buffer-identification
	  (format "Wanderlust: %s" 
		  (if wl-use-folder-petname-on-modeline
		      (wl-folder-get-petname folder)
		    folder)))
    (wl-summary-count-unread 
     (elmo-msgdb-get-mark-alist wl-summary-buffer-msgdb))
    (wl-summary-update-modeline)
    (if msg
	(if (eq wl-summary-buffer-view 'thread)
	    (wl-thread-jump-to-msg msg)
	  (wl-summary-jump-to-msg msg))
      (goto-char (point-max))
      (beginning-of-line))))

(defun wl-summary-get-buffer (folder)
  (or (and folder
	   (get-buffer (wl-summary-sticky-buffer-name folder)))
      (get-buffer wl-summary-buffer-name)))

(defun wl-summary-get-buffer-create (folder &optional force-sticky)
  (if force-sticky
      (get-buffer-create 
       (wl-summary-sticky-buffer-name folder))
    (or (get-buffer (wl-summary-sticky-buffer-name folder))
	(get-buffer-create wl-summary-buffer-name))))

(defun wl-summary-goto-folder-subr (&optional folder scan-type other-window 
					      sticky interactive)
  "Display target folder on summary"
  (interactive)
  (let* ((keep-cursor (memq this-command
			    wl-summary-keep-cursor-command))
	 (fld (or folder (wl-summary-read-folder wl-default-folder)))
	 (cur-fld wl-summary-buffer-folder-name)
	 (retval nil)
	 buf mes hilit reuse-buf
	 entity)
    (if (string= fld "")
	(setq fld wl-default-folder))
    (when (and (not (string= cur-fld fld)) ; folder is moved.
	       (eq major-mode 'wl-summary-mode)) ; called in summary.
      (setq wl-summary-last-visited-folder wl-summary-buffer-folder-name)
      (wl-summary-save-status 'keep)) ;; keep current buffer, anyway.
    (setq buf (wl-summary-get-buffer-create fld sticky))
    (setq reuse-buf
	  (save-excursion
	    (set-buffer buf)
	    (string= fld wl-summary-buffer-folder-name)))
    (unwind-protect
	(if reuse-buf
	    (if interactive
		(switch-to-buffer buf)
	      (set-buffer buf))
	  (if other-window
	      (delete-other-windows))
	  (set-buffer buf)
	  (wl-summary-mode fld)
	  (let ((case-fold-search nil)
		(inhibit-read-only t)
		(buffer-read-only nil))
	    (erase-buffer)
	    (setq mode-line-buffer-identification
		  (format "Wanderlust: %s" 
			  (if wl-use-folder-petname-on-modeline
			      (wl-folder-get-petname fld)
			    fld)))
	      ;; resume summary cache
	    (if wl-summary-cache-use
		(let* ((dir (elmo-msgdb-expand-path fld))
		       (cache (expand-file-name wl-summary-cache-file dir))
		       (view (expand-file-name wl-summary-view-file dir)))
		  (when (file-exists-p cache)
		    (as-binary-input-file
		     (insert-file-contents cache))
		    (elmo-set-buffer-multibyte default-enable-multibyte-characters)
		    (decode-mime-charset-region (point-min)(point-max)
						wl-summary-buffer-mime-charset))
		  (when (file-exists-p view)
		    (setq wl-summary-buffer-view 
			  (wl-summary-load-file-object view)))
		  (if (eq wl-summary-buffer-view 'thread)
		      (wl-thread-resume-entity fld)
		    )))
	    (if (and (not folder)
		     (setq entity
			   (wl-folder-search-entity-by-name fld
							    wl-folder-entity
							    'folder)))
		(save-excursion
		  ;; entity-id is unknown.
		  (wl-folder-set-current-entity-id
		   (wl-folder-get-entity-id entity))))
	    ;; Load msgdb
	    (setq wl-summary-buffer-msgdb nil) ; new msgdb
	    (setq wl-summary-buffer-msgdb 
		  (wl-summary-msgdb-load-async fld))
	    (if (null wl-summary-buffer-msgdb)
		(setq wl-summary-buffer-msgdb 
		      (elmo-msgdb-load (wl-string fld))))
	    (wl-summary-count-unread 
	     (elmo-msgdb-get-mark-alist wl-summary-buffer-msgdb))
	    (wl-summary-update-modeline)))
      (unless (and reuse-buf keep-cursor)
	(setq hilit wl-summary-highlight)
	(unwind-protect
	    (let ((wl-summary-highlight (if reuse-buf wl-summary-highlight)))
	      (if (and (not scan-type)
		       interactive
		       (not wl-ask-range))
		  (setq scan-type (wl-summary-get-sync-range fld)))
	      (cond 
	       ((eq scan-type nil)
		(wl-summary-sync 'unset-cursor))
	       ((eq scan-type 'all)
		(wl-summary-sync 'unset-cursor "all"))
	       ((eq scan-type 'no-sync))
	       ((or (eq scan-type 'force-update)
		    (eq scan-type 'update))
		(setq mes (wl-summary-sync-force-update 'unset-cursor)))))
	  (if interactive
	      (switch-to-buffer buf)
	    (set-buffer buf))
	  (run-hooks 'wl-summary-prepared-pre-hook)
	  (goto-char (point-min))
	  (if (wl-summary-cursor-down t)
	      (if (and wl-auto-select-first
		       (wl-thread-next-mark-p
			(wl-thread-entity-get-mark 
			 (wl-summary-message-number))
			wl-summary-move-order))
		  (setq retval 'disp-msg))
	    (goto-char (point-max))
	    (forward-line -1)
	    (setq retval 'more-next))
	  (setq wl-summary-highlight hilit)
	  (if (and wl-summary-highlight
		   (not reuse-buf))
	      (if (and wl-summary-highlight-partial-threshold
		       (> (count-lines (point-min) (point-max))
			  wl-summary-highlight-partial-threshold))
		  (save-excursion
		    (unless wl-summary-partial-highlight-above-lines
		      (goto-char (point-max)))
		    (forward-line (- 
				   0 
				   (or 
				    wl-summary-partial-highlight-above-lines
				    wl-summary-highlight-partial-threshold)))
		    (wl-highlight-summary (point) (point-max)))
		(wl-highlight-summary (point-min) (point-max))))
	  (if (null wl-summary-buffer-msgdb) ;; one more try.
	      (setq wl-summary-buffer-msgdb 
		    (elmo-msgdb-load (wl-string fld))))
	  (if (eq retval 'disp-msg)
	      (wl-summary-redisplay))
	  (if mes (message "%s" mes))
	  (if (and interactive wl-summary-recenter)
	      (recenter (/ (- (window-height) 2) 2)))
	  )))
    (run-hooks 'wl-summary-prepared-hook)
    (set-buffer-modified-p nil)
    retval))

(defun wl-summary-summary-line-already-exists-p (parent-number buffer)
  "returns the depth."
  (set-buffer buffer)
  (goto-char (point-max))
  (let ((depth 0))
    (when (re-search-backward (format "^ *%s..../..\(.*\)..:.. "
				      parent-number) nil t)
      (goto-char (match-end 0))
      (while (string-match wl-thread-indent-regexp 
			   (char-to-string
			    (char-after (point))))	  
	(setq depth (+ 1 depth))
	(forward-char))
      (/ depth wl-thread-indent-level-internal))))

(defun wl-summary-goto-bottom-of-current-thread ()
  (if (re-search-forward (concat "^" wl-summary-buffer-number-regexp 
				 "..../..\(.*\)..:.. \\[") nil t)
      ()
    (goto-char (point-max))))

(defun wl-summary-goto-top-of-current-thread ()
  (if (re-search-backward (concat "^" wl-summary-buffer-number-regexp 
				  "..../..\(.*\)..:.. \\[") nil t)
      ()
    (goto-char (point-min))))

(defun wl-summary-goto-bottom-of-sub-thread (&optional depth)
  (interactive)
  (let ((depth (or depth 
		   (wl-thread-get-depth-of-current-line))))
    (forward-line 1)
    (while (and (not (eobp))
		(>= (wl-thread-get-depth-of-current-line) 
		    depth))
      (forward-line 1))
    (beginning-of-line)))

(defun wl-summary-insert-summary (entity database mark-alist dummy)
  (let* (summary-line msg)
    (setq msg (elmo-msgdb-overview-entity-get-number entity))
    (when (setq summary-line
		(wl-summary-overview-create-summary-line 
		 msg entity nil 0 mark-alist))
      (let ((inhibit-read-only t)
	    buffer-read-only)
	(goto-char (point-max))
	(insert summary-line "\n")))))

(defun wl-summary-default-subject-filter (subject)
  (let ((case-fold-search t))
    (setq subject (elmo-replace-in-string subject "[ \t]*\\(re\\|was\\):" ""))
    (setq subject (elmo-replace-in-string subject "[ \t]" ""))
    (elmo-replace-in-string subject "^\\[.*\\]" "")))

(defun wl-summary-subject-equal (subject1 subject2)
  (string= (wl-summary-subject-filter-func-internal subject1)
	   (wl-summary-subject-filter-func-internal subject2)))

(defun wl-summary-insert-thread-entity (entity overview mark-alist update)
  (let* ((this-id (elmo-msgdb-overview-entity-get-id entity))
	 (parent-entity 
	  (elmo-msgdb-overview-get-parent-entity this-id overview));; temp
	 ;;(parent-id (elmo-msgdb-overview-entity-get-id parent-entity))
	 (parent-number (elmo-msgdb-overview-entity-get-number parent-entity))
	 msg
	 )
    (if (and parent-number
	     wl-summary-divide-thread-when-subject-changed
	     (not (wl-summary-subject-equal 
		   (or (elmo-msgdb-overview-entity-get-subject 
			entity) "")
		   (or (elmo-msgdb-overview-entity-get-subject 
			parent-entity) ""))))
	(setq parent-number nil))
    (setq msg (elmo-msgdb-overview-entity-get-number entity))
    (wl-thread-insert-message entity overview mark-alist
			      msg parent-number update)))

(defun wl-summary-update-thread (entity 
				 overview 
				 mark-alist 
				 thr-entity
				 parent-entity)
  (let* ((depth 0)
	 (this-id (elmo-msgdb-overview-entity-get-id entity))
	 ;(parent-entity 
	 ; (elmo-msgdb-overview-get-parent-entity this-id overview))
	 (parent-id (elmo-msgdb-overview-entity-get-id parent-entity))
	 (parent-number (elmo-msgdb-overview-entity-get-number parent-entity))
	 summary-line msg subject-differ
	 )
    (cond 
     ((or (not parent-id)
	  (string= this-id parent-id))
      (goto-char (point-max))
      (beginning-of-line))
     ;; parent already exists in buffer.
     ((setq depth (or (wl-summary-summary-line-already-exists-p 
		       parent-number (current-buffer)) -1))
      (setq depth (+ 1 depth))
      (wl-thread-goto-bottom-of-sub-thread)
      ))
    (if (and (elmo-msgdb-overview-entity-get-number entity))
	(if (setq summary-line
		  (wl-summary-overview-create-summary-line 
		   (elmo-msgdb-overview-entity-get-number entity)
		   entity parent-entity depth mark-alist nil nil
		   thr-entity))
	    (let ((inhibit-read-only t)
		  (buffer-read-only nil))
	      (insert summary-line "\n")
	      )))))

(defun wl-summary-mark-as-unread ()
  (interactive)
  (save-excursion
    (let* (eol
	  (inhibit-read-only t)
	  (buffer-read-only nil)
	  (folder wl-summary-buffer-folder-name)
	  (msgdb wl-summary-buffer-msgdb)
	  (mark-alist (elmo-msgdb-get-mark-alist msgdb))
	  ;;(number-alist (elmo-msgdb-get-number-alist msgdb))
	  number
	  )
      (end-of-line)
      (setq eol (point))
      (re-search-backward (concat "^" wl-summary-buffer-number-regexp
				  "..../..")) ; set cursor line
      (beginning-of-line)
      (if (re-search-forward 
	   (format "^ *\\([0-9]+\\)[^0-9]\\(%s\\)" 
		   " ") eol t)
	  (progn
	    (setq number (string-to-int (wl-match-buffer 1)))
	    ;; server side mark
	    (elmo-mark-as-unread folder (list number)
				 msgdb)
	    (delete-region (match-beginning 2) (match-end 2))
	    (insert wl-summary-unread-mark)
	    (setq mark-alist
		  (elmo-msgdb-mark-append mark-alist
					  (string-to-int (wl-match-buffer 1))
					;(cdr (assoc 
					;(wl-match-buffer 1)
					;number-alist))
					  wl-summary-unread-mark))
	    (elmo-msgdb-set-mark-alist msgdb mark-alist)
	    (setq wl-summary-buffer-unread-count 
		  (+ 1 wl-summary-buffer-unread-count))
	    (wl-summary-update-modeline)
	    (wl-folder-update-unread 
	     folder
	     (+ wl-summary-buffer-unread-count 
		wl-summary-buffer-new-count))
	    (wl-summary-set-mark-modified)
	    (if wl-summary-highlight
		(wl-highlight-summary-current-line))
	    )))
    )
  (set-buffer-modified-p nil)
  )

(defun wl-summary-delete ()
  (interactive)
  (let (num)
    (setq num (wl-summary-mark-line "D"))
    (if (null num)
	(message "No message.")
      (setq wl-summary-buffer-delete-list
	    (cons num wl-summary-buffer-delete-list))
      (if (interactive-p)
	  (if (eq wl-summary-move-direction-downward nil)
	      (wl-summary-prev)
	    (wl-summary-next)))
      num)))

(defun wl-summary-remove-destination ()
  (save-excursion
    (let ((inhibit-read-only t)
	  (buffer-read-only nil)
          (buf (current-buffer))
          sol eol rs re)
      (beginning-of-line)
      (setq sol (point))
      (end-of-line)
      (setq eol (point))
      (setq rs (next-single-property-change sol 'invisible buf eol))
      (setq re (next-single-property-change rs 'invisible buf eol))
      (set-text-properties rs re '(invisible nil))
      (goto-char re)
      (delete-char (- eol re)))))

(defun wl-summary-check-mark (msg mark)
  (let ((check-func (cond ((string= mark "o")
			   'wl-summary-msg-marked-as-refiled
			   )
			  ((string= mark "O")
			   'wl-summary-msg-marked-as-copied
			   )
			  ((string= mark "D")
			   'wl-summary-msg-marked-as-deleted
			   )
			  ((string= mark "*")
			   'wl-summary-msg-marked-as-temp))))
    (if check-func
	(funcall check-func msg))))

(defun wl-summary-mark-collect (mark &optional begin end)
  (save-excursion
    (save-restriction
      (let (msglist)
	(narrow-to-region (or begin (point-min))
			  (or end (point-max)))
	(goto-char (point-min))
	;; for thread...
	(if (eq wl-summary-buffer-view 'thread)
	    (progn
	      (while (not (eobp))
		(let* ((number (wl-summary-message-number))
		       (entity (wl-thread-get-entity number))
		       result)
		  ;; opened...only myself is checked.
		  (if (wl-summary-check-mark number mark)
		      (wl-append msglist (list number)))
		  (unless (wl-thread-entity-get-opened entity) 
		    ;; closed...children is also checked.
		    (if (setq result (wl-thread-get-children-msgs-with-mark
				      number
				      mark))
			(wl-append msglist result)))
		  (forward-line 1)))
	      (wl-uniq-list msglist))
	  (let* ((case-fold-search nil)
		 (re (format (concat wl-summary-message-regexp "%s") 
			     (regexp-quote mark))))
	    (while (re-search-forward re nil t)
	      (setq msglist (cons (wl-summary-message-number) msglist)))
	    (nreverse msglist)))))))

(defun wl-summary-exec ()
  (interactive)
  (wl-summary-exec-subr (mapcar 'car wl-summary-buffer-refile-list)
			(reverse wl-summary-buffer-delete-list)
			(mapcar 'car wl-summary-buffer-copy-list)))

(defun wl-summary-exec-region (beg end)
  (interactive "r")
  (message "Collecting marks ...")
  (save-excursion
    (goto-char beg)
    (beginning-of-line)
    (setq beg (point))
    (goto-char end)
    (end-of-line)
    (setq end (point))
    (wl-summary-exec-subr (wl-summary-mark-collect "o" beg end)
			  (wl-summary-mark-collect "D" beg end)
			  (wl-summary-mark-collect "O" beg end))))
  
(defun wl-summary-exec-subr (msgs dels cpys)
  (save-excursion
    (let* ((del-fld (wl-summary-get-delete-folder
		     wl-summary-buffer-folder-name))
	   (start (point))
	   dst tmp msg msgs2 cpys2
	   msg-dst dst-msgs len 
	   refile-failures 
	   copy-failures
	   succeeds result executed)
      (if (not (or msgs dels cpys))
	  (message "No marks")
	(message "Executing ...")
	(setq msgs (append msgs dels))
	(setq msgs2 msgs)
	(while dels
	  (when (not (assq (car dels) wl-summary-buffer-refile-list))
	    (wl-append wl-summary-buffer-refile-list
		       (list (cons (car dels) del-fld)))
	    (setq wl-summary-buffer-delete-list
		  (delete (car dels) wl-summary-buffer-delete-list)))
	  (setq dels (cdr dels)))
	(setq len (length wl-summary-buffer-refile-list))
	;; begin refile...
	(while msgs
	  (setq msg (car msgs))
	  (setq msgs (cdr msgs))
	  (setq msg-dst (assq msg wl-summary-buffer-refile-list))
	  (setq dst (cdr msg-dst))
	  (if dst
	      (if (setq tmp (assoc dst dst-msgs))
		  (setq dst-msgs (cons (append tmp (list msg))
					 (delete tmp dst-msgs)))
		(setq dst-msgs (cons (list dst msg) dst-msgs))
		)))
	(setq refile-failures 0)
	(goto-char start) ; avoid moving cursor to the bottom line.
	(setq executed 0)
	(while dst-msgs
	  (elmo-msgdb-add-msgs-to-seen-list 
	   (car (car dst-msgs)) ;dst-folder	   
	   (cdr (car dst-msgs)) ;msgs 
	   wl-summary-buffer-msgdb 
	   (concat wl-summary-important-mark
		   wl-summary-read-uncached-mark))
	  (setq result nil)
	  (condition-case nil
	      (setq result (elmo-move-msgs wl-summary-buffer-folder-name 
					   (cdr (car dst-msgs))
					   (car (car dst-msgs))
					   wl-summary-buffer-msgdb 
					   len executed (cdr dst-msgs)))
	    (error nil))
	  (if result ; succeeded.
	      (progn
		;; update buffer.
		(wl-summary-delete-messages-on-buffer 
		 (cdr (car dst-msgs)))
		;; update refile-alist.
		(mapcar 
		 (function 
		  (lambda (x)
		    (setq wl-summary-buffer-refile-list
			  (delq (assq x wl-summary-buffer-refile-list)
				wl-summary-buffer-refile-list))))
		 (cdr (car dst-msgs)))
		)
	    (setq refile-failures 
		  (+ refile-failures (length (cdr (car dst-msgs))))))
	  (setq executed (+ executed (length (cdr (car dst-msgs)))))
	  (setq dst-msgs (cdr dst-msgs)))
	;; end refile
	(setq len (length wl-summary-buffer-copy-list))
	;; begin cOpy...
	(setq cpys2 cpys)
	(while cpys
	  (setq msg (car cpys))
	  (setq cpys (cdr cpys))
	  (setq msg-dst (assq msg wl-summary-buffer-copy-list))
	  (setq dst (cdr msg-dst))
	  (if dst
	      (if (setq tmp (assoc dst dst-msgs))
		  (setq dst-msgs (cons (append tmp (list msg))
				       (delete tmp dst-msgs)))
		(setq dst-msgs (cons (list dst msg) dst-msgs))
		)))
	(setq copy-failures 0)
	(setq executed 0)
	(while dst-msgs
	  (elmo-msgdb-add-msgs-to-seen-list 
	   (car (car dst-msgs)) ;dst-folder	   
	   (cdr (car dst-msgs)) ;msgs 
	   wl-summary-buffer-msgdb 
	   (concat wl-summary-important-mark
		   wl-summary-read-uncached-mark))
	  (setq result nil)
	  (condition-case nil
	      (setq result (elmo-move-msgs wl-summary-buffer-folder-name 
					   (cdr (car dst-msgs))
					   (car (car dst-msgs))
					   wl-summary-buffer-msgdb 
					   len executed 
					   (cdr dst-msgs) t)) ; t is no-delete
	    (error nil))
	  (if result ; succeeded.
	      (progn
		;; update buffer.
		(wl-summary-delete-copy-marks-on-buffer (cdr (car dst-msgs)))
		;; update copy-alist
		(mapcar
		 (function
		  (lambda (x)
		    (setq wl-summary-buffer-copy-list
			  (delq (assq x wl-summary-buffer-copy-list)
				wl-summary-buffer-copy-list))))
		 (cdr (car dst-msgs)))
		)
	    (setq copy-failures
		  (+ copy-failures (length (cdr (car dst-msgs))))))
	  (setq executed (+ executed (length (cdr (car dst-msgs)))))
	  (setq dst-msgs (cdr dst-msgs)))
	;; end cOpy 
	(wl-summary-folder-info-update)
	(wl-summary-set-message-modified)
	(wl-summary-set-mark-modified)
	(run-hooks 'wl-summary-exec-hook)
	(set-buffer-modified-p nil)
	(message (concat "Executing ... done"
			 (if (> refile-failures 0)
			     (format " (%d refiling failed)" refile-failures)
			   "")
			 (if (> copy-failures 0)
			     (format " (%d copying failed)" copy-failures)
			   "")
			 "."))
	))))

(defun wl-summary-read-folder (default &optional purpose ignore-error)
  (let ((fld (completing-read 
	      (format "Folder name %s(%s): " (or purpose "")
		      default)
	      wl-folder-entity-hashtb nil nil wl-default-spec
	      'wl-read-folder-hist)))
    (if (string= fld wl-default-spec) (setq fld default))
    (if (wl-folder-entity-exists-p fld)
	()
      ;; not in folder-entity-hashtb
      (if ignore-error 
	  (condition-case err
	      (if (elmo-folder-exists-p fld)
		  ()
		(if (y-or-n-p 
		     (format "Folder %s does not exit, create it?" fld))
		    (elmo-create-folder fld)
		  (error "Folder is not created.")))
	    (error (ding)(wl-display-error err t)))
	(if (elmo-folder-exists-p fld)
	    ()
	  (if (y-or-n-p 
	       (format "Folder %s does not exit, create it?" fld))
	      (elmo-create-folder fld)
	    (error "Folder is not created.")))))
    fld))

(defun wl-summary-print-destination (msg-num &optional dest)
  "Print refile destination on line."
  (wl-summary-remove-destination)
  (let ((inhibit-read-only t)
	(buffer-read-only nil)
	folder len rs re c)
    (setq folder (or dest 
		     (cdr (assq msg-num wl-summary-buffer-refile-list))))
    (setq len (string-width folder))
    (if (< len 1) ()
      (end-of-line)
      (setq re (point))
      (setq c 0)
      (while (< c len)
	(forward-char -1)
	(setq c (+ c (char-width (following-char)))))
      (setq rs (point))
      (set-text-properties rs re '(invisible t))
      (goto-char re)
      (wl-highlight-refile-destination-string folder)
      (insert folder)
      (set-buffer-modified-p nil))))

(defun wl-summary-copy (&optional dst)
  (interactive)
  (let* ((msg-num (wl-summary-message-number))
	 (msgid (and msg-num 
		     (cdr (assq msg-num
				 (elmo-msgdb-get-number-alist
				  wl-summary-buffer-msgdb)))))
	 (entity (and msg-num
		      (elmo-msgdb-overview-get-entity-by-number
		       (elmo-msgdb-get-overview wl-summary-buffer-msgdb)
		       msg-num)))
	 (folder (and msg-num
		      (or dst (wl-summary-read-folder 
			       (wl-refile-guess entity)
			       "for copy"
			       ))))
	 already)
    (if (string= folder wl-summary-buffer-folder-name)
	(error "Same folder."))
    (if (null msg-num)
	(message "No message.")
      (if (string= folder wl-queue-folder)
	  (error "Don't copy to %s." folder))
      ;; learn
;      (wl-refile-learn entity folder)
      (when msg-num
	(setq already (assq msg-num wl-summary-buffer-copy-list))
	(if already
	    (setq wl-summary-buffer-copy-list
		  (delete already wl-summary-buffer-copy-list)))
	(setq wl-summary-buffer-copy-list
	      (append wl-summary-buffer-copy-list
		      (list (cons msg-num folder))))
	(wl-summary-mark-line "O")
	;; print copy destination
	(wl-summary-print-destination msg-num folder)
	(if (interactive-p)
 	    (if (eq wl-summary-move-direction-downward nil)
 		(wl-summary-prev)
 	      (wl-summary-next)))
	(run-hooks 'wl-summary-copy-hook)
	(setq wl-summary-buffer-prev-copy-destination folder)
	))))

(defun wl-summary-refile (&optional dst)
  (interactive)
  (let* ((msg-num (wl-summary-message-number))
	 (msgid (and msg-num 
		     (cdr (assq msg-num
				 (elmo-msgdb-get-number-alist
				  wl-summary-buffer-msgdb)))))
	 (entity (and msg-num
		      (elmo-msgdb-overview-get-entity-by-number
		       (elmo-msgdb-get-overview wl-summary-buffer-msgdb)
		       msg-num)))
	 (folder (and msg-num
		      (or dst (wl-summary-read-folder 
			       (wl-refile-guess entity)
			       "for refile"
			       ))))
	 already)
    (if (string= folder wl-summary-buffer-folder-name)
	(error "Same folder."))
    (if (and (elmo-use-cache-p folder msg-num)
	     (not wl-plugged)
	     (or (null msgid)
		 (not (elmo-cache-exists-p msgid))))
	(error "Unplugged."))
    (if (null msg-num)
	(message "No message.")
      (if (string= folder wl-queue-folder)
	  (error "Don't refile to %s." folder))
      ;; learn
      (wl-refile-learn entity folder)
      (when msg-num
	(setq already (assq msg-num wl-summary-buffer-refile-list))
	(when already
	  (setq wl-summary-buffer-refile-list
		(delete already wl-summary-buffer-refile-list))
	  (wl-summary-unmark-line))
	(setq wl-summary-buffer-refile-list
	      (append wl-summary-buffer-refile-list
		      (list (cons msg-num folder))))
	(wl-summary-mark-line "o")
	;; print refile destination
	(wl-summary-print-destination msg-num)
	(if (interactive-p)
 	    (if (eq wl-summary-move-direction-downward nil)
 		(wl-summary-prev)
 	      (wl-summary-next))))
      (run-hooks 'wl-summary-refile-hook)
      (setq wl-summary-buffer-prev-refile-destination folder)
      msg-num
      )))

(defun wl-summary-refile-prev-destination ()
  "Refile message to previously refiled destination"
  (interactive)
  (wl-summary-refile wl-summary-buffer-prev-refile-destination)
  (if (eq wl-summary-move-direction-downward nil)
      (wl-summary-prev)
    (wl-summary-next))
  )

(defun wl-summary-copy-prev-destination ()
  "Refile message to previously refiled destination"
  (interactive)
  (wl-summary-copy wl-summary-buffer-prev-copy-destination)
  (if (eq wl-summary-move-direction-downward nil)
      (wl-summary-prev)
    (wl-summary-next))
  )

(defsubst wl-summary-no-auto-refile-message-p (msg mark-alist) 
  (member (cadr (assq msg mark-alist)) wl-summary-auto-refile-skip-marks))

(defun wl-summary-auto-refile (&optional open-all)
  "Set refile mark automatically according to wl-refile-guess-by-rule."
  (interactive "P")
  (save-excursion
    (if (and (eq wl-summary-buffer-view 'thread)
	     open-all) 
	(wl-thread-open-all))
    (let* ((spec wl-summary-buffer-folder-name)
	   (overview (elmo-msgdb-get-overview
		      wl-summary-buffer-msgdb))
	   (mark-alist (elmo-msgdb-get-mark-alist
			wl-summary-buffer-msgdb))
	   (count 0)
	   number dst thr-entity)
      (goto-line 1)
      (while (not (eobp))
	(setq number (wl-summary-message-number))
	(when (and (not (wl-summary-no-auto-refile-message-p number 
							     mark-alist))
		   (setq dst
			 (wl-refile-guess-by-rule
			  (elmo-msgdb-overview-get-entity-by-number
			   overview number)))
		   (not (equal dst spec)))
	  (wl-summary-refile dst)
	  (incf count)
	  (message "Marking...%d message(s)." count)
	  )
	(if (eq wl-summary-buffer-view 'thread)
	    ;; process invisible children.
	    (if (not (wl-thread-entity-get-opened
		      (setq thr-entity (wl-thread-get-entity number))))
		(mapcar 
		 (function
		  (lambda (x)
		    (when 
			(and (setq dst 
				   (wl-refile-guess-by-rule
				    (elmo-msgdb-overview-get-entity-by-number 
				     overview x)))
			     (not (equal dst spec)))
		      (wl-append wl-summary-buffer-refile-list
				 (list (cons x dst)))
		      (incf count)
		      (message "Marking...%d message(s)." count))))
		 (delete-if
		  (function (lambda (x)
			      (wl-summary-no-auto-refile-message-p 
			       x
			       mark-alist)))
		  (wl-thread-entity-get-descendant thr-entity)))))
	(forward-line))
      (if (eq count 0)
	  (message "No message was marked.")
	(message "Marked %d message(s)." count)))))

(defun wl-summary-unmark-line ()
  (interactive)
  (save-excursion
    (beginning-of-line)
    (let ((inhibit-read-only t)
	  (buffer-read-only nil)
	  msg-num
	  cur-mark)
      (when (looking-at "^ *\\([0-9]+\\)\\([^0-9]\\)")
	(goto-char (match-end 2))
	(setq msg-num (string-to-int (wl-match-buffer 1)))
	(setq cur-mark (wl-match-buffer 2))
	(if (string= cur-mark " ")
	    ()
	  (delete-region (match-beginning 2) (match-end 2))
	  (insert " ")
	  (wl-summary-delete-mark-from-list msg-num cur-mark)
	  (if wl-summary-highlight	  
	      (wl-highlight-summary-current-line nil nil nil))
	  ))))
  (set-buffer-modified-p nil))

(defun wl-summary-msg-marked-as-temp (msg)
  (if (memq msg wl-summary-buffer-temp-mark-list)
      t)
  )

(defun wl-summary-msg-marked-as-copied (msg)
  (assq msg wl-summary-buffer-copy-list)
  )

(defun wl-summary-msg-marked-as-deleted (msg)
  (if (memq msg wl-summary-buffer-delete-list)
      t)
  )

(defun wl-summary-msg-marked-as-refiled (msg)
  (assq msg wl-summary-buffer-refile-list)
  )

(defun wl-summary-msg-unmark-temp-marks (msg)
  "Delete temp marks for MSG."
  (let (pair)
    (cond 
     ((memq msg wl-summary-buffer-temp-mark-list)
      (setq wl-summary-buffer-temp-mark-list
	    (delete msg wl-summary-buffer-temp-mark-list)))
     ((setq pair (assq msg wl-summary-buffer-copy-list))
      (setq wl-summary-buffer-copy-list
	    (delete pair wl-summary-buffer-copy-list)))
     ((setq pair (assq msg wl-summary-buffer-refile-list))
      (setq wl-summary-buffer-refile-list
	    (delete pair wl-summary-buffer-refile-list)))
     ((memq msg wl-summary-buffer-delete-list)
      (setq wl-summary-buffer-delete-list
	    (delete msg wl-summary-buffer-delete-list))))))

(defun wl-summary-temp-mark-line ()
  (interactive)
  (let (num)
    (setq num (wl-summary-mark-line "*"))
    (setq wl-summary-buffer-temp-mark-list
	  (cons num wl-summary-buffer-temp-mark-list))
    (if (interactive-p)
	(if (eq wl-summary-move-direction-downward nil)
	    (wl-summary-prev)
	  (wl-summary-next)))))

(defun wl-summary-copy-region (beg end)
  (interactive "r")
  (save-excursion
    (save-restriction
      (goto-char beg)
      ;; guess by first msg
      (let* ((msgid (cdr (assq (wl-summary-message-number)
				(elmo-msgdb-get-number-alist
				 wl-summary-buffer-msgdb))))
	     (entity (assoc msgid
			    (elmo-msgdb-get-overview
			     wl-summary-buffer-msgdb)))
	     folder)
	(if entity
	    (setq folder (wl-summary-read-folder (wl-refile-guess entity)
						 "for copy")))
	(narrow-to-region beg end)	;(save-excursion (goto-char end)
					;					      (end-of-line) (point)))
	(if (eq wl-summary-buffer-view 'thread)
	    (progn
	      (while (not (eobp))
		(let* ((number (wl-summary-message-number))
		       (entity (wl-thread-get-entity number))
		       children)
		  (if (wl-thread-entity-get-opened entity)
		      ;; opened...copy line.
		      (wl-summary-copy folder)
		    ;; closed
		    (wl-summary-copy folder) ; copy itself
		    (setq children 
			  (delete number 
				  (wl-thread-get-children-msgs number)))
		    (while children
		      (wl-append wl-summary-buffer-copy-list
				 (list (cons (car children) folder)))
		      (setq children (cdr children))))
		  (forward-line 1))))
	  (while (not (eobp))
	    (wl-summary-copy folder)
	    (forward-line 1)))))))

(defun wl-summary-refile-region (beg end)
  (interactive "r")
  (save-excursion
    (save-restriction
      (goto-char beg)
      ;; guess by first msg
      (let* ((msgid (cdr (assq (wl-summary-message-number)
				(elmo-msgdb-get-number-alist
				 wl-summary-buffer-msgdb))))
	     (entity (assoc msgid
			    (elmo-msgdb-get-overview
			     wl-summary-buffer-msgdb)))
	     folder)
  	(if entity
  	    (setq folder (wl-summary-read-folder (wl-refile-guess entity)
 						 "for refile")))
 	(narrow-to-region beg end)	;(save-excursion (goto-char end)
  					;(end-of-line) (point)))
 	(if (eq wl-summary-buffer-view 'thread)
 	    (progn
 	      (while (not (eobp))
 		(let* ((number (wl-summary-message-number))
 		       (entity (wl-thread-get-entity number))
 		       children)
 		  (if (wl-thread-entity-get-opened entity)
 		      ;; opened...refile line.
 		      (wl-summary-refile folder)
 		    ;; closed
 		    (wl-summary-refile folder) ; refile itself
 		    (setq children 
 			  (delete number 
 				  (wl-thread-get-children-msgs number)))
 		    (while children
 		      (wl-append wl-summary-buffer-refile-list
 				 (list (cons (car children) folder)))
 		      (setq children (cdr children))))
 		  (forward-line 1))))
 	  (while (not (eobp))
 	    (wl-summary-refile folder)
 	    (forward-line 1)))))))

(defun wl-summary-delete-region (beg end)
  (interactive "r")
  (save-excursion
    (save-restriction
;      (narrow-to-region (save-excursion (goto-char beg) 
;					(beginning-of-line)
;					(point))
;			(save-excursion (goto-char end)
;					(end-of-line)
;					(point)))
      (narrow-to-region beg end)
      (goto-char (point-min))
      (if (eq wl-summary-buffer-view 'thread)
	  (progn
	    (while (not (eobp))
	      (let* ((number (wl-summary-message-number))
		     (entity (wl-thread-get-entity number))
		     (wl-summary-move-direction-downward t)
		     children)
		(if (wl-thread-entity-get-opened entity)
		    ;; opened...delete line.
		    (wl-summary-delete)
		  ;; closed
		  (wl-summary-delete) ; delete itself
		  (setq children (wl-thread-get-children-msgs number))
		  (while children
		    (wl-append wl-summary-buffer-delete-list
			       (list (car children)))
		    (setq children (cdr children)))))
	      (forward-line 1)))
	(while (not (eobp))
	  (wl-summary-delete)
	  (forward-line 1))))))

(defun wl-summary-unmark-region (beg end)
  (interactive "r")
  (save-excursion
    (save-restriction
;       (narrow-to-region (save-excursion (goto-char beg) 
; 					(beginning-of-line)
; 					(point))
; 			(save-excursion (goto-char end)
; 					(end-of-line)
; 					(point)))
      (narrow-to-region beg end)
      (goto-char (point-min))
      (if (eq wl-summary-buffer-view 'thread)
	  (progn
	    (while (not (eobp))
	      (let* ((number (wl-summary-message-number))
		     (entity (wl-thread-get-entity number))
		     children)
		(if (wl-thread-entity-get-opened entity)
		    ;; opened...unmark line.
		    (wl-summary-unmark-line)
		  ;; closed
		  (wl-summary-unmark-line) ; unmark itself
		  (setq children (wl-thread-get-children-msgs number))
		  (while children
		    (wl-summary-msg-unmark-temp-marks (car children))
		    (setq children (cdr children))))
		(forward-line 1))))
	(while (not (eobp))
	  (wl-summary-unmark-line)
	  (forward-line 1))))))

(defun wl-summary-temp-mark-region (beg end)
  (interactive "r")
  (save-excursion
    (save-restriction
;      (narrow-to-region (save-excursion (goto-char beg) 
;					(beginning-of-line)
;					(point))
;			(save-excursion (goto-char end)
;					(end-of-line)
;					(point)))
      (narrow-to-region beg end)
      (goto-char (point-min))
      (if (eq wl-summary-buffer-view 'thread)
	  (progn
	    (while (not (eobp))
	      (let* ((number (wl-summary-message-number))
		     (entity (wl-thread-get-entity number)))
		(if (wl-thread-entity-get-opened entity)
		    ;; opened...mark line.
		    (progn
		      (wl-summary-mark-line "*")
		      (wl-append wl-summary-buffer-temp-mark-list
				 (list number)))
		  ;; closed
		  (wl-summary-mark-line "*")
		  (wl-append wl-summary-buffer-temp-mark-list
			     (wl-thread-get-children-msgs number)))
		(forward-line 1))))
	(while (not (eobp))
	  (let (msg-num)
	    (setq msg-num (wl-summary-mark-line "*"))
	    (setq wl-summary-buffer-temp-mark-list
		  (cons msg-num wl-summary-buffer-temp-mark-list)))
	  (forward-line 1))))))

(defun wl-summary-temp-mark-all ()
  (interactive)
  (wl-summary-temp-mark-region (point-min) (point-max))
  (setq wl-summary-buffer-temp-mark-list 
	(mapcar 'car 
		(elmo-msgdb-get-number-alist wl-summary-buffer-msgdb))))

(defun wl-summary-delete-all-mark (mark)
  (goto-char (point-min))
  (let ((case-fold-search nil))
    (while (re-search-forward (format "^ *[0-9]+%s" 
				      (regexp-quote mark)) nil t)
      (wl-summary-unmark-line))
    (cond ((string= mark "*")
	   (setq wl-summary-buffer-temp-mark-list nil))
	  ((string= mark "D")
	   (setq wl-summary-buffer-delete-list nil)
	   )
	  ((string= mark "O")
	   (setq wl-summary-buffer-copy-list nil))
	  ((string= mark "o")
	   (setq wl-summary-buffer-refile-list nil)))))

(defun wl-summary-unmark-all ()
  "Unmark all according to what you input."
  (interactive)
  (let ((unmarks (string-to-char-list (read-from-minibuffer "Unmark: ")))
	cur-mark)
    (save-excursion
      (while unmarks
	(setq cur-mark (char-to-string (car unmarks)))
	(wl-summary-delete-all-mark cur-mark)
	(setq unmarks (cdr unmarks))
	))))

(defun wl-summary-temp-mark-thread ()
  (interactive)
  (let (beg end)
    (end-of-line)
    (wl-summary-goto-top-of-current-thread)
    (wl-thread-force-open)
    (setq beg (point))
    (end-of-line)
    (wl-summary-goto-bottom-of-current-thread)
;    (forward-line -1)
    (beginning-of-line)
    (setq end (point))
    (wl-summary-temp-mark-region beg end)
    ))

(defun wl-summary-temp-mark-msgs (msgs)
  (while msgs
    (if (eq wl-summary-buffer-view 'thread)
	(wl-thread-jump-to-msg (car msgs))
      (wl-summary-jump-to-msg (car msgs)))
    (wl-summary-temp-mark-line)
    (setq msgs (cdr msgs))))

(defun wl-summary-pick ()
  (interactive)
  (let* ((completion-ignore-case t)
	 (field (completing-read
		 (format "Field name (%s): " wl-summary-pick-field-default)
		 (mapcar 'list
			 (append '("From" "Subject" "Date"
				   "To" "Cc" "Body" "Since" "Before")
				 elmo-msgdb-extra-fields))))
	 (value (if (string-match field "Since\\|Before")
		    (completing-read "Value: "
				     (mapcar (function
					      (lambda (x)
						(list (format "%s" (car x)))))
					     elmo-date-descriptions))
		  (read-from-minibuffer "Value: ")))
	 (overview (elmo-msgdb-get-overview wl-summary-buffer-msgdb))
	 (number-alist (elmo-msgdb-get-number-alist wl-summary-buffer-msgdb))
	 server-side-search
	 result get-func sum)
    (if (string= field "")
	(setq field wl-summary-pick-field-default))
    (setq field (downcase field))
    (cond 
     ((string-match field "from")
      (setq get-func 'elmo-msgdb-overview-entity-get-from))
     ((string-match field "subject")
      (setq get-func 'elmo-msgdb-overview-entity-get-subject))
     ((string-match field "date")
      (setq get-func 'elmo-msgdb-overview-entity-get-date))
     ((string-match field "to")
      (setq get-func 'elmo-msgdb-overview-entity-get-to))
     ((string-match field "cc")
      (setq get-func 'elmo-msgdb-overview-entity-get-cc))
     ((string-match field "since")
      (setq server-side-search (format "since %s" value)))
     ((string-match field "before")
      (setq server-side-search (format "before %s" value)))     
     ((string-match field "body")
      (setq server-side-search (format "body %s" value)))
     ((member field elmo-msgdb-extra-fields)
      (setq get-func
	    (lambda (entity)
	      (elmo-msgdb-overview-entity-get-extra-field entity field))))
     (t
      (error "Pick by %s is not supported." field)))
    (unwind-protect
	(if server-side-search
	    (progn
	      (message "searching...")
	      (let ((elmo-mime-charset wl-summary-buffer-mime-charset))
		(setq result (elmo-search wl-summary-buffer-folder-name 
					  server-side-search
					  )))
	      (message "picked %d message(s)." (length result)))
	  (setq sum 0)
	  (while overview
	    (when (string-match value
				(or 
				 (funcall get-func (car overview))
				 ""))
	      (setq result
		    (append result
			    (list
			     (car
			      (rassoc 
			       (elmo-msgdb-overview-entity-get-id 
				(car overview))
			       number-alist)))))
	      (message "picked %d message(s)." (setq sum (+ sum 1))))
	    (setq overview (cdr overview))))
      (wl-summary-temp-mark-msgs result))))

(defun wl-summary-unvirtual ()
  "Exit from current virtual folder."
  (interactive)
  (if (eq 'filter
	  (elmo-folder-get-type wl-summary-buffer-folder-name))
      (wl-summary-goto-folder-subr (nth 4 (elmo-folder-get-spec
					   wl-summary-buffer-folder-name))
				   'update nil nil t)
    (error "This folder is not filtered.")))

(defun wl-summary-virtual (&optional arg)
  "Goto virtual folder."
  (interactive "P")
  (if arg
      (wl-summary-unvirtual)
    (let* ((completion-ignore-case t)
	   (field (completing-read (format "Field name (%s): " 
					   wl-summary-pick-field-default)
				   '(("From" . "From") 
				     ("Subject" . "Subject")
				     ("To" . "To")
				     ("Cc" . "Cc")
				     ("Body" . "Body")
				     ("Since" . "Since")
				     ("Before" . "Before")
				     )))
	   (value (read-from-minibuffer "Value: ")))
      (if (string= field "")
	  (setq field wl-summary-pick-field-default))
      (wl-summary-goto-folder-subr (concat "/" (downcase field) "=" value "/" 
					   wl-summary-buffer-folder-name)
				   'update nil nil t))))

(defun wl-summary-delete-all-temp-marks ()
  (interactive)
  (save-excursion
    (goto-char (point-min))
    (message "Unmarking...")
    (while (not (eobp))
      (wl-summary-unmark-line)
      (forward-line))
    (message "Unmarking...done.")
    (setq wl-summary-buffer-temp-mark-list nil)
    (setq wl-summary-buffer-delete-list nil)
    (setq wl-summary-buffer-refile-list nil)
    (setq wl-summary-buffer-copy-list nil)))
    

(defun wl-summary-delete-mark-from-list (msg-num mark)
  (cond ((string= mark "*")
	 (setq wl-summary-buffer-temp-mark-list
	       (delete msg-num wl-summary-buffer-temp-mark-list))
	 )
	((string= mark "D")
	 (setq wl-summary-buffer-delete-list
	       (delete msg-num wl-summary-buffer-delete-list))
	 )
	((string= mark "O")
	 (wl-summary-remove-destination)
	 (setq wl-summary-buffer-copy-list
	       (delete 
		(assq msg-num wl-summary-buffer-copy-list)
		wl-summary-buffer-copy-list)))
	((string= mark "o")
	 (wl-summary-remove-destination)
	 (setq wl-summary-buffer-refile-list
	       (delete 
		(assq msg-num wl-summary-buffer-refile-list)
		wl-summary-buffer-refile-list)))
	))

(defun wl-summary-mark-line (mark)
  "return message number"
  (save-excursion
    (beginning-of-line)
    (let ((inhibit-read-only t)
	  (buffer-read-only nil)
	  msg-num
	  cur-mark)
      (when (looking-at "^ *\\([0-9]+\\)\\([^0-9]\\)")
	(setq msg-num  (string-to-int (wl-match-buffer 1)))
	(setq cur-mark (wl-match-buffer 2))
	(goto-char (match-end 1))
	(delete-region (match-beginning 2) (match-end 2))
	(wl-summary-delete-mark-from-list msg-num cur-mark)
	(insert mark)
	(if wl-summary-highlight
	    (wl-highlight-summary-current-line nil nil t))
	(set-buffer-modified-p nil)
	msg-num
	))))

(defun wl-summary-temp-mark-delete ()
  (interactive)
  (save-excursion
    (goto-char (point-min))
    (let ((regexp (concat "^" wl-summary-buffer-number-regexp "\\(\\*\\)"))
	  number mlist)
      (while (re-search-forward regexp nil t)
	(let (wl-summary-buffer-disp-msg)
	  (setq number (wl-summary-delete))
	  (if number
	      (setq wl-summary-buffer-temp-mark-list
		    (delete number wl-summary-buffer-temp-mark-list)))))
      (setq mlist wl-summary-buffer-temp-mark-list)
      (while mlist
	(wl-append wl-summary-buffer-delete-list (list (car mlist)))
	(setq wl-summary-buffer-temp-mark-list
	      (delete (car mlist) wl-summary-buffer-temp-mark-list))
	(setq mlist (cdr mlist))))))

(defun wl-summary-temp-mark-prefetch ()
  (interactive)
  (save-excursion
    (let ((mlist wl-summary-buffer-temp-mark-list)
	  (inhibit-read-only t)
	  (buffer-read-only nil)
	  new-mark)
      (while mlist
	(setq new-mark (wl-summary-prefetch-msg (car mlist)))
	(when (wl-summary-jump-to-msg (car mlist))
	  (wl-summary-unmark-line)
	  (when (looking-at "^ *[0-9]+[^0-9]\\([^0-9]\\)")
	    (delete-region (match-beginning 1) (match-end 1)))
	  (goto-char (match-beginning 1))
	  (insert new-mark)
	  (if wl-summary-highlight	
	      (wl-highlight-summary-current-line)))
	(setq mlist (cdr mlist)))
      (setq wl-summary-buffer-temp-mark-list nil))))
    
(defun wl-summary-temp-mark-copy ()
  (interactive)
  (save-excursion
    (goto-char (point-min))
    (let ((regexp (concat "^" wl-summary-buffer-number-regexp "\\(\\*\\)"))
	  number
	  )
      ;; guess by first mark
      (when (re-search-forward regexp nil t)
	(let* ((msgid (cdr (assq (setq number (wl-summary-message-number))
				 (elmo-msgdb-get-number-alist
				  wl-summary-buffer-msgdb))))
	       (entity (assoc msgid
			      (elmo-msgdb-get-overview
			       wl-summary-buffer-msgdb)))
	       folder mlist)
	  (setq number 
		(wl-summary-copy
		 (setq folder (wl-summary-read-folder (wl-refile-guess entity)
						      "for copy"))))
	  (if number
	      (setq wl-summary-buffer-temp-mark-list
		    (delete number wl-summary-buffer-temp-mark-list)))
	  (while (re-search-forward regexp nil t)
	    (let (wl-summary-buffer-disp-msg)
	      (setq number (wl-summary-copy folder))
	      (if number
		  (setq wl-summary-buffer-temp-mark-list
			(delete number wl-summary-buffer-temp-mark-list)))))
	  ;; process invisible messages.
	  (setq mlist wl-summary-buffer-temp-mark-list)
	  (while mlist
	    (wl-append wl-summary-buffer-copy-list
		       (list (cons (car mlist) folder)))
	    (setq wl-summary-buffer-temp-mark-list
		  (delete (car mlist) wl-summary-buffer-temp-mark-list))
	    (setq mlist (cdr mlist))))))))

(defun wl-summary-temp-mark-refile ()
  (interactive)
  (save-excursion
    (goto-char (point-min))
    (let ((regexp (concat "^" wl-summary-buffer-number-regexp "\\(\\*\\)"))
	  number
	  )
      ;; guess by first mark
      (when (re-search-forward regexp nil t)
	(let* ((msgid (cdr (assq (setq number (wl-summary-message-number))
				 (elmo-msgdb-get-number-alist
				  wl-summary-buffer-msgdb))))
	       (entity (assoc msgid
			      (elmo-msgdb-get-overview
			       wl-summary-buffer-msgdb)))
	       folder mlist)
	  (setq number 
		(wl-summary-refile 
		 (setq folder (wl-summary-read-folder (wl-refile-guess entity)
						      "for refile"))))
	  (if number
	      (setq wl-summary-buffer-temp-mark-list
		    (delete number wl-summary-buffer-temp-mark-list)))
	  (while (re-search-forward regexp nil t)
	    (let (wl-summary-buffer-disp-msg)
	      (setq number (wl-summary-refile folder))
	      (if number
		  (setq wl-summary-buffer-temp-mark-list
			(delete number wl-summary-buffer-temp-mark-list)))))
	  ;; process invisible messages.
	  (setq mlist wl-summary-buffer-temp-mark-list)
	  (while mlist
	    (wl-append wl-summary-buffer-refile-list
		       (list (cons (car mlist) folder)))
	    (setq wl-summary-buffer-temp-mark-list
		  (delete (car mlist) wl-summary-buffer-temp-mark-list))
	    (setq mlist (cdr mlist))))))))

(defun wl-summary-temp-mark-mark-as-read ()
  (interactive)
  (save-excursion
    (goto-char (point-min))
    (let ((regexp (concat "^" wl-summary-buffer-number-regexp "\\(\\*\\)"))
	  (inhibit-read-only t)
	  (buffer-read-only nil)
	  number mlist)
      (while (re-search-forward regexp nil t)
	(let (wl-summary-buffer-disp-msg)
	  ;; delete temp-mark from buffer.
	  (delete-region (match-beginning 1) (match-end 1))
	  (insert " ")
	  (setq number (wl-summary-mark-as-read t))
	  (if wl-summary-highlight	
	      (wl-highlight-summary-current-line))
	  (if number
	      (setq wl-summary-buffer-temp-mark-list
		    (delete number wl-summary-buffer-temp-mark-list)))))
      (setq mlist wl-summary-buffer-temp-mark-list)
      (while mlist
	(wl-thread-msg-mark-as-read (car mlist))
	(setq wl-summary-buffer-temp-mark-list
	      (delete (car mlist) wl-summary-buffer-temp-mark-list))
	(setq mlist (cdr mlist)))
      (wl-summary-count-unread 
       (elmo-msgdb-get-mark-alist wl-summary-buffer-msgdb))
      (wl-summary-update-modeline))))

(defun wl-summary-mark-as-read (&optional notcrosses 
					  leave-server-side-mark-untouched
					  displayed)
  (interactive)
  (save-excursion
    (let* (eol
	   (inhibit-read-only t)
	   (buffer-read-only nil)
	   (folder wl-summary-buffer-folder-name)
	   (msgdb wl-summary-buffer-msgdb)
	   (mark-alist (elmo-msgdb-get-mark-alist msgdb))
	   ;;(number-alist (elmo-msgdb-get-number-alist msgdb))
	   (case-fold-search nil)
	   mark number unread
	   )
;      (end-of-line)
;      (setq eol (point))
      ;(re-search-backward "^ *[0-9]+..[0-9]+/[0-9]+" nil t) ; set cursor line
      (beginning-of-line)
      (if (looking-at
	   (format "^ *\\([0-9]+\\)[^0-9]\\(%s\\|%s\\|%s\\|%s\\).*$" 
		   (regexp-quote wl-summary-read-uncached-mark)
		   (regexp-quote wl-summary-unread-uncached-mark)
		   (regexp-quote wl-summary-unread-cached-mark)
		   (regexp-quote wl-summary-new-mark)))
	  (progn
	    (setq mark (wl-match-buffer 2))
	    (when mark
	      (cond 
	       ((string= mark wl-summary-new-mark) ; N
		(setq wl-summary-buffer-new-count 
		      (- wl-summary-buffer-new-count 1))
		(setq unread t)
		)
	       ((or (string= mark wl-summary-unread-uncached-mark) ; U
		    (string= mark wl-summary-unread-cached-mark))  ; !
		(setq wl-summary-buffer-unread-count 
		      (- wl-summary-buffer-unread-count 1))
		(setq unread t)))
	      (wl-summary-update-modeline)
	      (wl-folder-update-unread 
	       folder
	       (+ wl-summary-buffer-unread-count 
		  wl-summary-buffer-new-count))
	      )
	    (setq number (string-to-int (wl-match-buffer 1)))
	    ;; set server side mark...
	    (if (not leave-server-side-mark-untouched)
		(elmo-mark-as-read folder
				   (list number) msgdb))
	    (goto-char (match-end 2))
	    (delete-region (match-beginning 2) (match-end 2))
	    (insert " ")
	    (setq mark-alist
		  (elmo-msgdb-mark-set mark-alist number nil))
	    (elmo-msgdb-set-mark-alist msgdb mark-alist)
	    (wl-summary-set-mark-modified)
	    (if wl-summary-highlight
		(wl-highlight-summary-current-line nil nil t))
	    (if (not notcrosses)
		(wl-summary-set-crosspost nil (interactive-p)))
	    ))
      (set-buffer-modified-p nil)
      (if unread
	  (run-hooks 'wl-summary-unread-message-hook))
      number ;return value
      ))
  )

(defun wl-summary-mark-as-important ()
  (interactive)
  (if (eq (elmo-folder-get-type wl-summary-buffer-folder-name)
	  'internal)
      (error "Cannot process mark in this folder."))
  (save-excursion
    (let* (eol
	  (inhibit-read-only t)
	  (buffer-read-only nil)
	  (folder wl-summary-buffer-folder-name)
	  (msgdb wl-summary-buffer-msgdb)
	  (mark-alist (elmo-msgdb-get-mark-alist msgdb))
	  (number-alist (elmo-msgdb-get-number-alist msgdb))
	  message-id number
	  )
      (end-of-line)
      (setq eol (point))
      (re-search-backward (concat "^" wl-summary-buffer-number-regexp
				  "..../..") nil t) ; set cursor line
      (beginning-of-line)
      (if (re-search-forward "^ *\\([0-9]+\\)[^0-9]\\([^0-9]\\)" eol t)
	  (progn
	    (setq number (string-to-int (wl-match-buffer 1)))
	    (setq message-id (cdr (assq number number-alist)))
	    (if (string= wl-summary-important-mark
			 (wl-match-buffer 2))
		(progn
		  ;; server side mark
		  (elmo-unmark-important folder (list number) msgdb)
		  (elmo-msgdb-global-mark-delete message-id)
		  (delete-region (match-beginning 2) (match-end 2))
		  (insert " ")
		  (setq mark-alist
			(elmo-msgdb-mark-set mark-alist
					     number
					;message-id
					     nil)))
	      ;; server side mark
	      (elmo-mark-as-important folder (list number) msgdb)
	      (delete-region (match-beginning 2) (match-end 2))
	      (insert wl-summary-important-mark)
	      (setq mark-alist
		    (elmo-msgdb-mark-set mark-alist
					 ;(cdr (assoc 
					 ;(wl-match-buffer 1)
					 ;number-alist)
					 (string-to-int (wl-match-buffer 1))
					 wl-summary-important-mark))
	      ;; Force cache message!!
	      (save-match-data
		(unless (elmo-cache-exists-p message-id)
		  (elmo-force-cache-msg folder number message-id)))
	      (elmo-msgdb-global-mark-set message-id 
					  wl-summary-important-mark))
	    (elmo-msgdb-set-mark-alist msgdb mark-alist)
	    (wl-summary-set-mark-modified)
	    ))))
  (if wl-summary-highlight
      (wl-highlight-summary-current-line nil nil t))
  (set-buffer-modified-p nil)
  )

(defun wl-summary-overview-create-summary-line (msg 
						entity 
						parent-entity 
						depth
						mark-alist
						&optional 
						children-num
						temp-mark thr-entity
						subject-differ
						)
  (let ((wl-mime-charset wl-summary-buffer-mime-charset)
	(elmo-mime-charset wl-summary-buffer-mime-charset)
	no-parent before-indent
	from subject parent-raw-subject parent-subject
	mark line
	(elmo-lang wl-summary-buffer-weekday-name-lang)
	(children-num (if children-num (int-to-string children-num)))
	(thr-str ""))
    (if thr-entity
	(setq thr-str (wl-thread-make-indent-string thr-entity)))
    (if (string= thr-str "")
	(setq no-parent t)) ; no parent
    (if (and wl-summary-width 
	     wl-summary-indent-length-limit
	     (< wl-summary-indent-length-limit
		(string-width thr-str)))
	(setq thr-str (wl-set-string-width 
		       wl-summary-indent-length-limit
		       thr-str)))
    (setq from 
	  (wl-set-string-width 
	   (if children-num
	       (- wl-from-width (length children-num) 2)
	     wl-from-width)
	   (elmo-delete-char ?\n
			     (wl-summary-from-func-internal
			      (elmo-msgdb-overview-entity-get-from entity)))))
    (setq subject
	  (elmo-delete-char ?\n
			    (or (elmo-msgdb-overview-entity-get-subject 
				 entity)
				wl-summary-no-subject-message)))
    (setq parent-raw-subject 
	  (elmo-msgdb-overview-entity-get-subject parent-entity))
    (setq parent-subject 
	  (if parent-raw-subject 
	      (elmo-delete-char ?\n parent-raw-subject)))
    (setq mark (or (cadr (assq msg mark-alist)) " "))
    (setq line 
	  (concat 
	   (setq before-indent
		 (format (concat "%" 
				 (int-to-string 
				  wl-summary-buffer-number-column)
				 "s%s%s%s %s")
			 msg
			 (or temp-mark " ")
			 mark
			 (wl-summary-format-date
			  (elmo-msgdb-overview-entity-get-date entity))
			 (if thr-str thr-str "")))
	   (format "[%s ] %s"
		   (if children-num
		       (concat "+" children-num ": " from)
		     (concat " " from))
		   (if (or no-parent
			   (null parent-subject)
			   (not (wl-summary-subject-equal 
				 subject parent-subject)))
		       (wl-summary-subject-func-internal subject) ""))))
    (if wl-summary-width (setq line 
			       (wl-set-string-width 
				(- wl-summary-width 1) line)))
    (if wl-summary-highlight
	(wl-highlight-summary-line-string line 
					  (or temp-mark
					      mark)
					  thr-str (length before-indent)))
    line))
       
(defsubst wl-summary-proc-wday (wday-str year month mday)
  (save-match-data
    (if (string-match "\\([A-Z][a-z][a-z]\\).*" wday-str)
	(wl-match-string 1 wday-str)
      (elmo-date-get-week year month mday))))

(defmacro wl-summary-cursor-move-regex ()
  (` (let ((mark-alist
	    (if wl-plugged
		(cond ((eq wl-summary-move-order 'new)
		       (list
			(list
			 wl-summary-new-mark)
			(list
			 wl-summary-unread-uncached-mark
			 wl-summary-unread-cached-mark
			 wl-summary-important-mark)))
		      ((eq wl-summary-move-order 'unread)
		       (list
		       (list
			wl-summary-unread-uncached-mark
			wl-summary-unread-cached-mark
			wl-summary-new-mark)
		       (list
			wl-summary-important-mark)))
		      (t
		       (list
		       (list
			wl-summary-unread-uncached-mark
			wl-summary-unread-cached-mark
			wl-summary-new-mark
			wl-summary-important-mark))))
	      (cond ((eq wl-summary-move-order 'unread)
		     (list
		     (list
		      wl-summary-unread-cached-mark)
		     (list
		      wl-summary-important-mark)))
		    (t
		     (list
		     (list
		      wl-summary-unread-cached-mark
		      wl-summary-important-mark)))))))
       (mapcar
	(function
	 (lambda (mark-list)
	   (concat wl-summary-message-regexp
		   ".\\("
		   (mapconcat 'regexp-quote
			      mark-list
			      "\\|")
		   "\\)\\|"
		   wl-summary-message-regexp "\\*")))
	mark-alist))))

;;
;; Goto unread or important
;; 
(defun wl-summary-cursor-up (&optional hereto)
  (interactive "P")
  (if (and (not wl-summary-buffer-temp-mark-list)
	   (eq wl-summary-buffer-view 'thread))
      (progn
	(if (eobp)
	    (forward-line -1))
	(wl-thread-jump-to-prev-unread hereto))
    (if hereto
	(end-of-line)
      (beginning-of-line))
    (let ((case-fold-search nil)
	  regex-list)
      (setq regex-list (wl-summary-cursor-move-regex))
      (catch 'done
	(while regex-list
	  (when (re-search-backward
		 (car regex-list)
		 nil t nil)
	    (beginning-of-line)
	    (throw 'done t))
	  (setq regex-list (cdr regex-list)))
	(beginning-of-line)
	(throw 'done nil)))))

;;
;; Goto unread or important
;; returns t if next message exists in this folder.
(defun wl-summary-cursor-down (&optional hereto)
  (interactive "P")
  (if (and (not wl-summary-buffer-temp-mark-list)
	   (eq wl-summary-buffer-view 'thread))
      (wl-thread-jump-to-next-unread hereto)
    (if hereto
	(beginning-of-line)
      (end-of-line))
    (let ((case-fold-search nil)
	  regex-list)
      (setq regex-list (wl-summary-cursor-move-regex))
      (catch 'done
	(while regex-list
	  (when (re-search-forward
		 (car regex-list)
		 nil t nil)
	    (beginning-of-line)
	    (throw 'done t))
	  (setq regex-list (cdr regex-list)))
	(beginning-of-line)
	(throw 'done nil)))))

(defun wl-summary-save-view-cache (&optional keep-current-buffer)
  (save-excursion
    (let* ((dir (elmo-msgdb-expand-path wl-summary-buffer-folder-name))
	   (cache (expand-file-name wl-summary-cache-file dir))
	   (view (expand-file-name wl-summary-view-file dir))
	   (coding-system-for-write wl-cs-cache)
	   (output-coding-system wl-cs-cache)
	   (save-view wl-summary-buffer-view)
	   (tmp-buffer(get-buffer-create " *wl-summary-save-view-cache*"))
	   charset)
      (if (file-directory-p dir)
	  (); ok.
	(if (file-exists-p dir)
	    (error "File %s already exists." dir)
	  (elmo-make-directory dir)))
      (if (eq save-view 'thread)
	  (wl-thread-save-entity dir))
      (unwind-protect
	  (progn
	    (when (file-writable-p cache)
	      (if keep-current-buffer
		  (progn
		    (save-excursion
		      (set-buffer tmp-buffer)
		      (erase-buffer))
		    (setq charset wl-summary-buffer-mime-charset)
		    (copy-to-buffer tmp-buffer (point-min) (point-max))
		    (save-excursion
		      (set-buffer tmp-buffer)
		      (widen)
		      (encode-mime-charset-region 
		       (point-min) (point-max) charset)
		      (as-binary-output-file
		       (write-region (point-min)
				     (point-max) cache nil 'no-msg))))
		(let (buffer-read-only)
		  (widen)
		  (encode-mime-charset-region (point-min) (point-max)
					      wl-summary-buffer-mime-charset)
		  (as-binary-output-file
		   (write-region (point-min) (point-max) cache nil 'no-msg)))))
	    (when (file-writable-p view) ; 'thread or 'sequence
	      (save-excursion
		(set-buffer tmp-buffer)
		(erase-buffer)
		(prin1 save-view tmp-buffer)
		(princ "\n" tmp-buffer)
		(write-region (point-min) (point-max) view nil 'no-msg))))
	;; kill tmp buffer.
	(kill-buffer tmp-buffer)))))

(defsubst wl-summary-get-sync-range (folder)
  (intern (or (wl-get-assoc-list-value 
	       wl-folder-sync-range-alist
	       folder) 
	      wl-default-sync-range)))

;; redefined for wl-summary-sync-update
(defun wl-summary-input-range (folder)
  "returns update or all or rescan."
  ;; for the case when parts are expanded in the bottom of the folder
  (let ((input-range-list '("update" "all" "rescan" "first:" "last:" 
			    "no-sync"))
	(default (or (wl-get-assoc-list-value 
		      wl-folder-sync-range-alist
		      folder) 
		     wl-default-sync-range))
	range)
    (setq range
	  (completing-read (format "Range (%s): " default)
			   (mapcar
			    (function (lambda (x) (cons x x)))
			    input-range-list)))
    (if (string= range "")
	default
      range)))

(defun wl-summary-toggle-disp-folder (&optional arg)
  (interactive)
  (let (fld-buf fld-win
	(view-message-buffer (wl-message-get-buffer-create))
	(cur-buf (current-buffer))
	summary-win
	)
    (cond 
     ((eq arg 'on)
      (setq wl-summary-buffer-disp-folder t)
      ;; hide your folder window
      (if (setq fld-buf (get-buffer wl-folder-buffer-name))
	  (if (setq fld-win (get-buffer-window fld-buf))
	      (delete-window fld-win)))
      )
     ((eq arg 'off)
      (setq wl-summary-buffer-disp-folder nil)
      ;; hide your wl-message window!
      (wl-select-buffer view-message-buffer)
      (delete-window)
      (select-window (get-buffer-window cur-buf))
      ;; display wl-folder window!!
      (if (setq fld-buf (get-buffer wl-folder-buffer-name))
	  (if (setq fld-win (get-buffer-window fld-buf))
	      ;; folder win is already displayed.
	      (select-window fld-win)
	    ;; folder win is not displayed.
	    (switch-to-buffer fld-buf))
	;; no folder buf
	(wl-folder))
      ;; temporarily delete summary-win.
      (if (setq summary-win (get-buffer-window cur-buf))
	  (delete-window summary-win))
      (split-window-horizontally wl-folder-window-width)
      (other-window 1)
      (switch-to-buffer cur-buf))
     (t
      (if (setq fld-buf (get-buffer wl-folder-buffer-name))
	  (if (setq fld-win (get-buffer-window fld-buf))      
	      (setq wl-summary-buffer-disp-folder nil)
	    (setq wl-summary-buffer-disp-folder t)))
      (if (not wl-summary-buffer-disp-folder)
	  (progn
	    ;; hide your folder window
	    (if (setq fld-buf (get-buffer wl-folder-buffer-name))
		(if (setq fld-win (get-buffer-window fld-buf))
		    (delete-window fld-win)))
	    (run-hooks 'wl-summary-toggle-disp-folder-off-hook))
	(save-excursion
	  ;; hide message window
	  (let ((mes-win (get-buffer-window view-message-buffer)))
	    (if mes-win (delete-window mes-win)))
	  (select-window (get-buffer-window cur-buf))
	  ;; display wl-folder window!!
	  (if (setq fld-buf (get-buffer wl-folder-buffer-name))
	      (if (setq fld-win (get-buffer-window fld-buf))
		  ;; folder win is already displayed.
		  (select-window fld-win)
		;; folder win is not displayed...occupy all.
		(switch-to-buffer fld-buf))
	    ;; no folder buf
	    (wl-folder))
	  (split-window-horizontally wl-folder-window-width)
	  (other-window 1)
	  (switch-to-buffer cur-buf)
	  (run-hooks 'wl-summary-toggle-disp-folder-on-hook)
	  ))))))
  
(defun wl-summary-toggle-disp-msg (&optional arg)
  (interactive)
  (let (fld-buf fld-win
	(view-message-buffer (wl-message-get-buffer-create))
	(cur-buf (current-buffer))
	summary-win
	)
    (cond 
     ((eq arg 'on)
      (setq wl-summary-buffer-disp-msg t)
      ;; hide your folder window
      (if (and (not wl-stay-folder-window)
	       (setq fld-buf (get-buffer wl-folder-buffer-name)))
	  (if (setq fld-win (get-buffer-window fld-buf))
	      (delete-window fld-win)))
      )
     ((eq arg 'off)
      (wl-delete-all-overlays)
      (setq wl-summary-buffer-disp-msg nil)
      (save-excursion
        (wl-select-buffer view-message-buffer)
        (delete-window)
        (select-window (get-buffer-window cur-buf))
        (run-hooks 'wl-summary-toggle-disp-off-hook)
        ))
     (t
      (if (get-buffer-window view-message-buffer) ; already displayed
	  (setq wl-summary-buffer-disp-msg nil)
	(setq wl-summary-buffer-disp-msg t))
      (if wl-summary-buffer-disp-msg
	  (progn
	    (wl-summary-redisplay)
	    ;; hide your folder window
;;		(setq fld-buf (get-buffer wl-folder-buffer-name))
;;		(if (setq fld-win (get-buffer-window fld-buf))
;;		    (delete-window fld-win)))	    
	    (run-hooks 'wl-summary-toggle-disp-on-hook))
	(wl-delete-all-overlays)
	(save-excursion
	  (wl-select-buffer view-message-buffer)
	  (delete-window)
	  (select-window (get-buffer-window cur-buf))
	  (run-hooks 'wl-summary-toggle-disp-off-hook)
	  )
					;(switch-to-buffer cur-buf))
	)))))

(defun wl-summary-next-line-content ()
  (interactive)
  (let ((cur-buf (current-buffer)))
    (wl-summary-toggle-disp-msg 'on)
    (when (wl-summary-set-message-buffer-or-redisplay)
      (set-buffer cur-buf)
      (wl-message-next-page 1))))

(defun wl-summary-prev-line-content ()
  (interactive)
  (let ((cur-buf (current-buffer)))
    (wl-summary-toggle-disp-msg 'on)
    (when (wl-summary-set-message-buffer-or-redisplay)
      (set-buffer cur-buf)
      (wl-message-prev-page 1))))

(defun wl-summary-next-page ()
  (interactive)
  (wl-message-next-page))

(defun wl-summary-prev-page ()
  (interactive)
  (wl-message-prev-page))
  
(defun wl-summary-set-message-buffer-or-redisplay ()
  ;; if current message is not displayed, display it.
  ;; return t if exists.
  (let ((folder wl-summary-buffer-folder-name)
	(number (wl-summary-message-number))
	cur-folder cur-number message-last-pos
	(view-message-buffer (wl-message-get-buffer-create)))
    (save-excursion
      (set-buffer view-message-buffer)
      (setq cur-folder wl-message-buffer-cur-folder)
      (setq cur-number wl-message-buffer-cur-number))
    (if (and (string= folder (or cur-folder ""))
	     (eq number (or cur-number 0)))
	(progn
	  (set-buffer view-message-buffer)
	  t)
      (wl-summary-redisplay-internal folder number)
      nil)))

(defun wl-summary-temp-mark-forward (&optional arg)
  (interactive "P")
  (let ((mlist wl-summary-buffer-temp-mark-list)
	(summary-buf (current-buffer))
	start-point
	draft-buf)
    (wl-summary-jump-to-msg (car mlist))
    (wl-summary-forward)
    (setq start-point (point))
    (setq draft-buf (current-buffer))
    (setq mlist (cdr mlist))
    (save-window-excursion
      (while mlist
	(set-buffer summary-buf)
	(wl-summary-jump-to-msg (car mlist))
	(wl-summary-redisplay)
	(set-buffer draft-buf)
	(goto-char (point-max))
	(funcall wl-insert-mail-func)
	(setq mlist (cdr mlist)))
      (goto-char start-point)
      (save-excursion
	(set-buffer summary-buf)
	(wl-summary-delete-all-temp-marks)))))
  
(defun wl-summary-temp-mark-reply-with-citation (&optional arg)
  (interactive "P")
  (let ((mlist (nreverse wl-summary-buffer-temp-mark-list))
	(summary-buf (current-buffer))
	start-point
	draft-buf)
    (wl-summary-jump-to-msg (car mlist))
    (wl-summary-reply arg t)
    (goto-char (point-max))
    (setq start-point (point))
    (setq draft-buf (current-buffer))
    (save-window-excursion
      (while mlist
	(set-buffer summary-buf)
	(wl-summary-jump-to-msg (car mlist))
	(wl-summary-redisplay)
	(set-buffer draft-buf)
	(goto-char (point-max))
	(wl-draft-yank-original)
	(setq mlist (cdr mlist)))
      (goto-char start-point)
      (save-excursion
	(set-buffer summary-buf)
	(wl-summary-delete-all-temp-marks)))
    (run-hooks 'wl-mail-setup-hook)))

(defun wl-summary-reply-with-citation (&optional arg)
  (interactive "P")
  (unwind-protect
      (let (wl-draft-prepared-config-alist)
	(wl-summary-reply arg t))
    (goto-char (point-max))
    (wl-draft-yank-original)
    (if wl-draft-prepared-config-alist
	(wl-draft-prepared-config-exec mail-reply-buffer))
    (run-hooks 'wl-mail-setup-hook)
    ))

(defun wl-summary-jump-to-msg-by-message-id (&optional id)
  (interactive)
  (let ((mes-id (wl-string (or id (read-from-minibuffer "Message-ID: "))))
	(number-alist (elmo-msgdb-get-number-alist wl-summary-buffer-msgdb))
	num otherfld)
    (if (setq num (car (rassoc mes-id number-alist)))
	(wl-thread-jump-to-msg num)
      (if elmo-use-database
	  ;; for XEmacs!
	  (if (setq otherfld (elmo-database-msgid-get mes-id))
	      (let (wl-auto-select-first)
		(if (y-or-n-p 
		     (format 
		      "Message was found in the folder \"%s\". Jump to it? "
		      (car otherfld)))
		    (unwind-protect
			(wl-summary-goto-folder-subr 
			 (car otherfld) 'no-sync nil nil t)
		      (wl-folder-set-current-entity-id
		       (wl-folder-get-entity-id
			(wl-folder-search-entity-by-name (car otherfld)
							 wl-folder-entity
							 'folder)))
		      (wl-thread-jump-to-msg (nth 1 otherfld)))))
	    (message "No message with id \"%s\"." mes-id))
	(message "No message with id \"%s\"in this folder." mes-id)))))

(defun wl-summary-jump-to-parent-message (arg)
  (interactive "P")
  (let ((cur-buf (current-buffer))
	(regexp "\\(<[^<>]*>\\)[ \t]*$")
	(i -1) ;; xxx
	msg-id ref-list ref irt)
    (wl-summary-set-message-buffer-or-redisplay)
    (set-buffer (wl-message-get-original-buffer))
    (message "Searching parent message...")
    (setq ref (std11-field-body "References")
	  irt (std11-field-body "In-Reply-To"))
    (cond
     ((and arg (not (numberp arg)) ref (not (string= ref ""))
	   (string-match regexp ref))
      ;; The first message of the thread.
      (setq msg-id (wl-match-string 1 ref)))
     ;; "In-Reply-To:" has only one msg-id.
     ((and irt (not (string= irt ""))
	   (string-match regexp irt))
      (setq msg-id (wl-match-string 1 irt)))
     ((and (or (null arg) (numberp arg)) ref (not (string= ref ""))
	   (string-match regexp ref))
      ;; "^" searching parent, "C-u 2 ^" looking for grandparent.
      (while (string-match regexp ref)
	(setq ref-list
	      (append (list
		       (wl-match-string 1 ref))
		      ref-list))
	(setq ref (substring ref (match-end 0)))
	(setq i (1+ i)))
      (setq msg-id
            (if (null arg) (nth 0 ref-list) ;; previous
              (if (<= arg i) (nth (1- arg) ref-list)
                (nth i ref-list))))))
    (set-buffer cur-buf)
    (if (null msg-id)
        (error "No parent message!")
      (wl-summary-jump-to-msg-by-message-id msg-id)
      (wl-summary-redisplay)
      (message "Searching parent message...done"))))

(defun wl-summary-reply (&optional arg without-setup-hook)
  "Reply to current message. Default is \"wide\" reply.
Reply to author if invoked with argument."
  (interactive "P")
  (let ((folder wl-summary-buffer-folder-name)
	(number (wl-summary-message-number))
	mes-buf
	)
    (wl-summary-redisplay-internal folder number)
    (wl-select-buffer (get-buffer wl-message-buf-name))
    (set-buffer wl-message-buf-name)
    (goto-char (point-min))
    (split-window-vertically)
    (other-window 1)
    (when (setq mes-buf (wl-message-get-original-buffer))
      (wl-draft-reply mes-buf (not arg))
      (unless without-setup-hook
	(run-hooks 'wl-mail-setup-hook))
      )))

(defun wl-summary-write-current-newsgroup ()
  (interactive)
  (let ((folder wl-summary-buffer-folder-name)
	newsgroup)
    (if (not (eq 'nntp (elmo-folder-get-type folder)))
	(error "%s is not newsgroup." folder)
      (setq newsgroup (nth 1 (elmo-folder-get-spec folder)))
      (wl-draft nil nil nil nil nil newsgroup))
    ))

(defun wl-summary-forward ()
  (interactive)
  (let ((folder wl-summary-buffer-folder-name)
	(number (wl-summary-message-number))
	(summary-buf (current-buffer))
	entity subject num
	)
    (wl-summary-redisplay-internal folder number)
    (wl-select-buffer (get-buffer wl-message-buf-name))
    (split-window-vertically)
    (other-window 1)
    ;; get original subject.
    (if summary-buf
	(save-excursion 
	  (set-buffer summary-buf)
	  (setq num (wl-summary-message-number))
	  (setq entity (assoc (cdr (assq num 
					 (elmo-msgdb-get-number-alist 
					  wl-summary-buffer-msgdb)))
			      (elmo-msgdb-get-overview 
			       wl-summary-buffer-msgdb)))
	  (and entity
	       (setq subject 
		     (or (elmo-msgdb-overview-entity-get-subject entity) 
			 "")))))
    (wl-draft-forward subject summary-buf)
    (run-hooks 'wl-mail-setup-hook)))

(defun wl-summary-click (e)
  (interactive "e")
  (mouse-set-point e)
  (wl-summary-read))

(defun wl-summary-read ()
  (interactive)
  (let ((folder wl-summary-buffer-folder-name)
	(number (wl-summary-message-number))
	cur-folder cur-number message-last-pos
	(view-message-buffer (get-buffer-create wl-message-buf-name)))
    (save-excursion
      (set-buffer view-message-buffer)
      (setq cur-folder wl-message-buffer-cur-folder)
      (setq cur-number wl-message-buffer-cur-number))
    (wl-summary-toggle-disp-msg 'on)
    (if (and (string= folder cur-folder)
	     (eq number cur-number))
	(progn
	  (if (wl-summary-next-page)
	      (wl-summary-down t)))
;	    (wl-summary-scroll-up-content)))
      (wl-summary-redisplay-internal folder number))))

(defmacro wl-summary-move-regex ()
  (` (if (eq wl-summary-move-order 'unread)
	 (list
	  (format "^%s.\\(%s\\| \\)"
		  wl-summary-buffer-number-regexp
		  (regexp-quote wl-summary-unread-cached-mark))
	  (format "^%s.\\(%s\\| \\)"
		  wl-summary-buffer-number-regexp
		  (regexp-quote wl-summary-important-mark)))
       (list
	(format "^.%s\\(%s\\|%s\\| \\)"
		wl-summary-buffer-number-regexp
		(regexp-quote wl-summary-unread-cached-mark)
		(regexp-quote wl-summary-important-mark))))))

(defun wl-summary-prev (&optional interactive)
  (interactive)
  (if wl-summary-move-direction-toggle
      (setq wl-summary-move-direction-downward nil))
  (when (not wl-plugged)
    (let (bob regex-list)
      (beginning-of-line)
      (setq bob (bobp))
      (setq regex-list (wl-summary-move-regex))
      (catch 'done
	(while regex-list
	  (if (re-search-backward (car regex-list) nil t)
	      (throw 'done t))
	  (setq regex-list (cdr regex-list))))
      (if (not bob) (forward-line 1))))
  (if (>= (forward-line -1) 0)
      (progn
	(if wl-summary-buffer-disp-msg
	    (wl-summary-redisplay)))
    (if (or interactive
	    (interactive-p))
	(let (next-entity finfo)
	  (when wl-auto-select-next
	    (progn
	      (setq next-entity (wl-summary-get-prev-folder))
	      (if next-entity
		  (setq finfo (wl-folder-get-entity-info next-entity)))))
	  (wl-ask-folder 
	   '(lambda () (wl-summary-next-folder-or-exit next-entity))
	   (format
	    "No more unread messages. Type SPC to go to %s."
	    (or (and next-entity
		     (concat 
		      (elmo-replace-in-string next-entity "%" "%%")
		      (if (null (car finfo))
			  " (? new/? unread)"
			(format 
			 " (%d new/%d unread)" 
			 (nth 0 finfo)
			 (+ (nth 0 finfo)
			    (nth 1 finfo))))))
		"folder mode")))))))

(defun wl-summary-next (&optional interactive)
  (interactive)
  (if wl-summary-move-direction-toggle
      (setq wl-summary-move-direction-downward t))
  (if wl-plugged
      (forward-line 1)
    (let (regex-list)
      (end-of-line)
      (setq regex-list (wl-summary-move-regex))
      (catch 'done
	(while regex-list
	  (if (re-search-forward (car regex-list) nil t)
	      (throw 'done t))
	  (setq regex-list (cdr regex-list)))
	(forward-line 1))
      (beginning-of-line)))
  (if (wl-summary-message-number)
      (if wl-summary-buffer-disp-msg
	  (wl-summary-redisplay))
    (if (or interactive
	    (interactive-p))
	(let (next-entity finfo)
	  (when wl-auto-select-next
	    (progn
	      (setq next-entity (wl-summary-get-next-folder))
	      (if next-entity
		  (setq finfo (wl-folder-get-entity-info next-entity)))))
	  (wl-ask-folder 
	   '(lambda () (wl-summary-next-folder-or-exit next-entity))
	   (format
	    "No more unread messages. Type SPC to go to %s."
	    (or (and next-entity
		     (concat 
		      (elmo-replace-in-string next-entity "%" "%%")
		      (if (null (car finfo))
			  " (? new/? unread)"
			(format 
			 " (%d new/%d unread)" 
			 (nth 0 finfo)
			 (+ (nth 0 finfo)
			    (nth 1 finfo))))))
		"folder mode")))))))

(defun wl-summary-up (&optional interactive)
  (interactive)
  (if wl-summary-move-direction-toggle
      (setq wl-summary-move-direction-downward nil))
  (if (wl-summary-cursor-up)
      (if wl-summary-buffer-disp-msg
	  (wl-summary-redisplay))
    (if (or interactive
	    (interactive-p))
	(let (next-entity finfo)
	  (when wl-auto-select-next
	    (progn
	      (setq next-entity (wl-summary-get-prev-unread-folder))
	      (if next-entity
		  (setq finfo (wl-folder-get-entity-info next-entity)))))
	  (wl-ask-folder 
	   '(lambda () (wl-summary-next-folder-or-exit next-entity))
	   (format
	    "No more unread messages. Type SPC to go to %s."
	    (or (and next-entity
		     (concat 
		      (elmo-replace-in-string next-entity "%" "%%")
		      (format 
		       " (%d new/%d unread)" 
		       (nth 0 finfo)
		       (+ (nth 0 finfo)
			  (nth 1 finfo)))))
		"folder mode")))))))

(defun wl-summary-get-prev-folder ()
  (let ((folder-buf (get-buffer wl-folder-buffer-name))
	last-entity cur-id)
    (when folder-buf
      (setq cur-id (save-excursion (set-buffer folder-buf)
				   wl-folder-buffer-cur-entity-id))
      (wl-folder-get-prev-folder cur-id))))

(defun wl-summary-get-next-folder ()
  (let ((folder-buf (get-buffer wl-folder-buffer-name))
	cur-id)
    (when folder-buf
      (setq cur-id (save-excursion (set-buffer folder-buf)
				   wl-folder-buffer-cur-entity-id))
      (wl-folder-get-next-folder cur-id))))

(defun wl-summary-get-next-unread-folder ()
  (let ((folder-buf (get-buffer wl-folder-buffer-name))
	cur-id)
    (when folder-buf
      (setq cur-id (save-excursion (set-buffer folder-buf)
				   wl-folder-buffer-cur-entity-id))
      (wl-folder-get-next-folder cur-id 'unread))))

 (defun wl-summary-get-prev-unread-folder ()
  (let ((folder-buf (get-buffer wl-folder-buffer-name))
	cur-id)
    (when folder-buf
      (setq cur-id (save-excursion (set-buffer folder-buf)
				   wl-folder-buffer-cur-entity-id))
      (wl-folder-get-prev-folder cur-id 'unread))))

(defun wl-summary-down (&optional interactive)
  (interactive)
  (if wl-summary-move-direction-toggle
      (setq wl-summary-move-direction-downward t))
  (if (wl-summary-cursor-down)
      (if wl-summary-buffer-disp-msg
	  (wl-summary-redisplay))
    (if (or interactive
	    (interactive-p))
	(let (next-entity finfo)
	  (when wl-auto-select-next
	    (setq next-entity (wl-summary-get-next-unread-folder))
	    (if next-entity
		(setq finfo (wl-folder-get-entity-info next-entity))))
	  (wl-ask-folder 
	   '(lambda () (wl-summary-next-folder-or-exit next-entity))
	   (format
	    "No more unread messages. Type SPC to go to %s."
	    (or (and next-entity
		     (concat 
		      (elmo-replace-in-string next-entity "%" "%%")
		      (format 
		       " (%d new/%d unread)" 
		       (nth 0 finfo)
		       (+ (nth 0 finfo)
			  (nth 1 finfo)))))
		"folder mode")))))))

(defun wl-summary-goto-last-displayed-msg ()
  (interactive)
  (when wl-summary-buffer-last-displayed-msg 
    (wl-summary-jump-to-msg wl-summary-buffer-last-displayed-msg)
    (if wl-summary-buffer-disp-msg
	(wl-summary-redisplay))
    ))

(defun wl-summary-redisplay (&optional arg)
  (interactive "P")
  (wl-summary-redisplay-internal nil nil arg)
  )

(defsubst wl-summary-redisplay-internal (&optional folder number force-reload)
  (interactive)
  (let* ((msgdb wl-summary-buffer-msgdb)
	 (fld (or folder wl-summary-buffer-folder-name))
	 (num (or number (wl-summary-message-number)))
	 (wl-mime-charset      wl-summary-buffer-mime-charset)
	 (default-mime-charset wl-summary-buffer-mime-charset)
	 (wl-message-redisplay-func 
	  wl-summary-buffer-message-redisplay-func)
	 fld-buf fld-win thr-entity)
    (if (and wl-thread-open-reading-thread
	     (eq wl-summary-buffer-view 'thread)
	     (not (wl-thread-entity-get-opened 
		   (setq thr-entity (wl-thread-get-entity 
				     num))))
	     (wl-thread-entity-get-children thr-entity))
	(wl-thread-force-open))
    (if num
	(progn
	  (setq wl-summary-buffer-disp-msg t)
	  (setq wl-summary-buffer-last-displayed-msg 
		wl-summary-buffer-current-msg)
	  ;; hide folder window
	  (if (and (not wl-stay-folder-window)
		   (setq fld-buf (get-buffer wl-folder-buffer-name)))
	      (if (setq fld-win (get-buffer-window fld-buf))
		  (delete-window fld-win)))	  
	  (if (wl-message-redisplay fld num 'mime msgdb force-reload) 
	      (wl-summary-mark-as-read nil t t) ;; displayed
	    )
	  (setq wl-summary-buffer-current-msg num)
	  (if wl-summary-recenter
	      (recenter (/ (- (window-height) 2) 2)))
	  (wl-highlight-summary-displaying)
	  (run-hooks 'wl-summary-redisplay-hook))
      (message "No message to display.")
      )))

(defun wl-summary-redisplay-no-mime (&optional folder number)
  (interactive)
  (let* ((msgdb wl-summary-buffer-msgdb)
	 (fld (or folder wl-summary-buffer-folder-name))
	 (num (or number (wl-summary-message-number)))
	 (wl-mime-charset      wl-summary-buffer-mime-charset)
	 (default-mime-charset wl-summary-buffer-mime-charset)	 
	 wl-break-pages)
    (if num
	(progn
	  (wl-normal-message-redisplay fld num 'no-mime msgdb)
	  (wl-summary-mark-as-read nil nil t)
	  (run-hooks 'wl-summary-redisplay-hook))
      (message "No message to display.")
      (wl-ask-folder 'wl-summary-exit
		     "No more messages. Type SPC to go to folder mode.")
      )))

(defun wl-summary-redisplay-all-header (&optional folder number)
  (interactive)
  (let* ((msgdb wl-summary-buffer-msgdb)
	 (fld (or folder wl-summary-buffer-folder-name))
	 (num (or number (wl-summary-message-number)))
	 (wl-mime-charset      wl-summary-buffer-mime-charset)
	 (default-mime-charset wl-summary-buffer-mime-charset)	 
	 (wl-message-redisplay-func wl-summary-buffer-message-redisplay-func)
	 )
    (if num
	(progn
	  (if (wl-message-redisplay fld num 'all-header msgdb); t if displayed.
	      (wl-summary-mark-as-read nil nil t))
	  (run-hooks 'wl-summary-redisplay-hook))
      (message "No message to display."))))
	 
(defun wl-summary-jump-to-current-message ()
  (interactive)
  (let (message-buf message-win)
    (if (setq message-buf (get-buffer wl-message-buf-name))
	(if (setq message-win (get-buffer-window message-buf))
	    (select-window message-win)
	  (wl-select-buffer (get-buffer wl-message-buf-name)))
      (wl-summary-redisplay)
      (wl-select-buffer (get-buffer wl-message-buf-name)))
    (goto-char (point-min))))

(defun wl-summary-cancel-message ()
  "Cancel an article on news."
  (interactive)
  (wl-summary-set-message-buffer-or-redisplay)
  (let (message-buf)
    (if (setq message-buf (wl-message-get-original-buffer))
	(set-buffer message-buf)))
  (unless (wl-message-news-p)
    (error "This is not a news article; canceling is impossible"))
  (when (yes-or-no-p "Do you really want to cancel this article? ")
    (let (from newsgroups message-id distribution buf)
      (save-excursion
	(setq from (std11-field-body "from")
	      newsgroups (std11-field-body "newsgroups")
	      message-id (std11-field-body "message-id")
	      distribution (std11-field-body "distribution"))
	;; Make sure that this article was written by the user.
	(unless (string=
		 (wl-address-header-extract-address
		  (car (wl-parse-addresses from)))
		 (wl-address-header-extract-address wl-from))
	  (error "This article is not yours"))
	;; Make control message.
	(setq buf (set-buffer (get-buffer-create " *message cancel*")))
	(buffer-disable-undo (current-buffer))
	(erase-buffer)
	(insert "Newsgroups: " newsgroups "\n"
		"From: " (wl-address-header-extract-address
			  wl-from) "\n"
		"Subject: cmsg cancel " message-id "\n"
		"Control: cancel " message-id "\n"
		(if distribution
		    (concat "Distribution: " distribution "\n")
		  "")
		mail-header-separator "\n"
		wl-summary-cancel-message)
	(message "Canceling your message...")
	(wl-draft-raw-send t t) ; kill when done, force-pre-hooks.
	(message "Canceling your message...done")))))

(defun wl-summary-save ()
  (interactive)
  (wl-summary-redisplay)
  (set-buffer (wl-message-get-original-buffer))
  (let ((filename (read-file-name "Save to file: " wl-tmp-dir))
	(mmelmo-force-fetch-entire-message t))
    (if (file-exists-p filename)
	(if (y-or-n-p "file already exists. override it?")
	    (write-region (point-min) (point-max) filename))
      (write-region (point-min) (point-max) filename))))

;; mew-summary-pipe-message()
(defun wl-summary-pipe-message (prefix command)
  "Send this message via pipe."
  (interactive
   (list current-prefix-arg
	 (read-string "Shell command on message: " wl-summary-shell-command-last)))
  (if (y-or-n-p "Send this message to pipe? ")
      (save-excursion
	(wl-summary-set-message-buffer-or-redisplay)
	(set-buffer (wl-message-get-original-buffer))
	(if (string= command "")
	    (setq command wl-summary-shell-command-last))
	(goto-char (point-min)) ; perhaps this line won't be necessary
	(if prefix
	    (search-forward "\n\n"))
	(shell-command-on-region (point) (point-max) command nil)
	(setq wl-summary-shell-command-last command))))

(defun wl-summary-print-message (&optional arg)
  (interactive "P")
  (save-excursion
    (wl-summary-set-message-buffer-or-redisplay)
    (if (or (not (interactive-p))
	    (y-or-n-p "Print ok?"))
	(progn
	  (let* ((message-buffer (get-buffer wl-message-buf-name))
		 ;; (summary-buffer (get-buffer wl-summary-buffer-name))
		 (buffer (generate-new-buffer " *print*")))
	    (set-buffer message-buffer)
	    (copy-to-buffer buffer (point-min) (point-max))
	    (set-buffer buffer)
	    (funcall wl-print-buffer-func)
	    (kill-buffer buffer)))
      (message ""))))

(defun wl-summary-print-message-with-ps-print (&optional filename)
  (interactive (list (ps-print-preprint current-prefix-arg)))
  (if (or (not (interactive-p))
	  (y-or-n-p "Print ok?"))
      (let ((summary-buffer (current-buffer)))
	(save-excursion
	  (wl-summary-set-message-buffer-or-redisplay)
	  (let* ((message-buffer (get-buffer wl-message-buf-name))
		 (buffer (generate-new-buffer " *print*"))
		 (entity (progn
			   (set-buffer summary-buffer)
			   (assoc (cdr (assq 
					(wl-summary-message-number)
					(elmo-msgdb-get-number-alist 
					 wl-summary-buffer-msgdb)))
				  (elmo-msgdb-get-overview 
				   wl-summary-buffer-msgdb))))
		 (wl-ps-subject
		  (and entity
		       (or (elmo-msgdb-overview-entity-get-subject entity) 
			   "")))
		 (wl-ps-from
		  (and entity
		       (or (elmo-msgdb-overview-entity-get-from entity) "")))
		 (wl-ps-date
		  (and entity
		       (or (elmo-msgdb-overview-entity-get-date entity) ""))))
	    (run-hooks 'wl-ps-preprint-hook)
	    (set-buffer message-buffer)
	    (copy-to-buffer buffer (point-min) (point-max))
	    (set-buffer buffer)
	    (unwind-protect
		(let ((ps-left-header
		       (list (concat "(" wl-ps-subject ")")
			     (concat "(" wl-ps-from ")")))
		      (ps-right-header 
		       (list "/pagenumberstring load" 
			     (concat "(" wl-ps-date ")"))))
		  (run-hooks 'wl-ps-print-hook)
		  (funcall wl-ps-print-buffer-func filename))
	      (kill-buffer buffer)))))
    (message "")))
  
(if (featurep 'ps-print) ; ps-print is available.
    (fset 'wl-summary-print-message 'wl-summary-print-message-with-ps-print))

(defun wl-summary-folder-info-update ()
  (let ((folder (wl-string wl-summary-buffer-folder-name))
	(num-db (elmo-msgdb-get-number-alist
		 wl-summary-buffer-msgdb)))
    (wl-folder-set-folder-updated folder
				  (list 0
					(+ wl-summary-buffer-unread-count 
					   wl-summary-buffer-new-count)
					(length num-db)))
    ))

(defun wl-summary-set-crosspost (&optional type redisplay)
  (let* ((number (wl-summary-message-number))
	 (spec (elmo-folder-number-get-spec wl-summary-buffer-folder-name
					    number))
	 (folder (nth 1 spec))
	 crosspost-folders
	 message-buf newsgroups message-id)
    (when (eq (car spec) 'nntp)
      (if redisplay
	  (wl-summary-redisplay))
      (save-excursion
	(if (setq message-buf (wl-message-get-original-buffer))
	    (set-buffer message-buf))
	(setq newsgroups (std11-field-body "newsgroups")))
      (when newsgroups
	(let* ((msgdb wl-summary-buffer-msgdb)
	       (num-db (elmo-msgdb-get-number-alist msgdb)))
	  (setq message-id (cdr (assq number num-db)))
	  (setq crosspost-folders (wl-parse-newsgroups newsgroups))
	  (if (> (length crosspost-folders) 1)
	      (elmo-crosspost-message-set message-id
					  (delete folder crosspost-folders)
					  type))
	  )))))

(defun wl-summary-is-crosspost-folder (spec-list fld-list)
  (let (fld)
    (catch 'found
      (while spec-list
	(if (and (eq (caar spec-list) 'nntp)
		 (member (setq fld (nth 1 (car spec-list))) fld-list))
	    (throw 'found fld))
	(setq spec-list (cdr spec-list))))))

(defun wl-summary-update-crosspost ()
  (let* ((msgdb wl-summary-buffer-msgdb) 
	 (number-alist (elmo-msgdb-get-number-alist msgdb))
	 (spec-list (elmo-folder-get-primitive-spec-list
		     (wl-string wl-summary-buffer-folder-name)))
	 (alist elmo-crosspost-message-alist)
	 (crossed 0)
	 fld num)
    (when (assq 'nntp spec-list)
      (while alist
	(when (and
	       (setq fld
		     (wl-summary-is-crosspost-folder
		      spec-list
		      (nth 1 (car alist))))
	       (setq num (car (rassoc (caar alist) number-alist))))
	  (if (wl-thread-jump-to-msg num)
	      (progn
		(if (eq (nth 2 (car alist)) 'delete)
		    (wl-summary-delete);; wl-summary-delete-exec (?)
		  (wl-summary-mark-as-read t))
		(setq crossed (1+ crossed)))
	    ;; in closed thread?
	    nil)
	  (elmo-crosspost-message-delete (caar alist)
					 fld))
	(setq alist (cdr alist))))
    (if (> crossed 0)
	crossed)))

(defun wl-summary-pack-number (&optional arg)
  (interactive "P")
  (setq wl-summary-buffer-msgdb
	(elmo-pack-number
	 wl-summary-buffer-folder-name wl-summary-buffer-msgdb arg))
  (wl-summary-rescan))

(defun wl-summary-temp-mark-uudecode ()
  (interactive)
  (let ((mlist (reverse wl-summary-buffer-temp-mark-list))
	(summary-buf (current-buffer))
	(tmp-buf (get-buffer-create "*WL UUENCODE*"))
	orig-buf i k filename rc errmsg)
    (setq i 1)
    (setq k (length mlist))
    (set-buffer tmp-buf)
    (erase-buffer)
    (save-window-excursion
      (while mlist
	(set-buffer summary-buf)
	(wl-summary-jump-to-msg (car mlist))
	(wl-summary-redisplay)
	(set-buffer (setq orig-buf (wl-message-get-original-buffer)))
	(goto-char (point-min))
	(cond ((= i 1) ; first
	       (setq filename (wl-message-uu-substring 
			       orig-buf tmp-buf t 
			       (= i k))))
	      ((< i k)
	       (wl-message-uu-substring orig-buf tmp-buf))
	      (t ; last
	       (wl-message-uu-substring orig-buf tmp-buf nil t)))
	(setq i (1+ i))
	(setq mlist (cdr mlist)))
      (set-buffer tmp-buf)
      (message "exec %s..." wl-prog-uudecode)
      (elmo-bind-directory
       wl-tmp-dir
       (setq rc
	     (apply 'call-process-region (point-min) (point-max)
		    wl-prog-uudecode t nil nil wl-prog-uudecode-arg)))
      (setq errmsg (buffer-substring (point-min)(point-max)))
      (when (not (= 0 rc))
	(kill-buffer tmp-buf)
	(error "%s" errmsg))
      (kill-buffer tmp-buf)
      (save-excursion
	(set-buffer summary-buf)
	(wl-summary-delete-all-temp-marks))
      (setq filename (expand-file-name filename wl-tmp-dir))
      (if (file-exists-p filename)
	  (message "saved as %s" filename)
	(error "decode error: failed to save %s" filename))
      )))

(defun wl-summary-drop-unsync ()
  "Drop all unsync messages."
  (interactive)
  (if (or (not (interactive-p))
	  (y-or-n-p "Drop all unsync messages?"))
      (let* ((folder-list (elmo-folder-get-primitive-folder-list
			   wl-summary-buffer-folder-name))
	     (is-multi (elmo-multi-p wl-summary-buffer-folder-name))
	     (sum 0)
	     (multi-num 0)
	     pair)
	(message "Dropping...")
	(while folder-list
	  (setq pair (elmo-max-of-folder (car folder-list)))
	  (when is-multi ;; dirty hack...
	    (incf multi-num)
	    (setcar pair (+ (* multi-num elmo-multi-divide-number)
			    (car pair))))
	  (elmo-msgdb-set-number-alist 
	   wl-summary-buffer-msgdb 
	   (nconc 
	    (elmo-msgdb-get-number-alist wl-summary-buffer-msgdb)
	    (list (cons (car pair) nil))))
	  (setq sum (+ sum (cdr pair)))
	  (setq folder-list (cdr folder-list)))
	(wl-summary-set-message-modified)
	(wl-folder-set-folder-updated wl-summary-buffer-folder-name
				      (list 0
					    (+ wl-summary-buffer-unread-count 
					       wl-summary-buffer-new-count)
					    sum))
	(message "Dropping...done."))))

(provide 'wl-summary)
