/* cp_repack.c = routines for computing radii of circle packings and
key subroutines. */

#include "cp_head.h"

extern void fillcurves();
extern float s_radcalc();

/* CAUTION: algorithm changes should be tested in Repack.c before
they are included here.*/

/* Compute radii of packing to meet angle_sum targets, specified as
"aims". A negative "aim" means that that circle should not have its
radius adjusted. aim of zero is possible only for boundary 
circles in hyp setting. 

All routines are iterative, inspired by routine suggested
by Thurston. This file includes recent algorithms, old reliable 
versions, and experimental stuff being tried in spherical case.

Radii are of type r,h, or s (for special): r is euclidean radius,
h is hyperbolic, and s=exp(-h). Other relationships are:
h=log((1+r)/(1-r)), s=(1-r)/(1+r), r=(1-s)/(1+s).
Will be storing radii as s_radii. If h is infinite, then will store
eucl radius with negative value as the s_radius so it can be
used in graphing the appropriate circle. 

There currently is no packing algorithm for spherical geometry. Only
approach is to force packing into unit disc, do maximal packing
there and project back to sphere. Often, this involves a "puncture"
of the complex; missing circle is replaced as northern hemisphere
after max packing is projected back to sph. 

Overlap angles specified > Pi/2 can lead to incompatibilities. To
avoid domain errors, routines computing angles in such cases return
angle Pi. */

/* ==================== euclidean ================= */

int 
e_riffle(p,passes)   /* CRC - modified 5/28/97 -- uses model 
with super steps, safety checks. Adjust eucl radii to meet 
curvature targets in 'aim'. Neg. aim means that vertex is 
free - not subject to adjustment. */

int passes;
struct p_data *p;
{
	int i,j,aimnum=0,*index,count=0,key=0,key0,N,flag;
	float r, r2, fbest, faim, del, bet;
        float ttoler, cut=100, fact=-1, cut0, fact0;
        float lmax, rat,rr,*R0,ftol=.05;
	struct R_data *pR_ptr;
        struct K_data *pK_ptr;
        extern void e_anglesum_overlap();

	index=(int *)malloc(5*(p->nodecount+1)*sizeof(int));
	R0 = (float *)malloc((p->nodecount+1)*sizeof(float));
	pR_ptr=p->packR_ptr;
        pK_ptr=p->packK_ptr;
	for (i=1;i<=p->nodecount;i++)
 		if (pR_ptr[i].aim>0)
		 {
			index[aimnum]=i;
			aimnum++;
		 }
	if (aimnum==0) 
         {free(index); free(R0);return count; }
        ttoler = 3*aimnum*toler; /* adjust tolerance */

while (cut >ttoler && count<passes)
 {
/* save values */
	cut0 = cut;key0 = key;fact0 = fact; 
	for (i=1;i<=p->nodecount;i++) R0[i] = pR_ptr[i].rad;
/* do update using uniform label model */
	cut = 0;
	for (j=0;j<aimnum;j++) 
	 {
		i = index[j]; 		/* point to active node */
		faim = pR_ptr[i].aim;	/* get target sum */
		r = pR_ptr[i].rad;	/* get present label */
		e_anglesum_overlap(p,i,r,&fbest,&flag);  /* compute sum */
/* use the model to predict the next value */
		N = 2*pK_ptr[i].num;
		del = sin(faim/N);
		bet = sin(fbest/N);
		r2 = r*bet*(1-del)/(del*(1-bet));
/* store as new radius label */
		pR_ptr[i].rad = r2; 
		pR_ptr[i].curv = fbest;	/* store new angle sum */
		fbest -= faim;
		cut += fbest*fbest;	/* accum abs error */
	 }
	cut = sqrt(cut);
/* do super step? */
	key = 1;
	if (key0==1)
	 {
		fact = cut/cut0;
		if (fabs(fact-fact0)<ftol) fact = fact/(1-fact);
		lmax = 1000;
		for (j=0;j<aimnum;j++)	 /* find max step */
		 {
		   i = index[j];
		   r = pR_ptr[i].rad;
		   rat = r - R0[i];
		   if (rat<0)
			lmax = (lmax < (rr=(-r/rat))) ? lmax : rr;
		 }
		fact = (fact < 0.5*lmax) ? fact : 0.5*lmax; 
			/* compute new step */
		for (j=0;j<aimnum;j++)     /* interpolate to new radius */
		 {
		   i = index[j];
		   pR_ptr[i].rad += fact*(pR_ptr[i].rad-R0[i]);
		 }
		key = 0;
	 }

/* show activity */
	textsw_insert(msg_sw,"^",1); 
	XFlush(display);		

	count++;
 } /* end of while */

	free(index);
	free(R0);
	return count;
} /* e_riffle */

	/* --------- pre 1996 riffle routine (slow, but dependable) -- */
int
old_e_riffle(p,passes) /* (Could use some reworking.) */
int passes;
struct p_data *p;
{
	int i,j,aimnum=0,*index,count=0,dummy,flag;
	float recip,accum=0,verr,err,cut;
	struct R_data *pR_ptr;
	extern float e_radcalc();
	extern void e_anglesum_overlap();

	index=(int *)malloc(5*(p->nodecount+1)*sizeof(int));
	pR_ptr=p->packR_ptr;
	for (i=1;i<=p->nodecount;i++)
	if (pR_ptr[i].aim>0)
	 {
		index[aimnum]=i;
		aimnum++;
		err=pR_ptr[i].curv-pR_ptr[i].aim;
		accum += (err<0) ? (-err) : err;
	 }
	if (aimnum==0) {free(index);return count;}
	recip=.333333/aimnum;
	cut=accum*recip;
	while (cut >toler && count<passes)
	 {
		for (j=0;j<aimnum;j++)
		 {
			i=index[j];
			e_anglesum_overlap(p,i,pR_ptr[i].rad,
				&pR_ptr[i].curv,&flag);
			verr=pR_ptr[i].curv-pR_ptr[i].aim;
			if (fabs(verr)>cut)
			 {
			   pR_ptr[i].rad=e_radcalc(p,i,
				pR_ptr[i].rad,pR_ptr[i].aim,&dummy);
			   count++;
			 }
		 }
		accum=0;
		for (j=0;j<aimnum;j++)
		 {
			i=index[j];
			err=pR_ptr[i].curv-pR_ptr[i].aim;
			accum += (err<0) ? (-err) : err;
		 }
		cut=accum*recip;

/* show activity */
		textsw_insert(msg_sw,"e",1); 
		XFlush(display);

	
	 } /* end of while */
	free(index);
	return count;
} /* old_e_riffle */

int
e_riffle_vert(p,i)
struct p_data *p;
int i;
{
	int n=0,dummy,flag;
	float diff;
	struct R_data *pR_ptr;
	extern float e_radcalc();
	extern void e_anglesum_overlap();

	pR_ptr=p->packR_ptr;
	e_anglesum_overlap(p,i,pR_ptr[i].rad,&pR_ptr[i].curv,&flag);
	diff=pR_ptr[i].curv-pR_ptr[i].aim;
	while (n<20 && (diff>okerr || diff<(-okerr)) )
		 /* limit number of attempts */
	 {
		pR_ptr[i].rad=e_radcalc(p,i,
			pR_ptr[i].rad,pR_ptr[i].aim,&dummy);
		e_anglesum_overlap(p,i,pR_ptr[i].rad,&pR_ptr[i].curv,&flag);
		diff=pR_ptr[i].curv-pR_ptr[i].aim;
	 	n++;
	 }
	return n;
} /* e_riffle_vert */

float
e_radcalc(p,i,r,aim,chgflag) /* returns best radius at vert i to get
anglesum aim, via binary search method.
First guess = r. */
int i,*chgflag;
float r,aim;
struct p_data *p;
{
	int n,flag;
	float bestcurv,lower,upper,upcurv,lowcurv,factor=0.5;
	extern void e_anglesum_overlap();

	e_anglesum_overlap(p,i,r,&bestcurv,&flag);
	if (bestcurv>(aim+okerr))
	 {
		upper=r/factor;
		e_anglesum_overlap(p,i,upper,&upcurv,&flag);
		(*chgflag)++;
		if (upcurv>aim) return upper;
	 }
	else if (bestcurv<(aim-okerr))
	 {
		lower=r*factor;
		e_anglesum_overlap(p,i,lower,&lowcurv,&flag);
		(*chgflag)++;
		if (lowcurv<aim) return lower;
	 }
	else return r;
	(*chgflag)++;
	for (n=1;n<=iterates;n++)
	 {
		if (bestcurv>(aim+okerr)) 
		 {
		   lower=r;
		   lowcurv=bestcurv;
	   	   r += (aim-bestcurv)*(upper-r)/(upcurv-bestcurv);
		 }
		else if (bestcurv<(aim-okerr))
		 {
		   upper=r;
		   upcurv=bestcurv;
		   r -= (bestcurv-aim)*(lower-r)/(lowcurv-bestcurv);
		 }
		else return r;
		e_anglesum_overlap(p,i,r,&bestcurv,&flag);
	 }
	return r;
} /* e_radcalc */

void
e_anglesum_overlap(p,v,r,c,flag) /* compute ang sum, allow overlaps. 
Radii/overlaps incompatible, flag gets set, results are not reliable. */
int v,*flag;
float r,*c;
struct p_data *p;
{
	int j1,j2,n;
	float o2,o1,ovlp,r1,r2,m1,m2;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern int nghb();
	extern float e_cos_overlap();

	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
	*c=0; 
	*flag=0;

	j2=pK_ptr[v].flower[0];
	r2=pR_ptr[j2].rad;
	if (!p->overlap_status)
         {
           m2 = r2/(r+r2);
	   for (n=1;n<=pK_ptr[v].num;n++)
	    {
                m1 = m2;
		r2 = pR_ptr[pK_ptr[v].flower[n]].rad;
                m2 = r2/(r+r2);
		*c +=acos(1-2*m1*m2);
            }
	 }
	else
	 {
		o2=pK_ptr[v].overlaps[0];
		for (n=1;n<=pK_ptr[v].num;n++)
		 {
			j1=j2;r1=r2;o1=o2;
			j2=pK_ptr[v].flower[n];
			r2=pR_ptr[j2].rad;
			o2=pK_ptr[v].overlaps[n];
			ovlp=pK_ptr[j1].overlaps[nghb(p,j1,j2)];
			*c +=acos(e_cos_overlap(r,r1,r2,ovlp,o2,o1,flag));
		 }
	 }
} /* e_anglesum_overlap */


/* ==================== hyperbolic ================= */

int
h_riffle(p,passes)  /* CRC - modified 5/28/97 -- uses model 
with super steps, safety checks. Anglesum calculations are
in-line. Adjust hyperbolic s-radii to meet 
curvature targets in 'aim'. Neg. aim means that vertex is 
free - not subject to adjustment. */
int passes;
struct p_data *p;
{
	int i,j,k,j1,j2,aimnum=0,*index,count=0,key=0,key0,N,flag;
	float r, r1,r2,r3,fbest,faim,del,bet;
        float ttoler,cut=100,fact=-1,cut0,fact0;
        float ftol=0.05,lmax,rat,twor,*R0;
        float m2,m3,sr,t1,t2,t3,tr,o1,o2,o3;
	struct R_data *pR_ptr;
        struct K_data *pK_ptr;
        extern float h_cos_overlap();
	extern int nghb();

	index=(int *)malloc(5*(p->nodecount+1)*sizeof(int));
        R0 = (float *)malloc((p->nodecount+1)*sizeof(float));
	pR_ptr=p->packR_ptr;
	pK_ptr=p->packK_ptr;
   /* list radii to be adjusted, store squared radii */
	for (i=1;i<=p->nodecount;i++)
	 {
		if (pK_ptr[i].bdry_flag && pR_ptr[i].aim>=0 
		   && pR_ptr[i].aim<.001)
			pR_ptr[i].rad = (-.2);
		else if (pR_ptr[i].aim>0)
		 {
			if (pR_ptr[i].rad<=0 && pR_ptr[i].aim>.00001)
				pR_ptr[i].rad = .01;
			index[aimnum]=i;
			aimnum++;
		 }
		if (pR_ptr[i].rad>0)
		pR_ptr[i].rad = pR_ptr[i].rad*pR_ptr[i].rad;
	 }
	if (aimnum==0) 
	 {free(index);free(R0);return count;}
	ttoler = 3*sqrt((float)aimnum)*toler;		/* adjust tolerance */
	cut = ttoler+1;
/*	cut =3*aimnum*10;*/
while (cut >ttoler && count<passes)         /* Begin Main Loop */
 {
	cut0 = cut;key0 = key;fact0 = fact;	/* save prev values */
	for (i=1;i<=p->nodecount;i++) R0[i] = pR_ptr[i].rad;
	cut = 0;				    
/* Begin Iteration Loop */
	for (j=0;j<aimnum;j++) 
	 {
	/* anglesum calc */
		fbest = 0;
		i = index[j];
		if ((r = pR_ptr[i].rad)<0) r = 0;
		sr = sqrt(r);
		N = pK_ptr[i].num;
		if (!p->overlap_status) /* no overlaps */
		 { 			
               		twor = 2*r; 
               		r2 = pR_ptr[pK_ptr[i].flower[0]].rad;
               		m2 = (r2>0) ? (1-r2)/(1-r*r2) : 1;
               		for (k=1;k<=N;k++)	/* loop through petals */
               		 {
               			r3 = pR_ptr[pK_ptr[i].flower[k]].rad;
                   		m3 = (r3>0) ? (1-r3)/(1-r*r3) : 1;
                   		fbest += acos(1-twor*m2*m3); /* angle calc */
                   		m2 = m3;
               		 } 
               	 }
		else /* with overlaps, old routine */
		 {
			j2 = pK_ptr[i].flower[0];
		 	((r2 = pR_ptr[j2].rad) > 0) ? r2=sqrt(r2) : 0;
			o2=pK_ptr[i].overlaps[0];
			for (k=1;k<=N;k++)
			 {
				r1=r2;
				o1=o2;
				j1=j2;
				j2=pK_ptr[i].flower[k];
				((r2 = pR_ptr[j2].rad) >0) ? r2=sqrt(r2) : 0;
				o2=pK_ptr[i].overlaps[k];
				o3=pK_ptr[j1].overlaps[nghb(p,j1,j2)];
				fbest += 
				  acos(h_cos_overlap(sr,r1,r2,o3,o2,o1,&flag));
			 }
		 }
		faim = pR_ptr[i].aim;	/* get target sum */
	/* set up for model */
		N = 2*N;
		del = sin(faim/N);
		bet = sin(fbest/N);
		r2 = (bet-sr)/(bet*r-sr);	/* reference radius */
		if (r2>0) 			/* calc new label */
		 {
			t1 = 1 - r2;
			t2 = 2*del;
			t3 = t2/(sqrt(t1*t1+t2*t2*r2)+t1);
			r2 = t3*t3; 
		 }
		else
			r2 = del*del;		/* use lower limit */
		pR_ptr[i].rad = r2;		/* store new label */ 
		pR_ptr[i].curv = fbest;		/* store new anglesum */
		fbest = fbest-faim;
		cut += fbest*fbest;		/* accumulate error */
	 }
/* End Iteration Loop */

	cut = sqrt(cut);	
	key = 1;
	fact = cut/cut0;
	if (key0==1 && fact < 1.0)		/* try to extrapolate */
	 {
		cut *= fact;
		if (fabs(fact-fact0)<ftol)	/* use a super step */
			fact = fact/(1-fact);
		lmax = 1000;
		for (j=0;j<aimnum;j++)		/* find max allowable step*/
		 {
		   i = index[j];
		   r = pR_ptr[i].rad;
		   rat = r - R0[i];
		   if (rat>0)			    
			lmax = (lmax < (tr=((1-r)/rat))) ? lmax : tr;    
						/* to keep R<1 */
		   else if (rat < 0)
			lmax = (lmax < (tr= (-r/rat))) ? lmax : tr;      
						/* to keep R>0 */
		 }
/* compute new step */
		fact = (fact < 0.5*lmax) ? fact : 0.5*lmax;
/* interpolate new labels */						
		for (j=0;j<aimnum;j++)
		 {
		   i = index[j];
		   pR_ptr[i].rad += fact*(pR_ptr[i].rad-R0[i]);
		 }
		key = 0;
	 }
		
/* show activity */
	textsw_insert(msg_sw,"*",1); 
	XFlush(display);
		
	count++;
 } /* end of main while loop */

	for (i=1;i<=p->nodecount;i++)               /* reset labels */
		if (pR_ptr[i].rad>0)
			pR_ptr[i].rad = sqrt(pR_ptr[i].rad);
	free(index);
	free(R0);
	return count;
} /* h_riffle */

	/* --------- pre 1996 riffle routine (slow, but dependable) -- */
int
old_h_riffle(p,passes) /* (Could use some reworking.) */
int passes;
struct p_data *p;
{
	int i,j,aimnum=0,*index,count=0,dummy,flag;
	float recip,accum,verr,err,cut;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern float h_radcalc();
	extern void h_anglesum_overlap();

	index=(int *)malloc(5*(p->nodecount+1)*sizeof(int));
	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	accum=0;
	for (i=1;i<=p->nodecount;i++)
	 {
		if (pK_ptr[i].bdry_flag &&
			  pR_ptr[i].aim>=0  && pR_ptr[i].aim<.001)
			pR_ptr[i].rad=(-.2);
		else if (pR_ptr[i].aim>0)
		 {
			if (pR_ptr[i].rad<=0 && pR_ptr[i].aim>.00001) 
				pR_ptr[i].rad=.01;
			index[aimnum]=i;
			aimnum++;
			err=pR_ptr[i].curv-pR_ptr[i].aim;
			accum += (err<0) ? (-err) : err;
		 }
	 }
	if (aimnum==0) {free(index);return count;}
	recip=.333333/aimnum;
	cut=accum*recip;
	while (cut >toler && count<passes)
	 {
		for (j=0;j<aimnum;j++)
		 {
			i=index[j];
			h_anglesum_overlap(p,i,pR_ptr[i].rad,
				&pR_ptr[i].curv,&flag);
			verr=pR_ptr[i].curv-pR_ptr[i].aim;
			if (fabs(verr)>cut)
			 {
			   pR_ptr[i].rad=h_radcalc(p,i,
				pR_ptr[i].rad,pR_ptr[i].aim,&dummy);
			   count++;
			 }
		 }
		accum=0;
		for (j=0;j<aimnum;j++)
		 {
			i=index[j];
			err=pR_ptr[i].curv-pR_ptr[i].aim;
			accum += (err<0) ? (-err) : err;
		 }
		cut=accum*recip;

/* show activity */
		textsw_insert(msg_sw,"o",1); 
		XFlush(display);
	
	 } /* end of while */
	free(index);
	return count;
} /* old_h_riffle */

int
h_riffle_vert(p,i)
struct p_data *p;
int i;
{
	int n=0,dummy,flag;
	float diff;
	struct R_data *pR_ptr;
	extern float h_radcalc();
	extern void h_anglesum_overlap();

	pR_ptr=p->packR_ptr;
	h_anglesum_overlap(p,i,pR_ptr[i].rad,&pR_ptr[i].curv,&flag);
	diff=pR_ptr[i].curv-pR_ptr[i].aim;
	while (n<20 && (diff>okerr || diff<(-okerr)) )
		 /* limit number of attempts */
	 {
		pR_ptr[i].rad=h_radcalc(p,i,
			pR_ptr[i].rad,pR_ptr[i].aim,&dummy);
		h_anglesum_overlap(p,i,pR_ptr[i].rad,&pR_ptr[i].curv,&flag);
		diff=pR_ptr[i].curv-pR_ptr[i].aim;
	 	n++;
	 }
	return n;
} /* h_riffle_vert */

float
h_radcalc(p,i,s,aim,chgflag) /* returns best radius at vert i to get
anglesum aim, via secant method. s = first guess. */
int i,*chgflag;
float s,aim;
struct p_data *p;
{
	int n,flag;
	float bestcurv,lower,upper,upcurv,lowcurv,factor=0.5;
	extern void h_anglesum_overlap();

	h_anglesum_overlap(p,i,s,&bestcurv,&flag);
	if (bestcurv>(aim+okerr))
	 {
		lower=s*factor;
		h_anglesum_overlap(p,i,lower,&lowcurv,&flag);
		(*chgflag)++;
		if (lowcurv>aim) return lower;
	 }
	else if (bestcurv<(aim-okerr))
	 {
		upper=1-factor+s*factor;
		h_anglesum_overlap(p,i,upper,&upcurv,&flag);
		(*chgflag)++;
		if (upcurv<aim) return upper;
	 }
	else return s;
	(*chgflag)++;
	for (n=1;n<=iterates;n++)
	 {
		if (bestcurv>(aim+okerr)) 
		 {
		   upper=s;
		   upcurv=bestcurv;
		   s -= (bestcurv-aim)*(lower-s)/(lowcurv-bestcurv);
		 }
		else if (bestcurv<(aim-okerr))
		 {
		   lower=s;
		   lowcurv=bestcurv;
	   	   s += (aim-bestcurv)*(upper-s)/(upcurv-bestcurv);
		 }
		else return s;
		h_anglesum_overlap(p,i,s,&bestcurv,&flag);
	 }
	return s;
} /* h_radcalc */

int 
h_anglesum(p,i,s,c,flag) /* compute ang sum; should not be called if
there are overlaps. */
/* version modified by Chuck Collins, Oct 1996 */
int i,*flag;
float s,*c;
struct p_data *p;
{
	int k,j2,j1;
	float s1,s2,s3,m2,m3,a1,o1,o2,o3;
	extern float h_cos_overlap();
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern int nghb();

	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
	*c=0;
	*flag=0; 
	if (s<=0) return 1; /* infinite radius at vertex of interest */
	j2=pK_ptr[i].flower[0];
	s2=pR_ptr[j2].rad;
        if (s2<=0)  m2 = 1;
        else  m2 = (1-s2)/(1-s*s2);
	if (!p->overlap_status)
	  for (k=1;k<=pK_ptr[i].num;k++)
	   {
		s3=pR_ptr[pK_ptr[i].flower[k]].rad;
		if (s3<=0)  m3 = 1;
		else   m3 = (1-s3)/(1-s*s3);
		a1 = 1 - 2*s*m2*m3;
		if (a1>1)  a1 = 1;
		if (a1<-1) a1 = -1;
		*c+=acos(a1);
		s2 = s3;
		m2 = m3;
	   }
	else
	 {
		o2=pK_ptr[i].overlaps[0];
		for (k=1;k<=pK_ptr[i].num;k++)
		 {
			s1=s2;
			o1=o2;
			j1=j2;
			j2=pK_ptr[i].flower[k];
			s2=pR_ptr[j2].rad;
			o2=pK_ptr[i].overlaps[k];
			o3=pK_ptr[j1].overlaps[nghb(p,j1,j2)];
			*c+=acos(h_cos_overlap(s,s1,s2,o3,o2,o1,&flag));
		 }
	 }
	return 1;
} /* h_anglesum */

void
h_anglesum_overlap(p,i,s,c,flag) /* compute ang sum, allow overlaps. 
Radii/overlaps incompatible, flag gets set, results are not reliable. */
int i,*flag;
float s,*c;
struct p_data *p;
{
	int k,j2,j1;
	float s1,s2,o1,o2,o3;
	extern float h_cos_overlap(),h_comp_cos();
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern int nghb();

	*flag=0;
	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
	*c=0;
	if (s<=0) return; /* infinite radius at vertex of interest */
	j2=pK_ptr[i].flower[0];
	s2=pR_ptr[j2].rad;
	if (!p->overlap_status)
		for (k=1;k<=pK_ptr[i].num;k++)
		 {
			s1=s2;
			s2=pR_ptr[pK_ptr[i].flower[k]].rad;
			*c+=acos(h_comp_cos(s,s1,s2));
		 }
	else
	 {
		o2=pK_ptr[i].overlaps[0];
		for (k=1;k<=pK_ptr[i].num;k++)
		 {
			s1=s2;
			o1=o2;
			j1=j2;
			j2=pK_ptr[i].flower[k];
			s2=pR_ptr[j2].rad;
			o2=pK_ptr[i].overlaps[k];
			o3=pK_ptr[j1].overlaps[nghb(p,j1,j2)];
			*c+=acos(h_cos_overlap(s,s1,s2,o3,o2,o1,flag));
		 }
	 }
} /* h_anglesum_overlap */

/* ==================== spherical attempts ================= */

int
s_riffle(p,passes) /* adjust sph radii to meet curvature targets in 'aim'.
A target less than zero means that vertex is free - no adjustments.
Bdry radii adjusted only if their aim >= 0. Geometry forces different
tactics. To normalize, we find first face having three default aims,
set its aims to -1 and its radii to pi/3. When we state the
process, may need preliminary changes in some radii to avoid
incompatibilities. Also, process slow, so accept greater error. 
Also, may need alternate methods for complexes with bdry. */
int passes;
struct p_data *p;
{
	int i,j,aimnum=0,*index,count=0,dummy,norm_face,flag,v;
	float max_err=0,err,valence,new_radius;
	struct R_data *pR_ptr;
	f_data *face_ptr;

	index=(int *)malloc(5*(p->nodecount+1)*sizeof(int));
	pR_ptr=p->packR_ptr;
	face_ptr=p->faces;
/* Pick face to normalize */
	flag=1;i=0;
	while (flag && i++<=p->facecount)
	 {
		flag=0;
		for (j=0;j<3;j++)
			if (fabs(p->packR_ptr[face_ptr[i].vert[j]].aim 
				- 2.0*M_PI) > okerr) flag=1;
	 }
	if (flag) /* no face to normalize */
		{free(index);return count;}
	norm_face=i;
	for (j=0;j<3;j++)
	 {
		pR_ptr[face_ptr[norm_face].vert[j]].rad = (M_PI-okerr)/3.0;
		pR_ptr[face_ptr[norm_face].vert[j]].aim = -1.0;
	 }
/* make sure radii compatibility is met */
	check_compatibility(p,norm_face);
/* identify verts subject to change */
	fillcurves(p);
	for (i=1;i<=p->nodecount;i++)
	if (pR_ptr[i].aim>0)
	 {
		index[aimnum]=i;
		aimnum++;
		err=pR_ptr[i].curv-pR_ptr[i].aim;
		max_err = (fabs(err)>max_err) ? fabs(err) : max_err;
	 }
	if (aimnum==0 || max_err < .0001) /* none to adjust or okay. */
	 {
		for (j=0;j<3;j++) pR_ptr[face_ptr[norm_face].vert[j]].aim =
			2.0*M_PI; /* reset aims of face used for norming */
		free(index);
		return count;
	 }

/* find valence of mapping from aim totals */
	valence=(branch_order(p)/2.0) + 1.0;
/* ultimately need to branch for complexes with bdry, and possibly
for branched complexes. */
/* if only one vertex subject to adjustment */
	if (aimnum==1) 
	 {
		for (j=0;j<3;j++) pR_ptr[face_ptr[norm_face].vert[j]].aim =
			2.0*M_PI; /* reset aims of face used for norming */
		v=s_riffle_vert(p,index[0]);
		free(index);
		return(v);
	 }
/* normalize area */
	normalize(p,valence);
	v=1;  /*initalize for record keeping */
	while ((!(v=select_vertex(p,index,aimnum))==0)
		&& (count < passes))
	  {
		count++;
		if ((count % 100)==0)
		 {
			normalize(p,valence);
			sprintf(msgbuf," %d-> % .8e ",v,pR_ptr[v].curv/M_PI);
			msg();
		 } 
		new_radius=s_radcalc(p,v,pR_ptr[v].rad,pR_ptr[v].aim,&dummy);
		s_changerad(p,v,new_radius);
		/* normalize(p,valence); */
	   }			
	for (j=0;j<3;j++) pR_ptr[face_ptr[norm_face].vert[j]].aim =
		2.0*M_PI; /* reset aims of face used for norming */
	free(index);
	return count;
} /* s_riffle */

int
s_riffle_vert(p,i)
struct p_data *p;
int i;
{
	int n=0,dummy,flag;
	float diff;
	struct R_data *pR_ptr;

	pR_ptr=p->packR_ptr;
	s_anglesum_overlap(p,i,pR_ptr[i].rad,&pR_ptr[i].curv,&flag);
	diff=pR_ptr[i].curv-pR_ptr[i].aim;
	while (n<20 && (diff>okerr || diff<(-okerr)) )
		 /* limit number of attempts */
	 {
		pR_ptr[i].rad=s_radcalc(p,i,
			pR_ptr[i].rad,pR_ptr[i].aim,&dummy);
		s_anglesum_overlap(p,i,pR_ptr[i].rad,&pR_ptr[i].curv,&flag);
		diff=pR_ptr[i].curv-pR_ptr[i].aim;
	 	n++;
	 }
	return n;
} /* s_riffle_vert */

float
s_radcalc(p,v,r,aim,chgflag) /* returns best radius at vert v to get
anglesum aim, via binary search method. (Doesn't use overlaps).
First guess = r. */
int v,*chgflag;
float r,aim;
struct p_data *p;
{
	int n,increase_r,decrease_r,r_too_big,r_too_small,sign,flag;
	float bestcurv,lower,upper,upcurv,lowcurv,target,factor=0.15;
	extern int derivative_sign();

	target=aim;
	s_anglesum_overlap(p,v,r,&bestcurv,&flag);
	sign=derivative_sign(p,v,r);
	r_too_small=(((bestcurv>(aim+okerr)) && (sign<0)) ||
		     ((bestcurv<(aim+okerr)) && (sign>0)));
	r_too_big = (((bestcurv<(aim+okerr)) && (sign<0)) ||
		     ((bestcurv>(aim+okerr)) && (sign>0)));
	target+=(1/(3*p->packK_ptr[v].num))*(aim-bestcurv)*sign;
	
	if (r_too_small)
	 {
		if ((r*(1.0+factor))<M_PI)
			upper=r*(1.0+factor);
		else upper=r + ((M_PI-r)/2.0);
		s_anglesum_overlap(p,v,upper,&upcurv,&flag);
		(*chgflag)++;
		if (upcurv>target) return upper;
	 }
	else if (r_too_big)
	 {
		lower=r*(1.0-factor);
		s_anglesum_overlap(p,v,lower,&lowcurv,&flag);
		(*chgflag)++;
		if (lowcurv<target) return lower;
	 }
	else return r;
	(*chgflag)++;
	for (n=1;n<=iterates;n++)
	 {
		sign=derivative_sign(p,v,r);
		increase_r=(((bestcurv>(target+okerr)) && (sign<0)) ||
				((bestcurv<(target+okerr)) && (sign>0)));
		decrease_r=(((bestcurv>(target+okerr)) && (sign>0))  ||
				((bestcurv<(target+okerr)) && (sign<0)));
		if (increase_r) 
		 {
		   lower=r;
		   lowcurv=bestcurv;
		   r+= (target-bestcurv)*(upper-r)/(upcurv-bestcurv);	  
		 }
		else if (decrease_r)
		 {
		   upper=r;
		   upcurv=bestcurv;
		   r -= (bestcurv-target)*(lower-r)/(lowcurv-bestcurv);
		 }
		else return r;
		s_anglesum_overlap(p,v,r,&bestcurv,&flag);
	 }
	return r;
} /* s_radcalc */

int
s_anglesum_overlap(p,i,r,c,flag) /* compute ang sum; can't yet do 
overlaps. flag will (eventually) indicate incompatibilities. */
int i,*flag;
float r,*c;
struct p_data *p;
{
	int k,j2;
	float r1,r2;
	extern float s_comp_cos();
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;

	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
	*c=0;
	*flag=0; /* not used yet */

	if (r<=0) return 0; 
	j2=pK_ptr[i].flower[0];
	r2=pR_ptr[j2].rad;
	for (k=1;k<=pK_ptr[i].num;k++)
	 {
		r1=r2;
		r2=pR_ptr[pK_ptr[i].flower[k]].rad;
		*c+=acos(s_comp_cos(r,r1,r2));
	 }
	return 1;
} /* s_anglesum_overlap */

int
select_vertex(p,index,aimnum)
struct p_data *p;
int *index;
int aimnum;
{
	struct R_data *pR_ptr;
	int i,vert;
	float max_error,error;
	
/* Choose v with biggest error. (May need to put in check to prevent
rechoosing same vertex continually.) */
	
	pR_ptr=p->packR_ptr;
	vert=index[0];
	max_error=fabs(pR_ptr[index[0]].curv-pR_ptr[index[0]].aim);
	
	for (i=1;i<aimnum;i++)
	  {
		error=fabs(pR_ptr[index[i]].curv-pR_ptr[index[i]].aim);
		if (error > max_error)
		 {
			vert=index[i];
			max_error=error;
		 }
	   }
	if (max_error>.001) /* should compare to okerr, but too
		time consuming. */
	 {  
		return vert;
	 }
	else return 0;
} /* select_vertex */

int
normalize(p,valence) /* Uses secant method. Sets up duplicate data area. 
Assumes curvatures updated. */
struct p_data *p;
float valence;
{
	struct R_data *pR_ptr,*lowR,*highR;
	float target,area,max_r,lowarea,higharea,error,factor=0.15;
	float norm_err=.00001;
	extern float s_area();
	extern float max_r_adjustment();
	int v;

	pR_ptr=p->packR_ptr;

	target=4*M_PI*valence;
		
/* set up dup data */
	highR=(struct R_data*)malloc((p->nodecount+2)*sizeof(struct R_data));
	lowR=(struct R_data *)malloc((p->nodecount+2)*sizeof(struct R_data));
	for (v=1;v<=p->nodecount;v++) highR[v]=lowR[v]=pR_ptr[v];
	
	area=s_area(p);
	lowarea=area;
	higharea=area;
	error=area - target;
	
	if (fabs(error)> norm_err)
	  {
		  while (lowarea>(target))
		    {
		       for (v=1;v<=p->nodecount;v++)
			     if (pR_ptr[v].aim>0) 
				lowR[v].rad-=lowR[v].rad * factor;
			p->packR_ptr=lowR;
			fillcurves(p);	
		        lowarea=s_area(p);
		     }

		   while (higharea<(target))
		     {
	        	for (v=1;v<=p->nodecount;v++)
			     if (pR_ptr[v].aim>0 
				&& (max_r=max_r_adjustment(p,v))>0.0) 
				{
				max_r= highR[v].rad+max_r;
				highR[v].rad = ((1+factor)*highR[v].rad<=max_r)
				 ? (1+factor)*highR[v].rad : max_r;
				}
			p->packR_ptr=highR;
			fillcurves(p);	
			higharea=s_area(p);		     
	    	      }
	    }	
	p->packR_ptr=pR_ptr; /* CAUTION: note, pointer has been fooled with! */

	while (fabs(error)>norm_err)
	  {
	     	for (v=1;v<=p->nodecount;v++)
		   if (pR_ptr[v].aim>=0)
		       pR_ptr[v].rad=(higharea*lowR[v].rad+lowarea*highR[v].rad)
					/(higharea + lowarea);
		
		fillcurves(p);
		area=s_area(p);
		error= area-target;
		
		if (error>norm_err)
		   {
			for (v=1;v<=p->nodecount;v++) highR[v]=pR_ptr[v];
			higharea=area;
		    }		
		else if (error<(-norm_err))
		    {
			for (v=1;v<=p->nodecount;v++) lowR[v]=pR_ptr[v];
			lowarea=area;
		    }
	   }
	free(highR);
	free(lowR);
	return 1;
} /* normalize */

int
check_compatibility(p,nf) /* sum of angles of each face must sum to
less than pi. Return num radii adjusted to meet conditions. nf is the
normalized face. */
struct p_data *p;
int nf;
{
	int v,count=0,num_adj=0,j,v0,v1,v2,w;
	float max;
	f_data *face;
	struct R_data *pR_ptr;

	face=p->faces;
	pR_ptr=p->packR_ptr;
	
	for (v=1;v<=p->facecount;v++)
	  {
		count=0;
		if (v==nf) continue;	
		v0=face[nf].vert[0];v1=face[nf].vert[1];v2=face[nf].vert[2];
		while ((pR_ptr[face[v].vert[0]].rad 
		    +pR_ptr[face[v].vert[1]].rad
		    +pR_ptr[face[v].vert[2]].rad)>=(M_PI-okerr))
		  {
			max=0.0;
			for (j=0;j<3;j++)
			   max = (pR_ptr[w=face[v].vert[j]].rad > max 
				&& w!=v0 && w!=v1 && w!=v2) ?
				pR_ptr[face[v].vert[j]].rad : max;
			for (j=0;j<3;j++)
			   if (pR_ptr[w=face[v].vert[j]].rad >= max
				&& w!=v0 && w!=v1 && w!=v2)
				pR_ptr[face[v].vert[j]].rad *= .8;
			count++;
		   }
		num_adj += (count>0);
	   }
	if (num_adj)
	 {
		sprintf(msgbuf,"%d sph radii adjusted in pack %d for compatibility",num_adj,pack_num(p));
		msg();
	 }
	return (num_adj);
} /* check_compatibility */

float
s_area(p)
struct p_data *p;
{
	float carrier_area=0;
	int v;

	for (v=1;v<=p->nodecount;v++)
		carrier_area+=p->packR_ptr[v].curv;

	carrier_area-= (M_PI * p->facecount);
	return carrier_area;
} /* s_area */

int
derivative_sign(p,i,r) /* chk sign of deriv of sph angle sum at i, rad=r. */
struct p_data *p;
int i;
float r;
{
	int n,neg_term=0;
	float r1,r2,deriv=0,top,bottom;
	struct R_data *pR_ptr;
	struct K_data *pK_ptr;

	pR_ptr=p->packR_ptr;
	pK_ptr=p->packK_ptr;
	
	for (n=0 ;n<=pK_ptr[i].num-1;n++)
	  if ((2*r + pR_ptr[pK_ptr[i].flower[n]].rad + 
		pR_ptr[pK_ptr[i].flower[n+1]].rad)>=M_PI)
		neg_term++;
	if (!neg_term) return -1;
	for (n=0;n<=pK_ptr[i].num-1;n++)
	 {
		r1=pR_ptr[pK_ptr[i].flower[n]].rad;
		r2=pR_ptr[pK_ptr[i].flower[n]].rad;
		top=sin(2*r + r1 +r2)*sqrt(sin(r1)*sin(r2));
		bottom=sin(r+r1)*sin(r+r2)*sqrt(sin(r+r1+r2)*sin(r));
		deriv += top/bottom;
	 }
	if (deriv >= 0) return -1;
	return 1;
} /* derivative_sign */	

int
branch_order(p)
struct p_data *p;
{
	int v,order=0;
	struct R_data *pR_ptr;
	struct K_data *pK_ptr;

	pR_ptr=p->packR_ptr;
	pK_ptr=p->packK_ptr;
	for (v=1;v<=p->nodecount;v++)
		if ((pR_ptr[v].aim>0) && !pK_ptr[v].bdry_flag)
			order+=(int)(pR_ptr[v].aim/(2*M_PI)) -1;
	return order;
} /* branch_order */


/* -------------- this next routine isn't being used right now */

h_curvcalc(p,i,s,c,d)  /* comp angle sum c and its derivative d (with
respect to the s_radius, for interior nodes in hyp setting. Deriv
depends on whether some of radii are infinite */
struct p_data *p;
int i;
float s,*c,*d;
{
	int k;
	float s1,s2,a,b,cc,e,ss,cs;
	extern float h_comp_cos();
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;

	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	*c=0;*d=0;
	if (s<=0) return; /* infinite radius at vertex of interest */
	s2=pR_ptr[pK_ptr[i].flower[0]].rad;
	for (k=1;k<=pK_ptr[i].num;k++)
	 {
		s1=s2;
		s2=pR_ptr[pK_ptr[i].flower[k]].rad;
		cs=h_comp_cos(s,s1,s2);
		*c+=acos(cs);
		if ((s1>0) && (s2>0))
		 {
			cc=(-s1*s1-s2*s2);
			b=s1*s1*s2*s2;
			a=(-2-cc-2*b);
			e=cc-a;
			ss=s*s;
			*d+=2*e*(1-b*ss*ss)/((1+cc*ss+b*ss*ss)*
				sqrt( e*(2+(cc+a)*ss+2*b*ss*ss) ));
		 }
		else if ((s1<=0) && (s2<=0)) *d+=2/sqrt(1-s*s);
		else if (s1<=0) *d+=sqrt(s2/s)/(1+s2*s);
		else *d+=sqrt(s1/s)/(1+s1*s);
	 }
} /*h_curvcalc */

