/*
The DsTool program is the property of:
 
                             Cornell University 
                        Center of Applied Mathematics 
                              Ithaca, NY 14853
                      dstool_bugs@macomb.tn.cornell.edu
 
and may be used, modified and distributed freely, subject to the following
restrictions:
 
       Any product which incorporates source code from the DsTool
       program or utilities, in whole or in part, is distributed
       with a copy of that source code, including this notice. You
       must give the recipients all the rights that you have with
       respect to the use of this software. Modifications of the
       software must carry prominent notices stating who changed
       the files and the date of any change.
 
DsTool is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of FITNESS FOR A PARTICULAR PURPOSE.
The software is provided as is without any obligation on the part of Cornell 
faculty, staff or students to assist in its use, correction, modification or
enhancement.
*/

/*
 * traj_plot.c
 */

#include <complib.h>
#include <math.h>
#include <tcl.h>

#include "view.h"
#include "utilities.h"
#include "dscolor.h"

static int geomview_traj_plot(int n, struct Prop_DataS *prop_cntl,
			      int start_ptr, int stop_ptr,
			      double *func_values);
static int twod_traj_plot(int n, TWOD_VIEW_ITEM *twod,
			  struct Prop_DataS *prop_cntl,
			  int start_ptr, int stop_ptr,
			  double *func_values);
static int oned_traj_plot(int n, VIEW_ITEM *v,
			  struct Prop_DataS *prop_cntl,
			  int start_ptr, int stop_ptr);


int
traj_plot(struct Prop_DataS *prop_cntl, int start_ptr, int stop_ptr,
	  double *func_values)
{
  VIEW_ITEM *v = Views;
  unsigned int filter = prop_cntl->plot_filter;

  while (v) 
    {
      switch (v->type)
	{
	case GEOMVIEW:
	  if (filter & ON_THREED)
	    geomview_traj_plot(v->number, prop_cntl, 
			       start_ptr, stop_ptr, func_values);
	  break;
	case TWOD:
	  if (filter & ON_TWOD)
	    twod_traj_plot(v->number, &(v->data.twod), prop_cntl, 
			   start_ptr, stop_ptr, func_values);
	  break;
	case ONED:
	  if (filter & ON_ONED)
	    oned_traj_plot(v->number, v, prop_cntl, 
			   start_ptr, stop_ptr);
	  break;
	default:
	  fprintf(stdout,"Unrecognized view device in traj_plot.\n");
	}
      v = v->next_View;
    }
  return 0;
}


static int
geomview_traj_plot(int n, struct Prop_DataS *prop_cntl,
		   int start_ptr, int stop_ptr, double *func_values)
{
  int i;

  if (n != 1)
    {
      fprintf(stdout,"Only one GEOMVIEW window currently implemented.\n");
      return 0;
    }

  for(i=start_ptr; i<=stop_ptr; i++)
    {
      if (prop_cntl->gv_n > 0)
        {
          prop_cntl->gv_function(func_values, prop_cntl->traj_segment[i],
                                 prop_cntl->parameters);
          geomview_transform(prop_cntl->gv_n, func_values);
        }
      if (NULL != prop_cntl->gv_custom)
	prop_cntl->gv_custom(prop_cntl->traj_segment[i],
                                 prop_cntl->parameters);
    }
  return 0;
}

static int
twod_traj_plot(int n, TWOD_VIEW_ITEM *twod, struct Prop_DataS *prop_cntl,
	       int start_ptr, int stop_ptr, double *func_values)
{
  int i, x, y, color;
  int fn = FALSE;
  double dx, dy;

  /* will need function values */
  if ( (twod->hortype == FUNCT) || (twod->vertype == FUNCT) ) fn = TRUE;
  
  for (i=start_ptr; i<=stop_ptr; i++)
    {
      if (fn == TRUE)
	prop_cntl->aux_function(func_values, prop_cntl->traj_segment[i],
				prop_cntl->parameters);
      switch(twod->hortype)
	{
	case VARB:
	  dx = prop_cntl->traj_segment[i][twod->horindex];
	  break;
	case PARAM:
	  dx = prop_cntl->parameters[twod->horindex];
	  break;
	case FUNCT:
	  dx = func_values[twod->horindex];
	  break;
	default:
	  fprintf(stdout,"twod_plot: error in hor type\n");
	}
      switch(twod->vertype)
	{
	case VARB:
	  dy = prop_cntl->traj_segment[i][twod->verindex];
	  break;
	case PARAM:
	  dy = prop_cntl->parameters[twod->verindex];
	  break;
	case FUNCT:
	  dy = func_values[twod->verindex];
	  break;
	default:
	  fprintf(stdout,"twod_plot: error in ver type\n");
	}
      dx = (dx - twod->hormin) / (twod->hormax - twod->hormin);
      dy = (twod->vermax - dy) / (twod->vermax - twod->vermin);
      if ((dx >= 0.0) && (dx < 1.0) && (dy > 0.0) && (dy <= 1.0))
	{
	  x = (int) floor(dx*twod->width);
	  y = (int) floor(dy*twod->height);

          color = DsColor_get_index( twod->colormode, 
			             prop_cntl->table_color, 
                                     prop_cntl->sys_color ); 
  
	  twod_point(n,x,y,color,prop_cntl->symbol);
	}
    }
  return 0;
}

static int
oned_traj_plot(int n, VIEW_ITEM *v, struct Prop_DataS *prop_cntl,
	       int start_ptr, int stop_ptr)
{
  int i, color_id;
  int fn = FALSE;
  double old_x;
  double new_y;
  ONED_VIEW_ITEM *oned = &(v->data.oned);

  color_id = DsColor_get_index( oned->colormode, 
			        prop_cntl->table_color, 
			        prop_cntl->sys_color ); 

  if (prop_cntl->start_to_save < 0) {
    /* this trajectory is a continuation */
    if (start_ptr == 0) {
      if (stop_ptr > 0) {
	old_x = prop_cntl->state[0];
	new_y = prop_cntl->traj_segment[0][0];
	oned_line(v, old_x, old_x, old_x, new_y, color_id);
	oned_line(v, old_x, new_y, new_y, new_y, color_id);
	old_x = new_y;
      }
      start_ptr = 1;
    }
    else 
      old_x = prop_cntl->traj_segment[start_ptr-1][0];
  }

  else {
    if (start_ptr <= 1) {
      if (stop_ptr > 0) {
	old_x = prop_cntl->traj_segment[0][0];
	new_y = prop_cntl->traj_segment[1][0];
	oned_line(v, old_x, oned->vermin, old_x, new_y, color_id);
	oned_line(v, old_x, new_y, new_y, new_y, color_id);
	old_x = new_y;
      }
      start_ptr = 2;
    }
    else
      old_x = prop_cntl->traj_segment[start_ptr-1][0];
  }

  for (i=start_ptr; i<=stop_ptr; i++)
    {
      new_y = prop_cntl->traj_segment[i][0];
      oned_line(v, old_x, old_x, old_x, new_y, color_id);
      oned_line(v, old_x, new_y, new_y, new_y, color_id);
      old_x = new_y;
    }

  return 0;
}

