/*********************************************************************

** This library uses the netcdf - version 2.4.3 which is
** Copyright 1993-1997 University Corporation for Atmospheric Research/Unidata

** The netcdf library is provided as per the terms of the
** UCAR/Unidata license

** See netcdf-2.4.3/COPYRIGHT for the full notice

********************************************************************/
/* $Id: absff_metadata.c,v 1.2 1997/07/26 00:15:24 venkat Exp $ */
/* $Log: absff_metadata.c,v $
 * Revision 1.2  1997/07/26 00:15:24  venkat
 * Added-Copyright-notice-and-RCS-headers
 * */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <sys/types.h>

#include "absff_ext.h"


/** Errors: -1 for library error or NULL params, non-negative for success **/

int Absff_CopyFromMetadataValues(dup_vals, orig_vals, type, size, start_index, count)
 void *dup_vals, *orig_vals;
 Abs_Type type;
 long size, start_index, count;
{
 
 register int i,j;
 
 unsigned char *dup_uchar_vals, *orig_uchar_vals;
 char *dup_char_vals, *orig_char_vals;
 unsigned short *dup_ushort_vals, *orig_ushort_vals;
 short *dup_short_vals, *orig_short_vals;
 unsigned int *dup_uint_vals, *orig_uint_vals;
 int *dup_int_vals, *orig_int_vals;
 unsigned long *dup_ulong_vals, *orig_ulong_vals;
 long *dup_long_vals, *orig_long_vals;
 float *dup_float_vals, *orig_float_vals;
 double *dup_double_vals, *orig_double_vals;
 char **dup_string_vals, **orig_string_vals;
 
	
  if(dup_vals == NULL || orig_vals==NULL)
	return -1;

  if (count < 0 || count > size || start_index < 0 || start_index > (size -1) ){
	/** Error: Overflow **/
	return -1;
  }

  switch (type) {
	case abs_uchar8:
	case abs_uint8:

		dup_uchar_vals = (unsigned char *)dup_vals;
		orig_uchar_vals = (unsigned char *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_uchar_vals[j]=orig_uchar_vals[i];
		break;


	case abs_char8:
	case abs_int8:

		dup_char_vals = (char *)dup_vals;
		orig_char_vals = (char *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_char_vals[j]=orig_char_vals[i];
		break;

	case abs_uint16:

		dup_ushort_vals = (unsigned short *)dup_vals;
		orig_ushort_vals = (unsigned short *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_ushort_vals[j]=orig_ushort_vals[i];
		break;

	case abs_int16:

		dup_short_vals = (short *)dup_vals;
		orig_short_vals = (short *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_short_vals[j]=orig_short_vals[i];
		break;

	case abs_uint32:
		dup_uint_vals = (unsigned int *)dup_vals;
		orig_uint_vals = (unsigned int *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_uint_vals[j]=orig_uint_vals[i];
		break;

	case abs_int32:

		dup_int_vals = (int *)dup_vals;
		orig_int_vals = (int *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_int_vals[j]=orig_int_vals[i];
		break;

	case abs_uint64:
		dup_ulong_vals = (unsigned long *)dup_vals;
		orig_ulong_vals = (unsigned long *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_ulong_vals[j]=orig_ulong_vals[i];
		break;

	case abs_int64:

		dup_long_vals = (long *)dup_vals;
		orig_long_vals = (long *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_long_vals[j]=orig_long_vals[i];
		break;

	case abs_float32:

		dup_float_vals = (float *)dup_vals;
		orig_float_vals = (float *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_float_vals[j]=orig_float_vals[i];
		break;

	case abs_float64:

		dup_double_vals = (double *)dup_vals;
		orig_double_vals = (double *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_double_vals[j]=orig_double_vals[i];
		break;

	case abs_string:
	
		dup_string_vals = (char **)dup_vals;
		orig_string_vals = (char **) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			strcpy(dup_string_vals[j], orig_string_vals[i]);
		break;

	default:
		/** Error: Invalid Type **/
		return -1;
		break;
  }
  return 1;
}
 
int Absff_CopyToMetadataValues(dup_vals, orig_vals, type, size, start_index, count)
 void *dup_vals, *orig_vals;
 Abs_Type type;
 long size, start_index, count;
{
 
 register int i,j;
 
 unsigned char *dup_uchar_vals, *orig_uchar_vals;
 char *dup_char_vals, *orig_char_vals;
 unsigned short *dup_ushort_vals, *orig_ushort_vals;
 short *dup_short_vals, *orig_short_vals;
 unsigned int *dup_uint_vals, *orig_uint_vals;
 int *dup_int_vals, *orig_int_vals;
 unsigned long *dup_ulong_vals, *orig_ulong_vals;
 long *dup_long_vals, *orig_long_vals;
 float *dup_float_vals, *orig_float_vals;
 double *dup_double_vals, *orig_double_vals;
 char **dup_string_vals, **orig_string_vals;
 
	
  if(dup_vals == NULL || orig_vals==NULL)
	return -1;

  if (count < 0 || count > size || start_index < 0 || start_index > (size -1) ){
	/** Error: Overflow **/
	return -1;
  }

  switch (type) {
	case abs_uchar8:
	case abs_uint8:

		dup_uchar_vals = (unsigned char *)dup_vals;
		orig_uchar_vals = (unsigned char *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_uchar_vals[i]=orig_uchar_vals[j];
		break;


	case abs_char8:
	case abs_int8:

		dup_char_vals = (char *)dup_vals;
		orig_char_vals = (char *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_char_vals[i]=orig_char_vals[j];
		break;

	case abs_uint16:

		dup_ushort_vals = (unsigned short *)dup_vals;
		orig_ushort_vals = (unsigned short *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_ushort_vals[i]=orig_ushort_vals[j];
		break;

	case abs_int16:

		dup_short_vals = (short *)dup_vals;
		orig_short_vals = (short *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_short_vals[i]=orig_short_vals[j];
		break;

	case abs_uint32:
		dup_uint_vals = (unsigned int *)dup_vals;
		orig_uint_vals = (unsigned int *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_uint_vals[i]=orig_uint_vals[j];
		break;

	case abs_int32:

		dup_int_vals = (int *)dup_vals;
		orig_int_vals = (int *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_int_vals[i]=orig_int_vals[j];
		break;

	case abs_uint64:
		dup_ulong_vals = (unsigned long *)dup_vals;
		orig_ulong_vals = (unsigned long *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_ulong_vals[i]=orig_ulong_vals[j];
		break;

	case abs_int64:

		dup_long_vals = (long *)dup_vals;
		orig_long_vals = (long *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_long_vals[i]=orig_long_vals[j];
		break;

	case abs_float32:

		dup_float_vals = (float *)dup_vals;
		orig_float_vals = (float *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_float_vals[i]=orig_float_vals[j];
		break;

	case abs_float64:

		dup_double_vals = (double *)dup_vals;
		orig_double_vals = (double *) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			dup_double_vals[i]=orig_double_vals[j];
		break;

	case abs_string:
	
		dup_string_vals = (char **)dup_vals;
		orig_string_vals = (char **) orig_vals;
		for(i=start_index, j=0; j<count; i++,j++)
			strcpy(dup_string_vals[i], orig_string_vals[j]);
		break;

	default:
		/** Error: Invalid Type **/
		return -1;
		break;
  }
  return 1;
}

void * Absff_AllocDefaultMetadataValues(type, size)
 Abs_Type type;
 long size;
{
	void *def_val;

        switch(type){
            /* Determine default value based on type and size*/

		case abs_uint8:
		case abs_uchar8:
			def_val = (unsigned char *) calloc((int)size, sizeof(unsigned char));
			break;

		case abs_int8:
		case abs_char8:

			def_val = (char *) calloc((int)size, sizeof(char));
			break;

		case abs_uint16:

			def_val = (unsigned short *) calloc((int)size, sizeof(unsigned short));
			break;

		case abs_int16:

			def_val = (short *) calloc((int)size, sizeof(short));
			break;

		case abs_uint32:
			def_val = (unsigned int *) calloc((int)size, sizeof(unsigned int));
			break;

		case abs_int32:
		
			def_val = (int *) calloc((int)size, sizeof(int));
			break;

		case abs_uint64:
			
			def_val = (unsigned long *) calloc((int)size, sizeof(unsigned long));
			break;
			
		case abs_int64:

			def_val = (long *) calloc((int)size, sizeof(long));
			break;

		case abs_float32:

			def_val = (float *) calloc((int)size, sizeof(float));
			break;

		case abs_float64:

			def_val = (double *) calloc((int)size, sizeof(double));
			break;

		case abs_string:

			def_val = (char **) calloc((int)size, sizeof(char*));
			break;

		default:
			/** Error: Invalid Type **/
			def_val = NULL;
			break;

        }
	return def_val;
}

void * Absff_ReallocDefaultMetadataValues(vals, type, size)
 void *vals;
 Abs_Type type;
 long size;
{
        switch(type){
            /* Determine default value based on type and size*/

		case abs_uint8:
		case abs_uchar8:
			vals = (unsigned char *) realloc(vals, size * sizeof(unsigned char));
			break;

		case abs_int8:
		case abs_char8:

			vals = (char *) realloc(vals, size*sizeof(char));
			break;

		case abs_uint16:
			vals = (unsigned short *) realloc(vals, size*sizeof(unsigned short));
			break;

		case abs_int16:

			vals = (short *) realloc(vals, size*sizeof(short));
			break;

		case abs_uint32:
			vals = (unsigned int *) realloc(vals, size*sizeof(unsigned int));
			break;

		case abs_int32:
		
			vals = (int *) realloc(vals, size*sizeof(int));
			break;
			
		case abs_int64:
			vals = (long *) realloc(vals, size*sizeof(long));
			break;

		case abs_uint64:

			vals = (unsigned long *) realloc(vals, size*sizeof(unsigned long));
			break;

		case abs_float32:

			vals = (float *) realloc(vals, size*sizeof(float));
			break;

		case abs_float64:

			vals = (double *) realloc(vals, size*sizeof(double));
			break;

		case abs_string:

			vals = (char **) realloc(vals, size*sizeof(char*));
			break;

		default:
			/** Error: Invalid Type **/
			break;

        }
	return vals;
}
 
 

int Absff_DestructMetadata(amdi)
 Abs_Metadata *amdi;
{
 if((amdi == NULL) ||
	(amdi->adi == NULL))
	return -1;

 if(amdi->vals != NULL) {
	free(amdi->vals);
	amdi->vals = NULL;
 }

 return 1;
}
 
int Absff_CreateMetadata(amdi, type, size)
 Abs_Metadata *amdi;
 Abs_Type type;
 long size;
{
 if((amdi == NULL) ||
	(amdi->adi == NULL))
	return -1;

 if(amdi->vals == NULL)
	amdi->vals = Absff_AllocDefaultMetadataValues(type, size); 

 if(amdi->vals == NULL) return -1;
 amdi->abstype = type;
 amdi->size = size; 
 return 1;
}

int Absff_SetMetadataType(amdi, type)
 Abs_Metadata *amdi;
 Abs_Type type;
{
 Abs_Diskio *adi;
 if((amdi == NULL) ||
	(amdi->adi == NULL))
	return -1;

 adi = (struct Abs_Diskio_Type *) amdi->adi;

  if(!adi->is_writable){
		/** Error: Not writable **/
		return -1;
  }
  
  if(amdi->vals != NULL){
	free(amdi->vals);
	amdi->vals = NULL;
  }
  amdi->vals = Absff_AllocDefaultMetadataValues(type, amdi->size);
  amdi->abstype = type;
  return 1;
}

int Absff_SetMetadataSize(amdi, size)
 Abs_Metadata *amdi;
 long size;
{
 Abs_Type type;
 long 	  old_size;
 Abs_Diskio *adi;

  if((amdi == NULL) ||
	(amdi->adi == NULL))
	return -1;

  adi = (struct Abs_Diskio_Type *) amdi->adi;

  if(!adi->is_writable){
		/** Error: Not writable **/
		return -1;
  }
  

  old_size =  amdi->size;
  type =  amdi->abstype;

  if(amdi->vals == NULL)
	amdi->vals = Absff_AllocDefaultMetadataValues(type, size);
  else{
	if(size > old_size)
		 amdi->vals = Absff_ReallocDefaultMetadataValues(amdi->vals, type, size);
  }

  amdi->size = size;

  return 1;
}

int Absff_SetMetadataValues(amdi, vals, start_index, count)
 Abs_Metadata *amdi;
 void *vals;
 long start_index;
 long count; 
{ 

 Abs_Type type;
 long 	  size;
 Abs_Diskio *adi;

 if((amdi == NULL) ||
	(amdi->adi == NULL))
	return -1;

  adi = amdi->adi; 

  if(!adi->is_writable){
		/** Error: Not writable **/
		return -1;
  }
  

  size =  amdi->size;
  type =  amdi->abstype;

  if(count < 0 || count > size || start_index < 0 || start_index > (size -1) ){
	/** Error: Invalid start_index or size **/
	 return -1;
  }

  if(amdi->vals == NULL)
	amdi->vals = Absff_AllocDefaultMetadataValues(type, size);
  if(vals != amdi->vals)
  	Absff_CopyToMetadataValues(amdi->vals, vals, type, size, start_index, count);

  return 1;
}

int Absff_SetMetadataValuesFromString(amdi, vals, start_index, count)
 Abs_Metadata *amdi;
 char **vals;
 long start_index, count;
{
 Abs_Type type;
 Abs_Diskio *adi;
 int i, ret; 
 long size;
 void *values;

 uchar *uchar_values; 
 char *char_values;
 ushort *ushort_values;
 short *short_values;
 uint *uint_values;
 int *int_values;
 ulong *ulong_values; 
 long *long_values;
 float *float_values;
 double *double_values;
 
 
 if(amdi == NULL ||
	amdi->adi == NULL || vals == NULL) 
	return -1;

  adi = amdi->adi;
  if(!adi->is_writable){
		/** Error: Not writable **/
		return -1;
  }
  

  size =  amdi->size;
  type =  amdi->abstype;

  if(count < 0 || count > size ){
	/** Error: Invalid size **/
	 return -1;
  }

  switch (type){
	case abs_uint8:
	case abs_uchar8:
		values = (uchar *) malloc(count * sizeof(unsigned char));
		uchar_values = (uchar*) values;
  		for(i=0; i< count; i++)
			uchar_values[i] = vals[i][0];
		break;

	case abs_int8:
	case abs_char8:
		values = (char *) malloc(count * sizeof(char));
		char_values = (char*) values;
  		for(i=0; i< count; i++)
			char_values[i] = vals[i][0];
		break;

	case abs_uint16:
		values = (ushort*) malloc(count * sizeof(unsigned short));
		ushort_values = (ushort *)values;
  		for(i=0; i< count; i++)
			ushort_values[i] = atoi(vals[i]);
		break;

	case abs_int16:
		values = (short*) malloc(count * sizeof(short));
		short_values = (short *)values;
  		for(i=0; i< count; i++)
			short_values[i] = atoi(vals[i]);
		break;

	case abs_uint32:
		values = (uint *) malloc(count * sizeof(unsigned int));
		uint_values = (uint*) values;
  		for(i=0; i< count; i++)
			uint_values[i] = atoi(vals[i]);
		break;

	case abs_int32:
		values = (int *) malloc(count * sizeof(int));
		int_values = (int*) values;
  		for(i=0; i< count; i++)
			int_values[i] = atoi(vals[i]);
		break;

	case abs_uint64:
		values = (ulong *) malloc(count * sizeof(unsigned long));
		ulong_values = (ulong*) values;
  		for(i=0; i< count; i++)
			ulong_values[i] = Absff_Atol(vals[i]);
		break;
	
	case abs_int64:
		values = (long *) malloc(count * sizeof(long));
		long_values = (long *) values;
  		for(i=0; i< count; i++)
			long_values[i] = Absff_Atol(vals[i]);
		break;
	
	case abs_float32:
		values = (float *) malloc(count * sizeof(float));
		float_values = (float *) values;
		for(i=0; i<count; i++)
			float_values[i] = Absff_Atof(vals[i]);
		break;

	case abs_float64:
		values = (double *) malloc(count * sizeof(double));
		double_values = (double *) values;
		for(i=0; i<count; i++)
			double_values[i] = Absff_Atod(vals[i]);
		break;

	case abs_string:
		/* No conversion necessary */
  		return Absff_SetMetadataValues(amdi, vals, start_index, count);
		break;

	default:
		/* Error: Invalid type */
		return -1;
		
  }
  
  ret = Absff_SetMetadataValues(amdi, values, start_index, count);
  free(values);
  return (ret);
}

/** Errors: -1 for NULL params or other library errors**/

Abs_Type Absff_GetMetadataType(amdi)
 Abs_Metadata *amdi;
{
 if(amdi == NULL || amdi->adi == NULL)
	return (Abs_Type)-1;

 return amdi->abstype;
}

long Absff_GetMetadataSize(amdi)
 Abs_Metadata *amdi;
{
 if(amdi == NULL || amdi->adi == NULL)
	return -1;

 return amdi->size;
}

int Absff_GetMetadataValues(amdi, vals, start_index, count)
 /** Note that vals is appropriately allocated at the call site **/
 Abs_Metadata 	*amdi;
 void 		*vals;
 long		start_index;
 long 		count;
{
 long size;
 Abs_Type type;

 if((amdi == NULL) ||
	(amdi->adi == NULL) || (amdi->vals == NULL))
	return -1;


  size = amdi->size;
  type = amdi->abstype;

  if (count < 0 || count > size || start_index < 0 || start_index > (size -1) ){
	/** Error: Overflow **/
	return -1;
  }


  if(vals != amdi->vals)
	Absff_CopyFromMetadataValues(vals, amdi->vals, type, size, start_index, count);

  return 1;
}

int Absff_GetMetadataValuesAsString(amdi, vals, start_index, count)
 /** Note that vals is appropriately allocated at the call site **/
 /** But the individual elements will be allocated here and freed at the call
	site **/
 Abs_Metadata 	*amdi;
 char 		**vals;
 long		start_index;
 long 		count;
{
 long size;
 Abs_Type type;
 int i, j;

 char buffer[50];

 uchar *uchar_values;
 char *char_values;
 ushort *ushort_values;
 short *short_values;
 uint *uint_values;
 int *int_values;
 ulong *ulong_values;
 long *long_values;
 float *float_values;
 double *double_values;
 char **string_values; 

 if((amdi == NULL) ||
	(amdi->adi == NULL) || (amdi->vals == NULL) || vals == NULL)
	return -1;


  size = amdi->size;
  type = amdi->abstype;

  if (count < 0 || count > size || start_index < 0 || start_index > (size -1) ){
	/** Error: Overflow **/
	return -1;
  }

  switch (type){
	case abs_uint8:
	case abs_uchar8:
		uchar_values = (uchar *)amdi->vals;
		for(i=start_index, j=0; j<count; i++, j++){
			sprintf(buffer, "%c", uchar_values[i]);
			vals[j] = Absff_CopyString(buffer);
		} 
	 	break;
		
	case abs_int8:
	case abs_char8:
		char_values = (char *)amdi->vals;
		for(i=start_index, j=0; j<count; i++, j++){
			sprintf(buffer, "%c", char_values[i]);
			vals[j] = Absff_CopyString(buffer);
		} 
	 	break;
	
	case abs_uint16:
		ushort_values = (ushort *)amdi->vals;
		for(i=start_index, j=0; j<count; i++, j++){
			sprintf(buffer, "%d", ushort_values[i]);
			vals[j] = Absff_CopyString(buffer);
		} 
	 	break;
		
	case abs_int16:
		short_values = (short *)amdi->vals;
		for(i=start_index, j=0; j<count; i++, j++){
			sprintf(buffer, "%d", short_values[i]);
			vals[j] = Absff_CopyString(buffer);
		} 
	 	break;
		
	case abs_uint32:
		uint_values = (uint *)amdi->vals;
		for(i=start_index, j=0; j<count; i++, j++){
			sprintf(buffer, "%d", uint_values[i]);
			vals[j] = Absff_CopyString(buffer);
		} 
	 	break;

	case abs_int32:
		int_values = (int *)amdi->vals;
		for(i=start_index, j=0; j<count; i++, j++){
			sprintf(buffer, "%d", int_values[i]);
			vals[j] = Absff_CopyString(buffer);
		} 
	 	break;

	case abs_uint64:
		ulong_values = (ulong *)amdi->vals;	
		for(i=start_index, j=0; j<count; i++, j++){
			sprintf(buffer, "%ld", ulong_values[i]);
			vals[j] = Absff_CopyString(buffer);
		} 
		break;

	case abs_int64:

		long_values = (long *)amdi->vals;	
		for(i=start_index, j=0; j<count; i++, j++){
			sprintf(buffer, "%ld", long_values[i]);
			vals[j] = Absff_CopyString(buffer);
		} 
		break;

	case abs_float32:
		float_values = (float *)amdi->vals;
		for(i=start_index, j=0; j<count; i++, j++){
			sprintf(buffer, "%g", float_values[i]);
			vals[j] = Absff_CopyString(buffer);
		} 
		break;

	case abs_float64:
		double_values = (double *)amdi->vals;
		for(i=start_index, j=0; j<count; i++, j++){
			sprintf(buffer, "%g", double_values[i]);
			vals[j] = Absff_CopyString(buffer);
		} 
		break;

	case abs_string:
		string_values = (char **)amdi->vals;
		for(i=start_index, j=0; j<count; i++, j++){
			vals[j]=(char *) malloc((strlen(string_values[i])+1) * sizeof(char));
			strcpy(vals[j], string_values[i]);
		} 
		break;
	
	default:
		/* Error: Invalid Type */
		return -1;
  }
  return 1;
}

