/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include "common.h"
#include "error.h"
#include "misc.h"
#include "str.h"
#include "lexer.h"
#include "var.h"
#include "avl.h"

/* Declarations. */

#undef DEBUGGING
/*#define DEBUGGING 1 */
#include "debug-print.h"

/* Variable list. */
static variable **v;

/* Number of variables. */
static int nv;

static int do_value_labels (int);
static int verify_val_labs (int erase);
static int get_label (void);

#if DEBUGGING
static void debug_print (void);
#endif

/* Stubs. */

static void
init (void)
{
  v = NULL;
}

static void
done (void)
{
  free (v);
}

int
cmd_value_labels (void)
{
  int code;
  init ();
  match_id (VALUE);
  match_id (LABELS);
  code = do_value_labels (1);
  done ();
  return code;
}

int
cmd_add_value_labels (void)
{
  int code;
  match_id (ADD);
  match_id (VALUE);
  match_id (LABELS);
  code = do_value_labels (0);
  done ();
  return code;
}

/* Do it. */

static int
do_value_labels (int erase)
{
  match_tok ('/');
  
  while (token != '.')
    {
      parse_variables (NULL, &v, &nv, PV_SAME_TYPE);
      if (!verify_val_labs (erase))
	return -1;
      while (token != '/' && token != '.')
	if (!get_label ())
	  return -1;

      if (token != '/')
	break;
      get_token ();

      free (v);
      v = NULL;
    }

  if (token != '.')
    return syntax_error (NULL);

#if DEBUGGING
  debug_print ();
#endif
  return 1;
}

static int
verify_val_labs (int erase)
{
  int i;

  if (!nv)
    return 1;

  for (i = 0; i < nv; i++)
    {
      variable *vp = v[i];

      if (vp->type == ALPHA && vp->width > 8)
	return msg (SE, _("It is not possible to assign value labels to long "
		    "string variables such as %s."), vp->name);

      if (erase && v[i]->val_lab)
	{
	  avl_destroy (vp->val_lab, free_val_lab);
	  vp->val_lab = NULL;
	}
    }
  return 1;
}

/* Parse all the labels for a particular set of variables and add the
   specified labels to those variables. */
static int
get_label (void)
{
  int i;

  /* Make sure there's some variables. */
  if (!nv)
    {
      if (token != STRING && token != NUM)
	return 0;
      get_token ();
      return 1;
    }

  /* Parse all the labels and add them to the variables. */
  do
    {
      value_label *label;

      /* Allocate label. */
      label = xmalloc (sizeof (value_label));
#if __CHECKER__
      memset (&label->v, 0, sizeof (label->v));
#endif
      label->ref_count = nv;

      /* Set label->v. */
      if (v[0]->type == ALPHA)
	{
	  if (token != STRING)
	    return msg (SE, _("String expected for value."));
	  strbarepadcpy (label->v.s, tokstr, 8, ' ');
	}
      else
	{
	  if (token != NUM)
	    return msg (SE, _("Number expected for value."));
	  if (tokint == NOT_INT)
	    msg (SW, _("Value label `%g' is not integer."), tokval);
	  label->v.f = tokval;
	}

      /* Set label->s. */
      get_token ();
      force_string ();
      if (strlen (tokstr) > 60)
	{
	  msg (SW, _("Truncating value label to 60 characters."));
	  tokstr[60] = 0;
	}
      label->s = xstrdup (tokstr);

      for (i = 0; i < nv; i++)
	{
	  if (!v[i]->val_lab)
	    v[i]->val_lab = avl_create (NULL, val_lab_cmp,
					(void *) (v[i]->width));
	  
	  {
	    value_label *old;
	    
	    old = avl_replace (v[i]->val_lab, label);
	    if (old)
	      free_value_label (old);
	  }
	}

      get_token ();
    }
  while (token != '/' && token != '.');

  return 1;
}

#if DEBUGGING
static void
debug_print ()
{
  int i;

  puts (_("Value labels:"));
  for (i = 0; i < nvar; i++)
    {
      AVLtraverser *t = NULL;
      value_label *val;

      printf ("  %s\n", var[i]->name);
      if (var[i]->val_lab)
	if (var[i]->type == NUMERIC)
	  for (val = avltrav (var[i]->val_lab, &t);
	       val; val = avltrav (var[i]->val_lab, &t))
	    printf ("    %g:  `%s'\n", val->v.f, val->s);
	else
	  for (val = avltrav (var[i]->val_lab, &t);
	       val; val = avltrav (var[i]->val_lab, &t))
	    printf ("    `%.8s':  `%s'\n", val->v.s, val->s);
      else
	printf (_("    (no value labels)\n"));
    }
}
#endif /* DEBUGGING */

/* Compares two value labels and returns a strcmp()-type result. */
int
val_lab_cmp (const void *a, const void *b, void *param)
{
  if ((int) param)
    return strncmp (((value_label *) a)->v.s, ((value_label *) b)->v.s,
		    (int) param);
  else
    {
      int temp = ((value_label *) a)->v.f - ((value_label *) b)->v.f;
      if (temp > 0)
	return 1;
      else if (temp < 0)
	return -1;
      else
	return 0;
    }
}

/* Callback function to increment the reference count for a value
   label. */
void *
inc_ref_count (void *_v, unused void *param)
{
  ((value_label *) _v)->ref_count++;
  return _v;
}

/* Copy the avl tree of value labels and return a pointer to the
   copy. */
avl_tree *
copy_value_labels (avl_tree *src)
{
  avl_tree *dest;

  if (src == NULL)
    return NULL;
  dest = avl_copy (NULL, src, inc_ref_count);

  return dest;
}
