/*
 * Copyright 1995,96,97 Thierry Bousch
 * Licensed under the Gnu Public License, Version 2
 *
 * $Id: mnode.h,v 3.6 1997/04/18 15:49:46 bousch Exp $
 *
 * Mnodes (math-nodes) are the generic structure for handling math
 * objects. Once an mnode is created, its value will never change, so
 * it can be safely referenced by mrefs or other mnodes.
 */

#ifndef _SAML_MNODE_H
#define _SAML_MNODE_H
#include <stdlib.h>
#include "saml.h"
#include "saml-util.h"

typedef struct mnode_header {
	int type;		/* superficial type  */
	int refs;		/* reference counter */
} s_mnode, *mn_ptr;

typedef struct {
	struct mnode_header hdr;	/* see above */
	int length;			/* length of the remaining data */
	mn_ptr x[0];			/* remaining data (appended) */
} std_mnode, *smn_ptr;

s_mnode* mnode_build (int typeid, const char *string);
gr_string* mnode_stringify (s_mnode*);
s_mnode* mnode_make (int, s_mnode*);
s_mnode* mnode_etc (s_mnode*, int, void*);
s_mnode* mnode_promote (s_mnode*, s_mnode*);
s_mnode* mnode_cast (s_mnode*, int);
s_mnode* mnode_add (s_mnode*, s_mnode*);
s_mnode* mnode_sub (s_mnode*, s_mnode*);
s_mnode* mnode_mul (s_mnode*, s_mnode*);
s_mnode* mnode_div (s_mnode*, s_mnode*);
s_mnode* mnode_mod (s_mnode*, s_mnode*);
s_mnode* mnode_gcd (s_mnode*, s_mnode*);
int mnode_notzero (s_mnode*);
int mnode_isneg (s_mnode*);
int mnode_info (s_mnode*, int);
int mnode_differ (s_mnode*, s_mnode*);
int mnode_lessthan (s_mnode*, s_mnode*);
s_mnode* mnode_zero (s_mnode*);
s_mnode* mnode_negate (s_mnode*);
s_mnode* mnode_one (s_mnode*);
s_mnode* mnode_invert (s_mnode*);
s_mnode* mnode_power (s_mnode*, int);
s_mnode* mnode_sqrt (s_mnode*);
s_mnode* mnode_det (s_mnode*);
s_mnode* mnode_diff (s_mnode*, s_mnode*);
s_mnode* mnode_subs (s_mnode*, s_mnode*, s_mnode*);
s_mnode* mnode_elim (s_mnode*, s_mnode*, s_mnode*);
s_mnode* mnode_move_lit (s_mnode*, s_mnode*, s_mnode*);
s_mnode* mn_std_sub (s_mnode*, s_mnode*);
s_mnode* mn_std_div (s_mnode*, s_mnode*);
s_mnode* mn_euclidean_gcd (s_mnode*, s_mnode*);
s_mnode* mn_field_gcd (s_mnode*, s_mnode*);
int mn_std_differ (s_mnode*, s_mnode*);
int mn_std_lessthan (s_mnode*, s_mnode*);
void destroy_mnode (s_mnode*);
void mstd_free (std_mnode*);

static inline void unlink_mnode(s_mnode *n)
{
	register int c = (n->refs) - 1;
	n->refs = c;
	if (!c) destroy_mnode(n);
}

static inline s_mnode* copy_mnode(s_mnode *n)
{
	++(n->refs);
	return n;
}

static inline s_mnode* __mnalloc (int typeid, size_t bytes)
{
	s_mnode *n = malloc(bytes);
#ifndef NDEBUG
	if (!n) panic_out_of_memory();
#endif
	++nb_mnodes_allocated;
	n->type = typeid;
	n->refs = 1;
	return n;
}

static inline std_mnode* mstd_alloc (int typeid, int length)
{
	size_t bytes = sizeof(std_mnode) + length * sizeof(mn_ptr);
	smn_ptr n = (smn_ptr) __mnalloc(typeid, bytes);
	n->length = length;
	return n;
}

/* Error handling */

typedef struct {
	struct mnode_header hdr;	/* see above */
	int number;			/* error number */
	char where[0];			/* where did it occur */
} void_mnode;
	
static inline int saml_errno (s_mnode *mn)
{
	if (mn->type == ST_VOID)
	    return ((void_mnode*)mn)->number;
	return 0;
}

s_mnode* mnode_error (int reason, const char *where);

/* If you really need to peek/poke into mrefs, use this */

mn_ptr mref_to_mnode(mref_t);
mref_t mnode_to_mref(mn_ptr);

/*
 * Here is the data structure used to describe an Mtype (math-type)
 */

typedef struct saml_mtype {
	const char *name;
	void	 (*free) (s_mnode*);
	s_mnode* (*build) (const char*);
	gr_string* (*stringify) (s_mnode*);
	s_mnode* (*make) (s_mnode*);
	s_mnode* (*etc) (s_mnode*, int, void*);
	s_mnode* (*add) (s_mnode*, s_mnode*);
	s_mnode* (*sub) (s_mnode*, s_mnode*);
	s_mnode* (*mul) (s_mnode*, s_mnode*);
	s_mnode* (*div) (s_mnode*, s_mnode*);
	s_mnode* (*gcd) (s_mnode*, s_mnode*);
	int	 (*notzero) (s_mnode*);
	int	 (*isneg) (s_mnode*);
	int	 (*info) (s_mnode*, int);
	int	 (*differ) (s_mnode*, s_mnode*);
	int	 (*lessthan) (s_mnode*, s_mnode*);
	s_mnode* (*zero)   (s_mnode*);
	s_mnode* (*negate) (s_mnode*);
	s_mnode* (*one)    (s_mnode*);
	s_mnode* (*invert) (s_mnode*);
	s_mnode* (*sqrt)   (s_mnode*);
} s_mtype;

typedef struct {
	const char *name;
	void *free, *build, *stringify;
	void *make, *etc;
	void *add, *sub, *mul, *div, *gcd;
	void *notzero, *isneg, *info, *differ, *lessthan;
	void *zero, *negate, *one, *invert, *sqrt;
} unsafe_s_mtype;

#define MAX_MTYPES 256
extern s_mtype* mtype_table[];

#define MTYPE_OK(x)	((unsigned)(x) < MAX_MTYPES)
#define MTYPE_INUSE(x)	(mtype_table[x] != NULL)
#define MTYPE_STD(x)	(mtype_table[x]->free == (void*)mstd_free)

#if 0
/* The real prototypes are the following... */
void register_mtype (int type, s_mtype* desc);
void register_CV_routine (int t1, int t2, mn_ptr (*fn)(mn_ptr,mn_ptr));
#else
/* ...but these ones avoid many spurious warnings. */
void register_mtype (int, void*);
void register_CV_routine (int, int, void*);
#endif

#endif
